/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.controls.dialog.sheetmenu;


import android.app.Activity;
import android.content.Context;
import android.content.DialogInterface;
import android.content.res.ColorStateList;
import android.graphics.Rect;
import android.graphics.Typeface;
import android.util.TypedValue;
import android.view.Gravity;
import android.view.View;
import android.view.ViewGroup;
import android.widget.ImageView;
import android.widget.PopupWindow;

import androidx.core.widget.NestedScrollView;
import androidx.fragment.app.FragmentActivity;

import com.foxit.uiextensions.R;
import com.foxit.uiextensions.controls.dialog.UIBottomSheetFragment;
import com.foxit.uiextensions.controls.dialog.UIPopoverFrag;
import com.foxit.uiextensions.controls.toolbar.BaseBar;
import com.foxit.uiextensions.controls.toolbar.IBaseItem;
import com.foxit.uiextensions.controls.toolbar.impl.BaseItemImpl;
import com.foxit.uiextensions.controls.toolbar.impl.TopBarImpl;
import com.foxit.uiextensions.theme.ThemeUtil;
import com.foxit.uiextensions.utils.AppDisplay;
import com.foxit.uiextensions.utils.AppResource;
import com.foxit.uiextensions.utils.AppUtil;
import com.foxit.uiextensions.utils.SystemUiHelper;

public class UIActionMenu implements IActionMenu {
    private static final String ACTION_MENU_NAME = "ACTION_MENU_NAME";

    private Context mContext;
    Activity mAct;
    private UIBottomSheetFragment mPhoneMenu;
    private UIPopoverFrag mPadMenu;
    private OnActionMenuDismissListener mDismissListener;
    private ViewGroup mContentViewContainer;
    private View mContentView;
    private View mTopBarView;
    private ImageView mDragView;
    private View rootView;
    private boolean mIsPad;
    private int mWidth;
    private int mHeight;
    private int mSliderHeight;
    private boolean mScrollEnabled;
    private int mArrowColor;
    private boolean mAutoResetSystemUiOnDismiss = true;
    private boolean mAutoResetSystemUiOnShow = true;
    private String mName;

    public static IActionMenu newInstance(FragmentActivity context) {
        return new UIActionMenu(context, ACTION_MENU_NAME);
    }

    public static IActionMenu newInstance(FragmentActivity context, String name) {
        return new UIActionMenu(context, name);
    }

    private UIActionMenu(FragmentActivity context, String name) {
        mAct = context;
        mContext = context.getApplicationContext();
        mName = name;
        mIsPad = AppDisplay.isPad();
        mSliderHeight = AppResource.getDimensionPixelSize(mContext, R.dimen.ux_menu_drag_view_height);

        rootView = View.inflate(mContext, R.layout.rd_action_menu_layout, null);
        mDragView = rootView.findViewById(R.id.rd_action_menu_top_line);
        mContentViewContainer = rootView.findViewById(R.id.rd_action_menu_content);
        if (mIsPad) {
            mDragView.setVisibility(View.GONE);
            initPadMenu();
        } else {
            mDragView.setVisibility(View.VISIBLE);
            initPhoneMenu();
        }
    }

    private void initPhoneMenu() {
        if (mPhoneMenu != null) return;
        mPhoneMenu = UIBottomSheetFragment.newInstance((FragmentActivity) mAct, rootView, mName);
        mPhoneMenu.setOnDismissListener(new DialogInterface.OnDismissListener() {
            @Override
            public void onDismiss(DialogInterface dialog) {
                if (mDismissListener != null)
                    mDismissListener.onDismiss(UIActionMenu.this);

                if (mAutoResetSystemUiOnDismiss && SystemUiHelper.getInstance().isFullScreen())
                    SystemUiHelper.getInstance().hideStatusBar(mAct);
                AppUtil.removeViewFromParent(rootView);
                mPhoneMenu = null;
            }
        });
    }

    private void initPadMenu() {
        if (mPadMenu != null) return;
        mPadMenu = UIPopoverFrag.create((FragmentActivity) mAct, rootView, mName, false, false);
        mPadMenu.setShowArrow(true);
        if (mArrowColor != 0)
            mPadMenu.setArrowColor(mArrowColor);
        mPadMenu.setShadowWidth(AppResource.getDimensionPixelSize(mContext, R.dimen.ux_margin_16dp));
        mPadMenu.setOnDismissListener(new PopupWindow.OnDismissListener() {
            @Override
            public void onDismiss() {
                if (mDismissListener != null)
                    mDismissListener.onDismiss(UIActionMenu.this);

                if (mAutoResetSystemUiOnDismiss && SystemUiHelper.getInstance().isFullScreen())
                    SystemUiHelper.getInstance().hideSystemUI(mAct);
                AppUtil.removeViewFromParent(rootView);
                mPadMenu = null;
            }
        });
    }

    public void setDarkModeColor() {
        rootView.setBackgroundColor(mContext.getResources().getColor(R.color.b1));
        mDragView.setImageResource(R.drawable.shape_drag_bg);
        if (topBar != null) {
            topBar.setBackgroundColor(AppResource.getColor(mContext, R.color.b1));
            topBar.setShowSolidLineColor(AppResource.getColor(mContext, R.color.p1));
            titleItem.setTextColor(AppResource.getColor(mContext, R.color.t4));
        }
        if (mIsPad) {
            initPadMenu();
            mPadMenu.setArrowColor(AppResource.getColor(mContext, R.color.b1));
        }
    }

    @Override
    public void show(ViewGroup parent, Rect rect, int arrowPos, int arrowDist) {
//        setDarkModeColor();
        if (!isShowing())
            mContentViewContainer.measure(0, 0);

        int showWidth = mContentViewContainer.getMeasuredWidth();
        int showHeight = mContentViewContainer.getMeasuredHeight();
        if (mIsPad) {
            initPadMenu();
            showWidth = mWidth == 0 ? showWidth : mWidth;
            showHeight = mHeight == 0 ? showHeight : mHeight;

            int rootWidth = parent.getWidth();
            int rootHeight = parent.getHeight();

            int width = Math.min(showWidth, rootWidth);
            int height = Math.min(showHeight, rootHeight - AppResource.getDimensionPixelSize(mContext, R.dimen.ux_topbar_height));
            if (isShowing())
                mPadMenu.update(parent, rect, width, height, arrowPos);
            else
                mPadMenu.showAtLocation(parent, rect, width, height, arrowPos, arrowDist);
        } else {
            initPhoneMenu();
            if (mHeight == 0) {
                mDragView.measure(0, 0);
                int dragHeight = mDragView.getHeight();
                if (dragHeight == 0)
                    dragHeight = mSliderHeight;
                showHeight += dragHeight;
            } else {
                showHeight = mHeight;
            }

            int rootWidth = parent.getWidth();
            int rootHeight = parent.getHeight();
            if (AppDisplay.isLandscape()) {
                int statusBarHeight = 0;
                if (SystemUiHelper.getInstance().isStatusBarShown(mAct)) {
                    statusBarHeight = SystemUiHelper.getInstance().getStatusBarHeight(mContext);
                }
                showWidth = Math.min(rootWidth, rootHeight) + statusBarHeight;
            } else {
                showWidth = rootWidth;
            }
            showHeight = Math.min(showHeight, rootHeight - AppResource.getDimensionPixelSize(mContext, R.dimen.ux_topbar_height));
            if (AppDisplay.getRawScreenHeight() != rootHeight)
                showHeight += AppDisplay.getNavBarHeight();


            mPhoneMenu.setWidth(showWidth);
            mPhoneMenu.setHeight(showHeight);
//            int y = 0;
//            if (AppDisplay.getRawScreenHeight() != rootHeight)
//                y = AppDisplay.getNavBarHeight();
            if (isShowing())
                mPhoneMenu.update(0, 0, showWidth, showHeight);
            else
                mPhoneMenu.showAtLocation(Gravity.BOTTOM, 0, 0);
        }
    }

    @Override
    public void update(ViewGroup parent, Rect rect, int width, int height) {
        if (isShowing()) {
            initPadMenu();
            mPadMenu.update(parent, rect, width, height);
        }
    }

    @Override
    public void update(ViewGroup parent, Rect rect, int width, int height, int arrowPos) {
        if (isShowing()) {
            initPadMenu();
            mPadMenu.update(parent, rect, width, height, arrowPos);
        }
    }

    @Override
    public void update(ViewGroup view, int x, int y, int width, int height) {
        if (isShowing()) {
            initPhoneMenu();
            mPhoneMenu.update(x, y, width, height);
        }
    }

    @Override
    public void dismiss() {
        if (mIsPad) {
            if (mPadMenu != null && mPadMenu.isShowing())
                mPadMenu.dismiss();
        } else if (mPhoneMenu != null) {
            mPhoneMenu.dismiss();
        }
    }

    @Override
    public int getSliderHeight() {
        return mSliderHeight;
    }

    @Override
    public boolean isShowing() {
        if (mIsPad) {
            if (mPadMenu != null)
                return mPadMenu.isShowing();
        } else {
            if (mPhoneMenu != null && mPhoneMenu.getDialog() != null)
                return mPhoneMenu.getDialog().isShowing();
        }
        return false;
    }

    @Override
    public void setWidth(int width) {
        mWidth = width;
    }

    @Override
    public void setHeight(int height) {
        mHeight = height;
    }

    @Override
    public void setOnDismissListener(OnActionMenuDismissListener dismissListener) {
        mDismissListener = dismissListener;
    }

    @Override
    public void setContentView(View view) {
        mContentView = view;
        refreshContentView();
    }

    public View getContentView() {
        return mContentView;
    }

    @Override
    public void setShowOnKeyboard(boolean show) {
        if (mIsPad) {
            initPadMenu();
            mPadMenu.setShowOnKeyboard(show);
        } else {
            initPhoneMenu();
            mPhoneMenu.setShowOnKeyboard(show);
        }
    }

    @Override
    public void setScrollingEnabled(boolean enabled) {
        if (mScrollEnabled != enabled) {
            mScrollEnabled = enabled;
            refreshContentView();
        }
    }

    public void setAutoResetSystemUiOnDismiss(boolean auto) {
        mAutoResetSystemUiOnDismiss = auto;
    }

    @Override
    public void setAutoResetSystemUiOnShow(boolean auto) {
        mAutoResetSystemUiOnShow = auto;
        if (mIsPad) {
            initPadMenu();
            mPadMenu.setAutoResetSystemUiOnShow(mAutoResetSystemUiOnShow);
        }
    }

    @Override
    public void setArrowColor(int color) {
        mArrowColor = color;
        if (mIsPad && mPadMenu != null)
            mPadMenu.setArrowColor(color);
    }

    TopBarImpl topBar;
    IBaseItem titleItem;
    IBaseItem backItem;

    public void setTitleBarView(String title, View.OnClickListener onBackListener) {
        topBar = new TopBarImpl(mContext);
        topBar.setStartMargin(AppResource.getDimensionPixelSize(mContext, R.dimen.ux_margin_4dp));
        topBar.setEndMargin(AppResource.getDimensionPixelSize(mContext, R.dimen.ux_margin_4dp));
        backItem = new BaseItemImpl(mContext);
        backItem.setDisplayStyle(IBaseItem.ItemType.Item_Image);
        backItem.setImageResource(R.drawable.rd_back_vector);
        backItem.setImageTintList(ThemeUtil.getItemIconColor(mContext));
        if (onBackListener == null) {
            backItem.setOnClickListener(new View.OnClickListener() {
                public void onClick(View v) {
                    dismiss();
                }
            });
        } else {
            backItem.setOnClickListener(onBackListener);
        }
        topBar.setBackgroundColor(AppResource.getColor(mContext, R.color.b1));
        topBar.setShowSolidLineColor(AppResource.getColor(mContext, R.color.p1));
        topBar.addView(backItem, BaseBar.TB_Position.Position_LT);
        titleItem = new BaseItemImpl(mContext);
        titleItem.setDisplayStyle(IBaseItem.ItemType.Item_Text);
        titleItem.setText(title);
        titleItem.setTextColor(AppResource.getColor(mContext, R.color.t4));
        titleItem.setTextSize(TypedValue.COMPLEX_UNIT_PX, AppResource.getDimensionPixelSize(mContext, R.dimen.ux_text_size_16sp));
        titleItem.setTypeface(Typeface.defaultFromStyle(Typeface.BOLD));
        topBar.addView(titleItem, BaseBar.TB_Position.Position_CENTER);
        topBar.setMiddleButtonCenter(true);
        mTopBarView = topBar.getContentView();
        refreshContentView();
    }

    public void setBackItemColor(ColorStateList tintColor) {
        if (backItem != null) {
            backItem.setImageTintList(tintColor);
        }
    }

    public void refreshContentView() {
        mContentViewContainer.removeAllViews();
        if (mTopBarView != null) {
            AppUtil.removeViewFromParent(mTopBarView);
            mContentViewContainer.addView(mTopBarView);
        }
        if (mContentView != null) {
            AppUtil.removeViewFromParent(mContentView);

            if (mScrollEnabled) {
                NestedScrollView scrollView = new NestedScrollView(mContext);
                ViewGroup.LayoutParams layoutParams = new ViewGroup.LayoutParams(ViewGroup.LayoutParams.MATCH_PARENT, ViewGroup.LayoutParams.WRAP_CONTENT);
                scrollView.setLayoutParams(layoutParams);
                scrollView.addView(mContentView);
                mContentViewContainer.addView(scrollView);
            } else {
                mContentViewContainer.addView(mContentView);
            }
        }
    }
}
