/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.controls.dialog.sheetmenu;


import android.app.Activity;
import android.content.Context;
import android.graphics.Rect;
import android.graphics.drawable.Drawable;
import android.view.View;
import android.view.ViewGroup;
import android.widget.AdapterView;
import android.widget.ImageView;
import android.widget.LinearLayout;
import android.widget.ListView;
import android.widget.TextView;

import androidx.fragment.app.FragmentActivity;

import com.foxit.uiextensions.R;
import com.foxit.uiextensions.utils.AppDisplay;
import com.foxit.uiextensions.utils.AppResource;

import java.util.ArrayList;
import java.util.List;

public abstract class UISheetMenu implements ISheetMenu {
    protected OnSheetDismissListener mOnSheetDismissListener;
    private OnSheetItemClickListener mOnSheetItemClickListener;

    protected Activity mAct;
    protected SheetMenuAdapter mSheetMenuAdapter;
    protected Context mContext;
    protected List<Integer> mItems;
    protected ListView mListView;
    protected View mContentView;
    protected LinearLayout mSheetContentView;
    protected TextView mTitleTv;
    protected ImageView mDragView;

    private SheetItemBean mSelectedItem;
    private int mWidth;

    public static ISheetMenu newInstance(FragmentActivity context) {
        ISheetMenu sheetMenu;
        if (AppDisplay.isPad())
            sheetMenu = UIPopSheetMenu.create(context);
        else
            sheetMenu = UIBottomSheetMenu.create(context);
        return sheetMenu;
    }

    public LinearLayout getTextDesLayout() {
        return mContentView.findViewById(R.id.rd_sheet_menu_text_des_ly);
    }

    public UISheetMenu(FragmentActivity context, boolean showTopShadow) {
        mAct = context;
        mContext = context.getApplicationContext();
        mItems = new ArrayList<>();

        mContentView = View.inflate(mContext, R.layout.bottom_sheet_layout, null);
        mSheetContentView = mContentView.findViewById(R.id.rd_sheet_menu_container);
        mTitleTv = mContentView.findViewById(R.id.rd_sheet_menu_title);
        mTitleTv.setVisibility(View.GONE);
        mDragView = mContentView.findViewById(R.id.rd_sheet_menu_top_line);
        mDragView.setVisibility(showTopShadow ? View.VISIBLE : View.GONE);
        mListView = mContentView.findViewById(R.id.rd_sheet_menu_list);
        mSheetMenuAdapter = new SheetMenuAdapter(mContext);
        mListView.setAdapter(mSheetMenuAdapter);
        mListView.setOnItemClickListener(new AdapterView.OnItemClickListener() {
            @Override
            public void onItemClick(AdapterView<?> parent, View view, int position, long id) {
                SheetItemBean bean = (SheetItemBean) parent.getItemAtPosition(position);

                if (mSheetMenuAdapter.getMode() == ISheetMenu.SELECTED_MODE) {
                    if (bean.enabled) {
                        if (mSelectedItem != null) {
                            if (mSelectedItem.type == bean.type)
                                return;
                            mSelectedItem.selected = false;
                        }
                        bean.selected = true;
                        mSelectedItem = bean;
                        mSheetMenuAdapter.notifyDataSetChanged();
                    }
                }

                if (mOnSheetItemClickListener != null) {
                    mOnSheetItemClickListener.onItemClick(bean.type);
                }
            }
        });
    }

    @Override
    public void addSheetItem(@ISheetMenu.SheetMenu int type) {
        if (mItems.contains(type))
            return;
        mItems.add(type);
        mSheetMenuAdapter.addSubMenu(type, true);
        mSheetMenuAdapter.notifyDataSetChanged();
    }

    @Override
    public void setSheetItemEnabled(@SheetMenu int type, boolean enabled) {
        mSheetMenuAdapter.setSheetItemEnabled(type, enabled);
    }

    @Override
    public void setSheetItemSelected(@SheetMenu int type, boolean selected) {
        for (SheetItemBean itemBean : mSheetMenuAdapter.getSheetItems()) {
            if (type == itemBean.type && itemBean.enabled) {
                if (mSelectedItem != null) {
                    if (mSelectedItem.type == type)
                        return;
                    mSelectedItem.selected = false;
                }
                itemBean.selected = true;
                mSelectedItem = itemBean;
                mSheetMenuAdapter.notifyDataSetChanged();
                break;
            }
        }
    }

    @Override
    public void addSheetItem(@ISheetMenu.SheetMenu int type, boolean enabled) {
        if (mItems.contains(type))
            return;
        mItems.add(type);
        mSheetMenuAdapter.addSubMenu(type, enabled);
        mSheetMenuAdapter.notifyDataSetChanged();
    }

    @Override
    public void addCustomSheetItem(SheetItemBean itemBean) {
        List<SheetItemBean> itemBeans = mSheetMenuAdapter.getSheetItems();
        if (itemBeans.contains(itemBean)) return;
        itemBeans.add(itemBean);
    }

    @Override
    public void setCustomSheetItem(List<SheetItemBean> itemBeans) {
        mSheetMenuAdapter.clear();
        for (SheetItemBean itemBean : itemBeans) {
            if (itemBean.selected) {
                if (mSelectedItem == null) {
                    mSelectedItem = itemBean;
                } else {
                    itemBean.selected = false;
                }
            }
        }
        mSheetMenuAdapter.setSheetItems(itemBeans);
        mSheetMenuAdapter.notifyDataSetChanged();
    }

    @Override
    public void setSheetItems(List<Integer> items) {
        if (items == null) return;
        mSheetMenuAdapter.clear();
        mItems.clear();
        mItems = items;
        for (int i = 0; i < items.size(); i++) {
            mSheetMenuAdapter.addSubMenu(items.get(i), true);
        }
        mSheetMenuAdapter.notifyDataSetChanged();
    }

    @Override
    public void setSheetItemClickListener(OnSheetItemClickListener itemClickListener) {
        this.mOnSheetItemClickListener = itemClickListener;
    }

    @Override
    public void setOnSheetDismissListener(OnSheetDismissListener dismissListener) {
        mOnSheetDismissListener = dismissListener;
    }

    @Override
    public void setHeight(int height) {
    }

    @Override
    public void update(ViewGroup parent, Rect rect, int arrowPosition) {
    }

    @Override
    public void setMode(int mode) {
        mSheetMenuAdapter.setMode(mode);
    }

    @Override
    public void setTitle(String title) {
        mTitleTv.setVisibility(View.VISIBLE);
        mTitleTv.setText(title);
    }

    private int mMeasureWidth;
    private int mMeasureHeight;

    public void measure() {
        SheetMenuAdapter listAdapter = (SheetMenuAdapter) mListView.getAdapter();
        if (listAdapter == null) {
            return;
        }
        int totalHeight = 0;
        int maxWidth = 0;
        for (int i = 0, len = listAdapter.getCount(); i < len; i++) {
            View listItem = listAdapter.getView(i, null, mListView);
            listItem.measure(0, 0);
            totalHeight += listItem.getMeasuredHeight() + mListView.getDividerHeight();

            int width = listItem.getMeasuredWidth();
            if (width > maxWidth)
                maxWidth = width;
        }
        ViewGroup.LayoutParams params = mListView.getLayoutParams();
        params.height = totalHeight;
        if (mWidth > maxWidth) {
            maxWidth = mWidth;
            params.width = mWidth;
        }
        mListView.setLayoutParams(params);

        if (mTitleTv.getVisibility() == View.VISIBLE) {
            mTitleTv.measure(0, 0);
            int titleHeight = mTitleTv.getHeight();
            if (titleHeight == 0)
                titleHeight = AppResource.getDimensionPixelSize(mContext, R.dimen.ux_divider_with_desc_height);
            totalHeight += titleHeight;

        }
        if (mDragView.getVisibility() == View.VISIBLE) {
            mDragView.measure(0, 0);
            int dragHeight = AppDisplay.dp2px(mDragView.getHeight());
            if (dragHeight == 0) {
                dragHeight = AppResource.getDimensionPixelSize(mContext, R.dimen.ux_menu_drag_view_height);
            }
            totalHeight += dragHeight;
        }

        mMeasureWidth = maxWidth;
        mMeasureHeight = totalHeight;

        if (getTextDesLayout().getVisibility() == View.VISIBLE) {
            getTextDesLayout().measure(0, 0);
            mMeasureHeight += getTextDesLayout().getMeasuredHeight();
        }
    }

    @Override
    public int getMeasureWidth() {
        return mMeasureWidth;
    }

    @Override
    public int getMeasureHeight() {
        return mMeasureHeight;
    }

    @Override
    public void setWidth(int width) {
        mWidth = width;
    }

    @Override
    public View getContentView() {
        return mContentView;
    }

    public void setDivider(Drawable divider) {
        mListView.setDivider(divider);
    }

    public void setDividerHeight(int height) {
        mListView.setDividerHeight(height);
    }

    @Override
    public void updateTheme() {
        mSheetContentView.setBackgroundColor(AppResource.getColor(mContext, R.color.b1));
        mDragView.setBackgroundResource(R.drawable.shape_drag_bg);
        mSheetMenuAdapter.notifyDataSetChanged();
    }

    @Override
    public void setAutoResetSystemUiOnShow(boolean auto) {
    }

    @Override
    public void removeSheetItem(int type){
        for (SheetItemBean itemBean : mSheetMenuAdapter.getSheetItems()) {
            if (type == itemBean.type) {
                mSheetMenuAdapter.getSheetItems().remove(itemBean);
                mSheetMenuAdapter.notifyDataSetChanged();
                break;
            }
        }
    }


}
