/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.controls.filebrowser.imp;

import android.content.Context;
import android.graphics.Bitmap;
import android.view.View;
import android.view.ViewGroup;
import android.widget.CheckBox;
import android.widget.CompoundButton;
import android.widget.ImageView;
import android.widget.TextView;

import com.foxit.uiextensions.R;
import com.foxit.uiextensions.theme.ThemeUtil;
import com.foxit.uiextensions.utils.AppDisplay;
import com.foxit.uiextensions.utils.AppFileUtil;
import com.foxit.uiextensions.utils.AppResource;
import com.foxit.uiextensions.utils.thread.AppThreadManager;

import java.util.Locale;

import static com.foxit.uiextensions.utils.AppResource.getDrawableByFileName;


class FileListAdapter extends FileAdapter {
    private Context mContext;
    private boolean mShowCheckBox = true;
    private boolean mShowFileCount = true;
    private boolean mShowRightArrow = false;
    private boolean mIsPad;

    protected FileListAdapter(IFB_FileAdapterDelegate delegate) {
        super(delegate);
        mContext = delegate.getContext();
        mIsPad = AppDisplay.isPad();
    }

    public void showCheckBox(boolean show) {
        mShowCheckBox = show;
    }

    public void showRightArrow(boolean show) {
        mShowRightArrow = show;
    }

    public void showFileCount(boolean show) {
        mShowFileCount = show;
    }

    @Override
    public int getCount() {
        if (mDelegate.getDataSource() == null) return 0;
        return mDelegate.getDataSource().size();
    }

    @Override
    public FileItem getItem(int position) {
        try {
            return mDelegate.getDataSource().get(position);
        } catch (Throwable t) {
            t.printStackTrace();
        }
        return null;
    }

    @Override
    public long getItemId(int position) {
        return position;
    }

    @Override
    public View getView(int position, View convertView, ViewGroup parent) {
        int sourceSize = mDelegate.getDataSource().size();
        position = (position < sourceSize) ? position : sourceSize - 1;

        final FileItem item = mDelegate.getDataSource().get(position);
        if (item == null) {
            throw new NullPointerException("item == null");
        }
        final ViewHolder holder;
        if (convertView == null) {
            holder = new ViewHolder();
            convertView = View.inflate(mContext, R.layout.fb_file_item_phone, null);

            holder.searchFolderLayout = convertView.findViewById(R.id.fb_item_search_layout);
            holder.searchFolderPathTextView = (TextView) convertView.findViewById(R.id.fb_item_search_path);

            holder.commonLayout = convertView.findViewById(R.id.fb_item_common_layout);
            holder.checkBox = (CheckBox) convertView.findViewById(R.id.fb_item_checkbox);

            holder.iconImageView = (ImageView) convertView.findViewById(R.id.fb_item_icon);
            holder.nameTextView = (TextView) convertView.findViewById(R.id.fb_item_name);
            holder.dateTextView = (TextView) convertView.findViewById(R.id.fb_item_date);
            holder.sizeTextView = (TextView) convertView.findViewById(R.id.fb_item_size);

            holder.fileCount = (TextView) convertView.findViewById(R.id.fb_item_filecount);
            holder.rightArrow = convertView.findViewById(R.id.fb_item_right_arrow);

            holder.checkBox.setTag(new ClickListener(holder.checkBox, position) {
                @Override
                public void onClick(View v) {
                    FileItem item = getItem(this.getPosition());
                    if (((CompoundButton) v).isChecked()) {
                        if (mDelegate.onItemChecked(true, this.getPosition(), item))
                            ((CompoundButton) v).setChecked(true);
                        else
                            ((CompoundButton) v).setChecked(item.checked);
                    } else {
                        if (mDelegate.onItemChecked(false, this.getPosition(), item))
                            ((CompoundButton) v).setChecked(false);
                        else
                            ((CompoundButton) v).setChecked(item.checked);
                    }
                }
            });
            convertView.setTag(holder);
        } else {
            holder = (ViewHolder) convertView.getTag();
            ((ClickListener) holder.checkBox.getTag()).update(position);
        }

        if (mDelegate.isEditState()) {

            if (item.type != FileItem.TYPE_ALL_PDF_FOLDER && item.type != FileItem.TYPE_FOLDER
                    && item.type != FileItem.TYPE_ROOT) {
                holder.checkBox.setVisibility(View.VISIBLE);
                holder.checkBox.setChecked(item.checked);
            } else if (item.type == FileItem.TYPE_FOLDER) {
                holder.checkBox.setVisibility(View.GONE);
            } else {
                holder.checkBox.setVisibility(View.GONE);
            }
        } else {
            holder.checkBox.setVisibility(View.GONE);
        }
        holder.iconImageView.setBackground(null);
        switch (item.type) {
            default:
            case FileItem.TYPE_FILE:
                setVisibility(holder.searchFolderLayout, false);
                setVisibility(holder.commonLayout, true);
//                setVisibility(holder.sizeTextView, true);
                setVisibility(holder.fileCount, false);
                setVisibility(holder.rightArrow, false);
                setIcon(holder.iconImageView, item.path);

                StringBuilder sb = new StringBuilder();
                if (mIsPad) {
                    sb.append(AppFileUtil.getFileExt(item.name).toUpperCase());
                    sb.append(" · ");
                }
                sb.append(item.date);
                sb.append(" · ");
                sb.append(item.size == null ? "" : item.size);
                holder.dateTextView.setText(sb.toString());
                break;
            case FileItem.TYPE_FOLDER:
                setVisibility(holder.searchFolderLayout, false);
                setVisibility(holder.commonLayout, true);
//                setVisibility(holder.sizeTextView, false);
                setVisibility(holder.fileCount, mShowFileCount);
                setVisibility(holder.rightArrow, mShowRightArrow);
                holder.fileCount.setText(item.fileCount + "");
                holder.iconImageView.setImageResource(R.drawable.fb_file_dir);
                holder.dateTextView.setText(item.date);
                break;
            case FileItem.TYPE_ROOT:
                setVisibility(holder.searchFolderLayout, false);
                setVisibility(holder.commonLayout, true);
//                setVisibility(holder.sizeTextView, false);
                setVisibility(holder.fileCount, mShowFileCount);
                setVisibility(holder.rightArrow, mShowRightArrow);
                holder.fileCount.setText(item.fileCount + "");
                holder.iconImageView.setImageResource(R.drawable.fb_file_dir);
                holder.dateTextView.setText(item.date);
                break;
            case FileItem.TYPE_TARGET_FOLDER:
                setVisibility(holder.searchFolderLayout, false);
                setVisibility(holder.commonLayout, true);
//                setVisibility(holder.sizeTextView, false);
                setVisibility(holder.fileCount, mShowFileCount);
                setVisibility(holder.rightArrow, mShowRightArrow);
                holder.fileCount.setText(item.fileCount + "");
                holder.iconImageView.setImageResource(R.drawable.fb_file_dir);
                holder.dateTextView.setText(item.date);
                break;
            case FileItem.TYPE_TARGET_FILE:
                setVisibility(holder.searchFolderLayout, false);
                setVisibility(holder.commonLayout, true);
//                setVisibility(holder.sizeTextView, true);
                setVisibility(holder.fileCount, false);
                setVisibility(holder.rightArrow, false);
                setIcon(holder.iconImageView, item.path);

                sb = new StringBuilder();
                if (mIsPad) {
                    sb.append(AppFileUtil.getFileExt(item.name).toUpperCase());
                    sb.append(" · ");
                }
                sb.append(item.date);
                sb.append(" · ");
                sb.append(item.size == null ? "" : item.size);
                holder.dateTextView.setText(sb.toString());
                break;
            case FileItem.TYPE_ALL_PDF_FILE:
                setVisibility(holder.searchFolderLayout, false);
                setVisibility(holder.commonLayout, true);
//                setVisibility(holder.sizeTextView, true);
                setIcon(holder.iconImageView, item.path);
                setVisibility(holder.fileCount, false);
                setVisibility(holder.rightArrow, false);

                sb = new StringBuilder();
                if (mIsPad) {
                    sb.append(AppFileUtil.getFileExt(item.name).toUpperCase());
                    sb.append(" · ");
                }
                sb.append(item.date);
                sb.append(" · ");
                sb.append(item.size == null ? "" : item.size);
                holder.dateTextView.setText(sb.toString());
                break;
            case FileItem.TYPE_ALL_PDF_FOLDER:
                setVisibility(holder.searchFolderLayout, true);
                setVisibility(holder.commonLayout, false);
                holder.searchFolderPathTextView.setText(item.path == null ? "" : item.path);
                setVisibility(holder.fileCount, false);
                setVisibility(holder.rightArrow, false);
                holder.dateTextView.setText(item.date);
                break;
            case FileItem.TYPE_CLOUD_SELECT_FILE:
                setVisibility(holder.searchFolderLayout, false);
                setVisibility(holder.commonLayout, true);
//                setVisibility(holder.sizeTextView, true);
                setVisibility(holder.fileCount, false);
                setVisibility(holder.rightArrow, false);
                setVisibility(holder.checkBox, mShowCheckBox);
                setIcon(holder.iconImageView, item.path);

                sb = new StringBuilder();
                if (mIsPad) {
                    sb.append(AppFileUtil.getFileExt(item.name).toUpperCase());
                    sb.append(" · ");
                }
                sb.append(item.date);
                sb.append(" · ");
                sb.append(item.size == null ? "" : item.size);
                holder.dateTextView.setText(sb.toString());
                break;
            case FileItem.TYPE_CLOUD_SELECT_FOLDER:
                setVisibility(holder.searchFolderLayout, false);
                setVisibility(holder.commonLayout, true);
//                setVisibility(holder.sizeTextView, false);
                setVisibility(holder.fileCount, mShowFileCount);
                setVisibility(holder.rightArrow, mShowRightArrow);
                setVisibility(holder.checkBox, false);
                holder.fileCount.setText(item.fileCount + "");
                holder.iconImageView.setImageResource(R.drawable.fb_file_dir);
                holder.dateTextView.setText(item.date);
                break;
        }
        holder.searchFolderLayout.setBackgroundColor(mContext.getResources().getColor(R.color.ux_bg_color_all_pdf_path));
        holder.searchFolderPathTextView.setTextColor(mContext.getResources().getColor(R.color.t4));
        convertView.setBackground(mContext.getResources().getDrawable(R.drawable.rd_list_item_bg));
        holder.fileCount.setTextColor(mContext.getResources().getColor(R.color.t3));
        holder.fileCount.setBackground(mContext.getResources().getDrawable(R.drawable.fb_file_item_count_bg));
        holder.nameTextView.setTextColor(mContext.getResources().getColor(R.color.t4));
        holder.dateTextView.setTextColor(mContext.getResources().getColor(R.color.t3));
        holder.sizeTextView.setTextColor(mContext.getResources().getColor(R.color.t3));
        ThemeUtil.setTintList(holder.checkBox, ThemeUtil.getCheckboxColor(mContext));
        String name = item.name == null ? "" : item.name;
        holder.nameTextView.setText(mIsPad ? AppFileUtil.getFileNameWithoutExt(name) : name);
//        holder.sizeTextView.setText(item.size == null ? "" : item.size);
//        holder.dateTextView.setText(item.date == null ? "" : item.date);
        return convertView;
    }

    public static boolean isSupportThumbnail(String name) {
        String extension = name.substring(name.lastIndexOf(".") + 1).toLowerCase(Locale.ROOT);
        if (extension == null || extension.length() == 0) {
            return false;
        } else if (extension.equals("ofd")) {
            return true;
        } else {
            return false;
        }
    }


    private void setIcon(ImageView iconImageView, String path) {
        if (isSupportThumbnail(path)) {
            Bitmap bitmap = FileThumbnail.getInstance(mContext).getThumbnail(path, mThumbnailCallback);
            if (bitmap != null) {
                iconImageView.setBackground(AppResource.getDrawable(mContext, R.drawable.fb_file_pdf_bg));
                iconImageView.setImageBitmap(bitmap);
            }
            return;
        }

        int drawableId = getDrawableByFileName(path);
        if (drawableId == -1) {
            Bitmap bitmap = FileThumbnail.getInstance(mContext).getThumbnail(path, mThumbnailCallback);
            if (bitmap == null) {
                iconImageView.setImageResource(R.drawable.fb_file_pdf);
            } else {
                iconImageView.setBackground(AppResource.getDrawable(mContext, R.drawable.fb_file_pdf_bg));
                iconImageView.setImageBitmap(bitmap);
            }
        } else {
            iconImageView.setImageResource(drawableId);
        }
    }

    private FileThumbnail.ThumbnailCallback mThumbnailCallback = new FileThumbnail.ThumbnailCallback() {
        @Override
        public void result(boolean succeed, final String filePath) {
            if (mDelegate != null && succeed) {
                AppThreadManager.getInstance().runOnUiThread(new Runnable() {
                    @Override
                    public void run() {
                        if (mDelegate != null)
                            mDelegate.updateItem(filePath);
                    }
                });
            }
        }
    };

    private void setVisibility(View view, boolean visible) {
        view.setVisibility(visible ? View.VISIBLE : View.GONE);
    }

    public void updateThumbnail(String filePath, FileThumbnail.ThumbnailCallback callback) {
        FileThumbnail.getInstance(mContext).updateThumbnail(filePath, callback);
    }
}
