/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.controls.menu;


import android.content.Context;
import android.graphics.drawable.Drawable;
import android.view.View;

import com.foxit.uiextensions.modules.more.MoreMenuConstants;

import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.util.List;

import androidx.annotation.ColorInt;
import androidx.annotation.DrawableRes;
import androidx.annotation.IntDef;

/**
 * An Item is returned by calling one of the {@link IMenuView#addGroup}
 * methods.
 * <p>
 * For a feature set of specific menu types, see {@link IMenuView}.
 */
public interface IMenuGroup {

    /*Default group items ui style, no choice*/
    int NONE = 0;
    /*Default group items ui style, single choice */
    int SINGLE_CHOICE = 1;

    @IntDef({
            NONE,
            SINGLE_CHOICE})
    @Retention(RetentionPolicy.SOURCE)
    @interface CheckMode {
    }

    /**
     * Returns the root view of the current menu group.
     *
     * @return the root view of the current menu group.
     */
    View getContentView();

    /**
     * Returns the current menu group id
     *
     * @return the current menu group id{@link MoreMenuConstants#GROUP_ACTION_MENU_PRIMARY}
     */
    int getGroupId();

    /**
     * Change the visibility for the menu group
     *
     * @param visible True if this group should visible, false otherwise.
     */
    void setVisible(boolean visible);

    /**
     * @return the visible status for this group item.
     */
    boolean isVisible();

    /**
     * Change the title associated with this group.
     *
     * @param text The new text to be displayed.
     */
    void setHeaderTitle(String text);

    /**
     * Change the title text color associated with this group.
     *
     * @param color A color value in the form 0xAARRGGBB.
     *              Do not pass a resource ID. To get a color value from a resource ID, call
     *              {@link android.support.v4.content.ContextCompat#getColor(Context, int) getColor}.
     */
    void setHeaderTitleColor(@ColorInt int color);

    /**
     * Sets the background color for this group item.
     *
     * @param color A color value in the form 0xAARRGGBB.
     *              Do not pass a resource ID. To get a color value from a resource ID, call
     *              {@link android.support.v4.content.ContextCompat#getColor(Context, int) getColor}.
     */
    void setHeaderBackgroundColor(@ColorInt int color);

    /**
     * Set header title to be visible or not
     *
     * @param visible True if this group should visible, false otherwise.
     */
    void setHeaderTitleVisible(boolean visible);

    /**
     * Sets the height for header title.
     *
     * @param height A pixel value specified for the header height.
     */
    void setHeaderTitleHeight(@ColorInt int height);

    /**
     * Set divider to be visible or not
     *
     * @param visible Return true to be visible
     */
    void setDividerVisible(boolean visible);

    /**
     * Set the default style of the group item.
     *
     * @param mode The {@link CheckMode} to use
     */
    void setCheckMode(@CheckMode int mode);

    /**
     * @return the default style of the group item.
     */
    @CheckMode
    int getCheckMode();

    /**
     * add menu item with {@link IMenuItem}
     *
     * @param item item to add
     * @return the added item
     */
    IMenuItem addItem(IMenuItem item);

    /**
     * Add a new item to the group. This item displays the given title for its
     * label.
     *
     * @param title The new text to be displayed.
     * @return The newly added menu item.
     */
    IMenuItem addItem(CharSequence title);

    /**
     * Add a new item to the group. This item displays the given icon and title for its
     * label.
     *
     * @param icon  The new icon to be diaplayed.
     * @param title The new text to be displayed.
     * @return The newly added menu item.
     */
    IMenuItem addItem(Drawable icon, CharSequence title);

    /**
     * Add a new item to the group. This item displays the given icon and title for its
     * label.
     *
     * @param itemId Unique item ID. Please refer to existing items {@link MoreMenuConstants#ITEM_PRIMARY_PROTECT}
     * @param icon   The new icon to be diaplayed.
     * @param title  The new text to be displayed.
     * @return The newly added menu item.
     */
    IMenuItem addItem(int itemId, Drawable icon, CharSequence title);

    /**
     * Add a new item to the group. This item displays the given title for its
     * label.
     *
     * @param itemId Unique item ID. Please refer to existing items {@link MoreMenuConstants#ITEM_PRIMARY_PROTECT}
     * @param title  The new text to be displayed.
     * @return The newly added menu item.
     */
    IMenuItem addItem(int itemId, CharSequence title);

    /**
     * Add a new item to the group. This item displays the given title for its
     * label.
     *
     * @param index The position at which to add menu item
     * @param title The new text to be displayed.
     * @return The newly added menu item.
     */
    IMenuItem addItemAt(int index, CharSequence title);

    /**
     * Add a new item to the group. This item displays the given icon and title for its
     * label.
     *
     * @param index   The position at which to add menu item
     * @param iconRes The new icon to be diaplayed.
     * @param title   The new text to be displayed.
     * @return The newly added menu item.
     */
    IMenuItem addItemAt(int index, @DrawableRes int iconRes, CharSequence title);

    /**
     * Add a new item to the group. This item displays the given icon and title for its
     * label.
     *
     * @param index   The position at which to add menu item
     * @param itemId  Unique item ID. Please refer to existing items {@link MoreMenuConstants#ITEM_PRIMARY_PROTECT}
     * @param iconRes The new icon to be diaplayed.
     * @param title   The new text to be displayed.
     * @return The newly added menu item.
     */
    IMenuItem addItemAt(int index, int itemId, @DrawableRes int iconRes, CharSequence title);

    /**
     * Add a custom item to the group.
     *
     * @param customView The custom view
     * @return The newly added menu item.
     */
    IMenuItem addItem(View customView);

    /**
     * Add a custom item to the group.
     *
     * @param itemId     Unique item ID. Please refer to existing items {@link MoreMenuConstants#ITEM_PRIMARY_PROTECT}
     * @param customView The custom view
     * @return The newly added menu item.
     */
    IMenuItem addItem(int itemId, View customView);

    /**
     * Add a custom item to the group.
     *
     * @param index      The position at which to add menu item
     * @param customView The custom view
     * @return The newly added menu item.
     */
    IMenuItem addItemAt(int index, View customView);

    /**
     * Add a custom item to the group.
     *
     * @param index      The position at which to add menu item
     * @param itemId     Unique item ID. Please refer to existing items {@link MoreMenuConstants#ITEM_PRIMARY_PROTECT}
     * @param customView The custom view
     * @return The newly added menu item.
     */
    IMenuItem addItemAt(int index, int itemId, View customView);

    /**
     * Remove the specified menu item.
     *
     * @param item The specified menu item to be removed.
     */
    void removeItem(IMenuItem item);

    /**
     * Remove the specified menu according to item id.
     *
     * @param itemId Unique item ID. Please refer to existing items {@link MoreMenuConstants#ITEM_PRIMARY_PROTECT}
     */
    void removeItem(int itemId);

    /**
     * Remove the menu according to specified position.
     *
     * @param index The position of the menu item
     */
    void removeItemAt(int index);

    /**
     * Gets the menu item by the given id.
     *
     * @param itemId Unique item ID. Please refer to existing items {@link MoreMenuConstants#ITEM_PRIMARY_PROTECT}
     * @return The menu item.
     */
    IMenuItem getItem(int itemId);

    /**
     * Gets the group item at the given index.
     *
     * @param index The index of the menu item to return.
     * @return The menu item.
     */
    IMenuItem getItemAt(int index);

    /**
     * @return the all menu items under the current menu group
     */
    List<IMenuItem> getItems();

    /**
     * Set parent item for current menu group
     *
     * @param parent Current menu group's parent
     * @return Current menu group
     */
    IMenuGroup setParentItem(IMenuItem parent);

    /**
     * Get parent menu item of current menu group
     *
     * @return Current group's parent menu item
     */
    IMenuItem getParentItem();
}
