/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.controls.menu;

import android.content.Context;
import android.graphics.drawable.Drawable;
import android.view.View;
import android.view.ViewGroup;
import android.widget.LinearLayout;
import android.widget.TextView;

import com.foxit.uiextensions.R;
import com.foxit.uiextensions.modules.more.MoreMenuConstants;
import com.foxit.uiextensions.utils.AppUtil;

import java.util.ArrayList;
import java.util.List;

import androidx.annotation.ColorInt;

/**
 * Class <CODE>MenuGroupImpl</CODE> represents the menu group.
 * The menu group can contains a menu item.
 */
public class MenuGroupImpl implements IMenuGroup {
    private Context mContext;
    private View mRootView;
    private LinearLayout mContentList_ly;
    private TextView mTitleView;

    private List<IMenuItem> mMenuItems;
    private int mId = 0;
    private int mMode;

    private View mMenuDividerView;
    private IMenuItem mParentItem;

    /**
     * Constructs a menu group.
     *
     * @param context the context.
     * @param title   The title of the menu group.
     */
    MenuGroupImpl(Context context, CharSequence title) {
        this(context, 0, IMenuGroup.NONE, title);
    }

    /**
     * Constructs a menu group.
     *
     * @param context the context.
     * @param groupId The group tag, {@link MoreMenuConstants#GROUP_ACTION_MENU_PRIMARY}
     * @param title   The title of the menu group.
     */
    MenuGroupImpl(Context context, int groupId, CharSequence title) {
        this(context, groupId, IMenuGroup.NONE, title);
    }

    /**
     * Constructs a menu group.
     *
     * @param context the context.
     * @param groupId The group tag, @
     * @param mode    {@link CheckMode}. if the mode is {@link IMenuGroup.CheckMode#SINGLE_CHOICE},
     *                the check icon will be displayed when item selected.
     * @param title   The title of the menu group.
     */
    MenuGroupImpl(Context context, int groupId, int mode, CharSequence title) {
        this.mId = groupId;
        this.mMode = mode;

        mContext = context;
        mMenuItems = new ArrayList<>();
        mRootView = View.inflate(mContext, R.layout.view_menu_more_group, null);
        mContentList_ly = (LinearLayout) mRootView.findViewById(R.id.menu_more_group_content_ly);

        mMenuDividerView = mRootView.findViewById(R.id.menu_more_item_divider);
        mMenuDividerView.setVisibility(View.GONE);

        //init title
        mTitleView = (TextView) mRootView.findViewById(R.id.menu_more_group_title);
        if (title == null)
            title = "";
        mTitleView.setText(title);
    }

    public void setDarkModeColor() {
        if (mRootView != null) {
            mRootView.setBackgroundColor(mContext.getResources().getColor(R.color.b2));
            mTitleView.setBackgroundColor(mContext.getResources().getColor(R.color.b2));
            mMenuDividerView.setBackgroundColor(mContext.getResources().getColor(R.color.p1));
            mContentList_ly.setBackgroundColor(mContext.getResources().getColor(R.color.b1));
            mTitleView.setTextColor(mContext.getResources().getColor(R.color.t3));
        }
    }

    @Override
    public IMenuItem addItem(CharSequence title) {
        MenuItemImpl menuItem = new MenuItemImpl(mContext, title);
        menuItem.setMenuGroup(this);
        mMenuItems.add(menuItem);
        resetItems();
        return menuItem;
    }

    @Override
    public IMenuItem addItem(IMenuItem menuItem) {
        if (null == menuItem) return null;
        menuItem.setMenuGroup(this);
        mMenuItems.add(menuItem);
        resetItems();
        return menuItem;
    }

    @Override
    public IMenuItem addItem(Drawable icon, CharSequence title) {
        MenuItemImpl menuItem = new MenuItemImpl(mContext, 0, icon, title);
        menuItem.setMenuGroup(this);
        mMenuItems.add(menuItem);
        resetItems();
        return menuItem;
    }

    @Override
    public IMenuItem addItem(int itemId, Drawable icon, CharSequence title) {
        for (IMenuItem menuItem : mMenuItems) {
            if (itemId != 0 && (menuItem.getItemId() == itemId)) {
                removeItem(menuItem);
                break;
            }
        }
        MenuItemImpl menuItem = new MenuItemImpl(mContext, itemId, icon, title);
        menuItem.setMenuGroup(this);
        mMenuItems.add(menuItem);
        resetItems();
        return menuItem;
    }

    @Override
    public IMenuItem addItem(int itemId, CharSequence title) {
        for (IMenuItem menuItem : mMenuItems) {
            if (itemId != 0 && (menuItem.getItemId() == itemId)) {
                removeItem(menuItem);
                break;
            }
        }
        MenuItemImpl menuItem = new MenuItemImpl(mContext, itemId, title);
        menuItem.setMenuGroup(this);
        mMenuItems.add(menuItem);
        resetItems();
        return menuItem;
    }

    @Override
    public IMenuItem addItemAt(int index, CharSequence title) {
        MenuItemImpl item = new MenuItemImpl(mContext, title);
        item.setMenuGroup(this);
        mMenuItems.add(index, item);
        resetItems();
        return item;
    }

    @Override
    public IMenuItem addItemAt(int index, int iconRes, CharSequence title) {
        MenuItemImpl item = new MenuItemImpl(mContext, 0, iconRes, title);
        item.setMenuGroup(this);
        // avoid index > size cause crash
        if (index >= mMenuItems.size())
            index = mMenuItems.size();
        mMenuItems.add(index, item);
        resetItems();
        return item;
    }

    @Override
    public IMenuItem addItemAt(int index, int itemId, int iconRes, CharSequence title) {
        for (IMenuItem menuItem : mMenuItems) {
            if (itemId != 0 && (menuItem.getItemId() == itemId)) {
                removeItem(menuItem);
                break;
            }
        }
        MenuItemImpl item = new MenuItemImpl(mContext, itemId, iconRes, title);
        item.setMenuGroup(this);
        // avoid index > size cause crash
        if (index >= mMenuItems.size())
            index = mMenuItems.size();
        mMenuItems.add(index, item);
        resetItems();
        return item;
    }

    @Override
    public IMenuItem addItem(View customView) {
        MenuItemImpl item = new MenuItemImpl(mContext, customView);
        item.setMenuGroup(this);
        mMenuItems.add(item);
        resetItems();
        return item;
    }

    @Override
    public IMenuItem addItem(int itemId, View customView) {
        MenuItemImpl item = new MenuItemImpl(mContext, itemId, customView);
        item.setMenuGroup(this);
        mMenuItems.add(item);
        resetItems();
        return item;
    }

    @Override
    public IMenuItem addItemAt(int index, View customView) {
        MenuItemImpl item = new MenuItemImpl(mContext, customView);
        item.setMenuGroup(this);
        mMenuItems.add(index, item);
        resetItems();
        return item;
    }

    @Override
    public IMenuItem addItemAt(int index, int itemId, View customView) {
        MenuItemImpl item = new MenuItemImpl(mContext, itemId, customView);
        item.setMenuGroup(this);
        mMenuItems.add(index, item);
        resetItems();
        return item;
    }

    @Override
    public void removeItem(IMenuItem item) {
        if (item == null) return;
        for (IMenuItem menuItem : mMenuItems) {
            if (menuItem == item) {
                mContentList_ly.removeView(item.getContentView());
                mMenuItems.remove(item);
                break;
            }
        }
    }

    @Override
    public void removeItem(int itemId) {
        int index = -1;
        for (int i = 0; i < mMenuItems.size(); i++) {
            IMenuItem menuItem = mMenuItems.get(i);
            if (itemId != 0 && menuItem.getItemId() == itemId) {
                index = i;
                break;
            }
        }

        if (index != -1) {
            IMenuItem menuItem = mMenuItems.get(index);
            mContentList_ly.removeView(menuItem.getContentView());
            mMenuItems.remove(menuItem);
        }
    }

    @Override
    public void removeItemAt(int index) {
        IMenuItem menuItem = mMenuItems.get(index);
        mContentList_ly.removeView(menuItem.getContentView());
        mMenuItems.remove(menuItem);
    }

    @Override
    public void setVisible(boolean visible) {
        getContentView().setVisibility(visible ? View.VISIBLE : View.GONE);
    }

    @Override
    public boolean isVisible() {
        return getContentView().getVisibility() == View.VISIBLE;
    }

    @Override
    public IMenuItem getItem(int itemId) {
        for (IMenuItem menuItem : mMenuItems) {
            if (itemId != 0 && (menuItem.getItemId() == itemId)) {
                return menuItem;
            }
        }
        return null;
    }

    @Override
    public IMenuItem getItemAt(int index) {
        return mMenuItems.get(index);
    }

    @Override
    public List<IMenuItem> getItems() {
        return mMenuItems;
    }

    @Override
    public IMenuGroup setParentItem(IMenuItem parent) {
        mParentItem = parent;
        return this;
    }

    @Override
    public IMenuItem getParentItem() {
        return mParentItem;
    }

    @Override
    public void setHeaderTitle(String text) {
        mTitleView.setText(text);
    }

    @Override
    public void setHeaderTitleColor(@ColorInt int color) {
        mTitleView.setTextColor(color);
    }

    @Override
    public void setHeaderBackgroundColor(@ColorInt int color) {
        mTitleView.setBackgroundColor(color);
    }

    @Override
    public void setHeaderTitleVisible(boolean visible) {
        mTitleView.setVisibility(visible ? View.VISIBLE : View.GONE);
    }

    @Override
    public void setHeaderTitleHeight(int height) {
        mTitleView.getLayoutParams().height = height;
    }

    @Override
    public void setDividerVisible(boolean visible) {
        if (visible)
            mMenuDividerView.setVisibility(View.VISIBLE);
        else
            mMenuDividerView.setVisibility(View.GONE);
    }

    @Override
    public void setCheckMode(int mode) {
        mMode = mode;
    }

    @Override
    public int getCheckMode() {
        return mMode;
    }

    @Override
    public View getContentView() {
        return mRootView;
    }

    @Override
    public int getGroupId() {
        return mId;
    }

    private void resetItems() {
        mContentList_ly.removeAllViews();
        for (IMenuItem item : mMenuItems) {
            addItemToMenu(item);
        }
    }

    private void addItemToMenu(IMenuItem item) {
        AppUtil.removeViewFromParent(item.getContentView());
        mContentList_ly.addView(item.getContentView(), new LinearLayout.LayoutParams(
                ViewGroup.LayoutParams.MATCH_PARENT,
                ViewGroup.LayoutParams.WRAP_CONTENT));
    }

}
