/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.controls.menu;

import android.content.Context;
import android.graphics.drawable.Drawable;
import android.view.View;
import android.widget.ImageView;
import android.widget.LinearLayout;
import android.widget.RelativeLayout;
import android.widget.TextView;

import androidx.annotation.DrawableRes;

import com.foxit.uiextensions.R;
import com.foxit.uiextensions.modules.more.MoreMenuConstants;
import com.foxit.uiextensions.theme.ThemeUtil;
import com.foxit.uiextensions.utils.AppResource;

import java.util.List;

/**
 * Class <CODE>MenuItemImpl</CODE> represents the menu item.
 */
public class MenuItemImpl implements IMenuItem {

    public View customView;
    private TextView mMenuTextView;
    private ImageView mMenuCheckView;
    private View mMenuDividerView;
    private ImageView mMenuImageView;
    private View mView;
    private OnMenuItemClickListener mItemClickListener;
    private IMenuGroup mMenuGroup;

    private int mId = 0;
    private boolean mIsCustomView = false;
    private boolean mIsSelected = false;
    private View mMenuArrowRightView;
    private RelativeLayout mMenuMoreItemLy;
    private Context mContext;

    /**
     * Constructs a menu item.
     *
     * @param context  the context.
     * @param itemText the text of the menu item.
     */
    MenuItemImpl(Context context, CharSequence itemText) {
        this(context, 0, 0, itemText);
    }

    /**
     * Constructs a menu item.
     *
     * @param context  the context.
     * @param itemId   the tag of the menu item.
     *                 constants are defined in {@link MoreMenuConstants}, eg: {@link MoreMenuConstants#ITEM_PRIMARY_PROTECT}.
     *                 it should related to the menu group.
     * @param itemText the text of the menu item.
     */
    MenuItemImpl(Context context, int itemId, CharSequence itemText) {
        this(context, itemId, 0, itemText);
    }

    /**
     * Constructs a menu item.
     *
     * @param context  the context.
     * @param itemId   the tag of the menu item.
     *                 constants are defined in {@link MoreMenuConstants}, eg: {@link MoreMenuConstants#ITEM_PRIMARY_PROTECT}.
     *                 it should related to the menu group.
     * @param itemText the text of the menu item.
     * @param imageID  the image id of the menu item.
     */
    MenuItemImpl(Context context, int itemId, int imageID, CharSequence itemText) {
        this(context, itemId, imageID == 0 ? null : AppResource.getDrawable(context, imageID), itemText);
    }

    public void setDarkModeColor() {
        if (mMenuMoreItemLy == null)
            return;
        mMenuMoreItemLy.setBackground(mContext.getResources().getDrawable(R.drawable.more_menu_item_selector));
        ThemeUtil.setTintList(mMenuImageView, ThemeUtil.getListIconColor(mContext));
        ThemeUtil.setTintList(mMenuTextView, ThemeUtil.getEnableTextColor(mContext));
        ThemeUtil.setTintList(mMenuCheckView, ThemeUtil.getItemIconColor(mContext, false));
        ((ImageView) mMenuArrowRightView).setColorFilter(mContext.getResources().getColor(R.color.i3));
        mMenuDividerView.setBackgroundColor(mContext.getResources().getColor(R.color.p1));
    }

    /**
     * Constructs a menu item.
     *
     * @param context  the context.
     * @param itemId   the tag of the menu item.
     *                 constants are defined in {@link MoreMenuConstants}, eg: {@link MoreMenuConstants#ITEM_PRIMARY_PROTECT}.
     *                 it should related to the menu group.
     * @param itemText the text of the menu item.
     * @param icon     the icon of the menu item.
     */
    MenuItemImpl(Context context, int itemId, Drawable icon, CharSequence itemText) {
        mContext=context;
        mId = itemId;
        mView = View.inflate(context, R.layout.view_menu_more_item, null);
        mMenuTextView = mView.findViewById(R.id.menu_item_text);
        if (itemText == null)
            mMenuTextView.setVisibility(View.INVISIBLE);
        else {
            mMenuTextView.setVisibility(View.VISIBLE);
            mMenuTextView.setText(itemText);
            ThemeUtil.setTintList(mMenuTextView, ThemeUtil.getEnableTextColor(mContext));
        }

        mMenuMoreItemLy = mView.findViewById(R.id.menu_more_item_ly);
        mMenuImageView = mView.findViewById(R.id.menu_item_icon);
        if (icon == null) {
            mMenuImageView.setVisibility(View.GONE);
        } else {
            mMenuImageView.setVisibility(View.VISIBLE);
            mMenuImageView.setImageDrawable(icon);
            ThemeUtil.setTintList(mMenuImageView, ThemeUtil.getListIconColor(context));
        }

        mMenuCheckView = mView.findViewById(R.id.menu_item_check);
        ThemeUtil.setTintList(mMenuCheckView, ThemeUtil.getItemIconColor(context, false));
        mMenuCheckView.setVisibility(View.GONE);
        mMenuDividerView = mView.findViewById(R.id.menu_more_item_divider);
        mMenuDividerView.setVisibility(View.GONE);
        mMenuArrowRightView = mView.findViewById(R.id.menu_item_right_arrow);
        mMenuArrowRightView.setVisibility(View.GONE);
        ((ImageView) mMenuArrowRightView).setColorFilter(context.getResources().getColor(R.color.i3));
        mView.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                if (mMenuGroup != null && mMenuGroup.getCheckMode() == IMenuGroup.SINGLE_CHOICE) {
                    if (mIsSelected) return;

                    List<IMenuItem> menuItems = mMenuGroup.getItems();
                    for (IMenuItem menuItem : menuItems) {
                        menuItem.setSelected(false);
                    }
                    MenuItemImpl.this.setSelected(true);
                }

                if (mItemClickListener != null) {
                    mItemClickListener.onClick(MenuItemImpl.this);
                }
            }
        });
    }

    /**
     * Constructs a menu item.
     *
     * @param context    the context.
     * @param customView The custom view.
     */
    MenuItemImpl(Context context, View customView) {
        this(context, 0, customView);
    }

    /**
     * Constructs a menu item.
     *
     * @param context    the context.
     * @param itemId     the tag of the menu item.
     *                   constants are defined in {@link MoreMenuConstants}, eg: {@link MoreMenuConstants#ITEM_PRIMARY_PROTECT}.
     *                   it should related to the menu group.
     * @param customView The custom view.
     */
    MenuItemImpl(Context context, int itemId, View customView) {
        mContext=context;
        mId = itemId;
        this.customView = customView;
        mView = View.inflate(context, R.layout.view_menu_more_item, null);
        LinearLayout ly = (LinearLayout) mView.getRootView();
        ly.removeAllViews();
        ly.addView(customView);

        mIsCustomView = true;
    }

    @Override
    public void setOnMenuItemClickListener(OnMenuItemClickListener listener) {
        this.mItemClickListener = listener;
    }

    @Override
    public void setDividerVisible(boolean visible) {
        if (mIsCustomView) return;
        if (visible)
            mMenuDividerView.setVisibility(View.VISIBLE);
        else
            mMenuDividerView.setVisibility(View.GONE);
    }

    public void setMenuArrowRightViewVisible(boolean visible) {
        if (mIsCustomView) return;
        mMenuArrowRightView.setVisibility(visible ? View.VISIBLE : View.GONE);
    }

    @Override
    public int getItemId() {
        return mId;
    }

    @Override
    public int getGroupId() {
        if (mMenuGroup != null)
            return mMenuGroup.getGroupId();
        return -1;
    }

    @Override
    public View getContentView() {
        return mView;
    }

    @Override
    public IMenuGroup getMenuGroup() {
        return mMenuGroup;
    }

    @Override
    public void setMenuGroup(IMenuGroup menuGroup) {
        mMenuGroup = menuGroup;
        if (mMenuGroup != null && mMenuGroup.getCheckMode() == IMenuGroup.SINGLE_CHOICE) {
            mMenuCheckView.setVisibility(mIsSelected ? View.VISIBLE : View.GONE);
        }
    }

    @Override
    public void setEnabled(boolean enabled) {
        mView.setEnabled(enabled);
        if (mIsCustomView) return;
        mMenuTextView.setEnabled(enabled);
        mMenuImageView.setEnabled(enabled);
    }

    @Override
    public void setSelected(boolean selected) {
        mIsSelected = selected;
        mView.setSelected(selected);
        if (mIsCustomView) return;
        mMenuTextView.setSelected(selected);
        mMenuImageView.setSelected(selected);
        if (mMenuGroup != null && mMenuGroup.getCheckMode() == IMenuGroup.SINGLE_CHOICE) {
            mMenuCheckView.setVisibility(selected ? View.VISIBLE : View.GONE);
        }
    }

    @Override
    public void setTitle(CharSequence title) {
        if (mIsCustomView) return;
        mMenuTextView.setText(title);
    }

    @Override
    public CharSequence getTitle() {
        if (mIsCustomView) return "";
        return mMenuTextView.getText();
    }

    @Override
    public void setIcon(@DrawableRes int resId) {
        if (mIsCustomView) return;
        mMenuImageView.setImageResource(resId);
    }

    @Override
    public void setVisible(boolean visible) {
        getContentView().setVisibility(visible ? View.VISIBLE : View.GONE);
    }

    @Override
    public boolean getVisible() {
        return getContentView().getVisibility() == View.VISIBLE;
    }

}
