/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.controls.menu;

import android.app.Activity;
import android.content.Context;
import android.content.res.Configuration;
import android.graphics.Rect;
import android.graphics.drawable.Drawable;
import android.view.View;

import com.foxit.uiextensions.R;
import com.foxit.uiextensions.UIExtensionsManager;
import com.foxit.uiextensions.controls.dialog.MatchDialog;
import com.foxit.uiextensions.controls.dialog.UIMatchDialog;
import com.foxit.uiextensions.controls.dialog.UIPopoverWin;
import com.foxit.uiextensions.controls.dialog.sheetmenu.IActionMenu;
import com.foxit.uiextensions.controls.dialog.sheetmenu.UIActionMenu;
import com.foxit.uiextensions.controls.toolbar.BaseBar;
import com.foxit.uiextensions.modules.more.MoreMenuConstants;
import com.foxit.uiextensions.theme.ThemeUtil;
import com.foxit.uiextensions.utils.AppDisplay;
import com.foxit.uiextensions.utils.AppResource;
import com.foxit.uiextensions.utils.SystemUiHelper;
import com.foxit.uiextensions.utils.thread.AppThreadManager;

import java.util.List;

import androidx.annotation.DrawableRes;
import androidx.fragment.app.FragmentActivity;

/**
 * Class <CODE>SubgroupMenuItemImpl</CODE> represents the menu item which contains subgroup.
 */
public class SubgroupMenuItemImpl extends MenuItemImpl {

    private IMenuGroup mSubgroup;
    private UIExtensionsManager mUIExtensionsManager;
    private MatchDialog mDialog;
    private UIMatchDialog dialog;
    private UIActionMenu mActionMenu;
    private String mTitleText;
    private OnActionMenuBackListener mActionMenuBackListener;
    private boolean mResetSystemUiOnDismiss = true;

    public SubgroupMenuItemImpl(Context context, int itemId, CharSequence itemText, UIExtensionsManager uiExtensionsManager) {
        this(context, itemId, null, itemText, uiExtensionsManager);
    }

    public SubgroupMenuItemImpl(Context context, int itemId, Drawable icon, CharSequence itemText, UIExtensionsManager uiExtensionsManager) {
        super(context, itemId, icon, itemText);
        initMenu(context, itemId, itemText, uiExtensionsManager);
    }

    private void initMenu(Context context, int itemId, CharSequence itemText, UIExtensionsManager uiExtensionsManager) {
        mUIExtensionsManager = uiExtensionsManager;
        mNightMode = context.getResources().getConfiguration().uiMode & Configuration.UI_MODE_NIGHT_MASK;
        setSubgroup(new MenuGroupImpl(context, itemId, itemText).setParentItem(this));
        getSubgroup().setHeaderTitleVisible(false);
        mTitleText = itemText == null ? "" : itemText.toString();
        setMenuArrowRightViewVisible(true);
        setOnMenuItemClickListener(new OnMenuItemClickListener() {
            @Override
            public void onClick(IMenuItem item) {
                showDialog();
            }
        });
    }

    /**
     * Set the subgroup of the menu item
     *
     * @param subgroup The subgroup of the menu item to set
     */
    public void setSubgroup(IMenuGroup subgroup) {
        mSubgroup = subgroup;
    }

    /**
     * Get the subgroup of the menu item
     *
     * @return The subgroup of the menu item
     */
    public IMenuGroup getSubgroup() {
        return mSubgroup;
    }

    /**
     * show full screen dialog which contain child menu items
     */
    public void showDialog() {
        if (null == mUIExtensionsManager) return;
        Activity activity = mUIExtensionsManager.getAttachedActivity();
        if (null == activity) return;
        if (null == mDialog) {
            initDialog(activity);
        }
        mDialog.showDialog();
    }

    public void setDartModeColor() {
        if (mSubgroup != null && mSubgroup instanceof MenuGroupImpl) {
            ((MenuGroupImpl) mSubgroup).setDarkModeColor();
            for (int i = 0; i < mSubgroup.getItems().size(); i++) {
                ((MenuItemImpl) mSubgroup.getItems().get(i)).setDarkModeColor();
            }
        }
        if (mActionMenu != null) {
            mActionMenu.setDarkModeColor();
        }
    }

    public void updateThemeColor() {
        if (mSubgroup != null && mSubgroup instanceof MenuGroupImpl) {
            ((MenuGroupImpl) mSubgroup).setDarkModeColor();
            for (int i = 0; i < mSubgroup.getItems().size(); i++) {
                ((MenuItemImpl) mSubgroup.getItems().get(i)).setDarkModeColor();
            }
        }
    }

    /**
     * The interface to handle action menu ui back click listener
     */
    public interface OnActionMenuBackListener {
        void onClick(SubgroupMenuItemImpl item);
    }

    /**
     * Set the action menu ui back click listener
     *
     * @param backListener Back click listener
     * @return The menu item itself
     */
    public SubgroupMenuItemImpl setOnActionMenuBackListener(OnActionMenuBackListener backListener) {
        mActionMenuBackListener = backListener;
        return this;
    }

    private UIExtensionsManager.ConfigurationChangedListener mConfigurationChangedListener;
    private int mNightMode = -1;

    /**
     * Show action menu dialog which contain child menu items
     */
    public void showActionMenu(final View anchorView) {
        if (null == mUIExtensionsManager) return;
        Activity context = mUIExtensionsManager.getAttachedActivity();
        if (null == context) return;
        if (mActionMenu == null) {
            initActionMenu(context);
        }
        if (mConfigurationChangedListener == null) {
            mConfigurationChangedListener = new UIExtensionsManager.ConfigurationChangedListener() {
                @Override
                public void onConfigurationChanged(Configuration newConfig) {
                    if (mNightMode != (newConfig.uiMode & Configuration.UI_MODE_NIGHT_MASK)) {
                        mNightMode = newConfig.uiMode & Configuration.UI_MODE_NIGHT_MASK;
//                        setDartModeColor();
//                        dismiss();
                        return;
                    }
                    if (mActionMenu.isShowing())
                        mUIExtensionsManager.getRootView().postDelayed(new Runnable() {
                            @Override
                            public void run() {
                                showActionMenu(anchorView);
                            }
                        }, 100);
                }
            };
            mUIExtensionsManager.registerConfigurationChangedListener(mConfigurationChangedListener);
        }

        Rect rect = new Rect();
        anchorView.getGlobalVisibleRect(rect);
        mActionMenu.setAutoResetSystemUiOnDismiss(false);
        mActionMenu.show(mUIExtensionsManager.getRootView(), rect, UIPopoverWin.ARROW_TOP, 0);
        mResetSystemUiOnDismiss = true;
    }

    public void initActionMenu(Activity context) {
        mActionMenu = (UIActionMenu) UIActionMenu.newInstance((FragmentActivity) mUIExtensionsManager.getAttachedActivity());
        final SubgroupMenuItemImpl item = this;
        View.OnClickListener clickListener = mActionMenuBackListener == null ? null : new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                mActionMenuBackListener.onClick(item);
            }
        };
        mActionMenu.setTitleBarView(getTitle().toString(), clickListener);
        mActionMenu.refreshContentView();
        mActionMenu.setContentView(mSubgroup.getContentView());
        mActionMenu.setOnDismissListener(new IActionMenu.OnActionMenuDismissListener() {
            @Override
            public void onDismiss(IActionMenu actionMenu) {
                if (mResetSystemUiOnDismiss) {
                    if (SystemUiHelper.getInstance().isFullScreen()) {
//                        if (AppDisplay.isPad())
                        SystemUiHelper.getInstance().hideSystemUI(mUIExtensionsManager.getAttachedActivity());
//                        else
//                            SystemUiHelper.getInstance().hideStatusBar(mUIExtensionsManager.getAttachedActivity());
                    }
                }
                AppThreadManager.getInstance().getMainThreadHandler().postDelayed(new Runnable() {
                    @Override
                    public void run() {
                        if (mConfigurationChangedListener != null) {
                            mUIExtensionsManager.unregisterConfigurationChangedListener(mConfigurationChangedListener);
                            mConfigurationChangedListener = null;
                        }
                    }
                }, 100);
            }
        });
        if (AppDisplay.isPad())
            mActionMenu.setWidth(AppResource.getDimensionPixelSize(context, R.dimen.ux_pad_top_menu_width));
        setActionMenu(mActionMenu);
    }

    private void initDialog(Activity activity) {
        dialog = new UIMatchDialog(activity);
        dialog.setTitlePosition(BaseBar.TB_Position.Position_CENTER);
        dialog.setBackButtonVisible(View.VISIBLE);
        dialog.setBackButtonStyle(MatchDialog.TEXT_BACK);
        dialog.setBackButtonText(AppResource.getString(mUIExtensionsManager.getAttachedActivity(), R.string.fx_string_close));
        dialog.setBackButtonTintList(ThemeUtil.getItemIconColor(activity));
        dialog.setTitle(mTitleText);
        dialog.getTitleBar().setMiddleButtonCenter(true);
        dialog.setContentView(mSubgroup.getContentView());
        dialog.setListener(new MatchDialog.DialogListener() {
            @Override
            public void onResult(long btType) {
            }

            @Override
            public void onBackClick() {
                mDialog.dismiss();
                if (mActionMenuBackListener != null) {
                    mActionMenuBackListener.onClick(SubgroupMenuItemImpl.this);
                }
            }

            @Override
            public void onTitleRightButtonClick() {
            }
        });
        setDialog(dialog);
    }

    public void setDialog(MatchDialog dialog) {
        mDialog = dialog;
    }

    public void setActionMenu(UIActionMenu actionMenu) {
        mActionMenu = actionMenu;
    }

    public void setUIExtensionsManager(UIExtensionsManager uiExtensionsManager) {
        this.mUIExtensionsManager = uiExtensionsManager;
    }

    /**
     * Hide view which contains child menu items
     */
    public void hide() {
        if (mDialog != null && mDialog.isShowing()) {
            mDialog.dismiss();
        }
        if (mActionMenu != null && mActionMenu.isShowing()) {
            mResetSystemUiOnDismiss = false;
            mActionMenu.dismiss();
        }
    }

    public void dismiss() {
        if (mDialog != null) {
            mDialog.dismiss();
            mDialog = null;
        }
        if (mActionMenu != null) {
            mResetSystemUiOnDismiss = true;
            mActionMenu.dismiss();
            mActionMenu = null;
        }
    }

    /**
     * Add sub menu item with another one
     *
     * @param item The item to add
     * @return The added item
     */
    public IMenuItem addSubItem(IMenuItem item) {
        return mSubgroup.addItem(item);
    }

    /**
     * Add sub item with title
     *
     * @param title The item's title
     * @return The added item
     */
    public IMenuItem addSubItem(CharSequence title) {
        return mSubgroup.addItem(title);
    }

    /**
     * Add sub item with icon, title
     *
     * @param title The sitem's title
     * @return The added item
     */
    public IMenuItem addSubItem(Drawable icon, CharSequence title) {
        return mSubgroup.addItem(icon, title);
    }

    /**
     * Add sub item with item id, icon, title
     *
     * @param itemId The item's id
     * @param icon   The item's icon
     * @param title  The item's title
     * @return The added item
     */
    public IMenuItem addSubItem(int itemId, Drawable icon, CharSequence title) {
        return mSubgroup.addItem(itemId, icon, title);
    }

    public IMenuItem addSubItem(int itemId, CharSequence title) {
        return mSubgroup.addItem(itemId, title);
    }

    public IMenuItem addSubItem(View customView) {
        return mSubgroup.addItem(customView);
    }

    public IMenuItem addSubItem(int tag, View customView) {
        return mSubgroup.addItem(tag, customView);
    }

    public IMenuItem addSubItemAt(int index, CharSequence title) {
        return mSubgroup.addItemAt(index, title);
    }

    public IMenuItem addSubItemAt(int index, @DrawableRes int iconRes, CharSequence title) {
        return mSubgroup.addItemAt(index, iconRes, title);
    }

    /**
     * Add sub item with item id, icon, title at the specify index
     *
     * @param itemId  The item's id
     * @param iconRes The item's icon resource id
     * @param title   The item's title
     * @return The added item
     */
    public IMenuItem addSubItemAt(int index, int itemId, @DrawableRes int iconRes, CharSequence title) {
        return mSubgroup.addItemAt(index, itemId, iconRes, title);
    }

    public IMenuItem addSubItemAt(int index, View customView) {
        return mSubgroup.addItemAt(index, customView);
    }

    public IMenuItem addSubItemAt(int index, int itemId, View customView) {
        return mSubgroup.addItemAt(index, itemId, customView);
    }

    /**
     * Remove the specified sub menu item.
     *
     * @param item The specified menu item to be removed.
     */
    public void removeSubItem(IMenuItem item) {
        mSubgroup.removeItem(item);
    }

    /**
     * Remove the specified menu according to item id.
     *
     * @param itemId Unique item ID. Please refer to existing items {@link MoreMenuConstants#ITEM_PROTECT_REDACTION}
     */
    public void removeSubItem(int itemId) {
        mSubgroup.removeItem(itemId);
    }

    /**
     * Remove the menu according to specified position.
     *
     * @param index The position of the menu item
     */
    public void removeSubItemAt(int index) {
        mSubgroup.removeItemAt(index);
    }

    /**
     * Get the sub item by the item's id
     *
     * @param itemId The item's id
     * @return The menu item
     */
    public IMenuItem getSubItem(int itemId) {
        return mSubgroup.getItem(itemId);
    }

    /**
     * Get the sub items of the menu item
     *
     * @return The items of the subgroup
     */
    public List<IMenuItem> getSubItems() {
        return mSubgroup.getItems();
    }
}
