/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.controls.panel;

import android.view.View;
import android.view.ViewGroup;

/**
 * The interface that defines information for Panel container,
 * thc container include topbar,tabbar and panel content.
 * <p>
 * Through this interface you can add/remove/get/set panel via {@link PanelSpec}
 */
public interface PanelHost {
    /**
     * Interface used to allow the creator of a PanelHost to run some code when the
     * specified view is closed.
     */
    interface ICloseDefaultPanelCallback {
        /**
         * This method will be invoked when the specified view is closed.
         *
         * @param v The specified view.
         */
        void closeDefaultPanel(View v);
    }

    /**
     * Retrieve the {@link View} attached to this PanelHost, if present.
     *
     * @return The View attached to the dialog or null if no View is present.
     */
    ViewGroup getContentView();

    /**
     * Add a specified {@link PanelSpec}  panelto the PanelHost.
     *
     * @param spec The specified PanelSpec.
     */
    void addSpec(PanelSpec spec);

    /**
     * <p>Adds a panel view. If the panel type{@link PanelSpec#getPanelType()} already exists, will not be added again.
     * reference{@link PanelSpec#BOOKMARKS}</p>
     *
     * @param index the position at which to add the panel
     * @param spec  the panel to add
     */
    void addSpec(int index, PanelSpec spec);

    /**
     * Remove the specified {@link PanelSpec} panel from the PanelHost.
     *
     * @param spec The specified PanelSpec.
     */
    void removeSpec(PanelSpec spec);

    /**
     * Remove the specified {@link PanelSpec} panel type from the PanelHost.
     *
     * @param panelType the panel type {@link PanelSpec#BOOKMARKS}
     */
    void removeSpec(int panelType);

    /**
     * Returns the position in the group of the specified panel.
     *
     * @param spec the panel for which to get the position
     * @return a positive integer representing the position of the panel in the
     * group, or -1 if the panel does not exist in the group
     */
    int indexOf(PanelSpec spec);

    /**
     * Set a panel type start with{@link PanelSpec#BOOKMARKS} as the current PanelSpec.
     *
     * @param panelType The specified PanelType.
     */
    void setCurrentSpec(int panelType);

    /**
     * Get the current {@link PanelSpec}.
     *
     * @return The current {@link PanelSpec}.
     */
    PanelSpec getCurrentSpec();

    /**
     * Get the {@link PanelSpec} by the panel type start with {@link PanelSpec#BOOKMARKS}
     *
     * @param panelType he specified panel type.
     * @return The {@link PanelSpec} associated with the panel type start with {@link PanelSpec#BOOKMARKS}.
     */
    PanelSpec getSpec(int panelType);

    /**
     * Set whether the panel title bar is displayed
     *
     * @param visible true means show the panel title bar
     */
    void setTabVisibility(boolean visible);

}
