/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.controls.panel.impl;

import android.content.Context;
import android.content.res.ColorStateList;
import android.graphics.PorterDuff;
import android.graphics.drawable.Drawable;
import android.graphics.drawable.LayerDrawable;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.FrameLayout;
import android.widget.ImageView;
import android.widget.LinearLayout;
import android.widget.RelativeLayout;
import android.widget.TextView;

import com.foxit.sdk.PDFViewCtrl;
import com.foxit.uiextensions.IUIInteractionEventListener;
import com.foxit.uiextensions.R;
import com.foxit.uiextensions.UIExtensionsManager;
import com.foxit.uiextensions.controls.ViewPagerAdapter;
import com.foxit.uiextensions.controls.panel.PanelContentViewPage;
import com.foxit.uiextensions.controls.panel.PanelHost;
import com.foxit.uiextensions.controls.panel.PanelSpec;
import com.foxit.uiextensions.modules.panel.bookmark.ReadingBookmarkModule;
import com.foxit.uiextensions.theme.ThemeConfig;
import com.foxit.uiextensions.theme.ThemeUtil;
import com.foxit.uiextensions.utils.AppDisplay;
import com.foxit.uiextensions.utils.AppResource;
import com.google.android.material.tabs.TabLayout;

import java.util.ArrayList;
import java.util.List;

import androidx.annotation.NonNull;
import androidx.viewpager.widget.ViewPager;

public class PanelHostImpl implements PanelHost {
    private Context mContext;
    private ViewGroup mRootView;
    private LinearLayout mTopLayout;
    private TabLayout mTabLayout;
    private TextView mNoPanelView;
    private FrameLayout mNoPanelTopbarView;
    private ArrayList<PanelSpec> mSpecs;
    private PanelSpec mCurSpec;
    private UIExtensionsManager mUIExtensionsManager;
    private PanelContentViewPage mContentViewPager;
    private List<View> mViewPagerList;
    private int[] mIdList = new int[]{R.id.rd_panel_tab_item_bookmarks,
            R.id.rd_panel_tab_item_outline, R.id.rd_panel_tab_item_comments, R.id.rd_panel_tab_item_attachments, R.id.rd_panel_tab_item_digital_signatures};
    private ICloseDefaultPanelCallback mColseDefaultPanelListener;
    private ViewPagerAdapter mViewPagerAdapter;

    public PanelHostImpl(Context context, PDFViewCtrl.UIExtensionsManager uiExtensionsManager, ICloseDefaultPanelCallback closeDefaultPanelListener) {
        mColseDefaultPanelListener = closeDefaultPanelListener;
        mContext = context.getApplicationContext();
        mUIExtensionsManager = (UIExtensionsManager) uiExtensionsManager;
        mRootView = (ViewGroup) View.inflate(context, R.layout.root_panel, null);
        mTopLayout = mRootView.findViewById(R.id.panel_topbar_layout);
        mTabLayout = mRootView.findViewById(R.id.panel_tabbar_layout);
        mNoPanelView = mRootView.findViewById(R.id.panel_content_noinfo);
        mContentViewPager = mRootView.findViewById(R.id.panel_content_viewpager);

        mSpecs = new ArrayList<PanelSpec>();
        mViewPagerList = new ArrayList<View>();
        mViewPagerAdapter = new ViewPagerAdapter(mViewPagerList);
        mContentViewPager.setAdapter(mViewPagerAdapter);
        mContentViewPager.addOnPageChangeListener(new ViewPager.OnPageChangeListener() {
            @Override
            public void onPageScrolled(int position, float positionOffset, int positionOffsetPixels) {

            }

            @Override
            public void onPageSelected(int position) {
                if (mCurSpec != null && mCurSpec.getPanelType() != mSpecs.get(position).getPanelType()) {
                    statisticsPanel();
                    setCurrentSpec(mSpecs.get(position).getPanelType());
                    TabLayout.Tab tab = mTabLayout.getTabAt(position);
                    if (tab != null)
                        tab.select();
                }
            }

            @Override
            public void onPageScrollStateChanged(int state) {

            }
        });

        updateTabLayoutBackgroundColor();
        mTabLayout.setTabMode(TabLayout.MODE_FIXED);
        mTabLayout.setTabGravity(TabLayout.GRAVITY_FILL);
        mTabLayout.setSelectedTabIndicatorColor(ThemeConfig.getInstance(mContext).getPrimaryColor());
        mTabLayout.addOnTabSelectedListener(new TabLayout.OnTabSelectedListener() {
            @Override
            public void onTabSelected(TabLayout.Tab tab) {
                int position = tab.getPosition();

                if (mCurSpec != null && mCurSpec.getPanelType() != mSpecs.get(position).getPanelType()) {
                    statisticsPanel();
                    setCurrentSpec(mSpecs.get(position).getPanelType());
                }
            }


            @Override
            public void onTabUnselected(TabLayout.Tab tab) {
            }

            @Override
            public void onTabReselected(TabLayout.Tab tab) {
            }
        });
    }

    @Override
    public ViewGroup getContentView() {
        return mRootView;
    }

    @Override
    public PanelSpec getSpec(int panelType) {
        for (PanelSpec spec : mSpecs) {
            if (spec.getPanelType() == panelType) {
                return spec;
            }
        }
        return null;
    }

    @Override
    public void setTabVisibility(boolean visible) {
        if (visible) {
            mTopLayout.setVisibility(View.VISIBLE);
            mTabLayout.setVisibility(View.VISIBLE);
            mContentViewPager.setScrollable(true);
        } else {
            mTopLayout.setVisibility(View.GONE);
            mTabLayout.setVisibility(View.GONE);
            mContentViewPager.setScrollable(false);
        }
    }

    @Override
    public void addSpec(PanelSpec spec) {
        if (getSpec(spec.getPanelType()) != null)
            return;
        int index = -1;
        for (int i = 0; i < mSpecs.size(); i++) {
            if (mSpecs.get(i).getPanelType() > spec.getPanelType()) {
                index = i;
                break;
            }
        }
        _addSpec(index, spec);
    }

    @Override
    public void addSpec(int index, PanelSpec spec) {
        if (getSpec(spec.getPanelType()) != null)
            return;
        _addSpec(index, spec);
    }

    private void _addSpec(int index, PanelSpec spec) {
        if (mNoPanelView.getVisibility() == View.VISIBLE) {
            mNoPanelView.setVisibility(View.GONE);
            mContentViewPager.setVisibility(View.VISIBLE);
        }

        if (index == -1 || index > mSpecs.size()) {
            mSpecs.add(spec);
            mViewPagerList.add(spec.getContentView());
        } else {
            mSpecs.add(index, spec);
            mViewPagerList.add(index, spec.getContentView());
        }
        mViewPagerAdapter.notifyDataSetChanged();

        int tabIndex = mSpecs.indexOf(spec);
        addTab(tabIndex, spec);
        if (mCurSpec == null || mSpecs.size() == 1) {
            setFocuses(0);
        } else {
            if (!(mCurSpec instanceof ReadingBookmarkModule)) {
                setFocuses(mSpecs.indexOf(mCurSpec));
            }
        }
    }

    @Override
    public void removeSpec(PanelSpec spec) {
        int index = mSpecs.indexOf(spec);
        removeSpec(index, spec);
    }

    @Override
    public void removeSpec(int panelType) {
        for (int i = 0; i < mSpecs.size(); i++) {
            PanelSpec spec = mSpecs.get(i);
            if (spec.getPanelType() == panelType) {
                removeSpec(i, spec);
                break;
            }
        }
    }

    @Override
    public int indexOf(PanelSpec spec) {
        return mSpecs.indexOf(spec);
    }

    private void removeSpec(int index, PanelSpec spec) {
        if (index < 0) return;

        mSpecs.remove(index);
        mViewPagerList.remove(index);
        mViewPagerAdapter.notifyDataSetChanged();

        removeTab(spec);
        if (mSpecs.size() > index) {
            setFocuses(index);
        } else {
            setFocuses(mSpecs.size() - 1);
        }
    }

    @Override
    public void setCurrentSpec(int panelType) {
        if (mCurSpec != null) {
//            if (mCurSpec.getPanelType() == panelType) {
//                mCurSpec.onActivated();
//                return;
//            }
            mCurSpec.onDeactivated();
            mTabLayout.selectTab(null);
        }

        if (mSpecs.size() == 0) {
            mNoPanelView.setVisibility(View.VISIBLE);
            mContentViewPager.setVisibility(View.GONE);
            mTabLayout.setVisibility(View.GONE);
            mTopLayout.removeAllViews();
            mTopLayout.addView(getNoPanelTopbar());
            return;
        }

        PanelSpec panelSpec = getSpec(panelType);
        if (panelSpec != null) {
            setFocuses(mSpecs.indexOf(panelSpec));
            panelSpec.onActivated();
        } else {
            setFocuses(0);
            mSpecs.get(0).onActivated();
        }
    }

    @Override
    public PanelSpec getCurrentSpec() {
        return mCurSpec;
    }

    private void addTab(int index, @NonNull PanelSpec spec) {
        // icon view
        ImageView iconView = new ImageView(mContext);
        iconView.setId(R.id.rd_panel_tab_item);
        iconView.setImageResource(spec.getIcon());
        ColorStateList colorStateList;
        if (spec.getIconTint() == null)
            colorStateList = ThemeUtil.getPrimaryIconColor(mContext);
        else
            colorStateList = spec.getIconTint();
        ThemeUtil.setTintList(iconView, colorStateList);

        RelativeLayout.LayoutParams iconLayoutParams = new RelativeLayout.LayoutParams(
                RelativeLayout.LayoutParams.MATCH_PARENT, RelativeLayout.LayoutParams.WRAP_CONTENT);
        iconLayoutParams.addRule(RelativeLayout.CENTER_IN_PARENT, RelativeLayout.TRUE);

        RelativeLayout tabItemView = new RelativeLayout(mContext);
        tabItemView.addView(iconView, iconLayoutParams);

        TabLayout.Tab tab = mTabLayout.newTab();
        tab.setTag(spec.getPanelType());
        tab.setCustomView(tabItemView);
        if (index < mIdList.length)
            tab.view.setId(mIdList[index]);
        mTabLayout.addTab(tab, index);
    }

    void removeTab(PanelSpec spec) {
        for (int i = 0; i < mTabLayout.getTabCount(); i++) {
            TabLayout.Tab tab = mTabLayout.getTabAt(i);
            if (tab != null && tab.getTag() != null) {
                if (((Integer) tab.getTag()) == spec.getPanelType()) {
                    mTabLayout.removeTab(tab);
                    break;
                }
            }
        }
    }

    private void setFocuses(int index) {
        if (index < 0 || index > mSpecs.size() - 1) {
            index = 0;
        }
        if (mSpecs.size() == 0) {
            mNoPanelView.setVisibility(View.VISIBLE);
            mContentViewPager.setVisibility(View.GONE);
            mTabLayout.setVisibility(View.GONE);
            mTopLayout.removeAllViews();
            mTopLayout.addView(getNoPanelTopbar());
            return;
        }

        mCurSpec = mSpecs.get(index);
        mTopLayout.removeAllViews();
        mTopLayout.addView(mCurSpec.getTopToolbar());
        mTabLayout.setVisibility(View.VISIBLE);
        mContentViewPager.setCurrentItem(index);
        TabLayout.Tab tab = mTabLayout.getTabAt(index);
        if (tab != null)
            tab.select();
    }

    private TextView mNoPanelCloseTv;
    private TextView mNoPanelTitle;

    private View getNoPanelTopbar() {
        if (mNoPanelTopbarView == null) {
            mNoPanelTopbarView = (FrameLayout) LayoutInflater.from(mContext).inflate(R.layout.panel_no_panel_topbar, null, false);

            mNoPanelCloseTv = mNoPanelTopbarView.findViewById(R.id.panel_no_panel_topbar_close);
            mNoPanelTitle = mNoPanelTopbarView.findViewById(R.id.panel_no_panel_title);
            mNoPanelCloseTv.setTextColor(ThemeUtil.getPrimaryTextColor(mContext));
            // set top side offset
            if (AppDisplay.isPad()) {
                mNoPanelCloseTv.setVisibility(View.GONE);
            } else {
                if (mColseDefaultPanelListener != null) {
                    mNoPanelCloseTv.setVisibility(View.VISIBLE);
                    mNoPanelCloseTv.setOnClickListener(new View.OnClickListener() {
                        @Override
                        public void onClick(View v) {
                            mColseDefaultPanelListener.closeDefaultPanel(v);
                        }
                    });
                } else {
                    mNoPanelCloseTv.setVisibility(View.GONE);
                }
            }
        }
        return mNoPanelTopbarView;
    }

    public void onThemeChange(String type, int color) {
        if (mTabLayout != null) {
            for (int i = 0; i < mTabLayout.getTabCount(); i++) {
                TabLayout.Tab tab = mTabLayout.getTabAt(i);
                if (tab != null && tab.getCustomView() != null) {
                    PanelSpec panelSpec = null;
                    for (PanelSpec spec : mSpecs) {
                        if (spec.getPanelType() == (int) tab.getTag()) {
                            panelSpec = spec;
                            break;
                        }
                    }
                    if (panelSpec == null) return;

                    ImageView iconView = (ImageView) ((RelativeLayout) tab.getCustomView()).getChildAt(0);
                    ColorStateList colorStateList;
                    if (panelSpec.getIconTint() == null)
                        colorStateList = ThemeUtil.getPrimaryIconColor(mContext);
                    else
                        colorStateList = panelSpec.getIconTint();
                    ThemeUtil.setTintList(iconView, colorStateList);
                }
            }
            updateTabLayoutBackgroundColor();
            mTabLayout.setSelectedTabIndicatorColor(ThemeConfig.getInstance(mContext).getPrimaryColor());
        }

        if (mNoPanelTopbarView != null) {
            mNoPanelTopbarView.setBackgroundColor(ThemeConfig.getInstance(mContext).getB2());
            mNoPanelCloseTv.setTextColor(ThemeUtil.getPrimaryTextColor(mContext));
            mNoPanelView.setTextColor(AppResource.getColor(mContext, R.color.t2));
            mNoPanelTitle.setTextColor(AppResource.getColor(mContext, R.color.t4));
        }
    }

    private void statisticsPanel() {
        switch (mCurSpec.getPanelType()) {
            case PanelSpec.BOOKMARKS:
                mUIExtensionsManager.onUIInteractElementClicked(IUIInteractionEventListener.Reading_Panel_Bookmark);
                break;
            case PanelSpec.OUTLINE:
                mUIExtensionsManager.onUIInteractElementClicked(IUIInteractionEventListener.Reading_Panel_Outline);
                break;
            case PanelSpec.ANNOTATIONS:
                mUIExtensionsManager.onUIInteractElementClicked(IUIInteractionEventListener.Reading_Panel_Comment);
                break;
            case PanelSpec.ATTACHMENTS:
                mUIExtensionsManager.onUIInteractElementClicked(IUIInteractionEventListener.Reading_Panel_Attachment);
                break;
            case PanelSpec.SIGNATURES:
                mUIExtensionsManager.onUIInteractElementClicked(IUIInteractionEventListener.Reading_Panel_DigitalSignature);
                break;
            default:
                break;
        }
    }

    private void updateTabLayoutBackgroundColor(){
        mTabLayout.setBackground(AppResource.getDrawable(mContext, R.drawable.panel_tab_bg));
        if (mTabLayout.getBackground() instanceof LayerDrawable) {
            LayerDrawable layerDrawable = (LayerDrawable) mTabLayout.getBackground();
            if (layerDrawable.getNumberOfLayers() > 1) {
                Drawable drawable = layerDrawable.getDrawable(1);
                drawable.setColorFilter(ThemeConfig.getInstance(mContext).getB2(), PorterDuff.Mode.SRC);
            }
        }
    }
}
