/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.controls.propertybar.imp;

import android.app.Activity;
import android.content.Context;
import android.content.DialogInterface;
import android.content.res.ColorStateList;
import android.graphics.Color;
import android.os.Build;
import android.util.SparseBooleanArray;
import android.view.View;
import android.widget.ImageView;
import android.widget.LinearLayout;
import android.widget.RelativeLayout;
import android.widget.TextView;

import androidx.annotation.NonNull;

import com.foxit.sdk.PDFViewCtrl;
import com.foxit.sdk.common.Renderer;
import com.foxit.uiextensions.IUIInteractionEventListener;
import com.foxit.uiextensions.R;
import com.foxit.uiextensions.UIExtensionsManager;
import com.foxit.uiextensions.annots.common.UIFillView;
import com.foxit.uiextensions.config.Config;
import com.foxit.uiextensions.controls.dialog.SimpleBottomWindow;
import com.foxit.uiextensions.controls.propertybar.IViewSettingsWindow;
import com.foxit.uiextensions.controls.propertybar.PropertyBar;
import com.foxit.uiextensions.pdfreader.MainCenterItemBean;
import com.foxit.uiextensions.theme.ThemeConfig;
import com.foxit.uiextensions.theme.ThemeUtil;
import com.foxit.uiextensions.utils.AppResource;

import java.util.HashMap;
import java.util.Map;

public class ViewSettingsWindow implements IViewSettingsWindow {
    private SimpleBottomWindow mBottomViewSettingsDialog;
    private SimpleBottomWindow mPageColorWindow;
    private SimpleColorView mColorView;
    private Integer mCustomPageColor;

    // private LinearLayout mPageModeLayout;

    private UIFillView mIvSingle;
    private UIFillView mIvFacing;
    private TextView mTvFacing;
    private UIFillView mIvCover;
    private TextView mTvCover;

    private UIFillView mIvDay;
    private UIFillView mIvPageColor;
    private UIFillView mIvNight;

    //    private LinearLayout mDayNightLayout;
//    private LinearLayout mDayLayout;
//    private LinearLayout mPageColorLayout;
//    private LinearLayout mNightLayout;
//
//    private LinearLayout mContinuousPageLayout;
    private LinearLayout mFitPageLayout;
    private LinearLayout mFitWidthLayout;
    private LinearLayout mReflowLayout;
    private LinearLayout mCropPageLayout;
    private LinearLayout mSpeakLayout;
    private LinearLayout mAutoFlipLayout;
    private LinearLayout mRotateViewLayout;
    private LinearLayout mPanZoomLayout;

    private RelativeLayout mContinuousPageButtonLayout;
    private ImageView mIvContinuousPage;
    private TextView mContinuousPageTv;
    private ImageView mContinuousPageIv;

    private ImageView mRTLTitleIv;
    private TextView mRTLTitleTv;
    private RelativeLayout mRTL_SwitchLayout;
    private ImageView mRTL_SwitchIcon;

    private ImageView mIvFitPageSelected;
    private ImageView mIvFitWidthSelected;
    private ImageView mIvCropPageSelected;

    private View mContentView;
    private final PDFViewCtrl mPdfView;
    private final Context mContext;

    private final SparseBooleanArray mSetVisibleViews;
    private final SparseBooleanArray mEnabledArray;
    private final Map<Integer, Integer> mIdsMap = new HashMap<>();
    private final Map<Integer, IViewSettingsWindow.IValueChangeListener> mListeners;
    private int mPageMode = PDFViewCtrl.PAGELAYOUTMODE_SINGLE;
    private @PageColorMode
    int mPageColorMode;
    private int mZoomMode = PDFViewCtrl.ZOOMMODE_FITPAGE;
    private OnDismissListener mOnDismissListener;
    private Context mActivityContext;

    private boolean mInitPageColor;

    public ViewSettingsWindow(@NonNull Context context, PDFViewCtrl pdfViewCtrl) {
        mPdfView = pdfViewCtrl;
        mContext = context.getApplicationContext();
        mListeners = new HashMap<>();
        mSetVisibleViews = new SparseBooleanArray();
        mEnabledArray = new SparseBooleanArray();
        mInitPageColor = false;

        //if (mPdfView.isNightMode() && mPdfView.getColorMode() == Renderer.e_ColorModeMappingGray) {
        Config config = ((UIExtensionsManager) mPdfView.getUIExtensionsManager()).getConfig();
        if (config.uiSettings.colorMode.equals("Night")) {
            mPageColorMode = IViewSettingsWindow.NIGHT;
        } else if (config.uiSettings.colorMode.equals("Map")) {
            mPageColorMode = IViewSettingsWindow.CUSTOM_COLOR;
        } else {
            mPageColorMode = IViewSettingsWindow.DAY;
        }
    }

    public void initView(Activity activity) {
        if (activity == null || mContentView != null) return;

        mActivityContext = activity;
        mContentView = View.inflate(mContext, R.layout.bottom_bar_view_setting_layout, null);
        mBottomViewSettingsDialog = new SimpleBottomWindow(mActivityContext, mContentView);
        mBottomViewSettingsDialog.setOnDismissListener(new DialogInterface.OnDismissListener() {
            @Override
            public void onDismiss(DialogInterface dialog) {
                if (mOnDismissListener != null) {
                    mOnDismissListener.onDismiss();
                }
            }
        });
        initialize(mContentView);

        int visiableArraySize = mSetVisibleViews.size();
        for (int i = 0; i < visiableArraySize; i++) {
            int type = mSetVisibleViews.keyAt(i);
            setVisible(type, mSetVisibleViews.get(type));
        }

        for (int i = 0; i < mEnabledArray.size(); i++) {
            enableBar(mEnabledArray.keyAt(i), mEnabledArray.valueAt(i));
        }
        mEnabledArray.clear();

        initializePageColor(mActivityContext);
    }

    private void initialize(View view) {
//        mPageModeLayout = view.findViewById(R.id.ll_page_mode);

        mIvSingle = view.findViewById(R.id.iv_single_page_icon);
        mIvSingle.setOnClickListener(mClickListener);
        mIvSingle.setTintList(getBorderColorStateList(), ThemeUtil.getPrimaryIconColor(mContext, true));

        mTvFacing = view.findViewById(R.id.tv_facing_page_icon);
        ThemeUtil.setTintList(mTvFacing, ThemeUtil.getToolbarTextColor(mContext));
        mIvFacing = view.findViewById(R.id.iv_facing_page_icon);
        mIvFacing.setOnClickListener(mClickListener);
        mIvFacing.setTintList(getBorderColorStateList(), ThemeUtil.getPrimaryIconColor(mContext, true));

        mTvCover = view.findViewById(R.id.tv_cover_page_icon);
        ThemeUtil.setTintList(mTvCover, ThemeUtil.getToolbarTextColor(mContext));
        mIvCover = view.findViewById(R.id.iv_cover_page_icon);
        mIvCover.setOnClickListener(mClickListener);
        mIvCover.setTintList(getBorderColorStateList(), ThemeUtil.getPrimaryIconColor(mContext, true));

//        mDayNightLayout = view.findViewById(R.id.ll_daynight);
//        mDayLayout = view.findViewById(R.id.ll_day);
//        mPageColorLayout= view.findViewById(R.id.ll_page_color);
//        mNightLayout = view.findViewById(R.id.ll_night);

        mIvDay = view.findViewById(R.id.iv_day_icon);
        mIvDay.setOnClickListener(mClickListener);
        mIvDay.setForceDarkAllowed(false);

        mIvPageColor = view.findViewById(R.id.iv_page_color_icon);
        mIvPageColor.setOnClickListener(mClickListener);
        mIvPageColor.setForceDarkAllowed(false);

        mIvNight = view.findViewById(R.id.iv_night_icon);
        mIvNight.setOnClickListener(mClickListener);
        mIvNight.setSrcTintList(getNightSrcStateList());
        mIvNight.setFillColorFilter(Color.parseColor("#FF2E2E2E"));

//        mContinuousPageLayout = view.findViewById(R.id.ll_continuous_page);
        mFitPageLayout = view.findViewById(R.id.ll_fit_page);
        mFitPageLayout.setOnClickListener(mClickListener);
        ThemeUtil.setViewTintList(mFitPageLayout, ThemeUtil.getListIconColor(mContext), ThemeUtil.getListTextColor(mContext));
        mFitWidthLayout = view.findViewById(R.id.ll_fit_width);
        mFitWidthLayout.setOnClickListener(mClickListener);
        ThemeUtil.setViewTintList(mFitWidthLayout, ThemeUtil.getListIconColor(mContext), ThemeUtil.getListTextColor(mContext));
        mReflowLayout = view.findViewById(R.id.ll_reflow);
        mReflowLayout.setOnClickListener(mClickListener);
        ThemeUtil.setViewTintList(mReflowLayout, ThemeUtil.getListIconColor(mContext), ThemeUtil.getListTextColor(mContext));
        mCropPageLayout = view.findViewById(R.id.ll_crop);
        mCropPageLayout.setOnClickListener(mClickListener);
        ThemeUtil.setViewTintList(mCropPageLayout, ThemeUtil.getListIconColor(mContext), ThemeUtil.getListTextColor(mContext));
        mSpeakLayout = view.findViewById(R.id.ll_speak);
        mSpeakLayout.setOnClickListener(mClickListener);
        ThemeUtil.setViewTintList(mSpeakLayout, ThemeUtil.getEnableIconColor(mContext), ThemeUtil.getEnableTextColor(mContext));
        mAutoFlipLayout = view.findViewById(R.id.ll_auto_flip);
        mAutoFlipLayout.setOnClickListener(mClickListener);
        ThemeUtil.setViewTintList(mAutoFlipLayout, ThemeUtil.getListIconColor(mContext), ThemeUtil.getListTextColor(mContext));

        mRotateViewLayout = view.findViewById(R.id.ll_rotate_view);
        mRotateViewLayout.setOnClickListener(mClickListener);
        ThemeUtil.setViewTintList(mRotateViewLayout, ThemeUtil.getListIconColor(mContext), ThemeUtil.getListTextColor(mContext));
        mPanZoomLayout = view.findViewById(R.id.ll_pan_zoom);
        mPanZoomLayout.setOnClickListener(mClickListener);
        ThemeUtil.setViewTintList(mPanZoomLayout, ThemeUtil.getListIconColor(mContext), ThemeUtil.getListTextColor(mContext));

        mContinuousPageIv = view.findViewById(R.id.iv_continuous_page_icon);
        ThemeUtil.setTintList(mContinuousPageIv, ThemeUtil.getPrimaryIconColor(mContext));
        mContinuousPageTv = view.findViewById(R.id.tv_continuous_page_title);
        ThemeUtil.setTintList(mContinuousPageTv, ThemeUtil.getToolbarTextColor(mContext));
        mContinuousPageButtonLayout = view.findViewById(R.id.ll_continuous_page_button);
        mContinuousPageButtonLayout.setOnClickListener(mClickListener);
        ThemeUtil.setBackgroundTintList(mContinuousPageButtonLayout, getSelectedButtonColorStateList());
        mIvContinuousPage = view.findViewById(R.id.continuous_page_button);

        mRTLTitleIv = view.findViewById(R.id.iv_rtl_icon);
        ThemeUtil.setTintList(mRTLTitleIv, ThemeUtil.getPrimaryIconColor(mContext));
        mRTLTitleTv = view.findViewById(R.id.tv_rtl_title);
        ThemeUtil.setTintList(mRTLTitleTv, ThemeUtil.getToolbarTextColor(mContext));
        mRTL_SwitchLayout = view.findViewById(R.id.rela_rtl_layout);
        mRTL_SwitchLayout.setOnClickListener(mClickListener);
        ThemeUtil.setBackgroundTintList(mRTL_SwitchLayout, getSelectedButtonColorStateList());
        mRTL_SwitchIcon = view.findViewById(R.id.iv_rtl_switch_button);

        mIvFitPageSelected = view.findViewById(R.id.iv_fit_page_selected_icon);
        mIvFitWidthSelected = view.findViewById(R.id.iv_fit_width_selected_icon);
        mIvCropPageSelected = view.findViewById(R.id.iv_crop_selected_icon);

        initializeMap();
        if (Build.VERSION.SDK_INT >= 29) { // 29 = Build.VERSION_CODES.Q
            mContinuousPageButtonLayout.setForceDarkAllowed(false);
            mRTL_SwitchLayout.setForceDarkAllowed(false);
            mFitPageLayout.setForceDarkAllowed(false);
            mFitWidthLayout.setForceDarkAllowed(false);
            mReflowLayout.setForceDarkAllowed(false);
            mCropPageLayout.setForceDarkAllowed(false);
            mSpeakLayout.setForceDarkAllowed(false);
            mAutoFlipLayout.setForceDarkAllowed(false);
            mRotateViewLayout.setForceDarkAllowed(false);
            mPanZoomLayout.setForceDarkAllowed(false);
        }
    }

    private ColorStateList getNightSrcStateList() {
        int disabled = AppResource.getColor(mContext, R.color.ux_color_white);
        int normal = AppResource.getColor(mContext, R.color.ux_color_white);
        return AppResource.createColorStateList(mContext, disabled, normal);
    }

    private void initializeMap() {
        mIdsMap.put(IViewSettingsWindow.TYPE_SINGLE_PAGE, R.id.ll_single_page);
        mIdsMap.put(IViewSettingsWindow.TYPE_FACING_PAGE, R.id.ll_facing_page);
        mIdsMap.put(IViewSettingsWindow.TYPE_COVER_PAGE, R.id.ll_cover_page);
        mIdsMap.put(IViewSettingsWindow.TYPE_DAY, R.id.ll_day);
        mIdsMap.put(IViewSettingsWindow.TYPE_PAGE_COLOR, R.id.ll_page_color);
        mIdsMap.put(IViewSettingsWindow.TYPE_NIGHT, R.id.ll_night);
        mIdsMap.put(IViewSettingsWindow.TYPE_CONTINUOUS_PAGE, R.id.ll_continuous_page);
        mIdsMap.put(IViewSettingsWindow.TYPE_FIT_WIDTH, R.id.ll_fit_width);
        mIdsMap.put(IViewSettingsWindow.TYPE_FIT_PAGE, R.id.ll_fit_page);
        mIdsMap.put(IViewSettingsWindow.TYPE_REFLOW, R.id.ll_reflow);
        mIdsMap.put(IViewSettingsWindow.TYPE_CROP, R.id.ll_crop);
        mIdsMap.put(IViewSettingsWindow.TYPE_TTS, R.id.ll_speak);
        mIdsMap.put(IViewSettingsWindow.TYPE_AUTO_FLIP, R.id.ll_auto_flip);
        mIdsMap.put(IViewSettingsWindow.TYPE_ROTATE_VIEW, R.id.ll_rotate_view);
        mIdsMap.put(IViewSettingsWindow.TYPE_PAN_ZOOM, R.id.ll_pan_zoom);
        mIdsMap.put(IViewSettingsWindow.TYPE_RIGHT_TO_LEFT, R.id.ll_rtl);
    }

    private View.OnClickListener mClickListener = new View.OnClickListener() {
        @Override
        public void onClick(View v) {
            int id = v.getId();
            if (id == R.id.iv_single_page_icon) {
                if (mListeners.get(IViewSettingsWindow.TYPE_SINGLE_PAGE) != null) {
                    if (mPageMode != PDFViewCtrl.PAGELAYOUTMODE_SINGLE) {
                        mPageMode = PDFViewCtrl.PAGELAYOUTMODE_SINGLE;
                        mListeners.get(IViewSettingsWindow.TYPE_SINGLE_PAGE).onValueChanged(IViewSettingsWindow.TYPE_SINGLE_PAGE, mPageMode);
                    }
                }

                mIvSingle.setSelected(true);
                mIvFacing.setSelected(false);
                mIvCover.setSelected(false);
                ((UIExtensionsManager) mPdfView.getUIExtensionsManager()).onUIInteractElementClicked(IUIInteractionEventListener.Reading_View_SinglePage);
            } else if (id == R.id.iv_facing_page_icon) {
                if (mListeners.get(IViewSettingsWindow.TYPE_SINGLE_PAGE) != null) {
                    if (mPageMode != PDFViewCtrl.PAGELAYOUTMODE_FACING) {
                        mPageMode = PDFViewCtrl.PAGELAYOUTMODE_FACING;
                        mListeners.get(IViewSettingsWindow.TYPE_SINGLE_PAGE).onValueChanged(IViewSettingsWindow.TYPE_SINGLE_PAGE, mPageMode);
                    }
                }

                mIvSingle.setSelected(false);
                mIvFacing.setSelected(true);
                mIvCover.setSelected(false);
                ((UIExtensionsManager) mPdfView.getUIExtensionsManager()).onUIInteractElementClicked(IUIInteractionEventListener.Reading_View_Facing);
            } else if (id == R.id.iv_cover_page_icon) {
                if (mListeners.get(IViewSettingsWindow.TYPE_SINGLE_PAGE) != null) {
                    if (mPageMode != PDFViewCtrl.PAGELAYOUTMODE_COVER) {
                        mPageMode = PDFViewCtrl.PAGELAYOUTMODE_COVER;
                        mListeners.get(IViewSettingsWindow.TYPE_SINGLE_PAGE).onValueChanged(IViewSettingsWindow.TYPE_SINGLE_PAGE, mPageMode);
                    }
                }
                mIvSingle.setSelected(false);
                mIvFacing.setSelected(false);
                mIvCover.setSelected(true);
                ((UIExtensionsManager) mPdfView.getUIExtensionsManager()).onUIInteractElementClicked(IUIInteractionEventListener.Reading_View_CoverFacing);
            } else if (id == R.id.ll_continuous_page_button) {
                if (mListeners.get(IViewSettingsWindow.TYPE_SINGLE_PAGE) != null) {
                    if (mContinuousPageButtonLayout.isSelected()) {
                        setSelectedButtonState(false, mContinuousPageButtonLayout, mIvContinuousPage);
                        mPdfView.setContinuous(false, true);
                    } else {
                        setSelectedButtonState(true, mContinuousPageButtonLayout, mIvContinuousPage);
                        mPdfView.setContinuous(true, true);
                    }
                    mPageMode = mPdfView.getPageLayoutMode();
                    mListeners.get(IViewSettingsWindow.TYPE_SINGLE_PAGE).onValueChanged(IViewSettingsWindow.TYPE_SINGLE_PAGE, mPageMode);
                }
                ((UIExtensionsManager) mPdfView.getUIExtensionsManager()).onUIInteractElementClicked(IUIInteractionEventListener.Reading_View_ContinuousScrolling);
            } else if (id == R.id.rela_rtl_layout) {
                if (mListeners.get(IViewSettingsWindow.TYPE_RIGHT_TO_LEFT) != null) {
                    if (mRTL_SwitchLayout.isSelected()) {
                        setSelectedButtonState(false, mRTL_SwitchLayout, mRTL_SwitchIcon);
                        mPdfView.setPageBinding(PDFViewCtrl.LEFT_EDGE);
                    } else {
                        setSelectedButtonState(true, mRTL_SwitchLayout, mRTL_SwitchIcon);
                        mPdfView.setPageBinding(PDFViewCtrl.RIGHT_EDGE);
                    }
                    mListeners.get(IViewSettingsWindow.TYPE_RIGHT_TO_LEFT)
                            .onValueChanged(IViewSettingsWindow.TYPE_RIGHT_TO_LEFT, mPdfView.getPageBinding());
                }
                ((UIExtensionsManager) mPdfView.getUIExtensionsManager()).onUIInteractElementClicked(IUIInteractionEventListener.Reading_View_Right_to_Left);
            } else if (id == R.id.iv_day_icon) {
                if (mListeners.get(IViewSettingsWindow.TYPE_DAY) != null) {
                    if (mPageColorMode != IViewSettingsWindow.DAY) {
                        mPageColorMode = IViewSettingsWindow.DAY;
                        mListeners.get(IViewSettingsWindow.TYPE_DAY).onValueChanged(IViewSettingsWindow.TYPE_DAY, mPageColorMode);
                    }
                    mIvNight.setBorderColor(AppResource.getColor(mContext, R.color.b2));
                    mIvDay.setBorderColor(ThemeConfig.getInstance(mContext).getPrimaryColor());
                    mIvPageColor.setBorderColor(AppResource.getColor(mContext, R.color.b2));
                }
                ((UIExtensionsManager) mPdfView.getUIExtensionsManager()).onUIInteractElementClicked(IUIInteractionEventListener.Reading_View_Day);
            } else if (id == R.id.iv_page_color_icon) {
                if (mListeners.get(IViewSettingsWindow.TYPE_DAY) != null) {
                    if (mPageColorMode != IViewSettingsWindow.CUSTOM_COLOR) {
                        mPageColorMode = IViewSettingsWindow.CUSTOM_COLOR;
                        mListeners.get(IViewSettingsWindow.TYPE_DAY).onValueChanged(IViewSettingsWindow.TYPE_DAY, mPageColorMode);

                        int selectedColor = mColorView.getSelectedColor();
                        for (int i = 0; i < bgColors.length; i++) {
                            int color = bgColors[i];
                            if (color == selectedColor) {
                                mPdfView.setMappingModeForegroundColor(textColors[i]);
                                break;
                            }
                        }
                        mPdfView.setMappingModeBackgroundColor(selectedColor);
                        if (selectedColor == Color.WHITE)
                            mPdfView.setColorMode(Renderer.e_ColorModeNormal);
                        else
                            //mPdfView.setColorMode(Renderer.e_ColorModeMappingGray);
                            mPdfView.setColorMode(
                                    ((UIExtensionsManager) mPdfView.getUIExtensionsManager()).getPageColorMode() == UIExtensionsManager.NIGHTCOLORMODE_MAPPINGGRAY
                                            ? Renderer.e_ColorModeMappingGray : Renderer.e_ColorModeMapping);
                        mPdfView.setBackgroundColor(AppResource.getColor(mContext, R.color.ux_bg_color_docviewer));
                    }
                    mIvDay.setBorderColor(AppResource.getColor(mContext, R.color.b2));
                    mIvPageColor.setBorderColor(ThemeConfig.getInstance(mContext).getPrimaryColor());
                    mIvNight.setBorderColor(AppResource.getColor(mContext, R.color.b2));
                    mPageColorWindow.show();
                    dismiss();
                }
                ((UIExtensionsManager) mPdfView.getUIExtensionsManager()).onUIInteractElementClicked(IUIInteractionEventListener.Reading_View_PageColor);
            } else if (id == R.id.iv_night_icon) {
                if (mListeners.get(IViewSettingsWindow.TYPE_DAY) != null) {
                    if (mPageColorMode != IViewSettingsWindow.NIGHT) {
                        mPageColorMode = IViewSettingsWindow.NIGHT;
                        mListeners.get(IViewSettingsWindow.TYPE_DAY).onValueChanged(IViewSettingsWindow.TYPE_DAY, mPageColorMode);
                    }
                    mIvDay.setBorderColor(AppResource.getColor(mContext, R.color.b2));
                    mIvNight.setBorderColor(ThemeConfig.getInstance(mContext).getPrimaryColor());
                    mIvPageColor.setBorderColor(AppResource.getColor(mContext, R.color.b2));
                }
                ((UIExtensionsManager) mPdfView.getUIExtensionsManager()).onUIInteractElementClicked(IUIInteractionEventListener.Reading_View_Night);
            } else if (id == R.id.ll_fit_page) {
                if (mListeners.get(IViewSettingsWindow.TYPE_FIT_PAGE) != null && mZoomMode != PDFViewCtrl.ZOOMMODE_FITPAGE) {
                    mZoomMode = PDFViewCtrl.ZOOMMODE_FITPAGE;
                    mListeners.get(IViewSettingsWindow.TYPE_FIT_PAGE).onValueChanged(IViewSettingsWindow.TYPE_FIT_PAGE, mZoomMode);
                }
                mFitWidthLayout.setSelected(false);
                mIvFitWidthSelected.setVisibility(View.INVISIBLE);

                mFitPageLayout.setSelected(true);
                mIvFitPageSelected.setVisibility(View.VISIBLE);
                ((UIExtensionsManager) mPdfView.getUIExtensionsManager()).onUIInteractElementClicked(IUIInteractionEventListener.Reading_View_FitPage);
            } else if (id == R.id.ll_fit_width) {
                if (mListeners.get(IViewSettingsWindow.TYPE_FIT_WIDTH) != null && mZoomMode != PDFViewCtrl.ZOOMMODE_FITWIDTH) {
                    mZoomMode = PDFViewCtrl.ZOOMMODE_FITWIDTH;
                    mListeners.get(IViewSettingsWindow.TYPE_FIT_WIDTH).onValueChanged(IViewSettingsWindow.TYPE_FIT_WIDTH, mZoomMode);
                }

                mFitPageLayout.setSelected(false);
                mIvFitPageSelected.setVisibility(View.INVISIBLE);

                mFitWidthLayout.setSelected(true);
                mIvFitWidthSelected.setVisibility(View.VISIBLE);
                ((UIExtensionsManager) mPdfView.getUIExtensionsManager()).onUIInteractElementClicked(IUIInteractionEventListener.Reading_View_FitWidth);
            } else if (id == R.id.ll_reflow) {
                if (mListeners.get(IViewSettingsWindow.TYPE_REFLOW) != null) {
                    mListeners.get(IViewSettingsWindow.TYPE_REFLOW).onValueChanged(IViewSettingsWindow.TYPE_REFLOW, true);
                }
                ((UIExtensionsManager) mPdfView.getUIExtensionsManager()).onUIInteractElementClicked(IUIInteractionEventListener.Reading_View_Reflow);
            } else if (id == R.id.ll_crop) {
                if (mListeners.get(IViewSettingsWindow.TYPE_CROP) != null) {
                    mListeners.get(IViewSettingsWindow.TYPE_CROP).onValueChanged(IViewSettingsWindow.TYPE_CROP, true);
                }
                ((UIExtensionsManager) mPdfView.getUIExtensionsManager()).onUIInteractElementClicked(IUIInteractionEventListener.Reading_View_CropMode);
            } else if (id == R.id.ll_speak) {
                if (mListeners.get(IViewSettingsWindow.TYPE_TTS) != null) {
                    mListeners.get(IViewSettingsWindow.TYPE_TTS).onValueChanged(IViewSettingsWindow.TYPE_TTS, true);
                }
                ((UIExtensionsManager) mPdfView.getUIExtensionsManager()).onUIInteractElementClicked(IUIInteractionEventListener.Reading_View_Speak);
            } else if (id == R.id.ll_auto_flip) {
                if (mListeners.get(IViewSettingsWindow.TYPE_AUTO_FLIP) != null) {
                    mListeners.get(IViewSettingsWindow.TYPE_AUTO_FLIP).onValueChanged(IViewSettingsWindow.TYPE_AUTO_FLIP, true);
                }
                ((UIExtensionsManager) mPdfView.getUIExtensionsManager()).onUIInteractElementClicked(IUIInteractionEventListener.Reading_View_AutoFlip);
            } else if (id == R.id.ll_rotate_view) {
                if (mListeners.get(IViewSettingsWindow.TYPE_ROTATE_VIEW) != null) {
                    mListeners.get(IViewSettingsWindow.TYPE_ROTATE_VIEW).onValueChanged(IViewSettingsWindow.TYPE_ROTATE_VIEW, true);
                }
                ((UIExtensionsManager) mPdfView.getUIExtensionsManager()).onUIInteractElementClicked(IUIInteractionEventListener.Reading_View_RotateView);
            } else if (id == R.id.ll_pan_zoom) {
                if (mListeners.get(IViewSettingsWindow.TYPE_PAN_ZOOM) != null) {
                    mListeners.get(IViewSettingsWindow.TYPE_PAN_ZOOM).onValueChanged(IViewSettingsWindow.TYPE_PAN_ZOOM, true);
                }
                ((UIExtensionsManager) mPdfView.getUIExtensionsManager()).onUIInteractElementClicked(IUIInteractionEventListener.Reading_View_PanZoom);
            }

        }
    };

    @Override
    public void setProperty(int property, Object value) {
        if (mContentView == null) return;
        if (property == IViewSettingsWindow.TYPE_SINGLE_PAGE) {
            this.mPageMode = (Integer) value;
            setSelectedButtonState(mPdfView.isContinuous(), mContinuousPageButtonLayout, mIvContinuousPage);

            if (this.mPageMode == PDFViewCtrl.PAGELAYOUTMODE_SINGLE) {
                mIvSingle.setSelected(true);
                mIvFacing.setSelected(false);
                mIvCover.setSelected(false);
            } else if (this.mPageMode == PDFViewCtrl.PAGELAYOUTMODE_FACING) {
                mIvSingle.setSelected(false);
                mIvFacing.setSelected(true);
                mIvCover.setSelected(false);
            } else if (this.mPageMode == PDFViewCtrl.PAGELAYOUTMODE_COVER) {
                mIvSingle.setSelected(false);
                mIvFacing.setSelected(false);
                mIvCover.setSelected(true);
            }
        } else if (property == IViewSettingsWindow.TYPE_RIGHT_TO_LEFT) {
            setSelectedButtonState(mPdfView.getPageBinding() == PDFViewCtrl.RIGHT_EDGE, mRTL_SwitchLayout, mRTL_SwitchIcon);
        } else if (property == IViewSettingsWindow.TYPE_DAY) {
            this.mPageColorMode = (Integer) value;
            if (this.mPageColorMode == IViewSettingsWindow.DAY) {
                mIvNight.setBorderColor(AppResource.getColor(mContext, R.color.b2));
                mIvDay.setBorderColor(ThemeConfig.getInstance(mContext).getPrimaryColor());
                mIvPageColor.setBorderColor(AppResource.getColor(mContext, R.color.b2));
            } else if (this.mPageColorMode == IViewSettingsWindow.CUSTOM_COLOR) {
                mIvDay.setBorderColor(AppResource.getColor(mContext, R.color.b2));
                mIvPageColor.setBorderColor(ThemeConfig.getInstance(mContext).getPrimaryColor());
                mIvNight.setBorderColor(AppResource.getColor(mContext, R.color.b2));
            } else if (this.mPageColorMode == IViewSettingsWindow.NIGHT) {
                mIvDay.setBorderColor(AppResource.getColor(mContext, R.color.b2));
                mIvNight.setBorderColor(ThemeConfig.getInstance(mContext).getPrimaryColor());
                mIvPageColor.setBorderColor(AppResource.getColor(mContext, R.color.b2));
            }

        } else if (property == IViewSettingsWindow.TYPE_FIT_PAGE) {
            mFitWidthLayout.setSelected(false);
            mIvFitWidthSelected.setVisibility(View.INVISIBLE);
            this.mZoomMode = (int) value;
            if (mZoomMode == PDFViewCtrl.ZOOMMODE_FITPAGE) {
                mFitPageLayout.setSelected(true);
                mIvFitPageSelected.setVisibility(View.VISIBLE);
            } else {
                mFitPageLayout.setSelected(false);
                mIvFitPageSelected.setVisibility(View.INVISIBLE);
            }
        } else if (property == IViewSettingsWindow.TYPE_FIT_WIDTH) {
            mFitPageLayout.setSelected(false);
            mIvFitPageSelected.setVisibility(View.INVISIBLE);
            this.mZoomMode = (int) value;
            if (mZoomMode == PDFViewCtrl.ZOOMMODE_FITWIDTH) {
                mFitWidthLayout.setSelected(true);
                mIvFitWidthSelected.setVisibility(View.VISIBLE);
            } else {
                mFitWidthLayout.setSelected(false);
                mIvFitWidthSelected.setVisibility(View.INVISIBLE);
            }
        } else if (property == IViewSettingsWindow.TYPE_CROP) {
            mCropPageLayout.setSelected((Boolean) value);
            mIvCropPageSelected.setVisibility((Boolean) value ? View.VISIBLE : View.INVISIBLE);

            enableBar(IViewSettingsWindow.TYPE_PAN_ZOOM, !mPdfView.isDynamicXFA() && !(Boolean) value);
        } else if (property == IViewSettingsWindow.TYPE_TTS) {
            mSpeakLayout.setSelected((Boolean) value);
        }
    }

    public void pageColorDismiss() {
        if (mPageColorWindow == null || !mPageColorWindow.getWindow().isAdded()) return;
        mPageColorWindow.dismiss();
        mPageColorWindow = null;
    }

    @Override
    public boolean isShowing() {
        return mBottomViewSettingsDialog.isShowing();
    }

    @Override
    public void show() {
        mBottomViewSettingsDialog.show();
    }

    @Override
    public boolean getVisible(int type) {
        if (mContentView == null) {
            return mSetVisibleViews.get(type, true);
        }

        Integer integer = mIdsMap.get(type);
        if (integer == null)
            return false;

        View view = mContentView.findViewById(integer);
        if (view != null)
            return view.getVisibility() == View.VISIBLE;
        return false;
    }

    @Override
    public void setVisible(int type, boolean visible) {
        mSetVisibleViews.put(type, visible);
        if (mContentView == null) {
            return;
        }
        Integer id = mIdsMap.get(type);
        if (null == id) return;
        int visibility = visible ? View.VISIBLE : View.GONE;
        if (id > 0) {
            mContentView.findViewById(id).setVisibility(visibility);
        }
    }

    @Override
    public void dismiss() {
        mBottomViewSettingsDialog.dismiss();
        if (mOnDismissListener != null) {
            mOnDismissListener.onDismiss();
        }
    }

    @Override
    public View getContentView() {
        return mContentView;
    }

    @Override
    public void setToolItems(MainCenterItemBean item) {
        //todo
    }

    @Override
    public MainCenterItemBean getToolItems() {
        //todo
        return null;
    }

    @Override
    public void onThemeColorChanged() {
    }

    @Override
    public void setEnabled(boolean enabled) {
        //todo
    }

    @Override
    public boolean isEnabled() {
        //todo
        return false;
    }

    @Override
    public void registerListener(IViewSettingsWindow.IValueChangeListener listener) {
        int type = listener.getType();
        if (!mListeners.containsKey(type)) {
            this.mListeners.put(type, listener);
        }
    }

    @Override
    public void unRegisterListener(IViewSettingsWindow.IValueChangeListener listener) {
        if (mListeners.containsKey(listener.getType())) {
            this.mListeners.remove(listener.getType());
        }
    }

    @Override
    public IValueChangeListener getValueChangeListener(int type) {
        return mListeners.get(type);
    }

    @Override
    public void enableBar(int property, boolean enable) {
        if (mContentView == null) {
            mEnabledArray.append(property, enable);
            return;
        }
        switch (property) {
            case TYPE_FACING_PAGE:
                mTvFacing.setEnabled(enable);
                mIvFacing.setEnabled(enable);
                break;
            case TYPE_COVER_PAGE:
                mTvCover.setEnabled(enable);
                mIvCover.setEnabled(enable);
                break;
            case TYPE_CONTINUOUS_PAGE:
                mContinuousPageTv.setEnabled(enable);
                mContinuousPageIv.setEnabled(enable);
                mContinuousPageButtonLayout.setEnabled(enable);
                break;
            case TYPE_REFLOW:
                mReflowLayout.setEnabled(enable);
                for (int i = 0; i < mReflowLayout.getChildCount(); i++) {
                    mReflowLayout.getChildAt(i).setEnabled(enable);
                }
                break;
            case TYPE_CROP:
                mCropPageLayout.setEnabled(enable);
                for (int i = 0; i < mCropPageLayout.getChildCount(); i++) {
                    mCropPageLayout.getChildAt(i).setEnabled(enable);
                }
                break;
            case TYPE_PAN_ZOOM:
                mPanZoomLayout.setEnabled(enable);
                for (int i = 0; i < mPanZoomLayout.getChildCount(); i++) {
                    mPanZoomLayout.getChildAt(i).setEnabled(enable);
                }
                break;
            case TYPE_FIT_PAGE:
                mFitPageLayout.setEnabled(enable);
                break;
            case TYPE_FIT_WIDTH:
                mFitWidthLayout.setEnabled(enable);
                break;
            case TYPE_ROTATE_VIEW:
                mRotateViewLayout.setEnabled(enable);
                break;
            case TYPE_TTS:
                mSpeakLayout.setEnabled(enable);
                for (int i = 0; i < mSpeakLayout.getChildCount(); i++) {
                    mSpeakLayout.getChildAt(i).setEnabled(enable);
                }
                break;
            case TYPE_RIGHT_TO_LEFT:
                mRTLTitleIv.setEnabled(enable);
                mRTLTitleTv.setEnabled(enable);
                mRTL_SwitchLayout.setEnabled(enable);
                break;
            default:
                break;
        }
    }

    @Override
    public void setOnDismissListener(OnDismissListener listener) {
        mOnDismissListener = listener;
    }

    @Override
    public IViewSettingsWindow updateTheme() {
        pageColorDismiss();
        if (isShowing()) {
            dismiss();
        }

        ViewSettingsWindow settingsWindow = new ViewSettingsWindow(mContext, mPdfView);
        settingsWindow.mPageMode = mPageMode;
        settingsWindow.mPageColorMode = mPageColorMode;
        settingsWindow.mZoomMode = mZoomMode;
        if (mCustomPageColor != null) {
            settingsWindow.mCustomPageColor = mCustomPageColor;
        }
        for (int i = 0; i < mSetVisibleViews.size(); i++) {
            int type = mSetVisibleViews.keyAt(i);
            boolean visible = mSetVisibleViews.valueAt(i);
            settingsWindow.mSetVisibleViews.put(type, visible);
        }

        settingsWindow.enableBar(TYPE_FACING_PAGE, mIvFacing.isEnabled());
        settingsWindow.enableBar(TYPE_COVER_PAGE, mIvCover.isEnabled());
        settingsWindow.enableBar(TYPE_CONTINUOUS_PAGE, mContinuousPageIv.isEnabled());
        settingsWindow.enableBar(TYPE_REFLOW, mReflowLayout.isEnabled());
        settingsWindow.enableBar(TYPE_CROP, mCropPageLayout.isEnabled());
        settingsWindow.enableBar(TYPE_PAN_ZOOM, mPanZoomLayout.isEnabled());
        settingsWindow.enableBar(TYPE_FIT_PAGE, mFitPageLayout.isEnabled());
        settingsWindow.enableBar(TYPE_FIT_WIDTH, mFitWidthLayout.isEnabled());
        settingsWindow.enableBar(TYPE_ROTATE_VIEW, mRotateViewLayout.isEnabled());
        settingsWindow.enableBar(TYPE_TTS, mSpeakLayout.isEnabled());

        for (IViewSettingsWindow.IValueChangeListener valueChangeListener : mListeners.values()) {
            settingsWindow.registerListener(valueChangeListener);
        }
        return settingsWindow;
    }

    @Override
    public int getColorMode() {
        return mPageColorMode;
    }

    @Override
    public boolean isSelected(int property) {
        Integer integer = mIdsMap.get(property);
        if (integer == null)
            return false;

        View view = mContentView.findViewById(integer);
        if (view != null)
            return view.isSelected();
        return false;
    }

    private ColorStateList getBorderColorStateList() {
        int disabled = AppResource.getColor(mContext, R.color.b2);
        int selected = ThemeConfig.getInstance(mContext).getPrimaryColor();
        int normal = AppResource.getColor(mContext, R.color.b2);
        return AppResource.createColorStateList(selected, disabled, normal);
    }

    private ColorStateList getSelectedButtonColorStateList() {
        int disabled = AppResource.getColor(mContext, R.color.p1);
        int selected = ThemeConfig.getInstance(mContext).getPrimaryColor();
        int normal = AppResource.getColor(mContext, R.color.p1);
        return AppResource.createColorStateList(selected, disabled, normal);
    }


    private static final int[] bgColors = new int[]{0xFFFFFFFF, 0xFFE8E8E8,
            0xFFF0E8D8, 0xFFCCC5B2,
            0xFF99CFA1, 0xFF2E5073,
            0xFF644F47, 0xFF9A9A9A,
    };

    private static final int[] textColors = {0xFF323232, 0xFF2B2B2B,
            0xFF323232, 0xFF5F3811,
            0xFF323232, 0xFF6D7E88,
            0xFF95938F, 0xFF000000};

    private void initializePageColor(Context context) {
        if (!mInitPageColor) {
            mInitPageColor = true;

            if (mPdfView.getColorMode() == Renderer.e_ColorModeMappingGray
                    || mPdfView.getColorMode() == Renderer.e_ColorModeMapping) {
                if (isNightMode()) {
                    mPageColorMode = IViewSettingsWindow.NIGHT;
                } else {
                    mPageColorMode = IViewSettingsWindow.CUSTOM_COLOR;
                    mCustomPageColor = mPdfView.getMappingModeBackgroundColor();
                }
            }
        }
        int selectColor = 0xFFFFFFFF;
        if (mCustomPageColor != null)
            selectColor = mCustomPageColor;
        if (mIvPageColor != null)
            mIvPageColor.setFillColorFilter(selectColor);
        mColorView = new SimpleColorView((Activity) context, selectColor, bgColors);
        mPageColorWindow = new SimpleBottomWindow(context, mColorView.getRootView());
        mColorView.setPropertyChangeListener(mPageColorListener);
    }

    private PropertyBar.PropertyChangeListener mPageColorListener = new PropertyBar.PropertyChangeListener() {
        @Override
        public void onValueChanged(long property, int value) {
            if (property == PropertyBar.PROPERTY_COLOR && mIvPageColor != null && mPageColorMode == CUSTOM_COLOR) {
                setPageBackgroundColor(value);
            }
        }

        @Override
        public void onValueChanged(long property, float value) {

        }

        @Override
        public void onValueChanged(long property, String value) {

        }
    };

    public void setPageBackgroundColor(int value) {
        mIvPageColor.setFillColorFilter(value);

        mCustomPageColor = value;
        for (int i = 0; i < bgColors.length; i++) {
            int color = bgColors[i];
            if (color == value) {
                mPdfView.setMappingModeForegroundColor(textColors[i]);
                break;
            }
        }
        mPdfView.setMappingModeBackgroundColor(value);
        mPdfView.setBackgroundColor(AppResource.getColor(mContext, R.color.ux_bg_color_docviewer));
        if (mCustomPageColor == Color.WHITE)
            mPdfView.setColorMode(Renderer.e_ColorModeNormal);
        else
            mPdfView.setColorMode(
                    ((UIExtensionsManager) mPdfView.getUIExtensionsManager()).getPageColorMode() == UIExtensionsManager.NIGHTCOLORMODE_MAPPINGGRAY
                            ? Renderer.e_ColorModeMappingGray : Renderer.e_ColorModeMapping);
    }

    public SimpleBottomWindow getPageColorWindow() {
        return mPageColorWindow;
    }

    private void setSelectedButtonState(boolean isSelected, RelativeLayout switchLayout, ImageView switchIcon) {
        if (switchLayout == null || switchIcon == null) return;
        switchLayout.setSelected(isSelected);
        RelativeLayout.LayoutParams params = (RelativeLayout.LayoutParams) switchIcon.getLayoutParams();
        if (params == null) return;
        params.removeRule(isSelected ? RelativeLayout.ALIGN_PARENT_LEFT : RelativeLayout.ALIGN_PARENT_RIGHT);
        params.addRule(isSelected ? RelativeLayout.ALIGN_PARENT_RIGHT : RelativeLayout.ALIGN_PARENT_LEFT, RelativeLayout.TRUE);
        switchIcon.setLayoutParams(params);
    }

    private boolean isNightMode() {
        if (mPdfView.getColorMode() == Renderer.e_ColorModeMappingGray || mPdfView.getColorMode() == Renderer.e_ColorModeMapping) {
            if (mPdfView.getMappingModeBackgroundColor() == IViewSettingsWindow.DEFAULT_NIGHT_BG_COLOR
                    && mPdfView.getMappingModeForegroundColor() == IViewSettingsWindow.DEFAULT_NIGHT_FG_COLOR) {
                return true;
            }
        }
        return false;
    }

    public void release() {
        mActivityContext = null;
    }
}
