/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.controls.toolbar.drag;

import android.animation.ArgbEvaluator;
import android.animation.ValueAnimator;
import android.content.Context;
import android.graphics.Color;
import android.graphics.Point;
import android.graphics.Rect;
import android.graphics.drawable.Drawable;
import android.graphics.drawable.GradientDrawable;
import android.os.Build;
import android.util.AttributeSet;
import android.view.MotionEvent;
import android.view.View;
import android.view.ViewGroup;
import android.widget.ImageView;
import android.widget.RelativeLayout;

import com.foxit.sdk.PDFViewCtrl;
import com.foxit.uiextensions.R;
import com.foxit.uiextensions.UIExtensionsManager;
import com.foxit.uiextensions.annots.common.UIBtnImageView;
import com.foxit.uiextensions.config.uisettings.UISettingsConfig;
import com.foxit.uiextensions.controls.toolbar.BaseBar;
import com.foxit.uiextensions.controls.toolbar.IBaseItem;
import com.foxit.uiextensions.controls.toolbar.ToolConstants;
import com.foxit.uiextensions.controls.toolbar.impl.BaseItemImpl;
import com.foxit.uiextensions.theme.ColorPair;
import com.foxit.uiextensions.theme.DynamicColorProvider;
import com.foxit.uiextensions.theme.ThemeConfig;
import com.foxit.uiextensions.utils.AppDevice;
import com.foxit.uiextensions.utils.AppDisplay;
import com.foxit.uiextensions.utils.AppResource;
import com.foxit.uiextensions.utils.AppUtil;

import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.util.ArrayList;

import androidx.annotation.ColorInt;
import androidx.annotation.IntDef;
import androidx.annotation.NonNull;
import androidx.annotation.RequiresApi;

/**
 * The drag tool bar.
 */
public class UIDragToolBar /*extends RelativeLayout*/ {

    public interface IToolbarEventListener {
        void onToolbarDragEventBegin();

        void onToolbarDragEventEnd();

        void onToolbarPositionChanged(@ToolbarPos int oldPos, @ToolbarPos int newPos);
    }

    public interface IToolbarLayoutListener {
        void onToolbarLayout(@ToolbarPos int curPos);
    }

    private ArrayList<IToolbarLayoutListener> mToolbarLayoutListeners;
    private ArrayList<IToolbarEventListener> mToolbarEventListeners;

    /**
     * The toolbar position: top
     */
    public static final int TOOLBAR_POS_TOP = 0;

//    /**
//     * The toolbar position: bottom
//     */
//    public static final int TOOLBAR_POS_BOTTOM = 1; // unsupported

    /**
     * The toolbar position: left
     */
    public static final int TOOLBAR_POS_LEFT = 1;

    /**
     * The toolbar position: right
     */
    public static final int TOOLBAR_POS_RIGHT = 2;

    /**
     * The toolbar position: any position. Set this, you can drag the toolbar to where you want.
     */
    public static final int TOOLBAR_POS_ANY = 3;

    @IntDef({TOOLBAR_POS_TOP, /*TOOLBAR_POS_BOTTOM,*/ TOOLBAR_POS_LEFT, TOOLBAR_POS_RIGHT, TOOLBAR_POS_ANY})
    @Retention(RetentionPolicy.SOURCE)
    public @interface ToolbarPos {
    }

    UIToolBaseBar mToolBar;
    UIExtensionsManager mUiExtensionsManager;
    Context mContext;

    UIDragToolBarRL mRootRl; // Optimize screen rotation speed
    RelativeLayout mTopRl;
    RelativeLayout mBottomRl;
    RelativeLayout mLeftRl;
    RelativeLayout mRightRl;
    RelativeLayout m1ContainerRl;
    RelativeLayout mContentRl;
    View mTopSolidView;
    //    View mBottomSolidView;
//    ViewGroup mScrollViewH;
//    ViewGroup mScrollViewV;
    @ToolbarPos
    int mBarPosition = TOOLBAR_POS_TOP;
    ImageView mDragView;
    BaseItemImpl mDragItem;
    boolean mDragItemVisible = false;
    static final int SHADOW_LENGTH = 10;
    int mShadowLength;
    boolean mIsFirstShow = true;
    boolean mAutoResizeHeight = false;

    private int mLastScreenOrientation;
    private int mCustomHorizontalHeight;
    private int mCustomVerticalWidth;

    public UIDragToolBar(Context context, PDFViewCtrl.UIExtensionsManager uiExtensionsManager) {
        this(context, null, uiExtensionsManager);
    }

    public UIDragToolBar(Context context, AttributeSet attrs, PDFViewCtrl.UIExtensionsManager uiExtensionsManager) {
        this(context, attrs, 0, uiExtensionsManager);
    }

    public UIDragToolBar(Context context, AttributeSet attrs, int defStyleAttr, PDFViewCtrl.UIExtensionsManager uiExtensionsManager) {
//        super(context, attrs, defStyleAttr); // Optimize screen rotation speed
        initialize(context, uiExtensionsManager);
    }

    @RequiresApi(api = Build.VERSION_CODES.LOLLIPOP)
    public UIDragToolBar(Context context, AttributeSet attrs, int defStyleAttr, int defStyleRes, PDFViewCtrl.UIExtensionsManager uiExtensionsManager) {
//        super(context, attrs, defStyleAttr, defStyleRes); // Optimize screen rotation speed
        initialize(context, uiExtensionsManager);
    }

    public View getRootView() {
        // Optimize screen rotation speed
        return mRootRl;
    }

    private void initialize(Context context, PDFViewCtrl.UIExtensionsManager uiExtensionsManager) {
        mContext = context;
        mLastScreenOrientation = context.getResources().getConfiguration().orientation;
        mUiExtensionsManager = (UIExtensionsManager) uiExtensionsManager;
        mToolbarEventListeners = new ArrayList<>();
        mToolbarLayoutListeners = new ArrayList<>();

        mRootRl = (UIDragToolBarRL) View.inflate(context, R.layout.ui_tool_bar_layout, null);
        mRootRl.setDragToolbar(this);  // Optimize screen rotation speed

        mTopRl = mRootRl.findViewById(R.id.rl_top_bar);
        mBottomRl = mRootRl.findViewById(R.id.rl_top_bar);
        mLeftRl = mRootRl.findViewById(R.id.rl_left_bar);
        mRightRl = mRootRl.findViewById(R.id.rl_right_bar);
        m1ContainerRl = mRootRl.findViewById(R.id.rl_tool_bar_container);
        m1ContainerRl.addOnLayoutChangeListener(new View.OnLayoutChangeListener() {

            @Override
            public void onLayoutChange(View v, int left, int top, int right, int bottom, int oldLeft, int oldTop, int oldRight, int oldBottom) {
                if (!mIsDragging && mLayoutRect != null) {
                    if (mDraggableBoundaryRect != null) {
                        int orientation = mContext.getResources().getConfiguration().orientation;
                        if (mBarPosition == TOOLBAR_POS_ANY && mLastScreenOrientation != orientation) {
                            float scaleLeft = (float) Math.max(0, mLayoutRect.left - mDraggableBoundaryRect.left) / mDraggableBoundaryRect.width();
                            float scaleRight = (float) Math.max(0, mDraggableBoundaryRect.right - mLayoutRect.right) / mDraggableBoundaryRect.width();
                            float scaleTop = (float) Math.max(mDraggableBoundaryInsetRect.top, mLayoutRect.top - mDraggableBoundaryInsetRect.top)
                                    / (mDraggableBoundaryRect.bottom + mDraggableBoundaryInsetRect.bottom - mLayoutRect.bottom);

                            mBoundaryView.getGlobalVisibleRect(mDraggableBoundaryRect);
                            mDraggableBoundaryRect.set(mDraggableBoundaryRect.left + mDraggableBoundaryInsetRect.left,
                                    mDraggableBoundaryRect.top + mDraggableBoundaryInsetRect.top,
                                    mDraggableBoundaryRect.right - mDraggableBoundaryInsetRect.right,
                                    mDraggableBoundaryRect.bottom - mDraggableBoundaryInsetRect.bottom);

                            if (scaleLeft == 0) {
                                mLayoutRect.left = Math.max(mDraggableBoundaryRect.left, Math.min(mLayoutRect.left, mDraggableBoundaryRect.right - m1ContainerRl.getWidth()));
                                mLayoutRect.right = mLayoutRect.left + m1ContainerRl.getWidth();
                            } else if (scaleLeft <= scaleRight) {
                                mLayoutRect.left = (int) (scaleLeft * mDraggableBoundaryRect.width() + mDraggableBoundaryRect.left);
                                mLayoutRect.right = mLayoutRect.left + m1ContainerRl.getWidth();
                            } else {
                                mLayoutRect.right = (int) (mDraggableBoundaryRect.width() - scaleRight * mDraggableBoundaryRect.width() + mDraggableBoundaryRect.left);
                                mLayoutRect.left = mLayoutRect.right - mContentRl.getWidth();
                            }

                            mLayoutRect.top = (int) ((mDraggableBoundaryRect.height() - m1ContainerRl.getHeight()) * scaleTop / (1 + scaleTop));
                            mLayoutRect.bottom = mLayoutRect.top + m1ContainerRl.getHeight();
                            mLastScreenOrientation = orientation;
                        } else {
                            mLayoutRect.left = Math.max(mDraggableBoundaryRect.left, Math.min(mLayoutRect.left, mDraggableBoundaryRect.right - m1ContainerRl.getWidth()));
                            mLayoutRect.top = Math.max(mDraggableBoundaryRect.top, Math.min(mLayoutRect.top, mDraggableBoundaryRect.bottom - m1ContainerRl.getHeight()));

                            mLayoutRect.right = mLayoutRect.left + m1ContainerRl.getWidth();
                            mLayoutRect.bottom = mLayoutRect.top + m1ContainerRl.getHeight();
                        }
                    }
                    m1ContainerRl.layout(mLayoutRect.left, mLayoutRect.top, mLayoutRect.right, mLayoutRect.bottom);
                    if (mToolBar instanceof UIHSToolBar) {
                        ((UIHSToolBar) mToolBar).updatePropertyBarPosition();
                    }
                }
            }
        });
        mContentRl = mRootRl.findViewById(R.id.rl_tool_bar_content_container);
        mTopSolidView = mRootRl.findViewById(R.id.view_tool_bar_top_solid);
//        mBottomSolidView = mRootRl.findViewById(R.id.view_tool_bar_bottom_solid);
//        mScrollViewH = mRootRl.findViewById(R.id.scrollview_tool_bar_content_h);
//        mScrollViewV = mRootRl.findViewById(R.id.scrollview_tool_bar_content_v);

        updateToolBarDragBackground();
        mTopSolidView.setBackgroundColor(ThemeConfig.getInstance(context).getPrimaryColor());
//        mBottomSolidView.setBackgroundResource(ThemeConfig.getInstance(context).getPrimaryColor());
//        addView(mRootRl); // Optimize screen rotation speed

        UISettingsConfig settingsConfig = ((UIExtensionsManager) uiExtensionsManager).getConfig().uiSettings;
        int toolBarDragType = settingsConfig.toolBarDragType;
        if (toolBarDragType == 3) {
            mDragItemVisible = true;
        } else if (AppDisplay.isPad()) {
            mDragItemVisible = toolBarDragType == 2;
        } else {
            mDragItemVisible = toolBarDragType == 1;
        }
    }

    boolean mIsDragging;

    BaseItemImpl createDragButtonItem(final Context context) {
        mDragView = new UIBtnImageView(context) {
            boolean mIsLayouting;
            Point mLastPt = new Point();

            @Override
            public boolean onTouchEvent(MotionEvent event) {
                if (mIsLayouting) {
                    return true;
                }

                switch (event.getAction()) {
                    case MotionEvent.ACTION_DOWN:
                        mLastPt.set((int) event.getRawX(), (int) event.getRawY());
                        setChecked(true);
                        triggerDragEvent();
                        break;
                    case MotionEvent.ACTION_MOVE: {
                        int x = (int) event.getRawX();
                        int y = (int) event.getRawY();
                        int dx = x - mLastPt.x;
                        int dy = y - mLastPt.y;
                        mLastPt.set(x, y);

                        if (mIsDragging) {
                            if (dx != 0 || dy != 0) {
                                int l = m1ContainerRl.getLeft();
                                int t = m1ContainerRl.getTop();
                                l += dx;
                                t += dy;
                                if (mDraggableBoundaryRect != null) {
                                    l = Math.max(mDraggableBoundaryRect.left, Math.min(l, mDraggableBoundaryRect.right - m1ContainerRl.getWidth()));
                                    t = Math.max(mDraggableBoundaryRect.top, Math.min(t, mDraggableBoundaryRect.bottom - m1ContainerRl.getHeight()));
                                }
                                m1ContainerRl.layout(l, t,
                                        l + m1ContainerRl.getWidth(), t + m1ContainerRl.getHeight());
                                if (mBarPosition == TOOLBAR_POS_ANY) {
                                    if (mLayoutRect == null) {
                                        setLayoutRect(new Rect());
                                    }
                                    mLayoutRect.set(m1ContainerRl.getLeft(), m1ContainerRl.getTop(), m1ContainerRl.getRight(), m1ContainerRl.getBottom());
                                }
                            }
                        }
                        break;
                    }
                    case MotionEvent.ACTION_UP:
                    case MotionEvent.ACTION_CANCEL:
                        if (mIsDragging) {
                            endDragEvent();
                            mIsDragging = false;
                        }
                        setChecked(false);
                        break;
                }

                return true;
            }

            void triggerDragEvent() {
                onToolbarDragEventBegin();
                mIsDragging = true;
                updateToolBarDragBackground();

                int l = m1ContainerRl.getLeft();
                int t = m1ContainerRl.getTop();
                int dd = 5;
                int dx = 0;
                int dy = 0;
                switch (mBarPosition) {
                    case TOOLBAR_POS_TOP:
                        dy = AppDisplay.dp2px(dd);
                        break;
            /*        case TOOLBAR_POS_BOTTOM:
                        dy = -AppDisplay.dp2px(dd);
                        break;*/
                    case TOOLBAR_POS_LEFT:
                        dx = AppDisplay.dp2px(dd);
                        break;
                    case TOOLBAR_POS_RIGHT:
                        dx = -AppDisplay.dp2px(dd);
                        break;
                    case UIDragToolBar.TOOLBAR_POS_ANY:
                        break;
                }
                m1ContainerRl.layout(l + dx, t + dy,
                        l + dx + m1ContainerRl.getWidth(), t + dy + m1ContainerRl.getHeight());

                if (mShowDraggableTargetZone) {
                    mLeftRl.setVisibility(View.VISIBLE);
                    mTopRl.setVisibility(View.VISIBLE);
                    mRightRl.setVisibility(View.VISIBLE);
//                  mBottomRl.setVisibility(View.VISIBLE);
                }
//                startAnim(AppResource.getColor(context, ThemeConfig.getInstance(context).getPrimaryColor()),
//                        AppResource.getColor(context, R.color.ux_color_list_item_pressed));
            }

            void endDragEvent() {
                Rect topRt = new Rect(mTopRl.getLeft(), mTopRl.getTop(), mTopRl.getRight(), mTopRl.getBottom());
                Rect bottomRt = new Rect(mBottomRl.getLeft(), mBottomRl.getTop(), mBottomRl.getRight(), mBottomRl.getBottom());
                Rect leftRt = new Rect(mLeftRl.getLeft(), mLeftRl.getTop(), mLeftRl.getRight(), mLeftRl.getBottom());
                Rect rightRt = new Rect(mRightRl.getLeft(), mRightRl.getTop(), mRightRl.getRight(), mRightRl.getBottom());
                Rect barRt = new Rect(m1ContainerRl.getLeft(), m1ContainerRl.getTop(), m1ContainerRl.getRight(), m1ContainerRl.getBottom());

                if (mBarPosition != TOOLBAR_POS_ANY) {
                    mTopRl.getGlobalVisibleRect(topRt);
                    mBottomRl.getGlobalVisibleRect(bottomRt);
                    mLeftRl.getGlobalVisibleRect(leftRt);
                    mRightRl.getGlobalVisibleRect(rightRt);
                    m1ContainerRl.getGlobalVisibleRect(barRt);
                }

                // dragger affect the position
                int oldPos = mBarPosition;
                if (barRt.intersect(rightRt)) {
                    mBarPosition = TOOLBAR_POS_RIGHT;
                } else if (barRt.intersect(leftRt)) {
                    mBarPosition = TOOLBAR_POS_LEFT;
//                } else if (barRt.intersect(bottomRt)) {
//                    mBarPosition = TOOLBAR_POS_BOTTOM;
                } else if (barRt.intersect(topRt)) {
                    mBarPosition = TOOLBAR_POS_TOP;
                }
//                setBarPositionToSp(mBarPosition);

                if (mBarPosition != oldPos) {
                    onToolbarPositionChanged(oldPos, mBarPosition);
                }

                mIsLayouting = true;
                layoutToolbar();
                mIsLayouting = false;

                endAnim();
                onToolbarDragEventEnd();
            }


            ArgbEvaluator argbEvaluator = new ArgbEvaluator();
            ValueAnimator valueAnimator;

            void startAnim(final int startColor, final int endColor) {
                valueAnimator = new ValueAnimator();
                valueAnimator.setIntValues(startColor, endColor);
                valueAnimator.setEvaluator(argbEvaluator);

                valueAnimator.setDuration(300);
                valueAnimator.addUpdateListener(new ValueAnimator.AnimatorUpdateListener() {
                    @Override
                    public void onAnimationUpdate(ValueAnimator animation) {
                        int frameValue = (Integer) animation.getAnimatedValue();
                        mLeftRl.setBackgroundColor(frameValue);
                        mTopRl.setBackgroundColor(frameValue);
                        mRightRl.setBackgroundColor(frameValue);
//                        mBottomRl.setBackgroundColor(frameValue);

                        if (frameValue == endColor && mIsDragging) {
                            startAnim(endColor, startColor);
                        }
                    }
                });
                valueAnimator.start();
            }

            void endAnim() {
                if (valueAnimator != null) {
                    valueAnimator.end();
                    valueAnimator = null;
                }
            }
        };
        mDragView.setImageResource(R.drawable.tool_bar_drager_vert);
        mDragItem = new BaseItemImpl(context, mDragView);
        mDragItem.setTag(ToolConstants.Drag);
        mDragItem.setId(R.id.id_comment_Drag);
        mDragItem.getContentView().setVisibility(mDragItemVisible ? View.VISIBLE : View.GONE);
        return mDragItem;
    }

    public void layoutToolbar() {
        onToolbarLayout(mBarPosition);
//        for (int i = 0; i < mBtnItems.size(); i ++) {
//            if (mDropDownTagSet.contains(mBtnItems.get(i).getContentView().getTag())) {
//                if (mBarPosition == 3) {
//                    mBtnItems.get(i).setCoverImageResource(R.drawable._70000_annot_bar_sanjiao_left);
//                } else {
//                    mBtnItems.get(i).setCoverImageResource(R.drawable._70000_annot_bar_sanjiao_right);
//                }
//            }
//        }
        if (mToolBar == null) return;
        if (mDragView != null) {
            if (mBarPosition == TOOLBAR_POS_TOP/* || mBarPosition == TOOLBAR_POS_BOTTOM*/) {
//            mDividerWidth = App.instance().getDisplay().dp2px(0.5f);
//            mDividerHeight = App.instance().getDisplay().dp2px(30);
                mDragView.setImageResource(R.drawable.tool_bar_drager_vert);
            } else {
//            mDividerWidth = App.instance().getDisplay().dp2px(30);
//            mDividerHeight = App.instance().getDisplay().dp2px(0.5f);
                mDragView.setImageResource(R.drawable.tool_bar_drager_horz);
            }
        }

//        mDividerView.setMinimumWidth((int)(mDividerWidth + 2));
//        mDividerView.setMinimumHeight((int)(mDividerHeight + 2));
//        mDividerView.invalidate();

        mTopSolidView.setVisibility(/*mBarPosition == TOOLBAR_POS_BOTTOM ? View.VISIBLE :*/ View.GONE);
//        mBottomSolidView.setVisibility(mBarPosition == 0 ? View.VISIBLE : View.GONE);

        if (mBarPosition == TOOLBAR_POS_TOP) {
            mToolBar.setOrientation(BaseBar.HORIZONTAL, ViewGroup.LayoutParams.MATCH_PARENT,
                    Math.max(mCustomHorizontalHeight, AppResource.getDimensionPixelSize(mContext, R.dimen.ui_bottombar_height)));
        }/* else if (mBarPosition == TOOLBAR_POS_BOTTOM) {
            mToolBar.setOrientation(BaseBar.HORIZONTAL, ViewGroup.LayoutParams.MATCH_PARENT,
                    Math.max(mCustomHorizontalHeight, AppResource.getDimensionPixelSize(mContext, R.dimen.ui_bottombar_height)));
        }*/ else {
            mToolBar.setOrientation(BaseBar.VERTICAL,
                    Math.max(mCustomVerticalWidth, AppResource.getDimensionPixelSize(mContext, R.dimen.ui_bottombar_height)), ViewGroup.LayoutParams.WRAP_CONTENT);
        }

        Point size = mToolBar.measureSize();
        RelativeLayout.LayoutParams containerLp = new RelativeLayout.LayoutParams(0, 0);

        int assumeHeight = 0, lrHeight = 0;
        int screenHeight = 0, screenWidth = 0;
        if (mUiExtensionsManager.getMainFrame() != null && mUiExtensionsManager.getMainFrame().getContentView().isShown()) { // Chrome OS
            screenHeight = mUiExtensionsManager.getMainFrame().getContentView().getHeight();
            screenWidth = mUiExtensionsManager.getMainFrame().getContentView().getWidth();
        }
        if (screenHeight == 0) {
            screenHeight = AppDisplay.getActivityHeight();
        }
        if (screenWidth == 0) {
            screenWidth = AppDisplay.getActivityWidth();
        }
        int topbarHeight = AppResource.getDimensionPixelSize(mContext, R.dimen.ui_bottombar_height);

        RelativeLayout leftRlParent = mLeftRl;
        RelativeLayout rightRlParent = mRightRl;

        switch (mBarPosition) {
            // 0, 1, 2, 3: top, bottom, left, right
            case TOOLBAR_POS_TOP:
            /*case TOOLBAR_POS_BOTTOM:*/ {
                mShadowLength = 0;
                mContentRl.setBackgroundResource(0);
                mContentRl.setPadding(0, 0, 0, 0);
                m1ContainerRl.setPadding(0, 0, 0, 0);

                containerLp.width = ViewGroup.LayoutParams.MATCH_PARENT;
                containerLp.height = size.y;// + AppResource.getDimensionPixelSize(getContext(), R.dimen.ux_toolbar_solidLine_height);
                containerLp.setMargins(0, 0, 0, 0);
                containerLp.addRule(RelativeLayout.CENTER_HORIZONTAL);
                if (mBarPosition == TOOLBAR_POS_TOP) {
                    containerLp.addRule(RelativeLayout.ALIGN_PARENT_TOP);
                    mTopSolidView.setVisibility(View.GONE);
//                    mBottomSolidView.setVisibility(View.VISIBLE);
                } else {
                    containerLp.addRule(RelativeLayout.ALIGN_PARENT_BOTTOM);
                    mTopSolidView.setVisibility(View.VISIBLE);
//                    mBottomSolidView.setVisibility(View.GONE);
                }
                mRealHeight = containerLp.height;
                m1ContainerRl.setLayoutParams(containerLp);

                if (AppDisplay.isPad()) {
                    assumeHeight = size.x + AppResource.getDimensionPixelSize(mContext, R.dimen.ux_view_setting_item_width) / 2 +
                            AppDisplay.dp2px(SHADOW_LENGTH) * 2;
                    lrHeight = size.x + AppResource.getDimensionPixelSize(mContext, R.dimen.ux_view_setting_item_width) / 2;
                } else {
                    lrHeight = screenHeight
                            - topbarHeight * 2 // topbar and bottombar height
                            - AppResource.getDimensionPixelSize(mContext, R.dimen.ux_margin_58dp) * 2 // margin
                            - mShadowLength * 2; // shadow
                    assumeHeight = lrHeight + mShadowLength * 2;
                }
                break;
            }
            case TOOLBAR_POS_ANY:
                mShadowLength = AppDisplay.dp2px(SHADOW_LENGTH);
                if (mAutoResizeHeight) {
                    ViewGroup toolBar = (ViewGroup) mToolBar.getContentView();
                    toolBar.measure(View.MeasureSpec.UNSPECIFIED, View.MeasureSpec.UNSPECIFIED);
                    int childCount = toolBar.getChildCount();
                    int realHeight = 0;
                    for (int i = 0; i < childCount; i++) {
                        realHeight += toolBar.getChildAt(i).getMeasuredHeight();
                    }

                    size.y = realHeight - AppResource.getDimensionPixelSize(mContext, R.dimen.ux_view_setting_item_width) / 2;
                    if (AppDisplay.isPad()) {
                        size.y = Math.min(size.y, AppResource.getDimensionPixelSize(mContext, R.dimen.ux_margin_492dp));
                        RelativeLayout.LayoutParams vsParams = (RelativeLayout.LayoutParams) toolBar.getChildAt(0).getLayoutParams();
                        vsParams.addRule(RelativeLayout.ABOVE, R.id.tool_right_container);
                        vsParams.removeRule(RelativeLayout.CENTER_VERTICAL);
                    } else {
                        if (screenHeight > screenWidth) {
                            size.y = Math.min(size.y, AppResource.getDimensionPixelSize(mContext, R.dimen.ux_margin_407dp));
                        } else {
                            size.y = Math.min(size.y, AppResource.getDimensionPixelSize(mContext, R.dimen.ux_margin_300dp));
                        }
                    }
                }
                mContentRl.setBackgroundResource(R.drawable.tool_bar_bg);
                mContentRl.setPadding(mShadowLength, mShadowLength, mShadowLength, mShadowLength);

                assumeHeight = size.y + AppResource.getDimensionPixelSize(mContext, R.dimen.ux_view_setting_item_width) / 2 + mShadowLength * 2;
                lrHeight = size.y + AppResource.getDimensionPixelSize(mContext, R.dimen.ux_view_setting_item_width) / 2;

                containerLp.width = size.x + mShadowLength * 2;
                containerLp.height = assumeHeight;

                if (mBarPosition == TOOLBAR_POS_LEFT) {
                    containerLp.addRule(RelativeLayout.ALIGN_PARENT_LEFT);
                    containerLp.setMargins(AppResource.getDimensionPixelSize(mContext, R.dimen.ux_margin_16dp) - mShadowLength, 0, 0, 0);
                } else {
                    containerLp.addRule(RelativeLayout.ALIGN_PARENT_RIGHT);
                    containerLp.setMargins(0, 0, AppResource.getDimensionPixelSize(mContext, R.dimen.ux_margin_16dp) - mShadowLength, 0);
                }

                mRealHeight = containerLp.height;

                m1ContainerRl.setLayoutParams(containerLp);

                if ((screenHeight - assumeHeight) / 2 < topbarHeight) {
                    containerLp.addRule(RelativeLayout.ALIGN_PARENT_TOP);
                    containerLp.topMargin = topbarHeight;
                } else {
                    containerLp.addRule(RelativeLayout.CENTER_VERTICAL);
                }

                break;
            case TOOLBAR_POS_LEFT:
            case TOOLBAR_POS_RIGHT: {
                mShadowLength = AppDisplay.dp2px(SHADOW_LENGTH);
                mContentRl.setBackgroundResource(R.drawable.tool_bar_bg);
                mContentRl.setPadding(mShadowLength, mShadowLength, mShadowLength, mShadowLength);

                if (AppDisplay.isPad()) {
                    assumeHeight = size.y + AppResource.getDimensionPixelSize(mContext, R.dimen.ux_view_setting_item_width) + mShadowLength * 2;
                    lrHeight = size.y + AppResource.getDimensionPixelSize(mContext, R.dimen.ux_view_setting_item_width) / 2;
                } else {
                    lrHeight = screenHeight
                            - topbarHeight * 2 // topbar and bottombar height
                            - AppResource.getDimensionPixelSize(mContext, R.dimen.ux_margin_58dp) * 2 // margin
                            - mShadowLength * 2; // shadow
                    assumeHeight = lrHeight + mShadowLength * 2;
                }

                containerLp.width = size.x + mShadowLength * 2;
                containerLp.height = assumeHeight;

                if (mBarPosition == TOOLBAR_POS_LEFT) {
                    containerLp.addRule(RelativeLayout.ALIGN_PARENT_LEFT);
                    containerLp.setMargins(AppResource.getDimensionPixelSize(mContext, R.dimen.ux_margin_16dp) - mShadowLength, 0, 0, 0);
                } else {
                    containerLp.addRule(RelativeLayout.ALIGN_PARENT_RIGHT);
                    containerLp.setMargins(0, 0, AppResource.getDimensionPixelSize(mContext, R.dimen.ux_margin_16dp) - mShadowLength, 0);
                }

                mRealHeight = containerLp.height;
                m1ContainerRl.setLayoutParams(containerLp);

                if ((screenHeight - assumeHeight) / 2 < topbarHeight) {
                    containerLp.addRule(RelativeLayout.ALIGN_PARENT_TOP);
                } else {
                    if (AppDisplay.isPad()) {
                        containerLp.addRule(RelativeLayout.ALIGN_PARENT_TOP, 0);
                        containerLp.addRule(RelativeLayout.CENTER_VERTICAL);
                        containerLp.topMargin = 0;
                    } else {
                        containerLp.addRule(RelativeLayout.CENTER_VERTICAL, 0);
                        containerLp.addRule(RelativeLayout.ALIGN_PARENT_TOP);
                        containerLp.topMargin = AppResource.getDimensionPixelSize(mContext, R.dimen.ux_margin_58dp) - SHADOW_LENGTH *2;
                    }
                }

                break;
            }
        }

        //suyu  拖拽时的停靠区域显示 ----------->
        leftRlParent.getLayoutParams().height = lrHeight;
        rightRlParent.getLayoutParams().height = lrHeight;

        RelativeLayout.LayoutParams leftLp = ((RelativeLayout.LayoutParams) leftRlParent.getLayoutParams());
        RelativeLayout.LayoutParams rightLp = ((RelativeLayout.LayoutParams) rightRlParent.getLayoutParams());
        if ((screenHeight - assumeHeight) / 2 < topbarHeight) {
            leftLp.addRule(RelativeLayout.CENTER_VERTICAL, 0);
            leftLp.addRule(RelativeLayout.ALIGN_PARENT_TOP);
            leftLp.topMargin = topbarHeight + AppDisplay.dp2px(5);

            rightLp.addRule(RelativeLayout.CENTER_VERTICAL, 0);
            rightLp.addRule(RelativeLayout.ALIGN_PARENT_TOP);
            rightLp.topMargin = topbarHeight + AppDisplay.dp2px(5);
        } else {
            if (AppDisplay.isPad()) {
                leftLp.addRule(RelativeLayout.ALIGN_PARENT_TOP, 0);
                leftLp.addRule(RelativeLayout.CENTER_VERTICAL);
                leftLp.topMargin = 0;

                rightLp.addRule(RelativeLayout.ALIGN_PARENT_TOP, 0);
                rightLp.addRule(RelativeLayout.CENTER_VERTICAL);
                rightLp.topMargin = 0;
            } else {
                leftLp.addRule(RelativeLayout.CENTER_VERTICAL, 0);
                leftLp.addRule(RelativeLayout.ALIGN_PARENT_TOP);
                leftLp.topMargin = AppResource.getDimensionPixelSize(mContext, R.dimen.ux_margin_58dp);

                rightLp.addRule(RelativeLayout.CENTER_VERTICAL, 0);
                rightLp.addRule(RelativeLayout.ALIGN_PARENT_TOP);
                rightLp.topMargin = AppResource.getDimensionPixelSize(mContext, R.dimen.ux_margin_58dp);
            }
        }

        mLeftRl.setVisibility(View.GONE);
        mTopRl.setVisibility(View.GONE);
        mRightRl.setVisibility(View.GONE);
        mBottomRl.setVisibility(View.GONE);

        // <------------------------------------------
//        if (AppDevice.isChromeOs(mUiExtensionsManager.getAttachedActivity())) {
//            if (mBarPosition == TOOLBAR_POS_TOP) {
//                mToolBar.setOrientation(BaseBar.HORIZONTAL, RelativeLayout.LayoutParams.MATCH_PARENT, AppResource.getDimensionPixelSize(mContext, R.dimen.ui_bottombar_height));
//            } else if (mBarPosition == TOOLBAR_POS_BOTTOM) {
//                mToolBar.setOrientation(BaseBar.HORIZONTAL, RelativeLayout.LayoutParams.MATCH_PARENT, AppResource.getDimensionPixelSize(mContext, R.dimen.ui_bottombar_height));
//            } else {
//                mToolBar.setOrientation(BaseBar.VERTICAL, AppResource.getDimensionPixelSize(mContext, R.dimen.ui_bottombar_height), RelativeLayout.LayoutParams.WRAP_CONTENT);
//            }
//
//            switch (mBarPosition) {
//                case TOOLBAR_POS_TOP:
//                case TOOLBAR_POS_BOTTOM: {
//                    int contentWidth = mRootRl.getMeasuredWidth();
//                    if (size.x > contentWidth) {
//                        mToolBar.setWidth(size.x);
//
//                        AppUtil.removeViewFromParent(mToolBar.getContentView());
//                        ((ViewGroup) mScrollViewH.getChildAt(0)).addView(mToolBar.getContentView());
//
//                        mScrollViewH.setVisibility(View.VISIBLE);
//                        mScrollViewV.setVisibility(View.INVISIBLE);
//                    } else {
//                        mToolBar.setWidth(ViewGroup.LayoutParams.MATCH_PARENT);
//
//                        AppUtil.removeViewFromParent(mToolBar.getContentView());
//                        mContentRl.addView(mToolBar.getContentView());
//                    }
//                    break;
//                }
//                case TOOLBAR_POS_ANY:
//                    break;
//                case TOOLBAR_POS_LEFT:
//                case TOOLBAR_POS_RIGHT: {
//                    int contentHeight = mLeftRl.getMeasuredHeight();
//                    if (size.y > contentHeight) {
//                        mToolBar.setHeight(size.y);
//
//                        AppUtil.removeViewFromParent(mToolBar.getContentView());
//                        ((ViewGroup) mScrollViewV.getChildAt(0)).addView(mToolBar.getContentView());
//
//                        mScrollViewH.setVisibility(View.INVISIBLE);
//                        mScrollViewV.setVisibility(View.VISIBLE);
//                    } else {
//                        mToolBar.setHeight(ViewGroup.LayoutParams.MATCH_PARENT);
//
//                        AppUtil.removeViewFromParent(mToolBar.getContentView());
//                        mContentRl.addView(mToolBar.getContentView());
//                    }
//                    break;
//                }
//            }
    }

    int mLastWidth = 0;
    int mLastHeight = 0;

    //    @Override // Optimize screen rotation speed
    protected void onLayout(boolean changed, int l, int t, int r, int b) {
//        super.onLayout(changed, l, t, r, b); // Optimize screen rotation speed

        if (changed && mDraggableBoundaryInsetRect != null) {
            if (mBoundaryView == null) {
                setBoundaryView(mUiExtensionsManager.getMainFrame().getContentView());
            }
            if (mDraggableBoundaryRect == null)
                mDraggableBoundaryRect = new Rect();
            mBoundaryView.getGlobalVisibleRect(mDraggableBoundaryRect);
            mDraggableBoundaryRect.set(mDraggableBoundaryRect.left + mDraggableBoundaryInsetRect.left, mDraggableBoundaryRect.top + mDraggableBoundaryInsetRect.top,
                    mDraggableBoundaryRect.right - mDraggableBoundaryInsetRect.right, mDraggableBoundaryRect.bottom - mDraggableBoundaryInsetRect.bottom);
        }
        if (mIsFirstShow) {
            layoutToolbar();
            mIsFirstShow = false;
            mLastWidth = r - l;
            mLastHeight = b - t;
        } else if (AppDevice.isChromeOs(mUiExtensionsManager.getAttachedActivity())) {
            if (mLastWidth != r - l || mLastHeight != b - t) {
                if (!mIsDragging) {
                    layoutToolbar();
                    mLastWidth = r - l;
                    mLastHeight = b - t;
                }
            }
        }
    }

    public void registerToolbarEventListener(IToolbarEventListener listener) {
        mToolbarEventListeners.add(listener);
    }

    public void unregisterToolbarEventListener(IToolbarEventListener listener) {
        mToolbarEventListeners.remove(listener);
    }

    public int getToolBarPosition() {
        return mBarPosition;
    }

    public void setToolBarPosition(@ToolbarPos int pos) {
        if (mBarPosition == pos) return;
        mBarPosition = pos;
        if (mBarPosition == TOOLBAR_POS_ANY) {
            mShowDraggableTargetZone = false;
        }
        layoutToolbar();
    }

    void onToolbarDragEventBegin() {
        for (IToolbarEventListener listener : mToolbarEventListeners) {
            listener.onToolbarDragEventBegin();
        }
    }

    void onToolbarDragEventEnd() {
        for (IToolbarEventListener listener : mToolbarEventListeners) {
            listener.onToolbarDragEventEnd();
        }
    }

    void onToolbarPositionChanged(@ToolbarPos int oldPos, @ToolbarPos int newPos) {
        for (IToolbarEventListener listener : mToolbarEventListeners) {
            listener.onToolbarPositionChanged(oldPos, newPos);
        }
    }

    private void updateToolBarDragBackground() {
        int strokeColor = ThemeConfig.getInstance(mContext).getPrimaryColor();
        int red = Color.red(strokeColor);
        int green = Color.green(strokeColor);
        int blue = Color.blue(strokeColor);
        int solidColor = Color.argb(26, red, green, blue); // alpha = 10%

        Drawable drawable = createDrawable(solidColor, strokeColor, false);
        Drawable dashDrawable = createDrawable(solidColor, strokeColor, true);
        if (mBarPosition == TOOLBAR_POS_TOP) {
            mTopRl.setBackground(drawable);
            mLeftRl.setBackground(dashDrawable);
            mRightRl.setBackground(dashDrawable);
        } else if (mBarPosition == TOOLBAR_POS_LEFT) {
            mTopRl.setBackground(dashDrawable);
            mLeftRl.setBackground(drawable);
            mRightRl.setBackground(dashDrawable);
        } else if (mBarPosition == TOOLBAR_POS_RIGHT) {
            mTopRl.setBackground(dashDrawable);
            mLeftRl.setBackground(dashDrawable);
            mRightRl.setBackground(drawable);
        }
    }

    private GradientDrawable createDrawable(int solidColor, int strokeColor, boolean isDash) {
        GradientDrawable drawable = new GradientDrawable();
        drawable.setColor(solidColor);
        if (isDash) {
            drawable.setStroke(1, strokeColor, 10, 5);
        } else {
            drawable.setStroke(1, strokeColor);
        }
        return drawable;
    }

    private int mRealHeight = 0;

    public int getToolbarRealHeight() {
        return Math.max(mRealHeight, AppResource.getDimensionPixelSize(mContext, R.dimen.ui_bottombar_height));
    }

    public void registerToolbarLayoutListener(IToolbarLayoutListener listener) {
        mToolbarLayoutListeners.add(listener);
    }

    public void unregisterToolbarLayoutListener(IToolbarLayoutListener listener) {
        mToolbarLayoutListeners.remove(listener);
    }

    private void onToolbarLayout(@ToolbarPos int curPos) {
        for (IToolbarLayoutListener listener : mToolbarLayoutListeners) {
            listener.onToolbarLayout(curPos);
        }
    }

    public boolean getTopLayoutGlobalVisibleRect(@NonNull Rect rect) {
        if (mTopRl != null) {
            return mTopRl.getGlobalVisibleRect(rect);
        }

        return false;
    }

    public void setToolBar(UIToolBaseBar toolBar) {
        this.mToolBar = toolBar;
        mContentRl.removeAllViews();
        AppUtil.removeViewFromParent(toolBar.getContentView());
        mContentRl.addView(mToolBar.getContentView());
        if (this.mDragItemVisible) {
            mToolBar.addDragItem();
        }
        layoutToolbar();
    }

    public UIToolBaseBar getToolBar() {
        return mToolBar;
    }

    public IBaseItem getDragItem() {
        if (mDragItem == null)
            mDragItem = createDragButtonItem(mContext);
        return mDragItem;
    }

    private boolean mShowDraggableTargetZone = true;
    private Rect mDraggableBoundaryInsetRect;
    private Rect mDraggableBoundaryRect;
    private Rect mLayoutRect;
    private View mBoundaryView;

    public void setDraggableBoundaryInset(Rect rect) {
        mDraggableBoundaryInsetRect = rect;
    }

    public void setBoundaryView(View view) {
        mBoundaryView = view;
    }

    public void setLayoutRect(Rect rect) {
        mLayoutRect = rect;
    }

    public Rect getLayoutRect() {
        return mLayoutRect;
    }

    public void setAutoResizeHeight(boolean flag) {
        mAutoResizeHeight = flag;
    }

    public void updateThemeColor() {
        mTopRl.setBackground(AppResource.getDrawable(mContext, R.drawable.ui_tool_bar_drag_bg));
        mBottomRl.setBackground(AppResource.getDrawable(mContext, R.drawable.ui_tool_bar_drag_bg));
        mLeftRl.setBackground(AppResource.getDrawable(mContext, R.drawable.ui_tool_bar_drag_bg));
        mRightRl.setBackground(AppResource.getDrawable(mContext, R.drawable.ui_tool_bar_drag_bg));
        mTopSolidView.setBackgroundColor(ThemeConfig.getInstance(mContext).getPrimaryColor());
        mContentRl.setBackgroundColor(AppResource.getColor(mContext, R.color.ux_color_toolbar));
        if (mToolBar != null)
            mToolBar.updateThemeColor();
    }

    public void removeAllViews() { // Optimize screen rotation speed
        if (mRootRl != null) {

        }
    }

    public int getWidth() { // Optimize screen rotation speed
        if (mRootRl != null) {
            return mRootRl.getWidth();
        }
        return 0;
    }

    public int getHeight() { // Optimize screen rotation speed
        if (mRootRl != null) {
            return mRootRl.getHeight();
        }
        return 0;
    }

    public void setHorizontalHeight(int height) {
        mCustomHorizontalHeight = height;
        layoutToolbar();
    }

    public int getHorizontalHeight() {
        return Math.max(mCustomHorizontalHeight, AppResource.getDimensionPixelSize(mContext, R.dimen.ui_bottombar_height));
    }

    public void setVerticalWidth(int width) {
        mCustomVerticalWidth = width;
        layoutToolbar();
    }

    public int getVerticalWidth() {
        return Math.max(mCustomVerticalWidth, AppResource.getDimensionPixelSize(mContext, R.dimen.ui_bottombar_height));
    }

    private DynamicColorProvider backgroundColorProvider = new DynamicColorProvider() {
        @Override
        public int getColor(boolean isDarkMode) {
            return ThemeConfig.getInstance(mContext).getB2();
        }
    };

    public void setBackgroundColorProvider(DynamicColorProvider provider) {
        this.backgroundColorProvider = provider;
    }

    public void setBackgroundColor(@ColorInt int lightColor, @ColorInt int darkColor){
        backgroundColorProvider = new ColorPair(lightColor, darkColor);
    }

    public DynamicColorProvider getBackgroundColorProvider(){
        return  this.backgroundColorProvider;
    }

}
