/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.controls.toolbar.drag;

import android.annotation.SuppressLint;
import android.content.Context;
import android.content.res.Configuration;
import android.graphics.Outline;
import android.graphics.Rect;
import android.os.Build;
import android.view.MotionEvent;
import android.view.View;
import android.view.ViewOutlineProvider;
import android.widget.FrameLayout;

import com.foxit.uiextensions.R;
import com.foxit.uiextensions.UIExtensionsManager;
import com.foxit.uiextensions.pdfreader.impl.MainFrame;
import com.foxit.uiextensions.utils.AppDisplay;
import com.foxit.uiextensions.utils.AppResource;
import com.foxit.uiextensions.utils.AppUtil;
import com.foxit.uiextensions.utils.SystemUiHelper;
import com.foxit.uiextensions.utils.thread.AppThreadManager;

import io.reactivex.annotations.NonNull;

public class UIDragView {
    private View mDragView;
    private UIRelativeLayout mDragContainer;
    private UIExtensionsManager mUiExtensionsManager;
    private FrameLayout mRootLayout;
    private int mLastX, mLastY;
    private Context mContext;
    private int mWidth;
    private int mScreenWidth;
    private int mScreenHeight;
    private int mStatusBarHeight;
    private int mScreenOrientation;

    public UIDragView(@NonNull View view, @NonNull UIExtensionsManager uiExtensionsManager) {
        this(view, uiExtensionsManager, 0);
    }

    public UIDragView(@NonNull View view, @NonNull UIExtensionsManager uiExtensionsManager, int width) {
        mDragView = view;
        mUiExtensionsManager = uiExtensionsManager;
        mWidth = width;
        mScreenOrientation = uiExtensionsManager.getAttachedActivity().getResources().getConfiguration().orientation;

        if (mUiExtensionsManager.getAttachedActivity() == null) {
            throw new NullPointerException("The attached activity is null.");
        }

        mUiExtensionsManager.registerConfigurationChangedListener(mConfigureChangedListener);
        mContext = mUiExtensionsManager.getAttachedActivity().getApplicationContext();
        mRootLayout = (FrameLayout) uiExtensionsManager.getAttachedActivity().getWindow().getDecorView();
//        mDragView.setOnTouchListener(mOnTouchListener);
        mDragView.setBackgroundResource(R.drawable.dlg_bg_4circle_corner_10dp_grayf4f4f4);
        initialize();
    }

    @SuppressLint("ClickableViewAccessibility")
    private void initialize() {
        int screenWidth = AppDisplay.getActivityWidth();
        int screenHeight = AppDisplay.getActivityHeight();

        if (mWidth == 0) {
            mWidth = Math.min(screenWidth, screenHeight) * 2 / 3;
            int mesureSpec_w = View.MeasureSpec.makeMeasureSpec(0, View.MeasureSpec.UNSPECIFIED);
            int mesureSpec_h = View.MeasureSpec.makeMeasureSpec(0, View.MeasureSpec.UNSPECIFIED);
            mDragView.measure(mesureSpec_w, mesureSpec_h);
            int viewWidth = mDragView.getMeasuredWidth();
            if (mWidth <= viewWidth)
                mWidth = viewWidth + AppDisplay.dp2px(10.0f);
        }
        FrameLayout.LayoutParams layoutParams = createLayoutParams((screenWidth - mWidth) / 2, screenHeight - 200- AppDisplay.getRealNavBarHeight(), 0, 0);

        AppUtil.removeViewFromParent(mDragView);
        mDragContainer = new UIRelativeLayout(mUiExtensionsManager, mContext);
        mDragContainer.addView(mDragView);
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
            mDragContainer.setElevation(10);
            mDragContainer.setOutlineProvider(new ViewOutlineProvider() {
                @Override
                public void getOutline(View view, Outline outline) {
                    outline.setRoundRect(0, 0, view.getWidth(), view.getHeight(), AppResource.getDimensionPixelSize(mContext, R.dimen.ux_margin_6dp));
                }
            });
        }
        mDragContainer.setOnTouchListener(mOnTouchListener);
        mRootLayout.addView(mDragContainer, layoutParams);

        mScreenWidth = AppDisplay.getActivityWidth();
        mScreenHeight = AppDisplay.getRawScreenHeight() - AppDisplay.getRealNavBarHeight();
        if (!SystemUiHelper.getInstance().isFullScreenMode(mUiExtensionsManager.getAttachedActivity()))
            mStatusBarHeight = SystemUiHelper.getInstance().getStatusBarHeight(mContext);
    }

    private View.OnTouchListener mOnTouchListener = new View.OnTouchListener() {
        @Override
        public boolean onTouch(View v, MotionEvent event) {
            switch (event.getAction()) {
                case MotionEvent.ACTION_DOWN:
                    mLastX = (int) event.getRawX();
                    mLastY = (int) event.getRawY();
                    break;
                case MotionEvent.ACTION_MOVE:
                    int left, top, right, bottom;
                    int dx = (int) event.getRawX() - mLastX;
                    int dy = (int) event.getRawY() - mLastY;
                    left = v.getLeft() + dx;
                    if (left < 0) {
                        left = 0;
                    }
                    right = left + v.getWidth();
                    if (right > mScreenWidth) {
                        right = mScreenWidth;
                        left = right - v.getWidth();
                    }
                    top = v.getTop() + dy;
                    if (top < mStatusBarHeight + 2) {
                        top = mStatusBarHeight + 2;
                    }
                    bottom = top + v.getHeight();
                    if (bottom > mScreenHeight) {
                        bottom = mScreenHeight;
                        top = bottom - v.getHeight();
                    }
                    v.layout(left, top, right, bottom);
                    mLastX = (int) event.getRawX();
                    mLastY = (int) event.getRawY();
                    break;
                case MotionEvent.ACTION_UP:
                    v.setLayoutParams(createLayoutParams(v.getLeft(), v.getTop(), 0, 0));
                    break;
            }
            return true;
        }
    };

    public void removeDragView() {
        mRootLayout.removeView(mDragContainer);
        mUiExtensionsManager.unregisterConfigurationChangedListener(mConfigureChangedListener);
    }

    public void setVisible(boolean visible) {
        mDragContainer.setVisibility(visible ? View.VISIBLE : View.GONE);
    }

    private FrameLayout.LayoutParams createLayoutParams(int left, int top, int right, int bottom) {
        FrameLayout.LayoutParams layoutParams = new FrameLayout.LayoutParams(mWidth, FrameLayout.LayoutParams.WRAP_CONTENT);
        layoutParams.setMargins(left, top, right, bottom);
        return layoutParams;
    }

    private final UIExtensionsManager.ConfigurationChangedListener mConfigureChangedListener = new UIExtensionsManager.ConfigurationChangedListener() {
        @Override
        public void onConfigurationChanged(Configuration newConfig) {
            if (mScreenOrientation == newConfig.orientation) {
                return;
            }
            mScreenOrientation = newConfig.orientation;

            final int rootWidth = mUiExtensionsManager.getRootView().getWidth();
            final int rootHeight = mUiExtensionsManager.getRootView().getHeight();

            final Rect rect = new Rect();
            mDragContainer.getGlobalVisibleRect(rect);

            AppThreadManager.getInstance().getMainThreadHandler().postDelayed(new Runnable() {
                @Override
                public void run() {
                    mScreenWidth = AppDisplay.getActivityWidth();
                    mScreenHeight = AppDisplay.getRawScreenHeight() - AppDisplay.getRealNavBarHeight();

                    int newRootWidth = mUiExtensionsManager.getRootView().getWidth();
                    int newRootHeight = mUiExtensionsManager.getRootView().getHeight();

                    float scaleL = (float) rect.left / (rootWidth - rect.width());
                    int newLeft = (int) (scaleL * (newRootWidth - rect.width()));

                    float scaleT = (float) rect.top / rootHeight;
                    int newTop = (int) (scaleT * newRootHeight);

                    FrameLayout.LayoutParams lp = (FrameLayout.LayoutParams) mDragContainer.getLayoutParams();
                    lp.setMargins(newLeft, newTop, 0, 0);
                    mDragContainer.setLayoutParams(lp);
                }
            }, 300);

            MainFrame mainFrame = (MainFrame) mUiExtensionsManager.getMainFrame();
            setVisible(!mainFrame.isShowFullScreenUI());
        }
    };

}
