/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.controls.toolbar.drag.addtools;


import android.app.Activity;
import android.content.Context;
import android.content.DialogInterface;
import android.content.res.Configuration;
import android.graphics.Point;
import android.graphics.Rect;
import android.graphics.RectF;
import android.text.TextUtils;
import android.util.SparseArray;
import android.view.View;
import android.widget.ImageView;
import android.widget.LinearLayout;
import android.widget.TextView;

import com.foxit.uiextensions.DocumentManager;
import com.foxit.uiextensions.Module;
import com.foxit.uiextensions.R;
import com.foxit.uiextensions.UIExtensionsManager;
import com.foxit.uiextensions.annots.stamp.StampConstants;
import com.foxit.uiextensions.annots.stamp.StampUtil;
import com.foxit.uiextensions.controls.dialog.UIMatchDialog;
import com.foxit.uiextensions.controls.dialog.UIPopoverFrag;
import com.foxit.uiextensions.controls.propertybar.PropertyBar;
import com.foxit.uiextensions.controls.toolbar.BaseBar;
import com.foxit.uiextensions.controls.toolbar.IToolSupply;
import com.foxit.uiextensions.controls.toolbar.ToolConstants;
import com.foxit.uiextensions.controls.toolbar.ToolItemBean;
import com.foxit.uiextensions.controls.toolbar.ToolProperty;
import com.foxit.uiextensions.controls.toolbar.UIToolView;
import com.foxit.uiextensions.controls.toolbar.impl.UIColorItem;
import com.foxit.uiextensions.pdfreader.AddToolsBean;
import com.foxit.uiextensions.pdfreader.config.ReadStateConfig;
import com.foxit.uiextensions.security.standard.PasswordModule;
import com.foxit.uiextensions.utils.AppDevice;
import com.foxit.uiextensions.utils.AppDisplay;
import com.foxit.uiextensions.utils.AppResource;
import com.foxit.uiextensions.utils.thread.AppThreadManager;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import androidx.annotation.NonNull;
import androidx.core.widget.NestedScrollView;
import androidx.recyclerview.widget.DefaultItemAnimator;
import androidx.recyclerview.widget.GridLayoutManager;
import androidx.recyclerview.widget.ItemTouchHelper;
import androidx.recyclerview.widget.LinearLayoutManager;
import androidx.recyclerview.widget.RecyclerView;

public class AddToolsDialog extends UIMatchDialog {
    private List<ToolItemBean> mAddedTools;
    private SparseArray<List<ToolItemBean>> mCustomTools;

    private final Context mContext;
    private final UIExtensionsManager mUIExtensionsManager;
    private PropertyBar mPropertyBar;

    private View mAddedToolsContainer;
    private TextView mNoAddedToolsTv;
    private RecyclerView mAddedToolsGridView;
    private GridLayoutManager mGridLayoutManager;
    private AddedToolsAdapter mAddedToolsAdapter;
    private NestedScrollView mToolsScrollView;

    private AddToolsAdapter mEditToolsAdapter;
    private AddToolsAdapter mCommentToolsAdapter;
    private AddToolsAdapter mDrawingToolsAdapter;
    private AddToolsAdapter mFormToolsAdapter;
    private AddToolsAdapter mProtectToolsAdapter;
    private RecyclerView mEditToolsListView;
    private RecyclerView mCommentToolsListView;
    private RecyclerView mDrawingToolsListView;
    private RecyclerView mFormToolsListView;
    private RecyclerView mProtectToolsListView;
    private View mEditToolsContentView;
    private View mCommentToolsContentView;
    private View mDrawingToolsContentView;
    private View mFormToolsContentView;
    private View mProtectToolsContentView;

    private RecyclerView mLastSelectedListView;
    private int mLastSelectedPosition;

    private int mItemHeight;
    private int mSpanCount = 6;
    private int mHorSpacing;
    private int mVerSpacing;
    private int mStatusBarHeight = 0;
    private boolean mIsToolItemsChanged = false;
    private final List<Integer> mIdEditList = new ArrayList<>(Arrays.asList(R.id.add_tools_edit_1,
            R.id.add_tools_edit_2, R.id.add_tools_edit_3, R.id.add_tools_edit_4, R.id.add_tools_edit_5
    ));
    private final List<Integer> mIdCommentList = new ArrayList<>(Arrays.asList(R.id.add_tools_comment_1, R.id.add_tools_comment_2, R.id.add_tools_comment_3, R.id.add_tools_comment_4, R.id.add_tools_comment_5,
            R.id.add_tools_comment_6, R.id.add_tools_comment_7, R.id.add_tools_comment_8, R.id.add_tools_comment_9, R.id.add_tools_comment_10,
            R.id.add_tools_comment_11, R.id.add_tools_comment_12, R.id.add_tools_comment_13, R.id.add_tools_comment_14, R.id.add_tools_comment_15
    ));
    private final List<Integer> mIdDrawingList = new ArrayList<>(Arrays.asList(R.id.add_tools_drawing_toolbar_1, R.id.add_tools_drawing_toolbar_2, R.id.add_tools_drawing_toolbar_3, R.id.add_tools_drawing_toolbar_4,
            R.id.add_tools_drawing_toolbar_5, R.id.add_tools_drawing_toolbar_6, R.id.add_tools_drawing_toolbar_7, R.id.add_tools_drawing_toolbar_8,
            R.id.add_tools_drawing_toolbar_9, R.id.add_tools_drawing_toolbar_10
    ));
    private final List<Integer> mIdFormList = new ArrayList<>(Arrays.asList(R.id.add_tools_form_toolbar_1, R.id.add_tools_form_toolbar_2,
            R.id.add_tools_form_toolbar_3, R.id.add_tools_form_toolbar_4, R.id.add_tools_form_toolbar_5, R.id.add_tools_form_toolbar_6, R.id.add_tools_form_toolbar_7
    ));
    private final List<Integer> mIdProtectList = new ArrayList<>(Arrays.asList(R.id.add_tools_protect_toolbar_1, R.id.add_tools_protect_toolbar_2, R.id.add_tools_protect_toolbar_3,
            R.id.add_tools_protect_toolbar_4, R.id.add_tools_protect_toolbar_5, R.id.add_tools_protect_toolbar_6, R.id.add_tools_protect_toolbar_7));

    public AddToolsDialog(Context context, UIExtensionsManager uiExtensionsManager) {
        super(context);
        if (context instanceof Activity) {
            if (AppDisplay.isStatusBarShown((Activity) context))
                mStatusBarHeight = AppDisplay.getStatusBarHeight();
        }

        mContext = context.getApplicationContext();
        mUIExtensionsManager = uiExtensionsManager;
        initView();
        initStyle();

        setOnShowListener(new OnShowListener() {
            @Override
            public void onShow(DialogInterface dialog) {
                refreshLayout();
            }
        });

        setOnDismissListener(new DialogInterface.OnDismissListener() {
            @Override
            public void onDismiss(DialogInterface dialog) {
                if (mUIExtensionsManager != null)
                    mUIExtensionsManager.unregisterConfigurationChangedListener(mConfigurationChangedListener);

                if (mDismissListener != null)
                    mDismissListener.onDismiss();
            }
        });
    }

    public void initData(List<ToolItemBean> addedTools, SparseArray<List<ToolItemBean>> customTools) {
        mIsToolItemsChanged = false;
        mAddedTools = addedTools;
        mCustomTools = customTools;

        //added tools
        if (mAddedTools.size() > 0) {
            mNoAddedToolsTv.setVisibility(View.GONE);
            mAddedToolsGridView.setVisibility(View.VISIBLE);
        } else {
            mNoAddedToolsTv.setVisibility(View.VISIBLE);
            mAddedToolsGridView.setVisibility(View.GONE);
        }
        mAddedToolsAdapter.setTools(mAddedTools);
        mAddedToolsAdapter.notifyUpdateData();

        //edit tools
        List<ToolItemBean> editTools = mCustomTools.get(AddToolsBean.EDIT_TOOLS);
        if (editTools == null)
            editTools = new ArrayList<>();
        mEditToolsAdapter.setTools(editTools);
        if (editTools.size() > 0) {
            if (editTools.size() == mIdEditList.size()) {
                mEditToolsAdapter.setIdsList(mIdEditList);
            } else {
                List<Integer> idList = new ArrayList<>();
                for (int i = 0; i < editTools.size(); i++) {
                    idList.add(mIdEditList.get(i));
                }
                mEditToolsAdapter.setIdsList(idList);
            }
        }
        mEditToolsAdapter.notifyDataSetChanged();
        mEditToolsContentView.setVisibility(editTools.size() > 0 ? View.VISIBLE : View.GONE);

        //comment tools
        List<ToolItemBean> commonTools = mCustomTools.get(AddToolsBean.COMMENT_TOOLS);
        if (commonTools == null)
            commonTools = new ArrayList<>();
        mCommentToolsAdapter.setTools(commonTools);
        if (commonTools.size() > 0) {
            if (commonTools.size() == mIdCommentList.size()) {
                mCommentToolsAdapter.setIdsList(mIdCommentList);
            } else {
                List<Integer> idList = new ArrayList<>();
                for (int i = 0; i < commonTools.size(); i++) {
                    idList.add(mIdCommentList.get(i));
                }
                mCommentToolsAdapter.setIdsList(idList);
            }
        }
        mCommentToolsAdapter.notifyDataSetChanged();
        mCommentToolsContentView.setVisibility(commonTools.size() > 0 ? View.VISIBLE : View.GONE);

        //drawing tools
        List<ToolItemBean> drawingTools = mCustomTools.get(AddToolsBean.DRAWING_TOOLS);
        if (drawingTools == null)
            drawingTools = new ArrayList<>();
        mDrawingToolsAdapter.setTools(drawingTools);
        if (drawingTools.size() > 0) {
            if (drawingTools.size() == mIdDrawingList.size()) {
                mDrawingToolsAdapter.setIdsList(mIdDrawingList);
            } else {
                List<Integer> idList = new ArrayList<>();
                for (int i = 0; i < drawingTools.size(); i++) {
                    idList.add(mIdDrawingList.get(i));
                }
                mDrawingToolsAdapter.setIdsList(idList);
            }
        }
        mDrawingToolsAdapter.notifyDataSetChanged();
        mDrawingToolsContentView.setVisibility(drawingTools.size() > 0 ? View.VISIBLE : View.GONE);

        //form tools
        List<ToolItemBean> formTools = mCustomTools.get(AddToolsBean.FORM_TOOLS);
        if (formTools == null)
            formTools = new ArrayList<>();
        mFormToolsAdapter.setTools(formTools);
        if (formTools.size() > 0) {
            if (formTools.size() == mIdFormList.size()) {
                mFormToolsAdapter.setIdsList(mIdFormList);
            } else {
                List<Integer> idList = new ArrayList<>();
                for (int i = 0; i < formTools.size(); i++) {
                    idList.add(mIdFormList.get(i));
                }
                mFormToolsAdapter.setIdsList(idList);
            }
        }
        mFormToolsAdapter.notifyDataSetChanged();
        mFormToolsContentView.setVisibility(formTools.size() > 0 ? View.VISIBLE : View.GONE);

        mEditToolsListView.setVisibility(View.VISIBLE);
        mCommentToolsListView.setVisibility(View.VISIBLE);
        mDrawingToolsListView.setVisibility(View.VISIBLE);
        mFormToolsListView.setVisibility(View.VISIBLE);

        //protect tools
        initData_ProtectTools();
    }

    private void initData_ProtectTools() {
        List<ToolItemBean> protectTools = mCustomTools.get(AddToolsBean.PROTECT_TOOLS);
        if (protectTools == null)
            protectTools = new ArrayList<>();
        mProtectToolsAdapter.setTools(protectTools);
        if (protectTools.size() > 0) {
            List<Integer> idList = new ArrayList<>();
            for (int i = 0; i < protectTools.size(); i++) {
                    idList.add(mIdProtectList.get(i));
                }
            mProtectToolsAdapter.setIdsList(idList);
        }
        mProtectToolsAdapter.notifyDataSetChanged();
        mProtectToolsContentView.setVisibility(protectTools.size() > 0 ? View.VISIBLE : View.GONE);
        mProtectToolsListView.setVisibility(View.VISIBLE);
    }

    private void initView() {
        View contentView = View.inflate(mContext, R.layout.add_tools_layout, null);
        initAddedTools(contentView);
        initToolList(contentView);
        setContentView(contentView);
    }

    private void initAddedTools(View contentView) {
        mItemHeight = AppResource.getDimensionPixelSize(mContext, R.dimen.ux_view_setting_item_width);
        mHorSpacing = AppResource.getDimensionPixelSize(mContext, R.dimen.ux_margin_8dp);

        mAddedToolsContainer = contentView.findViewById(R.id.added_tools_container);
        mNoAddedToolsTv = contentView.findViewById(R.id.tv_no_added_tools_prompt);
        mAddedToolsGridView = contentView.findViewById(R.id.rv_added_tools);
        mAddedToolsGridView.setHasFixedSize(true);
        mAddedToolsGridView.setItemAnimator(new DefaultItemAnimator());
        mAddedToolsAdapter = new AddedToolsAdapter(mContext);
        mAddedToolsGridView.setAdapter(mAddedToolsAdapter);
        SpacesItemDecoration spacesItemDecoration = new SpacesItemDecoration();
        mAddedToolsGridView.addItemDecoration(spacesItemDecoration);

        mAddedToolsAdapter.setOnItemClickListener(new AddedToolsAdapter.OnItemClickListener() {
            @Override
            public void onItemClick(int position, View view) {
                mIsToolItemsChanged = true;
                if (mAddedToolsAdapter.getItemCount() == 0) {
                    mNoAddedToolsTv.setVisibility(View.VISIBLE);
                    mAddedToolsGridView.setVisibility(View.GONE);
                }
            }

            @Override
            public void onMove() {
                mIsToolItemsChanged = true;
            }
        });

        ItemTouchHelper itemTouchHelper = new ItemTouchHelper(new AddedToolsTouchHelper(mAddedToolsAdapter));
        itemTouchHelper.attachToRecyclerView(mAddedToolsGridView);
    }

    private void initToolList(View contentView) {
        mEditToolsAdapter = new AddToolsAdapter(mContext);
        mCommentToolsAdapter = new AddToolsAdapter(mContext);
        mDrawingToolsAdapter = new AddToolsAdapter(mContext);
        mFormToolsAdapter = new AddToolsAdapter(mContext);
        mProtectToolsAdapter = new AddToolsAdapter(mContext);

        mToolsScrollView = contentView.findViewById(R.id.add_tools_scrollview);
        mEditToolsListView = contentView.findViewById(R.id.rv_edit_tools);
        mCommentToolsListView = contentView.findViewById(R.id.rv_comment_tools);
        mDrawingToolsListView = contentView.findViewById(R.id.rv_drawing_tools);
        mFormToolsListView = contentView.findViewById(R.id.rv_form_tools);
        mProtectToolsListView = contentView.findViewById(R.id.rv_protect_tools);

        mEditToolsContentView = contentView.findViewById(R.id.add_edit_tools_container);
        mCommentToolsContentView = contentView.findViewById(R.id.add_comment_tools_container);
        mDrawingToolsContentView = contentView.findViewById(R.id.add_drawing_tools_container);
        mFormToolsContentView = contentView.findViewById(R.id.add_form_tools_container);
        mProtectToolsContentView = contentView.findViewById(R.id.add_protect_tools_container);

        initListView(contentView,
                R.id.pack_edit_tools,
                R.id.iv_collapse_edit_tools,
                mEditToolsListView,
                mEditToolsAdapter);
        initListView(contentView,
                R.id.pack_comment_tools,
                R.id.iv_collapse_comment_tools,
                mCommentToolsListView,
                mCommentToolsAdapter);
        initListView(contentView,
                R.id.pack_drawing_tools,
                R.id.iv_collapse_drawing_tools,
                mDrawingToolsListView,
                mDrawingToolsAdapter);
        initListView(contentView,
                R.id.pack_form_tools,
                R.id.iv_collapse_form_tools,
                mFormToolsListView,
                mFormToolsAdapter);
        initListView(contentView,
                R.id.pack_protect_tools,
                R.id.iv_collapse_protect_tools,
                mProtectToolsListView,
                mProtectToolsAdapter);
    }

    private void initListView(View contentView,
                              int titleContainerId,
                              int expandViewId,
                              final RecyclerView toolListView,
                              final AddToolsAdapter adapter) {
        LinearLayout packView = contentView.findViewById(titleContainerId);
        final ImageView expandView = contentView.findViewById(expandViewId);
        expandView.setColorFilter(mContext.getResources().getColor(R.color.i3));
        toolListView.setNestedScrollingEnabled(false);
        final LinearLayoutManager linearLayoutManager = new LinearLayoutManager(mContext,
                LinearLayoutManager.VERTICAL, false) {
            @Override
            public boolean canScrollVertically() {
                return false;
            }
        };
        toolListView.setLayoutManager(linearLayoutManager);
        toolListView.setItemAnimator(new DefaultItemAnimator());
        toolListView.setAdapter(adapter);
        adapter.setOnItemClickListener(new AddToolsAdapter.OnItemClickListener() {
            @Override
            public void onItemClick(int type, final int postion, View view, ToolItemBean itemBean) {
                if (postion == -1) return;

                IToolSupply toolSupply = mUIExtensionsManager.getToolsManager().getToolSupply(itemBean.type);
                if (type == AddToolsAdapter.OPER_ADD_TOOL) {
                    if (mAddedToolsAdapter.getItemCount() == 0) {
                        mNoAddedToolsTv.setVisibility(View.GONE);
                        mAddedToolsGridView.setVisibility(View.VISIBLE);
                    }
                    mIsToolItemsChanged = true;

                    ToolItemBean newTool = new ToolItemBean();
                    newTool.itemStyle = ToolItemBean.DEFAULT_STYLE;
                    newTool.type = itemBean.type;
                    newTool.name = itemBean.name;
                    UIColorItem toolItem = toolSupply.createToolItem(itemBean.type, true);
                    if (itemBean.property != null) {
                        newTool.property = new ToolProperty(itemBean.property);
                        toolItem.setAlphaColorBg(itemBean.property.color);

                        if (itemBean.type == ToolConstants.Stamp) {
                            UIToolView toolView = (UIToolView) toolItem.getCustomView();
                            toolView.setFillBackgroundResource(StampUtil.getStampIconThumbnail(itemBean.property.style));
                            if (itemBean.property.style == StampConstants.CUSTOM_IMAGE_STAMP
                                    || itemBean.property.style == StampConstants.CUSTOM_TEXT_STAMP) {
                                toolView.setBackgroundColorFilter(AppResource.getColor(mContext, R.color.i3));
                            }
                        }
                    } else {
                        toolItem = toolSupply.createToolItem(itemBean.type, true);
                    }

                    boolean isReflow = mUIExtensionsManager.getState() == ReadStateConfig.STATE_REFLOW;
                    if (isReflow) {
                        toolItem.setEnable(false);
                    } else {
                        DocumentManager documentManager = mUIExtensionsManager.getDocumentManager();
                        if (itemBean.type == ToolConstants.TextField
                                || itemBean.type == ToolConstants.CheckBox
                                || itemBean.type == ToolConstants.RadioButton
                                || itemBean.type == ToolConstants.ComboBox
                                || itemBean.type == ToolConstants.ListBox
                                || itemBean.type == ToolConstants.SignatureField
                                || itemBean.type == ToolConstants.ImageField) {
                            toolItem.setEnable(!documentManager.isXFA()
                                    && documentManager.canModifyForm()
                                    && mUIExtensionsManager.isEnableModification()
                                    && !documentManager.isSign()
                                    && documentManager.withAddPermission());
                        } else if (itemBean.type == ToolConstants.Audio
                                || itemBean.type == ToolConstants.Video
                                || itemBean.type == ToolConstants.Link) {
                            toolItem.setEnable(documentManager.withAddPermission()
                                    && documentManager.canEdit() && mUIExtensionsManager.isEnableModification());
                        } else if (itemBean.type == ToolConstants.Text
                                || itemBean.type == ToolConstants.Image) {
                            toolItem.setEnable(documentManager.canEdit() && mUIExtensionsManager.isEnableModification());
                        } else if (itemBean.type == ToolConstants.ProtectTypeStandard
                                || itemBean.type == ToolConstants.ProtectTypeStandRemove) {
                            String filePath = mUIExtensionsManager.getPDFViewCtrl().getFilePath();
                            boolean isPPDF = false;
                            if (!TextUtils.isEmpty(filePath)) {
                                isPPDF = filePath.endsWith(".ppdf");
                            }
                            boolean isAvailable = false;
                            PasswordModule passwordModule = (PasswordModule) mUIExtensionsManager.getModuleByName(Module.MODULE_NAME_PASSWORD);
                            if (passwordModule != null){
                                isAvailable = passwordModule.getSecurityHandler().isAvailable();
                            }
                            toolItem.setEnable(isAvailable && !isPPDF);
                        } else if (itemBean.type == ToolConstants.ProtectTypeTrustCert) {
                            toolItem.setEnable(true);
                        } else {
                            toolItem.setEnable(documentManager.canAddAnnot()
                                    && mUIExtensionsManager.isEnableModification()
                                    && documentManager.withAddPermission());
                        }
                    }
                    newTool.toolItem = toolItem;
                    List<ToolItemBean> addedTools = mAddedToolsAdapter.getTools();
                    addedTools.add(newTool);
                    mAddedToolsAdapter.notifyItemInserted(addedTools.size());
                } else if (type == AddToolsAdapter.OPER_SHOW_PROPERTY) {
                    mLastSelectedListView = toolListView;
                    mLastSelectedPosition = postion;

                    mPropertyBar = toolSupply.getPropertyBar();
                    mPropertyBar.setTag(itemBean.type);
                    toolSupply.resetPropertyBar(itemBean, new PropertyBar.PropertyChangeListener() {
                        @Override
                        public void onValueChanged(long property, int value) {
                            adapter.notifyItemChanged(postion);
                        }

                        @Override
                        public void onValueChanged(long property, float value) {
                            adapter.notifyItemChanged(postion);
                        }

                        @Override
                        public void onValueChanged(long property, String value) {
                            adapter.notifyItemChanged(postion);
                        }
                    });
                    showPropertyBar(view);
                }
            }
        });

        packView.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                if (toolListView.getVisibility() == View.VISIBLE) {
                    toolListView.setVisibility(View.GONE);
                    expandView.setImageResource(R.drawable.comment_pack_up_icon);
                } else {
                    toolListView.setVisibility(View.VISIBLE);
                    expandView.setImageResource(R.drawable.rd_collapse_arrow);
                }
            }
        });
    }

    public boolean isToolItemsChanged() {
        return mIsToolItemsChanged;
    }

    private void initStyle() {
        setTitle(AppResource.getString(mContext, R.string.add_tools_title));
        setBackButtonText(AppResource.getString(mContext, R.string.fx_string_close));
        setBackButtonVisible(View.VISIBLE);
        setBackButtonStyle(TEXT_BACK);
        setTitlePosition(BaseBar.TB_Position.Position_CENTER);
        setStyle(DLG_TITLE_STYLE_BG_DEFAULT);
    }

    private void computeSize() {
        mAddedToolsContainer.measure(0, 0);
        int width = mAddedToolsContainer.getWidth();
        Point size = new Point(mItemHeight, mItemHeight);
        int margin = AppResource.getDimensionPixelSize(mContext, R.dimen.ux_margin_8dp);
        mSpanCount = Math.max(1, (width - margin) / (margin + size.x + 2));
        mVerSpacing = (width - size.x * mSpanCount) / (mSpanCount + 1);
    }

    private class SpacesItemDecoration extends RecyclerView.ItemDecoration {
        @Override
        public void getItemOffsets(@NonNull Rect outRect, @NonNull View view,
                                   @NonNull RecyclerView parent, @NonNull RecyclerView.State state) {
            if (mSpanCount > 0) {
                int position = parent.getChildAdapterPosition(view);
                int spanIndex = position % mSpanCount;

                outRect.left = mVerSpacing - spanIndex * mVerSpacing / mSpanCount;
                outRect.right = (spanIndex + 1) * mVerSpacing / mSpanCount;

                outRect.top = mHorSpacing;
                outRect.bottom = mHorSpacing;
            } else {
                outRect.setEmpty();
            }
        }
    }

    private void refreshLayout() {
        View rootView = mUIExtensionsManager.getRootView();
        rootView.measure(0, 0);
        int rootWidth = rootView.getWidth();
        int rootHeight = rootView.getHeight();
        LinearLayout.LayoutParams addedParams = (LinearLayout.LayoutParams) mAddedToolsContainer.getLayoutParams();
        if (rootWidth > rootHeight) {
            addedParams.width = LinearLayout.LayoutParams.MATCH_PARENT;
            addedParams.height = mItemHeight * 2 + mHorSpacing * 4;
        } else {
            addedParams.width = LinearLayout.LayoutParams.MATCH_PARENT;
            addedParams.height = mItemHeight * 3 + mHorSpacing * 6;
        }
        mAddedToolsContainer.setLayoutParams(addedParams);

        computeSize();
        if (mGridLayoutManager != null) {
            mGridLayoutManager.setSpanCount(mSpanCount);
            mGridLayoutManager.requestLayout();
        } else {
            mGridLayoutManager = new GridLayoutManager(mContext, mSpanCount);
            mAddedToolsGridView.setLayoutManager(mGridLayoutManager);
        }

        AppThreadManager.getInstance().getMainThreadHandler().post(new Runnable() {
            @Override
            public void run() {
                mAddedToolsAdapter.notifyUpdateData();
            }
        });
    }

    private void updatePropertyBar() {
        if (mPropertyBar != null && mPropertyBar.isShowing()) {
            AppThreadManager.getInstance().getMainThreadHandler().postDelayed(new Runnable() {
                @Override
                public void run() {
                    LinearLayoutManager layoutManager = (LinearLayoutManager) mLastSelectedListView.getLayoutManager();
                    if (layoutManager == null) return;

                    View contentView = (View) mLastSelectedListView.getParent();
                    final View itemView = layoutManager.findViewByPosition(mLastSelectedPosition);
                    if (itemView != null) {
                        mToolsScrollView.scrollTo(0, contentView.getTop() + itemView.getTop());

                        View propView = itemView.findViewById(R.id.add_tool_color_iv);
                        Rect rect = new Rect();
                        propView.getGlobalVisibleRect(rect);

                        int[] location = new int[2];
                        propView.getLocationOnScreen(location);
                        int x = location[0];
                        int y = location[1];

                        if (AppDevice.isChromeOs(mUIExtensionsManager.getAttachedActivity())) {
                            mUIExtensionsManager.getRootView().getLocationOnScreen(location);
                            x -= location[0];
                            y -= location[1];
                            y -= mStatusBarHeight;
                        }

                        RectF rectF = new RectF(x, y, x + rect.width(), y + rect.height());
                        mPropertyBar.update(rectF, UIPopoverFrag.ARROW_RIGHT);
                    } else {
                        mToolsScrollView.scrollTo(0, contentView.getTop());
                        dismissPropertyBar();
                    }
                }
            }, 300);
        }
    }

    private void showPropertyBar(View view) {
        Rect rect = new Rect();
        view.getGlobalVisibleRect(rect);

        int x = 0;
        int y = 0;
        String out = "";
        int[] location = new int[2];

        // chrome os debug logout
//            int sx, sy, wx, wy = 0;
//            mUIExtensionsManager.getRootView().getLocationOnScreen(location);
//            sx = location[0];
//            sy = location[1];
//            mUIExtensionsManager.getRootView().getLocationInWindow(location);
//            wx = location[0];
//            wy = location[1];
//            out += String.format("Read-Screen : %d, %d\nRead-inWindow : %d, %d\n------------\n", sx, sy, wx, wy);
//
//
//            getRootView().getLocationOnScreen(location);
//            sx = location[0];
//            sy = location[1];
//            getRootView().getLocationInWindow(location);
//            wx = location[0];
//            wy = location[1];
//            out += String.format("Dialog-Screen : %d, %d\nDialog-inWindow : %d, %d\n------------\n", sx, sy, wx, wy);
//            x = sx; y = sy;
//
//            view.getLocationOnScreen(location);
//            sx = location[0];
//            sy = location[1];
//            view.getLocationInWindow(location);
//            wx = location[0];
//            wy = location[1];
//            out += String.format("View-Screen : %d, %d\nView-inWindow : %d, %d\n------------\n", sx, sy, wx, wy);
//
//            x += wx;
//            y += wy + mStatusBarHeight;
//
//            out += String.format("show : %d - %d %d", x, y, mStatusBarHeight);
//            UIToast.getInstance(mContext).show(out);
// end

        view.getLocationOnScreen(location);
        x = location[0];
        y = location[1];

        if (AppDevice.isChromeOs(mUIExtensionsManager.getAttachedActivity())) {
            mUIExtensionsManager.getRootView().getLocationOnScreen(location);
            x -= location[0];
            y -= location[1];
            y -= mStatusBarHeight;
        }


        RectF rectF = new RectF(x, y, x + rect.width(), y + rect.height());
        mPropertyBar.show(rectF, true, UIPopoverFrag.ARROW_RIGHT);
    }

    public void notifyUpdateAddedTools(){
        if (mAddedToolsAdapter != null){
            AppThreadManager.getInstance().getMainThreadHandler().post(new Runnable() {
                @Override
                public void run() {
                    mAddedToolsAdapter.notifyUpdateData();
                }
            });
        }
    }

    private void dismissPropertyBar() {
        if (mPropertyBar != null && mPropertyBar.isShowing())
            mPropertyBar.dismiss();
    }

    @Override
    public void showDialog() {
        super.showDialog();
        if (mUIExtensionsManager != null)
            mUIExtensionsManager.registerConfigurationChangedListener(mConfigurationChangedListener);
    }

    private final UIExtensionsManager.ConfigurationChangedListener mConfigurationChangedListener = new UIExtensionsManager.ConfigurationChangedListener() {
        @Override
        public void onConfigurationChanged(Configuration newConfig) {
            if (isShowing()) {
                resetWH();
                updatePropertyBar();
                AppThreadManager.getInstance().getMainThreadHandler().postDelayed(new Runnable() {
                    @Override
                    public void run() {
                        refreshLayout();
                    }
                }, 200);
            }
        }
    };

}
