/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.controls.toolbar.impl;

import android.content.Context;
import android.content.res.ColorStateList;
import android.content.res.Resources;
import android.graphics.Color;
import android.graphics.PorterDuff;
import android.graphics.Typeface;
import android.graphics.drawable.Drawable;
import android.os.Build;
import android.text.InputFilter;
import android.text.TextUtils;
import android.view.Gravity;
import android.view.View;
import android.view.ViewGroup;
import android.widget.ImageView;
import android.widget.LinearLayout;
import android.widget.TextView;

import com.foxit.uiextensions.R;
import com.foxit.uiextensions.annots.common.UIBtnImageView;
import com.foxit.uiextensions.controls.toolbar.IBaseItem;
import com.foxit.uiextensions.theme.ColorPair;
import com.foxit.uiextensions.theme.DynamicColorProvider;
import com.foxit.uiextensions.utils.AppDisplay;
import com.foxit.uiextensions.utils.AppResource;
import com.foxit.uiextensions.utils.AppUtil;

import androidx.annotation.ColorInt;
import androidx.annotation.Nullable;

public class BaseItemImpl implements IBaseItem {
    protected Context mContext;

    protected ItemType mItemType;
    private int mTag;
    protected int mRelation;
    protected int mHighlightImageRes;

    protected LinearLayout mRootLayout;//content view
    protected UIBtnImageView mImage;
    protected TextView mTextView;//title
    private View mCustomView;

    protected LinearLayout.LayoutParams mTextParams;
    protected LinearLayout.LayoutParams mImgParams;//image or customView

    private ColorStateList mImageColorStateList;
    private ColorStateList mTextColorStateList;
    protected int mIconId;
    protected int mTextSizeUnit = -1;
    protected float mTextSize;
    protected int mSelectTextColor = Color.RED;
    protected int mNoSelectTextColor = Color.BLACK;
    protected int mDefinedTextSize = 12;
    protected boolean mUseTextColorRes = false;
    protected boolean mIsSelected = false;
    protected boolean mIsEnabled = true;
    private int mColor = 0;
    protected View.OnClickListener mClickListener;
    protected View.OnLongClickListener mLongClickListener;
    protected OnItemClickListener mItemClickListener;
    protected OnItemLongPressListener mItemLongClickListener;

    public BaseItemImpl(Context context) {
        this(context, null, 0, null, RELATION_LEFT, ItemType.Item_Text_Image,0);
    }

    public BaseItemImpl(Context context, String text) {
        this(context, text, 0, null, RELATION_LEFT, ItemType.Item_Text,0);
    }

    public BaseItemImpl(Context context, int imgRes) {
        this(context, null, imgRes, null, RELATION_LEFT, ItemType.Item_Image,0);
    }
    public BaseItemImpl(Context context, int imgRes, int color) {
        this(context, null, imgRes, null, RELATION_LEFT, ItemType.Item_Image, color);
    }
    public BaseItemImpl(Context context, View customView) {
        this(context, null, 0, customView, RELATION_LEFT, ItemType.Item_custom,0);
    }

    protected BaseItemImpl(Context context, String text, int imgRes, int relation) {
        this(context, text, imgRes, null, relation, ItemType.Item_Text_Image,0);
    }

    private BaseItemImpl(Context context, String text, int imgRes, View customView, int relation, ItemType type, int color) {
        initDimens(context);
        mContext = context;
        mRelation = relation;
        mItemType = type;
        mColor = color;
        mRootLayout = new LinearLayout(context) {
            @Override
            protected void onLayout(boolean changed, int l, int t, int r, int b) {
                super.onLayout(changed, l, t, r, b);
                onItemLayout(l, t, r, b);
            }
        };
        mRootLayout.setGravity(Gravity.CENTER);
        if (ItemType.Item_Image.equals(type) || ItemType.Item_Text_Image.equals(type)) {
            if(mColor == 0) {
                mImage = new UIBtnImageView(context, mRootLayout);
            }else{
                mImage = new UIBtnImageView(context, mColor, mRootLayout);
            }
            mImgParams = new LinearLayout.LayoutParams(LinearLayout.LayoutParams.WRAP_CONTENT, LinearLayout.LayoutParams.WRAP_CONTENT);
            mImgParams.gravity = Gravity.CENTER;
            if (imgRes != 0) {
                mIconId = imgRes;
                mImage.setImageResource(imgRes);
            }
        }

        if (ItemType.Item_Text.equals(type) || ItemType.Item_Text_Image.equals(type)) {
            mTextView = new TextView(context);
            mTextParams = new LinearLayout.LayoutParams(LinearLayout.LayoutParams.WRAP_CONTENT, LinearLayout.LayoutParams.WRAP_CONTENT);
            mTextParams.gravity = Gravity.CENTER;
            if (text != null) {
                mTextView.setText(text);
            }
        }

        if (customView != null) {
            mCustomView = customView;
            mRootLayout.setOrientation(LinearLayout.VERTICAL);
            mRootLayout.addView(customView, new LinearLayout.LayoutParams(ViewGroup.LayoutParams.WRAP_CONTENT, ViewGroup.LayoutParams.WRAP_CONTENT));
        } else {
            setTextImgRelation(relation);
        }
        if (mTextView != null) {//defined textSize and color
            mTextSize = mDefinedTextSize;
            setTextSize(mDefinedTextSize);
            setTextColor(AppResource.getColor(mContext,R.color.tb_item_text_color_selector));

        }
    }

    @Override
    public View getContentView() {
        return mRootLayout;
    }

    @Override
    public TextView getTextView() {
        return mTextView;
    }

    @Override
    public void setText(CharSequence text) {
        if (mTextView != null) {
            mTextView.setText(text);
        }
    }

    @Override
    public void setText(int res) {
        if (mTextView != null) {
            mTextView.setText(res);
        }
    }

    @Override
    public String getText() {
        if (mTextView != null) {
            if (mTextView.getText() != null) {
                return mTextView.getText().toString();
            }
        }
        return null;
    }

    @Override
    public void setTextColor(int selectedTextColor, int disSelectedTextColor) {
        mUseTextColorRes = false;
        mSelectTextColor = selectedTextColor;
        mNoSelectTextColor = disSelectedTextColor;
        if (mTextView != null) {
            mTextView.setTextColor(mNoSelectTextColor);
        }
    }

    @Override
    public void setTextColor(ColorStateList colors) {
        if (mTextView != null) {
            mUseTextColorRes = true;
            mTextColorStateList = colors;
            mTextView.setTextColor(colors);
        }
    }

    @Override
    public void setTextColor(int color) {
        setTextColor(color, color);
    }

    @Override
    public void setTypeface(Typeface typeface) {
        if (mTextView != null) {
            mTextView.setTypeface(typeface);
        }
    }

    @Override
    public void setTextSize(float size) {
        if (mTextView != null) {
            mTextSize = size;
            mTextView.setTextSize(size);
        }
    }

    @Override
    public void setTextSize(int unit, float size) {
        if (mTextView != null) {
            mTextSizeUnit = unit;
            mTextSize = size;
            mTextView.setTextSize(unit, size);
        }
    }

    @Override
    public void setTextColorResource(int res) {
        if (mTextView != null) {
            Resources resource = mTextView.getContext().getResources();
            ColorStateList csl = resource.getColorStateList(res);
            mTextColorStateList = csl;
            mTextView.setTextColor(csl);
            mUseTextColorRes = true;
        }
    }
    @Override
    public void setTextBackground(Drawable drawable) {
        if (mTextView != null) {
            mTextView.setBackground(drawable);
        }
    }
    @Override
    public boolean setImageResource(int res) {
        if (mImage != null) {
            mIconId = res;
            mImage.setImageResource(res);
            return true;
        } else {
            return false;
        }
    }

    @Override
    public void setHighlightImageResource(int res) {
        this.mHighlightImageRes = res;
    }

    @Override
    public int getHighlightImageResource() {
        return mHighlightImageRes;
    }

    @Override
    public boolean setImageDrawable(@Nullable Drawable drawable) {
        if (mImage != null) {
            mImage.setImageDrawable(drawable);
            return true;
        } else {
            return false;
        }
    }

    @Override
    public Drawable getImageDrawable() {
        if (mImage != null)
            return mImage.getDrawable();
        return null;
    }

    @Override
    public void setImageTintList(ColorStateList tint) {
        if (mImage != null) {
            mImageColorStateList = tint;
            mImage.setColorStateList(tint);
        }
    }

    @Override
    public void setForceDarkAllowed(boolean allowed) {
        if (mRootLayout != null) {
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.Q) {
                mRootLayout.setForceDarkAllowed(allowed);
            }
        }
    }

    @Override
    public void setContentView(View view) {
        if (view != null && mRootLayout != null) {
            mRootLayout.removeAllViews();
            mRootLayout.addView(view);
        }
    }

    @Override
    public void setBackgroundResource(int res) {
        mRootLayout.setBackgroundResource(res);
    }

    @Override
    public void setBackgroundColor(int color) {
        if (mItemType == ItemType.Item_custom) {
            if (mCustomView != null && mCustomView.getBackground() != null) {
                mCustomView.getBackground().setColorFilter(color, PorterDuff.Mode.SRC_IN);
            }
        } else if (mItemType == ItemType.Item_Text){
            if (mTextView != null && mTextView.getBackground() != null) {
                mTextView.getBackground().setColorFilter(color, PorterDuff.Mode.SRC_IN);
            }
        } else if (mItemType == ItemType.Item_Image){
            if (mImage != null) {
                mImage.invalidate(color);
            }
        } else if (mItemType == ItemType.Item_Text_Image){
            if (mTextView != null && mTextView.getBackground() != null) {
                mTextView.getBackground().setColorFilter(color, PorterDuff.Mode.SRC_IN);
            }

            if (mImage != null) {
                mImage.invalidate(color);
            }
        }else {
            if (mRootLayout != null && mRootLayout.getBackground() != null) {
                mRootLayout.getBackground().setColorFilter(color, PorterDuff.Mode.SRC_IN);
            }
        }
    }

    @Override
    public void setImageTextBackgroundResouce(int res) {
        if (mImage != null) {
            mImage.setBackgroundResource(res);
            return;
        }
        if (mTextView != null) {
            mTextView.setBackgroundResource(res);
            return;
        }

        View customView = mRootLayout.getChildAt(0);
        if (customView instanceof ImageView) {
            customView.setBackgroundResource(res);
        }
    }

    @Override
    public void setImagePadding(int l, int t, int r, int b) {
        if (mImage != null)
            mImage.setPadding(l, t, r, b);
    }

    @Override
    public void setTextLayoutParams(int width, int height) {
        if (mTextView != null) {
            mTextView.getLayoutParams().width = width;
            mTextView.getLayoutParams().height = height;
            mTextView.setGravity(Gravity.CENTER_VERTICAL);
        }
    }

    @Override
    public void setTextPadding(int l, int t, int r, int b) {
        if (mTextView != null)
            mTextView.setPadding(l, t, r, b);
    }

    @Override
    public void setFilters(InputFilter[] filters) {
        if (mTextView != null) {
            mTextView.setFilters(filters);
            mTextView.setText(mTextView.getText());
        }
    }

    @Override
    public void setEllipsize(TextUtils.TruncateAt where) {
        if (mTextView != null) {
            mTextView.setLines(1);
            mTextView.setEllipsize(where);
        }
    }

    @Override
    public void setMaxWidth(int maxWidth) {
        if (mTextView != null) {
            mTextView.setMaxWidth(maxWidth);
        }
    }

    @Override
    public void setRelation(int relation) {
        if (mRelation == relation) {
            return;
        }
        if (mItemType != ItemType.Item_Text_Image) {
            return;
        }
        mRelation = relation;
        mRootLayout.removeAllViews();
        setTextImgRelation(relation);
    }

    private void setTextImgRelation(int relation) {
        if (relation == RELATION_LEFT || relation == RELATION_RIGNT) {
            mRootLayout.setOrientation(LinearLayout.HORIZONTAL);
        } else {
            mRootLayout.setOrientation(LinearLayout.VERTICAL);
        }
        if (relation == RELATION_LEFT || relation == RELATION_TOP) {
            if (mTextView != null) {
                if (mTextParams != null) {
                    mRootLayout.addView(mTextView, mTextParams);
                } else {
                    mRootLayout.addView(mTextView);
                }
            }
            if (mImage != null) {
                if (mImgParams != null) {
                    mRootLayout.addView(mImage, mImgParams);
                } else {
                    mRootLayout.addView(mImage);
                }
            }
        } else {
            if (mImage != null) {
                if (mImgParams != null) {
                    mRootLayout.addView(mImage, mImgParams);
                } else {
                    mRootLayout.addView(mImage);
                }
            }
            if (mTextView != null) {
                if (mTextParams != null) {
                    mRootLayout.addView(mTextView, mTextParams);
                } else {
                    mRootLayout.addView(mTextView);
                }
            }
        }
    }

    @Override
    public void setEnable(boolean enabled) {
        mIsEnabled = enabled;
        mRootLayout.setEnabled(enabled);
        if (mTextView != null) {
            mTextView.setEnabled(enabled);
        }
        if (mImage != null) {
            mImage.setEnabled(enabled);
        }
    }

    @Override
    public void setClickable(boolean clickable) {
        mRootLayout.setClickable(clickable);
        if (mTextView != null) {
            mTextView.setClickable(clickable);
        }
        if (mImage != null) {
            mImage.setClickable(clickable);
        }
    }

    @Override
    public boolean isEnable() {
        return mIsEnabled;
    }

    @Override
    public void setSelected(boolean selected) {
        mIsSelected = selected;
        mRootLayout.setSelected(selected);
        if (mTextView != null) {
            mTextView.setSelected(selected);
            if (!mUseTextColorRes) {
                if (selected) {
                    mTextView.setTextColor(mSelectTextColor);
                } else {
                    mTextView.setTextColor(mNoSelectTextColor);
                }
            }
        }
        if (mImage != null) {
            mImage.setSelected(selected);
        }
    }

    @Override
    public boolean isSelected() {
        return mIsSelected;
    }

    @Override
    public void setOnClickListener(View.OnClickListener l) {
        mClickListener = l;
        mRootLayout.setOnClickListener(mClickListenerImp);
        if (mImage != null) {
            mImage.setOnClickListener(mClickListenerImp);
        }
    }

    @Override
    public void setOnLongClickListener(View.OnLongClickListener l) {
        mLongClickListener = l;
        mRootLayout.setOnLongClickListener(l);
    }

    @Override
    public void setOnItemClickListener(OnItemClickListener l) {
        mItemClickListener = l;
        mRootLayout.setOnClickListener(mClickListenerImp);
        if (mImage != null) {
            mImage.setOnClickListener(mClickListenerImp);
        }
    }

    @Override
    public void performClick() {
        if (mClickListener != null) {
            mClickListener.onClick(mRootLayout);
        } else if (mItemClickListener != null) {
            mItemClickListener.onClick(BaseItemImpl.this, mRootLayout);
        }
    }

    @Override
    public void performLongClick() {
        if (mLongClickListener != null) {
            mLongClickListener.onLongClick(mRootLayout);
        } else if (mItemLongClickListener != null) {
            mItemLongClickListener.onLongPress(BaseItemImpl.this, mRootLayout);
        }
    }

    @Override
    public void setOnItemLongPressListener(OnItemLongPressListener l) {
        mItemLongClickListener = l;
        mRootLayout.setOnLongClickListener(mLongClickListenerImp);
        if (mImage != null) {
            mImage.setOnLongClickListener(mLongClickListenerImp);
        }
    }

    @Override
    public void setChecked(boolean checked) {
        if (mImage != null) {
            mImage.setChecked(checked);
        }
    }

    @Override
    public boolean isChecked() {
        if (mImage != null)
            return mImage.isChecked();
        return false;
    }

    protected View.OnClickListener mClickListenerImp = new View.OnClickListener() {
        @Override
        public void onClick(View v) {
            if (mClickListener != null) {
                if (v == mImage)
                    mClickListener.onClick(mRootLayout);
                else
                    mClickListener.onClick(v);
            } else if (mItemClickListener != null) {
                mItemClickListener.onClick(BaseItemImpl.this, v);
            }
        }
    };

    protected View.OnLongClickListener mLongClickListenerImp = new View.OnLongClickListener() {
        @Override
        public boolean onLongClick(View v) {
            if (mLongClickListener != null) {
                return mLongClickListener.onLongClick(v);
            } else if (mItemLongClickListener != null) {
                return mItemLongClickListener.onLongPress(BaseItemImpl.this, v);
            }
            return false;
        }
    };

    @Override
    public void setTag(int tag) {
        mTag = tag;
    }

    @Override
    public int getTag() {
        return mTag;
    }

    @Override
    public void setId(int id) {
        mRootLayout.setId(id);
    }

    @Override
    public int getId() {
        if (mRootLayout != null) {
            return mRootLayout.getId();
        } else {
            return 0;
        }
    }

    @Override
    public void setInterval(int interval) {
        if (mTextView == null || mTextView.getLayoutParams() == null) {
            return;
        }
        if (!(mTextView.getLayoutParams() instanceof LinearLayout.LayoutParams)) {
            return;
        }
        ((LinearLayout.LayoutParams) mTextView.getLayoutParams()).setMargins(0, 0, 0, 0);
        if (mRelation == RELATION_LEFT) {
            ((LinearLayout.LayoutParams) mTextView.getLayoutParams()).rightMargin = interval;
        } else if (mRelation == RELATION_RIGNT) {
            ((LinearLayout.LayoutParams) mTextView.getLayoutParams()).leftMargin = interval;
        } else if (mRelation == RELATION_TOP) {
            ((LinearLayout.LayoutParams) mTextView.getLayoutParams()).bottomMargin = interval;
        } else {
            ((LinearLayout.LayoutParams) mTextView.getLayoutParams()).topMargin = interval;
        }
    }

    @Override
    public void setDisplayStyle(ItemType type) {
        if (ItemType.Item_Image.equals(type)) {
            if (mImage != null) {
                mImage.setVisibility(View.VISIBLE);
            }
            if (mTextView != null) {
                mTextView.setVisibility(View.GONE);
            }
        } else if (ItemType.Item_Text.equals(type)) {
            if (mImage != null) {
                mImage.setVisibility(View.GONE);
            }
            if (mTextView != null) {
                mTextView.setVisibility(View.VISIBLE);
            }
        } else if (ItemType.Item_Text_Image.equals(type)) {
            if (mImage != null) {
                mImage.setVisibility(View.VISIBLE);
            }
            if (mTextView != null) {
                mTextView.setVisibility(View.VISIBLE);
            }
        } else {
            if (mImage != null) {
                mImage.setVisibility(View.GONE);
            }
            if (mTextView != null) {
                mTextView.setVisibility(View.GONE);
            }
        }
    }

    @Override
    public void onItemLayout(int l, int t, int r, int b) {
        if (mItemLayoutListener != null) {
            mItemLayoutListener.onItemLayout(l, t, r, b);
        }
    }

    private IItemLayoutListener mItemLayoutListener;

    @Override
    public void setItemLayoutListener(IItemLayoutListener listener) {
        this.mItemLayoutListener = listener;
    }

    private void initDimens(Context context) {
        mDefinedTextSize = (int) context.getResources().getDimension(R.dimen.ux_text_height_toolbar);
        mDefinedTextSize = (int) AppDisplay.px2dp(mDefinedTextSize);
    }

    public void setImageContentDescription(String des) {
        if (mImage != null) {
            mImage.setContentDescription(des);
        }
    }

    @Override
    public IBaseItem cloneItem() {
        String text = null;
        if (ItemType.Item_Text.equals(mItemType) || ItemType.Item_Text_Image.equals(mItemType)) {
            text = mTextView.getText().toString();
        }

        IBaseItem baseItem = new BaseItemImpl(mContext, text, mIconId, mCustomView, mRelation, mItemType, 0);
        baseItem.setSelected(mIsSelected);
        baseItem.setChecked(isChecked());
        baseItem.setEnable(mIsEnabled);
        baseItem.setId(mRootLayout.getId());
        baseItem.setTag(mTag);
        baseItem.setHighlightImageResource(mHighlightImageRes);
        baseItem.setOnClickListener(mClickListener);
        baseItem.setOnItemClickListener(mItemClickListener);
        baseItem.setOnLongClickListener(mLongClickListener);
        baseItem.setOnItemLongPressListener(mItemLongClickListener);
        updateThemeColor(baseItem);
        return baseItem;
    }

    private void updateThemeColor(IBaseItem baseItem) {
        if (mTextView != null) {
            baseItem.setTypeface(mTextView.getTypeface());
            if (mUseTextColorRes) {
                baseItem.setTextColor(mTextView.getTextColors());
            } else {
                if (mIsSelected) {
                    baseItem.setTextColor(mSelectTextColor);
                } else {
                    baseItem.setTextColor(mNoSelectTextColor);
                }
            }
            if (mTextSizeUnit != -1)
                baseItem.setTextSize(mTextSizeUnit, mTextSize);
            else
                baseItem.setTextSize(mTextSize);
        }
        if (mImage != null) {
            if (mImageColorStateList != null)
                baseItem.setImageTintList(mImageColorStateList);
        }
    }

    public void updateThemeColor() {
        updateThemeColor(this);
    }


    public void invalidate(int color){
        mImage.invalidate(color);
    }

    protected DynamicColorProvider backgroundColorProvider;

    public void setBackgroundColorProvider(DynamicColorProvider provider) {
        this.backgroundColorProvider = provider;
        updateBackgroundColor();
    }

    public void setBackgroundColor(@ColorInt int lightColor, @ColorInt int darkColor){
        backgroundColorProvider = new ColorPair(lightColor, darkColor);
        updateBackgroundColor();
    }

    public void updateBackgroundColor() {
        if (backgroundColorProvider != null) {
            this.setBackgroundColor(backgroundColorProvider.getColor(AppUtil.isDarkMode(mContext)));
        }
    }
}

