/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.data;


import android.util.SparseArray;
import android.util.SparseBooleanArray;

import com.foxit.uiextensions.utils.AppUtil;

import java.util.ArrayList;
import java.util.Locale;

/**
 * Suggestions helper class, you can use this to save your search history when searching or filling out forms.
 */
public class SuggestionsHelper {
    private static SuggestionsHelper instance;
    private final SparseArray<ArrayList<String>> mSuggestionsArray;
    private final SparseBooleanArray mShowSuggestionsArray;

    private SuggestionsHelper() {
        mSuggestionsArray = new SparseArray<>();
        mShowSuggestionsArray = new SparseBooleanArray();
    }

    public static SuggestionsHelper getInstance() {
        if (instance == null)
            instance = new SuggestionsHelper();
        return instance;
    }

    /**
     * @param type Suggestions type{@link ISuggestionsType}
     * @return Get Suggestions by suggestions type {@link ISuggestionsType}
     */
    public ArrayList<String> getSuggestions(@ISuggestionsType.SuggestionsType int type) {
        return mSuggestionsArray.get(type);
    }

    /**
     * @param type     Suggestions type{@link ISuggestionsType}
     * @param keyWords Suggestions keywords
     * @return Get suggestions by suggestions type{@link ISuggestionsType} and keywords.
     */
    public ArrayList<String> getSuggestions(@ISuggestionsType.SuggestionsType int type, String keyWords) {
        return getSuggestions(type, keyWords, Integer.MAX_VALUE);
    }

    /**
     * @param type     Suggestions type{@link ISuggestionsType}
     * @param keyWords Suggestions keywords
     * @param maxCount the maximum suggestions count returned by the type and keywords.
     * @return Get suggestions by suggestions type{@link ISuggestionsType} and keywords.
     */
    public ArrayList<String> getSuggestions(@ISuggestionsType.SuggestionsType int type, String keyWords, int maxCount) {
        ArrayList<String> results = null;
        if (!AppUtil.isEmpty(keyWords)) {
            ArrayList<String> suggestions = mSuggestionsArray.get(type);
            if (suggestions != null) {
                keyWords = keyWords.toLowerCase();
                for (String str : suggestions) {
                    if (str.toLowerCase(Locale.ROOT).startsWith(keyWords)) {
                        if (results == null)
                            results = new ArrayList<>();
                        results.add(str);

                        if (results.size() >= maxCount) {
                            break;
                        }
                    }
                }
            }
        }
        return results;
    }

    /**
     * @param type        Suggestions type{@link ISuggestionsType}
     * @param searchFlags Search flags. Please refer to values starting from {@link ISuggestionsSearchType#e_SearchNormal} and this can be one or combination of these values.
     * @param keyWords    Suggestions keywords
     * @param maxCount    the maximum suggestions count returned by the type and keywords.
     * @return Get suggestions by suggestions type{@link ISuggestionsType} and keywords.
     */
    public ArrayList<String> getSuggestions(@ISuggestionsType.SuggestionsType int type, int searchFlags,
                                            String keyWords, int maxCount) {
        ArrayList<String> results = null;
        if (!AppUtil.isEmpty(keyWords)) {
            ArrayList<String> suggestions = mSuggestionsArray.get(type);
            if (suggestions != null) {
                boolean case_sensitive = (searchFlags & ISuggestionsSearchType.e_SearchMatchCase) == ISuggestionsSearchType.e_SearchMatchCase;
                boolean consecutive = (searchFlags & ISuggestionsSearchType.e_SearchConsecutive) == ISuggestionsSearchType.e_SearchConsecutive;
                if (case_sensitive && consecutive) {
                    for (String str : suggestions) {
                        if (str.startsWith(keyWords)) {
                            if (results == null)
                                results = new ArrayList<>();
                            results.add(str);

                            if (results.size() >= maxCount) {
                                break;
                            }
                        }
                    }
                } else if (case_sensitive) {
                    for (String str : suggestions) {
                        if (str.contains(keyWords)) {
                            if (results == null)
                                results = new ArrayList<>();
                            results.add(str);

                            if (results.size() >= maxCount) {
                                break;
                            }
                        }
                    }
                } else if (consecutive) {
                    keyWords = keyWords.toLowerCase();

                    for (String str : suggestions) {
                        if (str.toLowerCase(Locale.ROOT).startsWith(keyWords)) {
                            if (results == null)
                                results = new ArrayList<>();
                            results.add(str);

                            if (results.size() >= maxCount) {
                                break;
                            }
                        }
                    }
                } else {
                    keyWords = keyWords.toLowerCase();

                    for (String str : suggestions) {
                        if (str.toLowerCase().contains(keyWords)) {
                            if (results == null)
                                results = new ArrayList<>();
                            results.add(str);

                            if (results.size() >= maxCount) {
                                break;
                            }
                        }
                    }
                }
            }
        }
        return results;
    }

    /**
     * Set suggestions by the type{@link ISuggestionsType}.
     *
     * @param type        Suggestions type{@link ISuggestionsType}
     * @param suggestions suggestions array.
     */
    public void setSuggestions(@ISuggestionsType.SuggestionsType int type, ArrayList<String> suggestions) {
        mSuggestionsArray.put(type, suggestions);
    }

    /**
     * Add suggesition by the type{@link ISuggestionsType}.
     *
     * @param type       Suggestions type{@link ISuggestionsType}
     * @param suggestion A suggestion to add.
     */
    public void addSuggestion(@ISuggestionsType.SuggestionsType int type, String suggestion) {
        ArrayList<String> suggestions = mSuggestionsArray.get(type);
        if (suggestions == null) {
            suggestions = new ArrayList<>();
            suggestions.add(suggestion);
            mSuggestionsArray.put(type, suggestions);
        } else {
            for (String str : suggestions) {
                if (str.equals(suggestion)) {
                    suggestions.remove(str);
                    break;
                }
            }
            suggestions.add(0, suggestion);
        }
    }

    /**
     * Update the position of specific suggestion in the history suggestions.
     *
     * @param type       Suggestions type{@link ISuggestionsType}
     * @param index      Updated location
     * @param suggestion A suggestion to update.
     */
    public void updateSuggestion(@ISuggestionsType.SuggestionsType int type, int index, String suggestion) {
        ArrayList<String> suggestions = mSuggestionsArray.get(type);
        if (suggestions == null) {
            suggestions = new ArrayList<>();
            suggestions.add(suggestion);
            mSuggestionsArray.put(type, suggestions);
        } else {
            for (String str : suggestions) {
                if (str.equals(suggestion)) {
                    suggestions.remove(str);
                    break;
                }
            }
            suggestions.add(index, suggestion);
        }
    }

    /**
     * Remove a suggestion by the suggestions type{@link ISuggestionsType}.
     *
     * @param type       Suggestions type{@link ISuggestionsType}
     * @param suggestion A suggestion to remove.
     */
    public void removeSuggestion(@ISuggestionsType.SuggestionsType int type, String suggestion) {
        ArrayList<String> suggestions = mSuggestionsArray.get(type);
        if (suggestions != null) {
            suggestions.remove(suggestion);
        }
    }

    /**
     * Clear suggesions by the suggestions type{@link ISuggestionsType}.
     *
     * @param type Suggestions type{@link ISuggestionsType}
     */
    public void clearSuggestions(@ISuggestionsType.SuggestionsType int type) {
        ArrayList<String> suggestions = mSuggestionsArray.get(type);
        if (suggestions != null) {
            suggestions.clear();
        }
    }

    /**
     * Remove suggesions by the suggestions type{@link ISuggestionsType}.
     *
     * @param type Suggestions type{@link ISuggestionsType}
     */
    public void removeSuggestions(@ISuggestionsType.SuggestionsType int type) {
        ArrayList<String> suggestions = mSuggestionsArray.get(type);
        if (suggestions != null) {
            suggestions.clear();
            mSuggestionsArray.remove(type);
        }
    }

    /**
     * Set whether to display suggestions according to type{@link ISuggestionsType}.
     *
     * @param type Suggestions type{@link ISuggestionsType}
     * @param show True means show suggestions tool bar, false means do not display.
     */
    public void setShowSuggestions(@ISuggestionsType.SuggestionsType int type, boolean show) {
        mShowSuggestionsArray.put(type, show);
    }

    /**
     * Get whether to display suggestions according to type{@link ISuggestionsType}, the default is to display suggestions.
     *
     * @param type  Suggestions type{@link ISuggestionsType}
     * @return True means show suggestions tool bar, otherwise return false.
     */
    public boolean isShowSuggestions(@ISuggestionsType.SuggestionsType int type) {
        return mShowSuggestionsArray.get(type, true);
    }

}
