/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.modules;

import android.app.Activity;
import android.content.Context;
import android.graphics.Typeface;
import android.view.KeyEvent;
import android.view.View;
import android.view.ViewGroup;
import android.widget.RelativeLayout;

import com.foxit.sdk.PDFViewCtrl;
import com.foxit.sdk.common.Constants;
import com.foxit.sdk.pdf.PDFDoc;
import com.foxit.sdk.pdf.ReflowPage;
import com.foxit.uiextensions.IInteractionEventListener;
import com.foxit.uiextensions.IThemeEventListener;
import com.foxit.uiextensions.Module;
import com.foxit.uiextensions.R;
import com.foxit.uiextensions.UIExtensionsManager;
import com.foxit.uiextensions.config.modules.ModulesConfig;
import com.foxit.uiextensions.controls.panel.PanelHost;
import com.foxit.uiextensions.controls.panel.PanelSpec;
import com.foxit.uiextensions.controls.propertybar.IViewSettingsWindow;
import com.foxit.uiextensions.controls.toolbar.BaseBar;
import com.foxit.uiextensions.controls.toolbar.IBaseItem;
import com.foxit.uiextensions.controls.toolbar.ToolbarItemConfig;
import com.foxit.uiextensions.controls.toolbar.drag.UIDragView;
import com.foxit.uiextensions.controls.toolbar.impl.BaseBarImpl;
import com.foxit.uiextensions.controls.toolbar.impl.BaseItemImpl;
import com.foxit.uiextensions.controls.toolbar.impl.BottomBarImpl;
import com.foxit.uiextensions.controls.toolbar.impl.TopBarImpl;
import com.foxit.uiextensions.event.DocEventListener;
import com.foxit.uiextensions.event.PageEventListener;
import com.foxit.uiextensions.modules.crop.CropModule;
import com.foxit.uiextensions.modules.pagenavigation.PageNavigationModule;
import com.foxit.uiextensions.modules.panel.IPanelManager;
import com.foxit.uiextensions.modules.panel.annot.AnnotPanelModule;
import com.foxit.uiextensions.modules.panel.bookmark.ReadingBookmarkModule;
import com.foxit.uiextensions.modules.panel.filespec.FileSpecPanelModule;
import com.foxit.uiextensions.modules.panel.signature.SignaturePanelModule;
import com.foxit.uiextensions.pdfreader.ILifecycleEventListener;
import com.foxit.uiextensions.pdfreader.IMainFrame;
import com.foxit.uiextensions.pdfreader.IStateChangeListener;
import com.foxit.uiextensions.pdfreader.config.ReadStateConfig;
import com.foxit.uiextensions.pdfreader.impl.LifecycleEventListener;
import com.foxit.uiextensions.pdfreader.impl.MainFrame;
import com.foxit.uiextensions.theme.ThemeConfig;
import com.foxit.uiextensions.theme.ThemeUtil;
import com.foxit.uiextensions.utils.AppDevice;
import com.foxit.uiextensions.utils.AppDisplay;
import com.foxit.uiextensions.utils.AppResource;
import com.foxit.uiextensions.utils.AppUtil;
import com.foxit.uiextensions.utils.thread.AppThreadManager;

/**
 * Reflow the PDF page , so it will fit to the small screen such like cellphone.It is more convenient to read.
 */
public class ReflowModule implements Module {
    private Context mContext;
    private ViewGroup mParent;
    private PDFViewCtrl mPdfViewCtrl;

    private boolean mIsReflow;
    private IViewSettingsWindow mSettingWindow;
    private int mPreViewMode;
    private int mPreReflowMode;
    private int mPreReadState;
    private boolean mIsPreViewInCropMode;

    private BaseBar mReflowTopBar;
    private BaseBar mReflowBottomBar;
    private IBaseItem mBackItem;
    private IBaseItem mTitleItem;
    private IBaseItem mBookMarkItem;
    private IBaseItem mPicItem;
    private IBaseItem mZoomOutItem;//out(-)
    private IBaseItem mZoomInItem;//in(+)
    private IBaseItem mPrePageItem;
    private IBaseItem mNextPageItem;
    private IBaseItem mQuitPageItem;
    private IBaseItem mListItem;
    private float mLastScale = 1.0f;
    private float mScale = 1.0f;
    private float mMaxScale = 8.0f;
    private float mMinSale = 1.0f;
    private int mLastZoomMode = -1;
    private int mLastReflowPanelType = -1;
    private int mLastNormalPanelType = -1;
    private ModulesConfig mModuleConfig;
    private PDFViewCtrl.UIExtensionsManager mUiExtensionsManager;

    public ReflowModule(Context context, ViewGroup parent, PDFViewCtrl pdfViewCtrl, PDFViewCtrl.UIExtensionsManager uiExtensionsManager) {
        mContext = context;
        mParent = parent;
        mPdfViewCtrl = pdfViewCtrl;
        mUiExtensionsManager = uiExtensionsManager;
    }

    private IBaseItem mMultiTabItem;
    public void registMultiTabItem(IBaseItem item) {
        if (item == null) return;

        mMultiTabItem = item;
        if (mReflowTopBar != null) {
            mReflowTopBar.removeItem(mMultiTabItem);
            mReflowTopBar.addView(mMultiTabItem, BaseBar.TB_Position.Position_LT);
        }
    }

    @Override
    public boolean loadModule() {
        if (mUiExtensionsManager != null && mUiExtensionsManager instanceof UIExtensionsManager) {
            UIExtensionsManager uiExtensionsManager = (UIExtensionsManager) mUiExtensionsManager;

            mModuleConfig = uiExtensionsManager.getConfig().modules;
            mSettingWindow = uiExtensionsManager.getMainFrame().getSettingWindow();
            uiExtensionsManager.registerStateChangeListener(mStatusChangeListener);
            uiExtensionsManager.registerModule(this);
            uiExtensionsManager.registerThemeEventListener(mThemeEventListener);
            uiExtensionsManager.registerInteractionListener(mInteractionEventListener);
            uiExtensionsManager.getPanelManager().registerPanelEventListener(mPanelEventListener);
            if (AppDisplay.isPad())
                ((UIExtensionsManager) mUiExtensionsManager).registerLifecycleListener(mLifecycleEventListener);
        }

        mPdfViewCtrl.registerDocEventListener(mDocEventListener);
        return true;
    }

    private void addBar() {
        if (!AppDisplay.isPad()) {
            removeBar();

            RelativeLayout.LayoutParams reflowTopLp = new RelativeLayout.LayoutParams(ViewGroup.LayoutParams.MATCH_PARENT, ViewGroup.LayoutParams.WRAP_CONTENT);
            reflowTopLp.addRule(RelativeLayout.ALIGN_PARENT_TOP);
            mParent.addView(mReflowTopBar.getContentView(), reflowTopLp);
            RelativeLayout.LayoutParams reflowBottomLp = new RelativeLayout.LayoutParams(ViewGroup.LayoutParams.MATCH_PARENT, ViewGroup.LayoutParams.WRAP_CONTENT);
            reflowBottomLp.addRule(RelativeLayout.ALIGN_PARENT_BOTTOM);
            reflowBottomLp.addRule(RelativeLayout.CENTER_HORIZONTAL);
            mParent.addView(mReflowBottomBar.getContentView(), reflowBottomLp);
            mReflowTopBar.getContentView().setVisibility(View.INVISIBLE);
            mReflowBottomBar.getContentView().setVisibility(View.INVISIBLE);
        }
    }

    private void removeBar() {
        if (AppDisplay.isPad()) {
            if (mDragView != null) {
                mDragView.removeDragView();
                mDragView = null;
            }
        } else {
            if (mReflowBottomBar != null)
                mParent.removeView(mReflowBottomBar.getContentView());
            if (mReflowTopBar != null)
                mParent.removeView(mReflowTopBar.getContentView());
        }
    }

    private final PDFViewCtrl.IPageEventListener mPageEventListener = new PageEventListener() {
        @Override
        public void onPageChanged(int oldPageIndex, int curPageIndex) {
            resetNextPageItem();
            resetPrePageItem();
        }
    };

    private PDFViewCtrl.IDocEventListener mDocEventListener = new DocEventListener() {
        @Override
        public void onDocWillOpen() {
        }

        @Override
        public void onDocOpened(PDFDoc document, int errCode) {
            if (errCode != Constants.e_ErrSuccess)
                return;

            initReflowBar();
            if (!mPdfViewCtrl.isDynamicXFA()) {
                addBar();
            }
            initValue();
            if (!initViewSettingValue()) return;
            applyValue();
            registerViewSettingListener();
            mPdfViewCtrl.registerPageEventListener(mPageEventListener);
            onStatusChanged();

            String pageMode = ((UIExtensionsManager) mUiExtensionsManager).getConfig().uiSettings.pageMode;
            if ("Reflow".equals(pageMode)) {
                if (AppDisplay.isPad()) {
                    IMainFrame mainFrame = ((UIExtensionsManager) mUiExtensionsManager).getMainFrame();
                    if (mainFrame.getCurrentTab() != ToolbarItemConfig.ITEM_VIEW_TAB)
                        mainFrame.setCurrentTab(ToolbarItemConfig.ITEM_VIEW_TAB);
                }
                setReflowPageMode(true);
                ((UIExtensionsManager) mUiExtensionsManager).getConfig().uiSettings.pageMode = "Single";
            }
        }

        @Override
        public void onDocWillClose(PDFDoc document) {
            if (((UIExtensionsManager) mUiExtensionsManager).getConfig().modules.isLoadReadingBookmark) {
                ReadingBookmarkModule module = (ReadingBookmarkModule) ((UIExtensionsManager) mUiExtensionsManager).getModuleByName(MODULE_NAME_BOOKMARK);
                if (module != null)
                    module.removeMarkedItem(mBookMarkItem);
            }
        }

        @Override
        public void onDocClosed(PDFDoc document, int errCode) {
            removeBar();
            mPdfViewCtrl.unregisterPageEventListener(mPageEventListener);

            unregisterViewSettingListener();
        }

    };

    private boolean isLoadPanel() {
        return (mModuleConfig == null) || mModuleConfig.isLoadReadingBookmark
                || mModuleConfig.isLoadOutline || mModuleConfig.isLoadAttachment;
    }

    private void initReflowBar() {
        boolean isPad = AppDisplay.isPad();
        UIExtensionsManager uiManager = (UIExtensionsManager) mUiExtensionsManager;

        if (isPad) {
            mReflowBottomBar = new BaseBarImpl(mContext);
            mReflowBottomBar.setInterceptTouch(false);

            if (AppDevice.isChromeOs(mPdfViewCtrl.getAttachedActivity()))
                mReflowBottomBar.setItemInterval(AppResource.getDimensionPixelSize(mContext, R.dimen.ux_bottombar_button_space_phone));
            else
                mReflowBottomBar.setItemInterval(AppResource.getDimensionPixelSize(mContext, R.dimen.ux_bottombar_button_space_pad));

            mQuitPageItem = new BaseItemImpl(mContext);
            mQuitPageItem.setForceDarkAllowed(false);
        } else {
            mReflowTopBar = new TopBarImpl(mContext);
            mReflowTopBar.setMiddleButtonCenter(true);
            mReflowTopBar.setStartMargin(AppResource.getDimensionPixelSize(mContext, R.dimen.ux_margin_16dp));
            mReflowTopBar.setBackgroundColor(ThemeConfig.getInstance(mContext).getB2());

            mBackItem = new BaseItemImpl(mContext);
            mTitleItem = new BaseItemImpl(mContext);
            mBookMarkItem = new BaseItemImpl(mContext, R.drawable.rd_bottom_bookmark_selector);
            mBookMarkItem.setForceDarkAllowed(false);
            mBookMarkItem.setImageTintList(ThemeUtil.getPrimaryIconColor(mContext));
            mBookMarkItem.setId(R.id.id_bottom_bar_bookmark);
            boolean canAssemble = uiManager.getDocumentManager().canAssemble() && uiManager.isEnableModification();
            mBookMarkItem.setEnable(canAssemble);
            mReflowTopBar.addView(mBackItem, BaseBar.TB_Position.Position_LT);
            mReflowTopBar.addView(mTitleItem, BaseBar.TB_Position.Position_CENTER);
            if (uiManager.getConfig().modules.isLoadReadingBookmark) {
                mReflowTopBar.addView(mBookMarkItem, BaseBar.TB_Position.Position_RB);
                ReadingBookmarkModule module = (ReadingBookmarkModule) uiManager.getModuleByName(MODULE_NAME_BOOKMARK);
                if (module != null)
                    module.addMarkedItem(mBookMarkItem);
                try {
                    module.remarkItemState(mPdfViewCtrl.getCurrentPage());
                }catch (Exception e){

                }
            }

            mReflowBottomBar = new BottomBarImpl(mContext);
            mReflowBottomBar.setItemInterval(AppResource.getDimensionPixelSize(mContext, R.dimen.ux_reflow_bottombar_button_space_phone));

            mListItem = new BaseItemImpl(mContext);
            mListItem.setForceDarkAllowed(false);
        }
        mReflowBottomBar.setBackgroundColor(ThemeConfig.getInstance(mContext).getB2());
        mReflowBottomBar.setAutoCompressItemsInterval(true);

        mPicItem = new BaseItemImpl(mContext);
        mPicItem.setForceDarkAllowed(false);
        mZoomOutItem = new BaseItemImpl(mContext);
        mZoomOutItem.setForceDarkAllowed(false);
        mZoomInItem = new BaseItemImpl(mContext);
        mZoomInItem.setForceDarkAllowed(false);
        mPrePageItem = new BaseItemImpl(mContext);
        mPrePageItem.setForceDarkAllowed(false);
        mNextPageItem = new BaseItemImpl(mContext);
        mNextPageItem.setForceDarkAllowed(false);

        initItemsImgRes();
        initItemsOnClickListener();

        mReflowBottomBar.addView(mZoomInItem, BaseBar.TB_Position.Position_CENTER);
        mReflowBottomBar.addView(mZoomOutItem, BaseBar.TB_Position.Position_CENTER);
        mReflowBottomBar.addView(mPicItem, BaseBar.TB_Position.Position_CENTER);
        mReflowBottomBar.addView(mPrePageItem, BaseBar.TB_Position.Position_CENTER);
        mReflowBottomBar.addView(mNextPageItem, BaseBar.TB_Position.Position_CENTER);
        if (isPad) {
            mReflowBottomBar.addView(mQuitPageItem, BaseBar.TB_Position.Position_CENTER);
        } else {
            mModuleConfig = uiManager.getConfig().modules;
            if (mModuleConfig == null || isLoadPanel()) {
                mReflowBottomBar.addView(mListItem, BaseBar.TB_Position.Position_CENTER, 0);
            }
        }
        registMultiTabItem(mMultiTabItem);
    }

    private void initItemsImgRes() {
        boolean isPad = AppDisplay.isPad();
        if (AppUtil.isDarkMode(mContext)) {
            mPicItem.setImageResource(R.drawable.dark_reflow_no_image);
        } else {
            mPicItem.setImageResource(R.drawable.reflow_no_image);
        }
        mPicItem.setId(R.id.id_reflow_no_image);
        mZoomOutItem.setImageResource(R.drawable.reflow_zoom_out);
        mZoomOutItem.setId(R.id.id_reflow_zoom_out);
        mZoomInItem.setImageResource(R.drawable.reflow_zoom_in);
        mZoomInItem.setId(R.id.id_reflow_zoom_in);
        mPrePageItem.setImageResource(R.drawable.reflow_pre_page);
        mPrePageItem.setId(R.id.id_reflow_pre_page);
        mNextPageItem.setImageResource(R.drawable.reflow_next_page);
        mNextPageItem.setId(R.id.id_reflow_next_page);

        mZoomOutItem.setImageTintList(ThemeUtil.getEnableIconColor(mContext));
        mZoomInItem.setImageTintList(ThemeUtil.getEnableIconColor(mContext));
        mPrePageItem.setImageTintList(ThemeUtil.getEnableIconColor(mContext));
        mNextPageItem.setImageTintList(ThemeUtil.getEnableIconColor(mContext));
        if (isPad) {
            mQuitPageItem.setImageResource(R.drawable.ic_tool_bar_item_logout);
            mQuitPageItem.setId(R.id.id_reflow_tool_bar_item_logout);
            mQuitPageItem.setImageTintList(ThemeUtil.getPrimaryIconColor(mContext));
        } else {
            mListItem.setImageResource(R.drawable.rd_bottom_bar_pannel);
            mListItem.setId(R.id.id_reflow_bottom_bar_panel);
            mListItem.setImageTintList(ThemeUtil.getPrimaryIconColor(mContext));

            mBackItem.setText(R.string.fx_string_close);
            mBackItem.setTag(ToolbarItemConfig.ITEM_TOPBAR_BACK);
            mBackItem.setTextSize(AppDisplay.px2dp(mContext.getResources().getDimensionPixelOffset(R.dimen.ux_text_size_16sp)));
            mBackItem.setTextColor(ThemeConfig.getInstance(mContext).getPrimaryColor());

            mTitleItem.setText(AppResource.getString(mContext, R.string.rd_reflow_topbar_title));
            mTitleItem.setTag(ToolbarItemConfig.ITEM_TOPBAR_BACK + 1);
            mTitleItem.setTextSize(AppDisplay.px2dp(mContext.getResources().getDimensionPixelOffset(R.dimen.ux_text_size_16sp)));
            mTitleItem.setTextColorResource(R.color.t4);
            mTitleItem.setTypeface(Typeface.defaultFromStyle(Typeface.BOLD));


        }
    }

    private void resetPicItem() {
        if ((mPdfViewCtrl.getReflowMode() & ReflowPage.e_WithImage) == 1) {
            if (AppUtil.isDarkMode(mContext)) {
                mPicItem.setImageResource(R.drawable.dark_reflow_image);
            } else {
                mPicItem.setImageResource(R.drawable.reflow_image);
            }
        } else {
            if (AppUtil.isDarkMode(mContext)) {
                mPicItem.setImageResource(R.drawable.dark_reflow_no_image);
            } else {
                mPicItem.setImageResource(R.drawable.reflow_no_image);
            }
        }
    }

    private void resetZoomOutItem() {
        mZoomOutItem.setEnable(!isMinZoomScale());
    }

    private void resetZoomInItem() {
        mZoomInItem.setEnable(!isMaxZoomScale());
    }

    private void resetPrePageItem() {
        mPrePageItem.setEnable(mPdfViewCtrl.getCurrentPage() != 0);
    }

    private void resetNextPageItem() {
        mNextPageItem.setEnable(mPdfViewCtrl.getCurrentPage() + 1 < mPdfViewCtrl.getPageCount());
    }

    private void initItemsOnClickListener() {
        mPicItem.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                ((UIExtensionsManager) mUiExtensionsManager).resetHideToolbarsTimer();
                if ((mPdfViewCtrl.getReflowMode() & ReflowPage.e_WithImage) == 1) {
                    mPdfViewCtrl.setReflowMode(ReflowPage.e_Normal);
                    if (AppUtil.isDarkMode(mContext)) {
                        mPicItem.setImageResource(R.drawable.dark_reflow_no_image);
                    } else {
                        mPicItem.setImageResource(R.drawable.reflow_no_image);
                    }
                } else {
                    mPdfViewCtrl.setReflowMode(ReflowPage.e_WithImage | ReflowPage.e_Normal);
                    if (AppUtil.isDarkMode(mContext)) {
                        mPicItem.setImageResource(R.drawable.dark_reflow_image);
                    } else {
                        mPicItem.setImageResource(R.drawable.reflow_image);
                    }
                }
            }
        });
        mZoomOutItem.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                if (mLastScale == mScale && AppDisplay.isPad()) {
                    if (mSettingWindow.isSelected(IViewSettingsWindow.TYPE_FIT_WIDTH))
                        mSettingWindow.setProperty(IViewSettingsWindow.TYPE_FIT_WIDTH, PDFViewCtrl.ZOOMMODE_CUSTOMIZE);
                    if (mSettingWindow.isSelected(IViewSettingsWindow.TYPE_FIT_PAGE))
                        mSettingWindow.setProperty(IViewSettingsWindow.TYPE_FIT_PAGE, PDFViewCtrl.ZOOMMODE_CUSTOMIZE);
                }

                ((UIExtensionsManager) mUiExtensionsManager).resetHideToolbarsTimer();
                if (isMinZoomScale()) {
                    mZoomOutItem.setEnable(false);
                } else {
                    mScale = Math.max(mMinSale, mScale * 0.8f);
                    mPdfViewCtrl.setZoom(mScale);
                    resetZoomInItem();
                    resetZoomOutItem();
                }

            }
        });
        mZoomInItem.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                if (mLastScale == mScale && AppDisplay.isPad()) {
                    if (mSettingWindow.isSelected(IViewSettingsWindow.TYPE_FIT_WIDTH))
                        mSettingWindow.setProperty(IViewSettingsWindow.TYPE_FIT_WIDTH, PDFViewCtrl.ZOOMMODE_CUSTOMIZE);
                    if (mSettingWindow.isSelected(IViewSettingsWindow.TYPE_FIT_PAGE))
                        mSettingWindow.setProperty(IViewSettingsWindow.TYPE_FIT_PAGE, PDFViewCtrl.ZOOMMODE_CUSTOMIZE);
                }

                ((UIExtensionsManager) mUiExtensionsManager).resetHideToolbarsTimer();
                if (isMaxZoomScale()) {
                    mZoomInItem.setEnable(false);
                } else {
                    mScale = Math.min(mMaxScale, mScale * 1.25f);
                    mPdfViewCtrl.setZoom(mScale);
                    resetZoomInItem();
                    resetZoomOutItem();
                }
            }
        });
        mPrePageItem.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                ((UIExtensionsManager) mUiExtensionsManager).resetHideToolbarsTimer();
                if ((mPdfViewCtrl.getCurrentPage() - 1) >= 0) {
                    mPdfViewCtrl.gotoPrevPage();
                }
            }
        });
        mNextPageItem.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                ((UIExtensionsManager) mUiExtensionsManager).resetHideToolbarsTimer();
                if ((mPdfViewCtrl.getCurrentPage() + 1) < mPdfViewCtrl.getPageCount()) {
                    mPdfViewCtrl.gotoNextPage();
                }
            }
        });
        if (!AppDisplay.isPad()) {
            mBackItem.setOnClickListener(new View.OnClickListener() {
                @Override
                public void onClick(View v) {
                    resetZoomMode();
                    if (mPreReadState == ReadStateConfig.STATE_FILLSIGN)
                        ((UIExtensionsManager) mUiExtensionsManager).changeState(ReadStateConfig.STATE_FILLSIGN);
                    else
                        ((UIExtensionsManager) mUiExtensionsManager).changeState(ReadStateConfig.STATE_NORMAL);
                }
            });
            mListItem.setOnClickListener(new View.OnClickListener() {
                @Override
                public void onClick(View v) {
                    ((UIExtensionsManager) mUiExtensionsManager).triggerDismissMenuEvent();
                    ((UIExtensionsManager) mUiExtensionsManager).getPanelManager().showPanel();
                }
            });
        } else {
            mQuitPageItem.setOnClickListener(new View.OnClickListener() {
                @Override
                public void onClick(View v) {
                    resetZoomMode();
                    ((UIExtensionsManager) mUiExtensionsManager).changeState(ReadStateConfig.STATE_NORMAL);
                }
            });
        }
    }

    private void resetZoomMode() {
        if (!mPdfViewCtrl.isKeepZoomRatio() && mLastZoomMode != mPdfViewCtrl.getZoomMode()
                || mLastScale == mScale) {
            if (mLastZoomMode == PDFViewCtrl.ZOOMMODE_FITWIDTH) {
                mPdfViewCtrl.setZoomMode(PDFViewCtrl.ZOOMMODE_FITWIDTH);
                mSettingWindow.setProperty(IViewSettingsWindow.TYPE_FIT_WIDTH, PDFViewCtrl.ZOOMMODE_FITWIDTH);
            } else if (mLastZoomMode == PDFViewCtrl.ZOOMMODE_FITPAGE) {
                mPdfViewCtrl.setZoomMode(PDFViewCtrl.ZOOMMODE_FITPAGE);
                mSettingWindow.setProperty(IViewSettingsWindow.TYPE_FIT_PAGE, PDFViewCtrl.ZOOMMODE_FITPAGE);
            }
        }
    }

    private void initValue() {
        if (mPdfViewCtrl.getPageLayoutMode() == PDFViewCtrl.PAGELAYOUTMODE_REFLOW
                || ((UIExtensionsManager) mUiExtensionsManager).getConfig().uiSettings.pageMode.equals("Reflow")) {
            mIsReflow = true;
        } else {
            mIsReflow = false;
        }
        mPreViewMode = mPdfViewCtrl.getPageLayoutMode();
        mPreReflowMode = mPdfViewCtrl.getReflowMode();
    }

    private boolean initViewSettingValue() {
        mSettingWindow = ((UIExtensionsManager) mUiExtensionsManager).getMainFrame().getSettingWindow();
        if (mPdfViewCtrl.isDynamicXFA()) {
            mSettingWindow.enableBar(IViewSettingsWindow.TYPE_REFLOW, false);
            return false;
        }

        mSettingWindow.enableBar(IViewSettingsWindow.TYPE_REFLOW, true);
        mSettingWindow.setProperty(IViewSettingsWindow.TYPE_REFLOW, mIsReflow);
        return true;
    }

    private void applyValue() {
        if (mIsReflow) {
            ((UIExtensionsManager) mUiExtensionsManager).changeState(ReadStateConfig.STATE_REFLOW);
        }
    }

    private boolean isMaxZoomScale() {
        return mScale >= mMaxScale;
    }

    private boolean isMinZoomScale() {
        return mScale <= mMinSale;
    }

    private int mIndexOfAnnotPanel = -1;
    private int mIndexOfSignaturePanel = -1;

    private boolean mNeedResetPanelView = false;

    private void resetPanelView(boolean isReflowMode) {
        UIExtensionsManager uiExtensionsManager = ((UIExtensionsManager) mUiExtensionsManager);
        IPanelManager panelManager = uiExtensionsManager.getPanelManager();
        PanelHost panelHost = panelManager.getPanelHost();
        if (panelHost == null || panelHost.getCurrentSpec() == null) {
            mNeedResetPanelView = true;
            return;
        }

        int curPanelType = panelHost.getCurrentSpec().getPanelType();
        if (isReflowMode) {
            mLastNormalPanelType = curPanelType;
        } else {
            mLastReflowPanelType = curPanelType;
        }

        AnnotPanelModule annotPanel = (AnnotPanelModule) uiExtensionsManager.getModuleByName(Module.MODULE_NAME_ANNOTPANEL);
        if (mModuleConfig.isLoadAnnotations && annotPanel != null) {
            if (isReflowMode) {
                mIndexOfAnnotPanel = panelManager.getPanelHost().indexOf(annotPanel);
                panelManager.removePanel(annotPanel);
            } else {
                if (mIndexOfAnnotPanel >= 0) {
                    panelManager.addPanel(mIndexOfAnnotPanel, annotPanel);
                    mIndexOfAnnotPanel = -1;
                } else {
                    panelManager.addPanel(annotPanel);
                }
            }
        }

        SignaturePanelModule signaturePanel = (SignaturePanelModule) uiExtensionsManager.getModuleByName(Module.MODULE_NAME_SIGNATUREPANEL);
        if (mModuleConfig.isLoadSignature && signaturePanel != null) {
            if (isReflowMode) {
                mIndexOfSignaturePanel = panelManager.getPanelHost().indexOf(signaturePanel);
                panelManager.removePanel(signaturePanel);
                if (mIndexOfAnnotPanel >= 0)
                    mIndexOfSignaturePanel++;
            } else {
                if (mIndexOfSignaturePanel >= 0) {
                    panelManager.addPanel(mIndexOfSignaturePanel, signaturePanel);
                    mIndexOfSignaturePanel = -1;
                } else {
                    panelManager.addPanel(signaturePanel);
                }
            }
        }

        int panelType;
        if (isReflowMode) {
            panelType = mLastReflowPanelType == -1 ? PanelSpec.BOOKMARKS : mLastReflowPanelType;
        } else {
            panelType = mLastNormalPanelType == -1 ? PanelSpec.BOOKMARKS : mLastNormalPanelType;
        }
        panelManager.getPanelHost().setCurrentSpec(panelType);
    }

    private void registerViewSettingListener() {
        mSettingWindow.registerListener(mReflowChangeListener);
    }

    private void unregisterViewSettingListener() {
        mSettingWindow.unRegisterListener(mReflowChangeListener);
    }

    private UIDragView mDragView;
    private final IViewSettingsWindow.IValueChangeListener mReflowChangeListener = new IViewSettingsWindow.IValueChangeListener() {
        @Override
        public void onValueChanged(int type, Object value) {
            if (!(value instanceof Boolean)) return;

            if (type == IViewSettingsWindow.TYPE_REFLOW) {
                if ((Boolean) value) {
                    mLastZoomMode = mPdfViewCtrl.getZoomMode();
                    mLastScale = mPdfViewCtrl.getZoom();

                    if (AppDisplay.isPad()) {
                        mReflowBottomBar.getContentView().setVisibility(View.VISIBLE);
                        mDragView = createDragToolView();
                    }
                }

                setReflowPageMode((Boolean) value);
            }
        }

        @Override
        public int getType() {
            return IViewSettingsWindow.TYPE_REFLOW;
        }
    };

    private UIDragView createDragToolView() {
        if (mDragView != null) return mDragView;

        int width = 0;
        if (AppDevice.isChromeOs(((UIExtensionsManager) mUiExtensionsManager).getAttachedActivity()))
            width = AppResource.getDimensionPixelSize(mContext, R.dimen.ux_bottombar_button_space_phone) * 5 +
                    AppResource.getDimensionPixelSize(mContext, R.dimen.ux_tool_icon_size) * 6 +
                    AppDisplay.dp2px(+20);
        else
            width = AppResource.getDimensionPixelSize(mContext, R.dimen.ux_bottombar_button_space_pad) * 5 +
                    AppResource.getDimensionPixelSize(mContext, R.dimen.ux_tool_icon_size) * 6 +
                    AppDisplay.dp2px(+20);
        return new UIDragView(mReflowBottomBar.getContentView(), (UIExtensionsManager) mUiExtensionsManager, width);
    }

    private void setReflowPageMode(boolean isReflow) {
        setReflowPageMode(isReflow, true);
    }

    private void setReflowPageMode(boolean isReflow, boolean changeState) {
        UIExtensionsManager uiExtensionsManager = (UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager();

        mIsReflow = isReflow;
        int curState = uiExtensionsManager.getState();
        int curLayout = mPdfViewCtrl.getPageLayoutMode();
        int curReflowMode = mPdfViewCtrl.getReflowMode();
        if (mIsReflow) {
            if (curLayout != PDFViewCtrl.PAGELAYOUTMODE_REFLOW) {
                CropModule cropModule = (CropModule) uiExtensionsManager.getModuleByName(Module.MODULE_NAME_CROP);
                if (cropModule != null && cropModule.isCropMode()) {
                    mIsPreViewInCropMode = true;
                    cropModule.exitCrop();
                } else {
                    mIsPreViewInCropMode = false;
                }

                //hide annot menu.
                if (uiExtensionsManager.getDocumentManager().getCurrentAnnot() != null) {
                    uiExtensionsManager.getDocumentManager().setCurrentAnnot(null);
                }
                mPdfViewCtrl.setPageLayoutMode(PDFViewCtrl.PAGELAYOUTMODE_REFLOW);
                mPdfViewCtrl.setReflowMode(mPreReflowMode);
                uiExtensionsManager.triggerDismissMenuEvent();
                uiExtensionsManager.changeState(ReadStateConfig.STATE_REFLOW);
                IMainFrame mainFrame = uiExtensionsManager.getMainFrame();
                mainFrame.hideSettingWindow();
                if (!mainFrame.isToolbarsVisible()) {
                    mainFrame.showToolbars();
                }
                resetPanelView(true);
            }
        } else {
            if (mPreViewMode == PDFViewCtrl.PAGELAYOUTMODE_REFLOW) {
                mPdfViewCtrl.setPageLayoutMode(PDFViewCtrl.PAGELAYOUTMODE_SINGLE);
            } else {
                mPdfViewCtrl.setPageLayoutMode(mPreViewMode);
                if (!AppDisplay.isPad() && mIsPreViewInCropMode) {
                    CropModule cropModule = (CropModule) uiExtensionsManager.getModuleByName(Module.MODULE_NAME_CROP);
                    if (cropModule != null)
                        cropModule.restoreCrop();
                }
            }

            resetZoomMode();
            resetPanelView(false);

            if (changeState) {
                if (mPreReadState == ReadStateConfig.STATE_FILLSIGN) {
                    uiExtensionsManager.changeState(ReadStateConfig.STATE_FILLSIGN);
                } else {
                    if (!isReflow && uiExtensionsManager.getState() == ReadStateConfig.STATE_REDACT) {
                        uiExtensionsManager.changeState(ReadStateConfig.STATE_REDACT);
                    } else {
                        uiExtensionsManager.changeState(ReadStateConfig.STATE_NORMAL);
                    }
                }
            }
        }
        mPreReadState = curState;
        mPreViewMode = curLayout;
        mPreReflowMode = curReflowMode;
        PageNavigationModule module = (PageNavigationModule) uiExtensionsManager.getModuleByName(Module.MODULE_NAME_PAGENAV);
        if (module != null) {
            module.resetJumpView();
        }
    }

    private void onStatusChanged() {
        if (mPdfViewCtrl.getDoc() == null) {
            return;
        }

        if (mReflowBottomBar == null)
            initReflowBar();
        else {
            if (mBookMarkItem != null) {
                ReadingBookmarkModule module = (ReadingBookmarkModule) ((UIExtensionsManager) mUiExtensionsManager).getModuleByName(MODULE_NAME_BOOKMARK);
                if (module != null) {
                    module.addMarkedItem(mBookMarkItem);
                    mBookMarkItem.getContentView().setVisibility(View.VISIBLE);
                } else {
                    mBookMarkItem.getContentView().setVisibility(View.INVISIBLE);
                }
            }
        }

        if (mPdfViewCtrl.getPageLayoutMode() == PDFViewCtrl.PAGELAYOUTMODE_REFLOW) {
            UIExtensionsManager uiExtensionsManager = (UIExtensionsManager) mUiExtensionsManager;
            if (!AppDisplay.isPad()) {
                boolean isToolbarsVisible = uiExtensionsManager.getMainFrame().isToolbarsVisible();
                if (isToolbarsVisible && !uiExtensionsManager.getPanelManager().isShowingPanel()) {
                    mReflowBottomBar.getContentView().setVisibility(View.VISIBLE);
                    mReflowTopBar.getContentView().setVisibility(View.VISIBLE);
                } else {
                    mReflowBottomBar.getContentView().setVisibility(View.INVISIBLE);
                    mReflowTopBar.getContentView().setVisibility(View.INVISIBLE);
                }
                mPdfViewCtrl.offsetScrollBoundary(0, isToolbarsVisible ? mReflowTopBar.getContentView().getHeight() : 0,
                        0, isToolbarsVisible ? -mReflowBottomBar.getContentView().getHeight() : 0);
            } else {
                mDragView = createDragToolView();

                MainFrame mainFrame = (MainFrame) uiExtensionsManager.getMainFrame();
                mDragView.setVisible(!mainFrame.isShowFullScreenUI());
            }
            mScale = mPdfViewCtrl.getZoom();
            resetPicItem();
            resetZoomInItem();
            resetZoomOutItem();
            resetNextPageItem();
            resetPrePageItem();
        } else {
            if (AppDisplay.isPad()) {
                if (mDragView != null) {
                    mDragView.removeDragView();
                    mDragView = null;
                }
            } else {
                mReflowBottomBar.getContentView().setVisibility(View.INVISIBLE);
                mReflowTopBar.getContentView().setVisibility(View.INVISIBLE);
            }
        }
    }

    @Override
    public boolean unloadModule() {
        mPdfViewCtrl.unregisterDocEventListener(mDocEventListener);
        mDocEventListener = null;
        if (mUiExtensionsManager != null && mUiExtensionsManager instanceof UIExtensionsManager) {
            UIExtensionsManager uiExtensionsManager = (UIExtensionsManager) mUiExtensionsManager;
            uiExtensionsManager.unregisterStateChangeListener(mStatusChangeListener);
            uiExtensionsManager.unregisterThemeEventListener(mThemeEventListener);
            uiExtensionsManager.unregisterInteractionListener(mInteractionEventListener);
            uiExtensionsManager.getPanelManager().unregisterPanelEventListener(mPanelEventListener);
            if (AppDisplay.isPad())
                uiExtensionsManager.unregisterLifecycleListener(mLifecycleEventListener);
        }
        return true;
    }

    @Override
    public String getName() {
        return MODULE_NAME_REFLOW;
    }

    private final IStateChangeListener mStatusChangeListener = new IStateChangeListener() {
        @Override
        public void onStateChanged(int oldState, int newState) {
            int curLayout = mPdfViewCtrl.getPageLayoutMode();
            int curReflowMode = mPdfViewCtrl.getReflowMode();

            try {
                UIExtensionsManager uiExtensionsManager = (UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager();
                if (mSettingWindow == null) {
                    mSettingWindow = uiExtensionsManager.getMainFrame().getSettingWindow();
                }
                if (uiExtensionsManager.getState() == ReadStateConfig.STATE_REFLOW) {
                    if (oldState != ReadStateConfig.STATE_REFLOW) {
                        mMaxScale = mPdfViewCtrl.getMaxZoomLimit();
                        mMinSale = mPdfViewCtrl.getMinZoomLimit();

                        if (AppDisplay.isPad()) {
                            mSettingWindow.enableBar(IViewSettingsWindow.TYPE_FIT_WIDTH, false);
                            mSettingWindow.enableBar(IViewSettingsWindow.TYPE_FIT_PAGE, false);
                            mSettingWindow.enableBar(IViewSettingsWindow.TYPE_ROTATE_VIEW, false);
                        }
                    }

                    if (curLayout != PDFViewCtrl.PAGELAYOUTMODE_REFLOW) {
                        mIsReflow = true;
                        mSettingWindow.setProperty(IViewSettingsWindow.TYPE_REFLOW, true);
                        mPdfViewCtrl.setPageLayoutMode(PDFViewCtrl.PAGELAYOUTMODE_REFLOW);
                        mPdfViewCtrl.setReflowMode(mPreReflowMode);
                        if (uiExtensionsManager.getDocumentManager().getCurrentAnnot() != null) {
                            uiExtensionsManager.getDocumentManager().setCurrentAnnot(null);
                        }
                        IMainFrame mainFrame = uiExtensionsManager.getMainFrame();
                        mainFrame.hideSettingWindow();
                        if (!mainFrame.isToolbarsVisible()) {
                            mainFrame.showToolbars();
                        }
                        resetPanelView(true);
                        mPreViewMode = curLayout;
                        mPreReflowMode = curReflowMode;
                        PageNavigationModule pageNumberJump = (PageNavigationModule) uiExtensionsManager.getModuleByName(Module.MODULE_NAME_PAGENAV);
                        if (pageNumberJump != null)
                            pageNumberJump.resetJumpView();
                    }
                } else {
                    if (curLayout == PDFViewCtrl.PAGELAYOUTMODE_REFLOW) {
                        mSettingWindow.setProperty(IViewSettingsWindow.TYPE_REFLOW, false);
                        setReflowPageMode(false, false);

                        if (AppDisplay.isPad()) {
                            mSettingWindow.enableBar(IViewSettingsWindow.TYPE_FIT_WIDTH, true);
                            mSettingWindow.enableBar(IViewSettingsWindow.TYPE_FIT_PAGE, true);
                            mSettingWindow.enableBar(IViewSettingsWindow.TYPE_ROTATE_VIEW, true);
                        }
                    } else if (oldState == ReadStateConfig.STATE_REFLOW && newState != ReadStateConfig.STATE_REFLOW) {
                        mIsReflow = false;
                        mSettingWindow.setProperty(IViewSettingsWindow.TYPE_REFLOW, false);
                        if (AppDisplay.isPad()) {
                            mSettingWindow.enableBar(IViewSettingsWindow.TYPE_FIT_WIDTH, true);
                            mSettingWindow.enableBar(IViewSettingsWindow.TYPE_FIT_PAGE, true);
                            mSettingWindow.enableBar(IViewSettingsWindow.TYPE_ROTATE_VIEW, true);
                        }
                    }
                }
            } catch (Exception e) {
                e.printStackTrace();
            }
            onStatusChanged();
        }
    };

    public void clearData() {
        mIsPreViewInCropMode = false;
    }

    private final IInteractionEventListener mInteractionEventListener = new IInteractionEventListener() {
        @Override
        public boolean onKeyDown(Activity act, int keyCode, KeyEvent event) {
            UIExtensionsManager uiExtensionsManager = (UIExtensionsManager) mUiExtensionsManager;
            if (uiExtensionsManager.getState() == ReadStateConfig.STATE_REFLOW) {
                if (keyCode == KeyEvent.KEYCODE_BACK) {
                    if (AppDisplay.isPad()) {
                        if (uiExtensionsManager.getState() == ReadStateConfig.STATE_REFLOW) {
                            setReflowPageMode(false);
                            return true;
                        }
                    } else {
                        FileSpecPanelModule fileSpecPanelModule = (FileSpecPanelModule) uiExtensionsManager.getModuleByName(Module.MODULE_NAME_FILE_PANEL);
                        if (fileSpecPanelModule != null && fileSpecPanelModule.onKeyBack()) {
                            return true;
                        }

                        if (uiExtensionsManager.getPanelManager().isShowingPanel()) {
                            uiExtensionsManager.getPanelManager().hidePanel();
                            return true;
                        }

                        if (mPreReadState == ReadStateConfig.STATE_FILLSIGN)
                            uiExtensionsManager.changeState(ReadStateConfig.STATE_FILLSIGN);
                        else
                            uiExtensionsManager.changeState(ReadStateConfig.STATE_NORMAL);
                        resetZoomMode();
                        return true;
                    }
                }
            }
            return false;
        }
    };

    private final ILifecycleEventListener mLifecycleEventListener = new LifecycleEventListener() {
        @Override
        public void onHiddenChanged(boolean hidden) {
            if (hidden) {
                UIExtensionsManager extensionsManager = (UIExtensionsManager) mUiExtensionsManager;
                mInteractionEventListener.onKeyDown(extensionsManager.getAttachedActivity(), KeyEvent.KEYCODE_BACK, null);
            }
        }
    };

    private final IThemeEventListener mThemeEventListener = new IThemeEventListener() {
        @Override
        public void onThemeColorChanged(String type, int color) {
            if (mPdfViewCtrl.isDynamicXFA()) return;
            removeBar();
            initReflowBar();
            addBar();

            if (mBackItem != null)
                mBackItem.setTextColor(ThemeConfig.getInstance(mContext).getPrimaryColor());
            if (((UIExtensionsManager) mUiExtensionsManager).getState() == ReadStateConfig.STATE_REFLOW) {
                resetPicItem();
                resetZoomInItem();
                resetZoomOutItem();
                resetNextPageItem();
                resetPrePageItem();

                if (AppDisplay.isPad())
                    mDragView = createDragToolView();
            }
        }
    };

    private final IPanelManager.OnPanelEventListener mPanelEventListener = new IPanelManager.OnPanelEventListener() {
        @Override
        public void onPanelOpened() {
            if (mPdfViewCtrl.getPageLayoutMode() == PDFViewCtrl.PAGELAYOUTMODE_REFLOW) {
                if (mNeedResetPanelView) {
                    resetPanelView(mIsReflow);
                    mNeedResetPanelView = false;
                }
                if (!AppDisplay.isPad()) {
                    mReflowTopBar.getContentView().setVisibility(View.GONE);
                    mReflowBottomBar.getContentView().setVisibility(View.GONE);
                }
            }
        }

        @Override
        public void onPanelClosed() {
            if (mPdfViewCtrl.getPageLayoutMode() == PDFViewCtrl.PAGELAYOUTMODE_REFLOW) {
                if (!AppDisplay.isPad()) {
                    AppThreadManager.getInstance().getMainThreadHandler().postDelayed(new Runnable() {
                        @Override
                        public void run() {
                            mReflowTopBar.getContentView().setVisibility(View.VISIBLE);
                            mReflowBottomBar.getContentView().setVisibility(View.VISIBLE);
                        }
                    }, 300/* Animation time required to close panel*/);
                }
            }
        }
    };

    public void setDragToolbarVisiable(boolean visiable) {
        if (mDragView != null)
            mDragView.setVisible(visiable);
    }

}
