/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.modules;

import static com.foxit.uiextensions.utils.AppUtil.getSelectedButtonColorStateList;
import static com.foxit.uiextensions.utils.AppUtil.setSelectedButtonState;

import android.annotation.SuppressLint;
import android.app.Activity;
import android.app.SearchManager;
import android.content.Context;
import android.content.Intent;
import android.graphics.Rect;
import android.graphics.RectF;
import android.os.Build;
import android.text.Editable;
import android.text.SpannableString;
import android.text.Spanned;
import android.text.TextUtils;
import android.text.TextWatcher;
import android.text.style.BackgroundColorSpan;
import android.text.style.ForegroundColorSpan;
import android.util.DisplayMetrics;
import android.util.Log;
import android.view.KeyEvent;
import android.view.LayoutInflater;
import android.view.MotionEvent;
import android.view.View;
import android.view.ViewGroup;
import android.view.animation.Animation;
import android.view.animation.AnimationUtils;
import android.view.inputmethod.EditorInfo;
import android.widget.CheckBox;
import android.widget.EditText;
import android.widget.FrameLayout;
import android.widget.ImageView;
import android.widget.LinearLayout;
import android.widget.RelativeLayout;
import android.widget.TextView;

import androidx.annotation.NonNull;
import androidx.fragment.app.FragmentActivity;
import androidx.recyclerview.widget.DefaultItemAnimator;
import androidx.recyclerview.widget.LinearLayoutManager;
import androidx.recyclerview.widget.RecyclerView;

import com.foxit.sdk.PDFException;
import com.foxit.sdk.PDFViewCtrl;
import com.foxit.sdk.common.Constants;
import com.foxit.sdk.common.Progressive;
import com.foxit.sdk.common.fxcrt.RectFArray;
import com.foxit.sdk.pdf.PDFDoc;
import com.foxit.sdk.pdf.PDFPage;
import com.foxit.sdk.pdf.SearchCancelCallback;
import com.foxit.sdk.pdf.TextPage;
import com.foxit.sdk.pdf.TextSearch;
import com.foxit.uiextensions.Module;
import com.foxit.uiextensions.R;
import com.foxit.uiextensions.UIExtensionsManager;
import com.foxit.uiextensions.annots.common.UIBtnImageView;
import com.foxit.uiextensions.annots.redaction.RedactModule;
import com.foxit.uiextensions.browser.adapter.SuperAdapter;
import com.foxit.uiextensions.browser.adapter.viewholder.SuperViewHolder;
import com.foxit.uiextensions.controls.dialog.UIPopoverWin;
import com.foxit.uiextensions.controls.dialog.sheetmenu.UIActionMenu;
import com.foxit.uiextensions.modules.panel.bean.BaseBean;
import com.foxit.uiextensions.pdfreader.config.ReadStateConfig;
import com.foxit.uiextensions.theme.ThemeConfig;
import com.foxit.uiextensions.theme.ThemeUtil;
import com.foxit.uiextensions.utils.AppDisplay;
import com.foxit.uiextensions.utils.AppResource;
import com.foxit.uiextensions.utils.AppUtil;
import com.foxit.uiextensions.utils.thread.AppThreadManager;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.Callable;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.regex.PatternSyntaxException;

import io.reactivex.Single;
import io.reactivex.android.schedulers.AndroidSchedulers;
import io.reactivex.disposables.CompositeDisposable;
import io.reactivex.disposables.Disposable;
import io.reactivex.functions.Consumer;
import io.reactivex.schedulers.Schedulers;


public class SearchView {

    private final Context mContext;
    private final ViewGroup mParent;
    private final PDFViewCtrl mPdfViewCtrl;

    private View mSearchView = null;

    private boolean mIsBlank = true;
    private String mSearch_content;

    private LinearLayout mRd_search_ll_top;
    private EditText mTop_et_content;
    private ImageView mTop_iv_clear;
    private ImageView mIvSearchSettings;
    private TextView mTop_bt_cancel;

    private LinearLayout mRd_search_ll_center;
    private View mViewCenterLeft;
    private RelativeLayout mViewCenterRight;
    private LinearLayout mSearchCenterTop;
    private TextView mCenter_tv_total_number;
    private RecyclerView mCenter_lv_result_list;

    private LinearLayout mRd_search_ll_bottom;
    private ImageView mBottom_iv_prev;
    private ImageView mBottom_iv_next;
    private ImageView mBottom_iv_result;
    private LinearLayout mBottom_ll_shadow;

    protected List<RectF> mRect = new ArrayList<>();
    protected int mPageIndex = -1;//The page index of the search result
    protected boolean mIsCancel = true;
    private final DisplayMetrics mMetrics;
    private final LayoutInflater mInflater;
    private final SearchAdapter mAdapterSearch;
    private long mSearchId = 0;

    //    private PopupWindow mSearchSettingsPopup;
    private UIActionMenu mSearchSettingsWindow;
    private ImageView mCheckBoxMatchWholeWords;
    private ImageView mCheckBoxMatchCase;
    private int mSearchFlags = TextSearch.e_SearchNormal;
    protected List<List<RectF>> mSearchFoundRectList = new ArrayList<>();
    private View mNoContentView;

    private boolean isRedactSearch;
    private TextView mMarkRedactionButton;
    private final List<SearchResult> mSelectedResults = new ArrayList<>();
    private CheckBox mSelectAllCheckBox;
    private View mWholeWordsSelectLayout;
    private View mCaseSensitiveSelectLayout;
    private ImageView ivSearch;
    private RelativeLayout mRlTopLeft;
    private TextView mTvNoContent;
    private ImageView mTopSearchDivider;
    private ImageView mBottomIvSearchDivider;
    private TextView mTvSearchInInternet;
    private TextView mTvSearchCaseSensitive;
    private TextView mTvSearchWholeWords;

    protected SearchView(Context context, ViewGroup parent, PDFViewCtrl pdfViewCtrl) {
        mContext = context;
        mParent = parent;
        mPdfViewCtrl = pdfViewCtrl;
        mMetrics = context.getResources().getDisplayMetrics();
        mInflater = (LayoutInflater) context.getSystemService(Context.LAYOUT_INFLATER_SERVICE);
        mAdapterSearch = new SearchAdapter(context);
    }

    private void initView() {
        if (mSearchView != null) return;
        mSearchView = LayoutInflater.from(mContext).inflate(R.layout.search_layout, null, false);
        mSearchView.setVisibility(View.GONE);
        mParent.addView(mSearchView);
        ivSearch = mSearchView.findViewById(R.id.top_iv_search);
        mTopSearchDivider = mSearchView.findViewById(R.id.top_search_divider);
        mBottomIvSearchDivider = mSearchView.findViewById(R.id.bottom_iv_search_divider);
        mRlTopLeft = mSearchView.findViewById(R.id.top_ll_left);
        mRlTopLeft.setBackground(AppResource.getDrawable(mContext, R.drawable.shape_search_bg));
        ThemeUtil.setTintList(ivSearch, ThemeUtil.getPrimaryIconColor(mContext));
        mRd_search_ll_top = mSearchView.findViewById(R.id.rd_search_ll_top);
        mRd_search_ll_top.setBackgroundColor(ThemeConfig.getInstance(mContext).getB2());
        mTop_et_content = mSearchView.findViewById(R.id.top_et_content);
        if (AppDisplay.isPad()) { // SDKRD-9313
            mTop_et_content.setImeOptions(EditorInfo.IME_FLAG_NO_EXTRACT_UI | EditorInfo.IME_ACTION_SEARCH);
        }
        mTop_iv_clear = mSearchView.findViewById(R.id.top_iv_clear);
        ThemeUtil.setTintList(mTop_iv_clear, ThemeUtil.getPrimaryIconColor(mContext));
        mTop_bt_cancel = mSearchView.findViewById(R.id.top_bt_cancel);
        mTop_bt_cancel.setTextColor(ThemeUtil.getPrimaryTextColor(mContext));
        mIvSearchSettings = mSearchView.findViewById(R.id.top_search_settings);
        ThemeUtil.setTintList(mIvSearchSettings, ThemeUtil.getPrimaryIconColor(mContext));
        mRd_search_ll_center = mSearchView.findViewById(R.id.rd_search_ll_center);
        mViewCenterLeft = mSearchView.findViewById(R.id.rd_search_center_left);
        mViewCenterRight = mSearchView.findViewById(R.id.rd_search_center_right);
        mSearchCenterTop = mSearchView.findViewById(R.id.rd_search_center_top);
        mCenter_tv_total_number = mSearchView.findViewById(R.id.center_tv_total_number);
        mSelectAllCheckBox = mSearchView.findViewById(R.id.select_all_checkbox);
        ThemeUtil.setTintList(mSelectAllCheckBox, ThemeUtil.getCheckboxColor(mContext));
        mSelectAllCheckBox.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                if (isFastDoubleClick()) {
                    return;
                }

                mAdapterSearch.setSelectAll(mSelectAllCheckBox.isChecked());
                mMarkRedactionButton.setEnabled(mSelectAllCheckBox.isChecked());
            }
        });
        mCenter_lv_result_list = mSearchView.findViewById(R.id.center_lv_result_list);
        mCenter_lv_result_list.setHasFixedSize(true);
        mCenter_lv_result_list.setLayoutManager(new LinearLayoutManager(mContext, LinearLayoutManager.VERTICAL, false));
        mCenter_lv_result_list.setItemAnimator(new DefaultItemAnimator());

        mNoContentView = mSearchView.findViewById(R.id.rd_search_no_content);
        UIBtnImageView noContentImage = mSearchView.findViewById(R.id.iv_no_content);
        noContentImage.setColorStateList(ThemeUtil.getItemIconColor(mContext));
        mTvNoContent = mSearchView.findViewById(R.id.tv_no_content);
        mMarkRedactionButton = mSearchView.findViewById(R.id.mark_redaction_button);
        int disabled = ThemeConfig.getInstance(mContext).getI2();
        int normal = ThemeConfig.getInstance(mContext).getPrimaryColor();
        mMarkRedactionButton.setTextColor(AppResource.createColorStateList(mContext, disabled, normal));
        mMarkRedactionButton.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                RedactModule module = ((RedactModule) ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getModuleByName(Module.MODULE_NAME_REDACT));
                if (module != null) {
                    module.redactTextContent(mSelectedResults);
                }
                cancel();
            }
        });
        mMarkRedactionButton.setBackgroundColor(mContext.getResources().getColor(R.color.ui_color_grey_f4f4f4));
        mRd_search_ll_bottom = mSearchView.findViewById(R.id.rd_search_ll_bottom);
        if (AppDisplay.isPad()) {
            ViewGroup.MarginLayoutParams marginLayoutParams = (ViewGroup.MarginLayoutParams) mRd_search_ll_bottom.getLayoutParams();
            marginLayoutParams.width = AppResource.getDimensionPixelSize(mContext, R.dimen.ux_margin_184dp);
            marginLayoutParams.bottomMargin = AppResource.getDimensionPixelSize(mContext, R.dimen.ux_margin_24dp);
        }
        mBottom_iv_prev = mSearchView.findViewById(R.id.bottom_iv_prev);
        mBottom_iv_next = mSearchView.findViewById(R.id.bottom_iv_next);
        mBottom_iv_result = mSearchView.findViewById(R.id.bottom_iv_result);
        mBottom_ll_shadow = mSearchView.findViewById(R.id.bottom_ll_shadow);
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.Q) {
            mTop_et_content.setForceDarkAllowed(false);
            mBottom_iv_prev.setForceDarkAllowed(false);
            mBottom_iv_next.setForceDarkAllowed(false);
            mBottom_iv_result.setForceDarkAllowed(false);
            mBottom_ll_shadow.setForceDarkAllowed(false);
        }
        ThemeUtil.setTintList(mBottom_iv_prev, ThemeUtil.getPrimaryIconColor(mContext));
        ThemeUtil.setTintList(mBottom_iv_next, ThemeUtil.getPrimaryIconColor(mContext));
        ThemeUtil.setTintList(mBottom_iv_result, ThemeUtil.getPrimaryIconColor(mContext));
        mRd_search_ll_center.setVisibility(View.VISIBLE);
        mRd_search_ll_center.setBackgroundResource(R.color.ux_color_translucent);
        mViewCenterLeft.setVisibility(View.GONE);
        mViewCenterRight.setVisibility(View.GONE);
        mRd_search_ll_bottom.setVisibility(View.GONE);
        mBottom_ll_shadow.setVisibility(View.GONE);

        setSearchResultWidth();
        bindEvent();
    }

    private View getSettingsView() {
        View settingsView = mInflater.inflate(R.layout.search_settings, null);
        LinearLayout searchWholeWords = settingsView.findViewById(R.id.rela_search_whole_words);
        LinearLayout matchCase = settingsView.findViewById(R.id.rela_search_case_sensitive);
        mWholeWordsSelectLayout = settingsView.findViewById(R.id.whole_words_select);
        ThemeUtil.setBackgroundTintList(mWholeWordsSelectLayout, getSelectedButtonColorStateList(mContext));
        mCheckBoxMatchWholeWords = settingsView.findViewById(R.id.whole_words_select_image);
        mCaseSensitiveSelectLayout = settingsView.findViewById(R.id.case_sensitive_select);
        ThemeUtil.setBackgroundTintList(mCaseSensitiveSelectLayout, getSelectedButtonColorStateList(mContext));
        mCheckBoxMatchCase = settingsView.findViewById(R.id.case_sensitive_select_image);
        if (Build.VERSION.SDK_INT >= 29) {// 29 == Build.VERSION_CODES.Q
            mCheckBoxMatchCase.setForceDarkAllowed(false);
            mCheckBoxMatchWholeWords.setForceDarkAllowed(false);
            mCaseSensitiveSelectLayout.setForceDarkAllowed(false);
            mWholeWordsSelectLayout.setForceDarkAllowed(false);
        }
        mTvSearchInInternet = settingsView.findViewById(R.id.tv_search_in_internet);
        mTvSearchCaseSensitive = settingsView.findViewById(R.id.tv_search_case_sensitive);
        mTvSearchWholeWords = settingsView.findViewById(R.id.tv_search_whole_words);

        searchWholeWords.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                if (mCheckBoxMatchWholeWords.getTag() != null) {
                    mSearchFlags = mSearchFlags & (~TextSearch.e_SearchMatchWholeWord);
                } else {
                    mSearchFlags = mSearchFlags | TextSearch.e_SearchMatchWholeWord;
                }
                toggleSelectButtonState(mCheckBoxMatchWholeWords);
                if (!TextUtils.isEmpty(mTop_et_content.getText().toString()))
                    startSearch();
                mSearchSettingsWindow.dismiss();
            }
        });

        matchCase.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                if (mCheckBoxMatchCase.getTag() != null) {
                    mSearchFlags = mSearchFlags & (~TextSearch.e_SearchMatchCase);
                } else {
                    mSearchFlags = mSearchFlags | TextSearch.e_SearchMatchCase;
                }
                toggleSelectButtonState(mCheckBoxMatchCase);
                if (!TextUtils.isEmpty(mTop_et_content.getText().toString()))
                    startSearch();
                mSearchSettingsWindow.dismiss();
            }
        });

        mTvSearchInInternet.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                String searchContent = mTop_et_content.getText().toString();
                if (!TextUtils.isEmpty(searchContent)) {
                    UIExtensionsManager uiExtensionsManager = (UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager();
                    Activity activity = uiExtensionsManager.getAttachedActivity();
                    if (null != activity) {
                        Intent search = new Intent(Intent.ACTION_WEB_SEARCH);
                        search.putExtra(SearchManager.QUERY, searchContent);
                        activity.startActivity(search);
                    }
                }
                mSearchSettingsWindow.dismiss();
            }
        });
        return settingsView;
    }

    public void setFlag(boolean consecutive, boolean word) {
        initSearchSettings();
        if (consecutive && mCheckBoxMatchCase.getTag() == null) {
            mSearchFlags = mSearchFlags | TextSearch.e_SearchMatchCase;
            toggleSelectButtonState(mCheckBoxMatchCase);
        } else if (!consecutive && mCheckBoxMatchCase.getTag() != null){
            mSearchFlags = mSearchFlags & (~TextSearch.e_SearchMatchCase);
            toggleSelectButtonState(mCheckBoxMatchCase);
        }

        if (word && mCheckBoxMatchWholeWords.getTag() == null) {
            mSearchFlags = mSearchFlags | TextSearch.e_SearchMatchWholeWord;
            toggleSelectButtonState(mCheckBoxMatchWholeWords);
        } else if (!word && mCheckBoxMatchWholeWords.getTag() != null) {
            mSearchFlags = mSearchFlags & (~TextSearch.e_SearchMatchWholeWord);
            toggleSelectButtonState(mCheckBoxMatchWholeWords);
        }
    }

    public void toggleSelectButtonState(ImageView target) {
        if (target.getTag() != null) {
            setSelectedButtonState(false, target);
            target.setTag(null);
        } else {
            setSelectedButtonState(true, target);
            target.setTag(true);
        }
    }

    public boolean isRedactSearch() {
        return isRedactSearch;
    }

    protected View getView() {
        return mSearchView;
    }

    @SuppressLint("ClickableViewAccessibility")
    private void bindEvent() {
        AppUtil.dismissInputSoft(mTop_et_content);

        mTop_et_content.addTextChangedListener(new myTextWatcher());
        mTop_et_content.setOnKeyListener(mySearchListener);
        mTop_et_content.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).triggerDismissMenuEvent();
            }
        });

        mIvSearchSettings.setOnClickListener(searchModelListener);
        mTop_iv_clear.setOnClickListener(searchModelListener);
        mTop_bt_cancel.setOnClickListener(searchModelListener);
        mRd_search_ll_top.setOnClickListener(searchModelListener);

        mViewCenterLeft.setOnClickListener(searchModelListener);
        mViewCenterRight.setOnClickListener(searchModelListener);

        mBottom_iv_result.setOnClickListener(searchModelListener);
        mBottom_iv_prev.setOnClickListener(searchModelListener);
        mBottom_iv_next.setOnClickListener(searchModelListener);
        mBottom_iv_prev.setEnabled(false);
        mBottom_iv_next.setEnabled(false);

        mRd_search_ll_bottom.setOnTouchListener(new View.OnTouchListener() {

            @Override
            public boolean onTouch(View v, MotionEvent event) {
                return true;
            }
        });

        mCenter_lv_result_list.setAdapter(mAdapterSearch);
        mAdapterSearch.setOnItemClickListener(mItemClickListener);
    }

    protected void onLayoutChange(View v, int newWidth, int newHeight, int oldWidth, int oldHeight) {
        if (mSearchSettingsWindow != null && mSearchSettingsWindow.isShowing()) {
            UIExtensionsManager extensionsManager = (UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager();
            if (extensionsManager == null) return;
            Rect rect = new Rect();
            mIvSearchSettings.getGlobalVisibleRect(rect);
            mSearchSettingsWindow.show(extensionsManager.getRootView(), rect, UIPopoverWin.ARROW_TOP, 0);
        }
    }

    protected void setToolbarIcon() {
        mBottom_iv_prev.setEnabled(true);
        mBottom_iv_next.setEnabled(true);
        if (isFirstSearchResult())
            mBottom_iv_prev.setEnabled(false);
        if (isLastSearchResult())
            mBottom_iv_next.setEnabled(false);
    }

    private void setSearchNumber(int count) {
        mSelectAllCheckBox.setVisibility(View.GONE);
        mNoContentView.setVisibility(View.GONE);
        mCenter_tv_total_number.setVisibility(View.VISIBLE);
        mMarkRedactionButton.setVisibility(mSelectAllCheckBox.getVisibility());
        mCenter_tv_total_number.setText(AppResource.getString(mContext, R.string.searching_find_number, count));
    }

    private void setTotalNumber(int count) {
        if (count > 0) {
            mNoContentView.setVisibility(View.GONE);
            mSelectAllCheckBox.setVisibility(isRedactSearch ? View.VISIBLE : View.GONE);
            mMarkRedactionButton.setVisibility(mSelectAllCheckBox.getVisibility());
            mCenter_tv_total_number.setVisibility(View.VISIBLE);
            mCenter_tv_total_number.setText(String.format("%s  %s", AppResource.getString(mContext, R.string.search_find_number), count));
        } else {
            mCenter_tv_total_number.setVisibility(View.GONE);
            mSelectAllCheckBox.setVisibility(View.GONE);
            mMarkRedactionButton.setVisibility(View.GONE);
            mNoContentView.setVisibility(View.VISIBLE);
        }
    }

    private void setSearchResultWidth() {
        LinearLayout.LayoutParams leftParams = (LinearLayout.LayoutParams) mViewCenterLeft.getLayoutParams();
        LinearLayout.LayoutParams rightParams = (LinearLayout.LayoutParams) mViewCenterRight.getLayoutParams();
        if (AppDisplay.isPad()) {
            if (AppDisplay.isLandscape()) {
                leftParams.width = 0;
                leftParams.height = LinearLayout.LayoutParams.MATCH_PARENT;
                leftParams.weight = 2.0f;

                rightParams.width = 0;
                rightParams.height = LinearLayout.LayoutParams.MATCH_PARENT;
                rightParams.weight = 1.0f;

            } else {
                leftParams.width = 0;
                leftParams.height = LinearLayout.LayoutParams.MATCH_PARENT;
                leftParams.weight = 1.0f;

                rightParams.width = 0;
                rightParams.height = LinearLayout.LayoutParams.MATCH_PARENT;
                rightParams.weight = 1.0f;

            }
        } else {
            leftParams.width = 0;
            leftParams.height = LinearLayout.LayoutParams.MATCH_PARENT;
            leftParams.weight = 0;

            rightParams.width = 0;
            rightParams.height = LinearLayout.LayoutParams.MATCH_PARENT;
            rightParams.weight = 4.0f;
        }
        mViewCenterLeft.setLayoutParams(leftParams);
        mViewCenterRight.setLayoutParams(rightParams);
    }

    public void show() {
        UIExtensionsManager uiExtensionsManager = (UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager();
        if (mSearchView == null) {
            if (uiExtensionsManager.getState() == ReadStateConfig.STATE_REFLOW)
                launchRedactSearchView();
            else
                launchSearchView();
        }

        if (mSearchView != null) {
            uiExtensionsManager.changeState(ReadStateConfig.STATE_SEARCH);
            mSearchView.setVisibility(View.VISIBLE);
        }
    }

    public void dismiss() {
        if (mSearchView != null) {
            ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).changeState(ReadStateConfig.STATE_NORMAL);
            mSearchView.setVisibility(View.GONE);
        }
    }

    private ISearchItemClickListener mItemClickListener = new ISearchItemClickListener() {
        @Override
        public void onItemClick(int position, SearchResult result) {
            if (isFastDoubleClick()) {
                return;
            }

            onPreItemClick();

            if (mTagResultList.contains(mShowResultList.get(position))) {
                mCurrentPosition = position + 1;
                setCurrentPageX();
                RectF rectF = new RectF(mCurrentPageX, mCurrentPageY, mCurrentSearchR, mCurrentSearchB);
                RectF canvasRectF = new RectF();
                boolean transSuccess = mPdfViewCtrl.convertPageViewRectToDisplayViewRect(rectF, canvasRectF, mShowResultList.get(mCurrentPosition).mPageIndex);
                int screenWidth = getVisibleWidth().width();
                int screenHeight = getVisibleWidth().height();
                if (!transSuccess || canvasRectF.left < 0 || canvasRectF.right > screenWidth || canvasRectF.top < 0 || canvasRectF.bottom > screenHeight) {
                    int x = (int) (mCurrentPageX - getScreenWidth() / 4.0f);
                    int y = (int) (mCurrentPageY - getScreenHeight() / 4.0f);
                    mPdfViewCtrl.gotoPage(mShowResultList.get(mCurrentPosition).mPageIndex, x, y);
                }
            } else {
                mCurrentPosition = position;
                setCurrentPageX();
                RectF rectF = new RectF(mCurrentPageX, mCurrentPageY, mCurrentSearchR, mCurrentSearchB);
                RectF canvasRectF = new RectF();
                boolean transSuccess = mPdfViewCtrl.convertPageViewRectToDisplayViewRect(rectF, canvasRectF, mShowResultList.get(mCurrentPosition).mPageIndex);
                int screenWidth = getVisibleWidth().width();
                int screenHeight = getVisibleWidth().height();
                if (!transSuccess || canvasRectF.left < 0 || canvasRectF.right > screenWidth || canvasRectF.top < 0 || canvasRectF.bottom > screenHeight) {
                    int x = (int) (mCurrentPageX - getScreenWidth() / 4.0f);
                    int y = (int) (mCurrentPageY - getScreenHeight() / 4.0f);
                    mPdfViewCtrl.gotoPage(mShowResultList.get(mCurrentPosition).mPageIndex, x, y);
                }
            }
            mPageIndex = mShowResultList.get(mCurrentPosition).mPageIndex;
            refreshFoundRectList();
            mRect = mShowResultList.get(mCurrentPosition).mRects;
            setToolbarIcon();
            mPdfViewCtrl.invalidate();
        }
    };

    public void refreshFoundRectList() {
        int startPosition = 0;
        int endPosition = 0;
        SearchResult tag;
        for (int i = 0; i < mTagResultList.size(); i++) {
            tag = mTagResultList.get(i);
            startPosition += 1;
            if (tag.mPageIndex < mPageIndex) {
                startPosition += tag.mPatternStart;
            } else if (tag.mPageIndex == mPageIndex) {
                endPosition = startPosition + tag.mPatternStart;
                break;
            }
        }
        mSearchFoundRectList.clear();
        for (int i = startPosition; i < endPosition; i++) {
            if (mCurrentPosition == i)
                continue;
            mSearchFoundRectList.add(mShowResultList.get(i).mRects);
        }
    }

    private void onPreItemClick() {
        AppUtil.dismissInputSoft(mTop_et_content);
        Animation animationL2R = AnimationUtils.loadAnimation(mContext, R.anim.view_anim_rtol_hide);
        animationL2R.setAnimationListener(new Animation.AnimationListener() {
            @Override
            public void onAnimationStart(Animation animation) {
            }

            @Override
            public void onAnimationEnd(Animation animation) {
                mViewCenterRight.setVisibility(View.GONE);
                mViewCenterLeft.setVisibility(View.GONE);
                mRd_search_ll_center.setBackgroundResource(R.color.ux_color_translucent);
                mRd_search_ll_bottom.setVisibility(View.VISIBLE);
                mBottom_ll_shadow.setVisibility(View.VISIBLE);
            }

            @Override
            public void onAnimationRepeat(Animation animation) {

            }
        });
        animationL2R.setStartOffset(0);
        mViewCenterRight.startAnimation(animationL2R);
    }

    void onThemeChange(String type, int color) {
        if (mSearchSettingsWindow != null) {
            mSearchSettingsWindow.dismiss();
            mSearchSettingsWindow = null;
        }
        if (mTop_bt_cancel != null)
            mTop_bt_cancel.setTextColor(ThemeUtil.getPrimaryTextColor(mContext));
        if (mWholeWordsSelectLayout != null)
            ThemeUtil.setBackgroundTintList(mWholeWordsSelectLayout, getSelectedButtonColorStateList(mContext));
        if (mCaseSensitiveSelectLayout != null)
            ThemeUtil.setBackgroundTintList(mCaseSensitiveSelectLayout, getSelectedButtonColorStateList(mContext));
        UIBtnImageView noContentImage = mSearchView.findViewById(R.id.iv_no_content);
        noContentImage.setColorStateList(ThemeUtil.getItemIconColor(mContext));
        ThemeUtil.setTintList(mSelectAllCheckBox, ThemeUtil.getCheckboxColor(mContext));
        mMarkRedactionButton.setBackgroundColor(mContext.getResources().getColor(R.color.ui_color_grey_f4f4f4));
//        mMarkRedactionButton.setTextColor(AppResource.getColor(mContext, R.color.main_color));
        int disabled = ThemeConfig.getInstance(mContext).getI2();
        int normal = ThemeConfig.getInstance(mContext).getPrimaryColor();
        mMarkRedactionButton.setTextColor(AppResource.createColorStateList(mContext, disabled, normal));
        mAdapterSearch.notifyUpdateData();

        ThemeUtil.setTintList(ivSearch, ThemeUtil.getPrimaryIconColor(mContext));
        ThemeUtil.setTintList(mIvSearchSettings, ThemeUtil.getPrimaryIconColor(mContext));
        ThemeUtil.setTintList(mTop_iv_clear, ThemeUtil.getPrimaryIconColor(mContext));

        ThemeUtil.setTintList(mBottom_iv_prev, ThemeUtil.getPrimaryIconColor(mContext));
        ThemeUtil.setTintList(mBottom_iv_next, ThemeUtil.getPrimaryIconColor(mContext));
        ThemeUtil.setTintList(mBottom_iv_result, ThemeUtil.getPrimaryIconColor(mContext));

        mRd_search_ll_top.setBackgroundColor(ThemeConfig.getInstance(mContext).getB2());
        mRlTopLeft.setBackground(AppResource.getDrawable(mContext, R.drawable.shape_search_bg));
        mTop_et_content.setHintTextColor(AppResource.getColor(mContext, R.color.t3));
        mTop_et_content.setTextColor(AppResource.getColor(mContext, R.color.t4));
        mRd_search_ll_center.setBackgroundColor(AppResource.getColor(mContext, R.color.ux_color_translucent));
        mViewCenterRight.setBackgroundColor(AppResource.getColor(mContext, R.color.b1));
        mSearchCenterTop.setBackgroundColor(AppResource.getColor(mContext, R.color.b2));
        mCenter_tv_total_number.setTextColor(AppResource.getColor(mContext, R.color.t3));
        mTvNoContent.setTextColor(AppResource.getColor(mContext, R.color.t2));
        mRd_search_ll_bottom.setBackgroundColor(ThemeConfig.getInstance(mContext).getB2());
        mTopSearchDivider.setBackgroundColor(AppResource.getColor(mContext, R.color.p1));
        mBottomIvSearchDivider.setBackgroundColor(AppResource.getColor(mContext, R.color.p1));
        if (mSearchSettingsWindow != null) {
            mTvSearchInInternet.setTextColor(AppResource.getColor(mContext, R.color.t4));
            mTvSearchCaseSensitive.setTextColor(AppResource.getColor(mContext, R.color.t4));
            mTvSearchWholeWords.setTextColor(AppResource.getColor(mContext, R.color.t4));
            ThemeUtil.setBackgroundTintList(mWholeWordsSelectLayout, getSelectedButtonColorStateList(mContext));
            ThemeUtil.setBackgroundTintList(mCaseSensitiveSelectLayout, getSelectedButtonColorStateList(mContext));
        }
    }

    public void setRedactSearch(boolean redactSearch) {
        isRedactSearch = redactSearch;
        mAdapterSearch.setSelectable(redactSearch);
    }

    //search content changed "before" "on" and "after"
    private class myTextWatcher implements TextWatcher {
        @Override
        public void onTextChanged(CharSequence s, int start, int before, int count) {
        }

        @Override
        public void beforeTextChanged(CharSequence s, int start, int count, int after) {
        }

        @Override
        public void afterTextChanged(Editable s) {
            if (s.length() > 0) {
                mTop_iv_clear.setVisibility(View.VISIBLE);
                mIsBlank = false;
            } else {
                mTop_iv_clear.setVisibility(View.INVISIBLE);
                mIsBlank = true;
            }
        }
    }

    private SearchCancelListener mSearchCancelListener = null;

    public void setSearchCancelListener(SearchCancelListener listener) {
        mSearchCancelListener = listener;
    }

    public interface SearchCancelListener {
        void onSearchCancel();
    }

    public void startSearch(String text) {
        mTop_et_content.setText(text);
        mTop_et_content.setSelection(mTop_et_content.getText().length());
        startSearch();
    }
    private void startSearch() {
        mSearchCurrentPage = mPdfViewCtrl.getCurrentPage();
        ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).triggerDismissMenuEvent();
        if (!mIsBlank) {
            AppUtil.dismissInputSoft(mTop_et_content);
            AppThreadManager.getInstance().getMainThreadHandler().postDelayed(new Runnable() {
                @Override
                public void run() {

                    mSearch_content = mTop_et_content.getText().toString();
                    if (!AppUtil.isEmpty(mSearch_content)) {
                        if (mRd_search_ll_bottom.getVisibility() == View.VISIBLE) {
                            mRd_search_ll_bottom.setVisibility(View.GONE);
                            mBottom_ll_shadow.setVisibility(View.GONE);
                        }
                        if (mViewCenterRight.getVisibility() == View.VISIBLE) {
                            mIsCancel = false;
                            searchText(mSearch_content, mSearchFlags);
                        } else {
                            mRd_search_ll_center.setBackgroundResource(R.color.ux_color_mask_background);
                            mViewCenterLeft.setVisibility(View.VISIBLE);
                            mViewCenterLeft.setClickable(false);
                            mViewCenterRight.setVisibility(View.VISIBLE);
                            Animation animationR2L = AnimationUtils.loadAnimation(mContext, R.anim.view_anim_rtol_show);
                            animationR2L.setAnimationListener(new Animation.AnimationListener() {
                                @Override
                                public void onAnimationStart(Animation animation) {
                                }

                                @Override
                                public void onAnimationEnd(Animation animation) {
                                    mViewCenterLeft.setClickable(true);
                                    mIsCancel = false;
                                    searchText(mSearch_content, mSearchFlags);
                                }

                                @Override
                                public void onAnimationRepeat(Animation animation) {
                                }
                            });
                            animationR2L.setStartOffset(300);
                            mViewCenterRight.startAnimation(animationR2L);
                        }
                    }
                }
            }, 100);
        }
    }

    private final View.OnKeyListener mySearchListener = new View.OnKeyListener() {

        @Override
        public boolean onKey(View v, int keyCode, KeyEvent event) {
            if (KeyEvent.KEYCODE_ENTER == keyCode && event.getAction() == KeyEvent.ACTION_DOWN) {
                startSearch();
                return true;
            }
            return false;
        }
    };

    protected void onKeyBack() {
        cancel();
    }

    protected void cancel() {
        mRd_search_ll_center.setBackgroundResource(R.color.ux_color_translucent);
        mViewCenterLeft.setVisibility(View.GONE);
        mViewCenterRight.setVisibility(View.GONE);

        mIsCancel = true;
        mIsBlank = true;
        searchCancel();

        mSearchView.setVisibility(View.INVISIBLE);
        cancelSearchText();
        AppUtil.dismissInputSoft(mTop_et_content);
        if (mSearchCancelListener != null)
            mSearchCancelListener.onSearchCancel();
        setRedactSearch(false);
        clearSearchResult();
    }

    protected void cleanSearch() {
        mIsCancel = true;
        mIsBlank = true;
        searchCancel();
        mSearchView.setVisibility(View.INVISIBLE);
        mRd_search_ll_bottom.setVisibility(View.GONE);
        AppUtil.dismissInputSoft(mTop_et_content);
    }

    protected void setVisibility(int visibility) {
        mSearchView.setVisibility(visibility);
    }

    public void launchSearchView() {
        initView();
        mIsCancel = false;
        bCancelSearchText = false;
        mRect = null;
        if (mTop_et_content.getText().length() > 0) {
            mTop_et_content.selectAll();
            mTop_iv_clear.setVisibility(View.VISIBLE);
        }

        mTop_et_content.requestFocus();
        mTop_et_content.setFocusable(true);
        AppUtil.showSoftInput(mTop_et_content);
    }

    public void launchRedactSearchView() {
        setRedactSearch(true);
        launchSearchView();
    }

    private final View.OnClickListener searchModelListener = new View.OnClickListener() {

        @Override
        public void onClick(View v) {
            ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).triggerDismissMenuEvent();
            if (v.getId() == R.id.top_iv_clear) {
                searchCancel();
            } else if (v.getId() == R.id.top_bt_cancel) {
                cancel();
            } else if (v.getId() == R.id.rd_search_center_left) {
                if (isFastDoubleClick()) {
                    return;
                }
                AppUtil.dismissInputSoft(mTop_et_content);
                Animation animationL2R = AnimationUtils.loadAnimation(mContext, R.anim.view_anim_rtol_hide);
                animationL2R.setAnimationListener(new Animation.AnimationListener() {
                    @Override
                    public void onAnimationStart(Animation animation) {
                    }

                    @Override
                    public void onAnimationEnd(Animation animation) {
                        mRd_search_ll_center.setBackgroundResource(R.color.ux_color_translucent);
                        mViewCenterLeft.setVisibility(View.GONE);
                        mViewCenterRight.setVisibility(View.GONE);
                        mRd_search_ll_bottom.setVisibility(View.VISIBLE);
                        mBottom_ll_shadow.setVisibility(View.VISIBLE);
                    }

                    @Override
                    public void onAnimationRepeat(Animation animation) {
                    }
                });
                animationL2R.setStartOffset(0);
                mViewCenterRight.startAnimation(animationL2R);
            } else if (v.getId() == R.id.bottom_iv_result) {
                mRd_search_ll_bottom.setVisibility(View.GONE);
                mBottom_ll_shadow.setVisibility(View.GONE);
                mRd_search_ll_center.setBackgroundResource(R.color.ux_color_mask_background);
                mViewCenterLeft.setVisibility(View.VISIBLE);
                mViewCenterRight.setVisibility(View.VISIBLE);
                Animation animationR2L = AnimationUtils.loadAnimation(mContext, R.anim.view_anim_rtol_show);
                animationR2L.setAnimationListener(new Animation.AnimationListener() {
                    @Override
                    public void onAnimationStart(Animation animation) {
                    }

                    @Override
                    public void onAnimationEnd(Animation animation) {
                    }

                    @Override
                    public void onAnimationRepeat(Animation animation) {
                    }
                });
                animationR2L.setStartOffset(0);
                mViewCenterRight.startAnimation(animationR2L);

            } else if (v.getId() == R.id.bottom_iv_prev) {
                searchPre();
            } else if (v.getId() == R.id.bottom_iv_next) {
                searchNext();
            } else if (v.getId() == R.id.top_search_settings) {
                if (AppUtil.isFastDoubleClick()) return;
                showSearchSettings();
            }
        }
    };

    private void showSearchSettings() {
        UIExtensionsManager extensionsManager = (UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager();
        if (extensionsManager == null) return;

        Activity activity = extensionsManager.getAttachedActivity();
        if (mSearchSettingsWindow == null) {
            mSearchSettingsWindow = (UIActionMenu) UIActionMenu.newInstance((FragmentActivity) activity);
            mSearchSettingsWindow.setContentView(getSettingsView());
            mSearchSettingsWindow.setAutoResetSystemUiOnDismiss(false);
            mSearchSettingsWindow.setAutoResetSystemUiOnShow(false);
        }
        Rect rect = new Rect();
        mIvSearchSettings.getGlobalVisibleRect(rect);
        mSearchSettingsWindow.show(extensionsManager.getRootView(), rect, UIPopoverWin.ARROW_TOP, 0);
    }

    private void initSearchSettings() {
        UIExtensionsManager extensionsManager = (UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager();
        if (extensionsManager == null) return;

        Activity activity = extensionsManager.getAttachedActivity();
        if (mSearchSettingsWindow == null) {
            mSearchSettingsWindow = (UIActionMenu) UIActionMenu.newInstance((FragmentActivity) activity);
            mSearchSettingsWindow.setContentView(getSettingsView());
            mSearchSettingsWindow.setAutoResetSystemUiOnDismiss(false);
            mSearchSettingsWindow.setAutoResetSystemUiOnShow(false);
        }
    }

    private void searchCancel() {
        mTop_et_content.setText("");
        mTop_iv_clear.setVisibility(View.INVISIBLE);
    }

    private static long sLastTimeMillis;

    private boolean isFastDoubleClick() {
        long currentTimeMillis = System.currentTimeMillis();
        long delta = currentTimeMillis - sLastTimeMillis;
        if (Math.abs(delta) < 500) {
            return true;
        }
        sLastTimeMillis = currentTimeMillis;
        return false;
    }

    protected int getScreenWidth() {
        return mMetrics.widthPixels;
    }

    protected int getScreenHeight() {
        return mMetrics.heightPixels;
    }

    //search text function
    public interface TaskResult<T1, T2, T3> {
        void onResult(int errCode, T1 p1, T2 p2, T3 p3);

        void setTag(long tag);

        long getTag();
    }

    public static class SearchPageInfo {
        public int mErrCode;
        public int mPageIndex;
        public int mFlag;
        public String mPattern;
        public ArrayList<SearchResult> mSearchResults;
    }

    public static class SearchResult extends BaseBean {
        public final static int FLAG_SEARCH_TAG = 0;
        public final static int FLAG_SEARCH_CONTENT = 1;

        public int mPageIndex;
        public String mSentence;
        public int mPatternStart;
        public ArrayList<RectF> mRects;
        public boolean mSelected;

        public SearchResult(int pageIndex, int flag, String sentence, int patternStart) {
            mPageIndex = pageIndex;
            mSentence = sentence;
            mPatternStart = patternStart;
            mRects = new ArrayList<>();
            setFlag(flag);
        }

        public RectFArray getFxRectFArray() {
            RectFArray array = new RectFArray();
            for (RectF rectF : mRects) {
                array.add(AppUtil.toFxRectF(rectF));
            }
            return array;
        }
    }

    private CompositeDisposable mCompositeDisposable;
    private Disposable mDisposable;

    private void searchPageTask(int pageIndex, int flag, long taskId, String pattern,
                                final TaskResult<Integer, String, ArrayList<SearchResult>> taskResult) {
        if (mDisposable != null) {
            mCompositeDisposable.remove(mDisposable);
            mDisposable = null;
        }

        mDisposable = _searchPageTask(pageIndex, flag, taskId, pattern)
                .subscribeOn(Schedulers.io())
                .observeOn(AndroidSchedulers.mainThread())
                .subscribe(new Consumer<SearchPageInfo>() {
                    @Override
                    public void accept(SearchPageInfo searchInfo) throws Exception {
                        if (searchInfo.mErrCode == Constants.e_ErrSuccess) {
                            taskResult.onResult(searchInfo.mErrCode, searchInfo.mPageIndex,
                                    searchInfo.mPattern, searchInfo.mSearchResults);
                        } else {
                            setTotalNumber(mValueResultList.size());
                        }
                    }
                }, new Consumer<Throwable>() {
                    @Override
                    public void accept(Throwable throwable) throws Exception {
                        setTotalNumber(mValueResultList.size());
                    }
                });

        if (mCompositeDisposable == null)
            mCompositeDisposable = new CompositeDisposable();
        mCompositeDisposable.add(mDisposable);
    }

    private Single<SearchPageInfo> _searchPageTask(final int pageIndex, final int flag, final long taskId, final String pattern) {
        return Single.fromCallable(new Callable<SearchPageInfo>() {
            @Override
            public SearchPageInfo call() throws Exception {
                return searchPage(pageIndex, flag, taskId, pattern);
            }
        });
    }

    private SearchPageInfo searchPage(int pageIndex, int flag, long taskId, String pattern) {
        int errCode = Constants.e_ErrSuccess;
        ArrayList<SearchResult> searchResults = new ArrayList<>();
        PDFDoc document = mPdfViewCtrl.getDoc();
        try {
            SearchCancelCallback cancelCallback = new SearchCancelCallback() {
                @Override
                public boolean needToCancelNow() {
                    return true;
                }
            };
            TextSearch textSearch = new TextSearch(document, cancelCallback, TextPage.e_ParseTextNormal);

            textSearch.setStartPage(pageIndex);
            textSearch.setPattern(pattern, false);
            textSearch.setSearchFlags(flag);

            boolean bRet = textSearch.findNext();
            while (bRet) {
                if (mIsCancel || mSearchId != taskId) {
                    errCode = -1;
                    break;
                }
                if (textSearch.getMatchPageIndex() != pageIndex) {
                    break;
                }

                String sentence = textSearch.getMatchSentence();
                if (sentence == null) sentence = "";
                int sentencePos = textSearch.getMatchSentenceStartIndex();
                SearchResult searchResult = new SearchResult(pageIndex, SearchResult.FLAG_SEARCH_CONTENT, sentence, sentencePos);
                RectFArray rectFArray = textSearch.getMatchRects();
                for (int i = 0; i < rectFArray.getSize(); i++) {
                    searchResult.mRects.add(AppUtil.toRectF(rectFArray.getAt(i)));
                }
                searchResult.mSelected = true;
                mSelectedResults.add(searchResult);
                searchResults.add(searchResult);

                bRet = textSearch.findNext();
            }
            textSearch.delete();
            cancelCallback.delete();
        } catch (PDFException e) {
            errCode = e.getLastError();
        }

        SearchPageInfo searchPageInfo = new SearchPageInfo();
        searchPageInfo.mErrCode = errCode;
        searchPageInfo.mPageIndex = pageIndex;
        searchPageInfo.mFlag = flag;
        searchPageInfo.mPattern = pattern;
        searchPageInfo.mSearchResults = searchResults;
        return searchPageInfo;
    }

    private String mSearchText = null;
    private boolean bCancelSearchText = true;
    private int mCurrentPosition = -1;
    private float mCurrentPageX;
    private float mCurrentPageY;
    private float mCurrentSearchR;
    private float mCurrentSearchB;
    private final ArrayList<SearchResult> mTagResultList = new ArrayList<>();
    private final ArrayList<SearchResult> mValueResultList = new ArrayList<>();
    private final ArrayList<SearchResult> mShowResultList = new ArrayList<>();
    private final ArrayList<SearchResult> mSearchBeforeResultList = new ArrayList<>();
    private final ArrayList<SearchResult> mTagBeforeList = new ArrayList<>();
    private final ArrayList<SearchResult> mValuesBeforeList = new ArrayList<>();
    private int mSearchCurrentPage;
    private void _searchText(final String pattern, final int flag, final int pageIndex) {
        this.mSearchText = pattern.trim();
        TaskResult<Integer, String, ArrayList<SearchResult>> taskResult;
        searchPageTask(pageIndex, flag, mSearchId, pattern.trim(), taskResult = new TaskResult<Integer, String, ArrayList<SearchResult>>() {
            private long mTaskId;

            @Override
            public void onResult(int errCode, Integer p1, String p2, ArrayList<SearchResult> p3) {
                if (errCode == Constants.e_ErrOutOfMemory) {
                    mPdfViewCtrl.recoverForOOM();
                    setTotalNumber(mValueResultList.size());
                    return;
                }

                if (this.mTaskId != mSearchId) {
                    setTotalNumber(mValueResultList.size());
                    return;
                }

                if (p3 == null) {
                    setTotalNumber(mValueResultList.size());
                    return;
                }
                if (p3.size() > 0) {
                    int positionStart = mShowResultList.size();
                    SearchResult searchResult = new SearchResult(p1, SearchResult.FLAG_SEARCH_TAG, "tag", p3.size());
                    int insertCount = 0;

                    if (pageIndex < mSearchCurrentPage) {
                        positionStart = mSearchBeforeResultList.size();
                        mTagResultList.add(mTagBeforeList.size(),searchResult);
                        mShowResultList.add(mSearchBeforeResultList.size(),searchResult);
                        mSearchBeforeResultList.add(searchResult);
                        insertCount += 1;

                        mValueResultList.addAll(mValuesBeforeList.size(),p3);
                        mShowResultList.addAll(mSearchBeforeResultList.size(),p3);
                        mTagBeforeList.add(searchResult);
                        mValuesBeforeList.add(searchResult);
                        insertCount += p3.size();

                        mSearchBeforeResultList.addAll(p3);
                    }else{

                        mTagResultList.add(searchResult);
                        mShowResultList.add(searchResult);
                        insertCount += 1;
                        mValueResultList.addAll(p3);
                        mShowResultList.addAll(p3);
                        insertCount += p3.size();
                    }

                    if (mAdapterSearch != null){
                        mAdapterSearch.notifyItemRangeInserted(positionStart, insertCount);
                    }
                }

                setSearchNumber(mValueResultList.size());

                int count = 0;
                if(mSearchCurrentPage > 0){
                    count = mSearchCurrentPage - 1;
                }else{
                    count = mPdfViewCtrl.getPageCount() - 1;
                }

                if (pageIndex == count) {
                    setTotalNumber(mValueResultList.size());
                    mSelectAllCheckBox.setChecked(mSelectedResults.size() == mValueResultList.size());
                    mMarkRedactionButton.setEnabled(mSelectAllCheckBox.isChecked());
                    if (mCurrentPosition == -1 && mShowResultList.size() > 0) {
                        mCurrentPosition = mShowResultList.size() - 1;
                        mPageIndex = mShowResultList.get(mCurrentPosition).mPageIndex;
                        mRect = mShowResultList.get(mCurrentPosition).mRects;
                        setToolbarIcon();
                        refreshFoundRectList();
                        mPdfViewCtrl.invalidate();
                    }
                    return;
                }
                if (p1 >= mSearchCurrentPage && mCurrentPosition == -1 && p3.size() > 0) {
                    mCurrentPosition = mShowResultList.size() - p3.size();
                    mPageIndex = mShowResultList.get(mCurrentPosition).mPageIndex;
                    mRect = mShowResultList.get(mCurrentPosition).mRects;
                    setToolbarIcon();
                    refreshFoundRectList();
                    mPdfViewCtrl.invalidate();
                }
                setToolbarIcon();
                if (bCancelSearchText) {
                    bCancelSearchText = false;
                    setTotalNumber(mValueResultList.size());
                } else {
                    int index = pageIndex + 1;
                    if (pageIndex == mPdfViewCtrl.getPageCount() - 1 && mSearchCurrentPage > 0) {
                        index = 0;
                    }
                    _searchText(pattern, flag, index);
                }
            }

            @Override
            public void setTag(long taskId) {
                mTaskId = taskId;
            }

            @Override
            public long getTag() {
                return mTaskId;
            }
        });
        taskResult.setTag(mSearchId);
    }

    private void searchText(String pattern, int flag) {
        cancelSearchText();
        clearSearchResult();
        mCurrentPosition = -1;
        mSearchId++;
        mRect = null;
        mIsCancel = false;
        mSearchText = null;
        synchronized (this) {
            bCancelSearchText = false;
        }
        if (TextUtils.isEmpty(pattern.trim())) {
            setTotalNumber(0);
            return;
        }

        setSearchNumber(0);
        _searchText(pattern, flag, mSearchCurrentPage);
    }

    private void cancelSearchText() {
        synchronized (this) {
            if (!bCancelSearchText) {
                bCancelSearchText = true;

                // do cancel search text
                onCancelSearchText();
            }
        }
    }

    private void notifyDataSetChangedSearchAdapter() {
        if (mAdapterSearch != null) {
            mAdapterSearch.notifyDataSetChanged();
        }
    }

    private void clearSearchResult() {
        mShowResultList.clear();
        mTagResultList.clear();
        mValueResultList.clear();
        mSelectedResults.clear();
        mSearchFoundRectList.clear();
        mSearchBeforeResultList.clear();
        mTagBeforeList.clear();
        mValuesBeforeList.clear();

        releaseDisposable();
        notifyDataSetChangedSearchAdapter();
    }

    private void releaseDisposable() {
        if (mCompositeDisposable != null) {
            if (mDisposable != null) {
                mCompositeDisposable.remove(mDisposable);
                mDisposable = null;
            }
            mCompositeDisposable.clear();
            mCompositeDisposable = null;
        }
    }

    private void onCancelSearchText() {
        mRd_search_ll_bottom.setVisibility(View.GONE);
        mPdfViewCtrl.invalidate();
    }

    private Rect getVisibleWidth() {
        Rect rect = new Rect();
        mPdfViewCtrl.getGlobalVisibleRect(rect);
        return rect;
    }

    private void searchPre() {
        if (mSearchText == null || bCancelSearchText) {
            return;
        }

        if (mCurrentPosition <= 1) {
            mPageIndex = mShowResultList.get(mCurrentPosition).mPageIndex;
            mRect = mShowResultList.get(mCurrentPosition).mRects;
            setToolbarIcon();
            mPdfViewCtrl.invalidate();
            return;
        }
        mCurrentPosition--;
        if (mShowResultList.get(mCurrentPosition).mSentence.endsWith("tag")) {
            mCurrentPosition--;
        }
        setCurrentPageX();
        RectF rectF = new RectF(mCurrentPageX, mCurrentPageY, mCurrentSearchR, mCurrentSearchB);
        RectF canvasRectF = new RectF();
        boolean transSuccess = mPdfViewCtrl.convertPageViewRectToDisplayViewRect(rectF, canvasRectF, mShowResultList.get(mCurrentPosition).mPageIndex);
        int screenWidth = getVisibleWidth().width();
        int screenHeight = getVisibleWidth().height();
        if (!transSuccess || canvasRectF.left < 0 || canvasRectF.right > screenWidth || canvasRectF.top < 0 || canvasRectF.bottom > screenHeight) {
            int x = (int) (mCurrentPageX - getScreenWidth() / 4.0f);
            int y = (int) (mCurrentPageY - getScreenHeight() / 4.0f);
            mPdfViewCtrl.gotoPage(mShowResultList.get(mCurrentPosition).mPageIndex, x, y);
        }
        mPageIndex = mShowResultList.get(mCurrentPosition).mPageIndex;
        refreshFoundRectList();
        mRect = mShowResultList.get(mCurrentPosition).mRects;
        setToolbarIcon();
        mPdfViewCtrl.invalidate();
    }

    private void searchNext() {
        if (mSearchText == null || bCancelSearchText) {
            return;
        }

        if (mCurrentPosition >= mShowResultList.size() - 1) {
            mPageIndex = mShowResultList.get(mCurrentPosition).mPageIndex;
            mRect = mShowResultList.get(mCurrentPosition).mRects;
            setToolbarIcon();
            mPdfViewCtrl.invalidate();
            return;
        }
        mCurrentPosition++;
        if (mShowResultList.get(mCurrentPosition).mSentence.endsWith("tag")) {
            mCurrentPosition++;
        }
        setCurrentPageX();
        RectF rectF = new RectF(mCurrentPageX, mCurrentPageY, mCurrentSearchR, mCurrentSearchB);
        RectF canvasRectF = new RectF();
        boolean transSuccess = mPdfViewCtrl.convertPageViewRectToDisplayViewRect(rectF, canvasRectF, mShowResultList.get(mCurrentPosition).mPageIndex);
        int screenWidth = getVisibleWidth().width();
        int screenHeight = getVisibleWidth().height();
        if (!transSuccess || canvasRectF.left < 0 || canvasRectF.right > screenWidth || canvasRectF.top < 0 || canvasRectF.bottom > screenHeight) {
            int x = (int) (mCurrentPageX - getScreenWidth() / 4.0f);
            int y = (int) (mCurrentPageY - getScreenHeight() / 4.0f);
            mPdfViewCtrl.gotoPage(mShowResultList.get(mCurrentPosition).mPageIndex, x, y);
        }
        mPageIndex = mShowResultList.get(mCurrentPosition).mPageIndex;
        refreshFoundRectList();
        mRect = mShowResultList.get(mCurrentPosition).mRects;
        setToolbarIcon();
        mPdfViewCtrl.invalidate();
    }

    private boolean isFirstSearchResult() {
        return mCurrentPosition <= 1;
    }

    private boolean isLastSearchResult() {
        return mCurrentPosition < 1 || mCurrentPosition >= mShowResultList.size() - 1;
    }

    private void setCurrentPageX() {
        float x = 0, y = 0, r = 0, b = 0;
        for (int i = 0; i < mShowResultList.get(mCurrentPosition).mRects.size(); i++) {
            RectF pageRect = new RectF(mShowResultList.get(mCurrentPosition).mRects.get(i));
            RectF pageViewRect = new RectF();
            if (mPdfViewCtrl.convertPdfRectToPageViewRect(pageRect, pageViewRect, mShowResultList.get(mCurrentPosition).mPageIndex)) {
                if (i == 0) {
                    x = pageViewRect.left;
                    y = pageViewRect.top;
                    r = pageViewRect.right;
                    b = pageViewRect.bottom;
                } else {
                    if (pageViewRect.left < x) {
                        x = pageViewRect.left;
                    }
                    if (pageViewRect.top < y) {
                        y = pageViewRect.top;
                    }
                    if (pageViewRect.right > r) {
                        r = pageViewRect.right;
                    }
                    if (pageViewRect.bottom > b) {
                        b = pageViewRect.bottom;
                    }
                }
            }
        }
        mCurrentPageX = x;
        mCurrentPageY = y;
        mCurrentSearchR = r;
        mCurrentSearchB = b;
    }

    class SearchAdapter extends SuperAdapter<SearchResult> {
        private boolean mSelectable = false;
        private ISearchItemClickListener mItemClickListener;

        public SearchAdapter(Context context) {
            super(context);
        }

        @Override
        public void notifyUpdateData() {
            notifyDataSetChanged();
        }

        @Override
        public SearchResult getDataItem(int position) {
            return mShowResultList.get(position);
        }

        @NonNull
        @Override
        public SuperViewHolder onCreateViewHolder(@NonNull ViewGroup parent, int viewType) {
            SuperViewHolder viewHolder;
            if (viewType == SearchResult.FLAG_SEARCH_TAG) {
                viewHolder = new SearchPageTagViewHolder(inflateLayout(getContext(), parent, R.layout.search_item_tag));
            } else {
                viewHolder = new SearchContentViewHolder(inflateLayout(getContext(), parent, R.layout.search_item_content));
            }
            return viewHolder;
        }

        private View inflateLayout(Context context, ViewGroup parent, int layoutId) {
            return LayoutInflater.from(context).inflate(layoutId, parent, false);
        }

        @Override
        public int getItemCount() {
            return mShowResultList.size();
        }

        @Override
        public int getItemViewType(int position) {
            return mShowResultList.get(position).getFlag();
        }

        class SearchPageTagViewHolder extends SuperViewHolder {
            public FrameLayout search_bg;
            public TextView search_pageIndex;
            public TextView search_pageCount;

            public SearchPageTagViewHolder(View itemView) {
                super(itemView);
                search_bg = itemView.findViewById(R.id.rd_search_content_rl);
                search_pageIndex = itemView.findViewById(R.id.search_page_tv);
                search_pageCount = itemView.findViewById(R.id.search_curpage_count);
                if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.Q) {
                    search_bg.setForceDarkAllowed(false);
                    search_pageIndex.setForceDarkAllowed(false);
                    search_pageCount.setForceDarkAllowed(false);
                }
            }

            @Override
            public void bind(BaseBean data, int position) {
                SearchResult result = mShowResultList.get(position);
                search_bg.setBackgroundColor(AppResource.getColor(mContext,R.color.b2));
                search_pageIndex.setTextColor(AppResource.getColor(mContext,R.color.t3));
                search_pageCount.setTextColor(AppResource.getColor(mContext,R.color.t3));
                String pageNumber = AppResource.getString(mContext, R.string.search_page_number, result.mPageIndex + 1);
                search_pageIndex.setText(pageNumber);
                search_pageCount.setText(result.mPatternStart + "");
            }

            @Override
            public void onClick(View v) {
            }
        }

        class SearchContentViewHolder extends SuperViewHolder {
            private TextView search_content;
            private CheckBox selectCheckbox;

            public SearchContentViewHolder(View itemView) {
                super(itemView);
                search_content = itemView.findViewById(R.id.search_content_tv);

                if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.Q) {
                    search_content.setForceDarkAllowed(false);
                }
                selectCheckbox = itemView.findViewById(R.id.select_checkbox);


                if (isRedactSearch) {
                    selectCheckbox.setOnClickListener(this);
                } else {
                    selectCheckbox.setOnClickListener(null);
                }
            }

            @Override
            public void bind(BaseBean data, int position) {
                ThemeUtil.setTintList(selectCheckbox, ThemeUtil.getCheckboxColor(mContext));
                search_content.setTextColor(AppResource.getColor(mContext, R.color.t4));
                SearchResult result = mShowResultList.get(position);
                selectCheckbox.setVisibility(mSelectable ? View.VISIBLE : View.GONE);
                selectCheckbox.setChecked(result.mSelected);
                selectCheckbox.setTag(position);
                String mContent = result.mSentence;
                SpannableString searchContent = new SpannableString(mContent);

                String matchText = mSearchText.replaceAll("\r", " ");
                matchText = matchText.replaceAll("\n", " ");
                matchText = matchText.replaceAll("\\s+", " ");

                int patternStart = Math.max(0, result.mPatternStart);
                if (matchText.length() > mContent.length()) {
                    matchText = mContent.substring(patternStart);
                }

                try {
                    Pattern pattern = Pattern.compile(matchText, Pattern.CASE_INSENSITIVE);//ignore the case
                    Matcher matcher = pattern.matcher(searchContent);
                    boolean  isFind = false;
                        while (matcher.find()) {
                            isFind = true;
                            searchContent.setSpan(new BackgroundColorSpan(AppResource.getColor(mContext, R.color.ux_search_found_color)),
                                    patternStart,
                                    patternStart + matchText.length(),
                                    Spanned.SPAN_EXCLUSIVE_EXCLUSIVE);
                            searchContent.setSpan(new ForegroundColorSpan(AppResource.getColor(mContext, R.color.ux_color_ff2e2e2e)),
                                    patternStart,
                                    patternStart + matchText.length(),
                                    Spanned.SPAN_EXCLUSIVE_EXCLUSIVE);
                        }

                        if(!isFind) {
                            try {
                                PDFPage pdfPage = mPdfViewCtrl.getDoc().getPage(result.mPageIndex);
                                if(!pdfPage.isParsed()){
                                    Progressive progressive = pdfPage.startParse(PDFPage.e_ParsePageNormal, null, false);
                                        int state = Progressive.e_ToBeContinued;

                                        while (state == Progressive.e_ToBeContinued) {
                                            state = progressive.resume();
                                    }
                                }
                                TextPage textPage = new TextPage(pdfPage, TextPage.e_ParseTextNormal);
                                for (int i = 0; i < result.getFxRectFArray().getSize(); i++) {
                                    com.foxit.sdk.common.fxcrt.RectF fxRectF = result.getFxRectFArray().getAt(i);
                                    String text = textPage.getTextInRect(fxRectF);
                                    int firstIndex = result.mSentence.indexOf(text);
                                    searchContent.setSpan(new BackgroundColorSpan(AppResource.getColor(mContext, R.color.ux_search_found_color)),
                                            firstIndex,
                                            firstIndex + text.length(),
                                            Spanned.SPAN_EXCLUSIVE_EXCLUSIVE);
                                    searchContent.setSpan(new ForegroundColorSpan(AppResource.getColor(mContext, R.color.ux_color_ff2e2e2e)),
                                            firstIndex,
                                            firstIndex + text.length(),
                                            Spanned.SPAN_EXCLUSIVE_EXCLUSIVE);
                                }
                            } catch (PDFException e) {
                                e.printStackTrace();
                            }
                        }
                    search_content.setText(searchContent);
                } catch (PatternSyntaxException e) {
                    String splitContent = searchContent.subSequence(patternStart, patternStart + matchText.length()).toString();
                    if (splitContent.equalsIgnoreCase(matchText)) {
                        searchContent.setSpan(new BackgroundColorSpan(AppResource.getColor(mContext, R.color.ux_search_found_color)),
                                patternStart,
                                patternStart + matchText.length(),
                                Spanned.SPAN_EXCLUSIVE_EXCLUSIVE);
                        searchContent.setSpan(new ForegroundColorSpan(AppResource.getColor(mContext, R.color.ux_color_ff2e2e2e)),
                                patternStart,
                                patternStart + matchText.length(),
                                Spanned.SPAN_EXCLUSIVE_EXCLUSIVE);
                    }
                    search_content.setText(searchContent);
                }
            }

            @Override
            public void onClick(View v) {
                int position = getAdapterPosition();
                if (position == -1) return;
                SearchResult searchResult = mShowResultList.get(position);

                int id = v.getId();
                if (id == R.id.rd_search_item_content_ll) {
                    if (mItemClickListener != null) {
                        mItemClickListener.onItemClick(position, searchResult);
                    }
                } else if (id == R.id.select_checkbox){
                    searchResult.mSelected = !searchResult.mSelected;
                    if (searchResult.mSelected) {
                        if (!mSelectedResults.contains(searchResult))
                            mSelectedResults.add(searchResult);
                    } else {
                        mSelectedResults.remove(searchResult);
                    }
                    mSelectAllCheckBox.setChecked(mSelectedResults.size() == mValueResultList.size());
                    mMarkRedactionButton.setEnabled(mSelectedResults.size() > 0);
                }
            }
        }

        public void setOnItemClickListener(ISearchItemClickListener itemClickListener) {
            mItemClickListener = itemClickListener;
        }

        public void setSelectAll(boolean selectAll) {
            mSelectedResults.clear();
            SearchResult result;
            for (int i = 0; i < mShowResultList.size(); i++) {
                result = mShowResultList.get(i);
                if ("tag".equals(result.mSentence))
                    continue;
                result.mSelected = selectAll;
                if (result.mSelected) {
                    mSelectedResults.add(result);
                } else {
                    mSelectedResults.remove(result);
                }
            }
            notifyUpdateData();
        }

        public void setSelectable(boolean selectable) {
            mSelectable = selectable;
        }
    }

    protected void onDocumentClosed() {
        mTop_et_content.setText("");
        clearSearchResult();
        mCenter_tv_total_number.setText("");
    }

    interface ISearchItemClickListener {
        void onItemClick(int position, SearchResult result);
    }

}
