/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.modules;

import android.app.Activity;
import android.content.Context;
import android.graphics.Point;
import android.graphics.Rect;
import android.os.Handler;
import android.view.View;
import android.widget.PopupWindow;
import android.widget.RelativeLayout;

import com.foxit.sdk.PDFException;
import com.foxit.sdk.PDFViewCtrl;
import com.foxit.sdk.pdf.PDFPage;
import com.foxit.sdk.pdf.annots.Annot;
import com.foxit.sdk.pdf.annots.Markup;
import com.foxit.sdk.pdf.annots.Note;
import com.foxit.uiextensions.AbstractUndo;
import com.foxit.uiextensions.DocumentManager;
import com.foxit.uiextensions.IThemeEventListener;
import com.foxit.uiextensions.IUndoItem;
import com.foxit.uiextensions.Module;
import com.foxit.uiextensions.R;
import com.foxit.uiextensions.UIExtensionsManager;
import com.foxit.uiextensions.annots.AnnotHandler;
import com.foxit.uiextensions.annots.DefaultAnnotHandler;
import com.foxit.uiextensions.annots.IFlattenEventListener;
import com.foxit.uiextensions.controls.dialog.UIPopoverFrag;
import com.foxit.uiextensions.controls.toolbar.BaseBar;
import com.foxit.uiextensions.controls.toolbar.IBaseItem;
import com.foxit.uiextensions.controls.toolbar.ToolConstants;
import com.foxit.uiextensions.controls.toolbar.impl.BaseBarImpl;
import com.foxit.uiextensions.controls.toolbar.impl.BaseItemImpl;
import com.foxit.uiextensions.controls.toolbar.impl.UIColorItem;
import com.foxit.uiextensions.event.PageEventListener;
import com.foxit.uiextensions.pdfreader.ILayoutChangeListener;
import com.foxit.uiextensions.pdfreader.IStateChangeListener;
import com.foxit.uiextensions.theme.ThemeUtil;
import com.foxit.uiextensions.utils.AppAnnotUtil;
import com.foxit.uiextensions.utils.AppDisplay;
import com.foxit.uiextensions.utils.AppResource;
import com.foxit.uiextensions.utils.AppUtil;
import com.foxit.uiextensions.utils.SystemUiHelper;
import com.foxit.uiextensions.utils.thread.AppThreadManager;

import java.util.ArrayList;

import androidx.fragment.app.FragmentActivity;

/** Support undo/redo for annotation operations.(add/edit/delete..etc). */
public class UndoModule implements Module {
    private Context mContext;
    private DefaultAnnotHandler mDefAnnotHandler;

    private PDFViewCtrl mPdfViewCtrl;
    private PDFViewCtrl.UIExtensionsManager mUiExtensionsManager;

    private UIColorItem mUndoItem;
    private UIColorItem mRedoItem;
    private BaseItemImpl mMoreUndoItem;
    private BaseItemImpl mMoreRedoItem;

    private BaseBarImpl mMoreToolsBar;
    private UIPopoverFrag mBlackPopover;
    private RelativeLayout mBlackRootView;

    private IUndoItemCallback mUndoItemCallback;

    public UndoModule(Context context, PDFViewCtrl pdfViewCtrl, PDFViewCtrl.UIExtensionsManager uiExtensionsManager) {
        mContext = context;
        mPdfViewCtrl = pdfViewCtrl;
        mUiExtensionsManager = uiExtensionsManager;
    }

    @Override
    public String getName() {
        return MODULE_NAME_UNDO;
    }

    @Override
    public boolean loadModule() {
        mDefAnnotHandler = new DefaultAnnotHandler(mContext, mPdfViewCtrl);
        mPdfViewCtrl.registerPageEventListener(mPageEventListener);
        if (mUiExtensionsManager != null && mUiExtensionsManager instanceof UIExtensionsManager) {
            UIExtensionsManager uiExtensionsManager = ((UIExtensionsManager) mUiExtensionsManager);
            uiExtensionsManager.registerAnnotHandler(mDefAnnotHandler);
            uiExtensionsManager.registerModule(this);
            uiExtensionsManager.registerLayoutChangeListener(mLayoutChangeListener);
            uiExtensionsManager.registerThemeEventListener(mThemeEventListener);
            uiExtensionsManager.getDocumentManager().registerFlattenEventListener(mFlattenEventListener);
            initUndoItems();
        }
        return true;
    }

    private void initUndoItems() {
        UIExtensionsManager uiExtensionsManager = (UIExtensionsManager) mUiExtensionsManager;
        mUndoItem = createUndoButtonItem();
        mRedoItem = createRedoButtonItem();
        mMoreUndoItem = createMoreUndoButtonItem();
        mMoreRedoItem = createMoreRedoButtonItem();
        uiExtensionsManager.getDocumentManager().registerUndoEventListener(mUndoEventListener);
        uiExtensionsManager.registerStateChangeListener(mStateChangeListener);
    }

    @Override
    public boolean unloadModule() {
        mPdfViewCtrl.unregisterPageEventListener(mPageEventListener);
        if (mUiExtensionsManager != null && mUiExtensionsManager instanceof UIExtensionsManager) {
            UIExtensionsManager uiExtensionsManager = ((UIExtensionsManager) mUiExtensionsManager);
            uiExtensionsManager.unregisterAnnotHandler(mDefAnnotHandler);
            uiExtensionsManager.unregisterStateChangeListener(mStateChangeListener);
            uiExtensionsManager.unregisterThemeEventListener(mThemeEventListener);
            uiExtensionsManager.getDocumentManager().unregisterFlattenEventListener(mFlattenEventListener);
            uiExtensionsManager.getDocumentManager().unregisterUndoEventListener(mUndoEventListener);
            uiExtensionsManager.unregisterLayoutChangeListener(mLayoutChangeListener);
        }
        return true;
    }

    private AbstractUndo.IUndoEventListener mUndoEventListener = new AbstractUndo.IUndoEventListener() {
        @Override
        public void itemWillAdd(DocumentManager dm, IUndoItem item) {
        }

        @Override
        public void itemAdded(DocumentManager dm, IUndoItem item) {
            changeButtonStatus();
        }

        @Override
        public void itemWillRemoved(DocumentManager dm, IUndoItem item) {
        }

        @Override
        public void itemRemoved(DocumentManager dm, IUndoItem item) {
        }

        @Override
        public void willUndo(DocumentManager dm, IUndoItem item) {
        }

        @Override
        public void undoFinished(DocumentManager dm, IUndoItem item) {
            changeButtonStatus();
        }

        @Override
        public void willRedo(DocumentManager dm, IUndoItem item) {
        }

        @Override
        public void redoFinished(DocumentManager dm, IUndoItem item) {
            changeButtonStatus();
        }

        @Override
        public void willClearUndo(DocumentManager dm) {
        }

        @Override
        public void clearUndoFinished(DocumentManager dm) {
            changeButtonStatus();
        }
    };

    private IStateChangeListener mStateChangeListener = new IStateChangeListener() {
        @Override
        public void onStateChanged(int oldState, int newState) {
            changeButtonStatus();
        }
    };

    private ILayoutChangeListener mLayoutChangeListener = new ILayoutChangeListener() {
        @Override
        public void onLayoutChange(View v, int newWidth, int newHeight, int oldWidth, int oldHeight) {
            if (newWidth != oldWidth || newHeight != oldHeight) {
                updateUndoPop();
            }
        }
    };

    public void changeButtonStatus() {
        DocumentManager dm = ((UIExtensionsManager) mUiExtensionsManager).getDocumentManager();

        boolean canUndo = mUndoItemCallback != null ? mUndoItemCallback.canUndo() : dm.canUndo();
        boolean canRedo = mUndoItemCallback != null ? mUndoItemCallback.canRedo() : dm.canRedo();
        if (AppDisplay.isPad()) {
            mUndoItem.setEnable(canUndo);
            mRedoItem.setEnable(canRedo);
        } else {
            if (canUndo || canRedo)
                mUndoItem.setEnable(true);
            else
                mUndoItem.setEnable(false);

            mMoreUndoItem.setEnable(canUndo);
            mMoreRedoItem.setEnable(canRedo);
        }
    }

    public AnnotHandler getAnnotHandler() {
        return mDefAnnotHandler;
    }

    private PDFViewCtrl.IPageEventListener mPageEventListener = new PageEventListener() {
        @Override
        public void onPageMoved(boolean success, int index, int dstIndex) {
            ((UIExtensionsManager) mUiExtensionsManager).getDocumentManager().onPageMoved(success, index, dstIndex);
        }

        @Override
        public void onPagesRemoved(boolean success, int[] pageIndexes) {
            for (int i = 0; i < pageIndexes.length; i++)
                ((UIExtensionsManager) mUiExtensionsManager).getDocumentManager().onPageRemoved(success, pageIndexes[i] - i);

            AppThreadManager.getInstance().getMainThreadHandler().post(new Runnable() {
                @Override
                public void run() {
                    changeButtonStatus();
                }
            });
        }

        @Override
        public void onPagesInserted(boolean success, int dstIndex, int[] range) {
            ((UIExtensionsManager) mUiExtensionsManager).getDocumentManager().onPagesInsert(success, dstIndex, range);
        }
    };

    private ArrayList<String> mReplyLists = new ArrayList<>();

    private void getAnnotReplys(Annot annot) {
        try {
            if (annot.isMarkup()) {
                Markup markup = new Markup(annot);
                int count = markup.getReplyCount();

                if (count > 0) {
                    for (int i = 0; i < count; i++) {
                        Note note = markup.getReply(i);
                        if (note.getReplyCount() > 0) {
                            getAnnotReplys(note);
                        }
                        mReplyLists.add(AppAnnotUtil.getAnnotUniqueID(note));
                    }
                }
            }
        } catch (PDFException e) {
            e.printStackTrace();
        }
    }

    private IFlattenEventListener mFlattenEventListener = new IFlattenEventListener() {
        @Override
        public void onAnnotWillFlatten(PDFPage page, Annot annot) {
            try {
                mReplyLists.clear();
                getAnnotReplys(annot);
                mReplyLists.add(AppAnnotUtil.getAnnotUniqueID(annot));
                for (String uniqueId : mReplyLists) {
                    ((UIExtensionsManager) mUiExtensionsManager).getDocumentManager().removeFlattenUndoItems(page.getIndex(), uniqueId);
                }
            } catch (PDFException e) {
                e.printStackTrace();
            }
        }

        @Override
        public void onAnnotFlattened(PDFPage page, Annot annot) {
            changeButtonStatus();
        }
    };

    public UIColorItem createUndoButtonItem() {
        if (mUndoItem == null) {
            mUndoItem = new UIColorItem(mContext, R.drawable.tool_undo_normal);
            mUndoItem.setId(R.id.id_undo);
            mUndoItem.setImageTintList(ThemeUtil.getPrimaryIconColor(mContext));
            mUndoItem.setForceDarkAllowed(false);
            mUndoItem.setOnItemClickListener(new IBaseItem.OnItemClickListener() {
                @Override
                public void onClick(IBaseItem item, View v) {
                    ((UIExtensionsManager) mUiExtensionsManager).resetHideToolbarsTimer();
                    if (AppUtil.isFastDoubleClick()) return;
                    if (mUndoItemCallback != null && mUndoItemCallback.undo())
                        return;
                    if (((UIExtensionsManager) mUiExtensionsManager).getDocumentManager().canUndo()) {
                        ((UIExtensionsManager) mUiExtensionsManager).getDocumentManager().undo();
                    } else
                        mUndoItem.performLongClick();
                }
            });
            if (!AppDisplay.isPad()) {
                mUndoItem.setCoverImageResource(R.drawable.tool_bar_drop_right);
                mUndoItem.setCoverImageTintList(ThemeUtil.getPrimaryIconColor(mContext));
                mUndoItem.setOnItemLongPressListener(new IBaseItem.OnItemLongPressListener() {
                    @Override
                    public boolean onLongPress(IBaseItem item, View v) {
                        showUndoPop();
                        return true;
                    }
                });
            }
            mUndoItem.setTag(ToolConstants.Undo);
        }
        return mUndoItem;
    }

    public UIColorItem createRedoButtonItem() {
        if (mRedoItem == null) {
            mRedoItem = new UIColorItem(mContext, R.drawable.tool_redo_normal);
            mRedoItem.setId(R.id.id_redo);
            mRedoItem.setImageTintList(ThemeUtil.getPrimaryIconColor(mContext));
            mRedoItem.setForceDarkAllowed(false);
            mRedoItem.setOnItemClickListener(new IBaseItem.OnItemClickListener() {
                @Override
                public void onClick(IBaseItem item, View v) {
                    ((UIExtensionsManager) mUiExtensionsManager).resetHideToolbarsTimer();

                    if (AppUtil.isFastDoubleClick()) return;
                    if (mUndoItemCallback != null && mUndoItemCallback.redo())
                        return;
                    redo();
                }
            });
            mRedoItem.setTag(ToolConstants.Redo);
        }
        return mRedoItem;
    }

    private BaseItemImpl createMoreUndoButtonItem() {
        if (mMoreUndoItem == null) {
            mMoreUndoItem = new BaseItemImpl(mContext, R.drawable.tool_more_undo_normal);
            mMoreUndoItem.setId(R.id.id_more_undo);
            mMoreUndoItem.setForceDarkAllowed(false);
            mMoreUndoItem.setOnClickListener(new View.OnClickListener() {
                @Override
                public void onClick(View v) {
                    if (AppUtil.isFastDoubleClick()) return;
                    if (mUndoItemCallback != null && mUndoItemCallback.undo())
                        return;
                    undo();
                }
            });
            mMoreUndoItem.setImagePadding(AppDisplay.dp2px(5), 0,
                    AppDisplay.dp2px(5), 0);
            mMoreUndoItem.setImageTextBackgroundResouce(R.drawable.black_popover_bg_leftbtn);
            mMoreUndoItem.getContentView().setBackgroundColor(AppResource.getColor(mContext, R.color.ux_color_translucent));
            mMoreUndoItem.setTag(ToolConstants.MoreUndo);
        }
        return mMoreUndoItem;
    }

    private BaseItemImpl createMoreRedoButtonItem() {
        if (mMoreRedoItem == null) {
            mMoreRedoItem = new BaseItemImpl(mContext, R.drawable.tool_more_redo_normal);
            mMoreRedoItem.setId(R.id.id_more_redo);
            mMoreRedoItem.setForceDarkAllowed(false);
            mMoreRedoItem.setOnClickListener(new View.OnClickListener() {
                @Override
                public void onClick(View v) {
                    if (AppUtil.isFastDoubleClick()) return;
                    if (mUndoItemCallback != null && mUndoItemCallback.redo())
                        return;
                    redo();
                }
            });
            mMoreRedoItem.setImagePadding(AppDisplay.dp2px(5), 0,
                    AppDisplay.dp2px(5), 0);
            mMoreRedoItem.setImageTextBackgroundResouce(R.drawable.black_popover_bg_rightbtn);
            mMoreRedoItem.getContentView().setBackgroundColor(AppResource.getColor(mContext, R.color.ux_color_translucent));
            mMoreRedoItem.setTag(ToolConstants.MoreRedo);
        }
        return mMoreRedoItem;
    }

    public void showUndoPop() {
         Activity activity = ((UIExtensionsManager) mUiExtensionsManager).getAttachedActivity();
        ((UIExtensionsManager) mUiExtensionsManager).stopHideToolbarsTimer();
        if (SystemUiHelper.getInstance().isFullScreen())
            SystemUiHelper.getInstance().showNavigationBar(activity);

        if (mMoreToolsBar == null) {
            mMoreToolsBar = new BaseBarImpl(mContext);
            mMoreToolsBar.setHeight(AppDisplay.dp2px(UIPopoverFrag.mBlackPopoverHeightDp));
            mMoreToolsBar.setStartMargin(0);
            mMoreToolsBar.setEndMargin(0);
            mMoreToolsBar.setBackgroundColor(AppResource.getColor(mContext, R.color.ux_color_translucent));
            mMoreToolsBar.setItemInterval(AppDisplay.dp2px(0.5f));

            mMoreUndoItem = createMoreUndoButtonItem();
            mMoreRedoItem = createMoreRedoButtonItem();
            mMoreToolsBar.addView(mMoreUndoItem, BaseBar.TB_Position.Position_CENTER);
            mMoreToolsBar.addView(mMoreRedoItem, BaseBar.TB_Position.Position_CENTER);
        }

        if (mBlackPopover == null) {
            mBlackRootView = new RelativeLayout(mContext.getApplicationContext());
        }
        initBlackPopover();
        mBlackRootView.removeAllViews();
        mBlackRootView.setBackgroundColor(AppResource.getColor(mContext, R.color.ux_color_translucent, null));
        AppUtil.removeViewFromParent(mMoreToolsBar.getContentView());
        mBlackRootView.addView(mMoreToolsBar.getContentView());

        Point size = mMoreToolsBar.measureSize();
        Rect rect = new Rect();
        mUndoItem.getContentView().getGlobalVisibleRect(rect);
        int arrowPos = getArrowPosition();
        mBlackPopover.setArrowColor(AppResource.getColor(mContext, R.color.ux_color_black_popover_bg));
        mBlackPopover.setArrowForceDarkAllowed(false);
        mBlackPopover.showAtLocation(((UIExtensionsManager) mUiExtensionsManager).getRootView(), rect,
                size.x, AppDisplay.dp2px(UIPopoverFrag.mBlackPopoverHeightDp),
                arrowPos, 0);
    }

    private void initBlackPopover(){
        if (mBlackPopover != null) return;
        mBlackPopover = UIPopoverFrag.create((FragmentActivity) ((UIExtensionsManager) mUiExtensionsManager).getAttachedActivity(), mBlackRootView, true, true);
        mBlackPopover.setOnDismissListener(new PopupWindow.OnDismissListener() {
            @Override
            public void onDismiss() {
                ((UIExtensionsManager) mUiExtensionsManager).startHideToolbarsTimer();
                AppUtil.removeViewFromParent(mBlackRootView);
                mBlackPopover = null;
            }
        });
    }

    private int getArrowPosition() {
        return UIPopoverFrag.ARROW_TOP;
    }

    private void updateUndoPop() {
        if (mBlackPopover != null && mBlackPopover.isShowing()) {
            new Handler().postDelayed(new Runnable() {
                @Override
                public void run() {
                    Rect rect = new Rect();
                    mUndoItem.getContentView().getGlobalVisibleRect(rect);
                    Point size = mMoreToolsBar.measureSize();
                    initBlackPopover();
                    mBlackPopover.update(((UIExtensionsManager) mUiExtensionsManager).getRootView(), rect, size.x,
                            AppDisplay.dp2px(UIPopoverFrag.mBlackPopoverHeightDp));
                }
            }, 200);
        }
    }

    private void undo() {
        if (((UIExtensionsManager) mUiExtensionsManager).getDocumentManager().canUndo()) {
            ((UIExtensionsManager) mUiExtensionsManager).getDocumentManager().undo();
        }
    }

    private void redo() {
        if (((UIExtensionsManager) mUiExtensionsManager).getDocumentManager().canRedo()) {
            ((UIExtensionsManager) mUiExtensionsManager).getDocumentManager().redo();
        }
    }

    public void setUndoItemCallback(IUndoItemCallback undoItemCallback) {
        this.mUndoItemCallback = undoItemCallback;
    }

    public interface IUndoItemCallback {
        boolean undo();

        boolean canUndo();

        boolean redo();

        boolean canRedo();
    }

    private IThemeEventListener mThemeEventListener = new IThemeEventListener() {
        @Override
        public void onThemeColorChanged(String type, int color) {
            if (mUndoItem != null) {
                mUndoItem.setImageTintList(ThemeUtil.getPrimaryIconColor(mContext));
                mUndoItem.setCoverImageTintList(ThemeUtil.getPrimaryIconColor(mContext));
            }

            if (mRedoItem != null)
                mRedoItem.setImageTintList(ThemeUtil.getPrimaryIconColor(mContext));
        }
    };

}
