/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.modules.crop;

import android.app.AlertDialog;
import android.app.Dialog;
import android.content.Context;
import android.content.DialogInterface;
import android.content.res.Configuration;
import android.view.KeyEvent;
import android.view.ViewGroup;
import android.view.Window;
import android.view.WindowManager;

import com.foxit.sdk.PDFViewCtrl;
import com.foxit.sdk.common.Constants;
import com.foxit.sdk.pdf.PDFDoc;
import com.foxit.uiextensions.IThemeEventListener;
import com.foxit.uiextensions.Module;
import com.foxit.uiextensions.R;
import com.foxit.uiextensions.ToolHandler;
import com.foxit.uiextensions.UIExtensionsManager;
import com.foxit.uiextensions.annots.fillsign.FillSignModule;
import com.foxit.uiextensions.controls.propertybar.IViewSettingsWindow;
import com.foxit.uiextensions.event.DocEventListener;
import com.foxit.uiextensions.modules.ReflowModule;
import com.foxit.uiextensions.modules.textselect.BlankSelectToolHandler;
import com.foxit.uiextensions.modules.textselect.TextSelectToolHandler;
import com.foxit.uiextensions.pdfreader.config.ReadStateConfig;
import com.foxit.uiextensions.utils.AppDisplay;
import com.foxit.uiextensions.utils.AppResource;
import com.foxit.uiextensions.utils.SystemUiHelper;

/**
 * Crop module is for cutting white edge of PDF page, so the page content would become more prominent.
 */
public class CropModule implements Module {
    private Context mContext = null;
    private PDFViewCtrl mPdfViewCtrl = null;
    private ViewGroup mParent = null;
    private IViewSettingsWindow mSettingWindow;
    private CropView mCropView;
    private PDFViewCtrl.UIExtensionsManager mUiExtensionsManager;

    public CropModule(Context context, ViewGroup parent, PDFViewCtrl pdfViewCtrl, PDFViewCtrl.UIExtensionsManager uiExtensionsManager) {
        mContext = context;
        mParent = parent;
        mPdfViewCtrl = pdfViewCtrl;
        mUiExtensionsManager = uiExtensionsManager;
    }

    @Override
    public String getName() {
        return MODULE_NAME_CROP;
    }

    @Override
    public boolean loadModule() {
        if (mUiExtensionsManager != null && mUiExtensionsManager instanceof UIExtensionsManager) {
            ((UIExtensionsManager) mUiExtensionsManager).registerModule(this);
            ((UIExtensionsManager) mUiExtensionsManager).registerThemeEventListener(mThemeChangedListener);
            ((UIExtensionsManager) mUiExtensionsManager).registerConfigurationChangedListener(mConfigurationChangedListener);
            mSettingWindow = ((UIExtensionsManager) mUiExtensionsManager).getMainFrame().getSettingWindow();
        }
        mPdfViewCtrl.registerRecoveryEventListener(memoryEventListener);
        mPdfViewCtrl.registerDocEventListener(docEventListener);
        return true;
    }

    @Override
    public boolean unloadModule() {
        if (mUiExtensionsManager != null && mUiExtensionsManager instanceof UIExtensionsManager) {
            ((UIExtensionsManager) mUiExtensionsManager).unregisterThemeEventListener(mThemeChangedListener);
            ((UIExtensionsManager) mUiExtensionsManager).unregisterConfigurationChangedListener(mConfigurationChangedListener);
        }
        mPdfViewCtrl.unregisterDocEventListener(docEventListener);
        mPdfViewCtrl.unregisterRecoveryEventListener(memoryEventListener);
        if (mCropView != null){
            mCropView.setSettingWindow(null);
        }
        return true;
    }

    PDFViewCtrl.IDocEventListener docEventListener = new DocEventListener() {
        @Override
        public void onDocWillOpen() {
        }

        @Override
        public void onDocOpened(PDFDoc document, int errCode) {
            if (errCode != Constants.e_ErrSuccess) return;
            mSettingWindow = ((UIExtensionsManager) mUiExtensionsManager).getMainFrame().getSettingWindow();
            if (mPdfViewCtrl.isDynamicXFA()) {
                mSettingWindow.enableBar(IViewSettingsWindow.TYPE_CROP, false);
                return;
            }

            mSettingWindow.enableBar(IViewSettingsWindow.TYPE_CROP, true);
            registerViewSettingListener();
            if (mCropView == null) {
                mCropView = new CropView(mContext, mParent, mPdfViewCtrl);
                mCropView.setSettingWindow(mSettingWindow);
                mCropView.changeState(false);
            }
        }

        @Override
        public void onDocWillClose(PDFDoc document) {
        }

        @Override
        public void onDocClosed(PDFDoc document, int errCode) {
            exitCrop();
            unRegisterViewSettingListener();
        }

    };

    public boolean isCropMode() {
        return mCropView != null && mCropView.isCropMode();
    }

    public void exitCrop() {
        if (mCropView != null) {
            if (mCropView.isShow()) {
                mCropView.closeCropView();
            }

            if (mCropView.isCropMode()) {
                mPdfViewCtrl.setCropMode(PDFViewCtrl.CROPMODE_NONE);
                mCropView.changeState(false);
            }
        }
    }

    public void restoreCrop() {
        if (mCropView != null) {
            int cropMode = mCropView.getLastCropMode();
            if (cropMode == PDFViewCtrl.CROPMODE_NONE) return;
            if (PDFViewCtrl.CROPMODE_CUSTOMIZE == cropMode) {
                mPdfViewCtrl.setCropRect(-1, mCropView.getLastCropRect());
            }
            mPdfViewCtrl.setCropMode(cropMode);
            mCropView.changeState(true);
        }
    }

    private void registerViewSettingListener() {
        mSettingWindow.registerListener(mCropChangeListener);
    }

    private void unRegisterViewSettingListener() {
        if (mSettingWindow == null)
            return;
        mSettingWindow.unRegisterListener(mCropChangeListener);
    }

    private PDFViewCtrl.IRecoveryEventListener memoryEventListener = new PDFViewCtrl.IRecoveryEventListener() {
        @Override
        public void onWillRecover() {
            exitCrop();
        }

        @Override
        public void onRecovered() {
        }
    };

    private IViewSettingsWindow.IValueChangeListener mCropChangeListener = new IViewSettingsWindow.IValueChangeListener() {

        @Override
        public void onValueChanged(int type, Object value) {
            if (!(value instanceof Boolean)) return;
            if (type == IViewSettingsWindow.TYPE_CROP && (Boolean) value) {
                UIExtensionsManager uiExtensionsManager = (UIExtensionsManager) mUiExtensionsManager;
                if (uiExtensionsManager.getState() == ReadStateConfig.STATE_FILLSIGN) {
                    if (uiExtensionsManager.getCurrentToolHandler() != null)
                        uiExtensionsManager.setCurrentToolHandler(null);
                    else {
                        FillSignModule fillSignModule = (FillSignModule) uiExtensionsManager.getModuleByName(Module.MODULE_NAME_FIllSIGN);
                        fillSignModule.getToolHandler().onDeactivate();
                    }
                }

                showCropMenu();
                uiExtensionsManager.getMainFrame().hideSettingWindow();
                if (uiExtensionsManager.getDocumentManager().getCurrentAnnot() != null) {
                    uiExtensionsManager.getDocumentManager().setCurrentAnnot(null);
                }
            }
        }

        @Override
        public int getType() {
            return IViewSettingsWindow.TYPE_CROP;
        }
    };

    public boolean onKeyDown(int keyCode, KeyEvent event) {
        if (mCropView != null) {
            return mCropView.onKeyDown(keyCode, event);
        }
        return false;
    }

    public boolean onKeyBack() {
        if (mCropView != null) {
            return mCropView.onKeyBack();
        }
        return false;
    }

    Dialog mCropDialog;

    private void showCropMenu() {
        AlertDialog.Builder builder = new AlertDialog.Builder(((UIExtensionsManager) mUiExtensionsManager).getAttachedActivity());
        String[] items;
        if (mPdfViewCtrl.getCropMode() == PDFViewCtrl.CROPMODE_NONE) {
            items = new String[]{
                    AppResource.getString(mContext.getApplicationContext(), R.string.crop_menu_auto_crop),
                    AppResource.getString(mContext.getApplicationContext(), R.string.crop_menu_hand_crop)
            };
        } else {
            items = new String[]{
                    AppResource.getString(mContext.getApplicationContext(), R.string.crop_menu_auto_crop),
                    AppResource.getString(mContext.getApplicationContext(), R.string.crop_menu_hand_crop),
                    AppResource.getString(mContext.getApplicationContext(), R.string.crop_menu_remove_crop)
            };
        }


        builder.setItems(items, new DialogInterface.OnClickListener() {
            @Override
            public void onClick(DialogInterface dialog, int which) {
                UIExtensionsManager uiExtensionsManager = (UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager();
                if (uiExtensionsManager.getState() == ReadStateConfig.STATE_REFLOW) {
                    ReflowModule reflowModule = (ReflowModule) uiExtensionsManager.getModuleByName(Module.MODULE_NAME_REFLOW);
                    reflowModule.clearData();
                    uiExtensionsManager.changeState(ReadStateConfig.STATE_NORMAL);
                }
                switch (which) {
                    case 0: // auto crop
                        mPdfViewCtrl.setCropMode(PDFViewCtrl.CROPMODE_CONTENTSBOX);
                        mSettingWindow.setProperty(IViewSettingsWindow.TYPE_CROP, true);
                        break;
                    case 1: // manually crop
                        dismissSelectorMenu();
                        mCropView.openCropView();
                        SystemUiHelper.getInstance().setAllowedHideSystemUI(uiExtensionsManager.getAttachedActivity(), false);
                        uiExtensionsManager.getMainFrame().hideToolbars();
                        break;
                    case 2: // discard modify
                        mPdfViewCtrl.setCropMode(PDFViewCtrl.CROPMODE_NONE);
                        mCropView.changeState(false);
                        break;
                    default:
                        mSettingWindow.setProperty(IViewSettingsWindow.TYPE_CROP, false);
                        break;
                }
                dialog.dismiss();
            }
        });

        mCropDialog = builder.create();

        Window window = mCropDialog.getWindow();
        assert window != null;
        window.setBackgroundDrawableResource(R.drawable.dlg_bg_4circle_corner_10dp_grayf4f4f4);
        mCropDialog.setCanceledOnTouchOutside(true);
        mCropDialog.setOnDismissListener(new DialogInterface.OnDismissListener() {
            @Override
            public void onDismiss(DialogInterface dialog) {
                if (mPdfViewCtrl.getCropMode() == PDFViewCtrl.CROPMODE_NONE) {
                    mSettingWindow.setProperty(IViewSettingsWindow.TYPE_CROP, false);
                }
            }
        });
        mCropDialog.show();

        WindowManager.LayoutParams layoutParams = window.getAttributes();
        if (AppDisplay.isPad()) {
            layoutParams.width = Math.min(AppDisplay.getActivityHeight(), AppDisplay.getActivityWidth()) * 2 / 5;
        } else {
            if (AppDisplay.isLandscape()) {
                layoutParams.width = AppDisplay.getActivityHeight() * 2 / 3;
            } else {
                layoutParams.width = AppDisplay.getActivityWidth() * 2 / 3;
            }
        }
        window.setAttributes(layoutParams);
    }

    private IThemeEventListener mThemeChangedListener = new IThemeEventListener() {
        @Override
        public void onThemeColorChanged(String type, int color) {
            if (mCropView != null) {
                mCropView.onThemeColorChanged(type, color);
            }
            if (mCropDialog != null) {
                mCropDialog.dismiss();
                mCropDialog = null;
            }
        }
    };

    public void dismissSelectorMenu(){
        UIExtensionsManager uiExtensionsManager = (UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager();
        if (uiExtensionsManager == null)return;
        BlankSelectToolHandler blankSelectToolHandler = (BlankSelectToolHandler) uiExtensionsManager.getToolHandlerByType(ToolHandler.TH_TYPE_BLANKSELECT);
        if (blankSelectToolHandler != null)
            blankSelectToolHandler.dismissMenu();

        TextSelectToolHandler textSelectionTool = (TextSelectToolHandler) uiExtensionsManager.getToolHandlerByType(ToolHandler.TH_TYPE_TEXTSELECT);
        if (textSelectionTool != null)
            textSelectionTool.dismissMenu();
    }

    public CropInfo getManualCropInfo(){
        if (mCropView != null) {
            return mCropView.getManualCropInfo();
        }
        return null;
    }

    private UIExtensionsManager.ConfigurationChangedListener mConfigurationChangedListener = new UIExtensionsManager.ConfigurationChangedListener() {
        @Override
        public void onConfigurationChanged(Configuration newConfig) {
            if (mCropView != null && mCropView.isShow()) {
                mCropView.onConfigurationChanged(newConfig);
            }
        }
    };

}
