/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.modules.doc;


import android.graphics.RectF;

import com.foxit.sdk.PDFException;
import com.foxit.sdk.PDFViewCtrl;
import com.foxit.sdk.pdf.PDFDoc;
import com.foxit.sdk.pdf.PDFPage;
import com.foxit.uiextensions.Module;
import com.foxit.uiextensions.UIExtensionsManager;
import com.foxit.uiextensions.modules.crop.CropInfo;
import com.foxit.uiextensions.modules.crop.CropModule;
import com.foxit.uiextensions.utils.AppUtil;

import java.util.concurrent.Callable;

import io.reactivex.Observable;
import io.reactivex.Single;
import io.reactivex.android.schedulers.AndroidSchedulers;
import io.reactivex.disposables.Disposable;
import io.reactivex.functions.Consumer;

public class CropPDF {

    public static Single<Boolean> doCropPDF(final PDFViewCtrl viewCtrl,
                                            final OnProgressListener progressListener,
                                            final OnProgressCancelListener cancelListener) {
        return Single.fromCallable(new Callable<Boolean>() {
            @Override
            public Boolean call() throws Exception {
                return doCropPDFImpl(viewCtrl, progressListener, cancelListener);
            }
        });
    }

    public static boolean doCropPDFImpl(PDFViewCtrl viewCtrl,
                                         OnProgressListener progressListener,
                                         OnProgressCancelListener cancelListener) {
        if (viewCtrl.getCropMode() == PDFViewCtrl.CROPMODE_NONE) return true;
        PDFDoc doc = viewCtrl.getDoc();
        if (doc == null) return false;

        try {
            int pageCount = doc.getPageCount();
            for (int i = 0; i < pageCount; i++) {
                if (cancelListener != null && cancelListener.onCancel()) return false;

                if (progressListener != null) {
                    onProgress(progressListener, i);
                }

                PDFPage page = doc.getPage(i);
                RectF cropRectF = viewCtrl.getCropRect(i);
                if (cropRectF == null) {
                    if (viewCtrl.getColorMode() == PDFViewCtrl.CROPMODE_CONTENTSBOX) { // auto crop
                        cropRectF = getPageContentRect(page);
                    } else { //custom crop
                        UIExtensionsManager uiExtensionsManager = (UIExtensionsManager) viewCtrl.getUIExtensionsManager();
                        CropModule cropModule = (CropModule) uiExtensionsManager.getModuleByName(Module.MODULE_NAME_CROP);
                        if (cropModule != null) {
                            CropInfo cropInfo = cropModule.getManualCropInfo();
                            if (cropInfo != null && cropInfo.mType == 1) { // type： 0- CROP_SETTING_TYPE_ALL; 1-CROP_SETTING_TYPE_ODD_EVEN;
                                if (cropInfo.mIsOdd && i % 2 == 1 || !cropInfo.mIsOdd && i % 2 == 0) {
                                    cropRectF = new RectF(cropInfo.mOddCropRectF);
                                } else {
                                    cropRectF = new RectF(cropInfo.mCropRectF);
                                }
                            }
                        }
                    }
                }

                if (cropRectF == null || AppUtil.isEmptyPDFRect(cropRectF)) {
                    return false;
                }
                page.setBox(PDFPage.e_CropBox, AppUtil.toFxRectF(cropRectF));
            }
            return true;
        } catch (PDFException e) {
            e.printStackTrace();
        }
        return false;
    }

    private static RectF getPageContentRect(PDFPage page) {
        RectF rect = new RectF();
        try {
            if (page == null || page.isEmpty()) {
                return rect;
            }
            rect.set(0, page.getHeight(), page.getWidth(), 0);

            com.foxit.sdk.common.fxcrt.RectF contentBbox = page.calcContentBBox(PDFPage.e_CalcContentsBox);
            float left = contentBbox.getLeft();
            float right = contentBbox.getRight();
            float top = contentBbox.getTop();
            float bottom = contentBbox.getBottom();
            if (left != 0 || right != 0 || top != 0 || bottom != 0) {
                RectF newRect = new RectF(left, top, right, bottom);
                rect.set(newRect);
            }
            page.delete();
        } catch (PDFException e) {
            rect.setEmpty();
        }
        return rect;
    }

    private static void onProgress(final OnProgressListener progressListener, int pageIndex) {
        if (null != progressListener) {
            Disposable disposable = Observable.just(pageIndex)
                    .subscribeOn(AndroidSchedulers.mainThread())
                    .subscribe(new Consumer<Integer>() {
                        @Override
                        public void accept(Integer integer) throws Exception {
                            progressListener.onProgress(integer);
                        }
                    });
        }
    }

}
