/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.modules.doc;

import com.foxit.sdk.PDFException;
import com.foxit.sdk.addon.optimization.ImageSettings;
import com.foxit.sdk.addon.optimization.MonoImageSettings;
import com.foxit.sdk.addon.optimization.Optimizer;
import com.foxit.sdk.addon.optimization.OptimizerSettings;
import com.foxit.sdk.common.Progressive;
import com.foxit.sdk.common.fxcrt.PauseCallback;
import com.foxit.sdk.pdf.PDFDoc;
import com.foxit.uiextensions.utils.AppUtil;

import java.util.concurrent.Callable;

import io.reactivex.Observable;
import io.reactivex.Single;
import io.reactivex.android.schedulers.AndroidSchedulers;
import io.reactivex.functions.Consumer;

public class OptimizerPDF {

    public static Single<Boolean> doOptimizerPDF(PDFDoc doc, int imageQuality, int monoImageQuality, OnProgressListener progressListener) {
        return doOptimizerPDF(doc, imageQuality, monoImageQuality, progressListener, null);
    }

    public static Single<Boolean> doOptimizerPDF(final PDFDoc doc, final int imageQuality, final int monoImageQuality,
                                                 final OnProgressListener progressListener, final PauseCallbackImpl pauseCallback) {
        return Single.fromCallable(new Callable<Boolean>() {
            @Override
            public Boolean call() throws Exception {
                return doOptimizerPDFImpl(doc, imageQuality, monoImageQuality, progressListener, pauseCallback);
            }
        });
    }

    public static Single<Boolean> doOptimizerPDF(String path, int imageQuality, int monoImageQuality, OnProgressListener progressListener) {
        return doOptimizerPDF(path, null, imageQuality, monoImageQuality, progressListener);
    }

    public static Single<Boolean> doOptimizerPDF(final String path, final byte[] password, final int imageQuality,
                                                 final int monoImageQuality, final OnProgressListener progressListener) {
        return Single.fromCallable(new Callable<Boolean>() {
            @Override
            public Boolean call() throws Exception {
                return doOptimizerPDFImpl(path, password, imageQuality, monoImageQuality, progressListener);
            }
        });
    }

    public static boolean doOptimizerPDFImpl(String path, final byte[] password, int imageQuality, int monoImageQuality, OnProgressListener progressListener) {
        try {
            if (AppUtil.isEmpty(path))
                return false;
            PDFDoc doc = new PDFDoc(path);
            doc.load(password);
            return doOptimizerPDFImpl(doc, imageQuality, monoImageQuality, progressListener);
        } catch (PDFException e) {
            e.printStackTrace();
        }
        return false;
    }

    public static boolean doOptimizerPDFImpl(PDFDoc doc, int imageQuality, int monoImageQuality, OnProgressListener progressListener) {
        return doOptimizerPDFImpl(doc, imageQuality, monoImageQuality, progressListener, null);
    }

    public static boolean doOptimizerPDFImpl(PDFDoc doc, int imageQuality, int monoImageQuality,
                                             final OnProgressListener progressListener, PauseCallbackImpl callback) {
        try {
            if (doc == null)
                return false;
            OptimizerSettings optimizerSettings = new OptimizerSettings();

            ImageSettings imageSettings = new ImageSettings();
            imageSettings.setQuality(imageQuality);
            imageSettings.setCompressionMode(ImageSettings.e_ImageCompressjpeg);
            optimizerSettings.setColorGrayImageSettings(imageSettings);

            MonoImageSettings monoImageSettings = new MonoImageSettings();
            monoImageSettings.setQuality(monoImageQuality);
            monoImageSettings.setCompressionMode(MonoImageSettings.e_ImageCompressjbig2);
            optimizerSettings.setMonoImageSettings(monoImageSettings);

            optimizerSettings.setOptimizerOptions(OptimizerSettings.e_OptimizerCompressImages);
            Progressive progressive = Optimizer.optimize(doc, optimizerSettings, callback);
            onProgress(progressListener, progressive);
            int state = Progressive.e_ToBeContinued;
            while ((callback == null || !callback.isPause()) && state == Progressive.e_ToBeContinued) {
                onProgress(progressListener, progressive);
                state = progressive.resume();
            }
            return state == Progressive.e_Finished;
        } catch (PDFException e) {
            e.printStackTrace();
        }
        return false;
    }

    public static void onProgress(final OnProgressListener progressListener, Progressive progressive) {
        if (null != progressListener) {
            try {
                Observable.just(progressive.getRateOfProgress()).subscribeOn(AndroidSchedulers.mainThread())
                        .subscribe(new Consumer<Integer>() {
                            @Override
                            public void accept(Integer integer) throws Exception {
                                progressListener.onProgress(integer);
                            }
                        });
            } catch (PDFException e) {
                e.printStackTrace();
            }
        }
    }

    public static class PauseCallbackImpl extends PauseCallback {

        private boolean mPause;

        public PauseCallbackImpl(boolean pause) {
            mPause = pause;
        }

        @Override
        public boolean needToPauseNow() {
            return mPause;
        }

        public void setPause(boolean pause) {
            mPause = pause;
        }

        public boolean isPause() {
            return mPause;
        }
    }

}
