/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.modules.doc.docinfo;

import android.content.Context;
import android.content.DialogInterface;
import android.content.res.Configuration;
import android.graphics.Rect;
import android.text.TextUtils;
import android.view.View;
import android.view.ViewTreeObserver;
import android.view.inputmethod.EditorInfo;
import android.widget.EditText;
import android.widget.ImageView;
import android.widget.LinearLayout;
import android.widget.TextView;

import com.foxit.sdk.PDFException;
import com.foxit.sdk.PDFViewCtrl;
import com.foxit.sdk.common.WStringArray;
import com.foxit.sdk.pdf.Metadata;
import com.foxit.sdk.pdf.PDFDoc;
import com.foxit.sdk.pdf.PageBasicInfo;
import com.foxit.uiextensions.IPermissionProvider;
import com.foxit.uiextensions.R;
import com.foxit.uiextensions.UIExtensionsManager;
import com.foxit.uiextensions.annots.common.UIBtnImageView;
import com.foxit.uiextensions.controls.dialog.MatchDialog;
import com.foxit.uiextensions.controls.dialog.UIMatchDialog;
import com.foxit.uiextensions.controls.dialog.UIPopoverWin;
import com.foxit.uiextensions.controls.dialog.sheetmenu.IActionMenu;
import com.foxit.uiextensions.controls.dialog.sheetmenu.UIActionMenu;
import com.foxit.uiextensions.controls.filebrowser.imp.FileThumbnail;
import com.foxit.uiextensions.controls.toolbar.BaseBar;
import com.foxit.uiextensions.theme.ThemeUtil;
import com.foxit.uiextensions.utils.AppDisplay;
import com.foxit.uiextensions.utils.AppDmUtil;
import com.foxit.uiextensions.utils.AppFileUtil;
import com.foxit.uiextensions.utils.AppKeyboardUtil;
import com.foxit.uiextensions.utils.AppResource;
import com.foxit.uiextensions.utils.AppUtil;
import com.foxit.uiextensions.utils.SystemUiHelper;
import com.foxit.uiextensions.utils.UIMarqueeTextView;

import java.io.File;

import androidx.core.widget.NestedScrollView;
import androidx.fragment.app.FragmentActivity;

import static com.foxit.uiextensions.utils.AppUtil.pointsToCM;
import static com.foxit.uiextensions.utils.AppUtil.pointsToInches;
import static com.foxit.uiextensions.utils.AppUtil.pointsToMM;
import static com.foxit.uiextensions.utils.AppUtil.setThumbnailForImageView;
import static com.foxit.uiextensions.utils.AppUtil.upperCaseTextView;

/**
 * Class <CODE>DocInfoView</CODE> represents the basic information of pdf file.
 * <p/>
 * This class is used for showing the basic information of pdf file. It offers functions to initialize/show/FilePath Foxit PDF file basic information,
 * and also offers functions for global use.<br>
 * Any application should load Foxit PDF SDK by function {@link DocInfoView#init(String)} before calling any other Foxit PDF SDK functions.
 * When there is a need to show the basic information of pdf file, call function {@link DocInfoView#showActionMenu()}.
 */
public class DocInfoView {
    private Context mContext = null;
    private PDFViewCtrl mPdfViewCtrl = null;
    private boolean mIsPad = false;
    private String mFilePath = null;
    private SummaryInfo mSummaryInfo = null;
    private PermissionInfo mPermissionInfo = null;

    public void setAnchorView(View anchorView) {
        this.mAnchorView = anchorView;
    }

    private View mAnchorView;

    DocInfoView(Context context, PDFViewCtrl pdfViewCtrl) {
        mContext = context;
        mPdfViewCtrl = pdfViewCtrl;
        mIsPad = AppDisplay.isPad();
    }

    protected void init(String filePath) {
        setFilePath(filePath);
        mSummaryInfo = new SummaryInfo();
    }

    private void setFilePath(String path) {
        mFilePath = path;
    }

    public void show() {
        if (mSummaryInfo == null)
            return;
        mSummaryInfo.init();
        mSummaryInfo.show();
    }

    public void showActionMenu() {
        if (mSummaryInfo == null)
            return;
        mSummaryInfo.init();
        mSummaryInfo.showActionMenu(mAnchorView);
    }

    public void hideBottomDialog() {
        if (mSummaryInfo == null)
            return;
        mSummaryInfo.dismiss();
        hidePermissionBottomDialog();
    }

    public void hidePermissionBottomDialog() {
        if (mPermissionInfo == null) {
            return;
        }
        mPermissionInfo.dismiss();
    }

    public void setBackOnClickListener(View.OnClickListener onClickListener) {
        if (mSummaryInfo == null) return;
        mSummaryInfo.setBackOnClickListener(onClickListener);
    }

//    public void setDartModeColor() {
//        if (mSummaryInfo != null) {
//            mSummaryInfo.setDarkModeColor();
//        }
//        if (mPermissionInfo != null) {
//            mPermissionInfo.setDarkModeColor();
//        }
//    }

    abstract class DocInfo {
        protected View mRootLayout = null;
        protected View mTitleBarLayout;

        protected UIMatchDialog mDialog = null;
        protected UIActionMenu mBottomDialog = null;
        protected boolean mbDocResetSystemUiOnDismiss = true;

        protected UIMatchDialog mPermissionDialog = null;
        protected UIActionMenu mPermissionBottomDialog = null;
        protected boolean mbPermissionResetSystemUiOnDismiss = true;

        protected String mCaption = null;

        abstract void init();

        abstract void show();
    }

    /**
     * Class <CODE>SummaryInfo</CODE> represents the basic information of pdf file.
     * such as: file name, file path, file size and so on.
     * <p/>
     * This class is used for showing the basic information of pdf file. It offers functions to initialize/show/FilePath Foxit PDF file basic information,
     * and also offers functions for global use.<br>
     * Any application should load Foxit PDF SDK by function {@link SummaryInfo#init()} before calling any other Foxit PDF SDK functions.
     * When there is a need to show the basic information of pdf file, call function {@link SummaryInfo#show()}.
     */
    class SummaryInfo extends DocInfo {
        private DocumentInfo mDocumentInfo;
        private View.OnClickListener mBackClickListener;
        private ViewTreeObserver.OnGlobalLayoutListener mGlobalLayoutListener;
        private ImageView mShadow;
        private NestedScrollView mNestedScrollView;
        private TextView mTvFileNameValue;
        private TextView mTvDescriptionTitle;
        private TextView mTvSecurityTitle;
        private TextView mTvContent;
        private ImageView mIvSecurityDetail;
        private int mNightMode = -1;

        public void setBackOnClickListener(View.OnClickListener onClickListener) {
            mBackClickListener = onClickListener;
            if (mRootLayout == null) return;
            UIBtnImageView backImageView = mRootLayout.findViewById(R.id.rv_doc_info_top_bar_start);
            backImageView.setOnClickListener(onClickListener);
        }

        public class DocumentInfo {
            public String mFilePath = null;
            public String mFileName = null;
            public String mAuthor = null;
            public String mSubject = null;
            public String mCreateTime = null;
            public String mModTime = null;
            public long mFileSize = 0;
            public String mTitle = null;
            public String mHeader = null;
            public int mPageCount = 0;
            public String mPageSize = null;
            public String mCreator = null;
            public String mProducer = null;
            public String mKeywords = null;
            public static final String METADATA_KEY_TITLE = "Title";
            public static final String METADATA_KEY_SUBJECT = "Subject";
            public static final String METADATA_KEY_AUTHOR = "Author";
            public static final String METADATA_KEY_PRODUCER = "Producer";
            public static final String METADATA_KEY_CREATOR = "Creator";
            public static final String METADATA_KEY_KEYWORDS = "Keywords";
        }

        SummaryInfo() {
            mCaption = AppResource.getString(mContext.getApplicationContext(), R.string.rv_doc_info_fileinfo);
        }

//        void setDarkModeColor() {
//            mTvDocInfoTopBarCenter.setTextColor(mContext.getResources().getColor(R.color.t4));
//            mShadow.setBackgroundColor(mContext.getResources().getColor(R.color.p1));
//            mNestedScrollView.setBackgroundColor(mContext.getResources().getColor(R.color.b1));
//            mTvFileNameValue.setTextColor(mContext.getResources().getColor(R.color.t4));
//            mTvDescriptionTitle.setTextColor(mContext.getResources().getColor(R.color.t2));
//            mTvSecurityTitle.setTextColor(mContext.getResources().getColor(R.color.t2));
//            mTvContent.setTextColor(mContext.getResources().getColor(R.color.ux_text_color_body2_dark));
//            ThemeUtil.setTintList(mIvSecurityDetail, ThemeUtil.getPrimaryIconColor(mContext));
//            if (mDocumentInfo != null) {
//                initDocumentDescriptionView(mDocumentInfo);
//            }
//        }

        @Override
        void init() {
            String content = null;
            mNightMode = mContext.getResources().getConfiguration().uiMode & Configuration.UI_MODE_NIGHT_MASK;
            mRootLayout = View.inflate(mContext, R.layout.rv_doc_info, null);
            mTitleBarLayout = mRootLayout.findViewById(R.id.rv_doc_info_top_bar);
            mShadow = mRootLayout.findViewById(R.id.shadow);
            mNestedScrollView = mRootLayout.findViewById(R.id.rv_doc_info_nested_scroll_view);
            PDFDoc doc = mPdfViewCtrl.getDoc();

            if (doc == null) return;

            mDocumentInfo = getDocumentInfo();
            //file thumbnail image
            ImageView thumbnailImageView = mRootLayout.findViewById(R.id.rv_doc_info_thumbnail);
            setThumbnailForImageView(mContext, thumbnailImageView, mDocumentInfo.mFilePath, mThumbnailCallback);

            // file information
            mTvDescriptionTitle = mRootLayout.findViewById(R.id.rv_doc_info_description_title);
            mTvDescriptionTitle.setText(AppResource.getString(mContext.getApplicationContext(), R.string.rv_panel_edit_desc));
            upperCaseTextView(mTvDescriptionTitle);

            // filename
            mTvFileNameValue = (UIMarqueeTextView) mRootLayout.findViewById(R.id.rv_doc_info_file_name_value);
            mTvFileNameValue.setText(mDocumentInfo.mFileName);

            UIBtnImageView backImageView = mRootLayout.findViewById(R.id.rv_doc_info_top_bar_start);
            backImageView.setColorStateList(ThemeUtil.getItemIconColor(mContext));
            if (mBackClickListener == null) {
                mBackClickListener = new View.OnClickListener() {
                    @Override
                    public void onClick(View v) {
                        hideBottomDialog();
                    }
                };
            }
            backImageView.setOnClickListener(mBackClickListener);
            initDocumentDescriptionView(mDocumentInfo);
            mRootLayout.findViewById(R.id.rv_doc_info_root_ll).setOnClickListener(new View.OnClickListener() {
                @Override
                public void onClick(View v) {
                    if (Boolean.TRUE.equals(mRootLayout.getTag())) {
                        v.clearFocus();
                        AppKeyboardUtil.hideInputMethodWindow(mContext, v);
                    }
                }
            });

            mTvSecurityTitle = mRootLayout.findViewById(R.id.rv_doc_info_security_title);
            mTvSecurityTitle.setText(AppResource.getString(mContext.getApplicationContext(), R.string.rv_doc_info_security));
            upperCaseTextView(mTvSecurityTitle);

            mTvContent = mRootLayout.findViewById(R.id.rv_doc_info_security_content);

            mIvSecurityDetail = mRootLayout.findViewById(R.id.rv_doc_info_security_detail);
            ThemeUtil.setTintList(mIvSecurityDetail, ThemeUtil.getPrimaryIconColor(mContext));
            mRootLayout.findViewById(R.id.rv_doc_info_security_layout).setOnClickListener(new View.OnClickListener() {
                @Override
                public void onClick(View v) {
                    mPermissionInfo = new PermissionInfo();
                    mPermissionInfo.init();
                    if (mDialog != null && mDialog.isShowing()) {
                        mPermissionInfo.show();
                    } else {
                        mSummaryInfo.dismiss();
                        mPermissionInfo.show(mAnchorView);
                    }
                }
            });

            try {
                switch (doc.getEncryptionType()) {
                    case PDFDoc.e_EncryptPassword:
                        content = AppResource.getString(mContext.getApplicationContext(), R.string.rv_doc_info_security_standard);
                        break;
                    case PDFDoc.e_EncryptCertificate:
                        content = AppResource.getString(mContext.getApplicationContext(), R.string.rv_doc_info_security_pubkey);
                        break;
                    case PDFDoc.e_EncryptRMS:
                        content = AppResource.getString(mContext.getApplicationContext(), R.string.rv_doc_info_security_rms);
                        break;
                    case PDFDoc.e_EncryptCustom:
                        content = AppResource.getString(mContext.getApplicationContext(), R.string.rv_doc_info_security_custom);
                        break;
                    case PDFDoc.e_EncryptCDRM:
                        content = AppResource.getString(mContext.getApplicationContext(), R.string.rv_doc_info_security_cdrm);
                        break;
                    default:
                        if (mFilePath != null && mFilePath.toLowerCase().endsWith(".ppdf")) {
                            content = AppResource.getString(mContext.getApplicationContext(), R.string.rv_doc_info_security_rms);
                        } else {
                            content = AppResource.getString(mContext.getApplicationContext(), R.string.rv_doc_info_security_no);
                            mRootLayout.findViewById(R.id.rv_doc_info_security_layout).setOnClickListener(null);
                            mIvSecurityDetail.setVisibility(View.GONE);
                        }
                        break;
                }
            } catch (
                    PDFException e) {
                e.printStackTrace();
            }
            mTvContent.setText(content);
        }

        private void dismiss() {
            if (null != mDialog && mDialog.isShowing())
                mDialog.dismiss();
            if (null != mBottomDialog) {
                mbDocResetSystemUiOnDismiss = false;
                mBottomDialog.dismiss();
                mBottomDialog = null;
            }
        }

        private void initDocumentDescriptionView(DocumentInfo info) {
            LinearLayout descriptionLayout = mRootLayout.findViewById(R.id.rv_doc_info_description_layout);
            descriptionLayout.removeAllViews();
            descriptionLayout.addView(addDescriptionItem(R.string.menu_more_pdf_version, info.mHeader, null, null));
            descriptionLayout.addView(addDescriptionItem(R.string.menu_more_fast_web_view, mContext.getString(isFastWebViewPDF() ? R.string.fx_string_yes : R.string.fx_string_no), null, null));
            descriptionLayout.addView(addDescriptionItem(R.string.menu_more_tagged_pdf, mContext.getString(isTaggedPDF() ? R.string.fx_string_yes : R.string.fx_string_no), null, null));
            descriptionLayout.addView(addDescriptionItem(R.string.createpdf_new_pagesize, info.mPageSize, null, null));
            descriptionLayout.addView(addDescriptionItem(R.string.menu_more_pages, info.mPageCount + "", null, null));
            descriptionLayout.addView(addDescriptionItem(R.string.fm_history_file_name, info.mTitle,
                    AppResource.getString(mContext, R.string.menu_more_add_a_title), DocumentInfo.METADATA_KEY_TITLE));
            descriptionLayout.addView(addDescriptionItem(R.string.fm_history_pdf_subject, info.mSubject,
                    AppResource.getString(mContext, R.string.menu_more_add_a_subject), DocumentInfo.METADATA_KEY_SUBJECT));
            descriptionLayout.addView(addDescriptionItem(R.string.menu_more_creator, info.mCreator,
                    AppResource.getString(mContext, R.string.menu_more_add_a_creator), DocumentInfo.METADATA_KEY_CREATOR));
            descriptionLayout.addView(addDescriptionItem(R.string.fm_history_pdf_author, info.mAuthor,
                    AppResource.getString(mContext, R.string.menu_more_add_an_author), DocumentInfo.METADATA_KEY_AUTHOR));
            descriptionLayout.addView(addDescriptionItem(R.string.menu_more_producer, info.mProducer,
                    AppResource.getString(mContext, R.string.menu_more_add_a_producer), DocumentInfo.METADATA_KEY_PRODUCER));
            descriptionLayout.addView(addDescriptionItem(R.string.menu_more_keywords, info.mKeywords,
                    AppResource.getString(mContext, R.string.menu_more_add_keywords), DocumentInfo.METADATA_KEY_KEYWORDS));
            descriptionLayout.addView(addDescriptionItem(R.string.menu_more_document_size, AppUtil.fileSizeToString(info.mFileSize), null, null));
            descriptionLayout.addView(addDescriptionItem(R.string.fm_history_file_last_modify_time, info.mModTime, null, null));
            descriptionLayout.addView(addDescriptionItem(R.string.fm_history_file_create_time, info.mCreateTime, null, null));
            if (mIsPad) {
                descriptionLayout.getLayoutParams().width = AppResource.getDimensionPixelSize(mContext, R.dimen.ux_margin_375dp);
            }
        }

        private final View.OnFocusChangeListener mOnFocusChangeListener = new View.OnFocusChangeListener() {
            @Override
            public void onFocusChange(View v, boolean hasFocus) {
                if (hasFocus || v.getTag() == null) return;
                saveMetaValueOrNot((String) v.getTag(), ((TextView) v).getText().toString());
            }
        };

        View addDescriptionItem(int titleResId, String content, String editHint, String tag) {
            View item = View.inflate(mContext, R.layout.rv_doc_info_description_item, null);
            ((TextView) item.findViewById(R.id.doc_description_item_title)).setText(AppResource.getString(mContext, titleResId));
            TextView contentTextView = item.findViewById(R.id.doc_description_item_content);
            final EditText contentEditText = item.findViewById(R.id.doc_description_item_edit_content);
            if (null == editHint) {
                contentEditText.setVisibility(View.GONE);
                contentTextView.setVisibility(View.VISIBLE);
                contentTextView.setText(content);
            } else {
                contentTextView.setVisibility(View.GONE);
                contentEditText.setVisibility(View.VISIBLE);
                if (tag != null)
                    contentEditText.setTag(tag);
                contentEditText.setText(content);
                contentEditText.setHint(editHint);
                if (AppDisplay.isPad()) { // SDKRD-9313
                    contentEditText.setImeOptions(EditorInfo.IME_FLAG_NO_EXTRACT_UI);
                } else {
                    contentEditText.setImeOptions(EditorInfo.IME_ACTION_DONE);
                }
                contentEditText.setOnFocusChangeListener(mOnFocusChangeListener);
                if(!((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).isEnableModification()){
                    contentEditText.setEnabled(false);
                }
                contentEditText.setFocusable(false);
                contentEditText.setOnClickListener(new View.OnClickListener() {
                    @Override
                    public void onClick(View view) {
                        final Runnable doRunnable = new Runnable() {
                            @Override
                            public void run() {
                                contentEditText.setFocusable(true);
                                contentEditText.setFocusableInTouchMode(true);
                                contentEditText.requestFocus();
                                AppUtil.showSoftInput(contentEditText);
                            }
                        };

                        if (((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getPermissionProvider() != null) {
                            ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getPermissionProvider().checkPermission(
                                    IPermissionProvider.FUNCTION_DOC_INFO_EDIT, new IPermissionProvider.IPermissionState() {
                                        @Override
                                        public void onState(int state) {
                                            if (state == IPermissionProvider.PERMISSION_STATE_SHOW) {
                                                doRunnable.run();
                                            }
                                        }
                                    });
                        } else {
                            doRunnable.run();
                        }
                    }
                });
            }
            return item;
        }

        @Override
        void show() {
            if (mTitleBarLayout != null)
                mTitleBarLayout.setVisibility(View.GONE);

            extensionsManager = (UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager();
            mDialog = new UIMatchDialog(extensionsManager.getAttachedActivity());
            mDialog.setTitle(mCaption);
            mDialog.setTitlePosition(BaseBar.TB_Position.Position_CENTER);
            mDialog.setBackButtonVisible(View.VISIBLE);
            mDialog.setBackButtonStyle(MatchDialog.TEXT_BACK);
            mDialog.setBackButtonText(AppResource.getString(mContext, R.string.fx_string_close));
            mDialog.setBackButtonTintList(ThemeUtil.getItemIconColor(mContext));
            mDialog.getTitleBar().setMiddleButtonCenter(true);
            mDialog.setContentView(mRootLayout);
            mDialog.setListener(new MatchDialog.DialogListener() {
                @Override
                public void onResult(long btType) {
                }

                @Override
                public void onBackClick() {
                    mDialog.dismiss();
                }

                @Override
                public void onTitleRightButtonClick() {
                }
            });
            mDialog.setOnDLDismissListener(new MatchDialog.DismissListener() {
                @Override
                public void onDismiss() {
                    mRootLayout.clearFocus();
                    if (mConfigurationChangedListener != null) {
                        extensionsManager.unregisterConfigurationChangedListener(mConfigurationChangedListener);
                        mConfigurationChangedListener = null;
                    }
                }
            });
            mDialog.showDialog(true);

            if (mConfigurationChangedListener != null)
                extensionsManager.unregisterConfigurationChangedListener(mConfigurationChangedListener);
            mConfigurationChangedListener = new UIExtensionsManager.ConfigurationChangedListener() {
                @Override
                public void onConfigurationChanged(Configuration newConfig) {
                    if (mDialog != null && mDialog.isShowing()) {
                        mDialog.resetWH();
                        mDialog.showDialog();
                    }
                }
            };
            extensionsManager.registerConfigurationChangedListener(mConfigurationChangedListener);
        }

        private UIExtensionsManager.ConfigurationChangedListener mConfigurationChangedListener;
        UIExtensionsManager extensionsManager;

        public void showActionMenu(View anchorView) {
            if (anchorView == null) return;
            extensionsManager = (UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager();
            if (mBottomDialog != null && mBottomDialog.isShowing()) {
                if (mGlobalLayoutListener != null) {
                    mRootLayout.getViewTreeObserver().removeOnGlobalLayoutListener(mGlobalLayoutListener);
                    mGlobalLayoutListener = null;
                    mRootLayout.clearFocus();
                }
                if (mConfigurationChangedListener != null) {
                    extensionsManager.unregisterConfigurationChangedListener(mConfigurationChangedListener);
                    mConfigurationChangedListener = null;
                }
                mBottomDialog.setOnDismissListener(null);
                mBottomDialog.dismiss();
            }
            mBottomDialog = (UIActionMenu) UIActionMenu.newInstance((FragmentActivity) extensionsManager.getAttachedActivity());
            mBottomDialog.setContentView(mRootLayout);
            mBottomDialog.setOnDismissListener(new IActionMenu.OnActionMenuDismissListener() {
                @Override
                public void onDismiss(IActionMenu actionMenu) {
                    if (mbDocResetSystemUiOnDismiss) {
                        if (SystemUiHelper.getInstance().isFullScreen()) {
//                            if (AppDisplay.isPad())
                            SystemUiHelper.getInstance().hideSystemUI(extensionsManager.getAttachedActivity());
//                            else
//                                SystemUiHelper.getInstance().hideStatusBar(extensionsManager.getAttachedActivity());
                        }
                    }

                    if (mGlobalLayoutListener != null) {
                        mRootLayout.getViewTreeObserver().removeOnGlobalLayoutListener(mGlobalLayoutListener);
                        mGlobalLayoutListener = null;
                        mRootLayout.clearFocus();
                    }
                    if (mRootLayout != null) {
                        mRootLayout.postDelayed(new Runnable() {
                            @Override
                            public void run() {
                                if (mConfigurationChangedListener != null) {
                                    extensionsManager.unregisterConfigurationChangedListener(mConfigurationChangedListener);
                                    mConfigurationChangedListener = null;
                                }
                            }
                        }, 100);
                    }
                    AppUtil.removeViewFromParent(mRootLayout);
                    mBottomDialog = null;
                }
            });
            if (AppDisplay.isPad()) {
                mBottomDialog.setWidth(AppResource.getDimensionPixelSize(mContext, R.dimen.ux_margin_375dp));
            }
            if (mGlobalLayoutListener == null) {
                mGlobalLayoutListener = new ViewTreeObserver.OnGlobalLayoutListener() {
                    @Override
                    public void onGlobalLayout() {
                        Rect r = new Rect();
                        mRootLayout.getWindowVisibleDisplayFrame(r);
                        int heightDiff = mRootLayout.getRootView().getHeight() - (r.bottom - r.top);
                        if (heightDiff > 500) {
                            mRootLayout.setTag(true);
                        } else {
                            mRootLayout.setTag(null);
                        }
                    }
                };
                mRootLayout.getViewTreeObserver().addOnGlobalLayoutListener(mGlobalLayoutListener);
            }
            if (mConfigurationChangedListener == null) {
                mConfigurationChangedListener = new UIExtensionsManager.ConfigurationChangedListener() {
                    @Override
                    public void onConfigurationChanged(Configuration newConfig) {
                        if (mNightMode != (newConfig.uiMode & Configuration.UI_MODE_NIGHT_MASK)) {
                            mNightMode = newConfig.uiMode & Configuration.UI_MODE_NIGHT_MASK;
                            return;
                        }
                        mRootLayout.postDelayed(new Runnable() {
                            @Override
                            public void run() {
                                mSummaryInfo.showActionMenu(mAnchorView);
                            }
                        }, 100);
                    }
                };
                extensionsManager.registerConfigurationChangedListener(mConfigurationChangedListener);
            }
            Rect rect = new Rect();
            anchorView.getGlobalVisibleRect(rect);
            mBottomDialog.setAutoResetSystemUiOnDismiss(false);
            mBottomDialog.show(extensionsManager.getRootView(), rect, UIPopoverWin.ARROW_TOP, 0);
            mbDocResetSystemUiOnDismiss = true;
        }

        DocumentInfo getDocumentInfo() {
            DocumentInfo info = new DocumentInfo();
            PDFDoc doc = mPdfViewCtrl.getDoc();
            info.mFilePath = mFilePath;
            if (mFilePath != null) {
                info.mFileName = AppFileUtil.getFileNameWithoutExt(mFilePath);
                File file = new File(mFilePath);
                info.mFileSize = file.length();
            }

            try {
                Metadata metadata = new Metadata(doc);
                info.mAuthor = getMetaValue(metadata, DocumentInfo.METADATA_KEY_AUTHOR, "");
                info.mSubject = getMetaValue(metadata, DocumentInfo.METADATA_KEY_SUBJECT, "");
                info.mCreateTime = AppDmUtil.formatDocumentDate(AppDmUtil.FORMAT_MMM_DD_YYYY_HH_MM_SS, metadata.getCreationDateTime());
                info.mModTime = AppDmUtil.formatDocumentDate(AppDmUtil.FORMAT_MMM_DD_YYYY_HH_MM_SS, metadata.getModifiedDateTime());
                info.mTitle = getMetaValue(metadata, DocumentInfo.METADATA_KEY_TITLE, "");
                info.mHeader = doc.getHeader();
                info.mPageCount = doc.getPageCount();
                info.mProducer = getMetaValue(metadata, DocumentInfo.METADATA_KEY_PRODUCER, "");
                info.mCreator = getMetaValue(metadata, DocumentInfo.METADATA_KEY_CREATOR, "");
                info.mKeywords = getMetaValue(metadata, DocumentInfo.METADATA_KEY_KEYWORDS, "");
                PageBasicInfo pageBasicInfo = doc.getPageBasicInfo(mPdfViewCtrl.getCurrentPage());

                int measurementUnits = ((UIExtensionsManager)mPdfViewCtrl.getUIExtensionsManager()).getMeasurementUnits();
                switch (measurementUnits) {
                    case UIExtensionsManager.MEASUREMENT_UNIT_CM:
                        info.mPageSize = String.format("%1$.2f*%2$.2f cm",
                                pointsToCM(pageBasicInfo.getWidth()), pointsToCM(pageBasicInfo.getHeight()));
                        break;
                    case UIExtensionsManager.MEASUREMENT_UNIT_MM:
                        info.mPageSize = String.format("%1$.2f*%2$.2f mm",
                                pointsToMM(pageBasicInfo.getWidth()), pointsToMM(pageBasicInfo.getHeight()));
                        break;
                    default:
                        info.mPageSize = String.format(AppResource.getString(mContext, R.string.menu_more_page_size_display),
                                pointsToInches(pageBasicInfo.getWidth()), pointsToInches(pageBasicInfo.getHeight()));
                        break;
                }
            } catch (Exception e) {
                e.printStackTrace();
            }
            return info;
        }

        private String getMetaValue(Metadata metadata, String key, String defaultValue) throws
                PDFException {
            WStringArray subArray = metadata.getValues(key);
            return subArray.getSize() == 0 ? defaultValue : subArray.getAt(0);
        }


        private void saveMetaValueOrNot(String key, String value) {
            try {
                Metadata metadata = new Metadata(mPdfViewCtrl.getDoc());
                WStringArray oldValueArray = metadata.getValues(key);
                if (oldValueArray.getSize() == 0 && !TextUtils.isEmpty(value)) {
                    oldValueArray.add(value);
                    metadata.setValues(key, oldValueArray);
                    ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager().setDocModified(true);
                } else if (oldValueArray.getSize() > 0 && !oldValueArray.getAt(0).contentEquals(value)) {
                    oldValueArray.removeAll();
                    oldValueArray.add(value);
                    metadata.setValues(key, oldValueArray);
                    ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager().setDocModified(true);
                }
            } catch (PDFException e) {
                e.printStackTrace();
            }

        }

        private final FileThumbnail.ThumbnailCallback mThumbnailCallback = new FileThumbnail.ThumbnailCallback() {
            @Override
            public void result(boolean succeed, final String filePath) {
                if (succeed) {
                    mRootLayout.post(new Runnable() {
                        @Override
                        public void run() {
                            setThumbnailForImageView(mContext, (ImageView) mRootLayout.findViewById(R.id.rv_doc_info_thumbnail), filePath, mThumbnailCallback);
                        }
                    });
                }
            }
        };
    }

    private boolean isFastWebViewPDF() {
        PDFDoc doc = mPdfViewCtrl.getDoc();
        if (doc != null) {
            try {
                return doc.isLinearized();
            } catch (PDFException e) {
                e.printStackTrace();
            }
        }
        return false;
    }

    private boolean isTaggedPDF() {
        PDFDoc doc = mPdfViewCtrl.getDoc();
        if (doc != null) {
            try {
                return doc.isTaggedPDF();
            } catch (PDFException e) {
                e.printStackTrace();
            }
        }
        return false;
    }

    /**
     * Class <CODE>PermissionInfo</CODE> represents the permission information of pdf file.
     * such as: print, modify, fill form, extract and so on.
     * <p/>
     * This class is used for showing the permission information of pdf file. It offers functions to initialize/show Foxit PDF file basic information,
     * and also offers functions for global use.<br>
     * Any application should load Foxit PDF SDK by function {@link PermissionInfo#init()} before calling any other Foxit PDF SDK functions.
     * When there is a need to show the basic information of pdf file, call function {@link PermissionInfo#show()}.
     */
    class PermissionInfo extends DocInfo {

        private UIExtensionsManager.ConfigurationChangedListener mConfigurationChangedListener;
        private ImageView mShadow;
        private TextView mTvTopBarCenter;
        private TextView mTvPrint;
        private TextView mTvFillform;
        private TextView mTvAnnotform;
        private TextView mTvAssemble;
        private TextView mTvModify;
        private TextView mTvExtractaccess;
        private TextView mTvExtract;
        private TextView mTvSigning;


        private TextView mTvPrintOf;
        private TextView mTvFillformOf;
        private TextView mTvAnnotformOf;
        private TextView mTvAssembleOf;
        private TextView mTvModifyOf;
        private TextView mTvExtractaccessOf;
        private TextView mTvExtractOf;
        private TextView mTvSigningOf;

        private int mNightMode = -1;

        PermissionInfo() {
            mCaption = AppResource.getString(mContext.getApplicationContext(), R.string.menu_more_permission);
        }

//        void setDarkModeColor() {
//            mShadow.setBackgroundColor(mContext.getResources().getColor(R.color.p1));
//            mTvTopBarCenter.setTextColor(mContext.getResources().getColor(R.color.t4));
//
//            mTvPrint.setTextColor(mContext.getResources().getColor(R.color.t4));
//            mTvFillform.setTextColor(mContext.getResources().getColor(R.color.t4));
//            mTvAnnotform.setTextColor(mContext.getResources().getColor(R.color.t4));
//            mTvAssemble.setTextColor(mContext.getResources().getColor(R.color.t4));
//            mTvModify.setTextColor(mContext.getResources().getColor(R.color.t4));
//            mTvExtractaccess.setTextColor(mContext.getResources().getColor(R.color.t4));
//            mTvExtract.setTextColor(mContext.getResources().getColor(R.color.t4));
//            mTvSigning.setTextColor(mContext.getResources().getColor(R.color.t4));
//            mTvPrintOf.setTextColor(mContext.getResources().getColor(R.color.t4));
//            mTvFillformOf.setTextColor(mContext.getResources().getColor(R.color.t4));
//            mTvAnnotformOf.setTextColor(mContext.getResources().getColor(R.color.t4));
//            mTvAssembleOf.setTextColor(mContext.getResources().getColor(R.color.t4));
//            mTvModifyOf.setTextColor(mContext.getResources().getColor(R.color.t4));
//            mTvExtractaccessOf.setTextColor(mContext.getResources().getColor(R.color.t4));
//            mTvExtractOf.setTextColor(mContext.getResources().getColor(R.color.t4));
//            mTvSigningOf.setTextColor(mContext.getResources().getColor(R.color.t4));
//        }

        void init() {
            mNightMode = mContext.getResources().getConfiguration().uiMode & Configuration.UI_MODE_NIGHT_MASK;
            mRootLayout = View.inflate(mContext.getApplicationContext(), R.layout.rv_doc_info_permissioin, null);
            mTitleBarLayout = mRootLayout.findViewById(R.id.rv_doc_info_top_bar);

            PDFDoc doc = mPdfViewCtrl.getDoc();
            if (doc == null)
                return;
            mShadow = mRootLayout.findViewById(R.id.shadow);
            mTvTopBarCenter = mRootLayout.findViewById(R.id.rv_doc_info_top_bar_center);
            mTvPrint = mRootLayout.findViewById(R.id.rv_doc_info_permission_print);
            mTvPrint.setText(AppResource.getString(mContext.getApplicationContext(), R.string.rv_doc_info_permission_print));

            mTvFillform = mRootLayout.findViewById(R.id.rv_doc_info_permission_fillform);
            mTvFillform.setText(AppResource.getString(mContext.getApplicationContext(), R.string.rv_doc_info_permission_fillform));

            mTvAnnotform = mRootLayout.findViewById(R.id.rv_doc_info_permission_annotform);
            mTvAnnotform.setText(AppResource.getString(mContext.getApplicationContext(), R.string.rv_doc_info_permission_annotform));

            mTvAssemble = mRootLayout.findViewById(R.id.rv_doc_info_permission_assemble);
            mTvAssemble.setText(AppResource.getString(mContext.getApplicationContext(), R.string.rv_doc_info_permission_assemble));

            mTvModify = mRootLayout.findViewById(R.id.rv_doc_info_permission_modify);
            mTvModify.setText(AppResource.getString(mContext.getApplicationContext(), R.string.rv_doc_info_permission_modify));

            mTvExtractaccess = mRootLayout.findViewById(R.id.rv_doc_info_permission_extractaccess);
            mTvExtractaccess.setText(AppResource.getString(mContext.getApplicationContext(), R.string.rv_doc_info_permission_extractaccess));

            mTvExtract = mRootLayout.findViewById(R.id.rv_doc_info_permission_extract);
            mTvExtract.setText(AppResource.getString(mContext.getApplicationContext(), R.string.rv_doc_info_permission_extract));

            mTvSigning = mRootLayout.findViewById(R.id.rv_doc_info_permission_signing);
            mTvSigning.setText(AppResource.getString(mContext.getApplicationContext(), R.string.rv_doc_info_permission_signing));

            UIBtnImageView backImageView = mRootLayout.findViewById(R.id.rv_doc_info_top_bar_start);
            backImageView.setColorStateList(ThemeUtil.getItemIconColor(mContext));
            backImageView.setOnClickListener(new View.OnClickListener() {
                @Override
                public void onClick(View v) {
                    mbPermissionResetSystemUiOnDismiss = false;
                    mPermissionBottomDialog.dismiss();
                    showActionMenu();
                }
            });

            // on off switch
            mTvPrintOf = mRootLayout.findViewById(R.id.rv_doc_info_permission_print_of);
            mTvFillformOf = mRootLayout.findViewById(R.id.rv_doc_info_permission_fillform_of);
            mTvAnnotformOf = mRootLayout.findViewById(R.id.rv_doc_info_permission_annotform_of);
            mTvAssembleOf = mRootLayout.findViewById(R.id.rv_doc_info_permission_assemble_of);
            mTvModifyOf = mRootLayout.findViewById(R.id.rv_doc_info_permission_modify_of);
            mTvExtractaccessOf = mRootLayout.findViewById(R.id.rv_doc_info_permission_extractaccess_of);
            mTvExtractOf = mRootLayout.findViewById(R.id.rv_doc_info_permission_extract_of);
            mTvSigningOf = mRootLayout.findViewById(R.id.rv_doc_info_permission_signing_of);

            String allowed = AppResource.getString(mContext.getApplicationContext(), R.string.fx_string_allowed);
            String notAllowed = AppResource.getString(mContext.getApplicationContext(), R.string.fx_string_notallowed);

            mTvPrintOf.setText(((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager().canPrint() ? allowed : notAllowed);
            mTvFillformOf.setText(((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager().canFillForm() ? allowed : notAllowed);
            mTvAnnotformOf.setText(((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager().canAddAnnot() ? allowed : notAllowed);
            mTvAssembleOf.setText(((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager().canAssemble() ? allowed : notAllowed);
            mTvModifyOf.setText(((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager().canModifyContents() ? allowed : notAllowed);
            mTvExtractaccessOf.setText(((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager().canCopyForAssess() ? allowed : notAllowed);
            mTvExtractOf.setText(((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager().canCopy() ? allowed : notAllowed);
            boolean canSign = ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager().canSigning();
            mTvSigningOf.setText(canSign ? allowed : notAllowed);
        }

        private void dismiss() {
            if (null != mPermissionDialog)
                mPermissionDialog.dismiss();
            if (null != mPermissionBottomDialog) {
                mPermissionBottomDialog.dismiss();
                mPermissionBottomDialog = null;
            }
        }

        void show() {
            if (mPdfViewCtrl.getUIExtensionsManager() == null) return;
            final UIExtensionsManager uiExtensionsManager = ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager());
            Context context = uiExtensionsManager.getAttachedActivity();
            if (context == null) return;
            if (mTitleBarLayout != null)
                mTitleBarLayout.setVisibility(View.GONE);

            mPermissionDialog = new UIMatchDialog(context);
            mPermissionDialog.setTitle(mCaption);
            mPermissionDialog.setTitlePosition(BaseBar.TB_Position.Position_CENTER);
            mPermissionDialog.setBackButtonVisible(View.VISIBLE);
            mPermissionDialog.setBackButtonStyle(MatchDialog.TEXT_BACK);
            mPermissionDialog.setBackButtonText(AppResource.getString(mContext, R.string.fx_string_close));
            mPermissionDialog.setBackButtonTintList(ThemeUtil.getItemIconColor(mContext));
            mPermissionDialog.getTitleBar().setMiddleButtonCenter(true);
            mPermissionDialog.setContentView(mRootLayout);
            mPermissionDialog.setListener(new MatchDialog.DialogListener() {
                @Override
                public void onResult(long btType) {
                }

                @Override
                public void onBackClick() {
                    mPermissionDialog.dismiss();
                }

                @Override
                public void onTitleRightButtonClick() {
                }
            });
            mPermissionDialog.setOnDismissListener(new DialogInterface.OnDismissListener() {
                @Override
                public void onDismiss(DialogInterface dialog) {
                    if (mConfigurationChangedListener != null) {
                        uiExtensionsManager.unregisterConfigurationChangedListener(mConfigurationChangedListener);
                        mConfigurationChangedListener = null;
                    }
                }
            });
            mPermissionDialog.showDialog();

            if (mConfigurationChangedListener != null)
                uiExtensionsManager.unregisterConfigurationChangedListener(mConfigurationChangedListener);
            mConfigurationChangedListener = new UIExtensionsManager.ConfigurationChangedListener() {
                @Override
                public void onConfigurationChanged(Configuration newConfig) {
                    if (mPermissionDialog != null && mPermissionDialog.isShowing()) {
                        mPermissionDialog.resetWH();
                        mPermissionDialog.showDialog();
                    }
                }
            };
            uiExtensionsManager.registerConfigurationChangedListener(mConfigurationChangedListener);
        }

        public void show(View anchorView) {
            if (anchorView == null) return;
            final UIExtensionsManager extensionsManager = (UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager();
            if (mPermissionBottomDialog != null && mPermissionBottomDialog.isShowing()) {
                extensionsManager.unregisterConfigurationChangedListener(mConfigurationChangedListener);
                mConfigurationChangedListener = null;
                mPermissionBottomDialog.setOnDismissListener(null);
                mPermissionBottomDialog.dismiss();
            }
            mPermissionBottomDialog = (UIActionMenu) UIActionMenu.newInstance((FragmentActivity) extensionsManager.getAttachedActivity());
            mPermissionBottomDialog.setContentView(mRootLayout);
            mPermissionBottomDialog.setOnDismissListener(new IActionMenu.OnActionMenuDismissListener() {
                @Override
                public void onDismiss(IActionMenu actionMenu) {
                    if (mbPermissionResetSystemUiOnDismiss) {
                        if (SystemUiHelper.getInstance().isFullScreen()) {
//                            if (AppDisplay.isPad())
                            SystemUiHelper.getInstance().hideSystemUI(extensionsManager.getAttachedActivity());
//                            else
//                                SystemUiHelper.getInstance().hideStatusBar(extensionsManager.getAttachedActivity());
                        }
                    }

                    if (mRootLayout != null) {
                        mRootLayout.postDelayed(new Runnable() {
                            @Override
                            public void run() {
                                if (mConfigurationChangedListener != null) {
                                    extensionsManager.unregisterConfigurationChangedListener(mConfigurationChangedListener);
                                    mConfigurationChangedListener = null;
                                }
                            }
                        }, 100);
                    }
                }
            });
            if (AppDisplay.isPad()) {
                mPermissionBottomDialog.setWidth(AppResource.getDimensionPixelSize(mContext, R.dimen.ux_margin_375dp));
            }
            Rect rect = new Rect();
            anchorView.getGlobalVisibleRect(rect);
            if (mConfigurationChangedListener == null) {
                mConfigurationChangedListener = new UIExtensionsManager.ConfigurationChangedListener() {
                    @Override
                    public void onConfigurationChanged(Configuration newConfig) {
                        if (mNightMode != (newConfig.uiMode & Configuration.UI_MODE_NIGHT_MASK)) {
                            mNightMode = newConfig.uiMode & Configuration.UI_MODE_NIGHT_MASK;
                            return;
                        }
                        mRootLayout.postDelayed(new Runnable() {
                            @Override
                            public void run() {
                                mPermissionInfo.show(mAnchorView);
                            }
                        }, 100);
                    }
                };
                extensionsManager.registerConfigurationChangedListener(mConfigurationChangedListener);
            }
            mPermissionBottomDialog.setAutoResetSystemUiOnDismiss(false);
            mPermissionBottomDialog.show(extensionsManager.getRootView(), rect, UIPopoverWin.ARROW_TOP, 0);
            mbPermissionResetSystemUiOnDismiss = true;
        }
    }
}

