/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.modules.doc.saveas;


import android.app.Activity;
import android.content.Context;
import android.content.DialogInterface;
import android.content.res.Configuration;
import android.graphics.Rect;
import android.view.View;
import android.widget.ImageView;
import android.widget.TextView;

import com.foxit.sdk.PDFException;
import com.foxit.sdk.PDFViewCtrl;
import com.foxit.sdk.addon.optimization.ImageSettings;
import com.foxit.sdk.addon.optimization.MonoImageSettings;
import com.foxit.sdk.common.Constants;
import com.foxit.sdk.common.fxcrt.FileWriterCallback;
import com.foxit.sdk.pdf.PDFDoc;
import com.foxit.sdk.pdf.PDFPage;
import com.foxit.uiextensions.DocumentManager;
import com.foxit.uiextensions.IThemeEventListener;
import com.foxit.uiextensions.Module;
import com.foxit.uiextensions.R;
import com.foxit.uiextensions.UIExtensionsManager;
import com.foxit.uiextensions.annots.common.UIBtnImageView;
import com.foxit.uiextensions.annots.link.LinkModule;
import com.foxit.uiextensions.controls.dialog.CircleProgressBar;
import com.foxit.uiextensions.controls.dialog.FxProgressDialog;
import com.foxit.uiextensions.controls.dialog.MatchDialog;
import com.foxit.uiextensions.controls.dialog.UIDialog;
import com.foxit.uiextensions.controls.dialog.UITextEditDialog;
import com.foxit.uiextensions.controls.dialog.fileselect.UISaveAsDialog;
import com.foxit.uiextensions.controls.dialog.saveas.SaveAsBean;
import com.foxit.uiextensions.controls.dialog.saveas.UIDocSaveAsDialog;
import com.foxit.uiextensions.event.DocEventListener;
import com.foxit.uiextensions.home.local.LocalModule;
import com.foxit.uiextensions.modules.crop.CropModule;
import com.foxit.uiextensions.modules.doc.CropPDF;
import com.foxit.uiextensions.modules.doc.FlattenPDF;
import com.foxit.uiextensions.modules.doc.OnProgressCancelListener;
import com.foxit.uiextensions.modules.doc.OnProgressListener;
import com.foxit.uiextensions.modules.doc.OptimizerPDF;
import com.foxit.uiextensions.theme.ThemeConfig;
import com.foxit.uiextensions.theme.ThemeUtil;
import com.foxit.uiextensions.utils.AppDmUtil;
import com.foxit.uiextensions.utils.AppFileUtil;
import com.foxit.uiextensions.utils.AppResource;
import com.foxit.uiextensions.utils.AppStorageManager;
import com.foxit.uiextensions.utils.AppTheme;
import com.foxit.uiextensions.utils.AppUtil;
import com.foxit.uiextensions.utils.UIToast;
import com.foxit.uiextensions.utils.thread.AppThreadManager;

import java.io.File;

import io.reactivex.android.schedulers.AndroidSchedulers;
import io.reactivex.disposables.CompositeDisposable;
import io.reactivex.functions.Consumer;
import io.reactivex.schedulers.Schedulers;

public class DocSaveAsModule implements Module {

    private final PDFViewCtrl mPdfViewCtrl;
    private final PDFViewCtrl.UIExtensionsManager mUiExtensionsManager;
    private final Context mContext;
    private FxProgressDialog mProgressDialog;
    private UISaveAsDialog mSaveAsDialog;
    private UIDocSaveAsDialog mDocSaveAsDialog;
    private UITextEditDialog mFlattenDialog;
    private CompositeDisposable mDisposable;

    private String mFileExt = "pdf";
    private String mCacheSavePath;
    private String mSavePath;
    private String mPassword;
    private SaveAsBean mSaveAsBean;
    private int mFormat;
    private long mOriginSize;
    private boolean isSaveDocInCurPath = false;
    private boolean mIsSavingDoc = false;
    private CircleProgressBar mProgressBar;
    private int mNightMode = -1;
    private boolean mhasFileWriter;

    private OnProgressCancelListenerImpl mOnCancelListener;

    public DocSaveAsModule(Context context, PDFViewCtrl pdfViewCtrl, UIExtensionsManager uiExtensionsManager) {
        this.mContext = context;
        mPdfViewCtrl = pdfViewCtrl;
        mUiExtensionsManager = uiExtensionsManager;
        mNightMode = mContext.getResources().getConfiguration().uiMode & Configuration.UI_MODE_NIGHT_MASK;
    }

    @Override
    public String getName() {
        return MODULE_NAME_SAVE_AS;
    }

    @Override
    public boolean loadModule() {
        if (mUiExtensionsManager != null) {
            ((UIExtensionsManager) mUiExtensionsManager).registerModule(this);
            ((UIExtensionsManager) mUiExtensionsManager).registerConfigurationChangedListener(mConfigurationChangedListener);
            ((UIExtensionsManager) mUiExtensionsManager).registerThemeEventListener(mThemeEventListener);
        }
        mPdfViewCtrl.registerDocEventListener(mDocEventListener);
        return true;
    }

    @Override
    public boolean unloadModule() {
        if (mDisposable != null)
            mDisposable.clear();

        if (mUiExtensionsManager != null) {
            ((UIExtensionsManager) mUiExtensionsManager).unregisterConfigurationChangedListener(mConfigurationChangedListener);
            ((UIExtensionsManager) mUiExtensionsManager).unregisterThemeEventListener(mThemeEventListener);
        }
        mPdfViewCtrl.unregisterDocEventListener(mDocEventListener);
        return true;
    }

    public void showSaveAsDialog() {
        initSaveAsDialog();
        int formats = UIDocSaveAsDialog.FORMAT_ORIGINAL;
        DocumentManager documentManager = ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager();
        if (documentManager.canAddAnnot()
                && documentManager.canModifyContents()
                && ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).isEnableModification()
                && documentManager.withAllPermission(null)
                && documentManager.withModifyPermission()
                && documentManager.withDeletePermission()) {
            formats |= UIDocSaveAsDialog.FORMAT_FLATTEN;
        }
        if (mPdfViewCtrl.getCropMode() != PDFViewCtrl.CROPMODE_NONE) {
            formats |= UIDocSaveAsDialog.FORMAT_CROP;
        }
        if (!documentManager.isXFA()
                && !documentManager.simpleCheckPDFA(mPdfViewCtrl.getDoc())
                && documentManager.canModifyContents()
                && documentManager.canAddAnnot()
                && ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).isEnableModification()
                && documentManager.canFillForm()) {
            formats |= UIDocSaveAsDialog.FORMAT_OPTIMIZE;
        }
        mDocSaveAsDialog.setFormatItems(formats);
        mDocSaveAsDialog.setFileName(AppFileUtil.getFileNameWithoutExt(mPdfViewCtrl.getFilePath()));
        mDocSaveAsDialog.showDialog();
    }

    private void initSaveAsDialog() {
        if (mDocSaveAsDialog != null && mDocSaveAsDialog.isShowing()) {
            mDocSaveAsDialog.setListener(null);
            mDocSaveAsDialog.dismiss();
        }
        mDocSaveAsDialog = new UIDocSaveAsDialog(((UIExtensionsManager) mUiExtensionsManager).getAttachedActivity());
        mDocSaveAsDialog.setListener(new MatchDialog.DialogListener() {

            @Override
            public void onResult(long btType) {

            }

            @Override
            public void onBackClick() {
            }

            @Override
            public void onTitleRightButtonClick() {
                mFormat = mDocSaveAsDialog.getFormat();
                mFileExt = mDocSaveAsDialog.getFileExt();
                mSaveAsBean = mDocSaveAsDialog.getSaveAsBean();
                showSaveDocumentDialog(mDocSaveAsDialog.getFileName(), mFileExt);
                mDocSaveAsDialog.dismiss();
            }
        });
    }

    private void showSaveDocumentDialog(final String fileName, String fileExt) {
        mSaveAsDialog = new UISaveAsDialog(((UIExtensionsManager) mUiExtensionsManager).getAttachedActivity(), fileName, fileExt,
                new UISaveAsDialog.ISaveAsOnOKClickCallBack() {
                    @Override
                    public void onOkClick(String newFilePath) {
                        if (AppStorageManager.getInstance(mContext).isEquivalentExternalFilePath(newFilePath, mPdfViewCtrl.getFilePath())) {
                            isSaveDocInCurPath = true;
                            mSavePath = mPdfViewCtrl.getFilePath();
                            mCacheSavePath = getCacheFile();
                            doSaveDoc(mCacheSavePath);
                        } else {
                            isSaveDocInCurPath = false;
                            mSavePath = newFilePath;
                            if (AppFileUtil.needScopedStorageAdaptation()) {
                                mCacheSavePath = getCacheFile();
                                newFilePath = mCacheSavePath;
                            }
                            doSaveDoc(newFilePath);
                        }
                    }

                    @Override
                    public void onCancelClick() {
                    }
                });
        mSaveAsDialog.showDialog(true, fileName);
    }


    private String getCacheFile() {
        File file = new File(mPdfViewCtrl.getFilePath());
        String dir = file.getParent() + "/";
        String path = null;
        while (file.exists()) {
            path = dir + AppDmUtil.randomUUID(null) + "." + mFileExt;
            file = new File(path);
        }
        return path;
    }

    private void doSaveDoc(final String path) {
        final Activity activity = ((UIExtensionsManager) mUiExtensionsManager).getAttachedActivity();
        if (mFormat == UIDocSaveAsDialog.FORMAT_ORIGINAL) {
            showProgressDialog(AppResource.getString(mContext, R.string.fx_string_saving));

            int saveFlag = PDFDoc.e_SaveFlagNormal;
            try {
                saveFlag = mPdfViewCtrl.getDoc().getSignatureCount() > 0 ? PDFDoc.e_SaveFlagIncremental : PDFDoc.e_SaveFlagNormal;
            } catch (PDFException e) {
                e.printStackTrace();
            }
            mIsSavingDoc = true;
            mPdfViewCtrl.saveDoc(path, saveFlag);
        } else if (mFormat == UIDocSaveAsDialog.FORMAT_FLATTEN) {
            final UITextEditDialog dialog = new UITextEditDialog(activity, UIDialog.NO_INPUT);
            dialog.setTitle(AppResource.getString(mContext, R.string.rv_saveas_flatten));
            dialog.getPromptTextView().setText(AppResource.getString(mContext, R.string.fx_flatten_doc_toast));
            dialog.getOKButton().setOnClickListener(new View.OnClickListener() {
                @Override
                public void onClick(View v) {
                    doFlatten(path);
                    dialog.dismiss();
                }
            });
            dialog.show();
        } else if (mFormat == UIDocSaveAsDialog.FORMAT_CROP) {
            mOnCancelListener = new OnProgressCancelListenerImpl(false);
            doCrop(path, new OnProgressListener() {
                @Override
                public void onProgress(int progress) {
                    if (mProgressBar != null)
                        mProgressBar.setProgress(progress);
                }
            }, mOnCancelListener);
        } else if (mFormat == UIDocSaveAsDialog.FORMAT_OPTIMIZE) {
            if (mSaveAsBean == null)
                return;
            doOptimize(path, false, null, null);
        }
    }

    private void doCrop(final String path, OnProgressListener progressListener, OnProgressCancelListener cancelListener) {
        showProgressDialog(AppResource.getString(mContext, R.string.fx_string_saving));
        mProgressBar.setMax(mPdfViewCtrl.getPageCount());
        mProgressBar.setProgress(0);

        mIsSavingDoc = true;
        if (mDisposable == null)
            mDisposable = new CompositeDisposable();
        mDisposable.add(CropPDF.doCropPDF(mPdfViewCtrl, progressListener, cancelListener)
                .subscribeOn(Schedulers.io())
                .observeOn(AndroidSchedulers.mainThread())
                .subscribe(new Consumer<Boolean>() {
                    @Override
                    public void accept(Boolean b) {
                        if (mOnCancelListener != null && mOnCancelListener.isCancel()) return;

                        if (b != null && b) {
                            mPdfViewCtrl.saveDoc(path, PDFDoc.e_SaveFlagNormal);
                        } else {
                            mIsSavingDoc = false;
                            dismissProgressDialog();
                            UIToast.getInstance(mContext).show(AppResource.getString(mContext, R.string.fx_save_file_failed));
                        }
                    }
                }, new Consumer<Throwable>() {
                    @Override
                    public void accept(Throwable throwable) {
                        mIsSavingDoc = false;
                        dismissProgressDialog();
                        UIToast.getInstance(mContext).show(AppResource.getString(mContext, R.string.fx_save_file_failed));
                    }
                }));
    }

    private void doOptimize(final String path,
                            final boolean isReduce,
                            OnProgressListener listener,
                            OptimizerPDF.PauseCallbackImpl pauseCallback) {
        mIsSavingDoc = true;
        mOriginSize = AppFileUtil.getFileSize(mPdfViewCtrl.getFilePath());

        showProgressDialog(AppResource.getString(mContext, R.string.rv_saveas_optimize_optimizing));
        mProgressBar.setMax(100);
        mProgressBar.setProgress(0);

        if (mDisposable == null)
            mDisposable = new CompositeDisposable();
        mDisposable.add(OptimizerPDF.doOptimizerPDF(mPdfViewCtrl.getDoc(), ImageSettings.e_ImageCompressQualityMedium, MonoImageSettings.e_ImageCompressQualityLossy, listener, pauseCallback)
                .subscribeOn(Schedulers.io())
                .observeOn(AndroidSchedulers.mainThread())
                .subscribe(new Consumer<Boolean>() {
                    @Override
                    public void accept(Boolean b) throws Exception {
                        if (mOptimizePauseCallback != null && mOptimizePauseCallback.isPause()) {
                            mOptimizePauseCallback = null;
                            mIsSavingDoc = false;
                            return;
                        }

                        if (b != null && b) {
                            final FileWriterCallback writerCallback = ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getSaveWriter();
                            if (writerCallback != null) {
                                mhasFileWriter = true;
                                mPdfViewCtrl.saveDoc(writerCallback, PDFDoc.e_SaveFlagRemoveRedundantObjects);
                            } else {
                                mPdfViewCtrl.saveDoc(path, PDFDoc.e_SaveFlagRemoveRedundantObjects);
                            }
                        } else {
                            mIsSavingDoc = false;
                            dismissProgressDialog();
                            showOptimizerToast(2, !isReduce ? AppResource.getString(mContext, R.string.fx_save_file_failed) : AppResource.getString(mContext, R.string.menu_more_failed_to_reduce));
                        }
                    }
                }, new Consumer<Throwable>() {
                    @Override
                    public void accept(Throwable throwable) throws Exception {
                        if (mOptimizePauseCallback != null && mOptimizePauseCallback.isPause()) {
                            mOptimizePauseCallback = null;
                            mIsSavingDoc = false;
                            return;
                        }

                        mIsSavingDoc = false;
                        dismissProgressDialog();
                        showOptimizerToast(2, !isReduce ? AppResource.getString(mContext, R.string.fx_save_file_failed) : AppResource.getString(mContext, R.string.menu_more_failed_to_reduce));
                    }
                }));
    }

    private void doFlatten(final String path) {
        mIsSavingDoc = true;
        showProgressDialog(AppResource.getString(mContext, R.string.fx_string_saving));
        if (mDisposable == null)
            mDisposable = new CompositeDisposable();

        if (mPdfViewCtrl.isDynamicXFA()) {
            mDisposable.add(FlattenPDF.doFlattenXFADoc(mPdfViewCtrl.getXFADoc(), path)
                    .subscribeOn(Schedulers.io())
                    .observeOn(AndroidSchedulers.mainThread())
                    .subscribe(new Consumer<Boolean>() {
                        @Override
                        public void accept(Boolean b) {
                            if (b != null && b) {
                                openNewFile();
                            } else {
                                mIsSavingDoc = false;
                                dismissProgressDialog();
                                UIToast.getInstance(mContext).show(AppResource.getString(mContext, R.string.fx_save_file_failed));
                            }
                        }
                    }, new Consumer<Throwable>() {
                        @Override
                        public void accept(Throwable throwable) {
                            mIsSavingDoc = false;
                            dismissProgressDialog();
                            UIToast.getInstance(mContext).show(AppResource.getString(mContext, R.string.fx_save_file_failed));
                        }
                    }));
        } else {
            mDisposable.add(FlattenPDF.doFlattenPDFDoc(mPdfViewCtrl.getDoc(), true, PDFPage.e_FlattenAll)
                    .subscribeOn(Schedulers.io())
                    .observeOn(AndroidSchedulers.mainThread())
                    .subscribe(new Consumer<Boolean>() {
                        @Override
                        public void accept(Boolean b) throws Exception {
                            if (b != null && b) {
                                mPdfViewCtrl.saveDoc(path, PDFDoc.e_SaveFlagNormal);
                            } else {
                                mIsSavingDoc = false;
                                dismissProgressDialog();
                                UIToast.getInstance(mContext).show(AppResource.getString(mContext, R.string.fx_save_file_failed));
                            }
                        }
                    }, new Consumer<Throwable>() {
                        @Override
                        public void accept(Throwable throwable) {
                            mIsSavingDoc = false;
                            dismissProgressDialog();
                            UIToast.getInstance(mContext).show(AppResource.getString(mContext, R.string.fx_save_file_failed));
                        }
                    }));
        }
    }

    View container;
    TextView mProgressTips;
//    View mProgresscloseView;

    private void showProgressDialog(String message) {
        if (mProgressDialog == null) {
            final Activity activity = ((UIExtensionsManager) mUiExtensionsManager).getAttachedActivity();
            mProgressDialog = new FxProgressDialog(activity, AppResource.getString(mContext, R.string.rv_panel_annot_loading_start));
            container = mProgressDialog.getContentView().findViewById(R.id.progress_bar_container);
            container.setBackgroundResource(R.drawable.dialog_radian);
            container.getLayoutParams().width = AppResource.getDimensionPixelSize(mContext, R.dimen.ux_margin_280dp);
            container.getLayoutParams().height = AppResource.getDimensionPixelSize(mContext, R.dimen.ux_margin_190dp);
            mProgressDialog.getContentView().findViewById(R.id.progress_bar).setVisibility(View.GONE);
            mProgressBar = mProgressDialog.getContentView().findViewById(R.id.circle_progress_bar);
            mProgressBar.setMax(100);
            mProgressBar.setProgress(0);
            mProgressBar.setVisibility(View.VISIBLE);
            mProgressBar.setTextColor(ThemeConfig.getInstance(mContext).getPrimaryColor());
            mProgressBar.setSecondaryColor(ThemeConfig.getInstance(mContext).getPrimaryColor());

            mProgressTips = mProgressDialog.getContentView().findViewById(R.id.progress_tip);

            mProgressTips.setTextColor(AppResource.getColor(mContext, R.color.t4));
//            mProgresscloseView = mProgressDialog.getContentView().findViewById(R.id.progress_close);
//            mProgresscloseView.setVisibility(View.VISIBLE);
//            mProgresscloseView.setOnClickListener(new View.OnClickListener() {
//                @Override
//                public void onClick(View v) {
//                    mProgressDialog.dismiss();
//                }
//            });
        }
        mProgressBar.setProgress(0);
        mProgressDialog.setTips(message);
        container.setBackground(mContext.getResources().getDrawable(R.drawable.dialog_radian));
        mProgressBar.setTextColor(ThemeConfig.getInstance(mContext).getPrimaryColor());
        mProgressBar.setSecondaryColor(ThemeConfig.getInstance(mContext).getPrimaryColor());
        mProgressBar.setPrimaryColor(R.color.t2);
        mProgressTips.setTextColor(mContext.getResources().getColor(R.color.t4));
//        ThemeUtil.setTintList(mProgresscloseView, ThemeUtil.getPrimaryIconColor(mContext));
        mProgressDialog.show();

        if (mFormat == UIDocSaveAsDialog.FORMAT_OPTIMIZE || mFormat == UIDocSaveAsDialog.FORMAT_CROP) {
            mProgressDialog.setOnCancelListener(new DialogInterface.OnCancelListener() {
                @Override
                public void onCancel(DialogInterface dialog) {
                    if (mOptimizePauseCallback != null)
                        mOptimizePauseCallback.setPause(true);

                    if (mOnCancelListener != null)
                        mOnCancelListener.setCancel(true);
                }
            });
        }
    }

    private void dismissProgressDialog() {
        if (mProgressDialog != null && mProgressDialog.isShowing())
            mProgressDialog.dismiss();
    }

    private void openNewFile() {
        if (mhasFileWriter) {
            mhasFileWriter = false;
            mSavePath = mPdfViewCtrl.getFilePath();
        } else {
            if (isSaveDocInCurPath) {
                if (mCacheSavePath == null) return;
                AppStorageManager storageManager = AppStorageManager.getInstance(mContext);
                String target = storageManager.toExternalPathFromScopedCache(mSavePath);
                if (target != null) {
                    storageManager.copyDocument(mCacheSavePath, target, false);
                }
                File file = new File(mCacheSavePath);
                File docFile = new File(mPdfViewCtrl.getFilePath());
                if (file.exists()) {
                    if (!docFile.delete() || !file.renameTo(docFile)) {
                        UIToast.getInstance(mContext).show(AppResource.getString(mContext, R.string.fx_save_file_failed));
                        return;
                    }
                } else {
                    UIToast.getInstance(mContext).show(AppResource.getString(mContext, R.string.fx_save_file_failed));
                    return;
                }
            } else {
                if (AppFileUtil.needScopedStorageAdaptation() && mCacheSavePath != null) {
                    AppStorageManager storageManager = AppStorageManager.getInstance(mContext);
                    File file = new File(mCacheSavePath);
                    storageManager.copyDocument(mCacheSavePath, mSavePath, false);
                    File docFile = new File(storageManager.getScopedCacheFilePath(mSavePath));
                    if (file.exists()) {
                        if (docFile.exists()) {
                            if (!docFile.delete() || !file.renameTo(docFile)) {
                                UIToast.getInstance(mContext.getApplicationContext()).show(mContext.getApplicationContext().getString(R.string.fx_save_file_failed));
                            }
                        } else {
                            if (!file.renameTo(docFile)) {
                                UIToast.getInstance(mContext.getApplicationContext()).show(mContext.getApplicationContext().getString(R.string.fx_save_file_failed));
                            }
                        }
                    } else {
                        UIToast.getInstance(mContext.getApplicationContext()).show(mContext.getApplicationContext().getString(R.string.fx_save_file_failed));
                    }
                    mSavePath = docFile.getAbsolutePath();
                }
            }
        }

        mPdfViewCtrl.cancelAllTask();
        UIExtensionsManager uiExtensionsManager = (UIExtensionsManager) mUiExtensionsManager;
        uiExtensionsManager.triggerDismissMenuEvent();
        uiExtensionsManager.setCurrentToolHandler(null);
        uiExtensionsManager.getDocumentManager().setCurrentAnnot(null);
        uiExtensionsManager.getDocumentManager().clearUndoRedo();
        uiExtensionsManager.getDocumentManager().setDocModified(false);
        updateThumbnail(mSavePath);

        LinkModule linkModule = (LinkModule) uiExtensionsManager.getModuleByName(Module.MODULE_NAME_LINK);
        if (linkModule != null)
            linkModule.clear();
        byte[] password = null;
        if (getPassword() != null)
            password = getPassword().getBytes();
        mPdfViewCtrl.setKeepZoomRatioWhenReopenDoc(true);
        mPdfViewCtrl.openDoc(mSavePath, password);
    }

    private void updateThumbnail(String path) {
        LocalModule module = (LocalModule) ((UIExtensionsManager) mUiExtensionsManager)
                .getModuleByName(Module.MODULE_NAME_LOCAL);
        if (module != null)
            module.updateThumbnail(path);
    }

    public String getPassword() {
        return mPassword;
    }

    public void setPassword(String password) {
        this.mPassword = password;
    }

    private final PDFViewCtrl.IDocEventListener mDocEventListener = new DocEventListener() {
        @Override
        public void onDocWillOpen() {
        }

        @Override
        public void onDocOpened(PDFDoc document, int errCode) {
            dismissProgressDialog();
            if (mIsSavingDoc) {
                mIsSavingDoc = false;
                if (mFormat == UIDocSaveAsDialog.FORMAT_OPTIMIZE) {
                    long newSize = AppFileUtil.getFileSize(mSavePath);
                    String strOrigSize = AppFileUtil.formatFileSize(mOriginSize);
                    String strNewSize = AppFileUtil.formatFileSize(newSize);
                    String strFormat = AppResource.getString(mContext, R.string.rv_saveas_optimize_toast);
                    String strToast = String.format(strFormat, strOrigSize, strNewSize);
                    int state = newSize <= mOriginSize ? 0 : 1;
                    showOptimizerToast(state, strToast);
                    mFormat = UIDocSaveAsDialog.FORMAT_UNKNOWN;
                }
            }
        }

        @Override
        public void onDocWillClose(PDFDoc document) {
        }

        @Override
        public void onDocClosed(PDFDoc document, int errCode) {
        }

        @Override
        public void onDocWillSave(PDFDoc document) {
        }

        @Override
        public void onDocSaved(PDFDoc document, int errCode) {
            if (mIsSavingDoc) {
                if (mFormat == UIDocSaveAsDialog.FORMAT_CROP) {
                    UIExtensionsManager uiExtensionsManager = (UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager();
                    CropModule cropModule = (CropModule) uiExtensionsManager.getModuleByName(Module.MODULE_NAME_CROP);
                    if (cropModule != null) {
                        cropModule.exitCrop();
                    }
                }

                if (errCode == Constants.e_ErrSuccess) {
                    if ((mPdfViewCtrl.getViewRotation() % 2) != 0)
                        mPdfViewCtrl.rotateView(Constants.e_Rotation0);
                    mPdfViewCtrl.updatePagesLayout();
                    openNewFile();
                } else {
                    dismissProgressDialog();
                    UIToast.getInstance(mContext).show(AppResource.getString(mContext, R.string.fx_save_file_failed));
                }
                if (mCacheSavePath != null) {
                    mCacheSavePath = null;
                }
            }
        }
    };
    UIDialog dlg;
    View mDlgContentView;
    UIBtnImageView mDlgIvStart;
    UIBtnImageView mDlgIvCenter;
    UIBtnImageView mDlgIvEnd;
    TextView mDlgTvToast;
    ImageView mDlgClose;

    private void showOptimizerToast(int stateCode, String msg) {
        dlg = new UIDialog(((UIExtensionsManager) mUiExtensionsManager).getAttachedActivity(),
                R.layout.fx_saveas_optimize_toast, AppTheme.getDialogTheme());
        mDlgContentView = dlg.getContentView();
        mDlgContentView.getLayoutParams().width = AppResource.getDimensionPixelSize(mContext, R.dimen.ux_margin_280dp);
//        mDlgContentView.getLayoutParams().height = AppResource.getDimensionPixelSize(mContext, R.dimen.ux_margin_190dp);
        mDlgClose = mDlgContentView.findViewById(R.id.save_as_optimize_close);
        ThemeUtil.setTintList(mDlgClose, ThemeUtil.getPrimaryIconColor(mContext));
        mDlgClose.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                dlg.dismiss();
            }
        });
        mDlgIvStart = mDlgContentView.findViewById(R.id.optimize_icon_start);
        mDlgIvCenter = mDlgContentView.findViewById(R.id.optimize_icon_center);
        mDlgIvEnd = mDlgContentView.findViewById(R.id.optimize_icon_end);
        mDlgIvCenter.setColorStateList(ThemeUtil.getItemIconColor(mContext));
//        mDlgIvEnd.setColorStateList(ThemeUtil.getItemIconColor(mContext));
        if (stateCode == 0) {
            mDlgIvStart.setImageResource(R.drawable.ic_reduce_blank_big);
            mDlgIvCenter.setImageResource(R.drawable.ic_reduce_arrow);
//            mDlgIvEnd.setImageResource(R.drawable.ic_reduce_zip_small);
            if (AppUtil.isDarkMode(mContext)) {
                mDlgIvEnd.setImageResource(R.drawable.ic_reduce_zip_small_dark);
            }else{
                mDlgIvEnd.setImageResource(R.drawable.ic_reduce_zip_small);
            }
        } else if (stateCode == 1) {
            mDlgIvStart.setImageResource(R.drawable.ic_reduce_blank_small);
            mDlgIvCenter.setImageResource(R.drawable.ic_reduce_arrow);
//            mDlgIvEnd.setImageResource(R.drawable.ic_reduce_zip_big);
            if (AppUtil.isDarkMode(mContext)) {
                mDlgIvEnd.setImageResource(R.drawable.ic_reduce_zip_big_dark);
            }else{
                mDlgIvEnd.setImageResource(R.drawable.ic_reduce_zip_big);
            }
        } else {
            mDlgIvCenter.setImageResource(R.drawable.ic_failed_to_reduce);
        }

        mDlgTvToast = mDlgContentView.findViewById(R.id.saveas_optimize_toast);
        mDlgTvToast.setText(msg);
        mDlgTvToast.setTextColor(mContext.getResources().getColor(R.color.t4));
        if (dlg.getDialog().getWindow() != null) {
            dlg.getDialog().getWindow().setBackgroundDrawableResource(android.R.color.transparent);
        }
        dlg.show();
    }

//    private void setOptimizerToastDarkMode() {
//        mDlgContentView.setBackground(mContext.getResources().getDrawable(R.drawable.dialog_radian));
//        ThemeUtil.setTintList(mDlgIvStart, ThemeUtil.getPrimaryIconColor(mContext));
//        mDlgIvCenter.setColorStateList(ThemeUtil.getItemIconColor(mContext));
//        mDlgIvEnd.setColorStateList(ThemeUtil.getItemIconColor(mContext));
//        mDlgTvToast.setTextColor(mContext.getResources().getColor(R.color.t4));
//        ThemeUtil.setTintList(mDlgClose, ThemeUtil.getPrimaryIconColor(mContext));
//    }

    private IThemeEventListener mThemeEventListener = new IThemeEventListener() {
        @Override
        public void onThemeColorChanged(String type, int color) {
            if (dlg != null) {
                dlg.dismiss();
                dlg = null;
            }
            if (mDocSaveAsDialog != null) {
                mDocSaveAsDialog.dismiss();
                mDocSaveAsDialog = null;

            }
            if (mSaveAsDialog != null) {
                mSaveAsDialog.dismiss();
                mSaveAsDialog = null;
            }

            if (mFlattenDialog != null) {
                mFlattenDialog.dismiss();
                mFlattenDialog = null;
            }
        }
    };

    private final UIExtensionsManager.ConfigurationChangedListener mConfigurationChangedListener = new UIExtensionsManager.ConfigurationChangedListener() {
        @Override
        public void onConfigurationChanged(Configuration newConfig) {
            if (mNightMode != (newConfig.uiMode & Configuration.UI_MODE_NIGHT_MASK)) {
                mNightMode = newConfig.uiMode & Configuration.UI_MODE_NIGHT_MASK;
                return;
            }

            if (mSaveAsDialog != null && mSaveAsDialog.isShowing()) {
                mSaveAsDialog.resetWH();
                mSaveAsDialog.showDialog();
            }

            if (mDocSaveAsDialog != null && mDocSaveAsDialog.isShowing()) {
                mDocSaveAsDialog.resetWH();
                mDocSaveAsDialog.showDialog();
                AppThreadManager.getInstance().getMainThreadHandler().postDelayed(new Runnable() {
                    @Override
                    public void run() {
                        mDocSaveAsDialog.setFileName(mDocSaveAsDialog.getFileName());
                    }
                }, 200);
            }
        }
    };

    public void doFlattenCurrentFileWithoutSave() {
        mFlattenDialog = new UITextEditDialog(((UIExtensionsManager) mUiExtensionsManager).getAttachedActivity(), UIDialog.NO_INPUT);
        mFlattenDialog.setTitle(AppResource.getString(mContext, R.string.fx_string_flatten));
        mFlattenDialog.getPromptTextView().setText(AppResource.getString(mContext, R.string.fx_flatten_toast));
        mFlattenDialog.getCancelButton().setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                mFlattenDialog.dismiss();
            }
        });
        mFlattenDialog.getOKButton().setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                mFlattenDialog.dismiss();
                String path = mPdfViewCtrl.getFilePath();
                if (mDisposable == null)
                    mDisposable = new CompositeDisposable();
                if (mPdfViewCtrl.isDynamicXFA()) {
                    mDisposable.add(FlattenPDF.doFlattenXFADoc(mPdfViewCtrl.getXFADoc(), path)
                            .subscribeOn(Schedulers.io())
                            .observeOn(AndroidSchedulers.mainThread())
                            .subscribe(new Consumer<Boolean>() {
                                @Override
                                public void accept(Boolean b) {
                                    if (b) {
                                        UIToast.getInstance(mContext).show(AppResource.getString(mContext, R.string.menu_more_flattening_is_successful));
                                        DocumentManager documentManager = ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager();
                                        documentManager.onAnnotsFlattened(mPdfViewCtrl.getDoc());
                                        documentManager.setDocModified(true);
                                        documentManager.clearUndoRedo();

                                        for (int pageIndex : mPdfViewCtrl.getVisiblePages()) {
                                            int width = mPdfViewCtrl.getPageViewWidth(pageIndex);
                                            int height = mPdfViewCtrl.getPageViewHeight(pageIndex);
                                            mPdfViewCtrl.refresh(pageIndex, new Rect(0, 0, width, height));
                                        }
                                    } else {
                                        dismissProgressDialog();
                                    }

                                }
                            }, new Consumer<Throwable>() {
                                @Override
                                public void accept(Throwable throwable) {
                                    dismissProgressDialog();
                                }
                            }));
                } else {
                    mDisposable.add(FlattenPDF.doFlattenPDFDoc(mPdfViewCtrl.getDoc(), true, PDFPage.e_FlattenAll)
                            .subscribeOn(Schedulers.io())
                            .observeOn(AndroidSchedulers.mainThread())
                            .subscribe(new Consumer<Boolean>() {
                                @Override
                                public void accept(Boolean b) throws Exception {
                                    if (b) {
                                        UIToast.getInstance(mContext).show(AppResource.getString(mContext, R.string.menu_more_flattening_is_successful));
                                        DocumentManager documentManager = ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager();
                                        documentManager.onAnnotsFlattened(mPdfViewCtrl.getDoc());
                                        documentManager.setDocModified(true);
                                        documentManager.clearUndoRedo();

                                        for (int pageIndex : mPdfViewCtrl.getVisiblePages()) {
                                            int width = mPdfViewCtrl.getPageViewWidth(pageIndex);
                                            int height = mPdfViewCtrl.getPageViewHeight(pageIndex);
                                            mPdfViewCtrl.refresh(pageIndex, new Rect(0, 0, width, height));
                                        }
                                    } else {
                                        dismissProgressDialog();
                                    }
                                }
                            }, new Consumer<Throwable>() {
                                @Override
                                public void accept(Throwable throwable) {
                                    dismissProgressDialog();
                                }
                            }));
                }
            }
        });
        mFlattenDialog.show();
    }

    private OptimizerPDF.PauseCallbackImpl mOptimizePauseCallback;

    public void doOptimizeCurrentFile() {
        isSaveDocInCurPath = true;
        mFormat = UIDocSaveAsDialog.FORMAT_OPTIMIZE;
        mSavePath = mPdfViewCtrl.getFilePath();
        mCacheSavePath = getCacheFile();

        mOptimizePauseCallback = new OptimizerPDF.PauseCallbackImpl(false);
        doOptimize(mCacheSavePath, true, new OnProgressListener() {
            @Override
            public void onProgress(int progress) {
                mProgressBar.setProgress(progress);
            }
        }, mOptimizePauseCallback);
    }

    private static class OnProgressCancelListenerImpl implements OnProgressCancelListener {

        private boolean mIsCancel;

        public OnProgressCancelListenerImpl(boolean cancel) {
            mIsCancel = cancel;
        }

        public void setCancel(boolean cacel) {
            mIsCancel = cacel;
        }

        public boolean isCancel() {
            return mIsCancel;
        }

        @Override
        public boolean onCancel() {
            return mIsCancel;
        }
    }

}
