/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.modules.dynamicxfa;

import android.app.Activity;
import android.content.Context;
import android.content.DialogInterface;
import android.content.Intent;
import android.content.res.Configuration;
import android.graphics.Canvas;
import android.graphics.DashPathEffect;
import android.graphics.Matrix;
import android.graphics.Paint;
import android.graphics.PaintFlagsDrawFilter;
import android.graphics.PathEffect;
import android.graphics.PointF;
import android.graphics.Rect;
import android.graphics.RectF;
import android.os.Handler;
import android.text.Editable;
import android.text.TextUtils;
import android.text.TextWatcher;
import android.view.KeyEvent;
import android.view.MotionEvent;
import android.view.View;
import android.view.ViewGroup;
import android.view.inputmethod.EditorInfo;
import android.widget.EditText;

import androidx.annotation.NonNull;

import com.foxit.sdk.PDFException;
import com.foxit.sdk.PDFViewCtrl;
import com.foxit.sdk.Task;
import com.foxit.sdk.addon.xfa.XFAPage;
import com.foxit.sdk.addon.xfa.XFAWidget;
import com.foxit.sdk.common.Image;
import com.foxit.sdk.common.fxcrt.Matrix2D;
import com.foxit.sdk.pdf.PDFDoc;
import com.foxit.sdk.pdf.Signature;
import com.foxit.uiextensions.Module;
import com.foxit.uiextensions.R;
import com.foxit.uiextensions.UIExtensionsManager;
import com.foxit.uiextensions.annots.form.FormFillerUtil;
import com.foxit.uiextensions.annots.form.FormNavigationModule;
import com.foxit.uiextensions.controls.dialog.MatchDialog;
import com.foxit.uiextensions.controls.dialog.UIDialog;
import com.foxit.uiextensions.controls.dialog.UITextEditDialog;
import com.foxit.uiextensions.controls.dialog.fileselect.UIFileSelectDialog;
import com.foxit.uiextensions.controls.dialog.fileselect.UISaveAsDialog;
import com.foxit.uiextensions.controls.propertybar.AnnotMenu;
import com.foxit.uiextensions.controls.propertybar.imp.AnnotMenuImpl;
import com.foxit.uiextensions.data.ISuggestionsType;
import com.foxit.uiextensions.data.SuggestionsHelper;
import com.foxit.uiextensions.event.IUISaveasEventListener;
import com.foxit.uiextensions.home.local.LocalModule;
import com.foxit.uiextensions.pdfreader.config.ActRequestCode;
import com.foxit.uiextensions.security.certificate.CertificateFileInfo;
import com.foxit.uiextensions.security.digitalsignature.DigitalSignatureModule;
import com.foxit.uiextensions.security.digitalsignature.DigitalSignatureSecurityHandler;
import com.foxit.uiextensions.security.digitalsignature.DigitalSignatureUtil;
import com.foxit.uiextensions.security.digitalsignature.IDigitalSignatureCallBack;
import com.foxit.uiextensions.security.digitalsignature.IDigitalSignatureCreateCallBack;
import com.foxit.uiextensions.utils.AppDevice;
import com.foxit.uiextensions.utils.AppDisplay;
import com.foxit.uiextensions.utils.AppDmUtil;
import com.foxit.uiextensions.utils.AppFileUtil;
import com.foxit.uiextensions.utils.AppIntentUtil;
import com.foxit.uiextensions.utils.AppKeyboardUtil;
import com.foxit.uiextensions.utils.AppResource;
import com.foxit.uiextensions.utils.AppSQLite;
import com.foxit.uiextensions.utils.AppUtil;
import com.foxit.uiextensions.utils.SystemUiHelper;
import com.foxit.uiextensions.utils.thread.AppThreadManager;

import java.io.File;
import java.io.FileFilter;
import java.util.ArrayList;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;


public class DynamicXFAWidgetHandler implements IXFAWidgetHandler {
    private static final int UN_SINGED_WIDGET = 0;
    private static final int SIGNED_WIDGET = 1;
    private static final int IMAGE_WIDGET = 2;
    private static final Lock lock = new ReentrantLock();

    private final Context mContext;
    private PDFViewCtrl mPdfViewCtrl;
    private ViewGroup mParent;

    private XFAWidget mCurrentXFAWidget = null;
    private RectF mCurrentWidgetRect = new RectF();
    private int mCurrentWidgetType;

    private FormNavigationModule mFNModule = null;
    private int mPageOffset;
    private int mOffset;
    private EditText mEditView = null;
    private PointF mLastTouchPoint = new PointF(0, 0);
    //    private boolean mIsBackBtnPush = false; //for some input method, double backspace click
    private boolean mAdjustPosition = false;
    private boolean mIsShowEditText = false;
    private String mBeforeText = null;
    private Paint mPathPaint;
    private int mKeyBoardHeight;
    private int mAddSignPageIndex = -1;

    private boolean mShouldAddSuggestions = false;
    private boolean mShouldInterceptTextChanged = false;

    private final String mCaptureImgPath;
    private UIFileSelectDialog mFileSelectDialog = null;

    public DynamicXFAWidgetHandler(Context context, ViewGroup parent, PDFViewCtrl pdfViewCtrl) {
        mContext = context;
        mParent = parent;
        mPdfViewCtrl = pdfViewCtrl;

        mCaptureImgPath = AppFileUtil.getDiskCachePath(context) + "/camera_photos/xfa_capture_img";
    }

    public void initialize() {
        mPathPaint = new Paint(Paint.ANTI_ALIAS_FLAG);
        mPathPaint.setStyle(Paint.Style.STROKE);
        mPathPaint.setAntiAlias(true);
        mPathPaint.setDither(true);
        PathEffect effects = new DashPathEffect(new float[]{1, 2, 4, 8}, 1);
        mPathPaint.setPathEffect(effects);

        initFormNavigation();
    }

    private void initFormNavigation() {
        mFNModule = (FormNavigationModule) ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getModuleByName(Module.MODULE_NAME_FORM_NAVIGATION);
        if (mFNModule != null) {
            mFNModule.initView();
            mFNModule.hide();
            mFNModule.getPreView().setOnClickListener(new View.OnClickListener() {
                @Override
                public void onClick(View v) {
                    AppThreadManager.getInstance().startThread(preNavigation);
                }
            });

            mFNModule.getNextView().setOnClickListener(new View.OnClickListener() {
                @Override
                public void onClick(View v) {
                    AppThreadManager.getInstance().startThread(nextNavigation);
                }
            });

            mFNModule.getClearView().setOnClickListener(new View.OnClickListener() {
                @Override
                public void onClick(View v) {
                    if (mCurrentXFAWidget != null && !mCurrentXFAWidget.isEmpty()) {
                        resetXFAWidget();
                    }
                }
            });

            mFNModule.getFinishView().setOnClickListener(new View.OnClickListener() {
                @Override
                public void onClick(View v) {
                    if (mCurrentXFAWidget != null) {
                        if (shouldShowInputSoft(mCurrentXFAWidget)) {
                            AppUtil.dismissInputSoft(mEditView);
                            mParent.removeView(mEditView);
                        }
                        setCurrentXFAWidget(null);
                    }
                    mFNModule.hide();
                    resetDocViewerOffset();
                }
            });

            mFNModule.setClearEnable(false);
        }
    }

    private void resetXFAWidget() {
        try {
            PDFViewCtrl.lock();
            mPdfViewCtrl.getXFADoc().killFocus();
            //killFoucs may cause the xfawidget to be deleted
            if (mCurrentXFAWidget != null)
                mCurrentXFAWidget.resetData();
            if (mFNModule != null)
                mFNModule.hideSuggestionsbar();
            //resetData may cause the xfawidget to be deleted
            if (mCurrentXFAWidget != null) {
                mShouldAddSuggestions = AppUtil.isEmpty(mCurrentXFAWidget.getValue());

                mPdfViewCtrl.getXFADoc().setFocus(mCurrentXFAWidget);
                refresh(mCurrentXFAWidget);
            }
        } catch (PDFException e) {
            e.printStackTrace();
        } finally {
            PDFViewCtrl.unlock();
        }
    }

    private boolean isFind = false;
    private boolean isDocFinish = false;
    private int prePageIdx;
    private int preWidgetIdx;
    private int nextPageIdx;
    private int nextWidgetIdx;
    private CountDownLatch mCountDownLatch;
    private Runnable preNavigation = new Runnable() {

        @Override
        public void run() {
            try {
                if (mCurrentXFAWidget != null && !mCurrentXFAWidget.isEmpty()) {
//                    refresh(mCurrentXFAWidget);
                    XFAPage curPage = mCurrentXFAWidget.getXFAPage();
                    final int curPageIdx = curPage.getIndex();
                    prePageIdx = curPageIdx;
                    final int curWidgetIdx = mCurrentXFAWidget.getIndex();
                    preWidgetIdx = curWidgetIdx;
                    isFind = false;
                    isDocFinish = false;
                    while (prePageIdx >= 0) {
                        mCountDownLatch = new CountDownLatch(1);
                        curPage = mPdfViewCtrl.getXFADoc().getPage(prePageIdx);
                        if (prePageIdx == curPageIdx && !isDocFinish) {
                            preWidgetIdx = curWidgetIdx - 1;
                        } else {
                            preWidgetIdx = curPage.getWidgetCount() - 1;
                        }

                        while (curPage != null && preWidgetIdx >= 0) {
                            final XFAWidget preWidget = curPage.getWidget(preWidgetIdx);
                            final int preWidgetType = preWidget.getType();
                            if (preWidget != null
                                    && !preWidget.isEmpty()
                                    && preWidget.getPresence() == XFAWidget.e_PresenceVisible
                                    && isXfaWidgetSupportJump(preWidgetType)) {
                                isFind = true;
                                AppThreadManager.getInstance().getMainThreadHandler().post(new Runnable() {

                                    @Override
                                    public void run() {
                                        try {
                                            if (preWidgetType == XFAWidget.e_WidgetTypeChoiceList) {
                                                RectF rect = AppUtil.toRectF(preWidget.getRect());
                                                rect.left += 5;
                                                rect.top -= 5;
                                                mLastTouchPoint.set(rect.left, rect.top);
                                            }
                                            setCurrentXFAWidget(null);
                                            if (preWidget != null && !preWidget.isEmpty()) {
                                                RectF bbox = AppUtil.toRectF(preWidget.getRect());
                                                RectF rect = new RectF(bbox);

                                                if (mPdfViewCtrl.convertPdfRectToPageViewRect(rect, rect, prePageIdx)) {
                                                    float devX = rect.left - (mPdfViewCtrl.getWidth() - rect.width()) / 2;
                                                    float devY = rect.top - (mPdfViewCtrl.getHeight() - rect.height()) / 2;
                                                    mPdfViewCtrl.gotoPage(prePageIdx, devX, devY);
                                                } else {
                                                    mPdfViewCtrl.gotoPage(prePageIdx, new PointF(bbox.left, bbox.top));
                                                }
                                                setCurrentXFAWidget(preWidget);
                                                mPdfViewCtrl.getXFADoc().setFocus(preWidget);
                                                if (mFNModule != null) {
                                                    mFNModule.hideSuggestionsbar();
                                                }
                                            }
                                        } catch (PDFException e) {
                                            e.printStackTrace();
                                        }
                                    }
                                });

                                break;
                            } else {
                                preWidgetIdx--;
                            }
                        }
                        mCountDownLatch.countDown();

                        try {
                            if (mCountDownLatch.getCount() > 0)
                                mCountDownLatch.await();
                        } catch (InterruptedException e) {
                            e.printStackTrace();
                        }

                        if (isFind) break;
                        prePageIdx--;
                        if (prePageIdx < 0) {
                            prePageIdx = mPdfViewCtrl.getXFADoc().getPageCount() - 1;
                            isDocFinish = true;
                        }
                    }
                }
            } catch (PDFException e) {
                e.printStackTrace();
            }
        }
    };

    private Runnable nextNavigation = new Runnable() {

        @Override
        public void run() {
            try {
                if (mCurrentXFAWidget != null && !mCurrentXFAWidget.isEmpty()) {
                    XFAPage curPage = mCurrentXFAWidget.getXFAPage();

                    final int curPageIdx = curPage.getIndex();
                    nextPageIdx = curPageIdx;
                    final int curWidgetIdx = mCurrentXFAWidget.getIndex();
                    nextWidgetIdx = curWidgetIdx;
                    isFind = false;
                    isDocFinish = false;

                    while (nextPageIdx < mPdfViewCtrl.getXFADoc().getPageCount()) {

                        mCountDownLatch = new CountDownLatch(1);
                        curPage = mPdfViewCtrl.getXFADoc().getPage(nextPageIdx);
                        if (nextPageIdx == curPageIdx && !isDocFinish) {
                            nextWidgetIdx = curWidgetIdx + 1;
                        } else {
                            nextWidgetIdx = 0;
                        }

                        while (curPage != null && nextWidgetIdx < curPage.getWidgetCount()) {
                            final XFAWidget nextWidget = curPage.getWidget(nextWidgetIdx);
                            final int nextWidgetType = nextWidget.getType();
                            if (nextWidget != null
                                    && !nextWidget.isEmpty()
                                    && nextWidget.getPresence() == XFAWidget.e_PresenceVisible
                                    && isXfaWidgetSupportJump(nextWidgetType)) {
                                isFind = true;
                                AppThreadManager.getInstance().getMainThreadHandler().post(new Runnable() {

                                    @Override
                                    public void run() {
                                        try {
                                            if (nextWidgetType == XFAWidget.e_WidgetTypeChoiceList) {
                                                RectF rect = AppUtil.toRectF(nextWidget.getRect());
                                                rect.left += 5;
                                                rect.top -= 5;
                                                mLastTouchPoint.set(rect.left, rect.top);
                                            }
                                            setCurrentXFAWidget(null);
                                            if (nextWidget != null && !nextWidget.isEmpty()) {
                                                RectF bbox = AppUtil.toRectF(nextWidget.getRect());
                                                RectF rect = new RectF(bbox);

                                                if (mPdfViewCtrl.convertPdfRectToPageViewRect(rect, rect, nextPageIdx)) {
                                                    float devX = rect.left - (mPdfViewCtrl.getWidth() - rect.width()) / 2;
                                                    float devY = rect.top - (mPdfViewCtrl.getHeight() - rect.height()) / 2;
                                                    mPdfViewCtrl.gotoPage(nextPageIdx, devX, devY);
                                                } else {
                                                    mPdfViewCtrl.gotoPage(nextPageIdx, new PointF(bbox.left, bbox.top));
                                                }

                                                setCurrentXFAWidget(nextWidget);
                                                mPdfViewCtrl.getXFADoc().setFocus(nextWidget);
                                                if (mFNModule != null) {
                                                    mFNModule.hideSuggestionsbar();
                                                }
                                            }
                                        } catch (PDFException e) {
                                            e.printStackTrace();
                                        }
                                    }

                                });

                                break;
                            } else {
                                nextWidgetIdx++;
                            }
                        }
                        mCountDownLatch.countDown();

                        try {
                            if (mCountDownLatch.getCount() > 0)
                                mCountDownLatch.await();
                        } catch (InterruptedException e) {
                            e.printStackTrace();
                        }

                        if (isFind) break;
                        nextPageIdx++;
                        if (nextPageIdx >= mPdfViewCtrl.getXFADoc().getPageCount()) {
                            nextPageIdx = 0;
                            isDocFinish = true;
                        }
                    }
                }
            } catch (PDFException e) {
                e.printStackTrace();
            }
        }
    };

    private boolean isXfaWidgetSupportJump(int widgetType) {
        return widgetType != XFAWidget.e_WidgetTypeArc
                && widgetType != XFAWidget.e_WidgetTypeRectangle
                && widgetType != XFAWidget.e_WidgetTypeLine
                && widgetType != XFAWidget.e_WidgetTypePushButton
                && widgetType != XFAWidget.e_WidgetTypeSignature
                && widgetType != XFAWidget.e_WidgetTypeUnknown;
    }

    protected void showSoftInput() {
        AppUtil.showSoftInput(mEditView);
    }

    private void postDismissNavigation() {
        DismissNavigation dn = new DismissNavigation();
        dn.postDelayed(dn, 500);
    }

    private class DismissNavigation extends Handler implements Runnable {

        @Override
        public void run() {
            if (mPdfViewCtrl == null || mPdfViewCtrl.getDoc() == null || mPdfViewCtrl.getXFADoc() == null)
                return;
            if (mCurrentXFAWidget == null || mCurrentXFAWidget.isEmpty()) {
                if (mFNModule != null)
                    mFNModule.getLayout().setVisibility(View.INVISIBLE);
                AppUtil.dismissInputSoft(mEditView);
                resetDocViewerOffset();
            }
        }
    }

    private boolean shouldShowNavigation(XFAWidget xfaWidget) {
        if (xfaWidget == null || xfaWidget.isEmpty()) return false;
        try {
            int type = xfaWidget.getType();
            if (type == XFAWidget.e_WidgetTypeArc || type == XFAWidget.e_WidgetTypeRectangle
                    || type == XFAWidget.e_WidgetTypeLine || type == XFAWidget.e_WidgetTypePushButton)
                return false;
            return true;
        } catch (PDFException e) {
            e.printStackTrace();
        }

        return false;
    }

    public void navigationDismiss() {
        if (mFNModule != null) {
            mFNModule.hideFormNavigationBar();
        }

        if (mEditView != null) {
            mParent.removeView(mEditView);
        }
        resetDocViewerOffset();
        AppUtil.dismissInputSoft(mEditView);
    }


    private boolean isHitXFAWidget(XFAWidget xfaWidget, PointF pointF, int pageIndex) {
        try {
            if (xfaWidget.getXFAPage().getIndex() != pageIndex) return false;
            if (xfaWidget.getPresence() != XFAWidget.e_PresenceVisible) return false;
            int hitArea = xfaWidget.onHitTest(AppUtil.toFxPointF(pointF));
            if (hitArea != XFAWidget.e_HitTestAreaUnknown) return true;
        } catch (PDFException e) {
            e.printStackTrace();
        }
        return false;
    }

    private boolean isSameXFAWidget(XFAWidget xfaWidget, XFAWidget other) {
        if (xfaWidget == null || xfaWidget.isEmpty()) return false;
        if (other == null || other.isEmpty()) return false;
        return xfaWidget.equal(other);
    }

    protected void setCurrentXFAWidget(XFAWidget xfaWidget) {
        lock.lock();
        if (mCurrentXFAWidget == null && xfaWidget == null) return;
        if (isSameXFAWidget(mCurrentXFAWidget, xfaWidget)) return;
        if (xfaWidget == null) {
            ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).startHideToolbarsTimer();
        } else {
            ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).stopHideToolbarsTimer();
        }
        XFAWidget lastWidget = mCurrentXFAWidget;
        if (lastWidget != null && !lastWidget.isEmpty()) {
            onXFAWidgetDeselected(lastWidget, true);
        }

        mCurrentXFAWidget = xfaWidget;
        if (xfaWidget != null && !xfaWidget.isEmpty()) {
            onXFAWidgetSelected(xfaWidget, true);
        }
        lock.unlock();
    }

    protected void update(XFAWidget xfaWidget) {
        if (mCurrentXFAWidget == null && xfaWidget == null) return;
        try {
            if (xfaWidget.getIndex() != mCurrentXFAWidget.getIndex()) {
               return;
            }

            if (mCurrentXFAWidget != null) {
                setCurrentXFAWidget(null);
            } else {
                hideAnnotMenu();
                postDismissNavigation();
                if (mIsShowEditText) {
                    AppUtil.dismissInputSoft(mEditView);
                    mParent.removeView(mEditView);
                    mIsShowEditText = false;
                }
                mPdfViewCtrl.getXFADoc().killFocus();
//            refresh(xfaWidget);
            }
        } catch (PDFException e) {
            e.printStackTrace();
        }
    }

    protected boolean shouldShowInputSoft(XFAWidget xfaWidget) {
        if (xfaWidget == null || xfaWidget.isEmpty()) return false;
        try {
            int type = xfaWidget.getType();
            return shouldShowInputSoft(type);
        } catch (PDFException e) {
            e.printStackTrace();
        }

        return false;
    }

    protected boolean shouldShowInputSoft(int type) {
        return type == XFAWidget.e_WidgetTypeBarcode
                || type == XFAWidget.e_WidgetTypeDateTimeEdit
                || type == XFAWidget.e_WidgetTypeNumericEdit
                || type == XFAWidget.e_WidgetTypePasswordEdit
                || type == XFAWidget.e_WidgetTypeTextEdit
                || type == XFAWidget.e_WidgetTypeChoiceList;
    }

    public void resetDocViewerOffset() {
        if (mPageOffset != 0) {
            mPageOffset = 0;
            setBottomOffset(0);
        }
    }

    private void setBottomOffset(int offset) {
        if (mOffset == -offset)
            return;
        mOffset = -offset;
        mPdfViewCtrl.layout(0, 0 + mOffset, mPdfViewCtrl.getWidth(), mPdfViewCtrl.getHeight() + mOffset);
    }

    protected boolean onKeyBack() {
        if (mCurrentXFAWidget == null || mCurrentXFAWidget.isEmpty()) return false;
        try {
            int type = mCurrentXFAWidget.getType();
            if (type != XFAWidget.e_WidgetTypeSignature && type != XFAWidget.e_WidgetTypeUnknown) {
                setCurrentXFAWidget(null);
                navigationDismiss();
                return true;
            }
        } catch (PDFException e) {
            e.printStackTrace();
        }
        return false;
    }

    private static PointF getPageViewOrigin(PDFViewCtrl pdfViewCtrl, int pageIndex, float x, float y) {
        PointF pagePt = new PointF(x, y);
        pdfViewCtrl.convertPageViewPtToDisplayViewPt(pagePt, pagePt, pageIndex);
        RectF rect = new RectF(0, 0, pagePt.x, pagePt.y);
        pdfViewCtrl.convertDisplayViewRectToPageViewRect(rect, rect, pageIndex);
        PointF originPt = new PointF(x - rect.width(), y - rect.height());
        return originPt;
    }

    private double getDistanceOfPoints(PointF p1, PointF p2) {
        return Math.sqrt(Math.abs((p1.x - p2.x)
                * (p1.x - p2.x) + (p1.y - p2.y)
                * (p1.y - p2.y)));
    }

    private void refresh(XFAWidget xfaWidget) throws PDFException {
        if (xfaWidget == null || xfaWidget.isEmpty()) return;
        RectF rectF = AppUtil.toRectF(xfaWidget.getRect());
        int pageIndex = xfaWidget.getXFAPage().getIndex();
        mPdfViewCtrl.convertPdfRectToPageViewRect(rectF, rectF, pageIndex);
        mPdfViewCtrl.refresh(pageIndex, AppDmUtil.rectFToRect(rectF));
    }

    private void invalidate(XFAWidget xfaWidget) throws PDFException {
        if (xfaWidget == null || xfaWidget.isEmpty()) return;
        RectF rectF = AppUtil.toRectF(xfaWidget.getRect());
        int pageIndex = xfaWidget.getXFAPage().getIndex();
        mPdfViewCtrl.convertPdfRectToPageViewRect(rectF, rectF, pageIndex);
        mPdfViewCtrl.convertPageViewRectToDisplayViewRect(rectF, rectF, pageIndex);
        mPdfViewCtrl.invalidate(AppDmUtil.rectFToRect(rectF));
    }

    @Override
    public void onXFAWidgetSelected(final XFAWidget widget, boolean reRender) {
        final UIExtensionsManager uiExtensionsManager = (UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager();
        if (isXFAWidgetSignature(widget) && uiExtensionsManager.getConfig().modules.isLoadSignature) {
            try {
                Signature signature = widget.getSignature();
                if (signature.isEmpty()) return;

                RectF rectF = AppUtil.toRectF(widget.getRect());
                mPageIndex = widget.getXFAPage().getIndex();
                mPdfViewCtrl.convertPdfRectToPageViewRect(rectF, rectF, mPageIndex);
                mPdfViewCtrl.convertPageViewRectToDisplayViewRect(rectF, rectF, mPageIndex);
                if (uiExtensionsManager.getDocumentManager().canModifyXFAForm()) {
                    showAnnotMenu(signature.isSigned() ? SIGNED_WIDGET : UN_SINGED_WIDGET, rectF);
                }
                mPdfViewCtrl.invalidate(AppDmUtil.rectFToRect(rectF));
            } catch (PDFException e) {
                e.printStackTrace();
            }
            return;
        }

        if (shouldShowInputSoft(widget)) {
            try {
                mShouldAddSuggestions = AppUtil.isEmpty(widget.getValue());
            } catch (PDFException e) {
                e.printStackTrace();
            }

            SystemUiHelper.getInstance().showNavigationBar(uiExtensionsManager.getAttachedActivity());
            mIsShowEditText = true;
            mAdjustPosition = true;

            if (mEditView != null) {
                mParent.removeView(mEditView);
            }
            mEditView = new EditText(mContext);
            if (AppDisplay.isPad()) { // SDKRD-9313
                mEditView.setImeOptions(EditorInfo.IME_FLAG_NO_EXTRACT_UI);
            }
            mEditView.setLayoutParams(new ViewGroup.LayoutParams(1, 1));
            mEditView.setSingleLine(false);
            mEditView.setText("");

            mParent.addView(mEditView);
            showSoftInput();

            mEditView.setOnKeyListener(new View.OnKeyListener() {
                @Override
                public boolean onKey(View v, int keyCode, KeyEvent event) {
                    if (keyCode == KeyEvent.KEYCODE_DEL && event.getAction() == KeyEvent.ACTION_DOWN) {
                        if (AppUtil.isEmpty(mEditView.getText().toString())) {
                            onBackspaceBtnDown(widget);
//                            mIsBackBtnPush = true;
                        }
                    }
                    return false;
                }
            });

            mEditView.addTextChangedListener(new TextWatcher() {
                @Override
                public void beforeTextChanged(CharSequence s, int start, int count,
                                              int after) {
                    if (mShouldInterceptTextChanged) return;

                    if (count > 0) {
                        mBeforeText = s.subSequence(start, start + count).toString();
                    } else {
                        mBeforeText = "";
                    }
                }

                @Override
                public void onTextChanged(CharSequence s, int start, int before, int count) {
                    if (mShouldInterceptTextChanged) return;

                    try {
                        String charInput = "";

                        if (!mBeforeText.isEmpty()
                                && count >= before
                                && s.subSequence(start, start + before).toString().equals(mBeforeText)) {
                            charInput = s.subSequence(start + before, start + count).toString();
                        } else {
                            for (int i = 0; i < mBeforeText.length(); i++) {
                                charInput += (char) 8;
                            }

                            if (count > 0) {
                                String afterTxt = s.subSequence(start, start + count).toString();
                                charInput += afterTxt;
                            }
                        }

                        if (!charInput.isEmpty()) {
                            for (int i = 0; i < charInput.length(); i++) {
                                char c = charInput.charAt(i);
                                if (FormFillerUtil.isEmojiCharacter((int) c))
                                    break;

                                widget.onChar(c, 0);
                            }
                        }
                    } catch (PDFException ignored) {

                    }
                }

                @Override
                public void afterTextChanged(Editable s) {
                    if (mShouldInterceptTextChanged) {
                        mShouldInterceptTextChanged = false;
                        return;
                    }

                    if (mFNModule != null) {
                        mFNModule.updateSuggestionsbar(s.toString());
                    }
                }
            });

            ViewGroup parent = uiExtensionsManager.getRootView();
            AppKeyboardUtil.setKeyboardListener(parent, parent, new AppKeyboardUtil.IKeyboardListener() {
                @Override
                public void onKeyboardOpened(int keyboardHeight) {
                    if (uiExtensionsManager.getAttachedActivity() == null) return;

                    if (SystemUiHelper.getInstance().isFullScreen())
                        hideStatusBar(uiExtensionsManager.getAttachedActivity());

                    if (mFNModule != null) {
                        mFNModule.showFormNavigationBarWithKeyBoard(getFNBottomPadding());
                    }

                    try {
                        if (keyboardHeight != 0 && mKeyBoardHeight == 0) {
                            if (mCurrentXFAWidget != null && !mCurrentXFAWidget.isEmpty()) {
                                refresh(mCurrentXFAWidget);
                            }
                        }
                    } catch (PDFException e) {
                        e.printStackTrace();
                    }
                }

                @Override
                public void onKeyboardClosed() {
                    if (uiExtensionsManager.getAttachedActivity() == null) return;

                    if (SystemUiHelper.getInstance().isFullScreen())
                        hideStatusBar((uiExtensionsManager.getAttachedActivity()));

                    if (mFNModule != null) {
                        if (mCurrentXFAWidget == null) {
                            mFNModule.hideFormNavigationBar();
                        } else {
                            mFNModule.showFormNavigationBar();
                        }
                    }
                }
            });
        }

        try {
            int type = widget.getType();

            mCurrentWidgetRect = AppUtil.toRectF(widget.getRect());
            mCurrentWidgetType = type;
            if (mFNModule != null) {
                mFNModule.setClearEnable(true);
                if (type == XFAWidget.e_WidgetTypeImageEdit) {

                    RectF rectF = AppUtil.toRectF(widget.getRect());
                    mPageIndex = widget.getXFAPage().getIndex();
                    mPdfViewCtrl.convertPdfRectToPageViewRect(rectF, rectF, mPageIndex);
                    mPdfViewCtrl.convertPageViewRectToDisplayViewRect(rectF, rectF, mPageIndex);
                    showAnnotMenu(IMAGE_WIDGET, rectF);
                } else if (type != XFAWidget.e_WidgetTypeArc
                        && type != XFAWidget.e_WidgetTypeRectangle
                        && type != XFAWidget.e_WidgetTypeLine
                        && type != XFAWidget.e_WidgetTypePushButton) {
                    mFNModule.show();

                    mFNModule.setSuggestionApplyCallback(new FormNavigationModule.ISuggestionApplyCallback() {
                        @Override
                        public void onSuggestionApply(String content) {
                            if (mCurrentXFAWidget == null) return;

                            if (isEditingText()) {
                                try {
                                    resetXFAWidget();
                                    char[] charArray = content.toCharArray();

                                    for (char c : charArray) {
                                        if (FormFillerUtil.isEmojiCharacter((int) c))
                                            break;
                                        mCurrentXFAWidget.onChar(c, 0);
                                    }

                                    mShouldInterceptTextChanged = true;
                                    mEditView.setText(content);
                                    mEditView.setSelection(content.length());
                                } catch (Exception ignored) {
                                }
                            }
                        }
                    });
                }
            }
        } catch (PDFException e) {
            e.printStackTrace();
        }
    }

    @Override
    public void onXFAWidgetDeselected(XFAWidget widget, boolean reRender) {
        if (mCurrentXFAWidget == null) return;
        mCurrentXFAWidget = null;

        UIExtensionsManager uiExtensionsManager = (UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager();
        SystemUiHelper.getInstance().setAllowedHideSystemUI(uiExtensionsManager.getAttachedActivity(), true);
        hideAnnotMenu();

        if (isXFAWidgetSignature(widget)) {
            mPageIndex = -1;
            try {
                mPdfViewCtrl.getXFADoc().killFocus();
                if (!mCurrentWidgetRect.isEmpty()) {
                    float x = (mCurrentWidgetRect.left + mCurrentWidgetRect.right) / 2;
                    float y = (mCurrentWidgetRect.top + mCurrentWidgetRect.bottom) / 2;
                    int pageIndex = widget.getXFAPage().getIndex();
                    XFAWidget tempWidget = getXFAWidget(mPdfViewCtrl, pageIndex, new PointF(x, y));
                    invalidate(tempWidget);
                }
            } catch (PDFException e) {
                e.printStackTrace();
            }
        } else {
            postDismissNavigation();
            AppKeyboardUtil.removeKeyboardListener(uiExtensionsManager.getRootView());
            try {
                mPdfViewCtrl.getXFADoc().killFocus();
                if (!mCurrentWidgetRect.isEmpty()) {
                    float x = (mCurrentWidgetRect.left + mCurrentWidgetRect.right) / 2;
                    float y = (mCurrentWidgetRect.top + mCurrentWidgetRect.bottom) / 2;
                    int pageIndex = widget.getXFAPage().getIndex();
                    XFAWidget tempWidget = getXFAWidget(mPdfViewCtrl, pageIndex, new PointF(x, y));
                    refresh(tempWidget);
                }

                if (mCurrentWidgetType == XFAWidget.e_WidgetTypeTextEdit
                        && SuggestionsHelper.getInstance().isShowSuggestions(ISuggestionsType.Form_Filling)) {
                    if (mShouldAddSuggestions && !AppUtil.isEmpty(widget.getValue())) {
                        mShouldAddSuggestions = false;
                        SuggestionsHelper.getInstance().addSuggestion(ISuggestionsType.Form_Filling, widget.getValue());
                    }
                }
            } catch (PDFException e) {
                e.printStackTrace();
            }
            if (mIsShowEditText) {
                AppUtil.dismissInputSoft(mEditView);
                mParent.removeView(mEditView);
                mIsShowEditText = false;
            }
        }

        mShouldInterceptTextChanged = false;
        mLastTouchPoint.set(0, 0);
        mCurrentWidgetType = -1;
        mCurrentWidgetRect.setEmpty();
        if (mFNModule != null) {
            mFNModule.setSuggestionApplyCallback(null);
            mFNModule.hideFormNavigationBar();
        }
    }

    private boolean isDown = false;
    private PointF mLastPoint = new PointF();
    private PointF mDownPoint = new PointF();

    @Override
    public boolean onTouchEvent(int pageIndex, MotionEvent motionEvent, XFAWidget widget) {
        try {
            if (!((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager().canModifyXFAForm())
                return false;
            PointF devPt = new PointF(motionEvent.getX(), motionEvent.getY());
            PointF pageViewPt = new PointF();
            mPdfViewCtrl.convertDisplayViewPtToPageViewPt(devPt, pageViewPt, pageIndex);
            PointF pdfPointF = new PointF();
            mPdfViewCtrl.convertPageViewPtToPdfPt(pageViewPt, pdfPointF, pageIndex);

            int action = motionEvent.getActionMasked();
            switch (action) {
                case MotionEvent.ACTION_DOWN:
                    if (isSameXFAWidget(mCurrentXFAWidget, widget) && isHitXFAWidget(widget, pdfPointF, pageIndex)) {
                        isDown = true;
                        mDownPoint.set(pageViewPt);
                        mLastPoint.set(pageViewPt);
                        widget.onLButtonDown(AppUtil.toFxPointF(pdfPointF), 0);
                        return true;
                    }
                    return false;
                case MotionEvent.ACTION_MOVE:
                    if (getDistanceOfPoints(pageViewPt, mLastPoint) > 0 && isSameXFAWidget(mCurrentXFAWidget, widget)) {
                        mLastPoint.set(pageViewPt);
                        widget.onMouseMove(AppUtil.toFxPointF(pdfPointF), 0);
                        return true;
                    }
                    return false;
                case MotionEvent.ACTION_UP:
                case MotionEvent.ACTION_CANCEL:
                    if (isHitXFAWidget(widget, pdfPointF, pageIndex) || isDown) {
                        isDown = false;
                        if (getDistanceOfPoints(mDownPoint, mLastPoint) == 0) {
                            widget.onMouseMove(AppUtil.toFxPointF(pdfPointF), 0);
                        }
                        widget.onLButtonUp(AppUtil.toFxPointF(pdfPointF), 0);
                        if (isSameXFAWidget(mCurrentXFAWidget, widget) && shouldShowInputSoft(widget)) {
                            int keybordHeight = AppKeyboardUtil.getKeyboardHeight(mParent);
                            if (0 <= keybordHeight && keybordHeight < AppDisplay.getRawScreenHeight() / 5) {
                                showSoftInput();
                            }
                        }

                        mDownPoint.set(0, 0);
                        mLastPoint.set(0, 0);
                        return true;
                    }
                    mDownPoint.set(0, 0);
                    mLastPoint.set(0, 0);
                    return false;
                default:
            }
        } catch (PDFException e) {

        }
        return false;
    }

    @Override
    public boolean onLongPress(int pageIndex, MotionEvent motionEvent, XFAWidget widget) {
        return false;
    }

    @Override
    public boolean onSingleTapConfirmed(int pageIndex, MotionEvent motionEvent, XFAWidget widget) {
        try {
            if (!((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager().canModifyXFAForm())
                return false;
            PointF devPt = new PointF(motionEvent.getX(), motionEvent.getY());
            PointF pageViewPt = new PointF();
            mPdfViewCtrl.convertDisplayViewPtToPageViewPt(devPt, pageViewPt, pageIndex);
            PointF pdfPointF = new PointF();
            mPdfViewCtrl.convertPageViewPtToPdfPt(pageViewPt, pdfPointF, pageIndex);
            mLastTouchPoint.set(pdfPointF.x, pdfPointF.y);

            boolean ret = false;
            boolean isHit = isHitXFAWidget(widget, pdfPointF, pageIndex);
            if (isSameXFAWidget(mCurrentXFAWidget, widget)) {
                if (isHit) {
                    ret = true;
                } else {
                    if (shouldShowNavigation(widget)) {
                        if (mFNModule != null) {
                            mFNModule.hideFormNavigationBar();
                        }
                        resetDocViewerOffset();
                    }
                    setCurrentXFAWidget(null);
                    ret = false;
                }
            } else {
                if (isXFAWidgetSignature(widget)) {
                    if (((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getConfig().modules.isLoadSignature) {
                        setCurrentXFAWidget(widget);
                        ret = true;
                    } else {
                        setCurrentXFAWidget(null);
                        ret = false;
                    }
                } else {
                    setCurrentXFAWidget(widget);
                    ret = true;
                }
            }

            if (isSameXFAWidget(mCurrentXFAWidget, widget) && isHit) {
                widget.onMouseMove(AppUtil.toFxPointF(pdfPointF), 0);
                widget.onLButtonDown(AppUtil.toFxPointF(pdfPointF), 0);
                widget.onLButtonUp(AppUtil.toFxPointF(pdfPointF), 0);
            }

            return ret;
        } catch (PDFException e) {

        }
        return false;
    }

    @Override
    public void onDraw(int pageIndex, Canvas canvas) {
        if (!mPdfViewCtrl.isDynamicXFA() || !mPdfViewCtrl.isPageVisible(pageIndex)) return;
        if (mCurrentXFAWidget == null || mCurrentXFAWidget.isEmpty()) return;
        try {
            int index = mCurrentXFAWidget.getXFAPage().getIndex();
            if (index != pageIndex) return;
            RectF rect = new RectF(mCurrentWidgetRect);

            PointF viewpoint = new PointF(rect.left, rect.bottom);
            PointF point = new PointF(rect.left, rect.bottom);
            mPdfViewCtrl.convertPdfPtToPageViewPt(viewpoint, viewpoint, pageIndex);
            mPdfViewCtrl.convertPdfPtToPageViewPt(point, point, pageIndex);
            mPdfViewCtrl.convertPageViewPtToDisplayViewPt(viewpoint, viewpoint, pageIndex);
            int type = mCurrentWidgetType;
            if (shouldShowInputSoft(mCurrentWidgetType)) {
                mKeyBoardHeight = AppKeyboardUtil.getKeyboardHeight(mParent);
                if (mAdjustPosition && mKeyBoardHeight > AppDisplay.getRawScreenHeight() / 5) {
                    if (AppDisplay.getRawScreenHeight() - viewpoint.y < (mKeyBoardHeight + AppDisplay.dp2px(116))) {
                        int rawScreenHeight = AppDisplay.getRawScreenHeight();
                        int navBarHeight = AppDisplay.getRealNavBarHeight();
                        mPageOffset = (int) (mKeyBoardHeight - (rawScreenHeight - viewpoint.y));

                        if (mPageOffset != 0 && pageIndex == mPdfViewCtrl.getPageCount() - 1 ||
                                (!mPdfViewCtrl.isContinuous() && mPdfViewCtrl.getPageLayoutMode() == PDFViewCtrl.PAGELAYOUTMODE_SINGLE) ||
                                mPdfViewCtrl.getPageLayoutMode() == PDFViewCtrl.PAGELAYOUTMODE_FACING ||
                                mPdfViewCtrl.getPageLayoutMode() == PDFViewCtrl.PAGELAYOUTMODE_COVER) {

                            PointF point1 = new PointF(0, mPdfViewCtrl.getPageViewHeight(pageIndex));
                            mPdfViewCtrl.convertPageViewPtToDisplayViewPt(point1, point1, pageIndex);
                            float screenHeight = AppDisplay.getActivityHeight();
                            if (point1.y <= screenHeight) {
                                int offset = mPageOffset + AppDisplay.dp2px(116) + navBarHeight;
                                setBottomOffset(offset);
                            }
                        }
                        PointF oriPoint = getPageViewOrigin(mPdfViewCtrl, pageIndex, point.x, point.y);
                        mPdfViewCtrl.gotoPage(pageIndex,
                                oriPoint.x, oriPoint.y + mPageOffset + AppDisplay.dp2px(116) + navBarHeight);
                        mAdjustPosition = false;
                    } else {
                        resetDocViewerOffset();
                    }
                }
            }

            if (pageIndex != mPdfViewCtrl.getPageCount() - 1 &&
                    !(!mPdfViewCtrl.isContinuous() && mPdfViewCtrl.getPageLayoutMode() == PDFViewCtrl.PAGELAYOUTMODE_SINGLE) &&
                    mPdfViewCtrl.getPageLayoutMode() != PDFViewCtrl.PAGELAYOUTMODE_FACING &&
                    mPdfViewCtrl.getPageLayoutMode() != PDFViewCtrl.PAGELAYOUTMODE_COVER) {
                resetDocViewerOffset();
            }
            if (AppKeyboardUtil.getKeyboardHeight(mParent) < AppDisplay.getRawScreenHeight() / 5
                    && (pageIndex == mPdfViewCtrl.getPageCount() - 1 ||
                    (!mPdfViewCtrl.isContinuous() && mPdfViewCtrl.getPageLayoutMode() == PDFViewCtrl.PAGELAYOUTMODE_SINGLE) ||
                    mPdfViewCtrl.getPageLayoutMode() == PDFViewCtrl.PAGELAYOUTMODE_FACING ||
                    mPdfViewCtrl.getPageLayoutMode() == PDFViewCtrl.PAGELAYOUTMODE_COVER)) {
                resetDocViewerOffset();
            }

            final boolean b = type != XFAWidget.e_WidgetTypeArc
                    && type != XFAWidget.e_WidgetTypeRectangle
                    && type != XFAWidget.e_WidgetTypeLine
                    && type != XFAWidget.e_WidgetTypePushButton;
            if (mFNModule != null) {
                if (b) {
                    if (shouldShowInputSoft(mCurrentWidgetType)) {
                        mFNModule.showFormNavigationBarWithKeyBoard(getFNBottomPadding());
                    } else {
                        mFNModule.showFormNavigationBar();
                    }
                }
                if (mCurrentXFAWidget == null) {
                    mFNModule.hide();
                }
            }
            canvas.save();
            canvas.setDrawFilter(new PaintFlagsDrawFilter(0, Paint.ANTI_ALIAS_FLAG | Paint.FILTER_BITMAP_FLAG));
            if (b) {
                RectF bbox = new RectF();
                mPdfViewCtrl.convertPdfRectToPageViewRect(rect, bbox, pageIndex);
                bbox.sort();
                bbox.inset(-5, -5);

                canvas.drawLine(bbox.left, bbox.top, bbox.left, bbox.bottom, mPathPaint);
                canvas.drawLine(bbox.left, bbox.bottom, bbox.right, bbox.bottom, mPathPaint);
                canvas.drawLine(bbox.right, bbox.bottom, bbox.right, bbox.top, mPathPaint);
                canvas.drawLine(bbox.left, bbox.top, bbox.right, bbox.top, mPathPaint);
            }
            canvas.restore();
        } catch (PDFException e) {
            e.printStackTrace();
        }
    }


    public boolean onTouchEvent(int pageIndex, MotionEvent motionEvent) {
        if (!((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager().canModifyXFAForm())
            return false;
        int action = motionEvent.getActionMasked();
        XFAWidget xfaWidget = null;
        switch (action) {
            case MotionEvent.ACTION_DOWN: {
                xfaWidget = mCurrentXFAWidget;
                if (xfaWidget != null && !xfaWidget.isEmpty()) {
                    if (onTouchEvent(pageIndex, motionEvent, xfaWidget)) {
                        return true;
                    }
                }

                xfaWidget = getXFAWidget(mPdfViewCtrl, pageIndex, motionEvent);
                break;
            }
            case MotionEvent.ACTION_MOVE:
            case MotionEvent.ACTION_UP:
            case MotionEvent.ACTION_CANCEL:
                xfaWidget = mCurrentXFAWidget;
                break;
            default:
                return false;
        }
        if (xfaWidget != null && !xfaWidget.isEmpty()) {
            return onTouchEvent(pageIndex, motionEvent, xfaWidget);
        }

        return false;
    }

    public boolean onLongPress(int pageIndex, MotionEvent motionEvent) {
        if (!((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager().canModifyXFAForm())
            return false;
        boolean isCanceled = false;
        if (mCurrentXFAWidget != null && !mCurrentXFAWidget.isEmpty()) {
            if (onLongPress(pageIndex, motionEvent, mCurrentXFAWidget)) {
                return true;
            }
            if (mCurrentXFAWidget == null) {
                isCanceled = true;
            }
        }

        XFAWidget xfaWidget = getXFAWidget(mPdfViewCtrl, pageIndex, motionEvent);
        if (xfaWidget != null && !xfaWidget.isEmpty()) {
            if (onLongPress(pageIndex, motionEvent, xfaWidget)) {
                return true;
            }
        }

        if (isCanceled) {
            return true;
        }

        return false;
    }

    public boolean onSingleTapConfirmed(int pageIndex, MotionEvent motionEvent) {
        if (!((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager().canModifyXFAForm())
            return false;
        boolean isCanceled = false;
        if (mCurrentXFAWidget != null && !mCurrentXFAWidget.isEmpty()) {
            if (onSingleTapConfirmed(pageIndex, motionEvent, mCurrentXFAWidget)) {
                return true;
            }
            if (mCurrentXFAWidget == null) {
                isCanceled = true;
            }
        }

        XFAWidget xfaWidget = getXFAWidget(mPdfViewCtrl, pageIndex, motionEvent);
        if (xfaWidget != null && !xfaWidget.isEmpty()) {
            if (onSingleTapConfirmed(pageIndex, motionEvent, xfaWidget)) {
                return true;
            }
        }

        if (isCanceled) {
            return true;
        }
        return false;
    }

    private XFAWidget getXFAWidget(PDFViewCtrl pdfViewCtrl, int pageIndex, MotionEvent motionEvent) {
        return getXFAWidget(pdfViewCtrl, pageIndex, new PointF(motionEvent.getX(), motionEvent.getY()));
    }

    private XFAWidget getXFAWidget(PDFViewCtrl pdfViewCtrl, int pageIndex, PointF pointF) {
        try {
            PDFViewCtrl.lock();
            PointF devPt = new PointF(pointF.x, pointF.y);
            PointF pageViewPt = new PointF();
            pdfViewCtrl.convertDisplayViewPtToPageViewPt(devPt, pageViewPt, pageIndex);
            PointF pdfPointF = new PointF();
            pdfViewCtrl.convertPageViewPtToPdfPt(pageViewPt, pdfPointF, pageIndex);

            XFAPage xfaPage = pdfViewCtrl.getXFADoc().getPage(pageIndex);
            Matrix matrix = pdfViewCtrl.getDisplayMatrix(pageIndex);
            if (matrix == null) return null;
            Matrix2D matrix2D = AppUtil.toMatrix2D(matrix);
            return xfaPage.getWidgetAtDevicePoint(matrix2D, AppUtil.toFxPointF(pageViewPt), 10.0f);
        } catch (PDFException ignored) {
        } finally {
            PDFViewCtrl.unlock();
        }
        return null;
    }

    private void onBackspaceBtnDown(XFAWidget widget) {
        try {
            widget.onChar((char) 8, 0);
        } catch (PDFException e) {
            e.printStackTrace();
        }
    }

    protected XFAWidget getCurrentXFAWidget() {
        return mCurrentXFAWidget;
    }


    // For XFAWidget signature
    private AnnotMenu mAnnotMenu;
    private ArrayList<Integer> mMenuItems;
    private DigitalSignatureUtil mDSUtil;
    private UITextEditDialog mWillSignDialog;
    private boolean mDefaultAdd = false;
    private int mPageIndex = -1;
    private DigitalSignatureSecurityHandler mDSSecurityHandler;

    private boolean isXFAWidgetSignature(XFAWidget xfaWidget) {
        if (xfaWidget == null || xfaWidget.isEmpty()) return false;
        try {
            if (xfaWidget.getType() == XFAWidget.e_WidgetTypeSignature) return true;
        } catch (PDFException e) {
//            e.printStackTrace();
        }
        return false;
    }

    private void showAnnotMenu(int menuType, RectF rectF) {
        prepareMenuItems(menuType);

        if (mAnnotMenu == null) {
            mAnnotMenu = new AnnotMenuImpl(mContext, mPdfViewCtrl);
        }
        if (mMenuItems.size() > 0) {
            mAnnotMenu.setMenuItems(mMenuItems);
            mAnnotMenu.setListener(mMenuListener);
            mAnnotMenu.setShowAlways(true);
            mAnnotMenu.show(rectF);
        }
    }

    private void prepareMenuItems(int type) {
        if (mMenuItems == null) {
            mMenuItems = new ArrayList<>();
        } else {
            mMenuItems.clear();
        }

        UIExtensionsManager ui = (UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager();
        if (type == UN_SINGED_WIDGET) {
            if (ui.getDocumentManager().withAllPermission(null) && ui.isEnableModification()) {
                mMenuItems.add(AnnotMenu.AM_BT_SIGNATURE);
            }
        } else if (type == SIGNED_WIDGET) {
            mMenuItems.add(AnnotMenu.AM_BT_VERIFY_SIGNATURE);
        } else if (type == IMAGE_WIDGET) {
            mMenuItems.add(AnnotMenu.AM_BT_FILE);
            mMenuItems.add(AnnotMenu.AM_BT_GALERY);
            if (AppDevice.hasCamera(ui.getAttachedActivity())) {
                mMenuItems.add(AnnotMenu.AM_BT_CAMERA);
            }
        }
    }

    private void hideAnnotMenu() {
        if (mAnnotMenu == null) return;
        mAnnotMenu.dismiss();
        mMenuItems.clear();
    }

    private final AnnotMenu.ClickListener mMenuListener = new AnnotMenu.ClickListener() {

        @Override
        public void onAMClick(int id) {
            if (AppUtil.isFastDoubleClick()) return;
            if (id == AnnotMenu.AM_BT_SIGNATURE) {
                mAnnotMenu.dismiss();
                selectCertificate();
            } else if (AnnotMenu.AM_BT_VERIFY_SIGNATURE == id) {
                doVerify();
                setCurrentXFAWidget(null);
            } else if (id == AnnotMenu.AM_BT_GALERY) {
                UIExtensionsManager uiExtensionsManager = (UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager();
                AppIntentUtil.selectImageFromGallery(uiExtensionsManager.getAttachedActivity(), ActRequestCode.REQ_FILE_FROM_GALLERY);
            } else if (id == AnnotMenu.AM_BT_CAMERA) {
                UIExtensionsManager uiExtensionsManager = (UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager();
                AppIntentUtil.selectImageFromCamera(uiExtensionsManager,
                        mCaptureImgPath, ActRequestCode.REQ_CAMERA_PERMISSION, ActRequestCode.REQ_FILE_FROM_CAMERA);
            } else if (id == AnnotMenu.AM_BT_FILE) {
                showSelectFileDialog();
            }
        }
    };

    private void showSelectFileDialog() {
        mFileSelectDialog = getFileSelectDialog();
        mFileSelectDialog.init(new FileFilter() {
            @Override
            public boolean accept(File pathname) {
                return !(pathname.isHidden() || !AppFileUtil.canRead(pathname))
                        && !(pathname.isFile() && !isPictures(pathname));
            }
        }, true);

        mFileSelectDialog.setListener(new MatchDialog.DialogListener() {
            @Override
            public void onResult(long btType) {
            }

            @Override
            public void onBackClick() {
                setCurrentXFAWidget(null);
                mFileSelectDialog.dismiss();
            }

            @Override
            public void onTitleRightButtonClick() {
                if (mFileSelectDialog.getSelectedFiles().size() > 0) {
                    String filepath = mFileSelectDialog.getSelectedFiles().get(0).path;
                    AddImageTask addImageTask = new AddImageTask(filepath, getCurrentXFAWidget(), mPdfViewCtrl, new Task.CallBack() {
                        @Override
                        public void result(Task task) {
                            setCurrentXFAWidget(null);
                        }
                    });
                    mPdfViewCtrl.addTask(addImageTask);
                }
                mFileSelectDialog.dismiss();
            }
        });
        mFileSelectDialog.setOnDLCancelListener(new DialogInterface.OnCancelListener() {
            @Override
            public void onCancel(DialogInterface dialog) {
                setCurrentXFAWidget(null);
            }
        });

        mFileSelectDialog.resetWH();
        mFileSelectDialog.showDialog();
    }

    private UIFileSelectDialog getFileSelectDialog() {
        if (mFileSelectDialog == null) {
            UIExtensionsManager uiExtensionsManager = (UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager();
            mFileSelectDialog = new UIFileSelectDialog(uiExtensionsManager.getAttachedActivity());
            mFileSelectDialog.init(new FileFilter() {
                @Override
                public boolean accept(File pathname) {
                    return !(pathname.isHidden() || !AppFileUtil.canRead(pathname))
                            && !(pathname.isFile() && !isPictures(pathname));
                }
            }, true);
            mFileSelectDialog.setCanceledOnTouchOutside(true);
        } else {
            AppThreadManager.getInstance().getMainThreadHandler().post(new Runnable() {
                @Override
                public void run() {
                    mFileSelectDialog.notifyDataSetChanged();
                }
            });
        }
        return mFileSelectDialog;
    }

    private boolean isPictures(File file) {
        String pathName = file.getName().toLowerCase();
        return pathName.endsWith(".bmp")
                || pathName.endsWith(".jpg")
                || pathName.endsWith(".png")
                || pathName.endsWith(".gif")
                || pathName.endsWith(".tif")
                || pathName.endsWith(".jpeg");
    }

    void onActivityResult(Activity act, int requestCode, int resultCode, Intent data) {
        if (resultCode == Activity.RESULT_OK) {
            String path = null;
            if (requestCode == ActRequestCode.REQ_FILE_FROM_GALLERY && data != null) {
                path = AppFileUtil.getFilePathFromUri(mContext, data.getData());
            } else if (requestCode == ActRequestCode.REQ_FILE_FROM_CAMERA) {
                path = mCaptureImgPath;
            }

            if (!AppUtil.isEmpty(path)) {
                AddImageTask addImageTask = new AddImageTask(path, getCurrentXFAWidget(), mPdfViewCtrl, new Task.CallBack() {
                    @Override
                    public void result(Task task) {
                        setCurrentXFAWidget(null);
                    }
                });
                mPdfViewCtrl.addTask(addImageTask);
            }
        } else {
            setCurrentXFAWidget(null);
        }
    }

    private static class AddImageTask extends Task {

        private final String mImagePath;
        private final PDFViewCtrl mViewCtrl;
        private final XFAWidget mXfaWidget;
        private boolean mRet;

        public AddImageTask(String imagePath, final XFAWidget xfaWidget, PDFViewCtrl viewCtrl, final CallBack callBack) {
            super(new CallBack() {
                @Override
                public void result(Task task) {
                    try {
                        AddImageTask imageTask = (AddImageTask) task;
                        if (imageTask.mRet) {
                            int pageIndex = imageTask.mXfaWidget.getXFAPage().getIndex();
                            if (imageTask.mViewCtrl.isPageVisible(pageIndex)) {
                                RectF rectF = AppUtil.toRectF(imageTask.mXfaWidget.getRect());
                                imageTask.mViewCtrl.convertPdfRectToPageViewRect(rectF, rectF, pageIndex);
                                imageTask.mViewCtrl.refresh(pageIndex, AppDmUtil.rectFToRect(rectF));
                            }
                        }
                        UIExtensionsManager uiManager = (UIExtensionsManager) imageTask.mViewCtrl.getUIExtensionsManager();
                        uiManager.setSaveOptions(new UIExtensionsManager.SaveOptions(PDFDoc.e_SaveFlagRemoveRedundantObjects));

                        if (callBack != null) {
                            callBack.result(imageTask);
                        }
                    } catch (PDFException e) {
                        e.printStackTrace();
                    }
                }
            });

            mXfaWidget = xfaWidget;
            mViewCtrl = viewCtrl;
            mImagePath = imagePath;
        }

        @Override
        protected void execute() {
            if (mXfaWidget == null){
                mRet =false;
                return;
            }

            try {
                Image image = new Image(mImagePath);
                mXfaWidget.setImage(image);
                mRet = true;
            } catch (PDFException e) {
                mRet = false;
            }
        }
    }

    private void selectCertificate() {
        if (mDSUtil == null) {
            mDSUtil = new DigitalSignatureUtil(mContext, mPdfViewCtrl);
            initDBTableForDSG();
        }

        mDSUtil.addCertList(new IDigitalSignatureCallBack() {
            @Override
            public void onCertSelect(String path, CertificateFileInfo fileInfo) {
                if (!AppUtil.isEmpty(path) && !AppUtil.isEmpty(fileInfo.fileName)) {
                    doSign(path);
                } else {
                    setCurrentXFAWidget(null);
                }
            }
        });
    }


    private static final String DB_TABLE_DSG_PFX = "_pfx_dsg_cert";
    private static final String PUBLISHER = "publisher";
    private static final String ISSUER = "issuer";
    private static final String SERIALNUMBER = "serial_number";
    private static final String FILEPATH = "file_path";
    private static final String CHANGEFILEPATH = "file_change_path";
    private static final String FILENAME = "file_name";
    private static final String PASSWORD = "password";

    private void initDBTableForDSG() {
        if (!AppSQLite.getInstance(mContext).isTableExist(DB_TABLE_DSG_PFX)) {
            ArrayList<AppSQLite.FieldInfo> fieldInfos = new ArrayList<AppSQLite.FieldInfo>();
            fieldInfos.add(new AppSQLite.FieldInfo(SERIALNUMBER, AppSQLite.KEY_TYPE_VARCHAR));
            fieldInfos.add(new AppSQLite.FieldInfo(ISSUER, AppSQLite.KEY_TYPE_VARCHAR));
            fieldInfos.add(new AppSQLite.FieldInfo(PUBLISHER, AppSQLite.KEY_TYPE_VARCHAR));
            fieldInfos.add(new AppSQLite.FieldInfo(FILEPATH, AppSQLite.KEY_TYPE_VARCHAR));
            fieldInfos.add(new AppSQLite.FieldInfo(CHANGEFILEPATH, AppSQLite.KEY_TYPE_VARCHAR));
            fieldInfos.add(new AppSQLite.FieldInfo(FILENAME, AppSQLite.KEY_TYPE_VARCHAR));
            fieldInfos.add(new AppSQLite.FieldInfo(PASSWORD, AppSQLite.KEY_TYPE_VARCHAR));
            AppSQLite.getInstance(mContext).createTable(DB_TABLE_DSG_PFX, fieldInfos);
        }
        String filePath = mContext.getFilesDir() + "/DSGCert";
        File file = new File(filePath);
        if (!file.exists()) {
            file.mkdirs();
        }
    }

    private boolean doSign(final String certPath) {
        if (!isXFAWidgetSignature(mCurrentXFAWidget)) return false;
        final Signature signature;
        final RectF rectF;
        try {
            rectF = AppUtil.toRectF(mCurrentXFAWidget.getRect());
            signature = mCurrentXFAWidget.getSignature();
            if (signature == null || signature.isEmpty()) return false;
        } catch (PDFException e) {
            return false;
        }
        if (mDefaultAdd) {
            sign2Doc(signature, certPath, rectF);
            return true;
        }
        if (mWillSignDialog == null || mWillSignDialog.getDialog().getOwnerActivity() == null) {
            if (mPdfViewCtrl.getUIExtensionsManager() == null) return false;
            Context context = ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getAttachedActivity();
            if (context == null) return false;
            mWillSignDialog = new UITextEditDialog(context, UIDialog.NO_INPUT);
            mWillSignDialog.getCancelButton().setOnClickListener(new View.OnClickListener() {

                @Override
                public void onClick(View v) {
                    if (AppUtil.isFastDoubleClick()) return;
                    mWillSignDialog.dismiss();
                    setCurrentXFAWidget(null);
                    if (mPdfViewCtrl.isPageVisible(mPageIndex)) {
                        mPdfViewCtrl.invalidate();
                    }

                }
            });
            mWillSignDialog.setOnCancelListener(new DialogInterface.OnCancelListener() {

                @Override
                public void onCancel(DialogInterface dialog) {
                    setCurrentXFAWidget(null);

                }
            });
            mWillSignDialog.getPromptTextView().setText(AppResource.getString(mContext.getApplicationContext(), R.string.rv_sign_dialog_description));
            mWillSignDialog.setTitle(AppResource.getString(mContext.getApplicationContext(), R.string.rv_sign_dialog_title));
        }
        mWillSignDialog.getOKButton().setOnClickListener(new View.OnClickListener() {

            @Override
            public void onClick(View v) {
                if (AppUtil.isFastDoubleClick()) return;
                mWillSignDialog.dismiss();
                sign2Doc(signature, certPath, rectF);
                mDefaultAdd = true;
            }
        });
        mWillSignDialog.show();
        return true;
    }

    private void sign2Doc(Signature signature, String certPath, RectF rectF) {
        Context context = ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getAttachedActivity();
        if (context == null) return;
        if (!AppUtil.isEmpty(certPath)) {
            // do digital signature sign
            signDigitalSignature(context, signature, certPath, rectF);
        }
    }

    private UISaveAsDialog mSaveAsDialog;

    private void signDigitalSignature(@NonNull Context context, final Signature signature, final String certPath, final RectF rectF) {
        boolean isAutoSaveSignedDoc = ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).isAutoSaveSignedDoc();
        if (isAutoSaveSignedDoc) {
            String userSavePath = ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getSignedDocSavePath();
            if (TextUtils.isEmpty(userSavePath)) {
                // Get origin file path
                userSavePath = mPdfViewCtrl.getFilePath();
                if (TextUtils.isEmpty(userSavePath)) {
                    return;
                }
                int index = userSavePath.lastIndexOf('.');
                if (index < 0) index = userSavePath.length();
                userSavePath = userSavePath.substring(0, index) + "-signed.pdf";
            }
            saveSignFile(signature, certPath, rectF, userSavePath);
        } else {
            mSaveAsDialog = new UISaveAsDialog(context, "sign.pdf", "pdf", new UISaveAsDialog.ISaveAsOnOKClickCallBack() {
                @Override
                public void onOkClick(final String newFilePath) {
                    saveSignFile(signature, certPath, rectF, newFilePath);
                }

                @Override
                public void onCancelClick() {
                    mWillSignDialog.dismiss();
                    setCurrentXFAWidget(null);
                }
            });

            mSaveAsDialog.setOnCancelListener(new UISaveAsDialog.ICancelListener() {
                @Override
                public void onCancelListener() {
                    setCurrentXFAWidget(null);
                }
            });
            mSaveAsDialog.showDialog();
        }
    }

    private void updateThumbnail(String path) {
        LocalModule module = (LocalModule) ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager())
                .getModuleByName(Module.MODULE_NAME_LOCAL);
        if (module != null) {
            module.updateThumbnail(path);
        }
    }

    private void saveSignFile(final Signature signature, final String certPath, final RectF rectF, final String savePath) {
        if (TextUtils.isEmpty(savePath)) return;

        final UIExtensionsManager uiExtensionsManager = (UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager();
        final IUISaveasEventListener saveasListener = uiExtensionsManager.getSaveasEventListener(IUISaveasEventListener.SAVE_BY_SIGNATURE);
        if (saveasListener != null && !saveasListener.isAvailablePath(savePath)) {
            return;
        }

        String tmpPath = savePath;
        final File file = new File(savePath);
        if (file.exists()) {
            tmpPath = savePath + "_tmp.pdf";
        }
        final String finalTmpPath = tmpPath;
        if (saveasListener != null)
            saveasListener.saveasBegin(savePath);
        mDSUtil.addCertSignature(tmpPath, certPath, signature, rectF, mPageIndex, false,null, new IDigitalSignatureCreateCallBack() {
            @Override
            public void onCreateFinish(boolean success) {
                if (!success) {
                    File file = new File(finalTmpPath);
                    file.delete();
                    if (saveasListener != null)
                        saveasListener.saveasFinished(success, savePath);
                    return;
                }

                File newFile = new File(savePath);
                File file = new File(finalTmpPath);
                file.renameTo(newFile);

                if (!mPdfViewCtrl.isPageVisible(mPageIndex)) {
                    setCurrentXFAWidget(null);
                    if (saveasListener != null)
                        saveasListener.saveasFinished(success, savePath);
                    return;
                }

                mAddSignPageIndex = mPageIndex;
                mPdfViewCtrl.cancelAllTask();
                setCurrentXFAWidget(null);
                uiExtensionsManager.getDocumentManager().clearUndoRedo();
                uiExtensionsManager.getDocumentManager().setDocModified(false);
                String openPath = AppFileUtil.getAdaptedFilePath(mContext, savePath);
                mPdfViewCtrl.setKeepZoomRatioWhenReopenDoc(true);
                if (saveasListener != null) {
                    saveasListener.saveasFinished(success, savePath);
                    if (saveasListener.shouldReOpenDoc(savePath)) {
                        mPdfViewCtrl.openDoc(openPath, null);
                    }
                } else {
                    mPdfViewCtrl.openDoc(openPath, null);
                }
                updateThumbnail(savePath);
            }
        });
    }

    private void doVerify() {
        if (mCurrentXFAWidget == null || mCurrentXFAWidget.isEmpty()) return;
        try {
            Signature signature = mCurrentXFAWidget.getSignature();
            if (signature == null || signature.isEmpty() || !signature.isSigned()) return;
            verifySignature(signature);
        } catch (PDFException e) {
            e.printStackTrace();
        }
    }

    public void verifySignature(final Signature signature) {
        if (mDSSecurityHandler == null) {
//            mDSSecurityHandler = new DigitalSignatureSecurityHandler(mContext, mPdfViewCtrl, null);
            DigitalSignatureModule module = (DigitalSignatureModule) ((UIExtensionsManager) (mPdfViewCtrl.getUIExtensionsManager())).getModuleByName(Module.MODULE_NAME_DIGITALSIGNATURE);
            if (module == null) return;
            mDSSecurityHandler = module.getSecurityHandler();
        }

        mDSSecurityHandler.verifySignature(signature);
    }

    protected void onConfigurationChanged(Configuration newConfig) {
        if (mSaveAsDialog != null && mSaveAsDialog.isShowing()) {
            mSaveAsDialog.resetWH();
            mSaveAsDialog.showDialog();
        }

        if (mDSUtil != null && mDSUtil.mViewSupport != null)
            mDSUtil.mViewSupport.onConfigurationChanged();

        if (mFileSelectDialog != null && mFileSelectDialog.isShowing()) {
            mFileSelectDialog.resetWH();
            mFileSelectDialog.showDialog();
        }
    }

    public void onDrawForControls(Canvas canvas) {
        if (mAnnotMenu == null || mCurrentXFAWidget == null || mCurrentXFAWidget.isEmpty()) return;
        try {
            int pageIndex = mCurrentXFAWidget.getXFAPage().getIndex();
            if (mPdfViewCtrl.isPageVisible(pageIndex)) {
                RectF bbox = AppUtil.toRectF(mCurrentXFAWidget.getRect());
                mPdfViewCtrl.convertPdfRectToPageViewRect(bbox, bbox, pageIndex);
                mPdfViewCtrl.convertPageViewRectToDisplayViewRect(bbox, bbox, pageIndex);
                mAnnotMenu.update(bbox);
            }
        } catch (PDFException e) {
            e.printStackTrace();
        }
    }

    boolean isAddCertSignature() {
        return mAddSignPageIndex != -1;
    }

    void gotoSignPage() {
        if (mAddSignPageIndex != -1) {
            mPdfViewCtrl.gotoPage(mAddSignPageIndex);
            mAddSignPageIndex = -1;
        }
    }

    private void hideStatusBar(Activity activity) {
        SystemUiHelper.getInstance().hideStatusBar(activity);
    }

    private int getFNBottomPadding() {
        Rect rect = new Rect();
        mParent.getGlobalVisibleRect(rect);
        int padding;
        int top = rect.top;
        int bottom = rect.bottom;
        int screenHeight = AppDisplay.getRawScreenHeight();

        UIExtensionsManager uiExtensionsManager = (UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager();
        int keyboardHeight;
        if (SystemUiHelper.getInstance().isFullScreenMode(uiExtensionsManager.getAttachedActivity())) {
            keyboardHeight = AppKeyboardUtil.getKeyboardHeight(mParent, true);
        } else {
            if (SystemUiHelper.getInstance().isFullScreen())
                keyboardHeight = AppKeyboardUtil.getKeyboardHeight(mParent, true);
            else
                keyboardHeight = AppKeyboardUtil.getKeyboardHeight(mParent, false);
        }

        if ((screenHeight - bottom - AppDisplay.getNavBarHeight()) >= keyboardHeight) {
            padding = 0;
        } else if (screenHeight - top > keyboardHeight && screenHeight - bottom - AppDisplay.getNavBarHeight() < keyboardHeight) {
            if (screenHeight < (bottom + AppDisplay.getNavBarHeight())) {
                padding = keyboardHeight + AppDisplay.getNavBarHeight();
            } else {
                padding = keyboardHeight;
            }
        } else {
            padding = 0;
        }
        return padding;
    }

    private boolean isEditingText() {
        return mEditView != null && mEditView.getParent() != null;
    }

}
