/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.modules.more;

import android.content.Context;
import android.content.res.Configuration;
import android.view.View;
import android.view.ViewGroup;

import com.foxit.sdk.PDFException;
import com.foxit.sdk.PDFViewCtrl;
import com.foxit.sdk.common.Constants;
import com.foxit.sdk.pdf.PDFDoc;
import com.foxit.sdk.pdf.PDFPage;
import com.foxit.sdk.pdf.annots.Annot;
import com.foxit.uiextensions.IThemeEventListener;
import com.foxit.uiextensions.Module;
import com.foxit.uiextensions.R;
import com.foxit.uiextensions.UIExtensionsManager;
import com.foxit.uiextensions.annots.AnnotEventListener;
import com.foxit.uiextensions.annots.IFlattenEventListener;
import com.foxit.uiextensions.annots.IRedactionEventListener;
import com.foxit.uiextensions.controls.menu.IMenuView;
import com.foxit.uiextensions.controls.toolbar.BaseBar;
import com.foxit.uiextensions.controls.toolbar.IBaseItem;
import com.foxit.uiextensions.controls.toolbar.ToolbarItemConfig;
import com.foxit.uiextensions.controls.toolbar.impl.BaseItemImpl;
import com.foxit.uiextensions.event.DocEventListener;
import com.foxit.uiextensions.event.PageEventListener;
import com.foxit.uiextensions.pdfreader.config.ReadStateConfig;
import com.foxit.uiextensions.pdfreader.impl.MainFrame;
import com.foxit.uiextensions.security.standard.PasswordModule;
import com.foxit.uiextensions.theme.ThemeUtil;
import com.foxit.uiextensions.utils.AppUtil;

public class MoreMenuModule implements Module {
    private Context mContext;
    private PDFViewCtrl mPdfViewer;
    private ViewGroup mParent = null;
    private MoreMenuView mMoreMenuView = null;
    private IBaseItem mMoreMenuItem;
    private PasswordModule mPasswordModule = null;
    private UIExtensionsManager mUiExtensionsManager;

    private boolean mHasFormFillerModule = false;
    //for password
    private boolean mHasPasswordModule = false;
    private boolean mShouldResetFormItems = false;

    public MoreMenuModule(Context context, ViewGroup parent, PDFViewCtrl pdfViewer, PDFViewCtrl.UIExtensionsManager uiExtensionsManager) {
        mContext = context;
        mPdfViewer = pdfViewer;
        mParent = parent;
        mUiExtensionsManager = (UIExtensionsManager) uiExtensionsManager;
    }

    /**
     * Note: This method is only used within RDK
     */
    public MoreMenuView getView() {
        return mMoreMenuView;
    }

    /**
     * get more menu view
     *
     * @return {@link IMenuView}
     */
    public IMenuView getMenuView() {
        if (mMoreMenuView.getMoreMenu() == null)
            mMoreMenuView.initMoreMenuActionView();
        return mMoreMenuView.getMoreMenu();
    }

    @Override
    public String getName() {
        return Module.MODULE_MORE_MENU;
    }

    @Override
    public boolean loadModule() {
        if (mMoreMenuView == null) {
            mMoreMenuView = new MoreMenuView(mContext, mParent, mPdfViewer);
        }
        // Topbar Menu Button
        mMoreMenuItem = new BaseItemImpl(mContext, R.drawable.rd_item_more);
        mMoreMenuItem.setTag(ToolbarItemConfig.ITEM_TOPBAR_MORE);
        mMoreMenuItem.setImageTintList(ThemeUtil.getTopBarIconColorStateList(mContext));
        mMoreMenuItem.setId(R.id.id_top_bar_more_menu);
        mMoreMenuView.setPopupAnchorView(mMoreMenuItem.getContentView());
        mMoreMenuItem.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                if (mMoreMenuClickListener != null) {
                    mMoreMenuClickListener.onClick();
                    return;
                }
                if (AppUtil.isFastDoubleClick()) return;
                if (v.getTag(R.id.more_anchor_view_tag) instanceof View) {
                    mMoreMenuView.setPopupAnchorView((View) v.getTag(R.id.more_anchor_view_tag));
                } else {
                    mMoreMenuView.setPopupAnchorView(mMoreMenuItem.getContentView());
                }
                if (mUiExtensionsManager.getState() == ReadStateConfig.STATE_PANZOOM)
                    mUiExtensionsManager.changeState(ReadStateConfig.STATE_NORMAL);
                mUiExtensionsManager.triggerDismissMenuEvent();
                if (mMoreMenuView.getMoreMenu() == null){
                    configFormFillerModule(mUiExtensionsManager.getModuleByName(MODULE_NAME_FORMFILLER));
                    configPasswordModule(mUiExtensionsManager.getModuleByName(MODULE_NAME_PASSWORD));
                }
                mMoreMenuView.show();
            }
        });
        mUiExtensionsManager.getMainFrame().getTopToolbar().addView(mMoreMenuItem, BaseBar.TB_Position.Position_RB);
        mUiExtensionsManager.registerModule(this);
        mUiExtensionsManager.registerConfigurationChangedListener(mConfigurationChangedListener);
        mUiExtensionsManager.registerThemeEventListener(mThemeEventListener);
        mUiExtensionsManager.getDocumentManager().registerAnnotEventListener(mAnnotEventListener);
        mUiExtensionsManager.getDocumentManager().registerRedactionEventListener(mRedactionListener);
        mUiExtensionsManager.getDocumentManager().registerFlattenEventListener(mFlattenEventListener);

        mPdfViewer.registerDocEventListener(mDocumentEventListener);
        mPdfViewer.registerPageEventListener(mPageEventListener);
        return true;
    }

    @Override
    public boolean unloadModule() {
        mPdfViewer.unregisterDocEventListener(mDocumentEventListener);
        mPdfViewer.unregisterPageEventListener(mPageEventListener);
        mUiExtensionsManager.unregisterConfigurationChangedListener(mConfigurationChangedListener);
        mUiExtensionsManager.unregisterThemeEventListener(mThemeEventListener);
        mUiExtensionsManager.getDocumentManager().unregisterAnnotEventListener(mAnnotEventListener);
        mUiExtensionsManager.getDocumentManager().unregisterRedactionEventListener(mRedactionListener);
        mUiExtensionsManager.getDocumentManager().unregisterFlattenEventListener(mFlattenEventListener);

        mDocumentEventListener = null;
        mPageEventListener = null;
        return true;
    }

    /**
     * Note: This method is only used within RDK
     */
    public void setFilePath(String filePath) {
        mMoreMenuView.setFilePath(filePath);
    }

    private PDFViewCtrl.IDocEventListener mDocumentEventListener = new DocEventListener() {
        @Override
        public void onDocWillOpen() {
        }

        @Override
        public void onDocOpened(PDFDoc document, int errCode) {
            if (errCode != Constants.e_ErrSuccess) {
                return;
            }
            mMoreMenuView.reloadActionMenu();
            mMoreMenuView.onDocOpened();
        }

        @Override
        public void onDocWillClose(PDFDoc document) {
            //close file/folder select dialog
            mMoreMenuView.dismissDialogs();
        }

        @Override
        public void onDocClosed(PDFDoc document, int errCode) {
            if (errCode != Constants.e_ErrSuccess) {
                return;
            }

            if (mHasPasswordModule) {
                mPasswordModule.getPasswordSupport().setDocOpenAuthEvent(true);
                mPasswordModule.getPasswordSupport().setIsOwner(false);
            }
        }

    };

    private UIExtensionsManager.ConfigurationChangedListener mConfigurationChangedListener = new UIExtensionsManager.ConfigurationChangedListener() {
        @Override
        public void onConfigurationChanged(Configuration newConfig) {
            if (mMoreMenuView != null) {
                mMoreMenuView.onConfigurationChanged(newConfig);
//                mMoreMenuView.hide();
//                mMoreMenuView = null;
            }
        }
    };

    private IThemeEventListener mThemeEventListener = new IThemeEventListener() {
        @Override
        public void onThemeColorChanged(String type, int color) {
            if (mMoreMenuView != null)
                mMoreMenuView.updateThemeColor();
        }
    };

    public void AI_showProtectMenu() {
        mMoreMenuView.showProtectMenu();
    }

    private PDFViewCtrl.IPageEventListener mPageEventListener = new PageEventListener() {

        @Override
        public void onPageChanged(int oldPageIndex, int curPageIndex) {
            if (!mUiExtensionsManager.getMainFrame().isToolbarsVisible()) {
                if (mUiExtensionsManager.getMainFrame() instanceof MainFrame)
                    ((MainFrame) mUiExtensionsManager.getMainFrame()).updateFullScreenPageNumberText(curPageIndex);
            }
        }

        @Override
        public void onPagesInserted(boolean success, int dstIndex, int[] pageRanges) {
            if (success) {
                if (mHasFormFillerModule)
                    mMoreMenuView.reloadFormItems();
            }
        }

    };

    private void configFormFillerModule(Module module) {
        if (module == null)
            return;
        mHasFormFillerModule = true;
    }

    private void configPasswordModule(Module module) {
        if (module == null) {
            return;
        }

        mHasPasswordModule = true;
        mPasswordModule = (PasswordModule) module;
    }

    private AnnotEventListener mAnnotEventListener = new AnnotEventListener() {
        @Override
        public void onAnnotAdded(PDFPage page, Annot annot) {
            try {
                if (annot != null && !annot.isEmpty()) {
                    int type = annot.getType();
                    if (Annot.e_Widget == type && mHasFormFillerModule)
                        mMoreMenuView.reloadFormItems();
                }
            } catch (PDFException e) {
                e.printStackTrace();
            }
        }

        @Override
        public void onAnnotWillDelete(PDFPage page, Annot annot) {
            try {
                if (annot != null && !annot.isEmpty()) {
                    int type = annot.getType();
                    if (Annot.e_Widget == type && mHasFormFillerModule) {
                        mShouldResetFormItems = true;
                    }
                }
            } catch (PDFException e) {
                e.printStackTrace();
            }
        }

        @Override
        public void onAnnotDeleted(PDFPage page, Annot annot) {
            if (mShouldResetFormItems) {
                mShouldResetFormItems = false;
                mMoreMenuView.reloadFormItems();
            }
        }

        @Override
        public void onAnnotModified(PDFPage page, Annot annot) {
        }

        @Override
        public void onAnnotChanged(Annot lastAnnot, Annot currentAnnot) {
        }
    };

    private IFlattenEventListener mFlattenEventListener = new IFlattenEventListener() {
        @Override
        public void onAnnotsFlattened(PDFDoc doc) {
            if (mHasFormFillerModule)
                mMoreMenuView.reloadFormItems();
        }
    };

    private IRedactionEventListener mRedactionListener = new IRedactionEventListener() {
        @Override
        public void onAnnotWillApply(PDFPage page, Annot annot) {
        }

        @Override
        public void onAnnotApplied(PDFPage page, Annot annot) {
            if (mHasFormFillerModule)
                mMoreMenuView.reloadFormItems();
        }

        @Override
        public void onAnnotsApplied(PDFDoc doc) {
            if (mHasFormFillerModule)
                mMoreMenuView.reloadFormItems();
        }
    };

    /**
     * Interface definition for a callback to be invoked when the more menu is clicked.
     */
    public interface OnMoreMenuClickListener {
        /**
         * Called when the more menu has been clicked.
         */
        void onClick();
    }

    private OnMoreMenuClickListener mMoreMenuClickListener;

    public void setOnMoreMenuClickListener(OnMoreMenuClickListener listener) {
        mMoreMenuClickListener = listener;
    }
}

