/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.modules.more;

import android.app.ProgressDialog;
import android.content.Context;
import android.content.res.Configuration;
import android.graphics.Rect;
import android.os.Build;
import android.text.Selection;
import android.text.Spannable;
import android.text.TextUtils;
import android.view.View;
import android.view.ViewGroup;
import android.widget.Toast;

import com.foxit.sdk.PDFException;
import com.foxit.sdk.PDFViewCtrl;
import com.foxit.sdk.Task;
import com.foxit.sdk.addon.xfa.XFADoc;
import com.foxit.sdk.common.Constants;
import com.foxit.sdk.common.Progressive;
import com.foxit.sdk.common.Range;
import com.foxit.sdk.common.fxcrt.PauseCallback;
import com.foxit.sdk.common.fxcrt.RectF;
import com.foxit.sdk.common.fxcrt.RectFArray;
import com.foxit.sdk.fdf.FDFDoc;
import com.foxit.sdk.pdf.AnnotationSummary;
import com.foxit.sdk.pdf.AnnotationSummarySettings;
import com.foxit.sdk.pdf.PDFDoc;
import com.foxit.sdk.pdf.PDFPage;
import com.foxit.sdk.pdf.TextPage;
import com.foxit.sdk.pdf.annots.Annot;
import com.foxit.sdk.pdf.annots.Highlight;
import com.foxit.sdk.pdf.annots.QuadPoints;
import com.foxit.sdk.pdf.annots.QuadPointsArray;
import com.foxit.uiextensions.DocumentManager;
import com.foxit.uiextensions.IPermissionProvider;
import com.foxit.uiextensions.IUIInteractionEventListener;
import com.foxit.uiextensions.Module;
import com.foxit.uiextensions.R;
import com.foxit.uiextensions.ToolHandler;
import com.foxit.uiextensions.UIExtensionsManager;
import com.foxit.uiextensions.annots.form.FormFillerModule;
import com.foxit.uiextensions.annots.ink.InkToolHandler;
import com.foxit.uiextensions.annots.redaction.RedactModule;
import com.foxit.uiextensions.controls.dialog.AppDialogManager;
import com.foxit.uiextensions.controls.dialog.MatchDialog;
import com.foxit.uiextensions.controls.dialog.UIDialog;
import com.foxit.uiextensions.controls.dialog.UIMatchDialog;
import com.foxit.uiextensions.controls.dialog.UIPopoverWin;
import com.foxit.uiextensions.controls.dialog.UITextEditDialog;
import com.foxit.uiextensions.controls.dialog.fileselect.UIFileSelectDialog;
import com.foxit.uiextensions.controls.dialog.fileselect.UIFolderSelectDialog;
import com.foxit.uiextensions.controls.dialog.sheetmenu.IActionMenu;
import com.foxit.uiextensions.controls.dialog.sheetmenu.UIActionMenu;
import com.foxit.uiextensions.controls.filebrowser.imp.FileItem;
import com.foxit.uiextensions.controls.menu.IMenuGroup;
import com.foxit.uiextensions.controls.menu.IMenuItem;
import com.foxit.uiextensions.controls.menu.IMenuView;
import com.foxit.uiextensions.controls.menu.MenuViewImpl;
import com.foxit.uiextensions.controls.menu.MenuViewManager;
import com.foxit.uiextensions.controls.menu.SubgroupMenuItemImpl;
import com.foxit.uiextensions.controls.menu.action.IMenuPresenter;
import com.foxit.uiextensions.modules.doc.docinfo.DocInfoModule;
import com.foxit.uiextensions.modules.doc.docinfo.DocInfoView;
import com.foxit.uiextensions.modules.doc.saveas.DocSaveAsModule;
import com.foxit.uiextensions.modules.dynamicxfa.DynamicXFAModule;
import com.foxit.uiextensions.modules.print.PrintModule;
import com.foxit.uiextensions.modules.snapshot.SnapshotDialogFragment;
import com.foxit.uiextensions.modules.snapshot.SnapshotPresenter;
import com.foxit.uiextensions.security.standard.PasswordModule;
import com.foxit.uiextensions.security.trustcertificate.TrustCertificateModule;
import com.foxit.uiextensions.utils.AppAnnotUtil;
import com.foxit.uiextensions.utils.AppDisplay;
import com.foxit.uiextensions.utils.AppFileUtil;
import com.foxit.uiextensions.utils.AppResource;
import com.foxit.uiextensions.utils.AppStorageManager;
import com.foxit.uiextensions.utils.AppUtil;
import com.foxit.uiextensions.utils.Event;
import com.foxit.uiextensions.utils.IResult;
import com.foxit.uiextensions.utils.SystemUiHelper;
import com.foxit.uiextensions.utils.UIToast;
import com.foxit.uiextensions.utils.thread.AppThreadManager;

import java.io.File;
import java.io.FileFilter;
import java.io.FileOutputStream;
import java.util.List;

import androidx.fragment.app.FragmentActivity;

import static com.foxit.uiextensions.Module.MODULE_NAME_DOCINFO;
import static com.foxit.uiextensions.Module.MODULE_NAME_FORMFILLER;
import static com.foxit.uiextensions.Module.MODULE_NAME_PASSWORD;
import static com.foxit.uiextensions.utils.AppAnnotUtil.noAnnotation;
import static com.foxit.uiextensions.utils.AppAnnotUtil.noComment;

public class MoreMenuView {
    private final Context mContext;
    private final PDFViewCtrl mPdfViewCtrl;
    private final ViewGroup mParent;
    private MenuViewImpl mMoreMenuView = null;
    //    private PopupWindow mMenuPopupWindow = null;
//    private ViewGroup mRootView = null;
    private String mFilePath = null;
    private String mExportFilePath = null;
    private ProgressDialog mProgressDlg;
    private String mProgressMsg;
    private UIFileSelectDialog mFileSelectDialog;
    private UIFolderSelectDialog mFolderSelectDialog;
    private final UIExtensionsManager mUIExtensionsManager;
    private View mPopupAnchorView;
    private UIActionMenu mMoreActionMenuWindow;
    private FormFillerModule mFormFilterModule;
    private PasswordModule mPasswordModule;
    private boolean mNeedShowToolbarOnDismiss = true;
    private IMenuItem mProtectRedactionItem;
    private DocInfoView docInfoView;
    private UITextEditDialog mReduceDialog;
    SubgroupMenuItemImpl mProtectItem;
    SubgroupMenuItemImpl mCommentsFieldsItem;
    SubgroupMenuItemImpl summarizeCommentsItem;
    private int mNightMode = -1;
    private UITextEditDialog mReNameDialog;
    private UITextEditDialog mExportDialog;
    private UITextEditDialog mExistsRmDialog;
    private UITextEditDialog mResetDialog;
    private UITextEditDialog mExportDataDialog;
    private UITextEditDialog mFileNameDialog;
    private SubgroupMenuItemImpl mSummarizeCommentsItem;
    private boolean mResetSystemUiOnDismiss = true;

    public MoreMenuView(Context context, ViewGroup parent, PDFViewCtrl pdfViewCtrl) {
        mContext = context;
        mParent = parent;
        mPdfViewCtrl = pdfViewCtrl;
        mUIExtensionsManager = (UIExtensionsManager) pdfViewCtrl.getUIExtensionsManager();
        mNightMode = mContext.getResources().getConfiguration().uiMode & Configuration.UI_MODE_NIGHT_MASK;
    }

    /**
     * show popup window for pad devices and show bottom sheet dialog for phone devices
     */
    public void show() {
        if (mMoreMenuView == null) {
            initMoreMenuActionView();
        } else {
            boolean needReload = false;
            if (mPasswordModule == null) {
                mPasswordModule = (PasswordModule) mUIExtensionsManager.getModuleByName(MODULE_NAME_PASSWORD);
                needReload = true;
            }
            if (mFormFilterModule == null) {
                mFormFilterModule = (FormFillerModule) mUIExtensionsManager.getModuleByName(MODULE_NAME_FORMFILLER);
                needReload = true;
            }
            if (needReload)
                reloadActionMenu();
        }
        if (SystemUiHelper.getInstance().isFullScreen()) {
            mUIExtensionsManager.stopHideToolbarsTimer();
            if (!AppDisplay.isPad())
                SystemUiHelper.getInstance().showNavigationBar(mUIExtensionsManager.getAttachedActivity());
            AppThreadManager.getInstance().getMainThreadHandler().postDelayed(new Runnable() {
                @Override
                public void run() {
                    showMoreMenu();
                }
            }, 200);
        } else {
            mUIExtensionsManager.stopHideToolbarsTimer();
            this.showMoreMenu();
        }
    }

    public void setNeedShowToolbarOnDismiss(boolean flag) {
        mNeedShowToolbarOnDismiss = flag;
    }

    private void showMoreMenu() {
        initMoreActionMenuWindow();
        Rect rect = new Rect();
        mPopupAnchorView.getGlobalVisibleRect(rect);
        mMoreMenuView.setTitleInfo(mFilePath);
        DocumentManager documentManager = mUIExtensionsManager.getDocumentManager();
        if (mSecondaryActionMenuGroup.getItem(MoreMenuConstants.ITEM_SECONDARY_SAVE_AS) != null) {
            mSecondaryActionMenuGroup.getItem(MoreMenuConstants.ITEM_SECONDARY_SAVE_AS).
                    setEnabled(mUIExtensionsManager.isEnableModification());
        }
        if (mSecondaryActionMenuGroup.getItem(MoreMenuConstants.ITEM_SECONDARY_FLATTEN) != null) {
            mSecondaryActionMenuGroup.getItem(MoreMenuConstants.ITEM_SECONDARY_FLATTEN).
                    setEnabled(documentManager.canAddAnnot()
                            && documentManager.canModifyContents()
                            && mUIExtensionsManager.isEnableModification()
                            && documentManager.withAllPermission(null)
                            && documentManager.withModifyPermission()
                            && documentManager.withDeletePermission());
        }
        if (mSecondaryActionMenuGroup.getItem(MoreMenuConstants.ITEM_SECONDARY_REDUCE_FILE_SIZE) != null) {
            mSecondaryActionMenuGroup.getItem(MoreMenuConstants.ITEM_SECONDARY_REDUCE_FILE_SIZE).setEnabled(!documentManager.isXFA()
                    && !documentManager.simpleCheckPDFA(mPdfViewCtrl.getDoc())
                    && documentManager.canModifyContents()
                    && documentManager.canAddAnnot()
                    && documentManager.canFillForm()
                    && documentManager.withModifyPermission()
                    && documentManager.withDeletePermission()
                    && mUIExtensionsManager.isEnableModification());
        }

        if (mSecondaryActionMenuGroup.getItem(MoreMenuConstants.ITEM_SECONDARY_SCREEN) != null) {
            mSecondaryActionMenuGroup.getItem(MoreMenuConstants.ITEM_SECONDARY_SCREEN).setEnabled(
                    !(!mUIExtensionsManager.getDocumentManager().canCopy() ||
                            !mUIExtensionsManager.getDocumentManager().canCopyForAssess()));
        }

        mMoreMenuView.setOnTitleClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                hideMoreMenu(false);
                setNeedShowToolbarOnDismiss(false);
                mUIExtensionsManager.onUIInteractElementClicked(IUIInteractionEventListener.Reading_More_FileProperty);
                DocInfoModule docInfoModule = (DocInfoModule) mUIExtensionsManager.getModuleByName(MODULE_NAME_DOCINFO);
                if (docInfoModule != null) {
                    docInfoView = docInfoModule.getView();
                    docInfoView.setAnchorView(mPopupAnchorView);
                    docInfoView.setBackOnClickListener(new View.OnClickListener() {

                        @Override
                        public void onClick(View v) {
                            if (docInfoView != null) {
                                docInfoView.hideBottomDialog();
                            }
                            showMoreMenu();
                            setNeedShowToolbarOnDismiss(true);
                        }
                    });
                    docInfoView.showActionMenu();
                }
            }
        });
        mMoreActionMenuWindow.setAutoResetSystemUiOnDismiss(false);
        mMoreActionMenuWindow.show(mUIExtensionsManager.getRootView(), rect, UIPopoverWin.ARROW_TOP, 0);
        mResetSystemUiOnDismiss = true;
    }

    private void initMoreActionMenuWindow() {
        if (mMoreActionMenuWindow != null) return;
        mMoreActionMenuWindow = (UIActionMenu) UIActionMenu.newInstance((FragmentActivity) mUIExtensionsManager.getAttachedActivity());
        mMoreActionMenuWindow.setContentView(mMoreMenuView.getContentView());
        mMoreActionMenuWindow.setOnDismissListener(new IActionMenu.OnActionMenuDismissListener() {
            @Override
            public void onDismiss(IActionMenu actionMenu) {
                AppUtil.removeViewFromParent(mMoreMenuView.getContentView());
                mMoreActionMenuWindow = null;
                if (mResetSystemUiOnDismiss) {
                    if (SystemUiHelper.getInstance().isFullScreen()) {
//                            if (AppDisplay.isPad())
                        SystemUiHelper.getInstance().hideSystemUI(mUIExtensionsManager.getAttachedActivity());
//                            else
//                                SystemUiHelper.getInstance().hideStatusBar(mUIExtensionsManager.getAttachedActivity());
                    }
                }
                if (!mNeedShowToolbarOnDismiss) return;
                if (mUIExtensionsManager.getMainFrame().isToolbarsVisible() && mUIExtensionsManager.getConfig().uiSettings.fullscreen)
                    mUIExtensionsManager.startHideToolbarsTimer();
            }
        });
        if (AppDisplay.isPad())
            mMoreActionMenuWindow.setWidth(AppResource.getDimensionPixelSize(mContext, R.dimen.ux_margin_375dp));
    }

    public void hide() {
        this.hideMoreMenu(true);
    }

    public MenuViewImpl getMoreMenu() {
        return mMoreMenuView;
    }

    public void initView() {
        if (mMoreMenuView == null) {
            mMoreMenuView = new MenuViewImpl(mContext, new IMenuView.OnClosedListener() {
                @Override
                public void onClosed() {
                    hideMoreMenu(false);
                }
            });
        }
    }

    private IMenuItem mPrintItem = null;

    protected void reloadDocInfoItems() {
        if (mPrintItem != null) {
            mPrintItem.setEnabled(false);
            boolean canPrint = mUIExtensionsManager.getDocumentManager().canPrint();
            mPrintItem.setEnabled((Build.VERSION.SDK_INT > Build.VERSION_CODES.KITKAT) && canPrint);
        }
        if (mSummarizeCommentsItem != null) {
            mSummarizeCommentsItem.setEnabled(mUIExtensionsManager.getDocumentManager().canCopy());
        }
    }

    public void setFilePath(String filePath) {
        mFilePath = filePath;

        UIExtensionsManager uiExtensionsManager = mUIExtensionsManager;
        DocInfoModule docInfoModule = (DocInfoModule) uiExtensionsManager.getModuleByName(MODULE_NAME_DOCINFO);
        if (docInfoModule != null) {
            docInfoModule.setFilePath(mFilePath);
        }

        PasswordModule passwordModule = (PasswordModule) uiExtensionsManager.getModuleByName(Module.MODULE_NAME_PASSWORD);
        if (passwordModule != null && passwordModule.getPasswordSupport() != null) {
            passwordModule.getPasswordSupport().setFilePath(mFilePath);
        }
    }

    private IMenuItem mImportFormItem = null;
    private IMenuItem mExportFormItem = null;
    private IMenuItem mResetFormItem = null;
    private IMenuItem mCreateFormItem = null;
    private IMenuGroup mFormGroup;

    protected void reloadFormItems() {
        if (mMoreMenuView == null) return;
        if (mImportFormItem != null)
            mImportFormItem.setEnabled(false);
        if (mExportFormItem != null)
            mExportFormItem.setEnabled(false);
        if (mResetFormItem != null)
            mResetFormItem.setEnabled(false);
//        if (mCreateFormItem != null)
//            mCreateFormItem.setEnabled(false);

        PDFDoc doc = mPdfViewCtrl.getDoc();
        try {
            DocumentManager documentManager = mUIExtensionsManager.getDocumentManager();
            if (mPdfViewCtrl.isDynamicXFA()) {
                if (documentManager.withAllPermission(null) && mUIExtensionsManager.isEnableModification()) {
                    mImportFormItem.setEnabled(true);
                    mResetFormItem.setEnabled(true);
                }
                mExportFormItem.setEnabled(true);
            } else if (doc != null) {
//                if (!documentManager.isXFA() && documentManager.canModifyForm() && !documentManager.isSign()) {
//                    mCreateFormItem.setEnabled(true);
//                }
                if (doc.hasForm()) {
                    if (documentManager.canFillForm()
                            && mUIExtensionsManager.isEnableModification()
                            && documentManager.withAllPermission(null)) {
                        mImportFormItem.setEnabled(true);
                        mResetFormItem.setEnabled(true);
                    }

                    if (documentManager.canCopy()) {
                        mExportFormItem.setEnabled(true);
                    }
                }
            }
        } catch (PDFException e) {
            e.printStackTrace();
        }
    }

    /**
     * if rotate the screen, will reset the size of more menu.to fit the screen.
     */
    protected void onConfigurationChanged(Configuration newConfig) {
        if (mPasswordModule != null && mPasswordModule.getPasswordSupport() != null) {
            mPasswordModule.getPasswordSupport().onConfigurationChanged(newConfig);
        }


        if (mFileSelectDialog != null && mFileSelectDialog.isShowing()) {
            mFileSelectDialog.resetWH();
            mFileSelectDialog.showDialog();
        }
        if (mFolderSelectDialog != null && mFolderSelectDialog.isShowing()) {
            mFolderSelectDialog.resetWH();
            mFolderSelectDialog.showDialog();
        }
        //update popup window position when screen rotated
        if (mMoreActionMenuWindow != null) {
            if (mMoreActionMenuWindow.isShowing()) {
                mPdfViewCtrl.postDelayed(new Runnable() {
                    @Override
                    public void run() {
                        showMoreMenu();
                    }
                }, 100);
            }
        }

    }

    public void dismissDialogs() {
        if (mFileSelectDialog != null && mFileSelectDialog.isShowing())
            mFileSelectDialog.dismiss();
        if (mFolderSelectDialog != null && mFolderSelectDialog.isShowing())
            mFolderSelectDialog.dismiss();
    }

    private void hideMoreMenu(boolean resetSystemUiOnDismiss) {
        if (mMoreActionMenuWindow != null && mMoreActionMenuWindow.isShowing()) {
            mResetSystemUiOnDismiss = resetSystemUiOnDismiss;
            mMoreActionMenuWindow.dismiss();
        }
    }

    private MatchDialog.DialogListener mImportFormListener;
    private MatchDialog.DialogListener mImportCommentListener;

    private void importFormFromXML(final IMenuItem item) {
        if (null == mFileSelectDialog)
            mFileSelectDialog = new UIFileSelectDialog(mUIExtensionsManager.getAttachedActivity());
        if (null == mImportFormListener)
            mImportFormListener = new UIMatchDialog.DialogListener() {
                @Override
                public void onResult(long btType) {
                }

                @Override
                public void onBackClick() {
                }

                @Override
                public void onTitleRightButtonClick() {
                    List<FileItem> files = mFileSelectDialog.getSelectedFiles();
                    mFileSelectDialog.dismiss();
                    mProgressMsg = mContext.getString(R.string.fx_form_importing);
                    showProgressDlg();
                    String path = files.get(0).path;
                    path = AppFileUtil.getAdaptedFilePath(mContext, path);

                    String muduleName = mPdfViewCtrl.isDynamicXFA() ? Module.MODULE_NAME_DYNAMICXFA : MODULE_NAME_FORMFILLER;
                    Module module =  mUIExtensionsManager.getModuleByName(muduleName);
                    if (module instanceof FormFillerModule) {
                        ((FormFillerModule) module).importFormFromXML(path, new Event.Callback() {
                            @Override
                            public void result(Event event, boolean success) {
                                ((SubgroupMenuItemImpl) item.getMenuGroup().getParentItem()).hide();
                                dismissProgressDlg();
                                if (success) {
                                    int[] pages = mPdfViewCtrl.getVisiblePages();
                                    for (int pageIndex : pages) {
                                        mPdfViewCtrl.refresh(pageIndex, new Rect(0, 0, mPdfViewCtrl.getPageViewWidth(pageIndex), mPdfViewCtrl.getPageViewHeight(pageIndex)));
                                    }
                                    mUIExtensionsManager.getDocumentManager().setDocModified(true);
                                    UIToast.getInstance(mContext).show(AppResource.getString(mContext, R.string.menu_more_success_import_data));
                                } else {
                                    UIToast.getInstance(mContext).show(AppResource.getString(mContext, R.string.menu_more_fail_import_data));
                                }
                            }
                        });
                    } else if (module instanceof DynamicXFAModule) {
                        ((DynamicXFAModule) module).importData(path, new Event.Callback() {
                            @Override
                            public void result(Event event, boolean success) {
                                dismissProgressDlg();
                                if (success) {
                                    int[] pages = mPdfViewCtrl.getVisiblePages();
                                    for (int pageIndex : pages) {
                                        mPdfViewCtrl.refresh(pageIndex, new Rect(0, 0, mPdfViewCtrl.getPageViewWidth(pageIndex), mPdfViewCtrl.getPageViewHeight(pageIndex)));
                                    }
                                    mUIExtensionsManager.getDocumentManager().setDocModified(true);
                                    UIToast.getInstance(mContext).show(AppResource.getString(mContext, R.string.menu_more_success_import_data));
                                } else {
                                    UIToast.getInstance(mContext).show(AppResource.getString(mContext, R.string.menu_more_fail_import_data));
                                }
                            }
                        });
                    }
                }
            };
        mFileSelectDialog.init(new FileFilter() {
            @Override
            public boolean accept(File pathname) {
                if (pathname.isHidden() || !AppFileUtil.canRead(pathname)) return false;
                return !pathname.isFile() || pathname.getName().toLowerCase().endsWith(".xml");
            }
        }, true);
        mFileSelectDialog.setListener(mImportFormListener);
        mFileSelectDialog.showDialog();
    }

    public void exportFormToXML(final Module module) {
        mExportDataDialog = new UITextEditDialog(mUIExtensionsManager.getAttachedActivity());
        mExportDataDialog.setTitle(mContext.getString(R.string.formfiller_export_title));
        mExportDataDialog.setLengthFilters(UITextEditDialog.MAX_FILE_NAME_LENGTH);
        mExportDataDialog.getInputEditText().setVisibility(View.VISIBLE);
        mExportDataDialog.getPromptTextView().setVisibility(View.GONE);
        String fileNameWithoutExt = AppFileUtil.getFileNameWithoutExt(mFilePath);
        mExportDataDialog.getInputEditText().setText(String.format("%s.xml", fileNameWithoutExt));
        Spannable text = mExportDataDialog.getInputEditText().getText();
        if (text != null && fileNameWithoutExt != null) {
            Selection.setSelection(text, 0, fileNameWithoutExt.length());
        }
        AppUtil.showSoftInput(mExportDataDialog.getInputEditText());
        mExportDataDialog.getCancelButton().setOnClickListener(new View.OnClickListener() {

            @Override
            public void onClick(View v) {
                mExportDataDialog.dismiss();
            }
        });

        mExportDataDialog.getOKButton().setOnClickListener(new View.OnClickListener() {

            @Override
            public void onClick(View v) {
                String name = mExportDataDialog.getInputEditText().getText().toString();
                if (name.toLowerCase().endsWith(".xml")) {
                    mExportFilePath = mFolderSelectDialog.getCurrentPath() + "/" + name;
                } else {
                    mExportFilePath = mFolderSelectDialog.getCurrentPath() + "/" + name + ".xml";
                }
                File file = new File(mExportFilePath);
                if (!mExportFilePath.contentEquals(file.getPath())){
                    UIToast.getInstance(mContext).show(mContext.getString(R.string.menu_more_hint_invalid_name));
                    return;
                }
                mExportDataDialog.dismiss();
                if (file.exists()) {
                    mReNameDialog = new UITextEditDialog(mUIExtensionsManager.getAttachedActivity(), UIDialog.NO_INPUT);
                    mReNameDialog.setTitle(mContext.getString(R.string.fm_file_exist));
                    mReNameDialog.getPromptTextView().setText(mContext.getString(R.string.fx_string_filereplace_warning));
                    mReNameDialog.show();

                    mReNameDialog.getOKButton().setOnClickListener(new View.OnClickListener() {
                        @Override
                        public void onClick(View v) {
                            mReNameDialog.dismiss();
                            mProgressMsg = mContext.getString(R.string.fx_form_exporting);
                            showProgressDlg();
                            Event.Callback callback = new Event.Callback() {
                                @Override
                                public void result(Event event, boolean success) {
                                    dismissProgressDlg();
                                    if (success) {
                                        UIToast.getInstance(mContext).show(AppResource.getString(mContext, R.string.menu_more_success_export_data));
                                    } else {
                                        UIToast.getInstance(mContext).show(mContext.getString(R.string.formfiller_export_error));
                                    }
                                }
                            };
                            String path = AppFileUtil.getAdaptedFilePath(mContext, mExportFilePath);
                            if (module instanceof FormFillerModule) {
                                ((FormFillerModule) module).exportFormToXML(path, callback);
                            } else if (module instanceof DynamicXFAModule) {
                                ((DynamicXFAModule) module).exportData(path, XFADoc.e_ExportDataTypeXML, callback);
                            }
                        }
                    });

                    mReNameDialog.getCancelButton().setOnClickListener(new View.OnClickListener() {
                        @Override
                        public void onClick(View v) {
                            mReNameDialog.dismiss();
                            exportFormToXML(module);
                        }
                    });
                } else {
                    mProgressMsg = mContext.getString(R.string.fx_form_exporting);
                    showProgressDlg();
                    Event.Callback callback = new Event.Callback() {
                        @Override
                        public void result(Event event, boolean success) {
                            dismissProgressDlg();
                            if (success) {
                                UIToast.getInstance(mContext).show(AppResource.getString(mContext, R.string.menu_more_success_export_data));
                            } else {
                                UIToast.getInstance(mContext).show(mContext.getString(R.string.formfiller_export_error));
                            }
                        }
                    };
                    String path = AppFileUtil.getAdaptedFilePath(mContext, mExportFilePath);
                    if (module instanceof FormFillerModule) {
                        ((FormFillerModule) module).exportFormToXML(path, callback);
                    } else if (module instanceof DynamicXFAModule) {
                        ((DynamicXFAModule) module).exportData(path, XFADoc.e_ExportDataTypeXML, callback);
                    }
                }
            }
        });

        mExportDataDialog.show();
    }

    private void resetForm(final Module module) {
        mProgressMsg = mContext.getString(R.string.fx_form_reseting);
        showProgressDlg();
        Event.Callback callback = new Event.Callback() {
            @Override
            public void result(Event event, boolean success) {
                dismissProgressDlg();
                if (success) {
                    mUIExtensionsManager.getDocumentManager().setDocModified(true);
                    mUIExtensionsManager.getDocumentManager().clearUndoRedo();
                    int[] pages = mPdfViewCtrl.isContinuous() ? mPdfViewCtrl.getVisiblePages() : mPdfViewCtrl.getCachePages();
                    if (pages != null) {
                        for (int pageIndex : pages) {
                            Rect rect = new Rect(0, 0, mPdfViewCtrl.getPageViewWidth(pageIndex), mPdfViewCtrl.getPageViewHeight(pageIndex));
                            mPdfViewCtrl.refresh(pageIndex, rect);
                        }
                    }
                }
            }
        };

        if (module instanceof FormFillerModule) {
            ((FormFillerModule) module).resetForm(callback);
        } else if (module instanceof DynamicXFAModule) {
            ((DynamicXFAModule) module).resetForm(callback);
        }
    }


    // For Password Encryption
    private IMenuItem mProtectEncryptionItem;
    private IMenuItem mProtectRemoveItem;
    private IMenuItem mProtectTrustedCertificateItem;

    public void reloadPasswordItem(PasswordModule module) {
        if (mPdfViewCtrl.getDoc() != null) {
            boolean isPPDF = false;
            try {
                String filePath = mPdfViewCtrl.getFilePath();
                if (!TextUtils.isEmpty(filePath)) {
                    isPPDF = filePath.endsWith(".ppdf");
                }
                int encryptType = mPdfViewCtrl.getDoc().getEncryptionType();
                if (encryptType == PDFDoc.e_EncryptPassword && !isPPDF) {
                    mProtectEncryptionItem.setVisible(false);
                    mProtectRemoveItem.setVisible(true);
                } else {
                    mProtectRemoveItem.setVisible(false);
                    mProtectEncryptionItem.setVisible(true);
                }
            } catch (PDFException e) {
                e.printStackTrace();
            }
            if (module.getSecurityHandler().isAvailable() && !isPPDF) {
                if (mProtectEncryptionItem != null)
                    mProtectEncryptionItem.setEnabled(true);
                if (mProtectRemoveItem != null)
                    mProtectRemoveItem.setEnabled(true);
            } else {
                if (mProtectEncryptionItem != null)
                    mProtectEncryptionItem.setEnabled(false);
                if (mProtectRemoveItem != null)
                    mProtectRemoveItem.setEnabled(false);
            }
        }
    }

    private void showProgressDlg() {
        if (mProgressDlg == null && mUIExtensionsManager.getAttachedActivity() != null) {
            mProgressDlg = new ProgressDialog(mUIExtensionsManager.getAttachedActivity());
            mProgressDlg.setProgressStyle(ProgressDialog.STYLE_SPINNER);
            mProgressDlg.setCancelable(false);
            mProgressDlg.setIndeterminate(false);
        }

        if (mProgressDlg != null && !mProgressDlg.isShowing()) {
            mProgressDlg.setMessage(mProgressMsg);
            AppDialogManager.getInstance().showAllowManager(mProgressDlg, null);
        }
    }

    private void dismissProgressDlg() {
        if (mProgressDlg != null && mProgressDlg.isShowing()) {
            AppDialogManager.getInstance().dismiss(mProgressDlg);
            mProgressDlg = null;
        }
    }

    // for import/export comment
    private IMenuItem mImportCommentItem = null;
    private IMenuItem mExportCommentItem = null;
    private IMenuItem mExportHighlightTextItem = null;

    protected void reloadCommentItems() {
        if (mImportCommentItem != null)
            mImportCommentItem.setEnabled(false);
        if (mExportCommentItem != null)
            mExportCommentItem.setEnabled(false);
        if (mExportHighlightTextItem != null)
            mExportHighlightTextItem.setEnabled(false);

        PDFDoc doc = mPdfViewCtrl.getDoc();
        if (mPdfViewCtrl.isDynamicXFA()) {
            mImportCommentItem.setEnabled(false);
            mExportCommentItem.setEnabled(false);
            mExportHighlightTextItem.setEnabled(false);
        } else if (doc != null) {
            DocumentManager documentManager = mUIExtensionsManager.getDocumentManager();
            if (documentManager.canAddAnnot()
                    && documentManager.withAddPermission()
                    && mUIExtensionsManager.isEnableModification()) {
                mImportCommentItem.setEnabled(true);
            }

            if (documentManager.canCopy() || documentManager.canAddAnnot()) {
                mExportCommentItem.setEnabled(true);
                mExportHighlightTextItem.setEnabled(true);
            }
        }
    }

    private void importCommentFromFDF(final IMenuItem item) {
        if (null == mFileSelectDialog)
            mFileSelectDialog = new UIFileSelectDialog(mUIExtensionsManager.getAttachedActivity());
        if (null == mImportCommentListener)
            mImportCommentListener = new UIMatchDialog.DialogListener() {
                @Override
                public void onResult(long btType) {
                }

                @Override
                public void onBackClick() {

                }

                @Override
                public void onTitleRightButtonClick() {
                    List<FileItem> files = mFileSelectDialog.getSelectedFiles();
                    mFileSelectDialog.dismiss();
                    String path = files.get(0).path;
                    path = AppFileUtil.getAdaptedFilePath(mContext, path);
                    try {
                        if (noAnnotation(new FDFDoc(path))) {
                            UIToast.getInstance(mContext).show(AppResource.getString(mContext, R.string.menu_more_no_comments_imported_toast_content), Toast.LENGTH_SHORT);
                            return;
                        }
                        mProgressMsg = mContext.getString(R.string.fx_form_importing);
                        showProgressDlg();
                        MoreMenuView.ImportAndExportComments task = new MoreMenuView.ImportAndExportComments(path, MoreMenuView.ImportAndExportComments.TYPE_IMPORT, new Event.Callback() {
                            @Override
                            public void result(Event event, boolean success) {
                                dismissProgressDlg();
                                ((SubgroupMenuItemImpl) item.getMenuGroup().getParentItem()).hide();
                                if (success) {
                                    mUIExtensionsManager.getDocumentManager().setDocModified(true);
                                    mUIExtensionsManager.getDocumentManager().onAnnosImported();
                                    int[] pages = mPdfViewCtrl.getVisiblePages();
                                    if (pages != null) {
                                        for (int pageIndex : pages) {
                                            Rect rect = new Rect(0, 0, mPdfViewCtrl.getPageViewWidth(pageIndex), mPdfViewCtrl.getPageViewHeight(pageIndex));
                                            mPdfViewCtrl.refresh(pageIndex, rect);
                                        }
                                        UIToast.getInstance(mContext).show(AppResource.getString(mContext, R.string.menu_more_success_import_data));
                                        mUIExtensionsManager.getDocumentManager().clearUndoRedo();
                                    } else {
                                        UIToast.getInstance(mContext).show(AppResource.getString(mContext, R.string.rv_unknown_error));
                                    }

                                } else {
                                    UIToast.getInstance(mContext).show(AppResource.getString(mContext, R.string.menu_more_fail_import_data));
                                }
                            }
                        });
                        mPdfViewCtrl.addTask(task);
                    } catch (PDFException e) {
                        e.printStackTrace();
                    }

                }
            };
        mFileSelectDialog.init(new FileFilter() {
            @Override
            public boolean accept(File pathname) {
                if (pathname.isHidden() || !AppFileUtil.canRead(pathname)) return false;
                return !pathname.isFile() || pathname.getName().toLowerCase().endsWith(".fdf");
            }
        }, true);
        mFileSelectDialog.setListener(mImportCommentListener);
        mFileSelectDialog.showDialog();
    }

    private void onExportHTClicked() {
        showFolderSelectDialog(new MatchDialog.DialogListener() {
            @Override
            public void onResult(long btType) {

            }

            @Override
            public void onBackClick() {

            }

            @Override
            public void onTitleRightButtonClick() {
                mFolderSelectDialog.dismiss();
                mExportDialog = new UITextEditDialog(mUIExtensionsManager.getAttachedActivity());
                mExportDialog.setTitle(mContext.getString(R.string.formfiller_export_title));
                mExportDialog.setLengthFilters(UITextEditDialog.MAX_FILE_NAME_LENGTH);
                mExportDialog.getInputEditText().setVisibility(View.VISIBLE);
                mExportDialog.getPromptTextView().setVisibility(View.VISIBLE);
                mExportDialog.getPromptTextView().setText(mContext.getString(R.string.fx_rename_file));
                String fileNameWithoutExt = AppFileUtil.getFileNameWithoutExt(mFilePath);
                mExportDialog.getInputEditText().setText(String.format("%s.txt", fileNameWithoutExt));
                Spannable text = mExportDialog.getInputEditText().getText();
                if (text != null && fileNameWithoutExt != null) {
                    Selection.setSelection(text, 0, fileNameWithoutExt.length());
                }
                AppUtil.showSoftInput(mExportDialog.getInputEditText());
                mExportDialog.getCancelButton().setOnClickListener(new View.OnClickListener() {
                    @Override
                    public void onClick(View v) {
                        mExportDialog.dismiss();
                    }
                });

                mExportDialog.getOKButton().setOnClickListener(new View.OnClickListener() {

                    @Override
                    public void onClick(View v) {
                        if (mFolderSelectDialog == null) {
                            mExportDialog.dismiss();
                            return;
                        }
                        String name = mExportDialog.getInputEditText().getText().toString();
                        if (name.toLowerCase().endsWith(".txt")) {
                            mExportFilePath = mFolderSelectDialog.getCurrentPath() + "/" + name;
                        } else {
                            mExportFilePath = mFolderSelectDialog.getCurrentPath() + "/" + name + ".txt";
                        }
                        File file = new File(mExportFilePath);
                        if (!mExportFilePath.contentEquals(file.getPath())){
                            UIToast.getInstance(mContext).show(mContext.getString(R.string.menu_more_hint_invalid_name));
                            return;
                        }
                        mExportDialog.dismiss();
                        if (file.exists()) {
                            mExistsRmDialog = new UITextEditDialog(mUIExtensionsManager.getAttachedActivity(), UIDialog.NO_INPUT);
                            mExistsRmDialog.setTitle(mContext.getString(R.string.fm_file_exist));
                            mExistsRmDialog.getPromptTextView().setText(mContext.getString(R.string.fx_string_filereplace_warning));
                            mExistsRmDialog.show();

                            mExistsRmDialog.getOKButton().setOnClickListener(new View.OnClickListener() {
                                @Override
                                public void onClick(View v) {
                                    mExistsRmDialog.dismiss();
                                    mProgressMsg = mContext.getString(R.string.fx_form_exporting);
                                    showProgressDlg();
                                    MoreMenuView.ImportAndExportComments task = new MoreMenuView.ImportAndExportComments(AppFileUtil.getAdaptedFilePath(mContext, mExportFilePath), MoreMenuView.ImportAndExportComments.TYPE_EXPORT_HIGHLIGHT_TEXT, mExportHighlightCallback);
                                    mPdfViewCtrl.addTask(task);
                                }
                            });

                            mExistsRmDialog.getCancelButton().setOnClickListener(new View.OnClickListener() {
                                @Override
                                public void onClick(View v) {
                                    mExistsRmDialog.dismiss();
                                }
                            });
                        } else {
                            mProgressMsg = mContext.getString(R.string.fx_form_exporting);
                            showProgressDlg();

                            MoreMenuView.ImportAndExportComments task = new MoreMenuView.ImportAndExportComments(AppFileUtil.getAdaptedFilePath(mContext, mExportFilePath), MoreMenuView.ImportAndExportComments.TYPE_EXPORT_HIGHLIGHT_TEXT, mExportHighlightCallback);
                            mPdfViewCtrl.addTask(task);
                        }
                    }
                });
                mExportDialog.show();
            }
        });
    }

    private final IResult<Integer, Void, Void> mExportHighlightCallback = new IResult<Integer, Void, Void>() {
        @Override
        public void onResult(boolean success, Integer p1, Void p2, Void p3) {
            dismissProgressDlg();
            if (p1 == ImportAndExportComments.EXPORT_HIGHLIGHT_SUCCESS) {
                UIToast.getInstance(mContext).show(AppResource.getString(mContext, R.string.menu_more_success_export_data));
            } else {
                final UITextEditDialog  warningDlg = new UITextEditDialog(mUIExtensionsManager.getAttachedActivity(), UIDialog.NO_INPUT);
                warningDlg.setTitle(AppResource.getString(mContext, R.string.fx_string_warning));
                warningDlg.getOKButton().setOnClickListener(new View.OnClickListener() {
                    @Override
                    public void onClick(View v) {
                        warningDlg.dismiss();
                    }
                });
                warningDlg.getCancelButton().setVisibility(View.GONE);
                warningDlg.getPromptTextView().setText(p1 == ImportAndExportComments.NO_HIGHTLIGHT_TEXT ?
                        AppResource.getString(mContext, R.string.no_highlight_text_to_export)
                        : AppResource.getString(mContext, R.string.formfiller_export_error));
                warningDlg.show();
            }
        }
    };

    public void exportCommentToFDF(final IMenuItem item) {
        mExportDialog = new UITextEditDialog(mUIExtensionsManager.getAttachedActivity());
        mExportDialog.setTitle(mContext.getString(R.string.formfiller_export_title));
        mExportDialog.setLengthFilters(UITextEditDialog.MAX_FILE_NAME_LENGTH);
        mExportDialog.getInputEditText().setVisibility(View.VISIBLE);
        mExportDialog.getPromptTextView().setVisibility(View.GONE);
        String fileNameWithoutExt = AppFileUtil.getFileNameWithoutExt(mFilePath);
        mExportDialog.getInputEditText().setText(String.format("%s.fdf", fileNameWithoutExt));
        Spannable text = mExportDialog.getInputEditText().getText();
        if (text != null && fileNameWithoutExt != null) {
            Selection.setSelection(text, 0, fileNameWithoutExt.length());
        }
        AppUtil.showSoftInput(mExportDialog.getInputEditText());
        mExportDialog.getCancelButton().setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                mExportDialog.dismiss();
            }
        });

        mExportDialog.getOKButton().setOnClickListener(new View.OnClickListener() {

            @Override
            public void onClick(View v) {
                if (mFolderSelectDialog == null) {
                    mExportDialog.dismiss();
                    return;
                }
                String name = mExportDialog.getInputEditText().getText().toString();
                if (name.toLowerCase().endsWith(".fdf")) {
                    mExportFilePath = mFolderSelectDialog.getCurrentPath() + "/" + name;
                } else {
                    mExportFilePath = mFolderSelectDialog.getCurrentPath() + "/" + name + ".fdf";
                }
                File file = new File(mExportFilePath);
                if (!mExportFilePath.contentEquals(file.getPath())){
                    UIToast.getInstance(mContext).show(mContext.getString(R.string.menu_more_hint_invalid_name));
                    return;
                }
                mExportDialog.dismiss();
                if (file.exists()) {
                    mExistsRmDialog = new UITextEditDialog(mUIExtensionsManager.getAttachedActivity(), UIDialog.NO_INPUT);
                    mExistsRmDialog.setTitle(mContext.getString(R.string.fm_file_exist));
                    mExistsRmDialog.getPromptTextView().setText(mContext.getString(R.string.fx_string_filereplace_warning));
                    mExistsRmDialog.show();

                    mExistsRmDialog.getOKButton().setOnClickListener(new View.OnClickListener() {
                        @Override
                        public void onClick(View v) {
                            mExistsRmDialog.dismiss();
                            mProgressMsg = mContext.getString(R.string.fx_form_exporting);
                            showProgressDlg();
                            Event.Callback callback = new Event.Callback() {
                                @Override
                                public void result(Event event, boolean success) {
                                    dismissProgressDlg();
                                    ((SubgroupMenuItemImpl) item.getMenuGroup().getParentItem()).hide();
                                    if (success) {
                                        UIToast.getInstance(mContext).show(AppResource.getString(mContext, R.string.menu_more_success_export_data));
                                    } else {
                                        UIToast.getInstance(mContext).show(mContext.getString(R.string.formfiller_export_error));
                                    }
                                }
                            };

                            MoreMenuView.ImportAndExportComments task = new MoreMenuView.ImportAndExportComments(AppFileUtil.getAdaptedFilePath(mContext, mExportFilePath), MoreMenuView.ImportAndExportComments.TYPE_EXPORT, callback);
                            mPdfViewCtrl.addTask(task);
                        }
                    });

                    mExistsRmDialog.getCancelButton().setOnClickListener(new View.OnClickListener() {
                        @Override
                        public void onClick(View v) {
                            mExistsRmDialog.dismiss();
                            exportCommentToFDF(item);
                        }
                    });
                } else {
                    mProgressMsg = mContext.getString(R.string.fx_form_exporting);
                    showProgressDlg();
                    Event.Callback callback = new Event.Callback() {
                        @Override
                        public void result(Event event, boolean success) {
                            dismissProgressDlg();
                            ((SubgroupMenuItemImpl) item.getMenuGroup().getParentItem()).hide();
                            if (success) {
                                UIToast.getInstance(mContext).show(AppResource.getString(mContext, R.string.menu_more_success_export_data));
                            } else {
                                UIToast.getInstance(mContext).show(mContext.getString(R.string.formfiller_export_error));
                            }
                        }
                    };

                    MoreMenuView.ImportAndExportComments task = new MoreMenuView.ImportAndExportComments(AppFileUtil.getAdaptedFilePath(mContext, mExportFilePath), MoreMenuView.ImportAndExportComments.TYPE_EXPORT, callback);
                    mPdfViewCtrl.addTask(task);
                }
            }
        });
        mExportDialog.show();
    }

    private MatchDialog.DialogListener mExportCommentListener = null;
    private MatchDialog.DialogListener mExportFormListener = null;


    private void showFolderSelectDialog(UIMatchDialog.DialogListener listener) {
        if (null == mFolderSelectDialog) {
            mFolderSelectDialog = new UIFolderSelectDialog(mUIExtensionsManager.getAttachedActivity());
        }
        mFolderSelectDialog.setListener(listener);
        mFolderSelectDialog.setFileFilter(new FileFilter() {
            @Override
            public boolean accept(File pathname) {
                return !(pathname.isHidden() || !AppFileUtil.canRead(pathname));
            }
        });
        mFolderSelectDialog.showDialog();
    }

    /**
     * set the anchor view for the popup window to help locate its position
     *
     * @param anchorView anchor view to set
     */
    public void setPopupAnchorView(View anchorView) {
        this.mPopupAnchorView = anchorView;
    }

    public View getPopupAnchorView() {
        return mPopupAnchorView;
    }

    private IMenuGroup mPrimaryActionMenuGroup;
    private IMenuGroup mSecondaryActionMenuGroup;

    public void showProtectMenu() {
        setNeedShowToolbarOnDismiss(false);
        hideMoreMenu(false);
        mProtectItem.showActionMenu(mPopupAnchorView);
    }

    public void initMoreMenuActionView() {
        if (null == mPrimaryActionMenuGroup) {
            initView();
            mUIExtensionsManager.getMenuViewManager().setMenuPresenter(MenuViewManager.POSITION_MORE, new IMenuPresenter() {
                @Override
                public IMenuView getMenuView() {
                    return mMoreMenuView;
                }
            });
            mPrimaryActionMenuGroup = mMoreMenuView.addGroup(MoreMenuConstants.GROUP_ACTION_MENU_PRIMARY, null);
            mPrimaryActionMenuGroup.setHeaderTitleVisible(false);
            mPrimaryActionMenuGroup.setDividerVisible(true);
            IMenuItem.OnMenuItemClickListener mPrimaryActionListener = new IMenuItem.OnMenuItemClickListener() {
                @Override
                public void onClick(IMenuItem item) {
                    setNeedShowToolbarOnDismiss(false);
                    hideMoreMenu(false);
                    ((SubgroupMenuItemImpl) item).showActionMenu(mPopupAnchorView);
                }
            };
            SubgroupMenuItemImpl.OnActionMenuBackListener backListener = new SubgroupMenuItemImpl.OnActionMenuBackListener() {
                @Override
                public void onClick(SubgroupMenuItemImpl item) {
                    item.hide();
                    showMoreMenu();
                    setNeedShowToolbarOnDismiss(true);
                }
            };
            createProtectMenu().setOnActionMenuBackListener(backListener).setOnMenuItemClickListener(mPrimaryActionListener);
            createCommentsFieldsMenu().setOnActionMenuBackListener(backListener).setOnMenuItemClickListener(mPrimaryActionListener);
        }
        if (null == mSecondaryActionMenuGroup) {
            mSecondaryActionMenuGroup = mMoreMenuView.addGroup(MoreMenuConstants.GROUP_ACTION_MENU_SECONDARY, null);
            mSecondaryActionMenuGroup.setHeaderTitleVisible(false);
            initSecondaryActionMenu();
            reloadActionMenu();
        }
    }

    private void initSecondaryActionMenu() {
        final DocSaveAsModule saveAsModule = (DocSaveAsModule) mUIExtensionsManager.getModuleByName(Module.MODULE_NAME_SAVE_AS);
        final PrintModule printModule = (PrintModule) mUIExtensionsManager.getModuleByName(Module.MODULE_NAME_PRINT);
        final DocInfoModule docInfoModule = (DocInfoModule) mUIExtensionsManager.getModuleByName(MODULE_NAME_DOCINFO);
        boolean hasDocInfoModule = docInfoModule != null;
        IMenuItem.OnMenuItemClickListener menuItemClickListener = new IMenuItem.OnMenuItemClickListener() {
            @Override
            public void onClick(IMenuItem item) {
                hideMoreMenu(true);
                switch (item.getItemId()) {
                    case MoreMenuConstants.ITEM_SECONDARY_SAVE_AS:
                        mUIExtensionsManager.onUIInteractElementClicked(IUIInteractionEventListener.Reading_More_SaveAs);
                        saveAsModule.showSaveAsDialog();
                        break;
                    case MoreMenuConstants.ITEM_SECONDARY_REDUCE_FILE_SIZE:
                        mUIExtensionsManager.onUIInteractElementClicked(IUIInteractionEventListener.Reading_More_ReduceFileSize);
                        final Runnable doRunnable = new Runnable() {
                            @Override
                            public void run() {
                                if (!AppAnnotUtil.hasModuleLicenseRight(Constants.e_ModuleNameOptimizer)) {
                                    String message = AppUtil.getMessage(mContext, Constants.e_ErrNoOptimizerModuleRight);
                                    UIToast.getInstance(mContext).show(message);
                                    return;
                                }

                                mReduceDialog = UITextEditDialog.quickCreate(mUIExtensionsManager.getAttachedActivity(), R.string.menu_more_confirm, R.string.menu_more_reduce_files_warning_content);
                                mReduceDialog.getOKButton().setOnClickListener(new View.OnClickListener() {
                                    @Override
                                    public void onClick(View v) {
                                        DocSaveAsModule module = (DocSaveAsModule) mUIExtensionsManager.getModuleByName(Module.MODULE_NAME_SAVE_AS);
                                        module.doOptimizeCurrentFile();
                                        mReduceDialog.dismiss();
                                    }
                                });
                                mReduceDialog.getCancelButton().setOnClickListener(new View.OnClickListener() {
                                    @Override
                                    public void onClick(View v) {
                                        mReduceDialog.dismiss();
                                    }
                                });
                                mReduceDialog.show();
                            }
                        };

                        if (mUIExtensionsManager.getPermissionProvider() != null) {
                            mUIExtensionsManager.getPermissionProvider().checkPermission(
                                    IPermissionProvider.FUNCTION_REDUCE_FILE_SIZE, new IPermissionProvider.IPermissionState() {
                                        @Override
                                        public void onState(int state) {
                                            if (state == IPermissionProvider.PERMISSION_STATE_SHOW) {
                                                doRunnable.run();
                                            }
                                        }
                                    }
                            );
                        } else {
                            doRunnable.run();
                        }
                        break;
                    case MoreMenuConstants.ITEM_SECONDARY_PRINT:
                        mUIExtensionsManager.onUIInteractElementClicked(IUIInteractionEventListener.Reading_More_Print);
                        printModule.showPrintSettingOptions();
                        break;
                    case MoreMenuConstants.ITEM_SECONDARY_FLATTEN:
                        mUIExtensionsManager.onUIInteractElementClicked(IUIInteractionEventListener.Reading_More_Flatten);
                        if (mUIExtensionsManager.getPermissionProvider() != null) {
                            mUIExtensionsManager.getPermissionProvider().checkPermission(
                                    IPermissionProvider.FUNCTION_FLATTEN, new IPermissionProvider.IPermissionState() {
                                        @Override
                                        public void onState(int state) {
                                            if (state == IPermissionProvider.PERMISSION_STATE_SHOW) {
                                                DocSaveAsModule module = (DocSaveAsModule) mUIExtensionsManager.getModuleByName(Module.MODULE_NAME_SAVE_AS);
                                                module.doFlattenCurrentFileWithoutSave();
                                            }
                                        }
                                    }
                            );
                        } else {
                            DocSaveAsModule module = (DocSaveAsModule) mUIExtensionsManager.getModuleByName(Module.MODULE_NAME_SAVE_AS);
                            module.doFlattenCurrentFileWithoutSave();
                        }
                        break;
                    case MoreMenuConstants.ITEM_SECONDARY_SCREEN:
                        mUIExtensionsManager.onUIInteractElementClicked(IUIInteractionEventListener.Reading_More_ScreenCapture);
                        //there should verify the security info
//                        if (!mUIExtensionsManager.getDocumentManager().canCopy() ||
//                                !mUIExtensionsManager.getDocumentManager().canCopyForAssess()) {
//                            UIToast.getInstance(mContext).show(AppResource.getString(mContext, R.string.no_permission), Toast.LENGTH_LONG);
//                            return;
//                        }

                        FragmentActivity act = ((FragmentActivity) mUIExtensionsManager.getAttachedActivity());
                        SnapshotDialogFragment mFragment = (SnapshotDialogFragment) act.getSupportFragmentManager().findFragmentByTag("SnapshotFragment");
                        if (mFragment == null) {
                            mFragment = new SnapshotDialogFragment();
                            mFragment.setPdfViewCtrl(mPdfViewCtrl);
                            mFragment.setContext(mContext);
                            new SnapshotPresenter(mContext, mFragment);
                        }

                        AppDialogManager.getInstance().showAllowManager(mFragment, act.getSupportFragmentManager(), "SnapshotFragment", null);
                        break;
                    case MoreMenuConstants.ITEM_SECONDARY_INFO:
                        setNeedShowToolbarOnDismiss(false);
                        mUIExtensionsManager.onUIInteractElementClicked(IUIInteractionEventListener.Reading_More_FileProperty);
                        DocInfoModule docInfoModule = (DocInfoModule) mUIExtensionsManager.getModuleByName(MODULE_NAME_DOCINFO);
                        if (docInfoModule != null) {
                            docInfoView = docInfoModule.getView();
                            docInfoView.setAnchorView(mPopupAnchorView);
                            docInfoView.setBackOnClickListener(new View.OnClickListener() {

                                @Override
                                public void onClick(View v) {
                                    if (docInfoView != null) {
                                        docInfoView.hideBottomDialog();
                                    }
                                    showMoreMenu();
                                    setNeedShowToolbarOnDismiss(true);
                                }
                            });
                            docInfoView.showActionMenu();
                        }
                        break;
                    default:
                        break;
                }
            }
        };
        if (hasDocInfoModule && saveAsModule != null) {
            mSecondaryActionMenuGroup.addItem(MoreMenuConstants.ITEM_SECONDARY_SAVE_AS, AppResource.getDrawable(mContext, R.drawable.more_menu_save_as), AppResource.getString(mContext, R.string.fx_string_saveas))
                    .setOnMenuItemClickListener(menuItemClickListener);
        }
        if (hasDocInfoModule) {
            mSecondaryActionMenuGroup.addItem(MoreMenuConstants.ITEM_SECONDARY_REDUCE_FILE_SIZE, AppResource.getDrawable(mContext, R.drawable.more_menu_reduce_files), AppResource.getString(mContext, R.string.rv_doc_reduce_file_size))
                    .setOnMenuItemClickListener(menuItemClickListener);
        }
        if (hasDocInfoModule && printModule != null) {
            mPrintItem = mSecondaryActionMenuGroup.addItem(MoreMenuConstants.ITEM_SECONDARY_PRINT, AppResource.getDrawable(mContext, R.drawable.more_menu_print), AppResource.getString(mContext, R.string.menu_more_print));
            mPrintItem.setOnMenuItemClickListener(menuItemClickListener);
        }
        mSecondaryActionMenuGroup.addItem(MoreMenuConstants.ITEM_SECONDARY_FLATTEN, AppResource.getDrawable(mContext, R.drawable.more_menu_flatten), AppResource.getString(mContext, R.string.fx_string_flatten))
                .setOnMenuItemClickListener(menuItemClickListener);
        if (hasDocInfoModule) {
            mSecondaryActionMenuGroup.addItem(MoreMenuConstants.ITEM_SECONDARY_SCREEN, AppResource.getDrawable(mContext, R.drawable.more_menu_screen), AppResource.getString(mContext, R.string.menu_more_screen))
                    .setOnMenuItemClickListener(menuItemClickListener);
        }
        mSecondaryActionMenuGroup.addItem(MoreMenuConstants.ITEM_SECONDARY_INFO, AppResource.getDrawable(mContext, R.drawable.more_menu_info_icon), AppResource.getString(mContext, R.string.fx_string_show_info))
                .setOnMenuItemClickListener(menuItemClickListener);
//        if (mFormGroup == null) {
//            mFormGroup = mMoreMenuView.addGroup(MoreMenuConstants.GROUP_FORM, mContext.getString(R.string.menu_more_group_form));
//        }

//        if (mCreateFormItem == null) {
//            mCreateFormItem = mFormGroup.addItem(MoreMenuConstants.ITEM_CREATE_FORM,
//                    AppResource.getString(mContext, R.string.menu_more_item_create));
//            mCreateFormItem.setOnMenuItemClickListener(new IMenuItem.OnMenuItemClickListener() {
//                @Override
//                public void onClick(IMenuItem item) {
//                    hideMoreMenu();
//                    mUIExtensionsManager.triggerDismissMenuEvent();
//                    mUIExtensionsManager.changeState(ReadStateConfig.STATE_CREATE_FORM);
//                }
//            });
//        }
    }

    private SubgroupMenuItemImpl createProtectMenu() {
        mProtectItem = new SubgroupMenuItemImpl(mContext, MoreMenuConstants.ITEM_PRIMARY_PROTECT,
                AppResource.getString(mContext, R.string.menu_more_group_protect), mUIExtensionsManager);
        mPasswordModule = (PasswordModule) mUIExtensionsManager.getModuleByName(MODULE_NAME_PASSWORD);
        IMenuItem.OnMenuItemClickListener listener = new IMenuItem.OnMenuItemClickListener() {
            @Override
            public void onClick(IMenuItem item) {
                ((SubgroupMenuItemImpl) item.getMenuGroup().getParentItem()).hide();
                switch (item.getItemId()) {
                    case MoreMenuConstants.ITEM_PROTECT_REDACTION:
                        mUIExtensionsManager.onUIInteractElementClicked(IUIInteractionEventListener.Reading_More_Protect_Redaction);
                        Module module = mUIExtensionsManager.getModuleByName(Module.MODULE_NAME_REDACT);
                        if (module != null) {
                            ((RedactModule) module).onActivity();
                        }
                        break;
                    case MoreMenuConstants.ITEM_PROTECT_REMOVE_PASSWORD:
                        mPasswordModule.onDecrypt();
                        break;
                    case MoreMenuConstants.ITEM_PROTECT_FILE_ENCRYPTION:
                        mUIExtensionsManager.onUIInteractElementClicked(IUIInteractionEventListener.Reading_More_Protect_FileEncryption);
                        mPasswordModule.onEncrypt();
                        break;
                    case MoreMenuConstants.ITEM_PROTECT_TRUSTED_CERTIFICATES:
                        mUIExtensionsManager.onUIInteractElementClicked(IUIInteractionEventListener.Reading_More_Protect_TrustedCertificates);
                        protectCertificate();
                        break;
                    default:
                        break;
                }
            }
        };

        mProtectRedactionItem = mProtectItem.addSubItem(MoreMenuConstants.ITEM_PROTECT_REDACTION, AppResource.getDrawable(mContext, R.drawable.more_menu_redaction), AppResource.getString(mContext, R.string.fx_string_redaction));
        mProtectRedactionItem.setOnMenuItemClickListener(listener);
        mProtectRedactionItem.setVisible(false);
        mProtectRemoveItem = mProtectItem.addSubItem(MoreMenuConstants.ITEM_PROTECT_REMOVE_PASSWORD, AppResource.getDrawable(mContext, R.drawable.more_menu_remove_password), AppResource.getString(mContext, R.string.menu_more_item_remove_encrytion));
        mProtectRemoveItem.setOnMenuItemClickListener(listener);
        mProtectRemoveItem.setVisible(false);
        mProtectEncryptionItem = mProtectItem.addSubItem(MoreMenuConstants.ITEM_PROTECT_FILE_ENCRYPTION, AppResource.getDrawable(mContext, R.drawable.more_menu_file_encryption), AppResource.getString(mContext, R.string.menu_more_file_encryption));
        mProtectEncryptionItem.setOnMenuItemClickListener(listener);
        mProtectEncryptionItem.setVisible(false);
        mProtectTrustedCertificateItem = mProtectItem.addSubItem(MoreMenuConstants.ITEM_PROTECT_TRUSTED_CERTIFICATES, AppResource.getDrawable(mContext, R.drawable.more_menu_certificate_protect), AppResource.getString(mContext, R.string.menu_more_item_trust_certificate));
        mProtectTrustedCertificateItem.setOnMenuItemClickListener(listener);
        mProtectTrustedCertificateItem.setVisible(false);
        mPrimaryActionMenuGroup.addItem(mProtectItem);
        return mProtectItem;
    }

    private void protectCertificate() {
        TrustCertificateModule trustCertModule = (TrustCertificateModule) mUIExtensionsManager.getModuleByName(Module.MODULE_NAME_TRUST_CERTIFICATE);
        if (trustCertModule != null)
            trustCertModule.show();
    }

    private SubgroupMenuItemImpl createCommentsFieldsMenu() {
        mCommentsFieldsItem = new SubgroupMenuItemImpl(mContext, MoreMenuConstants.ITEM_PRIMARY_COMMENT_FIELDS,
                AppResource.getString(mContext, R.string.menu_more_comments_fields), mUIExtensionsManager);
        IMenuItem.OnMenuItemClickListener listener = new IMenuItem.OnMenuItemClickListener() {
            @Override
            public void onClick(final IMenuItem item) {
                ((SubgroupMenuItemImpl) item.getMenuGroup().getParentItem()).hide();
                switch (item.getItemId()) {
                    case MoreMenuConstants.ITEM_COMMENTS_FIELDS_EXPORT_HIGHLIGHT_TEXT:
                        onExportHTClicked();
                        break;
                    case MoreMenuConstants.ITEM_COMMENTS_FIELDS_IMPORT_COMMENTS:
                        mUIExtensionsManager.onUIInteractElementClicked(IUIInteractionEventListener.Reading_More_CommentField_ImortComments);
                        importCommentFromFDF(item);
                        break;
                    case MoreMenuConstants.ITEM_COMMENTS_FIELDS_EXPORT_COMMENTS:
                        mUIExtensionsManager.onUIInteractElementClicked(IUIInteractionEventListener.Reading_More_CommentField_ExportComments);
                        if (null == mExportCommentListener)
                            mExportCommentListener = new MatchDialog.DialogListener() {
                                @Override
                                public void onResult(long btType) {

                                }

                                @Override
                                public void onBackClick() {

                                }

                                @Override
                                public void onTitleRightButtonClick() {
                                    if (!AppStorageManager.getInstance(mContext).checkDirectoryPermission(mFolderSelectDialog.getCurrentPath()))
                                        return;
                                    mFolderSelectDialog.dismiss();
                                    exportCommentToFDF(item);
                                }
                            };
                        showFolderSelectDialog(mExportCommentListener);
                        break;
                    case MoreMenuConstants.ITEM_COMMENTS_FIELDS_RESET_FORM_FIELDS:
                        mUIExtensionsManager.onUIInteractElementClicked(IUIInteractionEventListener.Reading_More_CommentField_ResetFormFields);
                        mResetDialog = UITextEditDialog.quickCreate(mUIExtensionsManager.getAttachedActivity(), R.string.menu_more_warning, R.string.menu_more_reset_all_form_fields_warning);
                        mResetDialog.getOKButton().setOnClickListener(new View.OnClickListener() {
                            @Override
                            public void onClick(View v) {
                                ((SubgroupMenuItemImpl) item.getMenuGroup().getParentItem()).hide();
                                resetForm(mPdfViewCtrl.isDynamicXFA() ? mUIExtensionsManager.getModuleByName(Module.MODULE_NAME_DYNAMICXFA) : mUIExtensionsManager.getModuleByName(MODULE_NAME_FORMFILLER));
                                mResetDialog.dismiss();
                            }
                        });
                        mResetDialog.getCancelButton().setOnClickListener(new View.OnClickListener() {
                            @Override
                            public void onClick(View v) {
                                mResetDialog.dismiss();
                            }
                        });
                        mResetDialog.show();
                        break;
                    case MoreMenuConstants.ITEM_COMMENTS_FIELDS_IMPORT_FORM_DATA:
                        mUIExtensionsManager.onUIInteractElementClicked(IUIInteractionEventListener.Reading_More_CommentField_ImportFormFields);
                        importFormFromXML(item);
                        break;
                    case MoreMenuConstants.ITEM_COMMENTS_FIELDS_EXPORT_FORM_DATA:
                        mUIExtensionsManager.onUIInteractElementClicked(IUIInteractionEventListener.Reading_More_CommentField_ExportFormFields);
                        if (null == mExportFormListener)
                            mExportFormListener = new MatchDialog.DialogListener() {
                                @Override
                                public void onResult(long btType) {

                                }

                                @Override
                                public void onBackClick() {

                                }

                                @Override
                                public void onTitleRightButtonClick() {
                                    if (!AppStorageManager.getInstance(mContext).checkDirectoryPermission(mFolderSelectDialog.getCurrentPath()))
                                        return;
                                    mFolderSelectDialog.dismiss();
                                    exportFormToXML(mPdfViewCtrl.isDynamicXFA() ? mUIExtensionsManager.getModuleByName(Module.MODULE_NAME_DYNAMICXFA) : mUIExtensionsManager.getModuleByName(MODULE_NAME_FORMFILLER));
                                }
                            };
                        showFolderSelectDialog(mExportFormListener);
                        break;
                    default:
                        break;
                }
            }
        };

        mFormFilterModule = (FormFillerModule) mUIExtensionsManager.getModuleByName(MODULE_NAME_FORMFILLER);
        mImportCommentItem = mCommentsFieldsItem.addSubItem(MoreMenuConstants.ITEM_COMMENTS_FIELDS_IMPORT_COMMENTS, AppResource.getDrawable(mContext, R.drawable.more_menu_import_comments), AppResource.getString(mContext, R.string.menu_more_item_annot_import));
        mImportCommentItem.setOnMenuItemClickListener(listener);
        mExportCommentItem = mCommentsFieldsItem.addSubItem(MoreMenuConstants.ITEM_COMMENTS_FIELDS_EXPORT_COMMENTS, AppResource.getDrawable(mContext, R.drawable.more_menu_export_comments), AppResource.getString(mContext, R.string.menu_more_item_annot_export));
        mExportCommentItem.setOnMenuItemClickListener(listener);
        mCommentsFieldsItem.addSubItem(createSummarizeCommentsMenu());
        mExportHighlightTextItem = mCommentsFieldsItem.addSubItem(MoreMenuConstants.ITEM_COMMENTS_FIELDS_EXPORT_HIGHLIGHT_TEXT, AppResource.getDrawable(mContext, R.drawable.more_menu_export_highlight_text), AppResource.getString(mContext, R.string.menu_more_export_highlight_text));
        mExportHighlightTextItem.setOnMenuItemClickListener(listener);
        if (mUIExtensionsManager.getConfig().modules.isLoadForm) {
            mResetFormItem = mCommentsFieldsItem.addSubItem(MoreMenuConstants.ITEM_COMMENTS_FIELDS_RESET_FORM_FIELDS, AppResource.getDrawable(mContext, R.drawable.more_menu_reset_form_fields), AppResource.getString(mContext, R.string.menu_more_item_reset));
            mResetFormItem.setOnMenuItemClickListener(listener);
            mImportFormItem = mCommentsFieldsItem.addSubItem(MoreMenuConstants.ITEM_COMMENTS_FIELDS_IMPORT_FORM_DATA, AppResource.getDrawable(mContext, R.drawable.more_menu_import_form_data), AppResource.getString(mContext, R.string.menu_more_item_import));
            mImportFormItem.setOnMenuItemClickListener(listener);
            mExportFormItem = mCommentsFieldsItem.addSubItem(MoreMenuConstants.ITEM_COMMENTS_FIELDS_EXPORT_FORM_DATA, AppResource.getDrawable(mContext, R.drawable.more_menu_export_form_data), AppResource.getString(mContext, R.string.menu_more_item_export));
            mExportFormItem.setOnMenuItemClickListener(listener);
        }
        mPrimaryActionMenuGroup.addItem(mCommentsFieldsItem);
        return mCommentsFieldsItem;
    }

    private IMenuItem createSummarizeCommentsMenu() {
        IMenuItem.OnMenuItemClickListener listener = new IMenuItem.OnMenuItemClickListener() {
            @Override
            public void onClick(final IMenuItem item) {
                ((SubgroupMenuItemImpl) item.getMenuGroup().getParentItem()).hide();
                if (item.getItemId() == MoreMenuConstants.ITEM_COMMENTS_FIELDS_SUMMARIZE_COMMENTS) {
                    mUIExtensionsManager.onUIInteractElementClicked(IUIInteractionEventListener.Reading_More_CommentField_SummarizeComments);
                    SubgroupMenuItemImpl subgroupMenuItem = (SubgroupMenuItemImpl) item;
                    subgroupMenuItem.showDialog();
                } else {
                    showFolderSelectDialog(new MatchDialog.DialogListener() {
                        @Override
                        public void onResult(long btType) {

                        }

                        @Override
                        public void onBackClick() {

                        }

                        @Override
                        public void onTitleRightButtonClick() {
                            if (!AppStorageManager.getInstance(mContext).checkDirectoryPermission(mFolderSelectDialog.getCurrentPath()))
                                return;
                            mFolderSelectDialog.dismiss();
                            summarizeComments(item.getTitle().equals(AppResource.getString(mContext, R.string.menu_more_comments_only)), mFolderSelectDialog.getCurrentPath());
                        }
                    });
                }
            }
        };
        mSummarizeCommentsItem = new SubgroupMenuItemImpl(mContext, MoreMenuConstants.ITEM_COMMENTS_FIELDS_SUMMARIZE_COMMENTS,
                AppResource.getDrawable(mContext, R.drawable.more_menu_summarize_comment), AppResource.getString(mContext, R.string.menu_more_summarize_comments), mUIExtensionsManager);
        mSummarizeCommentsItem.setMenuArrowRightViewVisible(false);
        mSummarizeCommentsItem.setOnMenuItemClickListener(listener);
        mSummarizeCommentsItem.addSubItem(AppResource.getString(mContext, R.string.menu_more_document_comments)).setOnMenuItemClickListener(listener);
        mSummarizeCommentsItem.addSubItem(AppResource.getString(mContext, R.string.menu_more_comments_only)).setOnMenuItemClickListener(listener);
        mSummarizeCommentsItem.setEnabled(mUIExtensionsManager.getDocumentManager().canCopy());
        return mSummarizeCommentsItem;
    }

    private void summarizeComments(final boolean commentsOnly, final String folderPath) {
        mFileNameDialog = new UITextEditDialog(mUIExtensionsManager.getAttachedActivity());
        mFileNameDialog.setTitle(mContext.getString(R.string.menu_more_input_file_name));
        mFileNameDialog.setLengthFilters(UITextEditDialog.MAX_FILE_NAME_LENGTH);
        mFileNameDialog.getPromptTextView().setText(R.string.menu_more_input_file_name_prompt_text);
        mFileNameDialog.getInputEditText().setVisibility(View.VISIBLE);
        mFileNameDialog.getInputEditText().setText(String.format(AppResource.getString(mContext, R.string.menu_more_summarize_prefix_text), AppFileUtil.getFileNameWithoutExt(mFilePath)));
        Spannable text = mFileNameDialog.getInputEditText().getText();
        if (text != null) {
            Selection.setSelection(text, 0, text.length());
        }
        AppUtil.showSoftInput(mFileNameDialog.getInputEditText());
        mFileNameDialog.getCancelButton().setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                mFileNameDialog.dismiss();
            }
        });
        mProgressMsg = mContext.getString(R.string.fx_string_processing);
        mFileNameDialog.getOKButton().setOnClickListener(new View.OnClickListener() {

            @Override
            public void onClick(View v) {
                mFileNameDialog.dismiss();
                String name = mFileNameDialog.getInputEditText().getText().toString();
                final String destFilePath = folderPath + "/" + name + ".pdf";
                File file = new File(destFilePath);
                if (file.exists()) {
                    mExistsRmDialog = new UITextEditDialog(mUIExtensionsManager.getAttachedActivity(), UIDialog.NO_INPUT);
                    mExistsRmDialog.setTitle(mContext.getString(R.string.fm_file_exist));
                    mExistsRmDialog.getPromptTextView().setText(mContext.getString(R.string.fx_string_filereplace_warning));
                    mExistsRmDialog.show();

                    mExistsRmDialog.getOKButton().setOnClickListener(new View.OnClickListener() {
                        @Override
                        public void onClick(View v) {
                            mExistsRmDialog.dismiss();
                            showProgressDlg();
                            Event.Callback callback = new Event.Callback() {
                                @Override
                                public void result(Event event, boolean success) {
                                    dismissProgressDlg();
                                    UIToast.getInstance(mContext).show(mContext.getString(!success ? R.string.menu_more_failed_to_create_comment_summary : R.string.menu_more_created_comment_summary));
                                }
                            };
                            summarizeOrNot(destFilePath, callback, commentsOnly);
                        }
                    });

                    mExistsRmDialog.getCancelButton().setOnClickListener(new View.OnClickListener() {
                        @Override
                        public void onClick(View v) {
                            mExistsRmDialog.dismiss();
                            summarizeComments(commentsOnly, folderPath);
                        }
                    });
                } else {
                    showProgressDlg();
                    Event.Callback callback = new Event.Callback() {
                        @Override
                        public void result(Event event, boolean success) {
                            dismissProgressDlg();
                            UIToast.getInstance(mContext).show(mContext.getString(!success ? R.string.menu_more_failed_to_create_comment_summary : R.string.menu_more_created_comment_summary));
                        }
                    };
                    summarizeOrNot(destFilePath, callback, commentsOnly);
                }
            }

            private void summarizeOrNot(String destFilePath, Event.Callback callback, boolean commentsOnly) {
                if (noComment(mPdfViewCtrl.getDoc())) {
                    dismissProgressDlg();
                    UIToast.getInstance(mContext).show(AppResource.getString(mContext, R.string.menu_more_no_comments_found_toast_content));
                } else {
                    destFilePath = AppFileUtil.getAdaptedFilePath(mContext, destFilePath);
                    MoreMenuView.SummarizeCommentsTask task = new MoreMenuView.SummarizeCommentsTask(destFilePath, commentsOnly, callback);
                    mPdfViewCtrl.addTask(task);
                }
            }
        });

        mFileNameDialog.show();
    }

    public void reloadActionMenu() {
        if (mPrimaryActionMenuGroup == null) return;
        if (mFormFilterModule != null) {
            reloadFormItems();
        }
        reloadCommentItems();
        reloadDocInfoItems();
        if (mUIExtensionsManager.getModuleByName(Module.MODULE_NAME_REDACT) != null) {
            mProtectRedactionItem.setVisible(true);
            mProtectRedactionItem.setEnabled(mUIExtensionsManager.getDocumentManager().canAddAnnot()
                    && mUIExtensionsManager.isEnableModification());
        } else {
            mProtectRedactionItem.setVisible(false);
        }
        mProtectTrustedCertificateItem.setVisible(mUIExtensionsManager.getModuleByName(Module.MODULE_NAME_TRUST_CERTIFICATE) != null);
        if (mPasswordModule != null) {
            reloadPasswordItem(mPasswordModule);
            try {
                if (mPdfViewCtrl.getDoc() != null && mPdfViewCtrl.getDoc().getEncryptionType() == PDFDoc.e_EncryptPassword) {
                    mPasswordModule.getPasswordSupport().isOwner();
                }
            } catch (PDFException e) {
                e.printStackTrace();
            }
        }
    }

    public void updateThemeColor() {
        if (mMoreActionMenuWindow != null) {
//            mMoreActionMenuWindow.setDarkModeColor();
            mMoreActionMenuWindow.dismiss();
            mMoreActionMenuWindow = null;
        }
        if (mProtectItem != null) {
            mProtectItem.dismiss();
            mProtectItem.updateThemeColor();
        }
        if (mCommentsFieldsItem != null) {
            mCommentsFieldsItem.dismiss();
            mCommentsFieldsItem.updateThemeColor();
        }
        if (mSummarizeCommentsItem != null) {
            mSummarizeCommentsItem.dismiss();
            mSummarizeCommentsItem.updateThemeColor();
        }
        if (mMoreMenuView != null) {
            mMoreMenuView.setDarkModeColor();
        }
        if (docInfoView != null) {
            docInfoView.hideBottomDialog();
            docInfoView = null;
        }

        if (mReduceDialog != null) {
            mReduceDialog.dismiss();
            mReduceDialog = null;
        }
        if (mExportDialog != null) {
            mExportDialog.dismiss();
            mExportDialog = null;
        }
        if (mFileSelectDialog != null) {
            mFileSelectDialog.dismiss();
            mFileSelectDialog = null;
        }
        if (mFolderSelectDialog != null) {
            mFolderSelectDialog.dismiss();
            mFolderSelectDialog = null;
        }
        if (mReNameDialog != null) {
            mReNameDialog.dismiss();
            mReNameDialog = null;
        }
        if (mExistsRmDialog != null) {
            mExistsRmDialog.dismiss();
            mExistsRmDialog = null;
        }
        if (mResetDialog != null) {
            mResetDialog.dismiss();
            mResetDialog = null;
        }
        if (mExportDataDialog != null) {
            mExportDataDialog.dismiss();
            mExportDataDialog = null;
        }
        if (mFileNameDialog != null) {
            mFileNameDialog.dismiss();
            mFileNameDialog = null;
        }
    }

    class SummarizeCommentsTask extends Task {
        private boolean mOnlyComment;
        private boolean mSucceeded = false;
        private String mPath;

        public SummarizeCommentsTask(String path, Boolean onlyComment, final Event.Callback callBack) {
            super(new CallBack() {
                @Override
                public void result(Task task) {
                    callBack.result(null, ((MoreMenuView.SummarizeCommentsTask) task).mSucceeded);
                }
            });
            mPath = path;
            mOnlyComment = onlyComment;
        }

        @Override
        protected void execute() {
            try {
                PDFViewCtrl.lock();
                mSucceeded = doSummarizeImpl(mOnlyComment, mPdfViewCtrl.getDoc(), mPath, null);
            } finally {
                PDFViewCtrl.unlock();
            }
        }

        private boolean doSummarizeImpl(boolean onlyComment, PDFDoc srcDoc, String destPath, PauseCallback pause) {
            if (srcDoc == null || AppUtil.isEmpty(destPath))
                return false;
            try {
                AnnotationSummary annotationSummary = new AnnotationSummary(srcDoc);
                AnnotationSummarySettings summarySettings = new AnnotationSummarySettings();
                summarySettings.setAnnotType(0, true);
                setThumbnailForAnnotations(summarySettings);
                summarySettings.setFontSize(AnnotationSummarySettings.e_FontSizeMedium);
                summarySettings.setSortType(AnnotationSummarySettings.e_TypeSortByPage);
                summarySettings.setSummaryLayout(onlyComment ? AnnotationSummarySettings.e_SummaryLayoutAnnotationOnly : AnnotationSummarySettings.e_SummaryLayoutSeparatePagesWithSequenceNumber);
                Progressive progressive = annotationSummary.startSummarize(destPath, summarySettings, pause);
                int state = Progressive.e_ToBeContinued;
                while (state == Progressive.e_ToBeContinued) {
                    state = progressive.resume();
                }
                boolean done = state == Progressive.e_Finished;
                if (done) {
                    AppStorageManager storageManager = AppStorageManager.getInstance(mContext);
                    String target = storageManager.toExternalPathFromScopedCache(mPath);
                    if (target != null) {
                        done = storageManager.copyDocument(mPath, target, true);
                    }
                }
                return done;
            } catch (PDFException e) {
                e.printStackTrace();
                File file = new File(destPath);
                if (file.exists()) {
                    file.delete();
                }
            }
            return false;
        }

        private void setThumbnailForAnnotations(AnnotationSummarySettings settings) {
            try {
                settings.setAnnotThumbnail(Annot.e_Note, AppResource.getBitmap(mContext, R.drawable.ic_annotation_summary_note));
                settings.setAnnotThumbnail(Annot.e_FreeText, AppResource.getBitmap(mContext, R.drawable.ic_annotation_summary_typewriter));
                settings.setAnnotThumbnail(Annot.e_Line, AppResource.getBitmap(mContext, R.drawable.ic_annotation_summary_line));
                settings.setAnnotThumbnail(Annot.e_Square, AppResource.getBitmap(mContext, R.drawable.ic_annotation_summary_square));
                settings.setAnnotThumbnail(Annot.e_Circle, AppResource.getBitmap(mContext, R.drawable.ic_annotation_summary_circle));
                settings.setAnnotThumbnail(Annot.e_Polygon, AppResource.getBitmap(mContext, R.drawable.ic_annotation_summary_polygon));
                settings.setAnnotThumbnail(Annot.e_PolyLine, AppResource.getBitmap(mContext, R.drawable.ic_annotation_summary_polyline));
                settings.setAnnotThumbnail(Annot.e_Highlight, AppResource.getBitmap(mContext, R.drawable.ic_annotation_summary_highlight));
                settings.setAnnotThumbnail(Annot.e_Underline, AppResource.getBitmap(mContext, R.drawable.ic_annotation_summary_underline));
                settings.setAnnotThumbnail(Annot.e_Squiggly, AppResource.getBitmap(mContext, R.drawable.ic_annotation_summary_squiggly));
                settings.setAnnotThumbnail(Annot.e_StrikeOut, AppResource.getBitmap(mContext, R.drawable.ic_annotation_summary_strikeout));
                settings.setAnnotThumbnail(Annot.e_Stamp, AppResource.getBitmap(mContext, R.drawable.ic_annotation_summary_stamp));
                settings.setAnnotThumbnail(Annot.e_Caret, AppResource.getBitmap(mContext, R.drawable.ic_annotation_summary_insert_text));
                settings.setAnnotThumbnail(Annot.e_Ink, AppResource.getBitmap(mContext, R.drawable.ic_annotation_summary_pencil));
                settings.setAnnotThumbnail(Annot.e_FileAttachment, AppResource.getBitmap(mContext, R.drawable.ic_annotation_summary_attachment));
            } catch (PDFException e) {
                e.printStackTrace();
            }
        }
    }

    class ImportAndExportComments extends Task {
        public static final int EXPORT_HIGHLIGHT_SUCCESS = 0;
        public static final int NO_HIGHTLIGHT_TEXT = 1;

        public static final int TYPE_IMPORT = 0;
        public static final int TYPE_EXPORT = 1;
        public static final int TYPE_EXPORT_HIGHLIGHT_TEXT = 2;
        private boolean mRet;
        private int mErrorCode;
        private final int mType;
        private final String mPath;

        public ImportAndExportComments(String path, int type, final Event.Callback callBack) {
            super(new CallBack() {
                @Override
                public void result(Task task) {
                    callBack.result(null, ((MoreMenuView.ImportAndExportComments) task).mRet);
                }
            });
            mPath = path;
            mType = type;
        }

        public ImportAndExportComments(String path, int type, final IResult<Integer,Void, Void> callBack) {
            super(new CallBack() {
                @Override
                public void result(Task task) {
                    ImportAndExportComments task1 = (ImportAndExportComments) task;
                    callBack.onResult(task1.mRet, task1.mErrorCode, null, null);
                }
            });
            mPath = path;
            mType = type;
        }

        @Override
        protected void execute() {
            try {
                PDFViewCtrl.lock();
                if (mType == TYPE_IMPORT) {
                    FDFDoc fdfDoc = new FDFDoc(mPath);
                    mRet = mPdfViewCtrl.getDoc().importFromFDF(fdfDoc, PDFDoc.e_Annots, new com.foxit.sdk.common.Range());
                } else if (mType == TYPE_EXPORT) {
                    FDFDoc fdfDoc = new FDFDoc(FDFDoc.e_FDF);
                    mRet = mPdfViewCtrl.getDoc().exportToFDF(fdfDoc, PDFDoc.e_Annots, new com.foxit.sdk.common.Range());
                    if (mRet) {
                        fdfDoc.saveAs(mPath);
                        AppStorageManager storageManager = AppStorageManager.getInstance(mContext);
                        String target = storageManager.toExternalPathFromScopedCache(mPath);
                        if (target != null) {
                            mRet = storageManager.copyDocument(mPath, target, true);
                        }
                    }
                } else if (mType == TYPE_EXPORT_HIGHLIGHT_TEXT) {
                    mErrorCode = exportHighlightedText(mPath);
                }
            } catch (PDFException e) {
                mRet = false;
            } finally {
                PDFViewCtrl.unlock();
            }
        }

    }

    public void onDocOpened() {
        PrintModule printModule = (PrintModule) mUIExtensionsManager.getModuleByName(Module.MODULE_NAME_PRINT);
        if (printModule != null){
            printModule.dismissPrintSetting();
        }

        TrustCertificateModule trustCertModule = (TrustCertificateModule) mUIExtensionsManager.getModuleByName(Module.MODULE_NAME_TRUST_CERTIFICATE);
        if (trustCertModule != null){
            trustCertModule.dismissDialog();
        }

        if (mPasswordModule != null && mPasswordModule.getPasswordSupport() != null) {
            mPasswordModule.getPasswordSupport().hideDialog();
        }

        FragmentActivity act = ((FragmentActivity) mUIExtensionsManager.getAttachedActivity());
        if (act == null) return;
        SnapshotDialogFragment fragment = (SnapshotDialogFragment) act.getSupportFragmentManager().findFragmentByTag("SnapshotFragment");
        if (fragment != null && fragment.isVisible()) {
            AppDialogManager.getInstance().dismiss(fragment);
        }
    }

    private int exportHighlightedText(String path) {
        File file  = new File(path);
        if (file.exists()) {
            file.delete();
        }
        try {
            if (!file.createNewFile())
                return -1;
            FileOutputStream fos = new FileOutputStream(file);
            String pageStr = "";
            StringBuilder HLStr = new StringBuilder();
            for (int i = 0; i < mPdfViewCtrl.getDoc().getPageCount(); i++) {
                pageStr ="Page "+ (i + 1) + "\r\n";
                PDFPage page = mUIExtensionsManager.getDocumentManager().getPage(i, true);
                TextPage textPage = new TextPage(page, TextPage.e_ParseTextNormal);
                for (int j = 0; j < page.getAnnotCount(); j ++) {
                    if (page.getAnnot(j).getType() == Annot.e_Highlight) {
                        HLStr.append(getHighlightedText2(textPage, page.getAnnot(j)));
                    }
                }

                if (!AppUtil.isEmpty(HLStr.toString())) {
                    // write one page text data
                    fos.write(pageStr.getBytes());
                    fos.write(HLStr.toString().getBytes());
                }

                HLStr = new StringBuilder();
            }

            fos.flush();
            return file.length() == 0 ? ImportAndExportComments.NO_HIGHTLIGHT_TEXT : ImportAndExportComments.EXPORT_HIGHLIGHT_SUCCESS;
        } catch (Exception e) {
            return -1;
        }
    }

    private String getHighlightedText(TextPage textPage, Annot annot) {
        StringBuilder text = new StringBuilder();
        try {
            Highlight hlAnnot = new Highlight(annot);
            QuadPointsArray qpArray = hlAnnot.getQuadPoints();

            for (long i = 0; i < qpArray.getSize(); i++) {
                QuadPoints quad_points = qpArray.getAt(i);
                RectF tmp_rect = new RectF(quad_points.getThird().getX(), quad_points.getThird().getY(),
                        quad_points.getSecond().getX(), quad_points.getSecond().getY());
                tmp_rect.normalize();
                int ncount = 0;
                int rc_array_size = 0;
                RectFArray rc_array = null;
                while (ncount < 3) {
                    rc_array = textPage.getTextRectArrayByRect(tmp_rect);
                    if (rc_array.getSize() > 1) {
                        tmp_rect.deflate(0, 1);
                        ncount++;
                    } else {
                        break;
                    }
                }

                rc_array_size = rc_array.getSize();
                for (int j = 0; j < rc_array_size; j++) {
                    RectF item = rc_array.getAt(j);
                    if (item.getRight() > tmp_rect.getRight())
                        item.setRight(tmp_rect.getRight());
                    Range range = textPage.getCharRange(item);
                    int seg_count = range.getSegmentCount();
                    if (seg_count == 0)
                        continue;
                    int start = range.getSegmentStart(0);
                    int end = range.getSegmentEnd(0);
                    RectF rc1 = textPage.getCharInfo(start).getChar_box();
                    RectF rc2 =  textPage.getCharInfo(end).getChar_box();
                    RectF rc_temp = rc1;
                    rc_temp.intersect(tmp_rect);
                    if (!rc_temp.isEmpty()) {
                        float percentage_left = rc_temp.width() / rc1.width();
                        if (percentage_left < 0.5) {
                            start++;
                        }
                    } else {
                        start++;
                    }
                    rc_temp = rc2;
                    rc_temp.intersect(tmp_rect);
                    if (!rc_temp.isEmpty()) {
                        float percentage_right = rc_temp.width() / rc2.width();
                        if (percentage_right < 0.5) {
                            end--;
                        }
                    } else {
                        end--;
                    }

                    text.append(textPage.getChars(start, end - start + 1)).append("\r\n");
                }
            }

        } catch (Exception e) {
            e.printStackTrace();
            return "";
        }
        return text.toString();
    }

    private String getHighlightedText2(TextPage textPage, Annot annot) {
        StringBuilder text = new StringBuilder();
        try {
            String str = "";
            RectFArray rectArray = textPage.getTextRectArrayByRect(annot.getRect());
            for (int i = 0; i < rectArray.getSize(); i++) {
                str = textPage.getTextInRect(rectArray.getAt(i));
                if (!AppUtil.isEmpty(str)) {
                    text.append(str).append("\r\n");
                }
            }

        } catch (Exception e) {
            e.printStackTrace();
            return "";
        }
        return text.toString();
    }
}
