/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.modules.pagenavigation;

import android.app.Activity;
import android.content.Context;
import android.graphics.Color;
import android.graphics.drawable.GradientDrawable;
import android.os.Build;
import android.os.Handler;
import android.os.Message;
import android.text.Editable;
import android.text.InputType;
import android.text.TextUtils;
import android.text.TextWatcher;
import android.util.SparseArray;
import android.view.KeyEvent;
import android.view.View;
import android.view.ViewGroup;
import android.view.animation.AnimationUtils;
import android.widget.EditText;
import android.widget.ImageView;
import android.widget.LinearLayout;
import android.widget.RelativeLayout;
import android.widget.TextView;

import com.foxit.sdk.PDFException;
import com.foxit.sdk.PDFViewCtrl;
import com.foxit.sdk.common.Constants;
import com.foxit.sdk.pdf.PDFDoc;
import com.foxit.sdk.pdf.PageLabels;
import com.foxit.uiextensions.IUIInteractionEventListener;
import com.foxit.uiextensions.Module;
import com.foxit.uiextensions.R;
import com.foxit.uiextensions.ToolHandler;
import com.foxit.uiextensions.UIExtensionsManager;
import com.foxit.uiextensions.controls.dialog.UIDialog;
import com.foxit.uiextensions.controls.dialog.UITextEditDialog;
import com.foxit.uiextensions.controls.toolbar.drag.UIDragToolBar;
import com.foxit.uiextensions.event.DocEventListener;
import com.foxit.uiextensions.event.PageEventListener;
import com.foxit.uiextensions.modules.panel.IPanelManager;
import com.foxit.uiextensions.pdfreader.ILayoutChangeListener;
import com.foxit.uiextensions.pdfreader.ILifecycleEventListener;
import com.foxit.uiextensions.pdfreader.IMainFrame;
import com.foxit.uiextensions.pdfreader.IStateChangeListener;
import com.foxit.uiextensions.pdfreader.config.ReadStateConfig;
import com.foxit.uiextensions.pdfreader.impl.LifecycleEventListener;
import com.foxit.uiextensions.pdfreader.impl.MainFrame;
import com.foxit.uiextensions.utils.AppDisplay;
import com.foxit.uiextensions.utils.AppDmUtil;
import com.foxit.uiextensions.utils.AppKeyboardUtil;
import com.foxit.uiextensions.utils.AppResource;
import com.foxit.uiextensions.utils.AppUtil;
import com.foxit.uiextensions.utils.thread.AppThreadManager;

import java.lang.ref.WeakReference;
import java.util.Timer;
import java.util.TimerTask;

/**
 * The module enable user to navigate between pdf pages by index.
 */
public class PageNavigationModule implements Module {
    private final Context mContext;
    private ViewGroup mParent;
    private final PDFViewCtrl mPdfViewCtrl;

    private boolean mIsClosedState = true;
    private RelativeLayout mClosedRootLayout;
    private LinearLayout mClosedPageLabel;
    private TextView mClosedPageLabel_Total;
    private TextView mClosedPageLabel_Current;
    private ImageView mPreImageView;
    private ImageView mNextImageView;
    private MyHandler mHandler;
    private PDFViewCtrl.UIExtensionsManager mUiExtensionsManager;

    private final int mLastState = ReadStateConfig.STATE_NORMAL;
    private ToolHandler mLastToolHandler;
    private PageNavigationBar mPageNavBar;
    private boolean mShowPanel = false;
    UIDragToolBar mUIDragToolBar;

    public PageNavigationModule(Context context, ViewGroup parent, PDFViewCtrl pdfViewCtrl, PDFViewCtrl.UIExtensionsManager uiExtensionsManager) {
        mContext = context;
        mPdfViewCtrl = pdfViewCtrl;
        mUiExtensionsManager = uiExtensionsManager;
        mParent = parent;
    }

    @Override
    public String getName() {
        return MODULE_NAME_PAGENAV;
    }

    @Override
    public boolean loadModule() {
        if (mUiExtensionsManager != null && mUiExtensionsManager instanceof UIExtensionsManager) {
            ((UIExtensionsManager) mUiExtensionsManager).registerModule(this);
            ((UIExtensionsManager) mUiExtensionsManager).registerStateChangeListener(mStateChangeListener);
            ((UIExtensionsManager) mUiExtensionsManager).registerLifecycleListener(mLifecycleEventListener);
            ((UIExtensionsManager) mUiExtensionsManager).registerLayoutChangeListener(mLayoutChangeListener);
            ((UIExtensionsManager) mUiExtensionsManager).getPanelManager().registerPanelEventListener(mPanelEventListener);
            if (AppDisplay.isPad()) {
                if (((UIExtensionsManager) mUiExtensionsManager).getMainFrame() instanceof MainFrame) {
                    mUIDragToolBar = ((MainFrame) ((UIExtensionsManager) mUiExtensionsManager).getMainFrame()).getDragToolBar();
                    if (mUIDragToolBar != null) {
                        mUIDragToolBar.registerToolbarEventListener(mToolbarEventListener);
                    }
                }
            }
        }
        mHandler = new MyHandler(this);
        mHandler.postDelayed(runnable, 1000 * 5);

        initClosedUI();
//        initOpenedUI();
        mPdfViewCtrl.registerDocEventListener(mDocumentEventListener);
        mPdfViewCtrl.registerPageEventListener(mPageEventListener);
        mPdfViewCtrl.registerLayoutEventListener(layoutEventListener);
        mPdfViewCtrl.setOnKeyListener(mOnKeyKListener);
        onUIStatusChanged();
        return true;
    }

    @Override
    public boolean unloadModule() {
        if (mUiExtensionsManager != null && mUiExtensionsManager instanceof UIExtensionsManager) {
            ((UIExtensionsManager) mUiExtensionsManager).unregisterStateChangeListener(mStateChangeListener);
            ((UIExtensionsManager) mUiExtensionsManager).unregisterLifecycleListener(mLifecycleEventListener);
            ((UIExtensionsManager) mUiExtensionsManager).unregisterLayoutChangeListener(mLayoutChangeListener);
            ((UIExtensionsManager) mUiExtensionsManager).getPanelManager().registerPanelEventListener(mPanelEventListener);
            if (mUIDragToolBar != null) {
                mUIDragToolBar.unregisterToolbarEventListener(mToolbarEventListener);
            }
        }

        disInitClosedUI();
        mPdfViewCtrl.unregisterDocEventListener(mDocumentEventListener);
        mPdfViewCtrl.unregisterPageEventListener(mPageEventListener);
        mPdfViewCtrl.unregisterLayoutEventListener(layoutEventListener);
        mPdfViewCtrl.setOnKeyListener(null);
        mHandler.removeCallbacks(runnable);
        mParent = null;
        mUiExtensionsManager = null;
        return true;
    }

    private void initClosedUI() {
        mClosedRootLayout = (RelativeLayout) View.inflate(mContext, R.layout.rd_gotopage_close, null);
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.Q)
            mClosedRootLayout.setForceDarkAllowed(false);
        mClosedPageLabel = (LinearLayout) mClosedRootLayout.findViewById(R.id.rd_gotopage_pagenumber);
        mClosedPageLabel_Total = (TextView) mClosedRootLayout.findViewById(R.id.rd_gotopage_pagenumber_total);
        mClosedPageLabel_Current = (TextView) mClosedRootLayout.findViewById(R.id.rd_gotopage_pagenumber_current);
        mClosedPageLabel_Current.setText("");
        mClosedPageLabel_Current.setTextColor(Color.WHITE);
        mClosedPageLabel_Total.setText("-");
        mClosedPageLabel_Total.setTextColor(Color.WHITE);
        mClosedPageLabel.setEnabled(false);
        mPreImageView = mClosedRootLayout.findViewById(R.id.rd_jumppage_previous);
        mNextImageView = mClosedRootLayout.findViewById(R.id.rd_jumppage_next);

        setClosedUIClickListener();
        RelativeLayout.LayoutParams closedLP = new RelativeLayout.LayoutParams(RelativeLayout.LayoutParams.WRAP_CONTENT, RelativeLayout.LayoutParams.WRAP_CONTENT);
        closedLP.addRule(RelativeLayout.ALIGN_PARENT_BOTTOM);
        mParent.addView(mClosedRootLayout, closedLP);
        if (AppDisplay.isPad()) {
            RelativeLayout rootView = mClosedRootLayout.findViewById(R.id.rv_gotopage_relativeLayout);
            RelativeLayout.LayoutParams layoutParams = (RelativeLayout.LayoutParams) rootView.getLayoutParams();
            int height = AppResource.getDimensionPixelSize(mContext, R.dimen.ux_panel_page_text_width);
            layoutParams.height = height;

            GradientDrawable rootViewBackground = (GradientDrawable) rootView.getBackground();
            rootViewBackground.setCornerRadius(height / 2.0f);

            View leftPV= mClosedRootLayout.findViewById(R.id.id_left_placeholder_view);
            RelativeLayout.LayoutParams leftPVParams = (RelativeLayout.LayoutParams) leftPV.getLayoutParams();
            leftPVParams.width = AppResource.getDimensionPixelSize(mContext, R.dimen.ux_margin_16dp);

            View rightPV= mClosedRootLayout.findViewById(R.id.id_right_placeholder_view);
            RelativeLayout.LayoutParams rightPVParams = (RelativeLayout.LayoutParams) rightPV.getLayoutParams();
            rightPVParams.width = AppResource.getDimensionPixelSize(mContext, R.dimen.ux_margin_16dp);

            RelativeLayout.LayoutParams preImageParams = (RelativeLayout.LayoutParams) mPreImageView.getLayoutParams();
            preImageParams.setMarginStart(0);
            preImageParams.setMarginEnd(AppResource.getDimensionPixelSize(mContext, R.dimen.ux_margin_8dp));

            RelativeLayout.LayoutParams nextImageParams = (RelativeLayout.LayoutParams) mNextImageView.getLayoutParams();
            nextImageParams.setMarginStart(AppResource.getDimensionPixelSize(mContext, R.dimen.ux_margin_8dp));
            nextImageParams.setMarginEnd(0);
        }
        mPreImageView.setVisibility(View.GONE);
        mNextImageView.setVisibility(View.GONE);
    }


    private void disInitClosedUI() {
        mParent.removeView(mClosedRootLayout);
    }

    private final UIDragToolBar.IToolbarEventListener mToolbarEventListener = new UIDragToolBar.IToolbarEventListener() {
        int pos;

        @Override
        public void onToolbarDragEventBegin() {
        }

        @Override
        public void onToolbarDragEventEnd() {
            changPageNumberState(((UIExtensionsManager) mUiExtensionsManager).getMainFrame().isToolbarsVisible());
        }

        @Override
        public void onToolbarPositionChanged(int oldPos, int newPos) {
            pos = newPos;
            if (mPageNavBar != null) {
                mPageNavBar.setPadPos(pos);
            }

        }
    };

    private void triggerDismissMenu() {

        if (mUiExtensionsManager instanceof UIExtensionsManager) {
            ((UIExtensionsManager) mUiExtensionsManager).triggerDismissMenuEvent();
        }

        if (((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager().getCurrentAnnot() != null) {
            ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager().setCurrentAnnot(null);
        }
    }

    private void setClosedUIClickListener() {
        mClosedPageLabel.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                showInputNumDialog();
            }
        });

        mPreImageView.setOnClickListener(new View.OnClickListener() {

            @Override
            public void onClick(View v) {
                mPdfViewCtrl.gotoPrevView();
                triggerDismissMenu();
                if (mPdfViewCtrl.hasPrevView()) {
                    mPreImageView.setVisibility(View.VISIBLE);
                } else {
                    mPreImageView.setVisibility(View.GONE);
                }
                if (mPdfViewCtrl.hasNextView()) {
                    mNextImageView.setVisibility(View.VISIBLE);
                } else {
                    mNextImageView.setVisibility(View.GONE);
                }
                Message msg = new Message();
                msg.what = SHOW_RESET;
                mHandler.sendMessage(msg);
            }
        });
        mNextImageView.setOnClickListener(new View.OnClickListener() {

            @Override
            public void onClick(View v) {
                mPdfViewCtrl.gotoNextView();
                triggerDismissMenu();
                if (mPdfViewCtrl.hasPrevView()) {
                    mPreImageView.setVisibility(View.VISIBLE);
                } else {
                    mPreImageView.setVisibility(View.GONE);
                }
                if (mPdfViewCtrl.hasNextView()) {
                    mNextImageView.setVisibility(View.VISIBLE);
                } else {
                    mNextImageView.setVisibility(View.GONE);
                }
                Message msg = new Message();
                msg.what = SHOW_RESET;
                mHandler.sendMessage(msg);
            }
        });
    }

    private void showReset() {
        mHandler.removeCallbacks(runnable);
        mHandler.postDelayed(runnable, 1000 * 5);
    }

    private void showOver() {
        if (mPreImageView.getVisibility() == View.VISIBLE) {
            mPreImageView.setVisibility(View.GONE);
        }
        if (mNextImageView.getVisibility() == View.VISIBLE) {
            mNextImageView.setVisibility(View.GONE);
        }
        if (mClosedRootLayout.getVisibility() == View.VISIBLE) {
            mClosedRootLayout.setVisibility(View.GONE);
        }

        if (mPageNavBar != null && mPageNavBar.isVisible()) {
            mPageNavBar.hide();
        }
    }

    //The timer
    private static final int SHOW_OVER = 100;
    private static final int SHOW_RESET = 200;

    private static class MyHandler extends Handler {
        WeakReference<PageNavigationModule> mNavRef;

        public MyHandler(PageNavigationModule module) {
            mNavRef = new WeakReference<>(module);
        }

        @Override
        public void handleMessage(Message msg) {
            PageNavigationModule module = mNavRef.get();
            switch (msg.what) {
                case SHOW_RESET:
                    if (module != null) {
                        module.showReset();
                    }

                    break;
                case SHOW_OVER:
                    if (module != null) {
                        module.showOver();
                    }
                    break;
                default:
                    break;
            }
        }
    }

    private final Runnable runnable = new Runnable() {
        @Override
        public void run() {
            Message msg = new Message();
            msg.what = SHOW_OVER;
            mHandler.sendMessage(msg);
        }
    };

    private void onUIStatusChanged() {
        UIExtensionsManager uiExtensionsManager = ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager());
        if (uiExtensionsManager.getState() == ReadStateConfig.STATE_SEARCH || uiExtensionsManager.getPanelManager().isShowingPanel()) {
            if (mClosedRootLayout.getVisibility() != View.GONE) {
                endShow();
                mClosedRootLayout.setVisibility(View.GONE);
            }
            if (mPageNavBar != null && mPageNavBar.isVisible()) {
                mPageNavBar.hide();
            }
        } else {
            if (mPageNavBar != null) {
                mPageNavBar.onChangeReadMode();
            }
            if (mIsClosedState
                    || uiExtensionsManager.getCurrentToolHandler() != null
                    || uiExtensionsManager.getCurrentAnnotHandler() != null) {
                if (!mIsClosedState) {
                    mIsClosedState = true;
                }

                if (mPageNavBar != null) {
                    mPageNavBar.resetOnViewStyle();
                    mPageNavBar.show();
                }

                if (mClosedRootLayout.getVisibility() != View.VISIBLE && uiExtensionsManager.getMainFrame().isToolbarsVisible()) {
//                    startShow();
                    mClosedPageLabel.setEnabled(true);
//                    mClosedRootLayout.setVisibility(View.VISIBLE);
                    if (mPdfViewCtrl.hasPrevView()) {
                        mPreImageView.setVisibility(View.VISIBLE);
                    } else {
                        mPreImageView.setVisibility(View.GONE);
                    }
                    if (mPdfViewCtrl.hasNextView()) {
                        mNextImageView.setVisibility(View.VISIBLE);
                    } else {
                        mNextImageView.setVisibility(View.GONE);
                    }

                    if (mPageNavBar != null && !mPageNavBar.isFullWidth()) {
                        boolean isShowNum = false;
                        if ((mPdfViewCtrl.getPageCount() > 2 && mPdfViewCtrl.getPageLayoutMode() == PDFViewCtrl.PAGELAYOUTMODE_FACING)
                                || (mPdfViewCtrl.getPageCount() > 1 && mPdfViewCtrl.getPageLayoutMode() != PDFViewCtrl.PAGELAYOUTMODE_FACING)) {
                            isShowNum = true;
                        }
                        if (mPdfViewCtrl.getDoc() != null && isShowNum) {
                            startShow();
                            mClosedRootLayout.setVisibility(View.VISIBLE);
                        }
                    }
                }
                Message msg = new Message();
                msg.what = SHOW_RESET;
                mHandler.sendMessage(msg);
            } else {
                if (mClosedRootLayout.getVisibility() != View.GONE) {
                    endShow();
                    mClosedRootLayout.setVisibility(View.GONE);
                }
                if (mPageNavBar != null)
                    mPageNavBar.hide();
            }
        }
    }

    private final PDFViewCtrl.ILayoutEventListener layoutEventListener = new PDFViewCtrl.ILayoutEventListener() {
        @Override
        public void onLayoutModeChanged(int origin, int present) {
            mPageEventListener.onPageChanged(mPdfViewCtrl.getCurrentPage(), mPdfViewCtrl.getCurrentPage());
        }
    };

    private final IStateChangeListener mStateChangeListener = new IStateChangeListener() {
        @Override
        public void onStateChanged(int oldState, int newState) {
            if (!mIsClosedState) {
                mIsClosedState = true;
            }
            changPageNumberState(((UIExtensionsManager) mUiExtensionsManager).getMainFrame().isToolbarsVisible());
        }
    };

    private final PDFViewCtrl.IPageEventListener mPageEventListener = new PageEventListener() {
        @Override
        public void onPageChanged(int old, int current) {
            if (mPdfViewCtrl == null || mPdfViewCtrl.getDoc() == null) return;
            mClosedPageLabel.setEnabled(true);
            boolean isShowNum = false;
            if ((mPdfViewCtrl.getPageCount() > 2 && mPdfViewCtrl.getPageLayoutMode() == PDFViewCtrl.PAGELAYOUTMODE_FACING)
                    || (mPdfViewCtrl.getPageCount() > 1 && mPdfViewCtrl.getPageLayoutMode() != PDFViewCtrl.PAGELAYOUTMODE_FACING)) {
                isShowNum = true;
            }
            if (isShowNum && ((UIExtensionsManager) mUiExtensionsManager).getMainFrame().isToolbarsVisible()) {
                mClosedRootLayout.setVisibility(View.VISIBLE);
                if (mPageNavBar != null) {
                    mPageNavBar.show();
                }
            } else {
                mClosedRootLayout.setVisibility(View.GONE);
                if (mPageNavBar != null) {
                    mPageNavBar.hide();
                }
            }

            setCurrentPageIndex(current);
            //reset jumpView
            resetJumpView(old != current);
            if (mPageNavBar != null) {
                if (mPdfViewCtrl.isContinuous()) {
                    mPageNavBar.onCurrentPageChanged(current + 1);
//                    mPageNavBar.onPageScrolled(current); // index
                } else {
                    mPageNavBar.onCurrentPageChanged(current + 1); // page number
                }
            }
        }

        @Override
        public void onPageJumped() {
            resetJumpView(true);
            if (mPageNavBar != null)
                mPageNavBar.onCurrentPageChanged(mPdfViewCtrl.getCurrentPage() + 1);
//            mPageNavBar.onPageScrolled(mPdfViewCtrl.getCurrentPage());
        }

        @Override
        public void onPageVisible(int index) {

        }

        @Override
        public void onPageInvisible(int index) {

        }

        @Override
        public void onPagesInserted(boolean success, int dstIndex, int[] range) {
            if (mPageNavBar != null)
                mPageNavBar.reset();
        }

        @Override
        public void onPagesRemoved(boolean success, int[] pageIndexes) {
            AppThreadManager.getInstance().getMainThreadHandler().post(new Runnable() {
                @Override
                public void run() {
                    if (mPageNavBar != null)
                        mPageNavBar.reset();
                }
            });
        }
    };

    public void resetJumpView(boolean isSendMsg) {
        if (mPdfViewCtrl.hasPrevView()) {
            mPreImageView.setVisibility(View.VISIBLE);
        } else {
            mPreImageView.setVisibility(View.GONE);
        }
        if (mPdfViewCtrl.hasNextView()) {
            mNextImageView.setVisibility(View.VISIBLE);
        } else {
            mNextImageView.setVisibility(View.GONE);
        }
        if (mShowPanel) {
            mClosedRootLayout.setVisibility(View.GONE);
            if (mPageNavBar != null) {
                mPageNavBar.hide();
            }
            return;
        }
        UIExtensionsManager extensionsManager = (UIExtensionsManager) mUiExtensionsManager;
        if (extensionsManager.getState() != ReadStateConfig.STATE_SEARCH
                && mClosedRootLayout.getVisibility() != View.VISIBLE) {
            boolean notShowFullScreenUI = false;
            IMainFrame mainFrame = extensionsManager.getMainFrame();
            if (mainFrame.isSmallBottomToolbarEnabled()) {
                View fullScreenBottomBar = mainFrame.getContentView().findViewById(R.id.read_fullscreen_bottom_bar);
                if (fullScreenBottomBar != null)
                    notShowFullScreenUI = fullScreenBottomBar.getVisibility() != View.VISIBLE;
            }
            boolean isShowNum = false;
            if ((mPdfViewCtrl.getPageCount() > 2 && mPdfViewCtrl.getPageLayoutMode() == PDFViewCtrl.PAGELAYOUTMODE_FACING)
                    || (mPdfViewCtrl.getPageCount() > 1 && mPdfViewCtrl.getPageLayoutMode() != PDFViewCtrl.PAGELAYOUTMODE_FACING)) {
                isShowNum = true;
            }
            if ((mainFrame.isToolbarsVisible() || notShowFullScreenUI)
                    && mPdfViewCtrl.getDoc() != null && isShowNum) {
                startShow();
                mClosedRootLayout.setVisibility(View.VISIBLE);
            }
            if (mPageNavBar != null) {
                mPageNavBar.show();
            }
        }

        if (isSendMsg) {
            Message msg = new Message();
            msg.what = SHOW_RESET;
            mHandler.sendMessage(msg);
        }
    }

    public void resetJumpView() {
        if (mPdfViewCtrl.hasPrevView()) {
            mPreImageView.setVisibility(View.VISIBLE);
        } else {
            mPreImageView.setVisibility(View.GONE);
        }
        if (mPdfViewCtrl.hasNextView()) {
            mNextImageView.setVisibility(View.VISIBLE);
        } else {
            mNextImageView.setVisibility(View.GONE);
        }

        setCurrentPageIndex(mPdfViewCtrl.getCurrentPage());
        if (mPageNavBar != null && !mShowPanel) {
            mPageNavBar.resetOnViewStyle();
        }
        Message msg = new Message();
        msg.what = SHOW_RESET;
        mHandler.sendMessage(msg);
    }

    private void setCurrentPageIndex(int current) {
        if (mPdfViewCtrl.getPageLayoutMode() == PDFViewCtrl.PAGELAYOUTMODE_COVER) {
            if (current % 2 == 0) {
                current = current - 1;
            }

            if (current == -1) {
                mClosedPageLabel_Current.setText(String.format("%s", 1));
            } else if (current < mPdfViewCtrl.getPageCount() - 1) {
                mClosedPageLabel_Current.setText(String.format("%s,%s", current + 1, current + 2));
            } else {
                mClosedPageLabel_Current.setText(String.format("%s", current + 1));
            }
        } else if (mPdfViewCtrl.getPageLayoutMode() == PDFViewCtrl.PAGELAYOUTMODE_FACING) {
            if (current % 2 != 0) {
                current = current - 1;
            }

            if (current < mPdfViewCtrl.getPageCount() - 1 || mPdfViewCtrl.getPageCount() % 2 == 0) {
                mClosedPageLabel_Current.setText(String.format("%s,%s", current + 1, current + 2));
            } else {
                mClosedPageLabel_Current.setText(String.format("%s", current + 1));
            }
        } else {
            mClosedPageLabel_Current.setText(String.format("%s", current + 1));
        }

        mClosedPageLabel_Total.setText(String.format("/%s", mPdfViewCtrl.getPageCount()));

        if (LogicalPageHelper.getInstance().isSupportLogicalPage()) {
            setLogicalPageNumbers(current);
        }
    }

    private void setLogicalPageNumbers(int current) {
        String title = LogicalPageHelper.getInstance().getLogicalPageNumber(current, mPdfViewCtrl.getDoc());
        mClosedPageLabel_Current.setText(String.format("%s(%s", title, mClosedPageLabel_Current.getText()));
        mClosedPageLabel_Total.setText(String.format("%s)", mClosedPageLabel_Total.getText()));
    }

    private void gotoLogicalPageNumbers(String pageLabel) {
        try {
            PDFDoc doc = mPdfViewCtrl.getDoc();
            int pageCount = doc.getPageCount();
            PageLabels pageLabels = new PageLabels(doc);
            for (int i = 0; i < pageCount; i++) {
                String labelTitle = LogicalPageHelper.getInstance().getLogicalPageNumber(i, pageLabels);
                if (labelTitle.equals(pageLabel)) {
                    mPdfViewCtrl.gotoPage(i, 0, 0);
                    return;
                }
            }

            if (AppDmUtil.isNumer(pageLabel)) {
                int pageIndex = Integer.parseInt(pageLabel) - 1;
                if (pageIndex >= 0) {
                    mPdfViewCtrl.gotoPage(pageIndex, 0, 0);
                    return;
                }
            }

            showWarningDialog(pageLabel);
        } catch (PDFException e) {
            e.printStackTrace();
        }
    }

    private void showWarningDialog(String pageLabel) {
        final UITextEditDialog dialog = new UITextEditDialog(mPdfViewCtrl.getAttachedActivity(), UIDialog.NO_INPUT);
        dialog.setTitle(AppResource.getString(mContext, R.string.fx_string_warning));

        dialog.getPromptTextView().setText(AppResource.getString(mContext, R.string.fx_gotopage_page_label_not_exist, pageLabel));
        dialog.getCancelButton().setVisibility(View.GONE);
        dialog.getOKButton().setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                dialog.dismiss();
            }
        });
        dialog.show();
    }

    private void startShow() {
        mClosedRootLayout.startAnimation(AnimationUtils.loadAnimation(mContext, R.anim.view_anim_visible_show));
    }

    private void endShow() {
        mClosedRootLayout.startAnimation(AnimationUtils.loadAnimation(mContext, R.anim.view_anim_visible_hide));
    }

    private final View.OnKeyListener mOnKeyKListener = new View.OnKeyListener() {
        @Override
        public boolean onKey(View v, int keyCode, KeyEvent event) {
            if (keyCode == KeyEvent.KEYCODE_BACK && event.getRepeatCount() == 0) {
                if (!mIsClosedState) {
                    mIsClosedState = true;
                    if (mLastToolHandler != null)
                        ((UIExtensionsManager) mUiExtensionsManager).setCurrentToolHandler(mLastToolHandler);
                    else
                        ((UIExtensionsManager) mUiExtensionsManager).changeState(mLastState);
//                    onUIStatusChanged();
                    return true;
                }
            }
            return false;
        }
    };

    private final PDFViewCtrl.IDocEventListener mDocumentEventListener = new DocEventListener() {
        @Override
        public void onDocWillOpen() {
            if (mPageNavBar != null) {
                mPageNavBar.destroy();
                mPageNavBar = null;
            }
        }

        @Override
        public void onDocOpened(PDFDoc pdfDoc, int errCode) {
            if (pdfDoc == null || errCode != Constants.e_ErrSuccess) {
                return;
            }

            try {
                if (((UIExtensionsManager) mUiExtensionsManager).isUseLogicalPageNumbers()) {
                    if (!mPdfViewCtrl.isDynamicXFA()) {
                        PageLabels pageLabels = new PageLabels(mPdfViewCtrl.getDoc());
                        LogicalPageHelper.getInstance().setSupportLogicalPage(pageLabels.hasPageLabel(0));
                        pageLabels.delete();
                    }
                }
            } catch (PDFException e) {
                e.printStackTrace();
            }

            mPageNavBar = new PageNavigationBar(mContext, mPdfViewCtrl, mClosedRootLayout);
            mPageEventListener.onPageChanged(mPdfViewCtrl.getCurrentPage(), mPdfViewCtrl.getCurrentPage());
            onUIStatusChanged();
        }

        @Override
        public void onDocWillClose(PDFDoc pdfDoc) {
            LogicalPageHelper.getInstance().release();
        }

        @Override
        public void onDocClosed(PDFDoc pdfDoc, int i) {
            if (mPageNavBar != null) {
                mPageNavBar.destroy();
                mPageNavBar = null;
            }
            mHandler.removeCallbacksAndMessages(null);
        }

    };

    public void changPageNumberState(boolean isToolbarsVisible) {
        if (!isToolbarsVisible) {
            if (mClosedRootLayout.getVisibility() != View.GONE) {
                endShow();
                mClosedRootLayout.setVisibility(View.GONE);
            }
            if (mPageNavBar != null) {
                mPageNavBar.resetOnViewStyle();
                mPageNavBar.hide();
            }

        } else {
            onUIStatusChanged();
        }
    }

    private final ILifecycleEventListener mLifecycleEventListener = new LifecycleEventListener() {

        @Override
        public void onResume(Activity act) {
            changPageNumberState(((UIExtensionsManager) mUiExtensionsManager).getMainFrame().isToolbarsVisible());
        }
    };

    private final ILayoutChangeListener mLayoutChangeListener = new ILayoutChangeListener() {
        @Override
        public void onLayoutChange(View v, int newWidth, int newHeight, int oldWidth, int oldHeight) {
            if (mPageNavBar != null) {
                int bottom;
                int pageNavBarHeight = 0;
                int paddingLeft = AppDisplay.dp2px(16);

                if (mPageNavBar.mCurPosId <= 1) {
                    pageNavBarHeight = mPageNavBar.getPageBottomHeight();
                }

                if (AppDisplay.isPad()) {
                    bottom = pageNavBarHeight + AppDisplay.dp2px(16);
                    paddingLeft = AppDisplay.dp2px(44);
                } else {
                    UIExtensionsManager uiExtensionsManager = (UIExtensionsManager) mUiExtensionsManager;
                    int state = uiExtensionsManager.getState();

                    bottom = AppResource.getDimensionPixelSize(mContext, R.dimen.ui_bottombar_height) + pageNavBarHeight + AppDisplay.dp2px(16);
                    if (!uiExtensionsManager.getMainFrame().isToolbarsVisible()) {
                        bottom = pageNavBarHeight + AppDisplay.dp2px(16);
                    } else {
                        boolean readState = state == ReadStateConfig.STATE_REFLOW
                                || state == ReadStateConfig.STATE_PANZOOM
                                || state == ReadStateConfig.STATE_TTS
                                || state == ReadStateConfig.STATE_AUTOFLIP
                                || state == ReadStateConfig.STATE_REDACT;
                        if (!uiExtensionsManager.getMainFrame().isShowBottomToolbar() && !readState) {
                            bottom -= AppResource.getDimensionPixelSize(mContext, R.dimen.ui_bottombar_height);
                        }
                    }

                    if (state == ReadStateConfig.STATE_COMPARE) {
                        bottom = pageNavBarHeight + AppDisplay.dp2px(16);
                    }
                }

                RelativeLayout.LayoutParams closedLP = (RelativeLayout.LayoutParams) mClosedRootLayout.getLayoutParams();
                if (closedLP.leftMargin != paddingLeft || closedLP.bottomMargin != bottom) {
                    closedLP.width = RelativeLayout.LayoutParams.WRAP_CONTENT;
                    closedLP.height = RelativeLayout.LayoutParams.WRAP_CONTENT;
                    closedLP.addRule(RelativeLayout.ALIGN_PARENT_BOTTOM);
                    closedLP.setMargins(paddingLeft, 0, 0, bottom);
                    mClosedRootLayout.setLayoutParams(closedLP);
                }
            }

            if (mPdfViewCtrl != null && mPdfViewCtrl.getDoc() != null) {
                boolean isShowNum = false;
                if ((mPdfViewCtrl.getPageCount() > 2 && mPdfViewCtrl.getPageLayoutMode() == PDFViewCtrl.PAGELAYOUTMODE_FACING)
                        || (mPdfViewCtrl.getPageCount() > 1 && mPdfViewCtrl.getPageLayoutMode() != PDFViewCtrl.PAGELAYOUTMODE_FACING)) {
                    isShowNum = true;
                }
                if (mPageNavBar != null && !isShowNum) {
                    mPageNavBar.hide();
                }
            }

            if (oldWidth != newWidth || oldHeight != newHeight) {
                mClosedPageLabel.requestLayout();
                if (mPdfViewCtrl != null && mPdfViewCtrl.getDoc() != null) {
                    resetJumpView();
                    if (mClosedRootLayout.getVisibility() == View.VISIBLE) {
                        if (mPageNavBar != null) {
                            mPageNavBar.show();
                        }
                    } else {
                        if (mPageNavBar != null) {
                            mPageNavBar.hide();
                        }
                    }
                }
            }

        }
    };
    private final IPanelManager.OnPanelEventListener mPanelEventListener = new IPanelManager.OnPanelEventListener() {
        @Override
        public void onPanelOpened() {
            if (mPageNavBar != null) {
                mPageNavBar.hide();
                mShowPanel = true;
            }
        }

        @Override
        public void onPanelClosed() {
            if (mPageNavBar != null && mShowPanel && ((UIExtensionsManager) mUiExtensionsManager).getMainFrame().isToolbarsVisible()) {
                boolean isShowNum = false;
                if ((mPdfViewCtrl.getPageCount() > 2 && mPdfViewCtrl.getPageLayoutMode() == PDFViewCtrl.PAGELAYOUTMODE_FACING)
                        || (mPdfViewCtrl.getPageCount() > 1 && mPdfViewCtrl.getPageLayoutMode() != PDFViewCtrl.PAGELAYOUTMODE_FACING)) {
                    isShowNum = true;
                }
                if (mPdfViewCtrl.getDoc() != null && isShowNum) {
                    mClosedRootLayout.setVisibility(View.VISIBLE);
                    mPageNavBar.show();
                    Message msg = new Message();
                    msg.what = SHOW_RESET;
                    mHandler.sendMessage(msg);
                }
            }
            mShowPanel = false;
        }
    };

    public void showInputNumDialog() {
        final UIExtensionsManager uiExtensionsManager = (UIExtensionsManager) mUiExtensionsManager;
        uiExtensionsManager.onUIInteractElementClicked(IUIInteractionEventListener.Reading_GotoPage);
        final UITextEditDialog dialog = new UITextEditDialog(uiExtensionsManager.getAttachedActivity());
        dialog.setTitle(AppResource.getString(mContext, R.string.rd_gotopage_toolbar_go));
        final TextView promptText = dialog.getPromptTextView();
        final EditText editText = dialog.getInputEditText();
        editText.requestFocus();
        final View okButton = dialog.getOKButton();

        String entet_str;
        if (LogicalPageHelper.getInstance().isSupportLogicalPage()) {
            String pageLabel = LogicalPageHelper.getInstance().getLogicalPageNumber(mPdfViewCtrl.getCurrentPage(), mPdfViewCtrl.getDoc());
            entet_str = AppResource.getString(mContext, R.string.rv_gotopage_error_toast) + " " +
                    pageLabel + "(" + (mPdfViewCtrl.getCurrentPage() + 1) + "/" + mPdfViewCtrl.getPageCount() + ")";
            editText.setInputType(InputType.TYPE_CLASS_TEXT);
        } else {
            entet_str = AppResource.getString(mContext, R.string.rv_gotopage_error_toast) +
                    " (" + (mPdfViewCtrl.getCurrentPage() + 1) + "/" + mPdfViewCtrl.getPageCount() + ")";
            editText.setInputType(InputType.TYPE_CLASS_NUMBER);
        }
        editText.addTextChangedListener(new TextWatcher() {
            @Override
            public void beforeTextChanged(CharSequence charSequence, int i, int i1, int i2) {

            }

            @Override
            public void onTextChanged(CharSequence charSequence, int i, int i1, int i2) {
                if (!editText.getText().toString().trim().equals("")) {
                    int inputType = editText.getInputType();
                    String content = editText.getText().toString();
                    if (inputType == InputType.TYPE_CLASS_NUMBER) {
                        int number = Integer.parseInt(content);
                        if (0 < number && number <= mPdfViewCtrl.getPageCount()) {
                            okButton.setEnabled(true);
                        } else {
                            editText.selectAll();
                            okButton.setEnabled(false);
                        }
                    } else {
                        okButton.setEnabled(!AppUtil.isEmpty(content));
                    }
                }
            }

            @Override
            public void afterTextChanged(Editable editable) {
                if (editText.getText().toString().trim().equals("")) {
                    okButton.setEnabled(false);
                }
            }
        });

        okButton.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View view) {
                if (TextUtils.isEmpty(editText.getText().toString())) {
                    return;
                }
                if (LogicalPageHelper.getInstance().isSupportLogicalPage()) {
                    gotoLogicalPageNumbers(editText.getText().toString());
                } else {
                    int number = Integer.parseInt(editText.getText().toString());
                    mPdfViewCtrl.gotoPage(number - 1, 0, 0);
                }
                dialog.dismiss();
            }
        });

        promptText.setText(entet_str);
        dialog.getOKButton().setEnabled(false);
        dialog.show();
//        AppKeyboardUtil.showSoftInput(editText);
        Timer timer = new Timer();
        timer.schedule(new TimerTask() {
            @Override
            public void run() {
                AppKeyboardUtil.showInputMethodWindow(mContext, editText);
            }
        }, 120);
    }

    public void setPageNavigationBarVisible(boolean visible) {
        if (mPageNavBar == null) return;

        mPageNavBar.setVisible(visible);
        if (visible) {
            changPageNumberState(((UIExtensionsManager) mUiExtensionsManager).getMainFrame().isToolbarsVisible());
        } else {
            mPageNavBar.hide();
        }
    }

}
