/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.modules.panel;

import android.app.Activity;
import android.content.Context;
import android.os.Build;
import android.view.View;
import android.view.ViewGroup;
import android.view.animation.Animation;
import android.view.animation.AnimationUtils;
import android.widget.RelativeLayout;

import com.foxit.sdk.PDFViewCtrl;
import com.foxit.uiextensions.IThemeEventListener;
import com.foxit.uiextensions.R;
import com.foxit.uiextensions.UIExtensionsManager;
import com.foxit.uiextensions.controls.panel.PanelHost;
import com.foxit.uiextensions.controls.panel.PanelSpec;
import com.foxit.uiextensions.controls.panel.impl.PanelHostImpl;
import com.foxit.uiextensions.pdfreader.ILayoutChangeListener;
import com.foxit.uiextensions.utils.ActManager;
import com.foxit.uiextensions.utils.AppDisplay;
import com.foxit.uiextensions.utils.AppResource;
import com.foxit.uiextensions.utils.AppUtil;
import com.foxit.uiextensions.utils.LayoutConfig;

import java.util.ArrayList;
import java.util.List;

/** The panel management, which manage all kinds of panels including bookmark, outline, annotations, file attachment... */
public class PanelManager implements IPanelManager {
    private PanelHostImpl mPanel;

    private List<OnPanelEventListener> mPanelEventListeners;
    private UIExtensionsManager mUIExtensionsManager;

    private View mRightShadow;
    private View mTopDivider;
    private ViewGroup mParent;
    private ViewGroup mPanelContainer;
    private Context mContext;
    private Context mPanelContext;

    public PanelManager(Context context,
                        PDFViewCtrl.UIExtensionsManager uiExtensionsManager,
                        ViewGroup parent,
                        ViewGroup panelContainer) {
        mPanelEventListeners = new ArrayList<>();
        mContext = context;
        mParent = parent;
        mPanelContainer = panelContainer;
        mUIExtensionsManager = (UIExtensionsManager) uiExtensionsManager;

        mUIExtensionsManager.registerLayoutChangeListener(new ILayoutChangeListener() {
            @Override
            public void onLayoutChange(View v, int newWidth, int newHeight, int oldWidth, int oldHeight) {
                if (isShowingPanel()) {
                    if (oldWidth != newWidth || oldHeight != newHeight) {
                        updatePanelLayout(newWidth, newHeight);
                    }
                }
            }
        });

        mUIExtensionsManager.registerThemeEventListener(new IThemeEventListener() {
            @Override
            public void onThemeColorChanged(String type, int color) {
                if (mPanel != null)
                    mPanel.onThemeChange(type, color);
            }
        });

        mPanelContext = context;
        if (Build.VERSION.SDK_INT <= 19) {
            Activity activity = ActManager.getInstance().getCurrentActivity();
            if (activity == null)
                activity = mUIExtensionsManager.getMainFrame().getAttachedActivity();

            if (activity != null)
                mPanelContext = activity;
        }
    }

    private void initPanel() {
        if (mPanel != null) return;
        mPanel = new PanelHostImpl(mPanelContext, mUIExtensionsManager, new PanelHost.ICloseDefaultPanelCallback() {
            @Override
            public void closeDefaultPanel(View v) {
                if (isShowingPanel()) {
                    hidePanel();
                    onPanelClosed();
                }
            }
        });

        if (AppDisplay.isPad()) {
            mRightShadow = new View(mContext);
            mRightShadow.setId(R.id.rd_panel_right_shadow);
            RelativeLayout.LayoutParams rightParams = new RelativeLayout.LayoutParams(
                    AppDisplay.dp2px(3), RelativeLayout.LayoutParams.MATCH_PARENT);
            rightParams.addRule(RelativeLayout.ALIGN_PARENT_RIGHT);
            mRightShadow.setBackgroundResource(R.drawable.panel_right_shadow);
            mRightShadow.setLayoutParams(rightParams);
            mPanelContainer.addView(mRightShadow);

            RelativeLayout.LayoutParams contentParams = new RelativeLayout.LayoutParams(
                    ViewGroup.LayoutParams.MATCH_PARENT, ViewGroup.LayoutParams.MATCH_PARENT);
            contentParams.addRule(RelativeLayout.LEFT_OF, mRightShadow.getId());
            mPanelContainer.addView(mPanel.getContentView(), contentParams);

            mTopDivider = new View(mContext);
            RelativeLayout.LayoutParams topParams = new RelativeLayout.LayoutParams(RelativeLayout.LayoutParams.MATCH_PARENT,
                    AppDisplay.dp2px(1));
            topParams.addRule(RelativeLayout.ALIGN_PARENT_TOP);
            topParams.addRule(RelativeLayout.LEFT_OF, mRightShadow.getId());

            mTopDivider.setBackgroundColor(AppResource.getColor(mContext, R.color.p1));
            mTopDivider.setLayoutParams(topParams);
            mPanelContainer.addView(mTopDivider);
        } else {
            mPanelContainer.addView(mPanel.getContentView(), new RelativeLayout.LayoutParams(
                    ViewGroup.LayoutParams.MATCH_PARENT, ViewGroup.LayoutParams.MATCH_PARENT));
        }
    }

    private ArrayList<PanelSpec> mSpecsToAdd = new ArrayList<>();
    private ArrayList<Integer> mSpecTypesToRemove = new ArrayList<>();

    @Override
    public void addPanel(PanelSpec panelSpec) {
        if (panelSpec == null) return;
        if (mPanel == null){
            mSpecsToAdd.add(panelSpec);
            return;
        }
        if (mSpecTypesToRemove.contains(panelSpec.getPanelType())){
            return;
        }
        if (!mSpecsToAdd.isEmpty()){
            for (PanelSpec spec : mSpecsToAdd) {
                mPanel.addSpec(spec);
            }
            mSpecsToAdd.clear();
        }
        mPanel.addSpec(panelSpec);
    }

    @Override
    public void addPanel(int index, PanelSpec panelSpec) {
        if (panelSpec == null) return;
        if (mPanel == null){
            if (index == -1 || index > mSpecsToAdd.size()) {
                mSpecsToAdd.add(panelSpec);
            } else {
                mSpecsToAdd.add(index, panelSpec);
            }
            mSpecsToAdd.add(index,panelSpec);
            return;
        }
        if (mSpecTypesToRemove.contains(panelSpec.getPanelType())){
            return;
        }
        if (!mSpecsToAdd.isEmpty()){
            for (PanelSpec spec : mSpecsToAdd) {
                mPanel.addSpec(spec);
            }
            mSpecsToAdd.clear();
        }
        mPanel.addSpec(index, panelSpec);
    }

    @Override
    public void removePanel(PanelSpec panelSpec) {
        if (panelSpec == null) return;
        if (mPanel == null){
            removePanel(panelSpec.getPanelType());
        }else {
            mPanel.removeSpec(panelSpec);
        }
    }

    @Override
    public void removePanel(int panelType) {
        if (mPanel == null) {
            if (!mSpecTypesToRemove.contains(panelType)){
                mSpecTypesToRemove.add(panelType);
            }
            return;
        }
        mPanel.removeSpec(panelType);
    }

    @Override
    public void registerPanelEventListener(OnPanelEventListener listener) {
        if (mPanelEventListeners.contains(listener)) return;
        mPanelEventListeners.add(listener);
    }

    @Override
    public void unregisterPanelEventListener(OnPanelEventListener listener) {
        if (mPanelEventListeners.contains(listener)) {
            mPanelEventListeners.remove(listener);
        }
    }

    private void onPanelOpened() {
        for (OnPanelEventListener listener : mPanelEventListeners) {
            listener.onPanelOpened();
        }
    }

    private void onPanelClosed() {
        for (OnPanelEventListener listener : mPanelEventListeners) {
            listener.onPanelClosed();
        }
    }

    @Override
    public PanelHost getPanelHost() {
        if (mPanel == null)
            initPanel();
        return mPanel;
    }

    @Override
    public void showPanel() {
        if (mPanel == null || mPanel.getCurrentSpec() == null) {
            showPanel(PanelSpec.OUTLINE);
        } else {
            showPanel(mPanel.getCurrentSpec().getPanelType());
        }
    }

    @Override
    public void showPanel(int panelType) {
        initPanel();
        showPanelLayout();
        mPanel.setCurrentSpec(panelType);
    }

    @Override
    public boolean isShowingPanel() {
        return mPanelContainer != null && mPanelContainer.getVisibility() == View.VISIBLE;
    }

    @Override
    public void hidePanel() {
        if (mPanel == null) return;
        if (mPanelContainer.getVisibility() == View.VISIBLE) {
            Animation animation = AnimationUtils.loadAnimation(mContext, R.anim.view_anim_ltor_hide);
            mPanelContainer.setAnimation(animation);
            mPanelContainer.setVisibility(View.INVISIBLE);
        }
        if (mPanel.getCurrentSpec() != null)
            mPanel.getCurrentSpec().onDeactivated();
        onPanelClosed();
    }

    private void showPanelLayout() {
        int width = AppDisplay.getActivityWidth();
        int height = AppDisplay.getActivityHeight();
        if (AppDisplay.isPad()) {
            float scale = LayoutConfig.RD_PANEL_WIDTH_SCALE_V;
            if (width > height)
                scale = LayoutConfig.RD_PANEL_WIDTH_SCALE_H;
            width = (int) (width * scale);
        }
        RelativeLayout.LayoutParams layoutParams = (RelativeLayout.LayoutParams) mPanelContainer.getLayoutParams();
        layoutParams.width = width;
        layoutParams.height = RelativeLayout.LayoutParams.MATCH_PARENT;
        mPanelContainer.setLayoutParams(layoutParams);

        Animation animation = AnimationUtils.loadAnimation(mContext, R.anim.view_anim_ltor_show);
        mPanelContainer.setAnimation(animation);
        mPanelContainer.setVisibility(View.VISIBLE);

        onPanelOpened();
        mSpecTypesToRemove.clear();
        mSpecsToAdd.clear();
    }

    private void updatePanelLayout(int width, int height) {
        if (AppDisplay.isPad()) {
            width = AppDisplay.getActivityWidth();
            height = AppDisplay.getActivityHeight();
            float scale = LayoutConfig.RD_PANEL_WIDTH_SCALE_V;
            if (width > height) {
                scale = LayoutConfig.RD_PANEL_WIDTH_SCALE_H;
            }
            width = (int) (width * scale);
        }
        RelativeLayout.LayoutParams layoutParams = (RelativeLayout.LayoutParams) mPanelContainer.getLayoutParams();
        layoutParams.width = width;
        layoutParams.height = RelativeLayout.LayoutParams.MATCH_PARENT;
        mPanelContainer.setLayoutParams(layoutParams);
    }

    public void updateTheme() {
        if (mPanel == null) return;

        if (AppDisplay.isPad()) {
            mRightShadow.setBackground(AppResource.getDrawable(mContext, R.drawable.panel_right_shadow));
            mTopDivider.setBackgroundColor(AppResource.getColor(mContext, R.color.p1));
        }
    }

}
