/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.modules.panel;


import android.content.Context;
import android.os.Handler;
import android.text.Editable;
import android.text.TextWatcher;
import android.view.CollapsibleActionView;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.view.inputmethod.EditorInfo;
import android.widget.EditText;
import android.widget.ImageView;
import android.widget.LinearLayout;
import android.widget.TextView;

import com.foxit.uiextensions.R;
import com.foxit.uiextensions.theme.ThemeUtil;
import com.foxit.uiextensions.utils.AppDisplay;
import com.foxit.uiextensions.utils.AppKeyboardUtil;
import com.foxit.uiextensions.utils.AppResource;
import com.foxit.uiextensions.utils.AppUtil;

public class PanelSearchView extends LinearLayout implements CollapsibleActionView {
    private static final int INTERVAL_TIME = 300;

    private Context mContext;
    private OnSearchEventListener mSearchEventListener;
    private Handler mSearchHandler;

    private View mSearchContainer;
    private LinearLayout mSearchFrame;
    private View mSearchbarDivider;
    private TextView mCancelTv;
    private ImageView mRightIv;
    private ImageView mClearSearchIv;
    private EditText mSearchEt;

    public PanelSearchView(Context context) {
        super(context);
        mContext = context;
        mSearchHandler = new Handler();

        final View searchView = LayoutInflater.from(mContext).inflate(R.layout.rd_search_view, this);
        mRightIv = searchView.findViewById(R.id.panel_search_right_view);
        mRightIv.setVisibility(GONE);
        mSearchContainer = searchView.findViewById(R.id.panel_search_bar_container);
        mSearchFrame = searchView.findViewById(R.id.panel_search_view_ll);

        LinearLayout searchContainer = searchView.findViewById(R.id.panel_search_view_container);
        searchContainer.setOnClickListener(new OnClickListener() {
            @Override
            public void onClick(View v) {
                if (!mSearchEt.hasFocus()) {
                    mSearchEt.requestFocus();
                    AppKeyboardUtil.showInputMethodWindow(mContext, mSearchEt);
                }
            }
        });
        mSearchbarDivider = searchView.findViewById(R.id.panel_search_bar_divider);
        mSearchbarDivider.setVisibility(View.GONE);
        mCancelTv = searchView.findViewById(R.id.panel_search_view_cancel);
        mCancelTv.setOnClickListener(new OnClickListener() {
            @Override
            public void onClick(View v) {
                if (mSearchEventListener != null)
                    mSearchEventListener.onCancel();
            }
        });
        mCancelTv.setTextColor(ThemeUtil.getPrimaryTextColor(mContext));
        mCancelTv.setVisibility(View.GONE);

        mSearchEt = searchView.findViewById(R.id.panel_search_view_edit);
        if (AppDisplay.isPad()) { // SDKRD-9313
            mSearchEt.setImeOptions(EditorInfo.IME_FLAG_NO_EXTRACT_UI);
        }
        mSearchEt.setHint(AppResource.getString(mContext, R.string.fx_string_search));
        mSearchEt.addTextChangedListener(new TextWatcher() {
            @Override
            public void beforeTextChanged(CharSequence s, int start, int count, int after) {
            }

            @Override
            public void onTextChanged(CharSequence s, int start, int before, int count) {
                if (AppUtil.isEmpty(s))
                    mClearSearchIv.setVisibility(View.GONE);
                else
                    mClearSearchIv.setVisibility(VISIBLE);

                if (mSearchRunnable != null) {
                    mSearchHandler.removeCallbacks(mSearchRunnable);
                    mSearchHandler.postDelayed(mSearchRunnable, INTERVAL_TIME);
                }
            }

            @Override
            public void afterTextChanged(Editable s) {
            }
        });
        mSearchEt.setOnFocusChangeListener(new OnFocusChangeListener() {
            @Override
            public void onFocusChange(View v, boolean hasFocus) {
                if (mSearchEventListener != null) {
                    mSearchEventListener.onFocusChange(v, hasFocus);
                }

                if (!hasFocus)
                    AppKeyboardUtil.hideInputMethodWindow(mContext, mSearchEt);
            }
        });

        mClearSearchIv = searchView.findViewById(R.id.panel_search_view_clear);
        mClearSearchIv.setOnClickListener(new OnClickListener() {
            @Override
            public void onClick(View v) {
                mSearchEt.setText("");
            }
        });
    }

    private Runnable mSearchRunnable = new Runnable() {
        @Override
        public void run() {
            if (mSearchEventListener != null)
                mSearchEventListener.onQueryTextChange(mSearchEt.getText().toString());
        }
    };

    @Override
    public void onActionViewExpanded() {
        LayoutParams layoutParams = new LayoutParams(ViewGroup.LayoutParams.MATCH_PARENT, ViewGroup.LayoutParams.WRAP_CONTENT);
        mSearchEt.setLayoutParams(layoutParams);
        mSearchEt.setHint(AppResource.getString(mContext, R.string.rv_search_hint));

        mCancelTv.setVisibility(View.VISIBLE);
        mSearchFrame.setBackgroundResource(R.drawable.rd_search_input_bg);
        mSearchbarDivider.setVisibility(View.VISIBLE);
        mSearchContainer.setBackgroundColor(AppResource.getColor(mContext, R.color.b2));
    }

    @Override
    public void onActionViewCollapsed() {
        if (!AppUtil.isEmpty(mSearchEt.getText()))
            mSearchEt.setText("");
        if (mSearchEt.hasFocus())
            mSearchEt.clearFocus();
        LayoutParams layoutParams = new LayoutParams(ViewGroup.LayoutParams.WRAP_CONTENT, ViewGroup.LayoutParams.WRAP_CONTENT);
        mSearchEt.setLayoutParams(layoutParams);
        mSearchEt.setHint(AppResource.getString(mContext, R.string.fx_string_search));

        mCancelTv.setVisibility(View.GONE);
        mSearchFrame.setBackgroundResource(R.drawable.rd_search_normal_bg);
        mSearchbarDivider.setVisibility(View.GONE);
        mSearchContainer.setBackgroundColor(AppResource.getColor(mContext, R.color.b1));
    }

    public void setSearchEventListener(OnSearchEventListener searchEventListener) {
        this.mSearchEventListener = searchEventListener;
    }

    public interface OnSearchEventListener {
        void onQueryTextChange(String newText);

        void onFocusChange(View v, boolean hasFocus);

        void onCancel();
    }

    public String getSearchText() {
        return mSearchEt.getText().toString();
    }

    public EditText getEditText() {
        return mSearchEt;
    }

    public ImageView getRightImageView(){
        return mRightIv;
    }

    public void onThemeColorChanged() {
        mCancelTv.setTextColor(ThemeUtil.getPrimaryTextColor(mContext));
        mSearchContainer.setBackgroundColor(AppResource.getColor(mContext,R.color.b1));
        mSearchFrame.setBackground(AppResource.getDrawable(mContext,R.drawable.rd_search_normal_bg));
        mSearchEt.setHintTextColor(AppResource.getColor(mContext,R.color.t3));
        mSearchEt.setTextColor(AppResource.getColor(mContext,R.color.t4));
        ThemeUtil.setTintList(mClearSearchIv,ThemeUtil.getPrimaryIconColor(mContext));
        ThemeUtil.setTintList(mRightIv,ThemeUtil.getPrimaryIconColor(mContext));
        mSearchbarDivider.setBackgroundColor(AppResource.getColor(mContext,R.color.p1));
    }

}
