/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.modules.panel.annot;

import android.app.Activity;
import android.content.Context;
import android.content.res.ColorStateList;
import android.content.res.Configuration;
import android.view.KeyEvent;
import android.view.View;
import android.widget.ImageView;
import android.widget.RelativeLayout;
import android.widget.TextView;

import com.foxit.sdk.PDFViewCtrl;
import com.foxit.sdk.common.Constants;
import com.foxit.sdk.pdf.PDFDoc;
import com.foxit.uiextensions.IInteractionEventListener;
import com.foxit.uiextensions.IThemeEventListener;
import com.foxit.uiextensions.Module;
import com.foxit.uiextensions.R;
import com.foxit.uiextensions.UIExtensionsManager;
import com.foxit.uiextensions.browser.PullRecyclerView;
import com.foxit.uiextensions.browser.behavior.NestedHeaderBehavior;
import com.foxit.uiextensions.controls.dialog.MatchDialog;
import com.foxit.uiextensions.controls.dialog.UIDialog;
import com.foxit.uiextensions.controls.dialog.UITextEditDialog;
import com.foxit.uiextensions.controls.panel.PanelSpec;
import com.foxit.uiextensions.event.DocEventListener;
import com.foxit.uiextensions.event.PageEventListener;
import com.foxit.uiextensions.modules.panel.IPanelManager;
import com.foxit.uiextensions.modules.panel.PanelSearchView;
import com.foxit.uiextensions.pdfreader.config.ReadStateConfig;
import com.foxit.uiextensions.theme.ThemeConfig;
import com.foxit.uiextensions.theme.ThemeUtil;
import com.foxit.uiextensions.utils.AppDevice;
import com.foxit.uiextensions.utils.AppDisplay;
import com.foxit.uiextensions.utils.AppResource;
import com.foxit.uiextensions.utils.AppUtil;
import com.foxit.uiextensions.utils.thread.AppThreadManager;

import java.util.ArrayList;
import java.util.List;

import androidx.annotation.NonNull;
import androidx.recyclerview.widget.DefaultItemAnimator;
import androidx.recyclerview.widget.LinearLayoutManager;
import androidx.recyclerview.widget.RecyclerView;

/**
 * All the annotations on the document would be listed on the annotation panel, so they could be easily reviewed, edited or deleted.
 */
public class AnnotPanelModule implements Module, PanelSpec {
    private final PDFViewCtrl mPdfViewCtrl;
    private final Context mContext;
    private UITextEditDialog mDialog;
    private AnnotFilterDialog mAnnotFilterDialog;
    private final UIExtensionsManager mUiExtensionsManager;
    private AnnotPanel mAnnotPanel;
    private AnnotAdapter mAnnotAdapter;
    private LinearLayoutManager mLayoutManager;

    private final List<String> mFilterTypes = new ArrayList<>();
    private List<AnnotNode> mNodeList = new ArrayList<>();

    private View mBottomFilterContainer;
    private TextView mBottomFilterResultTv;
    private ImageView mBottomFilterResultIv;

    private View mContentView;
    private View mBottomView;
    private ImageView mDeleteIv;
    private TextView mSelectedAllTv;
    private PanelSearchView mSearchView;
    private TextView mEditViewTv;
    private TextView mCloseTv;
    private View mTopBarView;
    private ImageView mNoInfoIv;
    private TextView mNoInfoTv;
    private TextView mLoadingTv;
    private PullRecyclerView mPullListView;
    private TextView mTopTitle;
    private int mPausedPageIndex = 0;
    private RelativeLayout mTopContainer;

    public AnnotPanelModule(Context context, PDFViewCtrl pdfViewCtrl, PDFViewCtrl.UIExtensionsManager uiExtensionsManager) {
        if (context == null || pdfViewCtrl == null) {
            throw new NullPointerException();
        }
        mContext = context.getApplicationContext();
        mPdfViewCtrl = pdfViewCtrl;
        mUiExtensionsManager = (UIExtensionsManager) uiExtensionsManager;
    }

    @Override
    public boolean loadModule() {
        mAnnotPanel = new AnnotPanel(this, mContext, mPdfViewCtrl);

        mUiExtensionsManager.registerThemeEventListener(mThemeEventListener);
        mUiExtensionsManager.registerInteractionListener(mInteractionEventListener);
        mUiExtensionsManager.registerConfigurationChangedListener(mConfigurationChangedListener);
        mUiExtensionsManager.registerModule(this);
        mUiExtensionsManager.getDocumentManager().registerAnnotEventListener(mAnnotPanel.getAnnotEventListener());
        mUiExtensionsManager.getDocumentManager().registerFlattenEventListener(mAnnotPanel.getFlattenEventListener());
        mUiExtensionsManager.getDocumentManager().registerRedactionEventListener(mAnnotPanel.getRedactionEventListener());
        mUiExtensionsManager.getDocumentManager().registerImportedAnnotsEventListener(mAnnotPanel.getImportAnnotsListener());
        mUiExtensionsManager.getDocumentManager().registerGroupEventListener(mAnnotPanel.getGroupEventListener());
        mPdfViewCtrl.registerDocEventListener(mDocEventListener);
        mPdfViewCtrl.registerPageEventListener(mPageEventListener);
        mUiExtensionsManager.getPanelManager().registerPanelEventListener(mPanelEventListener);
        return true;
    }

    private final IPanelManager.OnPanelEventListener mPanelEventListener = new IPanelManager.OnPanelEventListener() {
        @Override
        public void onPanelOpened() {
            if (mTopBarView != null) return;
            initPanelView();
            showLoadingView();
            mAnnotPanel.prepare();
            mAnnotPanel.startSearch(0);
        }

        @Override
        public void onPanelClosed() {

        }
    };

    private void initPanelView() {
        mTopBarView = createTopView();
        mSearchView = createSearchView();
        mContentView = createContentView();
        if (mUiExtensionsManager.getState() != ReadStateConfig.STATE_REFLOW)
            mUiExtensionsManager.getPanelManager().addPanel(this);
    }

    private View createTopView() {
        View topView = View.inflate(mContext, R.layout.panel_topbar_layout, null);
        mTopContainer = topView.findViewById(R.id.panel_rl_top_container);
        mTopContainer.setBackgroundColor(ThemeConfig.getInstance(mContext).getB2());
        mCloseTv = topView.findViewById(R.id.panel_close_tv);
        mCloseTv.setTextColor(ThemeUtil.getPrimaryTextColor(mContext));
        if (AppDevice.isChromeOs(mPdfViewCtrl.getAttachedActivity())) {
            mCloseTv.setVisibility(View.VISIBLE);
        } else {
            if (AppDisplay.isPad())
                mCloseTv.setVisibility(View.INVISIBLE);
            else
                mCloseTv.setVisibility(View.VISIBLE);
        }
        mCloseTv.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                if (mUiExtensionsManager.getPanelManager().isShowingPanel())
                    mUiExtensionsManager.getPanelManager().hidePanel();
                if (mAnnotAdapter.getState() == AnnotAdapter.EDIT_STATE)
                    switchEditState(AnnotAdapter.NORMAL_STATE);
                else if (mAnnotAdapter.getState() == AnnotAdapter.SEARCH_STATE)
                    switchSearchState(AnnotAdapter.NORMAL_STATE);
                updateUIState();
            }
        });
        mTopTitle = topView.findViewById(R.id.panel_title_tv);
        mTopTitle.setText(AppResource.getString(mContext, R.string.rv_panel_annots_title));

        mEditViewTv = topView.findViewById(R.id.panel_edit_tv);
        mEditViewTv.setTextColor(ThemeUtil.getPrimaryTextColor(mContext));
        mEditViewTv.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                int state;
                if (mAnnotAdapter.getState() == AnnotAdapter.EDIT_STATE)
                    state = AnnotAdapter.NORMAL_STATE;
                else
                    state = AnnotAdapter.EDIT_STATE;
                switchEditState(state);
            }
        });
        return topView;
    }

    private String mLastSearchText;

    private PanelSearchView createSearchView() {
        PanelSearchView searchView = new PanelSearchView(mContext);
        searchView.setSearchEventListener(new PanelSearchView.OnSearchEventListener() {
            @Override
            public void onQueryTextChange(String newText) {
                if (!AppUtil.isEqual(mLastSearchText, newText)) {
                    List<AnnotNode> searchItems = mAnnotAdapter.getSearchItems();
                    List<AnnotNode> nodeList = mAnnotAdapter.getNodeList();
                    searchItems.clear();
                    mLastSearchText = newText.toLowerCase();

                    if (AppUtil.isEmpty(mLastSearchText)) {
                        if (mAnnotAdapter.getState() == AnnotAdapter.SEARCH_STATE) {
                            searchItems.addAll(mNodeList);
//                            resetSearchCount(searchItems);
                        }
                    } else {
                        for (int index = 0; index < nodeList.size(); index++) {
                            AnnotNode node = nodeList.get(index);

                            String author = node.getAuthor().toLowerCase();
                            String content = node.getContent().toString().toLowerCase();
                            if (author.contains(mLastSearchText) || content.contains(mLastSearchText)) {
                                searchItems.add(node);
                            }
                        }
                        resetSearchCount(searchItems);
                    }

                    mAnnotAdapter.setSearchPattern(mLastSearchText);
                    updateUIState();
                }
            }

            @Override
            public void onFocusChange(View v, boolean hasFocus) {
                if (hasFocus) {
                    switchSearchState(AnnotAdapter.SEARCH_STATE);
                    mAnnotAdapter.establishNodeList();
                    mAnnotAdapter.getSearchItems().clear();
                    mAnnotAdapter.getSearchItems().addAll(generateNodeList());
//                    List<AnnotNode> annotNodes = mAnnotAdapter.getNodeList();
//                    mAnnotAdapter.getSearchItems().addAll(annotNodes);
//                    resetSearchCount(mAnnotAdapter.getSearchItems());
                    updateUIState();
                } else {
                    if (AppUtil.isEmpty(mSearchView.getSearchText())) {
                        if (mPullListView != null)
                            mPullListView.hideHeaderView();
                        switchSearchState(AnnotAdapter.NORMAL_STATE);
                    } else {
                        updateUIState();
                    }
                }
            }

            @Override
            public void onCancel() {
                if (mPullListView != null)
                    mPullListView.hideHeaderView();
                switchSearchState(AnnotAdapter.NORMAL_STATE);
            }
        });
        return searchView;
    }

    private void resetSearchCount(List<AnnotNode> nodeList) {
        int size = nodeList.size();
        if (size > 0) {
            AnnotNode totalCountNode = new AnnotNode(-1);
            totalCountNode.setFlag(AnnotNode.FLAG_TOTAL_COUNT);
            totalCountNode.counter = size;
            nodeList.add(0, totalCountNode);
        }
//        AnnotNode lastPageDivider = null;
//        int nodeCount = 0;
//        for (int index = 0; index < nodeList.size(); index++) {
//            AnnotNode node = nodeList.get(index);
//            if (node.isPageDivider()) {
//                if (lastPageDivider != null && nodeCount > 0) {
//                    lastPageDivider.counter = nodeCount;
//                }
//
//                if (index == nodeList.size() - 1) {
//                    nodeList.remove(node);
//                    break;
//                }
//
//                if (nodeList.get(index + 1).isPageDivider()) {
//                    nodeList.remove(node);
//                    index--;
//                } else {
//                    nodeCount = 0;
//                    lastPageDivider = node;
//                }
//            } else {
//                nodeCount++;
//            }
//        }
    }

    private View createContentView() {
        View contentView = View.inflate(mContext, R.layout.panel_annot_contentview, null);
        initBottomView(contentView);
        initAnnotListView(contentView);
        return contentView;
    }

    private void initBottomView(View contentView) {
        mBottomView = contentView.findViewById(R.id.outline_panel_bottom_view);
        mDeleteIv = contentView.findViewById(R.id.panel_bottom_delete_iv);
        ThemeUtil.setTintList(mDeleteIv, ThemeUtil.getPrimaryIconColor(mContext));
        mDeleteIv.setEnabled(false);
        mDeleteIv.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                if (mAnnotAdapter.isSelectedAll()) {
                    if (mDialog == null) {
                        final Context context = mUiExtensionsManager.getAttachedActivity();
                        String title = AppResource.getString(mContext, R.string.menu_more_confirm);
                        String prompt = AppResource.getString(mContext, R.string.rd_panel_clear_comment);
                        mDialog = new UITextEditDialog(context, UIDialog.NO_INPUT);
                        mDialog.setTitle(title);
                        mDialog.getPromptTextView().setText(prompt);
                    }
                    mDialog.getOKButton().setOnClickListener(new View.OnClickListener() {
                        @Override
                        public void onClick(View v) {
                            mAnnotPanel.clearNodes();
                            mDialog.dismiss();
                        }
                    });
                    mDialog.show();
                } else {
                    mAnnotPanel.clearNodes();
                }
            }
        });

        mSelectedAllTv = contentView.findViewById(R.id.panel_bottom_select_all_tv);
        mSelectedAllTv.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                boolean isSelectedAll = mAnnotAdapter.isSelectedAll();
                if (isSelectedAll) {
                    mDeleteIv.setEnabled(false);
                    mAnnotAdapter.selectAll(false);
                    mSelectedAllTv.setText(AppResource.getString(mContext, R.string.fx_string_select_all));
                } else {
                    mDeleteIv.setEnabled(true);
                    mAnnotAdapter.selectAll(true);
                    mSelectedAllTv.setText(AppResource.getString(mContext, R.string.fx_string_deselect_all));
                }
            }
        });
    }

    private void initAnnotListView(View contentView) {
        mBottomFilterContainer = contentView.findViewById(R.id.panel_annot_filter_container);
        mBottomFilterContainer.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                if (mAnnotFilterDialog == null) {
                    mAnnotFilterDialog = new AnnotFilterDialog(mUiExtensionsManager.getAttachedActivity(), mPdfViewCtrl);
                }
                mAnnotFilterDialog.resetWH();
                mAnnotFilterDialog.showDialog();
                mAnnotFilterDialog.notifyDataSetChanged();
                mAnnotFilterDialog.setOnDLDismissListener(new MatchDialog.DismissListener() {
                    @Override
                    public void onDismiss() {
                        if (mAnnotFilterDialog == null) {
                            return;
                        }
                        List<String> types = mAnnotFilterDialog.getSelectedTypes();
                        if (!types.equals(mFilterTypes)) {
                            mFilterTypes.clear();
                            mFilterTypes.addAll(types);
                            mAnnotAdapter.setFilterTypes(mFilterTypes);
                            mAnnotAdapter.establishNodeList();
                            updateUIState();

                            if (mFilterTypes.size() > 0) {
                                mBottomFilterResultIv.setImageResource(R.drawable.panel_filter_selected);
                                mBottomFilterResultTv.setText(AppResource.getString(mContext, R.string.fx_search_filter_result));
                            } else {
                                mBottomFilterResultIv.setImageResource(R.drawable.panel_filter_normal);
                                mBottomFilterResultTv.setText(AppResource.getString(mContext, R.string.fx_search_filter));
                            }
                        }
                    }
                });
            }
        });
        mBottomFilterResultTv = contentView.findViewById(R.id.panel_annot_filter_tv);
        mBottomFilterResultIv = contentView.findViewById(R.id.panel_annot_filter_iv);
        ThemeUtil.setTintList(mBottomFilterResultIv, ThemeUtil.getPrimaryIconColor(mContext));

        mLoadingTv = contentView.findViewById(R.id.tv_annot_panel_loading);
        mNoInfoTv = contentView.findViewById(R.id.rv_panel_annot_noinfo);
        mNoInfoIv = contentView.findViewById(R.id.panel_no_annot_iv);
        mNoInfoIv.setColorFilter(ThemeConfig.getInstance(mContext).getPrimaryColor());
        mPullListView = contentView.findViewById(R.id.rv_panel_annot_list);
        mPullListView.addHeaderView(mSearchView);

        RecyclerView recyclerView = mPullListView.getContentRV();
        mLayoutManager = new LinearLayoutManager(mContext, LinearLayoutManager.VERTICAL, false);
        recyclerView.setLayoutManager(mLayoutManager);
        recyclerView.setItemAnimator(new DefaultItemAnimator());
        mAnnotAdapter = mAnnotPanel.getAdapter();
        recyclerView.setAdapter(mAnnotAdapter);
        if (AppDisplay.isPad()) {
            recyclerView.addOnScrollListener(new RecyclerView.OnScrollListener() {
                @Override
                public void onScrolled(@NonNull RecyclerView recyclerView, int dx, int dy) {
                    mAnnotAdapter.onScrolled(mLayoutManager);
                }
            });
        }
        mAnnotAdapter.setItemClickListener(new AnnotAdapter.OnItemClickListener() {
            @Override
            public void onItemClick(int position, AnnotNode node) {
                if (mAnnotAdapter.getState() == AnnotAdapter.EDIT_STATE) {
                    mDeleteIv.setEnabled(mAnnotAdapter.getEditItems().size() > 0);
                    if (mAnnotAdapter.isSelectedAll())
                        mSelectedAllTv.setText(AppResource.getString(mContext, R.string.fx_string_deselect_all));
                    else
                        mSelectedAllTv.setText(AppResource.getString(mContext, R.string.fx_string_select_all));
                }
            }
        });
    }

    private void switchSearchState(int state) {
        if (mAnnotAdapter.getState() == state) return;

        mAnnotAdapter.setState(state);
        if (state == AnnotAdapter.SEARCH_STATE) {
            mNodeList.clear();
            mNodeList = new ArrayList<>(mAnnotAdapter.getNodeList());
            mPullListView.setHeaderState(NestedHeaderBehavior.FIXED);
            mUiExtensionsManager.getPanelManager().getPanelHost().setTabVisibility(false);
            mSearchView.onActionViewExpanded();
        } else {
            mAnnotAdapter.getNodeList().clear();
            mAnnotAdapter.getNodeList().addAll(mNodeList);

            mPullListView.setHeaderState(NestedHeaderBehavior.SHOW);
            mUiExtensionsManager.getPanelManager().getPanelHost().setTabVisibility(true);
            mSearchView.onActionViewCollapsed();
        }
        updateUIState();
    }

    private void switchEditState(int state) {
        if (mAnnotAdapter.getState() == state) return;

        mDeleteIv.setEnabled(false);
        mAnnotAdapter.clearSelectedNodes();
        mAnnotAdapter.setState(state);
        if (state == AnnotAdapter.EDIT_STATE) {
            mEditViewTv.setText(AppResource.getString(mContext, R.string.fx_string_done));
            mPullListView.setHeaderState(NestedHeaderBehavior.HIDE);
            mBottomView.setVisibility(View.VISIBLE);
            mSelectedAllTv.setText(AppResource.getString(mContext, R.string.fx_string_select_all));
            mSelectedAllTv.setEnabled(mAnnotAdapter.getCanEditAnnotCount() > 0);
        } else {
            mEditViewTv.setText(AppResource.getString(mContext, R.string.fx_string_edit));
            mPullListView.setHeaderState(NestedHeaderBehavior.SHOW);
            mBottomView.setVisibility(View.GONE);
        }
        updateUIState();
    }

    void updateUIState() {
        if (mAnnotAdapter == null) return;

        resetEditButton();
        int state = mAnnotAdapter.getState();
        switch (mAnnotPanel.getCurrentStatus()) {
            case AnnotPanel.STATUS_CANCEL:
                showLoadingView();
                mPullListView.setHeaderState(NestedHeaderBehavior.HIDE);
                mBottomFilterContainer.setVisibility(View.GONE);
                mLoadingTv.setVisibility(View.GONE);
                break;
            case AnnotPanel.STATUS_DONE:
            case AnnotPanel.STATUS_FAILED:
                if (mAnnotPanel.getCount() == 0) {
                    if (state == AnnotAdapter.EDIT_STATE)
                        switchEditState(AnnotAdapter.NORMAL_STATE);
                    else if (mFilterTypes.size() > 0 && !(state == AnnotAdapter.SEARCH_STATE))
                        mBottomFilterContainer.setVisibility(View.VISIBLE);
                    else
                        mBottomFilterContainer.setVisibility(View.GONE);
                    showNoAnnotsView();
                } else {
                    if (state == AnnotAdapter.EDIT_STATE) {
                        if (mAnnotAdapter.getCanEditAnnotCount() > 0) {
                            mDeleteIv.setEnabled(mAnnotAdapter.getEditItems().size() > 0);
                            mBottomFilterContainer.setVisibility(View.GONE);
                            mPullListView.setHeaderState(NestedHeaderBehavior.HIDE);
                        } else {
                            switchEditState(AnnotAdapter.NORMAL_STATE);
                        }
                    } else if (state == AnnotAdapter.SEARCH_STATE) {
                        mBottomFilterContainer.setVisibility(View.GONE);
                        mPullListView.setHeaderState(NestedHeaderBehavior.FIXED);
                    } else {
                        mBottomFilterContainer.setVisibility(View.VISIBLE);
                        mPullListView.setHeaderState(NestedHeaderBehavior.SHOW);
                    }
                    hideNoAnnotsView();
                }
                mLoadingTv.setVisibility(View.GONE);
                break;
            case AnnotPanel.STATUS_LOADING:
            case AnnotPanel.STATUS_PAUSED:
            case AnnotPanel.STATUS_DELETING:
            default:
                hideNoAnnotsView();
                mBottomFilterContainer.setVisibility(View.GONE);
                mPullListView.setHeaderState(NestedHeaderBehavior.HIDE);
                mLoadingTv.setVisibility(View.VISIBLE);
                break;
        }

        mAnnotAdapter.notifyUpdateData();
    }

    private final PDFViewCtrl.IPageEventListener mPageEventListener = new PageEventListener() {

        @Override
        public void onPagesRemoved(boolean success, int[] pageIndexes) {
            if (mAnnotAdapter == null) return;

            mAnnotAdapter.onPagesRemoved(success, pageIndexes);
        }

        @Override
        public void onPageMoved(boolean success, int index, int dstIndex) {
            if (mAnnotAdapter == null) return;
            mAnnotAdapter.onPageMoved(success, index, dstIndex);
        }

        @Override
        public void onPagesInserted(boolean success, int dstIndex, int[] range) {
            //when we inserted pages or docs, the annotlist should be reloaded.
            mAnnotPanel.prepare();
            mAnnotPanel.startSearch(0);
        }
    };

    private final PDFViewCtrl.IDocEventListener mDocEventListener = new DocEventListener() {
        @Override
        public void onDocWillOpen() {
        }

        @Override
        public void onDocOpened(PDFDoc document, int errCode) {
            if (mAnnotAdapter == null) return;

            if (errCode == Constants.e_ErrSuccess) {
                prepare();
                IPanelManager panelManager = mUiExtensionsManager.getPanelManager();
                if (panelManager.isShowingPanel()
                        && panelManager.getPanelHost().getCurrentSpec() == AnnotPanelModule.this) {
                    panelManager.getPanelHost().setCurrentSpec(getPanelType());
                }
            }
        }

        @Override
        public void onDocWillClose(PDFDoc document) {
            if (mTopBarView == null) return;
            mAnnotPanel.onDocWillClose();

            mLoadingTv.setVisibility(View.GONE);
            mPullListView.setHeaderState(NestedHeaderBehavior.HIDE);
            if (mAnnotAdapter.getState() == AnnotAdapter.EDIT_STATE)
                switchEditState(AnnotAdapter.NORMAL_STATE);
            else if (mAnnotAdapter.getState() == AnnotAdapter.SEARCH_STATE)
                switchSearchState(AnnotAdapter.NORMAL_STATE);

            mFilterTypes.clear();
            mAnnotFilterDialog = null;
            mBottomFilterResultIv.setImageResource(R.drawable.panel_filter_normal);
            mBottomFilterResultTv.setText(AppResource.getString(mContext, R.string.fx_search_filter));
            updateUIState();
        }

        @Override
        public void onDocClosed(PDFDoc document, int errCode) {
            if (mTopBarView == null) return;
            mLoadingTv.setVisibility(View.GONE);
            mLoadingTv.setText(AppResource.getString(mContext, R.string.rv_panel_annot_loading_start));
        }

    };

    public void prepare() {
        mAnnotPanel.prepare();
    }

    @Override
    public boolean unloadModule() {
        mPdfViewCtrl.unregisterPageEventListener(mPageEventListener);
        mPdfViewCtrl.unregisterDocEventListener(mDocEventListener);
        mUiExtensionsManager.unregisterInteractionListener(mInteractionEventListener);
        mUiExtensionsManager.unregisterConfigurationChangedListener(mConfigurationChangedListener);
        mUiExtensionsManager.unregisterThemeEventListener(mThemeEventListener);

        mUiExtensionsManager.getPanelManager().removePanel(this);
        mUiExtensionsManager.getPanelManager().unregisterPanelEventListener(mPanelEventListener);
        mUiExtensionsManager.getDocumentManager().unregisterAnnotEventListener(mAnnotPanel.getAnnotEventListener());
        mUiExtensionsManager.getDocumentManager().unregisterFlattenEventListener(mAnnotPanel.getFlattenEventListener());
        mUiExtensionsManager.getDocumentManager().unregisterRedactionEventListener(mAnnotPanel.getRedactionEventListener());
        mUiExtensionsManager.getDocumentManager().unregisterImportedAnnotsEventListener(mAnnotPanel.getImportAnnotsListener());
        mUiExtensionsManager.getDocumentManager().unregisterGroupEventListener(mAnnotPanel.getGroupEventListener());
        return true;
    }

    @Override
    public String getName() {
        return MODULE_NAME_ANNOTPANEL;
    }

    private void resetEditButton() {
        if ((!mUiExtensionsManager.getDocumentManager().canAddAnnot()
                || !mUiExtensionsManager.isEnableModification())
                || mAnnotPanel.getCurrentStatus() == AnnotPanel.STATUS_LOADING
                || mAnnotPanel.getCurrentStatus() == AnnotPanel.STATUS_PAUSED) {
            mEditViewTv.setEnabled(false);
        } else {
            mEditViewTv.setEnabled(mAnnotAdapter.getCanEditAnnotCount() > 0);
        }
    }

    public void pauseSearch(int pageIndex) {
        mPausedPageIndex = pageIndex;
    }

    private void showNoAnnotsView() {
        mEditViewTv.setEnabled(false);

        if (mAnnotAdapter.getState() == AnnotAdapter.SEARCH_STATE) {
            mNoInfoIv.setImageResource(R.drawable.panel_no_search_result);
            mNoInfoTv.setText(AppResource.getString(mContext, R.string.fx_no_search_result));
        } else {
            mNoInfoIv.setImageResource(R.drawable.panel_no_annotation);
            mNoInfoTv.setText(AppResource.getString(mContext, R.string.rv_panel_annots_noinformation));
            mPullListView.setHeaderState(NestedHeaderBehavior.HIDE);
        }
        mLoadingTv.setVisibility(View.GONE);
        mNoInfoIv.setVisibility(View.VISIBLE);
        mNoInfoTv.setVisibility(View.VISIBLE);
    }

    private void hideNoAnnotsView() {
        mLoadingTv.setVisibility(View.GONE);

        if (mNoInfoTv.getVisibility() == View.GONE && mNoInfoIv.getVisibility() == View.GONE)
            return;
        mNoInfoIv.setVisibility(View.GONE);
        mNoInfoTv.setVisibility(View.GONE);
    }

    private void showLoadingView() {
        mEditViewTv.setEnabled(false);
        mPullListView.setHeaderState(NestedHeaderBehavior.HIDE);
        mNoInfoIv.setVisibility(View.GONE);
        mNoInfoTv.setVisibility(View.GONE);
        mLoadingTv.setVisibility(View.VISIBLE);
    }

    public void updatePageAnnots(int pageIndex) {
        if (mBottomFilterContainer == null) return;

        mBottomFilterContainer.setVisibility(View.GONE);
        hideNoAnnotsView();
        mAnnotAdapter.clearPageNodes(pageIndex);
        mAnnotPanel.startSearch(pageIndex, false);
    }

    public void updateLoadedPage(int curPageIndex, int total) {
        if (mLoadingTv == null) return;
        if (mAnnotPanel.getCurrentStatus() != AnnotPanel.STATUS_DONE && mAnnotPanel.getCurrentStatus() != AnnotPanel.STATUS_FAILED) {
            if (!mLoadingTv.isShown()) mLoadingTv.setVisibility(View.VISIBLE);
            mLoadingTv.setText(AppResource.getString(mContext, R.string.rv_panel_annot_loading_pagenum, curPageIndex, total));
        }
    }

    @Override
    public int getPanelType() {
        return PanelSpec.ANNOTATIONS;
    }

    @Override
    public int getIcon() {
        return R.drawable.panel_tab_annotation;
    }

    @Override
    public ColorStateList getIconTint() {
        return null;
    }

    @Override
    public View getTopToolbar() {
        return mTopBarView;
    }

    @Override
    public View getContentView() {
        return mContentView;
    }

    @Override
    public void onActivated() {
        resetEditButton();
        switch (mAnnotPanel.getCurrentStatus()) {
            case AnnotPanel.STATUS_CANCEL:
                if (mTopBarView != null) {
                    showLoadingView();
                    mAnnotPanel.startSearch(0);
                }
                break;
            case AnnotPanel.STATUS_LOADING:
                mAnnotPanel.setStatusPause(false);
                break;
            case AnnotPanel.STATUS_DONE:
                if (mLoadingTv.getVisibility() != View.GONE)
                    mLoadingTv.setVisibility(View.GONE);
                if (mAnnotPanel.getCount() > 0)
                    hideNoAnnotsView();
                else
                    showNoAnnotsView();
                break;
            case AnnotPanel.STATUS_PAUSED:
                mAnnotPanel.setStatusPause(false);
                mAnnotPanel.startSearch(mPausedPageIndex);
                break;
            case AnnotPanel.STATUS_FAILED:
            case AnnotPanel.STATUS_DELETING:
            default:
                break;
        }
        if (mAnnotAdapter.hasDataChanged()) {
            if (mAnnotAdapter.getItemCount() == 0)
                showNoAnnotsView();
            mAnnotAdapter.notifyUpdateData();
            mAnnotAdapter.resetDataChanged();
        }
    }

    @Override
    public void onDeactivated() {
        mAnnotAdapter.dismissGroupList();
        if (mAnnotAdapter.getState() == AnnotAdapter.EDIT_STATE) {
            switchEditState(AnnotAdapter.NORMAL_STATE);
        } else if (mAnnotAdapter.getState() == AnnotAdapter.SEARCH_STATE) {
            if (mPullListView != null)
                mPullListView.hideHeaderView();
            switchSearchState(AnnotAdapter.NORMAL_STATE);
        }

        if (mAnnotPanel.getCurrentStatus() == AnnotPanel.STATUS_LOADING) {
            mAnnotPanel.setStatusPause(true);
        }
    }

    private final IInteractionEventListener mInteractionEventListener = new IInteractionEventListener() {
        @Override
        public boolean onKeyDown(Activity act, int keyCode, KeyEvent event) {
            if (mAnnotAdapter == null) return false;
            if (keyCode == KeyEvent.KEYCODE_BACK) {
                if (mAnnotAdapter.getState() == AnnotAdapter.SEARCH_STATE) {
                    switchSearchState(AnnotAdapter.NORMAL_STATE);
                    return true;
                }
                if (mAnnotAdapter.getState() == AnnotAdapter.EDIT_STATE) {
                    switchEditState(AnnotAdapter.NORMAL_STATE);
                    return true;
                }
            }
            return false;
        }
    };

    private final UIExtensionsManager.ConfigurationChangedListener mConfigurationChangedListener = new UIExtensionsManager.ConfigurationChangedListener() {
        @Override
        public void onConfigurationChanged(Configuration newConfig) {
            if (mAnnotAdapter == null) return;
            mAnnotAdapter.onConfigurationChanged(mLayoutManager);

            if (mAnnotPanel.getCurrentStatus() == AnnotPanel.STATUS_LOADING) {
                mAnnotPanel.setStatusPause(true);
                AppThreadManager.getInstance().getMainThreadHandler().postDelayed(new Runnable() {
                    @Override
                    public void run() {
                        IPanelManager panelManager = mUiExtensionsManager.getPanelManager();
                        if (panelManager.isShowingPanel()
                                && panelManager.getPanelHost().getCurrentSpec() == AnnotPanelModule.this) {
                            mAnnotPanel.setStatusPause(false);
                            mAnnotPanel.startSearch(mPausedPageIndex);
                        }
                    }
                }, 300);
            }

            if (mAnnotFilterDialog != null && mAnnotFilterDialog.isShowing()) {
                mAnnotFilterDialog.resetWH();
                mAnnotFilterDialog.showDialog();
            }
        }
    };

    private final IThemeEventListener mThemeEventListener = new IThemeEventListener() {
        @Override
        public void onThemeColorChanged(String type, int color) {
            if (mTopBarView == null) return;
            if (mAnnotFilterDialog != null) {
                mAnnotFilterDialog.dismiss();
                mAnnotFilterDialog = null;
            }
            if (mDialog != null) {
                mDialog.dismiss();
                mDialog = null;
            }
            mSearchView.onThemeColorChanged();
            mCloseTv.setTextColor(ThemeUtil.getPrimaryTextColor(mContext));
            mEditViewTv.setTextColor(ThemeUtil.getPrimaryTextColor(mContext));
            mNoInfoIv.setColorFilter(ThemeConfig.getInstance(mContext).getPrimaryColor());
            mNoInfoTv.setTextColor(AppResource.getColor(mContext, R.color.t2));
            mLoadingTv.setTextColor(AppResource.getColor(mContext, R.color.t2));
            mContentView.setBackgroundColor(AppResource.getColor(mContext, R.color.b1));
            mPullListView.setBackgroundColor(AppResource.getColor(mContext, R.color.b1));
            mSelectedAllTv.setTextColor(AppResource.getColor(mContext, R.color.fx_menu_text_selector));
            ThemeUtil.setTintList(mDeleteIv, ThemeUtil.getPrimaryIconColor(mContext));
            mAnnotAdapter.notifyUpdateData();
            mAnnotAdapter.dismissGroupList();
            mBottomView.setBackground(AppResource.getDrawable(mContext, R.drawable.bottom_menu_bg));
            mBottomFilterContainer.setBackground(AppResource.getDrawable(mContext, R.drawable.bottom_menu_bg));
            ThemeUtil.setTintList(mBottomFilterResultIv, ThemeUtil.getPrimaryIconColor(mContext));
            mBottomFilterResultTv.setTextColor(AppResource.getColor(mContext, R.color.fx_menu_text_selector));
            mTopTitle.setTextColor(AppResource.getColor(mContext, R.color.t4));
            mTopContainer.setBackgroundColor(ThemeConfig.getInstance(mContext).getB2());
            if (mAnnotFilterDialog != null) {
                mAnnotFilterDialog.dismiss();
                mAnnotFilterDialog = null;
            }
            mLoadingTv.setTextColor(ThemeConfig.getInstance(mContext).getT4());
        }
    };

    private List<AnnotNode> generateNodeList() {
        List<AnnotNode> list = new ArrayList<>(mNodeList);
        AnnotNode node;
        int size = list.size();
        for (int i = 0; i < size; i++) {
            node = list.get(i);
            if (!node.isLeafNode() && node.getLevel() == 1 && !node.isExpanded()) {
                int index = list.indexOf(node);
                if (index == -1) break;
                node.setExpanded(true);
                List<AnnotNode> replyList = new ArrayList<>();
                mAnnotAdapter.establishReplyNode(replyList, node);
                list.addAll(index + 1, replyList);
                node.setExpanded(false);
                size = list.size();
                i = index + replyList.size();
            }
        }
        return list;
    }
}
