/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.modules.panel.filespec;

import static com.foxit.uiextensions.controls.dialog.fileselect.UISaveAsDialog.ISaveAsOnOKClickCallBack;

import android.content.Context;
import android.content.DialogInterface;
import android.content.res.Configuration;
import android.graphics.Rect;
import android.os.Environment;
import android.os.Handler;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.CheckBox;
import android.widget.ImageView;
import android.widget.TextView;

import androidx.annotation.NonNull;
import androidx.appcompat.app.AlertDialog;
import androidx.fragment.app.FragmentActivity;
import androidx.recyclerview.widget.LinearLayoutManager;

import com.foxit.sdk.PDFViewCtrl;
import com.foxit.sdk.pdf.annots.Annot;
import com.foxit.uiextensions.IPermissionProvider;
import com.foxit.uiextensions.R;
import com.foxit.uiextensions.UIExtensionsManager;
import com.foxit.uiextensions.annots.common.UIAnnotReply;
import com.foxit.uiextensions.browser.adapter.SuperAdapter;
import com.foxit.uiextensions.browser.adapter.viewholder.SuperViewHolder;
import com.foxit.uiextensions.controls.dialog.fileselect.UISaveAsDialog;
import com.foxit.uiextensions.controls.dialog.sheetmenu.ISheetMenu;
import com.foxit.uiextensions.controls.dialog.sheetmenu.UIBottomSheetMenu;
import com.foxit.uiextensions.controls.dialog.sheetmenu.UISheetMenu;
import com.foxit.uiextensions.modules.panel.IPanelManager;
import com.foxit.uiextensions.modules.panel.bean.BaseBean;
import com.foxit.uiextensions.modules.panel.bean.FileBean;
import com.foxit.uiextensions.theme.ThemeUtil;
import com.foxit.uiextensions.utils.AppDisplay;
import com.foxit.uiextensions.utils.AppFileUtil;
import com.foxit.uiextensions.utils.AppResource;
import com.foxit.uiextensions.utils.AppStorageManager;
import com.foxit.uiextensions.utils.AppUtil;
import com.foxit.uiextensions.utils.thread.AppThreadManager;

import java.util.ArrayList;
import java.util.List;


public class FileAttachmentAdapter extends SuperAdapter<FileBean> implements FileAttachmentPresenter.FileAttachmentViewer {
    public final static int FLAG_NORMAL = 0;
    public final static int FLAG_ANNOT = 2;

    private List<FileBean> mFileList;
    private List<FileBean> mEditList;

    private final FileAttachmentPresenter presenter;
    private final FileSpecModuleCallback callback;
    private UISaveAsDialog saveAsDialog;
    private ISheetMenu mSheetMenu;
    private final PDFViewCtrl mPdfViewCtrl;
    private final UIExtensionsManager mUIExtensionsManager;
    private OnItemClickListener mOnItemClickListener;

    private int mCanEditAnnotCount = 0;
    private int mSelectedPosition = -1;
    private boolean mIsPad;
    private boolean mIsEditState = false;
    private Context mContext;

    public FileAttachmentAdapter(Context context, PDFViewCtrl pdfViewCtrl, FileSpecModuleCallback callback) {
        super(context);
        this.mContext = context;
        this.mFileList = new ArrayList<>();
        this.mEditList = new ArrayList<>();
        this.callback = callback;
        this.mPdfViewCtrl = pdfViewCtrl;
        mIsPad = AppDisplay.isPad();

        mUIExtensionsManager = (UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager();
        presenter = new FileAttachmentPresenter(context, pdfViewCtrl, this);
    }

    void initPDFNameTree(boolean reInit) {
        presenter.initPDFNameTree(reInit);
    }

    void startSearchFileAttachment(boolean isLoadAnnotation, boolean isLoadNameTree) {
        presenter.searchFileAttachment(isLoadAnnotation, isLoadNameTree);
    }

    void switchEditState(boolean toEditState) {
        mIsEditState = toEditState;
    }

    boolean isEditState() {
        return mIsEditState;
    }

    List<FileBean> getEditList() {
        return mEditList;
    }

    List<FileBean> getList() {
        return mFileList;
    }

    void setList(List<FileBean> list) {
        this.mFileList = list;
    }

    boolean isEmpty() {
        return mFileList == null || mFileList.size() == 0;
    }

    boolean isSelectedAll() {
        return mEditList.size() > 0 && mEditList.size() == getCanEditItemCount();
    }

    void selectAll(boolean select) {
        mEditList.clear();

        if (select) {
            for (FileBean item : mFileList) {
                if (item.getFlag() == FLAG_ANNOT) {
                    if (mUIExtensionsManager.getDocumentManager().canAddAnnot() && mUIExtensionsManager.isEnableModification() && item.canDelete() && item.isWithDeletePermission())
                        mEditList.add(item);
                } else {
                    if (mUIExtensionsManager.getDocumentManager().canModifyContents() && mUIExtensionsManager.isEnableModification())
                        mEditList.add(item);
                }
            }
            mCanEditAnnotCount = mEditList.size();
        }
        notifyDataSetChanged();
    }

    int getCanEditItemCount() {
        return mCanEditAnnotCount;
    }

    void resetCanEditAnnotCount() {
        int count = 0;
        for (FileBean item : mFileList) {
            if (item.getFlag() == FLAG_ANNOT) {
                if (mUIExtensionsManager.getDocumentManager().canAddAnnot() && mUIExtensionsManager.isEnableModification() && item.canDelete() && item.isWithDeletePermission())
                    count++;
            } else {
                if (mUIExtensionsManager.getDocumentManager().canModifyContents() && mUIExtensionsManager.isEnableModification())
                    count++;
            }
        }
        mCanEditAnnotCount = count;
    }

    /**
     * add a file spec to pdfdoc nametree
     *
     * @param name file name
     * @param file file path
     */
    void add(String name, String file) {
        presenter.add(name, file);
        mUIExtensionsManager.getDocumentManager().setDocModified(true);
    }

    void add(Annot annot) {
        presenter.add(annot);
        mUIExtensionsManager.getDocumentManager().setDocModified(true);
    }

    void delete(List<FileBean> fileBeans) {
        presenter.delete(fileBeans);
        mUIExtensionsManager.getDocumentManager().setDocModified(true);
    }

    void removeItemBean(int pageIndex, String uuid) {
        presenter.removeItemBean(pageIndex, uuid);
    }

    void groupAnnot(Annot annot) {
        presenter.groupAnnot(annot);
        mUIExtensionsManager.getDocumentManager().setDocModified(true);
    }

    boolean hasName(String name) {
        return presenter.hasName(name);
    }

    void unGroupAnnot(FileBean fileBean) {
        presenter.unGroupAnnot(fileBean);
    }

    void updateByOutside(Annot annot) {
        presenter.updateByOutside(annot);
        mUIExtensionsManager.getDocumentManager().setDocModified(true);
    }

    int getCurrentStatus() {
        return presenter.getCurrentStatus();
    }

    private void delete(FileBean fileBean) {
        presenter.delete(fileBean);
        mUIExtensionsManager.getDocumentManager().setDocModified(true);
    }

    private void flatten(FileBean fileBean) {
        presenter.flatten(fileBean);
    }

    private void setDesc(FileBean fileBean, String content) {
        presenter.update(fileBean, content);
        mUIExtensionsManager.getDocumentManager().setDocModified(true);
    }

    private void save(FileBean fileBean, String path) {
        presenter.save(fileBean, path);
    }

    private void open(FileBean fileBean, String path) {
        presenter.open(fileBean, path);
    }

    @Override
    public void notifyUpdateData() {
        notifyDataSetChanged();
    }

    @NonNull
    @Override
    public SuperViewHolder onCreateViewHolder(@NonNull ViewGroup parent, int viewType) {
        return new ItemViewHolder(inflateLayout(getContext(), parent, R.layout.panel_item_fileattachment));
    }

    private View inflateLayout(Context context, ViewGroup parent, int layoutId) {
        return LayoutInflater.from(context).inflate(layoutId, parent, false);
    }

    @Override
    public int getItemCount() {
        return mFileList.size();
    }

    @Override
    public FileBean getDataItem(int position) {
        return mFileList.get(position);
    }

    @Override
    public void success(List<FileBean> list) {
        this.mFileList = list;
        resetCanEditAnnotCount();
        callback.success(this.mFileList);
        notifyDataSetChanged();
    }

    @Override
    public void loadDataChanged(List<FileBean> list) {
        this.mFileList = list;
        resetCanEditAnnotCount();
        notifyDataSetChanged();
    }

    @Override
    public void loading(int pageIndex, int total) {
        callback.loading(pageIndex, total);
    }

    @Override
    public void pause(int pageIndex) {
        callback.pause(pageIndex);
    }

    @Override
    public void notifyUpdateData(int type, int index, List<FileBean> items) {
        this.mFileList = items;
        resetCanEditAnnotCount();
        callback.success(this.mFileList);

        if (type == FileAttachmentPresenter.FileAttachmentViewer.ADD) {
            notifyItemInserted(index);
        } else if (type == FileAttachmentPresenter.FileAttachmentViewer.MODIFY) {
            notifyItemChanged(index);
        } else if (type == FileAttachmentPresenter.FileAttachmentViewer.DELETE) {
            notifyItemRemoved(index);
        } else {
            notifyUpdateData();
        }
    }

    @Override
    public void fail(int errCode) {
        callback.fail(errCode);
    }

    @Override
    public void showProgressDlg(int type) {
        callback.showProgressDlg(type);
    }

    @Override
    public void open(String path, String filename) {
        callback.openDoc(path, filename);
    }

    @Override
    public void dismissProgressDlg(int type) {
        callback.dismissProgressDlg(type);
    }

    class ItemViewHolder extends SuperViewHolder {

        private final ImageView icon;
        private final ImageView more;
        private final TextView title;
        private final TextView fileInfo;
        private final TextView desc;
        private final CheckBox checkView;
        private final View container;
        private final View labelContainer;
        private final TextView labelName;
        private final TextView labelCount;
        private final View labelTopDivider;
        private final View labelBottomDivider;

        public ItemViewHolder(View viewHolder) {
            super(viewHolder);
            container = viewHolder.findViewById(R.id.panel_attachment_container);
            icon = viewHolder.findViewById(R.id.panel_item_fileattachment_icon);
            more = viewHolder.findViewById(R.id.panel_item_fileattachment_more);
            title = viewHolder.findViewById(R.id.panel_item_fileattachment_title);
            fileInfo = viewHolder.findViewById(R.id.panel_item_fileattachment_info);
            desc = viewHolder.findViewById(R.id.panel_item_fileattachment_desc);
            checkView = viewHolder.findViewById(R.id.rd_bookmark_item_checkbox);
            labelContainer = viewHolder.findViewById(R.id.panel_label_container);
            labelName = labelContainer.findViewById(R.id.rv_panel_item_page_tv);
            labelCount = labelContainer.findViewById(R.id.rv_panel_item_count_tv);
            labelTopDivider = labelContainer.findViewById(R.id.panel_label_top_divider);
            labelBottomDivider = labelContainer.findViewById(R.id.panel_label_bottom_divider);

            desc.setOnClickListener(this);
            more.setOnClickListener(this);
            container.setOnClickListener(this);
        }

        @Override
        public void bind(BaseBean data, int position) {
            container.setSelected(!mIsPad && mSelectedPosition == position);

            FileBean item = (FileBean) data;
            icon.setImageResource(getIconResource(item.getTitle()));
            if (mIsPad) {
                title.setText(AppFileUtil.getFileNameWithoutExt(item.getTitle()));
                fileInfo.setText(String.format("%s · %s · %s", AppFileUtil.getFileExt(item.getTitle()).toUpperCase(), item.getDate(), item.getSize()));
            } else {
                title.setText(item.getTitle());
                fileInfo.setText(String.format("%s · %s", item.getDate(), item.getSize()));
            }
            desc.setText(item.getDesc());
            desc.setVisibility(AppUtil.isBlank(item.getDesc()) ? View.GONE : View.VISIBLE);

            if (item.isShowLabel()) {
                if (item.getFlag() == FLAG_NORMAL)
                    labelName.setText(AppResource.getString(getContext(), R.string.rv_panel_attachment_label));
                else
                    labelName.setText(AppResource.getString(getContext(), R.string.attachment_page_tab, item.getPageIndex() + 1));
                labelCount.setText(String.valueOf(item.getCount()));
                labelContainer.setVisibility(View.VISIBLE);
            } else {
                labelContainer.setVisibility(View.GONE);
            }

            boolean canCopy = mUIExtensionsManager.getDocumentManager().canCopyForAssess()
                    && mUIExtensionsManager.getDocumentManager().canCopy();   //3. when copy access.. == not allow: all can not save
            if (item.getFlag() == FLAG_ANNOT) {
                //2.when comment ==not allow:annotation should not show more view! but fileattachment can!
                boolean canEdit = mUIExtensionsManager.getDocumentManager().canAddAnnot() && mUIExtensionsManager.isEnableModification();
                more.setEnabled(canEdit || canCopy);
                desc.setEnabled(canEdit && item.isWithModificationPermission());
            } else if (item.getFlag() == FLAG_NORMAL) {
                //1.when modify ==not allow:fileattachment should not show more view! but annotation can!
                boolean canModify = mUIExtensionsManager.getDocumentManager().canModifyContents() && mUIExtensionsManager.isEnableModification();
                more.setEnabled(canCopy || canModify);
                desc.setEnabled(canModify);
            }

            if (isEditState()) {
                more.setVisibility(View.GONE);
                int checkVisibility;
                if (item.getFlag() == FLAG_NORMAL)
                    checkVisibility = mUIExtensionsManager.getDocumentManager().canModifyContents() && mUIExtensionsManager.isEnableModification() ?
                            View.VISIBLE : View.INVISIBLE;
                else
                    checkVisibility = mUIExtensionsManager.getDocumentManager().canAddAnnot() && mUIExtensionsManager.isEnableModification() && item.canDelete() && item.isWithDeletePermission() ?
                            View.VISIBLE : View.INVISIBLE;
                checkView.setVisibility(checkVisibility);
                checkView.setChecked(mEditList.contains(item));
                ThemeUtil.setTintList(checkView, ThemeUtil.getCheckboxColor(getContext()));
            } else {
                checkView.setVisibility(View.GONE);
                more.setVisibility(View.VISIBLE);
            }

            labelContainer.setBackgroundColor(AppResource.getColor(mContext, R.color.b2));
            labelTopDivider.setBackgroundColor(AppResource.getColor(mContext, R.color.p1));
            labelBottomDivider.setBackgroundColor(AppResource.getColor(mContext, R.color.p1));
            labelName.setTextColor(AppResource.getColor(mContext, R.color.t3));
            labelCount.setTextColor(AppResource.getColor(mContext, R.color.t3));
            container.setBackground(AppResource.getDrawable(mContext, R.drawable.rd_list_item_bg));
            title.setTextColor(AppResource.getColor(mContext, R.color.t4));
            fileInfo.setTextColor(AppResource.getColor(mContext, R.color.t3));
            desc.setTextColor(AppResource.getColor(mContext, R.color.t2));
            ThemeUtil.setTintList(more, ThemeUtil.getPrimaryIconColor(mContext));
        }

        @Override
        public void onClick(View v) {
            int id = v.getId();

            final int position = getAdapterPosition();
            if (position == -1) return;

            final FileBean item = mFileList.get(position);
            if (id == R.id.panel_item_fileattachment_more) {
                if (AppUtil.isFastDoubleClick())
                    return;
                mSelectedPosition = position;
                if (!mIsPad && mSelectedPosition >= 0)
                    notifyItemChanged(mSelectedPosition);

                final List<Integer> items = new ArrayList<>();
                final boolean canCopy = mUIExtensionsManager.getDocumentManager().canCopyForAssess()
                        && mUIExtensionsManager.getDocumentManager().canCopy();   //3. when copy access.. == not allow: all can not save
                if (item.getFlag() == FLAG_ANNOT) {
                    //2.when comment ==not allow:annotation should not show more view! but fileattachment can!
                    boolean canEdit = mUIExtensionsManager.getDocumentManager().canAddAnnot() && mUIExtensionsManager.isEnableModification();
                    //4.when copy access or extract == not allow: annot can save,but attachment can not;
                    if (item.isWithModificationPermission() && canEdit || canCopy)
                        items.add(ISheetMenu.SAVE_MENU);
                    if (item.isWithModificationPermission() && canEdit && item.canComment())
                        items.add(ISheetMenu.DESC_MENU);//lock & readonly flag
                    if (item.isWithModificationPermission()
                            && canEdit && item.canFlatten()
                            && mUIExtensionsManager.canAssemble()
                            && mUIExtensionsManager.getDocumentManager().withDeletePermission())
                        items.add(ISheetMenu.FLATTEN_MENU);
                    if (canEdit && item.canDelete() && item.isWithDeletePermission())
                        items.add(ISheetMenu.DELETE_MENU);

                } else if (item.getFlag() == FLAG_NORMAL) {
                    //1.when modify ==not allow:fileattachment should not show more view! but annotation can!
                    final boolean canModify = mUIExtensionsManager.getDocumentManager().canModifyContents() && mUIExtensionsManager.isEnableModification();

                    IPermissionProvider permissionProvider = mUIExtensionsManager.getPermissionProvider();
                    if (permissionProvider != null) {
                        permissionProvider.checkPermission(IPermissionProvider.FUNCTION_ATTACHMENT_MODIFY, new IPermissionProvider.IPermissionState() {
                            @Override
                            public void onState(int state) {
                                if (canCopy)
                                    items.add(ISheetMenu.SAVE_MENU);
                                if (canModify && state == 0)
                                    items.add(ISheetMenu.DESC_MENU);
                                if (canModify && state == 0)
                                    items.add(ISheetMenu.DELETE_MENU);
                            }
                        });
                    } else {
                        if (canCopy)
                            items.add(ISheetMenu.SAVE_MENU);
                        if (canModify)
                            items.add(ISheetMenu.DESC_MENU);
                        if (canModify)
                            items.add(ISheetMenu.DELETE_MENU);
                    }
                }
                mSheetMenu = UISheetMenu.newInstance((FragmentActivity) mUIExtensionsManager.getAttachedActivity());
                if (mIsPad)
                    mSheetMenu.setWidth(AppResource.getDimensionPixelSize(getContext(), R.dimen.ux_pad_more_menu_width));
                mSheetMenu.setSheetItemClickListener(new UIBottomSheetMenu.OnSheetItemClickListener() {
                    @Override
                    public void onItemClick(int type) {
                        dismissSheetMenu();
                        if (ISheetMenu.SAVE_MENU == type) {
                            final String fileName = item.getTitle();
                            if (fileName == null || fileName.trim().length() < 1) {
                                AlertDialog dialog = new AlertDialog.Builder(mUIExtensionsManager.getAttachedActivity()).
                                        setMessage(AppResource.getString(getContext(), R.string.save_failed_by_incorrect_file_name)).
                                        setPositiveButton(AppResource.getString(getContext(), R.string.fx_string_ok), new DialogInterface.OnClickListener() {
                                            @Override
                                            public void onClick(DialogInterface dialog, int which) {
                                            }
                                        }).create();
                                dialog.setCanceledOnTouchOutside(false);
                                dialog.show();
                                return;
                            }
                            String suffix = fileName.substring(fileName.lastIndexOf(".") + 1);
                            saveAsDialog = new UISaveAsDialog(mUIExtensionsManager.getAttachedActivity(), item.getTitle(), suffix, new ISaveAsOnOKClickCallBack() {
                                @Override
                                public void onOkClick(final String newFilePath) {
                                    save(item, newFilePath);
                                }

                                @Override
                                public void onCancelClick() {
                                }
                            });
                            saveAsDialog.getFolderDialog().setRightButtonText(AppResource.getString(getContext(), R.string.fx_string_save));
                            saveAsDialog.showDialog();
                        } else if (ISheetMenu.COMMENT_MENU == type) {
                            setDesc(item);
                        } else if (ISheetMenu.FLATTEN_MENU == type) {
                            if (item.getFlag() == FLAG_ANNOT) {
                                if (mUIExtensionsManager.getPermissionProvider() != null) {
                                    mUIExtensionsManager.getPermissionProvider().checkPermission(
                                            IPermissionProvider.FUNCTION_FLATTEN, new IPermissionProvider.IPermissionState() {
                                                @Override
                                                public void onState(int state) {
                                                    if (state == IPermissionProvider.PERMISSION_STATE_SHOW) {
                                                        flatten(item);
                                                    }
                                                }
                                            }
                                    );
                                } else {
                                    flatten(item);
                                }
                            }
                        } else if (ISheetMenu.DELETE_MENU == type) {
                            delete(item);
                        }
                    }
                });
                mSheetMenu.setOnSheetDismissListener(new ISheetMenu.OnSheetDismissListener() {
                    @Override
                    public void onDismiss(ISheetMenu sheetMenu) {
                        if (mSelectedPosition != -1) {
                            int tempPosition = mSelectedPosition;
                            mSelectedPosition = -1;
                            if (!mIsPad && tempPosition >= 0)
                                notifyItemChanged(tempPosition);
                        }
                    }
                });
                mSheetMenu.setSheetItems(items);

                showSheetMenu(v);
            } else if (id == R.id.panel_attachment_container) {
                if (isEditState()) {
                    if (item.getFlag() == FLAG_ANNOT) {
                        if ((!mUIExtensionsManager.getDocumentManager().canAddAnnot() || !mUIExtensionsManager.isEnableModification()) || !item.canDelete() || !item.isWithDeletePermission())
                            return;
                    } else {
                        if (!mUIExtensionsManager.getDocumentManager().canModifyContents() || !mUIExtensionsManager.isEnableModification())
                            return;
                    }

                    if (mEditList.contains(item))
                        mEditList.remove(item);
                    else
                        mEditList.add(item);
                    notifyItemChanged(position);
                } else {
                    if (AppUtil.isFastDoubleClick())
                        return;

                    String tempPath = Environment.getExternalStorageDirectory() + "/FoxitSDK/AttaTmp/";
                    if (AppFileUtil.needScopedStorageAdaptation())
                        tempPath = AppStorageManager.getInstance(mContext).getScopedCacheDir() + "/FoxitSDK/AttaTmp/";
                    open(item, tempPath);
                }
                if (mOnItemClickListener != null)
                    mOnItemClickListener.onItemClick(position, item);
            } else if (id == R.id.panel_item_fileattachment_desc) {
                setDesc(item);
            }
        }

        private void setDesc(final FileBean fileBean) {
            UIAnnotReply.replyToAnnot(mPdfViewCtrl, mUIExtensionsManager.getRootView(), true, UIAnnotReply.TYPE_DESC, new UIAnnotReply.ReplyCallback() {
                @Override
                public void result(final String content) {
                    FileAttachmentAdapter.this.setDesc(fileBean, content);
                }

                @Override
                public String getContent() {
                    return fileBean.getDesc();
                }
            });
        }

        private int getIconResource(String filename) {
            String ext = "";
            if (filename != null) {
                ext = filename.substring(filename.lastIndexOf('.') + 1).toLowerCase();
            }

            if (ext.equals("pdf")) {
                return R.drawable.fb_file_pdf;
            } else if (ext.equals("ofd")) {
                return R.drawable.fb_file_ofd;
            } else if (ext.equals("ppdf")) {
                return R.drawable.fb_file_ppdf;
            } else if (ext.equals("jpg")
                    || ext.equals("jpeg")
                    || ext.equals("png")
                    || ext.equals("bmp")
                    || ext.equals("gif")) {
                return R.drawable.fb_file_picture;
            } else if (ext.equals("doc") || ext.equals("docx")) {
                return R.drawable.fb_file_doc;
            } else if (ext.equals("txt")) {
                return R.drawable.fb_file_txt;
            } else if (ext.equals("xls") || ext.equals("xlsx")) {
                return R.drawable.fb_file_xls;
            } else if (ext.equals("ppt")) {
                return R.drawable.fb_file_ppt;
            } else if (ext.equals("html") || ext.equals("xml")) {
                return R.drawable.fb_file_html;
            } else if (ext.equals("zip") || ext.equals("gz") || ext.equals("rar")) {
                return R.drawable.fb_file_zip;
            } else {
                return R.drawable.fb_file_other;
            }
        }
    }

    public void dismissDialog() {
        if (saveAsDialog != null) {
            saveAsDialog.dismiss();
            saveAsDialog = null;
        }
    }

    public void onConfigurationChanged(Configuration newConfig) {
        if (saveAsDialog != null && saveAsDialog.isShowing()) {
            saveAsDialog.resetWH();
            saveAsDialog.showDialog();
        }

        if (getCurrentStatus() == FileAttachmentPresenter.FileAttachmentViewer.STATUS_LOADING) {
            setPause(true);
            AppThreadManager.getInstance().getMainThreadHandler().postDelayed(new Runnable() {
                @Override
                public void run() {
                    IPanelManager panelManager = mUIExtensionsManager.getPanelManager();
                    if (panelManager.isShowingPanel()
                            && panelManager.getPanelHost().getCurrentSpec() instanceof FileSpecPanelModule) {
                        reStart();
                    }
                }
            }, 300);
        }
    }

    void release() {
        mFileList.clear();
        presenter.release();
        notifyDataSetChanged();
    }

    void setPause(boolean pause) {
        presenter.setPause(pause);
    }

    void reStart() {
        presenter.reStart();
    }

    void onPageRemoved(int index) {
        presenter.onPageRemoved(index);
    }

    void onPageMoved(int index, int dstIndex) {
        presenter.onPageMoved(index, dstIndex);
    }

    private void showSheetMenu(View view) {
        Rect rect = new Rect();
        view.getGlobalVisibleRect(rect);
        mSheetMenu.show(mUIExtensionsManager.getRootView(), rect);
    }

    private void dismissSheetMenu() {
        if (mSheetMenu != null && mSheetMenu.isShowing()) {
            mSheetMenu.dismiss();
        }
    }

    void updateMenu(final LinearLayoutManager layoutManager) {
        if (mSheetMenu != null && mSheetMenu.isShowing()) {
            if (mIsPad) {
                new Handler().postDelayed(new Runnable() {
                    @Override
                    public void run() {
                        View view = layoutManager.findViewByPosition(mSelectedPosition);
                        if (view != null) {
                            View moreView = view.findViewById(R.id.panel_item_fileattachment_more);
                            showSheetMenu(moreView);
                        } else {
                            mShouldUpdateMenu = true;

                            int scrollPosition = Math.min(getItemCount() - 1, mSelectedPosition + 1);
                            layoutManager.scrollToPosition(scrollPosition);
                        }
                    }
                }, 380);
            } else {
                showSheetMenu(mUIExtensionsManager.getRootView());
            }
        }
    }

    private boolean mShouldUpdateMenu = false;

    void onScrolled(LinearLayoutManager layoutManager) {
        if (mSelectedPosition == -1 || !mShouldUpdateMenu) return;

        mShouldUpdateMenu = false;
        View view = layoutManager.findViewByPosition(mSelectedPosition);
        if (view != null) {
            View moreView = view.findViewById(R.id.panel_item_fileattachment_more);
            showSheetMenu(moreView);
        } else {
            dismissSheetMenu();
        }
    }

    void setOnItemClickListener(OnItemClickListener onItemClickListener) {
        this.mOnItemClickListener = onItemClickListener;
    }

    interface OnItemClickListener {
        void onItemClick(int position, FileBean fileBean);
    }

}
