/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.modules.panel.filespec;

import android.app.Activity;
import android.content.Context;
import android.graphics.RectF;
import android.util.SparseArray;

import com.foxit.sdk.PDFException;
import com.foxit.sdk.PDFViewCtrl;
import com.foxit.sdk.Task;
import com.foxit.sdk.common.DateTime;
import com.foxit.sdk.pdf.FileSpec;
import com.foxit.sdk.pdf.PDFDoc;
import com.foxit.sdk.pdf.PDFPage;
import com.foxit.sdk.pdf.annots.Annot;
import com.foxit.sdk.pdf.annots.FileAttachment;
import com.foxit.sdk.pdf.objects.PDFDictionary;
import com.foxit.sdk.pdf.objects.PDFNameTree;
import com.foxit.sdk.pdf.objects.PDFObject;
import com.foxit.uiextensions.DocumentManager;
import com.foxit.uiextensions.UIExtensionsManager;
import com.foxit.uiextensions.annots.fileattachment.FileAttachmentUtil;
import com.foxit.uiextensions.annots.fileattachment.IFileAttachmentAnnotContent;
import com.foxit.uiextensions.annots.multiselect.GroupManager;
import com.foxit.uiextensions.modules.panel.bean.FileBean;
import com.foxit.uiextensions.utils.AnnotPermissionUtil;
import com.foxit.uiextensions.utils.AppAnnotUtil;
import com.foxit.uiextensions.utils.AppDmUtil;
import com.foxit.uiextensions.utils.AppFileUtil;
import com.foxit.uiextensions.utils.AppIntentUtil;
import com.foxit.uiextensions.utils.AppUtil;
import com.foxit.uiextensions.utils.Event;
import com.foxit.uiextensions.utils.IResult;
import com.foxit.uiextensions.utils.thread.AppThreadManager;

import java.io.File;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

public class FileAttachmentPresenter {
    private final FileAttachmentViewer viewer;
    private final PDFViewCtrl mPdfViewCtrl;
    private final UIExtensionsManager mUiExtensionsManager;
    private PDFNameTree pdfNameTree;
    private Context mContext;

    private final SparseArray<List<FileBean>> mPageAnnots;
    private final List<FileBean> mTotalList;
    private final List<FileBean> mDocAttachmentList;

    private int mLoadedStatus = FileAttachmentViewer.STATUS_CANCEL;
    private int mLoadedIndex;

    private SearchNameTreeTask mSearchNameTreeTask;
    private SearchFileAttachmentTask mSearchFileAttachmentTask;

    public FileAttachmentPresenter(Context context, PDFViewCtrl pdfViewCtrl, FileAttachmentViewer viewer) {
        this.viewer = viewer;

        mPageAnnots = new SparseArray<>();
        mTotalList = new ArrayList<>();
        mDocAttachmentList = new ArrayList<>();
        mPdfViewCtrl = pdfViewCtrl;
        mUiExtensionsManager = (UIExtensionsManager) pdfViewCtrl.getUIExtensionsManager();
        mContext = context;
    }

    private boolean hasNameTree() {
        if (mPdfViewCtrl.getDoc() != null) {
            try {
                PDFDictionary catalog = mPdfViewCtrl.getDoc().getCatalog();
                return catalog.hasKey("Names");
            } catch (PDFException e) {
                e.printStackTrace();
            }
        }
        return false;
    }

    void initPDFNameTree(boolean reInit) {
        if ((this.pdfNameTree == null || reInit)) {
            pdfNameTree = null;
            if (hasNameTree()) {
                createNameTree();
            }
        }
    }

    private void createNameTree() {
        try {
            this.pdfNameTree = new PDFNameTree(mPdfViewCtrl.getDoc(), PDFNameTree.e_EmbeddedFiles);
        } catch (PDFException e) {
            e.printStackTrace();
        }
    }

    public void searchFileAttachment(boolean isLoadAnnotation, boolean isLoadNameTree) {
        if (mPdfViewCtrl.getDoc() == null) return;

        if (isLoadNameTree) {
            clearData();
            cancelNameTreeTask();
            startSearchNameTree();
        } else {
            mTotalList.clear();
            mPageAnnots.clear();

            mLoadedStatus = FileAttachmentViewer.STATUS_CANCEL;
            mbPause = false;
            mLoadedIndex = 0;

            mTotalList.addAll(mDocAttachmentList);
        }

        if (isLoadAnnotation) {
            int pageCount = 0;
            try {
                pageCount = mPdfViewCtrl.getPageCount();
            } catch (Exception e) {
                mLoadedStatus = FileAttachmentViewer.STATUS_FAILED;
                viewer.fail(FileAttachmentViewer.STATUS_FAILED);
                return;
            }

            canelFileAttachmentTask();
            startSearchFileAttachment(0, pageCount);
        }
    }

    void startSearchNameTree() {
        mSearchNameTreeTask = new SearchNameTreeTask(mPdfViewCtrl, pdfNameTree,
                new IResult<Void, Void, List<FileBean>>() {
                    @Override
                    public void onResult(boolean success, Void p1, Void p2, List<FileBean> docAttachments) {
                        if (success) {
                            mDocAttachmentList.addAll(docAttachments);
                            mTotalList.addAll(docAttachments);
                        }
                        viewer.success(mTotalList);
                    }
                });
        mPdfViewCtrl.addTask(mSearchNameTreeTask);
    }

    private void startSearchFileAttachment(int pageIndex, final int pageCount) {
        mSearchFileAttachmentTask = new SearchFileAttachmentTask(mPdfViewCtrl, pageIndex,
                new IResult<Void, Integer, List<FileBean>>() {
                    @Override
                    public void onResult(boolean success, Void p1, Integer pageIndex, List<FileBean> annots) {
                        mLoadedIndex = pageIndex;

                        if (!success) {
                            mLoadedStatus = FileAttachmentViewer.STATUS_FAILED;
                            viewer.fail(FileAttachmentViewer.STATUS_FAILED);
                            return;
                        }

                        if (mbPause) {
                            mPauseIndex = pageIndex;
                            mLoadedStatus = FileAttachmentViewer.STATUS_PAUSED;
                            viewer.pause(pageIndex);
                            return;
                        }

                        mTotalList.addAll(annots);
                        mPageAnnots.append(pageIndex, annots);

                        viewer.loadDataChanged(mTotalList);
                        viewer.loading(pageIndex + 1, pageCount);

                        if (pageIndex < pageCount - 1) {
                            if (mLoadedStatus != FileAttachmentViewer.STATUS_CANCEL) {
                                startSearchFileAttachment(pageIndex + 1, pageCount);
                            }
                        } else {
                            mLoadedStatus = FileAttachmentViewer.STATUS_DONE;
                            viewer.success(mTotalList);
                        }
                    }
                });

        mPdfViewCtrl.addTask(mSearchFileAttachmentTask);
    }

    void clearData() {
        mTotalList.clear();
        mDocAttachmentList.clear();
        mPageAnnots.clear();

        mLoadedStatus = FileAttachmentViewer.STATUS_CANCEL;
        mbPause = false;
        mLoadedIndex = 0;
    }

    void release() {
        cancelNameTreeTask();
        canelFileAttachmentTask();

        clearData();
        pdfNameTree = null;
    }

    private void cancelNameTreeTask() {
        if (mSearchNameTreeTask != null && mSearchNameTreeTask.isRunning()) {
            mPdfViewCtrl.removeTask(mSearchNameTreeTask);
            mSearchNameTreeTask = null;
        }
    }

    private void canelFileAttachmentTask() {
        if (mSearchFileAttachmentTask != null && mSearchFileAttachmentTask.isRunning()) {
            mPdfViewCtrl.removeTask(mSearchFileAttachmentTask);
            mSearchFileAttachmentTask = null;
        }
    }

    private class SearchNameTreeTask extends Task {
        private final PDFViewCtrl pdfViewCtrl;
        private final PDFNameTree nameTree;
        private final List<FileBean> docAttachmentList;
        private boolean mRet = false;
        private boolean mIsRunning = false;

        private SearchNameTreeTask(PDFViewCtrl viewCtrl,
                                   PDFNameTree pdfNameTree,
                                   final IResult<Void, Void, List<FileBean>> onResultCallback) {
            super(new CallBack() {
                @Override
                public void result(Task task) {
                    SearchNameTreeTask task1 = (SearchNameTreeTask) task;
                    task1.mIsRunning = false;
                    onResultCallback.onResult(task1.mRet, null, null, task1.docAttachmentList);
                }
            });
            this.pdfViewCtrl = viewCtrl;
            this.nameTree = pdfNameTree;

            docAttachmentList = new ArrayList<>();
        }

        @Override
        protected void execute() {
            mIsRunning = true;
            if (nameTree == null) {
                mRet = false;
                return;
            }

            try {
                mLoadedStatus = FileAttachmentViewer.STATUS_LOADING;

                int nOrgCount = nameTree.getCount();
                for (int o = 0; o < nOrgCount; o++) {
                    if (isCanceled()) {
                        mRet = false;
                        return;
                    }

                    String name = nameTree.getName(o);
                    PDFObject object = nameTree.getObj(name);
                    FileBean item = new FileBean();
                    FileSpec fs = new FileSpec(pdfViewCtrl.getDoc(), object);
                    if (!fs.isEmpty()) {
                        item.setName(name);
                        item.setTitle(fs.getFileName());
                        item.setDate(getModifiedDateTimeString(fs));
                        item.setSize(AppFileUtil.formatFileSize(fs.getFileSize()));
                        item.setFlag(FileAttachmentAdapter.FLAG_NORMAL);
                        item.setDesc(fs.getDescription());
                        docAttachmentList.add(item);
                    }
                }

                if (docAttachmentList.size() > 0) {
                    docAttachmentList.get(0).setShowLable(true);
                    docAttachmentList.get(0).setCount(docAttachmentList.size());
                }
                mRet = true;
            } catch (PDFException e) {
                mRet = false;
            }
        }

        boolean isRunning() {
            return mIsRunning;
        }
    }

    private class SearchFileAttachmentTask extends Task {
        private final int mPageIndex;
        private final PDFViewCtrl pdfViewCtrl;
        private final List<FileBean> mAnnotList;
        private boolean mRet = false;
        private boolean mIsRunning = false;

        private SearchFileAttachmentTask(PDFViewCtrl viewCtrl,
                                         int pageIndex,
                                         final IResult<Void, Integer, List<FileBean>> onResultCallback) {
            super(new CallBack() {
                @Override
                public void result(Task task) {
                    SearchFileAttachmentTask task1 = (SearchFileAttachmentTask) task;
                    task1.mIsRunning = false;
                    onResultCallback.onResult(task1.mRet, null, task1.mPageIndex, task1.mAnnotList);
                }
            });
            this.pdfViewCtrl = viewCtrl;

            mPageIndex = pageIndex;
            mAnnotList = new ArrayList<>();
        }

        @Override
        protected void execute() {
            mIsRunning = true;
            //load annot
            if (pdfViewCtrl.getDoc() != null) {
                mLoadedStatus = FileAttachmentViewer.STATUS_LOADING;

                try {
                    PDFPage pdfPage = pdfViewCtrl.getDoc().getPage(mPageIndex);
                    if (pdfPage.isEmpty()) {
                        mRet = false;
                        return;
                    }

                    int annotcount = pdfPage.getAnnotCount();
                    List<FileBean> annots = new ArrayList<>();
                    for (int i = 0; i < annotcount; i++) {
                        if (isCanceled()) {
                            mRet = false;
                            return;
                        }

                        Annot annot = pdfPage.getAnnot(i);
                        if (annot != null && annot.getType() == Annot.e_FileAttachment) {
                            FileAttachment attachment = new FileAttachment(annot);
                            FileSpec fileSpec = attachment.getFileSpec();
                            if (!fileSpec.isEmpty()) {
                                FileBean item = new FileBean();
                                item.setTitle(fileSpec.getFileName());
                                item.setName(fileSpec.getFileName());
                                item.setDate(getModifiedDateTimeString(fileSpec));
                                item.setSize(AppFileUtil.formatFileSize(fileSpec.getFileSize()));
                                item.setFlag(FileAttachmentAdapter.FLAG_ANNOT);
                                item.setDesc(attachment.getContent());
                                item.setUuid(AppAnnotUtil.getAnnotUniqueID(attachment));
                                item.setPageIndex(attachment.getPage().getIndex());

                                if (GroupManager.getInstance().isGrouped(mPdfViewCtrl, attachment)) {
                                    item.setCanDelete(false);
                                    item.setCanFlatten(false);
                                } else {
                                    item.setCanDelete(!(AppAnnotUtil.isLocked(attachment) || AppAnnotUtil.isReadOnly(attachment)));
                                    item.setCanFlatten(true);
                                }
                                item.setCanComment(!(AppAnnotUtil.isLocked(attachment) || AppAnnotUtil.isReadOnly(attachment)));
                                DocumentManager dm = mUiExtensionsManager.getDocumentManager();
                                item.setWithModificationPermission(AnnotPermissionUtil.canModifyAnnot(dm, attachment));
                                item.setWithDeletePermission(AnnotPermissionUtil.canDeleteAnnot(dm, annot));
                                annots.add(item);
                            }
                        }
                    }
                    if (annots.size() > 0) {
                        annots.get(0).setShowLable(true);
                        annots.get(0).setCount(annots.size());
                        mAnnotList.addAll(annots);
                    }
                    mRet = true;
                } catch (Exception e) {
                    mRet = false;
                }
            }
        }

        boolean isRunning() {
            return mIsRunning;
        }
    }

    private String getModifiedDateTimeString(FileSpec fileSpec) {
        String date;
        try {
            date = AppDmUtil.formatDocumentDate(AppDmUtil.FORMAT_MMM_DD_YYYY, fileSpec.getModifiedDateTime());
        } catch (PDFException e) {
            date = AppDmUtil.DEFAULT_MMM_DD_YYYY;
        }
        return date;
    }

    boolean hasName(String name) {
        if (pdfNameTree == null) {
            createNameTree();
            if (pdfNameTree == null) return false;
        }
        try {
            return pdfNameTree.hasName(name);
        } catch (PDFException e) {
            e.printStackTrace();
        }
        return false;
    }

    public void add(String name, String path) {
        if (pdfNameTree == null) {
            createNameTree();
            if (pdfNameTree == null) return;
        }
        try {
            FileSpec pNewFile = new FileSpec(mPdfViewCtrl.getDoc());
            pNewFile.setFileName(name);
            pNewFile.embed(path);
            pNewFile.setCreationDateTime(AppDmUtil.currentDateToDocumentDate());
            pNewFile.setModifiedDateTime(AppDmUtil.javaDateToDocumentDate(new File(path).lastModified()));
            PDFDictionary dict = pNewFile.getDict();
            pdfNameTree.add(pNewFile.getFileName(), dict);

            FileBean item = new FileBean();
            item.setTitle(pNewFile.getFileName());
            item.setName(pNewFile.getFileName());
            item.setDate(getModifiedDateTimeString(pNewFile));
            item.setSize(AppFileUtil.formatFileSize(pNewFile.getFileSize()));
            item.setFlag(FileAttachmentAdapter.FLAG_NORMAL);
            item.setDesc(pNewFile.getDescription());

            int addIndex = mDocAttachmentList.size();
            mTotalList.add(addIndex, item);
            viewer.notifyUpdateData(FileAttachmentViewer.ADD, addIndex, mTotalList);

            mDocAttachmentList.add(item);
            FileBean labelBean = mDocAttachmentList.get(0);
            labelBean.setShowLable(true);
            labelBean.setCount(mDocAttachmentList.size());
            if (mDocAttachmentList.size() > 1)
                viewer.notifyUpdateData(FileAttachmentViewer.MODIFY, mTotalList.indexOf(labelBean), mTotalList);
        } catch (PDFException e) {
            viewer.fail(FileAttachmentViewer.ADD);
        }
    }

    public void add(Annot annot) {
        try {
            if (annot == null || annot.isEmpty()) return;

            int pageIndex = annot.getPage().getIndex();
            if (!isPageLoaded(pageIndex)) return;

            FileSpec pNewFile = ((FileAttachment) annot).getFileSpec();
            FileBean item = new FileBean();
            item.setTitle(pNewFile.getFileName());
            item.setName(pNewFile.getFileName());
            item.setDate(getModifiedDateTimeString(pNewFile));
            item.setSize(AppFileUtil.formatFileSize(pNewFile.getFileSize()));
            item.setFlag(FileAttachmentAdapter.FLAG_ANNOT);
            item.setDesc(annot.getContent());
            item.setUuid(AppAnnotUtil.getAnnotUniqueID(annot));
            item.setPageIndex(pageIndex);
            if (GroupManager.getInstance().isGrouped(mPdfViewCtrl, annot)) {
                item.setCanDelete(false);
                item.setCanFlatten(false);
            } else {
                item.setCanDelete(!(AppAnnotUtil.isLocked(annot) || AppAnnotUtil.isReadOnly(annot)));
                item.setCanFlatten(true);
            }
            item.setCanComment(!(AppAnnotUtil.isLocked(annot) || AppAnnotUtil.isReadOnly(annot)));
            item.setWithModificationPermission(AnnotPermissionUtil.canModifyAnnot(mUiExtensionsManager.getDocumentManager(), annot));
            item.setWithDeletePermission(AnnotPermissionUtil.canDeleteAnnot(mUiExtensionsManager.getDocumentManager(), annot));
            int addIndex = getAddIndex(pageIndex);
            mTotalList.add(addIndex, item);
            viewer.notifyUpdateData(FileAttachmentViewer.ADD, addIndex, mTotalList);

            List<FileBean> pageAnnots = mPageAnnots.get(pageIndex);
            if (pageAnnots == null) {
                pageAnnots = new ArrayList<>();
                mPageAnnots.append(pageIndex, pageAnnots);
            }
            pageAnnots.add(item);
            FileBean labelItem = pageAnnots.get(0);
            labelItem.setShowLable(true);
            labelItem.setCount(pageAnnots.size());
            int lableIndex = mTotalList.indexOf(labelItem);
            if (pageAnnots.size() > 1)
                viewer.notifyUpdateData(FileAttachmentViewer.MODIFY, lableIndex, mTotalList);
        } catch (PDFException e) {
            viewer.fail(FileAttachmentViewer.ADD);
        }
    }

    public void update(final FileBean fileBean, final String content) {
        if (fileBean == null) return;

        try {
            //update file panel file
            if (fileBean.getFlag() == FileAttachmentAdapter.FLAG_NORMAL) {
                if (pdfNameTree == null) return;
                String name = fileBean.getName();
                PDFObject obj = pdfNameTree.getObj(name);
                FileSpec pNewFile = new FileSpec(mPdfViewCtrl.getDoc(), obj);
                pNewFile.setDescription(content);

                pdfNameTree.setObj(name, pNewFile.getDict());
                fileBean.setDesc(content);
                viewer.notifyUpdateData(FileAttachmentViewer.MODIFY, mTotalList.indexOf(fileBean), mTotalList);
            } else if (fileBean.getFlag() == FileAttachmentAdapter.FLAG_ANNOT) {
                //update annot file
                String uuid = fileBean.getUuid();
                Annot annot = mUiExtensionsManager.getDocumentManager().getAnnot(fileBean.getPageIndex(), uuid);
                mUiExtensionsManager.getDocumentManager().
                        modifyAnnot(annot, new FileAttachmentContent((FileAttachment) annot, content), true, new Event.Callback() {
                            @Override
                            public void result(Event event, boolean success) {
                                if (success) {
                                    fileBean.setDesc(content);
                                    viewer.notifyUpdateData(FileAttachmentViewer.MODIFY, mTotalList.indexOf(fileBean), mTotalList);
                                } else {
                                    viewer.fail(FileAttachmentViewer.MODIFY);
                                }
                            }
                        });
            }
        } catch (PDFException e) {
            viewer.fail(FileAttachmentViewer.MODIFY);
        }
    }

    public void updateByOutside(Annot annot) {
        try {
            List<FileBean> pageAnnots = mPageAnnots.get(annot.getPage().getIndex());
            if (pageAnnots == null) return;

            int size = pageAnnots.size();
            for (int i = 0; i < size; i++) {
                FileBean fileBean = pageAnnots.get(i);
                if (!AppUtil.isBlank(fileBean.getUuid()) && fileBean.getUuid().equals(AppAnnotUtil.getAnnotUniqueID(annot))) {
                    if (!fileBean.getDesc().equals(annot.getContent())) {
                        fileBean.setDesc(annot.getContent());
                        int index = mTotalList.indexOf(fileBean);
                        viewer.notifyUpdateData(FileAttachmentViewer.MODIFY, index, mTotalList);
                    }
                    break;
                }
            }
        } catch (PDFException e) {
            viewer.fail(FileAttachmentViewer.MODIFY);
        }
    }

    private int getAddIndex(int pageIndex) {
        int index = 0;
        for (int i = 0; i <= pageIndex; i++) {
            List<FileBean> pageAnnots = mPageAnnots.get(pageIndex);
            if (pageAnnots == null) continue;
            index += mPageAnnots.get(i).size();
        }
        return index + mDocAttachmentList.size();
    }

    private static class FileAttachmentContent implements IFileAttachmentAnnotContent {

        private final FileAttachment mAttachment;
        private final String mContent;

        private FileAttachmentContent(FileAttachment fileAttachment, String content) {
            mAttachment = fileAttachment;
            mContent = content;
        }

        @Override
        public String getIconName() {
            try {
                return mAttachment.getIconName();
            } catch (PDFException e) {
                e.printStackTrace();
            }
            return null;
        }

        @Override
        public String getFilePath() {
            return null;
        }

        @Override
        public String getFileName() {
            try {
                return mAttachment.getFileSpec().getFileName();
            } catch (PDFException e) {
                e.printStackTrace();
            }
            return null;
        }

        @Override
        public int getPageIndex() {
            try {
                return mAttachment.getPage().getIndex();
            } catch (PDFException e) {
                e.printStackTrace();
            }
            return 0;
        }

        @Override
        public int getType() {
            try {
                return mAttachment.getType();
            } catch (PDFException e) {
                e.printStackTrace();
            }
            return Annot.e_UnknownType;
        }

        @Override
        public String getNM() {
            return AppAnnotUtil.getAnnotUniqueID(mAttachment);
        }

        @Override
        public RectF getBBox() {
            try {
                return AppUtil.toRectF(mAttachment.getRect());
            } catch (PDFException e) {
                e.printStackTrace();
            }
            return null;
        }

        @Override
        public int getColor() {
            try {
                return mAttachment.getBorderColor();
            } catch (PDFException e) {
                e.printStackTrace();
            }
            return 0;
        }

        @Override
        public int getFillColor() {
            return 0;
        }

        @Override
        public int getOpacity() {
            try {
                return (int) (mAttachment.getOpacity() * 255f + 0.5f);
            } catch (PDFException e) {
                e.printStackTrace();
            }
            return 0;
        }

        @Override
        public float getLineWidth() {
            try {
                if (mAttachment.getBorderInfo() != null) {
                    return mAttachment.getBorderInfo().getWidth();
                }
            } catch (PDFException e) {
                e.printStackTrace();
            }
            return 0;
        }

        @Override
        public String getSubject() {
            try {
                return mAttachment.getSubject();
            } catch (PDFException e) {
                e.printStackTrace();
            }
            return null;
        }

        @Override
        public DateTime getModifiedDate() {
            return AppDmUtil.currentDateToDocumentDate();
        }

        @Override
        public String getContents() {
            return mContent;
        }

        @Override
        public String getIntent() {
            try {
                return mAttachment.getIntent();
            } catch (PDFException e) {
                e.printStackTrace();
            }
            return null;
        }

        @Override
        public String getAuthor() {
            return null;
        }
    }

    public void groupAnnot(Annot annot) {
        try {
            List<FileBean> pageAnnots = mPageAnnots.get(annot.getPage().getIndex());
            if (pageAnnots == null) return;

            int size = pageAnnots.size();
            for (int i = 0; i < size; i++) {
                FileBean b = pageAnnots.get(i);
                if (!AppUtil.isBlank(b.getUuid()) && b.getUuid().equals(AppAnnotUtil.getAnnotUniqueID(annot))) {
                    if (GroupManager.getInstance().isGrouped(mPdfViewCtrl, annot)) {
                        b.setCanDelete(false);
                        b.setCanFlatten(false);
                    } else {
                        b.setCanDelete(!(AppAnnotUtil.isLocked(annot) || AppAnnotUtil.isReadOnly(annot)));
                        b.setCanFlatten(true);
                    }
                    b.setCanComment(!(AppAnnotUtil.isLocked(annot) || AppAnnotUtil.isReadOnly(annot)));

                    int index = mTotalList.indexOf(b);
                    viewer.notifyUpdateData(FileAttachmentViewer.MODIFY, index, mTotalList);
                    break;
                }
            }
        } catch (PDFException e) {
            viewer.fail(FileAttachmentViewer.GROUP);
        }
    }

    public void delete(List<FileBean> fileBeans) {
        delete(0, fileBeans);
    }

    public void delete(final FileBean fileBean) {
        if (fileBean == null) return;

        List<FileBean> fileBeans = new ArrayList<>();
        fileBeans.add(fileBean);
        delete(0, fileBeans);
    }

    private void delete(final int index, final List<FileBean> fileBeans) {
        viewer.showProgressDlg(FileAttachmentViewer.OPER_DELETE);
        if (index < fileBeans.size()) {
            final FileBean fileBean = fileBeans.get(index);
            if (fileBean.getFlag() == FileAttachmentAdapter.FLAG_NORMAL) {
                try {
                    if (pdfNameTree == null) return;
                    if (mUiExtensionsManager.canModifyContents()&&mUiExtensionsManager.isEnableModification()) {
                        if (pdfNameTree.removeObj(fileBean.getName())) {
                            removeItemBean(fileBean);
                            mUiExtensionsManager.setSaveOptions(new UIExtensionsManager.SaveOptions(PDFDoc.e_SaveFlagRemoveRedundantObjects));

                            delete(index + 1, fileBeans);
                        }
                    } else {
                        viewer.fail(FileAttachmentViewer.DELETE);
                        delete(index + 1, fileBeans);
                    }
                } catch (PDFException e) {
                    viewer.fail(FileAttachmentViewer.DELETE);
                    delete(index + 1, fileBeans);
                }
            } else if (fileBean.getFlag() == FileAttachmentAdapter.FLAG_ANNOT) {
                if (mUiExtensionsManager.getDocumentManager().canAddAnnot() && mUiExtensionsManager.isEnableModification() && fileBean.canDelete()) {
                    Annot annot = mUiExtensionsManager.getDocumentManager().getAnnot(fileBean.getPageIndex(), fileBean.getUuid());
                    mUiExtensionsManager.getDocumentManager().removeAnnot(annot, true, new Event.Callback() {
                        @Override
                        public void result(Event event, boolean success) {
                            if (success)
                                removeItemBean(fileBean);
                            else
                                viewer.fail(FileAttachmentViewer.DELETE);
                            delete(index + 1, fileBeans);
                        }
                    });
                } else {
                    viewer.fail(FileAttachmentViewer.DELETE);
                    delete(index + 1, fileBeans);
                }
            }
        } else {
            viewer.dismissProgressDlg(FileAttachmentViewer.OPER_DELETE);
            fileBeans.clear();
        }
    }

    public void removeItemBean(int pageIndex, String uuid) {
        List<FileBean> pageAnnots = mPageAnnots.get(pageIndex);
        if (pageAnnots == null) return;

        for (FileBean fileBean : pageAnnots) {
            if (fileBean.getPageIndex() == pageIndex
                    && fileBean.getUuid().equals(uuid)) {
                removeItemBean(fileBean);
                return;
            }
        }
    }

    private void removeItemBean(FileBean fileBean) {
        if (!mTotalList.contains(fileBean)) return;

        int index = mTotalList.indexOf(fileBean);
        mTotalList.remove(fileBean);
        viewer.notifyUpdateData(FileAttachmentViewer.DELETE, index, mTotalList);

        if (fileBean.getFlag() == FileAttachmentAdapter.FLAG_NORMAL) {
            mDocAttachmentList.remove(fileBean);
            if (mDocAttachmentList.size() > 0) {
                FileBean labelItem = mDocAttachmentList.get(0);
                labelItem.setShowLable(true);
                labelItem.setCount(mDocAttachmentList.size());
                viewer.notifyUpdateData(FileAttachmentViewer.MODIFY, mTotalList.indexOf(labelItem), mTotalList);
            }

        } else if (fileBean.getFlag() == FileAttachmentAdapter.FLAG_ANNOT) {
            int pageIndex = fileBean.getPageIndex();
            List<FileBean> pageAnnots = mPageAnnots.get(pageIndex);
            if (pageAnnots != null) {
                pageAnnots.remove(fileBean);
                if (pageAnnots.size() > 0) {
                    FileBean labelItem = pageAnnots.get(0);
                    labelItem.setShowLable(true);
                    labelItem.setCount(pageAnnots.size());
                    viewer.notifyUpdateData(FileAttachmentViewer.MODIFY, mTotalList.indexOf(labelItem), mTotalList);
                }
            }
        }
    }

    public void unGroupAnnot(FileBean fileBean) {
        List<FileBean> pageAnnots = mPageAnnots.get(fileBean.getPageIndex());
        if (pageAnnots == null) return;

        for (FileBean b : pageAnnots) {
            if (!AppUtil.isBlank(b.getUuid()) && b.getUuid().equals(fileBean.getUuid())) {
                b.setCanDelete(fileBean.canDelete());
                b.setCanFlatten(true);

                int index = mTotalList.indexOf(b);
                viewer.notifyUpdateData(FileAttachmentViewer.MODIFY, index, mTotalList);
                break;
            }
        }
    }

    public void flatten(final FileBean fileBean) {
        final int pageIndex = fileBean.getPageIndex();
        String uuid = fileBean.getUuid();
        Annot annot = mUiExtensionsManager.getDocumentManager().getAnnot(pageIndex, uuid);
        if (annot == null || annot.isEmpty() || !(annot instanceof FileAttachment)) return;

        mUiExtensionsManager.getDocumentManager().flattenAnnot(annot, new Event.Callback() {
            @Override
            public void result(Event event, boolean success) {
                if (success)
                    removeItemBean(fileBean);
                else
                    viewer.fail(FileAttachmentViewer.FLATTEN);
            }
        });
    }

    public void save(FileBean fileBean, final String path) {
        if (fileBean == null) return;

        if (fileBean.getFlag() == FileAttachmentAdapter.FLAG_NORMAL) {
            try {
                if (pdfNameTree == null) return;
                FileSpec fileSpec = new FileSpec(mPdfViewCtrl.getDoc(), pdfNameTree.getObj(fileBean.getName()));
                FileAttachmentUtil.saveAttachment(mPdfViewCtrl, path, fileSpec, new Event.Callback() {
                    @Override
                    public void result(Event event, boolean success) {
                    }

                });
            } catch (PDFException e) {
                e.printStackTrace();
            }
        } else if (fileBean.getFlag() == FileAttachmentAdapter.FLAG_ANNOT) {
            try {
                String uuid = fileBean.getUuid();
                int pageIndex = fileBean.getPageIndex();
                Annot annot = mUiExtensionsManager.getDocumentManager().getAnnot(pageIndex, uuid);
                if (annot == null || annot.isEmpty() || !(annot instanceof FileAttachment)) return;

                FileSpec fileSpec = ((FileAttachment) annot).getFileSpec();
                FileAttachmentUtil.saveAttachment(mPdfViewCtrl, path, fileSpec, new Event.Callback() {
                    @Override
                    public void result(Event event, boolean success) {
                    }

                });
            } catch (PDFException e) {
                e.printStackTrace();
            }
        }
    }

    public void open(final FileBean fileBean, String p) {
        if (fileBean == null) return;

        viewer.showProgressDlg(FileAttachmentViewer.OPER_OPEN_DOC);
        final String path = p + fileBean.getTitle();
        if (fileBean.getFlag() == FileAttachmentAdapter.FLAG_NORMAL) {
            try {
                if (pdfNameTree == null) return;
                final FileSpec fileSpec = new FileSpec(mPdfViewCtrl.getDoc(), pdfNameTree.getObj(fileBean.getName()));
                FileAttachmentUtil.saveAttachment(mPdfViewCtrl, path, fileSpec, new Event.Callback() {
                    @Override
                    public void result(Event event, boolean success) {
                        if (success) {
                            String ExpName = path.substring(path.lastIndexOf('.') + 1).toLowerCase();
                            if (ExpName.equals("pdf") || ExpName.equals("ppdf")) {
                                viewer.open(path, fileBean.getTitle());
                            } else {
                                viewer.dismissProgressDlg(FileAttachmentViewer.OPER_OPEN_DOC);
                                if (mPdfViewCtrl.getUIExtensionsManager() == null) return;
                                Activity context = mUiExtensionsManager.getAttachedActivity();
                                if (context == null) return;
                                AppIntentUtil.openFile(context, path);
                            }
                        }
                    }

                });

            } catch (PDFException e) {
                e.printStackTrace();
            }
        } else if (fileBean.getFlag() == FileAttachmentAdapter.FLAG_ANNOT) {
            try {
                final int pageIndex = fileBean.getPageIndex();
                String uuid = fileBean.getUuid();
                Annot annot = mUiExtensionsManager.getDocumentManager().getAnnot(pageIndex, uuid);
                if (annot == null || annot.isEmpty() || !(annot instanceof FileAttachment)) return;

                FileSpec fileSpec = ((FileAttachment) annot).getFileSpec();
                FileAttachmentUtil.saveAttachment(mPdfViewCtrl, path, fileSpec, new Event.Callback() {
                    @Override
                    public void result(Event event, boolean success) {
                        if (success) {
                            String ExpName = path.substring(path.lastIndexOf('.') + 1).toLowerCase();
                            if (ExpName.equals("pdf")) {
                                viewer.open(path, fileBean.getTitle());
                            } else {
                                viewer.dismissProgressDlg(FileAttachmentViewer.OPER_OPEN_DOC);
                                if (mPdfViewCtrl.getUIExtensionsManager() == null) return;
                                Activity context = mUiExtensionsManager.getAttachedActivity();
                                if (context == null) return;
                                AppIntentUtil.openFile(context, path);
                            }
                        }
                    }

                });
            } catch (PDFException e) {
                e.printStackTrace();
            }
        }
    }

    private boolean mbPause;
    private int mPauseIndex;

    void setPause(boolean pause) {
        mbPause = pause;
    }

    void reStart() {
        mbPause = false;
        int pageCount = 0;
        try {
            pageCount = mPdfViewCtrl.getPageCount();
        } catch (Exception e) {
            mLoadedStatus = FileAttachmentViewer.STATUS_FAILED;
            viewer.fail(FileAttachmentViewer.STATUS_FAILED);
            return;
        }

        canelFileAttachmentTask();
        startSearchFileAttachment(mPauseIndex, pageCount);
    }

    int getCurrentStatus() {
        return mLoadedStatus;
    }

    synchronized void onPageRemoved(int index) {
        if (!isPageLoaded(index)) return;

        List<FileBean> removeBeans = mPageAnnots.get(index);
        mTotalList.removeAll(removeBeans);
        mPageAnnots.remove(index);

        for (int i = index; i < mPageAnnots.size(); i++) {
            int key = mPageAnnots.keyAt(i);
            List<FileBean> fileBeans = mPageAnnots.valueAt(i);
            for (FileBean bean : fileBeans) {
                bean.setPageIndex(i);
            }
            mPageAnnots.remove(key);
            mPageAnnots.append(i, fileBeans);
        }

        AppThreadManager.getInstance().getMainThreadHandler().post(new Runnable() {
            @Override
            public void run() {
                viewer.notifyUpdateData(FileAttachmentViewer.PAGE_REMOVED, -1, mTotalList);
            }
        });
    }

    synchronized void onPageMoved(int index, int dstIndex) {
        if (!isPageLoaded(index) || !isPageLoaded(dstIndex)) return;

        List<List<FileBean>> annotList = new ArrayList<>();
        for (int i = 0; i < mPageAnnots.size(); i++) {
            List<FileBean> fileBeans = mPageAnnots.valueAt(i);
            annotList.add(fileBeans);
        }

        if (index < dstIndex) {
            for (int i = index; i < dstIndex; i++) {
                Collections.swap(annotList, i, i + 1);
            }
        } else {
            for (int i = index; i > dstIndex; i--) {
                Collections.swap(annotList, i, i - 1);
            }
        }

        int start = Math.min(index, dstIndex);
        int end = Math.max(index, dstIndex) + 1;
        for (int i = start; i < end; i++) {
            List<FileBean> beans = annotList.get(i);
            if (beans != null) {
                for (FileBean bean : beans) {
                    bean.setPageIndex(i);
                }
            }
        }

        mPageAnnots.clear();
        mTotalList.clear();
        mTotalList.addAll(mDocAttachmentList);
        for (int i = 0; i < annotList.size(); i++) {
            List<FileBean> fileBeans = annotList.get(i);
            mPageAnnots.append(i, fileBeans);
            mTotalList.addAll(fileBeans);
        }

        AppThreadManager.getInstance().getMainThreadHandler().post(new Runnable() {
            @Override
            public void run() {
                viewer.notifyUpdateData(FileAttachmentViewer.PAGE_MOVED, -1, mTotalList);
            }
        });
    }

    private boolean isPageLoaded(int pageIndex) {
        return pageIndex < mLoadedIndex || mLoadedStatus == FileAttachmentViewer.STATUS_DONE;
    }

    /**
     * viewer
     */
    public interface FileAttachmentViewer {
        int ADD = 1;
        int MODIFY = 2;
        int DELETE = 3;
        int FLATTEN = 4;
        int GROUP = 5;
        int UNGROUP = 6;
        int PAGE_REMOVED = 7;
        int PAGE_MOVED = 8;

        int OPER_OPEN_DOC = 11;
        int OPER_DELETE = 22;

        int STATUS_PREPARE = 100;
        int STATUS_LOADING = 101;
        int STATUS_CANCEL = 102;
        int STATUS_PAUSED = 103;
        int STATUS_DONE = 104;
        int STATUS_FAILED = 105;

        void success(List<FileBean> list);

        void fail(int errCode);

        void loading(int pageIndex, int total);

        void loadDataChanged(List<FileBean> list);

        void pause(int pageIndex);

        void showProgressDlg(int type);

        void dismissProgressDlg(int type);

        void open(String path, String name);

        void notifyUpdateData(int type, int index, List<FileBean> items);
    }
}
