/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.modules.panel.outline;

import android.annotation.SuppressLint;
import android.content.Context;
import android.graphics.PointF;
import android.os.Handler;
import android.os.Message;
import android.text.Spannable;
import android.text.SpannableStringBuilder;
import android.text.style.ForegroundColorSpan;
import android.view.LayoutInflater;
import android.view.MotionEvent;
import android.view.View;
import android.view.View.OnClickListener;
import android.view.ViewGroup;
import android.widget.CheckBox;
import android.widget.ImageView;
import android.widget.LinearLayout;
import android.widget.TextView;

import com.foxit.sdk.PDFException;
import com.foxit.sdk.PDFViewCtrl;
import com.foxit.sdk.Task;
import com.foxit.sdk.common.Constants;
import com.foxit.sdk.pdf.Bookmark;
import com.foxit.sdk.pdf.PDFDoc;
import com.foxit.sdk.pdf.actions.Destination;
import com.foxit.uiextensions.R;
import com.foxit.uiextensions.UIExtensionsManager;
import com.foxit.uiextensions.browser.adapter.SuperAdapter;
import com.foxit.uiextensions.browser.adapter.viewholder.SuperViewHolder;
import com.foxit.uiextensions.modules.panel.bean.BaseBean;
import com.foxit.uiextensions.theme.ThemeConfig;
import com.foxit.uiextensions.theme.ThemeUtil;
import com.foxit.uiextensions.utils.AppResource;
import com.foxit.uiextensions.utils.AppUtil;
import com.foxit.uiextensions.utils.IResult;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;

import androidx.annotation.NonNull;
import androidx.recyclerview.widget.ItemTouchHelper;

public abstract class OutlineSupport {
    private static final int UPDATEUI = 100;

    public static final int STATE_NORMAL = 0;
    public static final int STATE_LOADING = 1;
    public static final int STATE_LOAD_FINISH = 2;

    private ArrayList<OutlineNode> mAllOutlineList = new ArrayList<>();
    private ArrayList<OutlineNode> mShowOutlineList = new ArrayList<>();

    private Context mContext;
    private PDFViewCtrl mPDFViewCtrl;
    private MyHandler mHandler;
    private OutlineAdapter mAdapter;
    private LoadAllOutlinesTask mLoadAllOutlinesTask;
    private OnItemClickListener mItemClickListener;

    private boolean mIsNeedLoadAllOutlines = true;
    private int mLevel = 0;
    private int mCurrentState = STATE_NORMAL;
    private OutlineNode mRootOutlineNode;

    int getCurrentState() {
        return mCurrentState;
    }

    public OutlineSupport(Context context, PDFViewCtrl pdfViewCtrl, ImageView back) {
        mContext = context;
        mPDFViewCtrl = pdfViewCtrl;
        mHandler = new MyHandler();
        mAdapter = new OutlineAdapter(context);
        outlineBindingListView(mAdapter);

        back.setOnClickListener(new OnClickListener() {
            @Override
            public void onClick(View v) {
                mAdapter.getEditItems().clear();
                OutlineNode parentNode = mCurrentParentNode;
                if (parentNode != null) {
                    OutlineNode parent = parentNode.mParent;
                    mCurrentParentNode = parent;
                    if (parent != null) {
                        mLevel = mLevel - 1;
                        mShowOutlineList.clear();
                        if (mMovedParentList.contains(parent)){
                            refreshCurrentList();
                            mMovedParentList.remove(parent);
                            return;
                        }
                        List<OutlineNode> parentNodes = parent.mChildren;
                        mShowOutlineList.addAll(parentNodes);
                        setShowOutline(mShowOutlineList);
                        updateUI(mLevel, STATE_NORMAL);
                        mAdapter.notifyDataSetChanged();
                    }
                }
            }
        });
    }

    void init() {
        try {
            if (mPDFViewCtrl.getDoc() == null) return;
            Bookmark bookmark = mPDFViewCtrl.getDoc().getRootBookmark();
            mLevel = -1;
            if (bookmark == null || bookmark.isEmpty()) {
                updateUI(mAllOutlineList, STATE_LOAD_FINISH);
                return;
            }
            if (bookmark.hasChild()) {
                Bookmark firstBookmark = bookmark.getFirstChild();
                if (firstBookmark == null || firstBookmark.isEmpty()) {
                    updateUI(mAllOutlineList, STATE_LOAD_FINISH);
                    return;
                }
            }
            mCurrentState = STATE_LOADING;
            updateUI(mLevel, STATE_LOADING);

            mRootOutlineNode = createRootOutlineNode();
            getOutList(mRootOutlineNode);
        } catch (PDFException e) {
            e.printStackTrace();
        }
    }

    OutlineNode createRootOutlineNode(){
        try {
            Bookmark bookmark = mPDFViewCtrl.getDoc().getRootBookmark();
            if (bookmark.isEmpty())
                bookmark = mPDFViewCtrl.getDoc().createRootBookmark();
            OutlineNode rootOutlineNode = new OutlineNode();
            rootOutlineNode.mBookmark = bookmark;
            rootOutlineNode.mId = "0";
            rootOutlineNode.mLevel = 0;
            rootOutlineNode.mIsExpanded = true;
            rootOutlineNode.mIsLeaf = !bookmark.hasChild();
            return rootOutlineNode;
        } catch (PDFException e) {
            e.printStackTrace();
        }
        return null;
    }

    void gotoRootOutlineNode() {
        if (mRootOutlineNode == null || mLevel == 0) return;
        mLevel = 0;
        mShowOutlineList.clear();
        mCurrentParentNode = mRootOutlineNode;
        if (mMovedParentList.contains(mCurrentParentNode)){
            refreshCurrentList();
            mMovedParentList.remove(mCurrentParentNode);
            return;
        }
        List<OutlineNode> rootNodes = mRootOutlineNode.mChildren;
        mShowOutlineList.addAll(rootNodes);
        setShowOutline(mShowOutlineList);
        updateUI(mLevel, STATE_NORMAL);
        mAdapter.notifyDataSetChanged();
    }

    ArrayList<OutlineNode> getShowOutLineList() {
        return mShowOutlineList;
    }

    List<OutlineNode> getAllOutLineItems() {
        return mAllOutlineList;
    }

    class OutlineTask extends Task {
        OutlineNode mOutlineNode;
        boolean bSuccess;

        public OutlineTask(OutlineNode outlineNode, CallBack callBack) {
            super(callBack);
            mOutlineNode = outlineNode;
        }

        @Override
        protected void execute() {
            try {
                if (mOutlineNode == null || mOutlineNode.mBookmark == null || mOutlineNode.mBookmark.isEmpty()) {
                    bSuccess = true;
                    return;
                }
                Bookmark rootBookmark = mOutlineNode.mBookmark;
                Bookmark current = rootBookmark.getFirstChild();

                int position = 0;
                while (!isCanceled() && current != null && !current.isEmpty()) {
                    position++;

                    OutlineNode childItem = new OutlineNode();

                    childItem.mParent = mOutlineNode;
                    childItem.mId = mOutlineNode.mId + "-" + position;
                    childItem.mPid = mOutlineNode.mId;
                    childItem.mPosition = position;
                    childItem.mLevel = mOutlineNode.mLevel + 1;
                    childItem.mIsLeaf = !current.hasChild();

                    childItem.mTitle = current.getTitle();
                    childItem.mBookmark = current;
                    Destination dest = current.getDestination();
                    if (dest != null && !dest.isEmpty()) {
                        childItem.mPageIndex = dest.getPageIndex(mPDFViewCtrl.getDoc());

                        PointF destPt = AppUtil.getDestinationPoint(mPDFViewCtrl.getDoc(), dest);
                        childItem.mX = destPt.x;
                        childItem.mY = destPt.y;
                    }
                    current = current.getNextSibling();
                    mOutlineNode.mChildren.add(childItem);
                }
                bSuccess = true;
            } catch (PDFException e) {
                if (e.getLastError() == Constants.e_ErrOutOfMemory) {
                    mPDFViewCtrl.recoverForOOM();
                    return;
                }
                bSuccess = false;
            }
        }
    }

    private Task getOutlineInfo(OutlineNode outlineNode, final IResult<Void, Void, OutlineNode> result) {
        Task.CallBack callBack = new Task.CallBack() {
            @Override
            public void result(Task task) {
                OutlineTask task1 = (OutlineTask) task;
                if (result != null) {
                    result.onResult(task1.bSuccess, null, null, task1.mOutlineNode);
                }
            }
        };

        Task task = new OutlineTask(outlineNode, callBack);
        mPDFViewCtrl.addTask(task);
        return task;
    }

    void loadAllOutlines(final IResult<Void, Void, List<OutlineNode>> result) {
        if (mIsNeedLoadAllOutlines) {
            mAllOutlineList.clear();
            updateUI(mAllOutlineList, STATE_LOADING);

            Task.CallBack callBack = new Task.CallBack() {
                @Override
                public void result(Task task) {
                    LoadAllOutlinesTask task1 = (LoadAllOutlinesTask) task;
                    mAllOutlineList.addAll(task1.mOutlines);
                    if (result != null) {
                        result.onResult(task1.mRet, null, null, mAllOutlineList);
                    }
                    updateUI(mAllOutlineList, STATE_LOAD_FINISH);
                    mIsNeedLoadAllOutlines = false;
                    mLoadAllOutlinesTask = null;
                }
            };

            mLoadAllOutlinesTask = new LoadAllOutlinesTask(mPDFViewCtrl.getDoc(), callBack);
            mPDFViewCtrl.addTask(mLoadAllOutlinesTask);
        } else {
            if (result != null) {
                result.onResult(true, null, null, mAllOutlineList);
            }
        }
    }

    void cancelLoadAllOutlinesTask() {
        if (mLoadAllOutlinesTask != null)
            mPDFViewCtrl.removeTask(mLoadAllOutlinesTask);
    }

    void removeBookmark(final List<OutlineNode> outlineItems, final IResult<Void, Integer, ArrayList<OutlineNode>> result) {
        if (outlineItems == null || outlineItems.size() == 0) return;

        Task.CallBack callBack = new Task.CallBack() {
            @Override
            public void result(Task task) {
                mIsNeedLoadAllOutlines = true;

                OutlineNode parentNode = outlineItems.get(0).mParent;
                if (parentNode != null) {
                    parentNode.mChildren.removeAll(outlineItems);
                    parentNode.mIsLeaf = parentNode.mChildren.size() == 0;
                    parentNode.mIsExpanded = false;
                }

                mShowOutlineList.removeAll(outlineItems);
                outlineItems.clear();

                if (result != null) {
                    RemoveBookmarkTask removeTask = (RemoveBookmarkTask) task;
                    result.onResult(removeTask.mRet, null, mLevel, mShowOutlineList);
                }
            }
        };
        RemoveBookmarkTask removeBookmarkTask = new RemoveBookmarkTask(mPDFViewCtrl.getDoc(), outlineItems, callBack);
        mPDFViewCtrl.addTask(removeBookmarkTask);
    }

    private static class RemoveBookmarkTask extends Task {

        private PDFDoc mDoc;
        private List<OutlineNode> mItems;
        private boolean mRet;

        public RemoveBookmarkTask(PDFDoc doc, List<OutlineNode> outlineItems, CallBack callBack) {
            super(callBack);
            this.mDoc = doc;
            this.mItems = outlineItems;
        }

        @Override
        protected void execute() {
            if (mDoc == null || mItems == null) {
                mRet = false;
                return;
            }

            try {
                for (OutlineNode outlineItem : mItems) {
                    Bookmark bookmark = outlineItem.mBookmark;
                    if (bookmark == null || bookmark.isEmpty()) continue;
                    mDoc.removeBookmark(outlineItem.mBookmark);
                }
                mRet = true;
            } catch (PDFException e) {
                mRet = false;
            }
        }
    }

    private void updateUI(ArrayList<OutlineNode> lineList, int state) {
        mCurrentState = state;
        setShowOutline(lineList);
        if (mAdapter != null) {
            Message msg = new Message();
            msg.arg1 = state;
            msg.what = UPDATEUI;
            mHandler.sendMessage(msg);
        }
    }

    private static class LoadAllOutlinesTask extends Task {

        private boolean mRet;
        private PDFDoc mPDFDoc;
        private Bookmark mRootBookmark;
        private ArrayList<OutlineNode> mOutlines;

        public LoadAllOutlinesTask(PDFDoc pdfDoc, Task.CallBack callBack) {
            super(callBack);
            mPDFDoc = pdfDoc;
            try {
                if (pdfDoc != null)
                    mRootBookmark = pdfDoc.getRootBookmark();
            } catch (PDFException e) {
                e.printStackTrace();
            }
            mOutlines = new ArrayList<>();
        }

        @Override
        protected void execute() {
            if (mPDFDoc == null || mRootBookmark == null || mRootBookmark.isEmpty()) {
                mRet = false;
                return;
            }
            loadChildBookmark(mRootBookmark);
        }

        private void loadChildBookmark(Bookmark rootBookmark) {
            try {
                Bookmark current = rootBookmark.getFirstChild();
                while (current != null && !current.isEmpty()) {
                    OutlineNode outlineItem = new OutlineNode();

                    outlineItem.mTitle = current.getTitle();
                    outlineItem.mBookmark = current;

                    Destination dest = current.getDestination();
                    if (dest != null && !dest.isEmpty()) {
                        outlineItem.mPageIndex = dest.getPageIndex(mPDFDoc);

                        PointF destPt = AppUtil.getDestinationPoint(mPDFDoc, dest);
                        outlineItem.mX = destPt.x;
                        outlineItem.mY = destPt.y;
                    }
                    boolean hasChild = current.hasChild();
                    outlineItem.mIsLeaf = !hasChild;
                    mOutlines.add(outlineItem);
                    if (hasChild) {
                        loadChildBookmark(current);
                    }
                    current = current.getNextSibling();
                }
                mRet = true;
            } catch (PDFException e) {
                mRet = false;
            }
        }
    }

    private Task mGetOutlineTask;
    private OutlineNode mCurrentParentNode;

    private void getOutList(OutlineNode outlineNode) {
        mCurrentParentNode = outlineNode;
        if (outlineNode == null || outlineNode.mBookmark == null || outlineNode.mBookmark.isEmpty()) {
            updateUI(new ArrayList<OutlineNode>(), STATE_LOAD_FINISH);
            return;
        }
        if (!outlineNode.mIsLeaf) {
            if (outlineNode.mChildren.size() == 0) {
                if (mGetOutlineTask != null){
                    mPDFViewCtrl.removeTask(mGetOutlineTask);
                }
                mGetOutlineTask = getOutlineInfo(outlineNode, new IResult<Void, Void, OutlineNode>() {
                    @Override
                    public void onResult(boolean success, Void p1, Void p2, OutlineNode outlineItem) {
                        if (success) {
                            mShowOutlineList.clear();
                            mLevel = outlineItem.mLevel;
                            mShowOutlineList.addAll(outlineItem.mChildren);
                            updateUI(mShowOutlineList, STATE_LOAD_FINISH);
                        }
                    }
                });
            } else {
                mLevel = outlineNode.mLevel;
                mShowOutlineList.clear();
                mShowOutlineList.addAll(outlineNode.mChildren);
                updateUI(mShowOutlineList, STATE_LOAD_FINISH);
            }
        } else {
            updateUI(new ArrayList<OutlineNode>(), STATE_LOAD_FINISH);
        }
    }

    abstract void outlineBindingListView(OutlineAdapter adapter);

    abstract void setShowOutline(ArrayList<OutlineNode> mOutlineList);

    abstract void updateUI(int level, int state);

    class OutlineAdapter extends SuperAdapter<OutlineNode> {
        private String mSearchPattern;
        private boolean mIsEditState = false;
        private boolean mIsSearchState = false;
        private List<OutlineNode> mSearchItems;
        private List<OutlineNode> mEditItems;

        private OutlineAdapter(Context context) {
            super(context);
            mSearchItems = new ArrayList<>();
            mEditItems = new ArrayList<>();
        }

        List<OutlineNode> getSearchItems() {
            return mSearchItems;
        }

        List<OutlineNode> getEditItems() {
            return mEditItems;
        }

        List<OutlineNode> getSortedEditItems() {
            Collections.sort(mEditItems, new Comparator<OutlineNode>() {
                @Override
                public int compare(OutlineNode o1, OutlineNode o2) {
                    return o1.mPosition - o2.mPosition;
                }
            });
            return mEditItems;
        }

        void switchSearchState(boolean toSearthState) {
            mIsSearchState = toSearthState;
        }

        boolean isSearchState() {
            return mIsSearchState;
        }

        void switchEditState(boolean toEditState) {
            mIsEditState = toEditState;
        }

        boolean isEditState() {
            return mIsEditState;
        }

        void setSearchPattern(String Pattern) {
            mSearchPattern = Pattern;
        }

        boolean isSelectedAll() {
            return mEditItems.size() > 0 && mShowOutlineList.size() == mEditItems.size();
        }

        void selectAll(boolean select) {
            mEditItems.clear();
            if (select)
                mEditItems.addAll(mShowOutlineList);
            notifyDataSetChanged();
        }

        @Override
        public void notifyUpdateData() {
            notifyDataSetChanged();
        }

        @Override
        public OutlineNode getDataItem(int position) {
            if (isSearchState()) {
                return mSearchItems.get(position);
            } else {
                return mShowOutlineList.get(position);
            }
        }

        @Override
        public int getItemCount() {
            if (isSearchState())
                return mSearchItems != null ? mSearchItems.size() : 0;
            else
                return mShowOutlineList != null ? mShowOutlineList.size() : 0;
        }

        @Override
        public long getItemId(int position) {
            return position;
        }

        @NonNull
        @Override
        public SuperViewHolder onCreateViewHolder(@NonNull ViewGroup parent, int viewType) {
            return new OutlineViewHolder(LayoutInflater.from(getContext()).inflate(R.layout.rv_panel_outline_item, parent, false));
        }

        private class OutlineViewHolder extends SuperViewHolder {
            private LinearLayout mItemView;
            private TextView mChapterTv;
            private TextView mChapterIndexTv;
            private CheckBox mChapterCheck;
            private ImageView mMoreIv;
            private ImageView mDragIv;

            public OutlineViewHolder(View itemView) {
                super(itemView);
                mItemView = itemView.findViewById(R.id.panel_outline_item_container);
                mChapterTv = itemView.findViewById(R.id.panel_outline_item_chapter);
                mChapterIndexTv = itemView.findViewById(R.id.panel_outline_item_index);
                mChapterCheck = itemView.findViewById(R.id.rd_outline_item_checkbox);
                mMoreIv = itemView.findViewById(R.id.panel_outline_item_more);
                mDragIv = itemView.findViewById(R.id.panel_outline_item_drag);
                mMoreIv.setOnClickListener(this);
                mItemView.setOnClickListener(this);

                mChapterTv.setOnClickListener(this);
                mChapterIndexTv.setOnClickListener(this);
                mDragIv.setOnClickListener(this);
            }

            @Override
            public void bind(BaseBean data, int position) {
                OutlineNode outlineItem = (OutlineNode) data;
                mChapterIndexTv.setText(String.valueOf(outlineItem.mPageIndex + 1));
                mChapterIndexTv.setVisibility(outlineItem.mPageIndex >= 0 ? View.VISIBLE : View.GONE);
                mChapterTv.setClickable(isEditState());
                mChapterIndexTv.setClickable(isEditState());
                if (isSearchState()) {
                    mMoreIv.setVisibility(View.GONE);
                    mChapterCheck.setVisibility(View.GONE);
                    mDragIv.setVisibility(View.GONE);

                    Object text = mChapterTv.getText();
                    SpannableStringBuilder spannable;
                    if (text instanceof SpannableStringBuilder) {
                        spannable = (SpannableStringBuilder) text;
                        spannable.clearSpans();
                        spannable.clear();
                        spannable.append(outlineItem.mTitle);
                    } else {
                        spannable = new SpannableStringBuilder(outlineItem.mTitle);
                    }

                    if (!AppUtil.isEmpty(mSearchPattern)) {
                        int start = outlineItem.mTitle.toLowerCase().indexOf(mSearchPattern, -1);
                        if (start != -1) {
                            spannable.setSpan(new ForegroundColorSpan(ThemeConfig.getInstance(getContext()).getPrimaryColor()),
                                    start, start + mSearchPattern.length(),
                                    Spannable.SPAN_EXCLUSIVE_INCLUSIVE
                            );
                        }
                    }
                    mChapterTv.setText(spannable);
                } else if (isEditState()) {
                    mChapterCheck.setVisibility(View.VISIBLE);
                    mDragIv.setVisibility(mShowOutlineList.size() == 1 ? View.INVISIBLE : View.VISIBLE);
                    mChapterCheck.setChecked(mEditItems.contains(outlineItem));
                    mMoreIv.setVisibility(outlineItem.mIsLeaf ? View.INVISIBLE : View.VISIBLE);
                    mChapterTv.setText(outlineItem.mTitle);
                    ThemeUtil.setTintList(mChapterCheck, ThemeUtil.getCheckboxColor(getContext()));
                } else {
                    mChapterCheck.setVisibility(View.GONE);
                    mDragIv.setVisibility(View.GONE);
                    mMoreIv.setVisibility(outlineItem.mIsLeaf ? View.INVISIBLE : View.VISIBLE);
                    mChapterTv.setText(outlineItem.mTitle);
                }
                mItemView.setBackground(AppResource.getDrawable(mContext,R.drawable.rd_list_item_bg));
                ThemeUtil.setTintList(mMoreIv,ThemeUtil.getPrimaryIconColor(mContext));
                mChapterTv.setTextColor(AppResource.getColor(mContext,R.color.t4));
                mChapterIndexTv.setTextColor(AppResource.getColor(mContext,R.color.t3));
                bindDragIv();
            }

            @SuppressLint("ClickableViewAccessibility")
            private void bindDragIv() {
                if (mDragIv.getVisibility() == View.VISIBLE){
                    mDragIv.setOnTouchListener(new View.OnTouchListener() {
                        @Override
                        public boolean onTouch(View v, MotionEvent event) {
                            int actionMask = event.getActionMasked();
                            if(actionMask == MotionEvent.ACTION_DOWN){
                                if (mItemTouchHelper != null)
                                    mItemTouchHelper.startDrag(OutlineViewHolder.this);
                            }
                            return true;
                        }
                    });
                }
            }

            @Override
            public void onClick(View v) {
                int id = v.getId();
                int position = getAdapterPosition();
                OutlineNode outlineItem;
                if (id == R.id.panel_outline_item_container) {
                    if (isEditState()) {
                        outlineItem = mShowOutlineList.get(position);
                        if (mEditItems.contains(outlineItem))
                            mEditItems.remove(outlineItem);
                        else
                            mEditItems.add(outlineItem);
                        notifyItemChanged(position);
                    } else {
                        if (isSearchState())
                            outlineItem = mSearchItems.get(position);
                        else
                            outlineItem = mShowOutlineList.get(position);
                        int index = outlineItem.mPageIndex;
                        if (index < 0) return;
                        float x = outlineItem.mX;
                        float y = outlineItem.mY;
                        mPDFViewCtrl.gotoPage(index, new PointF(x, y));
                        if (((UIExtensionsManager) mPDFViewCtrl.getUIExtensionsManager()).getPanelManager().isShowingPanel())
                            ((UIExtensionsManager) mPDFViewCtrl.getUIExtensionsManager()).getPanelManager().hidePanel();
                    }

                    if (mItemClickListener != null)
                        mItemClickListener.onItemClick(position, outlineItem);
                } else if (id == R.id.panel_outline_item_more) {
                    mEditItems.clear();
                    OutlineNode currentNode = mShowOutlineList.get(position);
                    currentNode.mIsExpanded = true;
                    mCurrentState = STATE_LOADING;
                    getOutList(currentNode);
                }else if (isEditState() && (v == mChapterTv || v == mChapterIndexTv)){
                    mEditingOutlinePosition = position;
                    showEditOutlineDialog(v == mChapterTv);
                }
            }
        }
    }

    private class MyHandler extends Handler {
        @Override
        public void handleMessage(Message msg) {
            if (msg.what == UPDATEUI) {
                updateUI(mLevel, msg.arg1);
                mAdapter.notifyDataSetChanged();
            }
        }
    }

    void clear() {
        cancelLoadAllOutlinesTask();
        mIsNeedLoadAllOutlines = true;
        mAllOutlineList.clear();
        mShowOutlineList.clear();
        mMovedParentList.clear();
        mRootOutlineNode = null;
        mCurrentParentNode = null;
        mLevel = -1;
        cancelTask();
    }

    void setItemClickListener(OnItemClickListener itemClickListener) {
        mItemClickListener = itemClickListener;
    }

    interface OnItemClickListener {
        void onItemClick(int position, OutlineNode outlineItem);
    }

    public void appendOutline(int index, OutlineNode target) {
        if (target == null) return;
        mShowOutlineList.add(index, target);
        int size = mShowOutlineList.size();
        OutlineNode outlineNode;
        for (int i = index + 1; i < size; i++) {
            outlineNode = mShowOutlineList.get(i);
            outlineNode.mId = outlineNode.mPid + "-" + ++outlineNode.mPosition;
        }
        if (mRootOutlineNode == null){
            mRootOutlineNode = createRootOutlineNode();
            getOutList(mRootOutlineNode);
            setShowOutline(mShowOutlineList);
            mAdapter.notifyItemInserted(index);
            return;
        }
        updateListForParent(target.mParent);
        setShowOutline(mShowOutlineList);
        mAdapter.notifyItemInserted(index);
    }

    public void setMovedParents(OutlineNode fromParent, OutlineNode toParent) {
        mMovedParentList.clear();
        if (fromParent != null)
            mMovedParentList.add(fromParent);
        if (toParent != null)
            mMovedParentList.add(toParent);
    }

     void moveOutline(int fromPosition, int toPosition) {
         if (fromPosition == toPosition || mShowOutlineList == null || mShowOutlineList.isEmpty()) return;
         setShowOutline(mShowOutlineList);
         boolean isMovedUp = fromPosition > toPosition;
         updateListForParent(mShowOutlineList.get(0).mParent);
         OutlineNode from = mShowOutlineList.get(toPosition);
         OutlineNode to = mShowOutlineList.get(isMovedUp ? toPosition + 1 : toPosition -1);

         int startIndex = Math.min(fromPosition, toPosition);
         for (int i = startIndex; i < mShowOutlineList.size(); i++) {
             OutlineNode outlineNode = mShowOutlineList.get(i);
             outlineNode.mPosition = (i + 1);
             outlineNode.mId = outlineNode.mPid + "-" + outlineNode.mPosition;
         }
         try {
             Bookmark bookmark = from.mBookmark;
             if (bookmark == null || bookmark.isEmpty()) return;
             bookmark.moveTo(to.mBookmark, isMovedUp ?  Bookmark.e_PosPrevSibling : Bookmark.e_PosNextSibling);
             ((UIExtensionsManager)mPDFViewCtrl.getUIExtensionsManager()).getDocumentManager().setDocModified(true);
         } catch (PDFException e) {
             e.printStackTrace();
         }
    }

    private void updateListForParent(OutlineNode parent) {
        if (parent == null) return;
        if (parent.mChildren != null ){
            parent.mChildren.clear();
            parent.mChildren.addAll(mShowOutlineList);
            parent.mIsLeaf = parent.mChildren.isEmpty();
        }
    }

    private Task mGetOutlineInfoTask;

    public void getOutline(OutlineNode node, IResult<Void, Void, OutlineNode> iResult) {
        if (mGetOutlineInfoTask != null)
            mPDFViewCtrl.removeTask(mGetOutlineInfoTask);
        mGetOutlineInfoTask = getOutlineInfo(node, iResult);
    }

    private void cancelTask(){
        if (mGetOutlineTask != null)
            mPDFViewCtrl.removeTask(mGetOutlineTask);
        if (mGetOutlineInfoTask != null)
            mPDFViewCtrl.removeTask(mGetOutlineInfoTask);
    }

    void refreshCurrentList(){
        if (mCurrentParentNode == null) return;
        mCurrentParentNode.mChildren.clear();
        mAdapter.getEditItems().clear();
        getOutline(mCurrentParentNode, new IResult<Void, Void, OutlineNode>(){
            @Override
            public void onResult(boolean success, Void p1, Void p2, OutlineNode node) {
                mShowOutlineList.clear();
                mShowOutlineList.addAll(node.mChildren);
                setShowOutline(mShowOutlineList);
                updateUI(mShowOutlineList, STATE_LOAD_FINISH);
            }
        });
    }

    private List<OutlineNode> mMovedParentList = new ArrayList<>();

    void moveBookmark(OutlineNode from,OutlineNode to, int position) {
        try {
            Bookmark bookmark = from.mBookmark;
            if (bookmark == null || bookmark.isEmpty()) return;
            bookmark.moveTo(to.mBookmark, position);
        } catch (PDFException e) {
            e.printStackTrace();
        }
    }

    private AddOutlineDialog mEditOutlineDialog;
    private AddOutlineDialog.OnAddOutlineListener mEditOutlineListener;
    private int mEditingOutlinePosition = -1;

    private void showEditOutlineDialog(boolean firstOnFocus) {
        if (mEditOutlineDialog != null) {
            mEditOutlineDialog.dismiss();
        }
        mEditOutlineDialog = new AddOutlineDialog(mPDFViewCtrl.getAttachedActivity(), mPDFViewCtrl);
        mEditOutlineDialog.changeToEditUI();
        OutlineNode outlineNode = mShowOutlineList.get(mEditingOutlinePosition);
        mEditOutlineDialog.setOutlineValue(outlineNode.mTitle, outlineNode.mPageIndex, firstOnFocus);
        if (mEditOutlineListener == null)
            mEditOutlineListener = new AddOutlineDialog.OnAddOutlineListener() {
                @Override
                public void onAddOutline(String name, int pageIndex) {
                    editOutline(name, pageIndex);
                }
            };
        mEditOutlineDialog.setOnAddOutlineListener(mEditOutlineListener);
        mEditOutlineDialog.show();
    }

    private void editOutline(String name, int pageIndex) {
        if (mEditingOutlinePosition == -1) return;
        OutlineNode editingOutline = mShowOutlineList.get(mEditingOutlinePosition);
        try {
            Bookmark bookmark = editingOutline.mBookmark;
            boolean needUpdate = false;
            if (!name.contentEquals(editingOutline.mTitle)){
                bookmark.setTitle(name);
                needUpdate = true;
            }
            if (pageIndex != editingOutline.mPageIndex){
                bookmark.setDestination(Destination.createFitPage(mPDFViewCtrl.getDoc(), pageIndex));
                needUpdate = true;
            }
            if (needUpdate){
                editingOutline.mTitle = name;
                editingOutline.mPageIndex = pageIndex;
                mAdapter.notifyItemChanged(mEditingOutlinePosition);
                ((UIExtensionsManager)mPDFViewCtrl.getUIExtensionsManager()).getDocumentManager().setDocModified(true);
            }
        } catch (PDFException e) {
            e.printStackTrace();
        }finally {
            mEditingOutlinePosition = -1;
        }
    }

    public OutlineNode getCurrentParentNode(){
        return mCurrentParentNode;
    }

    private ItemTouchHelper mItemTouchHelper;

    public void setItemTouchHelper(ItemTouchHelper itemTouchHelper){
        mItemTouchHelper = itemTouchHelper;
    }
}
