/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.modules.panel.signature;

import android.content.Context;
import android.graphics.Rect;
import android.os.Handler;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.ImageView;
import android.widget.TextView;

import com.foxit.sdk.PDFViewCtrl;
import com.foxit.uiextensions.R;
import com.foxit.uiextensions.UIExtensionsManager;
import com.foxit.uiextensions.browser.adapter.SuperAdapter;
import com.foxit.uiextensions.browser.adapter.viewholder.PageFlagViewHolder;
import com.foxit.uiextensions.browser.adapter.viewholder.SuperViewHolder;
import com.foxit.uiextensions.controls.dialog.sheetmenu.ISheetMenu;
import com.foxit.uiextensions.controls.dialog.sheetmenu.UIBottomSheetMenu;
import com.foxit.uiextensions.controls.dialog.sheetmenu.UISheetMenu;
import com.foxit.uiextensions.modules.panel.ILoadPanelItemListener;
import com.foxit.uiextensions.modules.panel.bean.BaseBean;
import com.foxit.uiextensions.modules.panel.bean.SignatureBean;
import com.foxit.uiextensions.theme.ThemeUtil;
import com.foxit.uiextensions.utils.AppDisplay;
import com.foxit.uiextensions.utils.AppResource;
import com.foxit.uiextensions.utils.AppUtil;
import com.foxit.uiextensions.utils.IResult;

import java.util.ArrayList;
import java.util.List;

import androidx.annotation.NonNull;
import androidx.fragment.app.FragmentActivity;
import androidx.recyclerview.widget.LinearLayoutManager;


public class SignatureAdapter extends SuperAdapter<SignatureBean> {
    public final static int FLAG_SIGNATURE = 0;
    public final static int FLAG_PAGE_TAG = 1;

    private ISheetMenu mSheetMenu;
    private SignaturePresenter mSigPresenter;
    private ArrayList<SignatureBean> mItems = new ArrayList<>();
    private ILoadPanelItemListener mLoadPanelItemListener = null;
    private PDFViewCtrl mPdfViewCtrl;
    private UIExtensionsManager mUiExtensionsManager;

    private boolean mIsPad;
    private int mSelectedPosition = -1;
    private Context mContext;
    public SignatureAdapter(Context context, PDFViewCtrl pdfViewCtrl, SignaturePresenter.ISignedVersionCallBack callBack) {
        super(context);
        mContext=context;
        mPdfViewCtrl = pdfViewCtrl;
        mUiExtensionsManager = (UIExtensionsManager) pdfViewCtrl.getUIExtensionsManager();
        mSigPresenter = new SignaturePresenter(context, pdfViewCtrl, callBack);
        mIsPad = AppDisplay.isPad();
    }

    public void load(final ILoadPanelItemListener listener) {
        mLoadPanelItemListener = listener;
        mSigPresenter.loadSignatures(new IResult<ArrayList<SignatureBean>, Object, Object>() {
            @Override
            public void onResult(boolean success, ArrayList<SignatureBean> p1, Object p2, Object p3) {
                mItems.clear();
                boolean hasSignatures = success && p1.size() > 0;
                if (hasSignatures) {
                    mItems.addAll(p1);
                }
                notifyDataSetChanged();
                if (listener != null) {
                    listener.onResult(hasSignatures);
                }
            }
        });
    }

    @Override
    public void notifyUpdateData() {
        notifyDataSetChanged();
    }

    @Override
    public SignatureBean getDataItem(int position) {
        return mItems.get(position);
    }

    @NonNull
    @Override
    public SuperViewHolder onCreateViewHolder(@NonNull ViewGroup parent, int viewType) {
        SuperViewHolder viewHolder;
        switch (viewType) {
            case FLAG_SIGNATURE:
                viewHolder = new SignatureAdapter.ItemViewHolder(inflateLayout(getContext(), parent, R.layout.panel_item_signature));
                break;
            case FLAG_PAGE_TAG:
                viewHolder = new PageFlagViewHolder(inflateLayout(getContext(), parent, R.layout.panel_item_flag));
                break;
            default:
                viewHolder = new SignatureAdapter.ItemViewHolder(inflateLayout(getContext(), parent, R.layout.panel_item_signature));
                break;
        }

        return viewHolder;
    }

    @Override
    public int getItemCount() {
        return mItems.size();
    }

    @Override
    public int getItemViewType(int position) {
        return mItems.get(position).getFlag();
    }

    private View inflateLayout(Context context, ViewGroup parent, int layoutId) {
        return LayoutInflater.from(context).inflate(layoutId, parent, false);
    }

    private void view(int position) {
        mSigPresenter.view(position, mItems.get(position).getPageIndex(), mItems.get(position).getUuid());
    }

    private void verify(int position) {
        mSigPresenter.verify(position, mItems.get(position).getPageIndex(), mItems.get(position).getUuid());
    }

    private void sign(int position) {
        mSigPresenter.gotoPage(position, mItems.get(position).getPageIndex(), mItems.get(position).getUuid(), true);
    }

    private void gotoPage(int position) {
        mSigPresenter.gotoPage(position, mItems.get(position).getPageIndex(), mItems.get(position).getUuid(), false);
    }

    public void delete(String uuid) {
        for (SignatureBean bean : mItems) {
            if (bean.getUuid().equals(uuid)) {
                mItems.remove(bean);
                break;
            }
        }
        boolean hasSignatures = mItems.size() > 0;
        notifyDataSetChanged();
        if (mLoadPanelItemListener != null) {
            mLoadPanelItemListener.onResult(hasSignatures);
        }
    }

    public void clearItems() {
        mItems.clear();
        notifyDataSetChanged();
    }

    class ItemViewHolder extends SuperViewHolder {
        private ImageView mIcon;
        private ImageView mMore;
        private TextView mTitle;
        private TextView mDate;
        private View mSigContainer;

        public ItemViewHolder(View viewHolder) {
            super(viewHolder);
            mSigContainer = viewHolder.findViewById(R.id.panel_signature_container);
            mIcon = viewHolder.findViewById(R.id.panel_item_signature_icon);
            mMore = (ImageView) viewHolder.findViewById(R.id.panel_item_signature_more);
            mTitle = (TextView) viewHolder.findViewById(R.id.panel_item_signature_title);
            mDate = (TextView) viewHolder.findViewById(R.id.panel_item_signature_date);
            ThemeUtil.setTintList(mMore,ThemeUtil.getPrimaryIconColor(context));
            mMore.setOnClickListener(this);
            mSigContainer.setOnClickListener(this);
        }

        @Override
        public void bind(BaseBean data, int position) {
            mSigContainer.setSelected(!mIsPad && mSelectedPosition == position);

            SignatureBean item = (SignatureBean) data;
            String title = AppResource.getString(context.getApplicationContext(), R.string.rv_panel_signature_unsigned);
            String dateOrName = item.getSigner();
            if (item.isSigned()) {
                title = AppResource.getString(context, R.string.rv_panel_signature_signed_by, item.getSigner());
                dateOrName = item.getDate();

                mIcon.setImageDrawable(AppResource.getDrawable(context.getApplicationContext(), R.drawable.rv_panel_signature_type, null));
                mMore.setVisibility(View.VISIBLE);
            } else {
                mIcon.setImageDrawable(AppResource.getDrawable(context.getApplicationContext(), R.drawable.rv_panel_signature_unsigned_type, null));
                boolean isVisible = mUiExtensionsManager.isEnableModification() && ((mPdfViewCtrl.isDynamicXFA() && mUiExtensionsManager.getDocumentManager().canModifyXFAForm())
                        || (mUiExtensionsManager.getDocumentManager().canSigning() && !item.isReadOnly()));
                isVisible = isVisible && mUiExtensionsManager.getDocumentManager().withAllPermission(null);
                mMore.setVisibility(isVisible ? View.VISIBLE : View.GONE);
            }
            mTitle.setText(title);
            mDate.setText(dateOrName);
            mSigContainer.setBackground(AppResource.getDrawable(mContext,R.drawable.rd_list_item_bg));
            ThemeUtil.setTintList(mIcon,ThemeUtil.getPrimaryIconColor(mContext));
            ThemeUtil.setTintList(mMore,ThemeUtil.getPrimaryIconColor(mContext));
            mTitle.setTextColor(AppResource.getColor(mContext,R.color.t4));
            mDate.setTextColor(AppResource.getColor(mContext,R.color.t3));
        }

        @Override
        public void onClick(View v) {
            if (AppUtil.isFastDoubleClick()) {
                return;
            }
            final int position = getAdapterPosition();
            if (v.getId() == R.id.panel_item_signature_more) {
                mSelectedPosition = position;
                if (mSelectedPosition >= 0)
                    notifyItemChanged(mSelectedPosition);

                SignatureBean item = mItems.get(position);
                List<Integer> menus = new ArrayList<>();
                if (item.isSigned()) {
                    menus.add(ISheetMenu.CHECK_SIGNATURE_MENU);
                    menus.add(ISheetMenu.VERIFY_SIGNATURE_MENU);
                } else {
                    boolean canSign = (mPdfViewCtrl.isDynamicXFA() && mUiExtensionsManager.getDocumentManager().canModifyXFAForm())
                            || (mUiExtensionsManager.getDocumentManager().canSigning() && mUiExtensionsManager.isEnableModification() && !item.isReadOnly());
                    if (canSign) {
                        menus.add(ISheetMenu.SIGN_MENU);
                    }
                }
                mSheetMenu = UISheetMenu.newInstance((FragmentActivity) mUiExtensionsManager.getAttachedActivity());
                if (mIsPad)
                    mSheetMenu.setWidth(AppResource.getDimensionPixelSize(getContext(), R.dimen.ux_pad_more_menu_width));
                mSheetMenu.setSheetItemClickListener(new UIBottomSheetMenu.OnSheetItemClickListener() {
                    @Override
                    public void onItemClick(int type) {
                        dismissSheetMenu();
                        if (ISheetMenu.SIGN_MENU == type) {
                            sign(position);
                        } else if (ISheetMenu.CHECK_SIGNATURE_MENU == type) {
                            view(position);
                        } else if (ISheetMenu.VERIFY_SIGNATURE_MENU == type) {
                            verify(position);
                        }
                    }
                });
                mSheetMenu.setOnSheetDismissListener(new ISheetMenu.OnSheetDismissListener() {
                    @Override
                    public void onDismiss(ISheetMenu sheetMenu) {
                        if (mSelectedPosition != -1) {
                            int tempPosition = mSelectedPosition;
                            mSelectedPosition = -1;
                            if (tempPosition >= 0)
                                notifyItemChanged(tempPosition);
                        }
                    }
                });
                mSheetMenu.setSheetItems(menus);
                showSheetMenu(v);
            } else if (v.getId() == R.id.panel_signature_container) {
                gotoPage(position);
            }
        }
    }

    private void showSheetMenu(View view) {
        Rect rect = new Rect();
        view.getGlobalVisibleRect(rect);
        mSheetMenu.show(mUiExtensionsManager.getRootView(), rect);
    }

    private void dismissSheetMenu() {
        if (mSheetMenu != null && mSheetMenu.isShowing()) {
            mSheetMenu.dismiss();
        }
    }

    void updateMenu(final LinearLayoutManager layoutManager) {
        if (mSheetMenu != null && mSheetMenu.isShowing()) {
            if (mIsPad) {
                new Handler().postDelayed(new Runnable() {
                    @Override
                    public void run() {
                        View view = layoutManager.findViewByPosition(mSelectedPosition);
                        if (view != null) {
                            View moreView = view.findViewById(R.id.panel_item_signature_more);
                            showSheetMenu(moreView);
                        } else {
                            mShouldUpdateMenu = true;
                            int scrollPosition = Math.min(getItemCount() - 1, mSelectedPosition + 1);
                            layoutManager.scrollToPosition(scrollPosition);
                        }
                    }
                }, 380);
            } else {
                showSheetMenu(mUiExtensionsManager.getRootView());
            }
        }
    }

    private boolean mShouldUpdateMenu = false;

    void onScrolled(LinearLayoutManager layoutManager) {
        if (mSelectedPosition == -1 || !mShouldUpdateMenu) return;

        mShouldUpdateMenu = false;
        View view = layoutManager.findViewByPosition(mSelectedPosition);
        if (view != null) {
            View moreView = view.findViewById(R.id.panel_item_signature_more);
            showSheetMenu(moreView);
        } else {
            dismissSheetMenu();
        }
    }

}
