/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.modules.panel.signature;

import android.content.Context;
import android.content.res.ColorStateList;
import android.content.res.Configuration;
import android.os.Handler;
import android.os.Looper;
import android.os.Message;
import android.view.View;
import android.view.ViewGroup;
import android.widget.ImageView;
import android.widget.RelativeLayout;
import android.widget.TextView;

import com.foxit.sdk.PDFException;
import com.foxit.sdk.PDFViewCtrl;
import com.foxit.sdk.common.Constants;
import com.foxit.sdk.pdf.PDFDoc;
import com.foxit.sdk.pdf.PDFPage;
import com.foxit.sdk.pdf.annots.Annot;
import com.foxit.sdk.pdf.annots.Widget;
import com.foxit.sdk.pdf.interform.Field;
import com.foxit.uiextensions.IThemeEventListener;
import com.foxit.uiextensions.Module;
import com.foxit.uiextensions.R;
import com.foxit.uiextensions.UIExtensionsManager;
import com.foxit.uiextensions.annots.AnnotEventListener;
import com.foxit.uiextensions.annots.IFlattenEventListener;
import com.foxit.uiextensions.annots.IRedactionEventListener;
import com.foxit.uiextensions.controls.dialog.FxProgressDialog;
import com.foxit.uiextensions.controls.panel.PanelSpec;
import com.foxit.uiextensions.event.DocEventListener;
import com.foxit.uiextensions.event.PageEventListener;
import com.foxit.uiextensions.modules.panel.ILoadPanelItemListener;
import com.foxit.uiextensions.modules.panel.IPanelManager;
import com.foxit.uiextensions.pdfreader.config.ReadStateConfig;
import com.foxit.uiextensions.pdfreader.impl.SimpleViewer;
import com.foxit.uiextensions.theme.ThemeConfig;
import com.foxit.uiextensions.theme.ThemeUtil;
import com.foxit.uiextensions.utils.AppDevice;
import com.foxit.uiextensions.utils.AppDisplay;
import com.foxit.uiextensions.utils.AppResource;
import com.foxit.uiextensions.utils.UIToast;

import androidx.annotation.NonNull;
import androidx.recyclerview.widget.DefaultItemAnimator;
import androidx.recyclerview.widget.LinearLayoutManager;
import androidx.recyclerview.widget.RecyclerView;


public class SignaturePanelModule implements Module, PanelSpec {
    private PDFViewCtrl mPdfViewCtrl;
    private Context mContext;
    private UIExtensionsManager mUiExtensionsManager;

    private View mTopBarView;
    private View mContentView;
    private View mNoInfoContainer;
    private ImageView mNoInfoIv;
    private TextView mNoInfoTv;
    private TextView mLoadingView;
    private TextView mCloseView;
    private TextView mTitleView;
    private TextView mEditViewTv;
    private RelativeLayout mTopContainer;
    private RelativeLayout mRLContainer;

    private SignatureAdapter mSigAdapter;
    private boolean mIsNeedRefreshPanel = false;
    private boolean bDocClosed = false;
    private ViewGroup mParent;
    private SimpleViewer mSignedVersionView;
    private LinearLayoutManager mLayoutManager;

    public SignaturePanelModule(Context context, ViewGroup parent, PDFViewCtrl pdfViewCtrl, PDFViewCtrl.UIExtensionsManager uiExtensionsManager) {
        mContext = context.getApplicationContext();
        mParent = parent;
        mPdfViewCtrl = pdfViewCtrl;
        mUiExtensionsManager = (UIExtensionsManager) uiExtensionsManager;
        SignaturePresenter.ISignedVersionCallBack signedVersionCallback = new SignaturePresenter.ISignedVersionCallBack() {
            @Override
            public void onPrepare() {
                showProgressDialog();
            }

            @Override
            public void onOpen(PDFDoc pdfDoc, String fileName) {
                if (mSignedVersionView != null) {
                    mSignedVersionView.open(pdfDoc, fileName, new SimpleViewer.ISimpleViewerCallBack() {
                        @Override
                        public void onDocWillOpen() {

                        }

                        @Override
                        public void onDocOpened() {
                            dismissProgressDialog();
                        }

                        @Override
                        public void onDocClosed() {
                        }
                    });
                }
            }

            @Override
            public void onFinish() {
                dismissProgressDialog();
            }
        };
        mSigAdapter = new SignatureAdapter(mContext, pdfViewCtrl, signedVersionCallback);
    }

    @Override
    public String getName() {
        return MODULE_NAME_SIGNATUREPANEL;
    }

    @Override
    public boolean loadModule() {

        mUiExtensionsManager.registerModule(this);

        //register event listener
        mPdfViewCtrl.registerDocEventListener(mDocEventListener);
        mPdfViewCtrl.registerPageEventListener(mPageEventListener);

        mUiExtensionsManager.registerThemeEventListener(mThemeEventListener);
        mUiExtensionsManager.registerConfigurationChangedListener(mConfigurationChangedListener);
        mUiExtensionsManager.getDocumentManager().registerAnnotEventListener(mAnnotEventListener);
        mUiExtensionsManager.getDocumentManager().registerRedactionEventListener(mRedactionEventListener);
        mUiExtensionsManager.getDocumentManager().registerFlattenEventListener(mFlattenEventListener);
        mUiExtensionsManager.getPanelManager().registerPanelEventListener(mPanelEventListener);
        return true;
    }

    private final IPanelManager.OnPanelEventListener mPanelEventListener = new IPanelManager.OnPanelEventListener() {
        @Override
        public void onPanelOpened() {
            if (mTopBarView != null) return;
            initPanelView();
            mIsNeedRefreshPanel = true;
        }

        @Override
        public void onPanelClosed() {

        }
    };

    private void initPanelView() {
        mTopBarView = createTopView();
        mContentView = createContentView();
        if (mUiExtensionsManager.getState() != ReadStateConfig.STATE_REFLOW)
            mUiExtensionsManager.getPanelManager().addPanel(this);
    }

    @Override
    public boolean unloadModule() {
        mUiExtensionsManager.getPanelManager().removePanel(this);
        mUiExtensionsManager.getPanelManager().unregisterPanelEventListener(mPanelEventListener);
        mPdfViewCtrl.unregisterDocEventListener(mDocEventListener);
        mPdfViewCtrl.unregisterPageEventListener(mPageEventListener);
        mUiExtensionsManager.unregisterThemeEventListener(mThemeEventListener);
        mUiExtensionsManager.unregisterConfigurationChangedListener(mConfigurationChangedListener);

        mUiExtensionsManager.getDocumentManager().unregisterAnnotEventListener(mAnnotEventListener);
        mUiExtensionsManager.getDocumentManager().unregisterRedactionEventListener(mRedactionEventListener);
        mUiExtensionsManager.getDocumentManager().unregisterFlattenEventListener(mFlattenEventListener);
        return true;
    }

    @Override
    public int getIcon() {
        return R.drawable.panel_tab_signature;
    }

    @Override
    public ColorStateList getIconTint() {
        return null;
    }

    @Override
    public int getPanelType() {
        return PanelSpec.SIGNATURES;
    }

    @Override
    public View getTopToolbar() {
        return mTopBarView;
    }

    @Override
    public View getContentView() {
        return mContentView;
    }

    @Override
    public void onActivated() {
        if (mPdfViewCtrl.getDoc() != null) {
            if (mIsNeedRefreshPanel) {
                mLoadingView.setVisibility(View.VISIBLE);
                mNoInfoContainer.setVisibility(View.GONE);
                mSigAdapter.load(mItemListener);
                mIsNeedRefreshPanel = false;
            }

            if (mSignedVersionView == null) {
                mSignedVersionView = new SimpleViewer(mContext, mPdfViewCtrl, mParent);
            }
        }
    }

    @Override
    public void onDeactivated() {
        mSignedVersionView = null;
    }

    private final ILoadPanelItemListener mItemListener = new ILoadPanelItemListener() {
        @Override
        public void onResult(boolean success) {
            if (success) {
                mLoadingView.setVisibility(View.GONE);
                mNoInfoContainer.setVisibility(View.GONE);
            } else {
                mLoadingView.setVisibility(View.GONE);
                mNoInfoContainer.setVisibility(View.VISIBLE);
            }
        }
    };

    private View createContentView() {
        View contentView = View.inflate(mContext, R.layout.panel_signature_content, null);
        mRLContainer = contentView.findViewById(R.id.rv_panel_signature_layout);
        mNoInfoContainer = contentView.findViewById(R.id.panel_signature_noinfo_container);
        mNoInfoIv = contentView.findViewById(R.id.panel_signature_noinfo_iv);
        mNoInfoTv = contentView.findViewById(R.id.panel_signature_noinfo_tv);
        mNoInfoIv.setColorFilter(ThemeConfig.getInstance(mContext).getPrimaryColor());
        mLoadingView = contentView.findViewById(R.id.rv_panel_signature_loading);
        RecyclerView recyclerView = contentView.findViewById(R.id.rv_panel_signature_list);
        recyclerView.setAdapter(mSigAdapter);
        mLayoutManager = new LinearLayoutManager(mContext, LinearLayoutManager.VERTICAL, false);
        recyclerView.setLayoutManager(mLayoutManager);
        recyclerView.setItemAnimator(new DefaultItemAnimator());
        if (AppDisplay.isPad()) {
            recyclerView.addOnScrollListener(new RecyclerView.OnScrollListener() {
                @Override
                public void onScrolled(@NonNull RecyclerView recyclerView, int dx, int dy) {
                    mSigAdapter.onScrolled(mLayoutManager);
                }
            });
        }
        return contentView;
    }

    private View createTopView() {
        View topView = View.inflate(mContext, R.layout.panel_topbar_layout, null);
        mTopContainer = topView.findViewById(R.id.panel_rl_top_container);
        mTopContainer.setBackgroundColor(ThemeConfig.getInstance(mContext).getB2());
        mCloseView = topView.findViewById(R.id.panel_close_tv);
        mCloseView.setTextColor(ThemeUtil.getPrimaryTextColor(mContext));
        if (AppDevice.isChromeOs(mPdfViewCtrl.getAttachedActivity())) {
            mCloseView.setVisibility(View.VISIBLE);
        } else {
            if (AppDisplay.isPad())
                mCloseView.setVisibility(View.INVISIBLE);
            else
                mCloseView.setVisibility(View.VISIBLE);
        }
        mCloseView.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                if (mUiExtensionsManager.getPanelManager().isShowingPanel())
                    mUiExtensionsManager.getPanelManager().hidePanel();
            }
        });
        mTitleView = topView.findViewById(R.id.panel_title_tv);
        mTitleView.setText(AppResource.getString(mContext, R.string.rv_panel_signature_title));
        mEditViewTv = topView.findViewById(R.id.panel_edit_tv);
        mEditViewTv.setVisibility(View.INVISIBLE);
        return topView;
    }

    private final IThemeEventListener mThemeEventListener = new IThemeEventListener() {
        @Override
        public void onThemeColorChanged(String type, int color) {
            if (mTopBarView == null) return;
            mCloseView.setTextColor(ThemeUtil.getPrimaryTextColor(mContext));
            mNoInfoIv.setColorFilter(ThemeConfig.getInstance(mContext).getPrimaryColor());
            mTopContainer.setBackgroundColor(ThemeConfig.getInstance(mContext).getB2());
            mTitleView.setTextColor(AppResource.getColor(mContext, R.color.t4));
            mRLContainer.setBackgroundColor(AppResource.getColor(mContext, R.color.b1));
            mLoadingView.setTextColor(AppResource.getColor(mContext, R.color.t3));
            mNoInfoTv.setTextColor(AppResource.getColor(mContext, R.color.t2));
            mSigAdapter.notifyUpdateData();
        }
    };

    private final UIExtensionsManager.ConfigurationChangedListener mConfigurationChangedListener = new UIExtensionsManager.ConfigurationChangedListener() {
        @Override
        public void onConfigurationChanged(Configuration newConfig) {
            mSigAdapter.updateMenu(mLayoutManager);
        }
    };

    private final PDFViewCtrl.IDocEventListener mDocEventListener = new DocEventListener() {
        @Override
        public void onDocWillOpen() {
            mSigAdapter.clearItems();
        }

        @Override
        public void onDocOpened(PDFDoc document, int errCode) {
            if (errCode == Constants.e_ErrSuccess) {
                mIsNeedRefreshPanel = true;
                bDocClosed = false;
                IPanelManager panelManager = mUiExtensionsManager.getPanelManager();
                if (panelManager.isShowingPanel()
                        && panelManager.getPanelHost().getCurrentSpec() == SignaturePanelModule.this) {
                    panelManager.getPanelHost().setCurrentSpec(getPanelType());
                }
            } else {
                mIsNeedRefreshPanel = false;
            }
        }

        @Override
        public void onDocWillClose(PDFDoc document) {
        }

        @Override
        public void onDocClosed(PDFDoc document, int errCode) {
            bDocClosed = true;
            mSigAdapter.clearItems();
        }

    };

    private final PDFViewCtrl.IPageEventListener mPageEventListener = new PageEventListener() {

        @Override
        public void onPagesRemoved(boolean success, int[] pageIndexes) {
            if (mNoInfoContainer == null) return;
            if (mNoInfoContainer.getVisibility() == View.GONE) {
                mSigAdapter.load(mItemListener);
            }
        }

        @Override
        public void onPageMoved(boolean success, int index, int dstIndex) {
            if (mNoInfoContainer == null) return;
            if (mNoInfoContainer.getVisibility() == View.GONE) {
                mSigAdapter.load(mItemListener);
            }
        }

        @Override
        public void onPagesInserted(boolean success, int dstIndex, int[] pageRanges) {
            if (mNoInfoContainer == null) return;
            mSigAdapter.load(mItemListener);
        }
    };

    private final IRedactionEventListener mRedactionEventListener = new IRedactionEventListener() {
        @Override
        public void onAnnotWillApply(PDFPage page, Annot annot) {
        }

        @Override
        public void onAnnotApplied(PDFPage page, Annot annot) {
            if (mNoInfoContainer == null) return;
            if (mNoInfoContainer.getVisibility() == View.GONE) {
                mSigAdapter.load(mItemListener);
            }
        }
    };

    private final IFlattenEventListener mFlattenEventListener = new IFlattenEventListener() {

        @Override
        public void onAnnotsFlattened(PDFDoc doc) {
            if (mNoInfoContainer == null) return;
            if (mNoInfoContainer.getVisibility() == View.GONE) {
                mSigAdapter.load(mItemListener);
            }
        }
    };

    private final AnnotEventListener mAnnotEventListener = new AnnotEventListener() {
        @Override
        public void onAnnotAdded(PDFPage page, Annot annot) {
            try {
                if (annot.getType() == Annot.e_Widget &&
                        ((Widget) annot).getField().getType() == Field.e_TypeSignature) {
                    Message message = Message.obtain();
                    message.what = SIGNATURE_ADD;
                    mHandler.sendMessage(message);

                }
            } catch (PDFException e) {
                e.printStackTrace();
            }
        }

        @Override
        public void onAnnotWillDelete(PDFPage page, Annot annot) {
            try {
                if (annot.getType() == Annot.e_Widget &&
                        ((Widget) annot).getField().getType() == Field.e_TypeSignature) {
                    Message message = Message.obtain();
                    message.what = SIGNATURE_DELETE;
                    message.obj = String.valueOf(annot.getDict().getObjNum());
                    mHandler.sendMessage(message);

                }
            } catch (PDFException e) {
                e.printStackTrace();
            }
        }

        @Override
        public void onAnnotDeleted(PDFPage page, Annot annot) {

        }

        @Override
        public void onAnnotModified(PDFPage page, Annot annot) {

        }

        @Override
        public void onAnnotChanged(Annot lastAnnot, Annot currentAnnot) {

        }
    };

    private static final int SIGNATURE_ADD = 0;
    private static final int SIGNATURE_DELETE = 1;
    private final Handler mHandler = new Handler(Looper.getMainLooper()) {
        @Override
        public void handleMessage(Message msg) {
            super.handleMessage(msg);

            switch (msg.what) {
                case SIGNATURE_ADD:
                    IPanelManager panelManager = mUiExtensionsManager.getPanelManager();
                    if (AppDisplay.isPad()
                            && panelManager.isShowingPanel()
                            && panelManager.getPanelHost().getCurrentSpec() == SignaturePanelModule.this) {
                        if (mLoadingView != null) {
                            mLoadingView.setVisibility(View.VISIBLE);
                            mNoInfoContainer.setVisibility(View.GONE);
                        }
                        mSigAdapter.load(mItemListener);
                        mIsNeedRefreshPanel = false;
                    } else {
                        mIsNeedRefreshPanel = true;
                    }
                    break;
                case SIGNATURE_DELETE:
                    if (!bDocClosed) {
                        mSigAdapter.delete((String) msg.obj);
                    }
                    break;
                default:
                    break;
            }
        }
    };

    private FxProgressDialog mProgressDialog;

    private void showProgressDialog() {
        UIExtensionsManager uiExtensionsManager = (UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager();
        if (uiExtensionsManager.getAttachedActivity() == null) {
            UIToast.getInstance(mContext).show(AppResource.getString(mContext.getApplicationContext(), R.string.rv_unknown_error));
            return;
        }
        mProgressDialog = new FxProgressDialog(uiExtensionsManager.getAttachedActivity(),
                AppResource.getString(mContext.getApplicationContext(), R.string.fx_string_opening));

        mProgressDialog.show();
    }

    private void dismissProgressDialog() {
        if (mProgressDialog != null && mProgressDialog.isShowing()) {
            mProgressDialog.dismiss();
            mProgressDialog = null;
        }
    }

    public boolean onKeyBack() {
        if (mSignedVersionView == null) {
            return false;
        } else if (mSignedVersionView.getVisibility() == View.VISIBLE) {
            mSignedVersionView.close();
            return true;
        }
        return false;
    }
}
