/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.modules.panzoom;

import android.content.Context;
import android.graphics.Bitmap;
import android.graphics.Canvas;
import android.graphics.Color;
import android.graphics.Paint;
import android.graphics.Point;
import android.graphics.PointF;
import android.graphics.Rect;
import android.graphics.RectF;
import android.os.Build;
import android.util.Log;
import android.view.KeyEvent;
import android.view.LayoutInflater;
import android.view.MotionEvent;
import android.view.View;
import android.view.ViewGroup;
import android.view.WindowManager;
import android.widget.LinearLayout;
import android.widget.RelativeLayout;

import com.foxit.sdk.PDFException;
import com.foxit.sdk.PDFViewCtrl;
import com.foxit.sdk.common.Progressive;
import com.foxit.sdk.common.Renderer;
import com.foxit.sdk.common.fxcrt.Matrix2D;
import com.foxit.sdk.pdf.PDFPage;
import com.foxit.uiextensions.DocumentManager;
import com.foxit.uiextensions.R;
import com.foxit.uiextensions.UIExtensionsManager;
import com.foxit.uiextensions.controls.propertybar.IViewSettingsWindow;
import com.foxit.uiextensions.theme.ThemeConfig;
import com.foxit.uiextensions.utils.AppDisplay;
import com.foxit.uiextensions.utils.AppDmUtil;
import com.foxit.uiextensions.utils.AppResource;

public class PanZoomView extends RelativeLayout {
    private Context mContext = null;
    private PDFViewCtrl mPdfViewCtrl = null;

    private RelativeLayout mPanZoomView = null;
    private LinearLayout mPanZoom_ll_center;

    private float mTouchStartX;
    private float mTouchStartY;
//    private final WindowManager mWindowManager; suyu
//    private RelativeLayout.LayoutParams mWmParams; suyu
    private RelativeLayout.LayoutParams mRlp;
    private ViewGroup mParent;
    private OverlayView mOverlayView;
    private int mMaxWidth;
    private int mMaxHeight;

    public PanZoomView(Context context, PDFViewCtrl pdfViewCtrl) {
        super(context);
        mContext = context;
        mPdfViewCtrl = pdfViewCtrl;
//        mWindowManager = (WindowManager) context.getSystemService(Context.WINDOW_SERVICE); suyu
        mPanZoomView = (RelativeLayout) LayoutInflater.from(context).inflate(R.layout.pan_zoom_layout, this);

        initView();
        bindEvent();
    }

    public PanZoomView(Context context, ViewGroup parent, PDFViewCtrl pdfViewCtrl) {
        super(context);
        mContext = context;
        mPdfViewCtrl = pdfViewCtrl;
        mParent = parent;
        mPanZoomView = (RelativeLayout) LayoutInflater.from(context).inflate(R.layout.pan_zoom_layout, this);

        initView();
        bindEvent();
    }

    private void initView() {
        float dpi = mContext.getResources().getDisplayMetrics().densityDpi;
        if (dpi == 0) {
            dpi = 240;
        }
        mMaxWidth = (int) (dpi / 4 * 3.5f);
        mMaxHeight = (int) (dpi / 4 * 5);

        int pageIndex = mPdfViewCtrl.getCurrentPage();
        if (mPdfViewCtrl.getPageViewWidth(pageIndex) > mPdfViewCtrl.getPageViewHeight(pageIndex)) {
            int tmp = mMaxHeight;
            mMaxHeight = mMaxWidth;
            mMaxWidth = tmp;
        }

        mPanZoom_ll_center = (LinearLayout) mPanZoomView.findViewById(R.id.rd_panzoom_ll_center);
        mOverlayView = new OverlayView(mContext, mPdfViewCtrl);
        mPanZoom_ll_center.removeAllViews();
        mPanZoom_ll_center.addView(mOverlayView);
        mPanZoomView.setVisibility(View.VISIBLE);

        RelativeLayout.LayoutParams centerParams = (LayoutParams) mPanZoom_ll_center.getLayoutParams();
        centerParams.width = mMaxWidth;
        centerParams.height = mMaxHeight;
        mPanZoom_ll_center.setLayoutParams(centerParams);

        RelativeLayout.LayoutParams zoomlp = new RelativeLayout.LayoutParams(ViewGroup.LayoutParams.WRAP_CONTENT, ViewGroup.LayoutParams.WRAP_CONTENT);
        zoomlp.setMargins(200, 400, 0, 0);
        mParent.addView(mPanZoomView, zoomlp);

        updateTheme();
    }

    private void bindEvent() {
        mOverlayView.setPanZoomRectEvent(mPanZoomRectEventListener);
    }

    PanZoomView.IPanZoomRectEventListener mPanZoomRectEventListener = new PanZoomView.IPanZoomRectEventListener() {
        @Override
        public void onPanZoomRectMove(float offsetX, float offsetY) {
            mPdfViewCtrl.scrollView(offsetX, offsetY);
        }
    };

    protected void onBack() {
        if (mOverlayView != null) {
            mOverlayView.setPanZoomRectEvent(null);
            mOverlayView = null;
        }

        if (((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager().getCurrentAnnot() != null) {
            ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager().setCurrentAnnot(null);
        }

        mParent.removeView(mPanZoomView);

    }

    public boolean onKeyDown(int keyCode, KeyEvent event) {
        if (keyCode == KeyEvent.KEYCODE_BACK) {
            exit();
        }
        return false;
    }

    protected boolean exit() {
        if (mPanZoomView.getVisibility() == View.VISIBLE) {
            onBack();
            return true;
        }
        return false;
    }

    @Override
    public boolean onTouchEvent(MotionEvent event) {
//        mWmParams = (WindowManager.LayoutParams) getLayoutParams();
        mRlp = (RelativeLayout.LayoutParams)getLayoutParams();
        int x = (int) event.getRawX();
        int y = (int) event.getRawY();
        int action = event.getAction();
        switch (action) {
            case MotionEvent.ACTION_DOWN:
                ((UIExtensionsManager)mPdfViewCtrl.getUIExtensionsManager()).stopHideToolbarsTimer();
                mTouchStartX = event.getX();
                mTouchStartY = event.getY();

                View rootView = ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getRootView();
                mRootWidth = rootView.getWidth();
                mRootHeight = rootView.getHeight();
                break;
            case MotionEvent.ACTION_MOVE:
                float mMoveStartX = event.getX();
                float mMoveStartY = event.getY();

                if (Math.abs(mTouchStartX - mMoveStartX) > 3
                        && Math.abs(mTouchStartY - mMoveStartY) > 3) {
//                    mWmParams.x = (int) (x - mTouchStartX);
//                    mWmParams.y = (int) (y - mTouchStartY);
//                    mWindowManager.updateViewLayout(this, mWmParams); suyu
                    mRlp.setMargins(0, 0, 0, 0);
                    mRlp.leftMargin = (int)(x - mTouchStartX);
                    mRlp.topMargin = (int)(y - mTouchStartY);

                    Log.d("suyu", String.format("PanZoom width : %d", getWidth()));

                    if (mRlp.leftMargin < 0)
                        mRlp.leftMargin = 0;
                    if (mRlp.topMargin < 0)
                        mRlp.topMargin = 0;
                    if (mRlp.leftMargin > mRootWidth - getWidth())
                        mRlp.leftMargin = mRootWidth - getWidth();
                    if (mRlp.topMargin > mRootHeight - getHeight())
                        mRlp.topMargin = mRootHeight - getHeight();

                    mParent.updateViewLayout(this, mRlp);
                }
                break;
            case MotionEvent.ACTION_UP:
                ((UIExtensionsManager)mPdfViewCtrl.getUIExtensionsManager()).startHideToolbarsTimer();
                break;
            default:
        }
        return true;
    }

    private int mRootWidth;
    private int mRootHeight;

    protected void onLayoutChange(View v, int newWidth, int newHeight, int oldWidth, int oldHeight) {
//        mWmParams = (ViewGroup.LayoutParams) getLayoutParams();
        mRlp = (RelativeLayout.LayoutParams)getLayoutParams();
        View rootView = ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getRootView();
        int[] location = new int[2];
        rootView.getLocationOnScreen(location);
        int x = location[0];
        int y = location[1];

        mRootWidth = rootView.getWidth();
        mRootHeight = rootView.getHeight();
//        if (AppDisplay.isLandscape()) {
//            mWmParams.x = x + width / 2;
//            mWmParams.y = y + height / 4;
//        } else {
//            mWmParams.x = x + width / 4;
//            mWmParams.y = y + height / 2;
//        }
//        mWindowManager.updateViewLayout(this, mWmParams);
        mRlp.setMargins(0, 0, 0, 0);
        if (AppDisplay.isLandscape()) {
            mRlp.leftMargin = x + mRootWidth / 2;
            mRlp.topMargin = y + mRootHeight / 4;
        } else {
            mRlp.leftMargin = x + mRootWidth / 4;
            mRlp.topMargin = y + mRootHeight / 2;
        }
        mParent.updateViewLayout(this, mRlp);

        if (mOverlayView == null) return;
        mOverlayView.calcPdfViewerRect();
        reCalculatePanZoomRect(mOverlayView.curPageIndex);
    }


    public void reDrawPanZoomView(int pageIndex) {
        if (mOverlayView == null) return;
        if (mPdfViewCtrl.getPageLayoutMode() == PDFViewCtrl.PAGELAYOUTMODE_FACING ||
                mPdfViewCtrl.getPageLayoutMode() == PDFViewCtrl.PAGELAYOUTMODE_COVER) {
            Rect pageRect = mPdfViewCtrl.getPageViewRect(pageIndex);
            if (!pageRect.intersect(mOverlayView.mPdfViewerRect)) {
                pageIndex = pageIndex + 1;
            }
        }

        float pageW = mPdfViewCtrl.getPageViewWidth(pageIndex);
        float pageH = mPdfViewCtrl.getPageViewHeight(pageIndex);

        if ((pageW > pageH && mMaxWidth < mMaxHeight) ||
                (pageW < pageH && mMaxWidth > mMaxHeight)) {
            int tmp = mMaxHeight;
            mMaxHeight = mMaxWidth;
            mMaxWidth = tmp;

            RelativeLayout.LayoutParams centerParams = (LayoutParams) mPanZoom_ll_center.getLayoutParams();
            centerParams.width = mMaxWidth;
            centerParams.height = mMaxHeight;
            mPanZoom_ll_center.setLayoutParams(centerParams);
        }

        mOverlayView.setCurPageIndex(pageIndex);
        mOverlayView.calcCurPageViewRect();
        mOverlayView.calcPanZoomRect();
        post(mOverlayView);
    }

    public int getCurPageIndex() {
        if (mOverlayView == null)
            return -1;
        return mOverlayView.getCurPageIndex();
    }

    public void reCalculatePanZoomRect(int pageIndex) {
        if (mOverlayView == null) return;

        if (mPdfViewCtrl.getPageLayoutMode() == PDFViewCtrl.PAGELAYOUTMODE_FACING ||
                mPdfViewCtrl.getPageLayoutMode() == PDFViewCtrl.PAGELAYOUTMODE_COVER) {
            int tmpIndex = mOverlayView.curPageIndex;
            int curPageIndex = mPdfViewCtrl.getCurrentPage();
            Rect pageRect = mPdfViewCtrl.getPageViewRect(curPageIndex);
            if (pageRect.intersect(mOverlayView.mPdfViewerRect)) {
                tmpIndex = curPageIndex;
            } else if (pageIndex != mOverlayView.curPageIndex) {
                tmpIndex = pageIndex;
            }
            reDrawPanZoomView(tmpIndex);
            return;
        }

        RectF dirtyRect = mOverlayView.mPanZoomRect;
        mOverlayView.calcCurPageViewRect();
        mOverlayView.calcPanZoomRect();

        dirtyRect.union(mOverlayView.mPanZoomRect);
        mOverlayView.invalidate(AppDmUtil.rectFToRect(dirtyRect));
    }

    public boolean isPanZoomRectMoving() {
        if (mOverlayView == null) return false;
        return mOverlayView.mTouchCaptured;
    }

    public boolean onTouchEvent(int pageIndex, MotionEvent motionEvent) {
        if (mOverlayView == null) return false;
        if (mPdfViewCtrl.getPageLayoutMode() == PDFViewCtrl.PAGELAYOUTMODE_FACING ||
                mPdfViewCtrl.getPageLayoutMode() == PDFViewCtrl.PAGELAYOUTMODE_COVER) {
            Rect pageRect = mPdfViewCtrl.getPageViewRect(mOverlayView.curPageIndex);
            if (pageRect.intersect(mOverlayView.mPdfViewerRect)) return false;
            if (pageIndex != mOverlayView.curPageIndex) {
                reDrawPanZoomView(pageIndex);
            }
        }
        return false;
    }

    class OverlayView extends View implements Runnable {
        private PDFViewCtrl mViewCtrl;
        private Bitmap mBitmap = null;
        private Paint mBitmapPaint;
        private int curPageIndex;

        private RectF mPanZoomRect = new RectF(0, 0, 0, 0);
        private Paint mRectPaint;
        private int mRectColor;
        private int mLineWith = 4;

        private Rect mPdfViewerRect = new Rect(0, 0, 0, 0);
        private Rect mCurPageViewRect = new Rect(0, 0, 0, 0);

        private float mPanZoomWidth;
        private float mPanZoomHeight;

        public OverlayView(Context context, PDFViewCtrl pdfViewCtrl) {
            super(context);
            mViewCtrl = pdfViewCtrl;
            curPageIndex = mViewCtrl.getCurrentPage();

            mPanZoomWidth = mMaxWidth;
            mPanZoomHeight = mMaxHeight;

            mBitmapPaint = new Paint();
            mBitmapPaint.setAntiAlias(true);
            mBitmapPaint.setFilterBitmap(true);

            mRectPaint = new Paint();
            mRectPaint.setStyle(Paint.Style.STROKE);
            mRectPaint.setAntiAlias(true);
            mRectPaint.setDither(true);
            mRectPaint.setStrokeWidth(mLineWith);
            mRectColor = ThemeConfig.getInstance(mContext).getPrimaryColor();

            setBackgroundColor(Color.argb(0xff, 0xe1, 0xe1, 0xe1));

            initialize();

            if (Build.VERSION.SDK_INT < 24) {
                post(this);
            }
        }

        private void initialize() {
            calcPdfViewerRect();
            calcCurPageViewRect();
            calcPanZoomRect();
        }

        private void calcPdfViewerRect() {
            int viewerLeft = mViewCtrl.getLeft();
            int viewerRight = mViewCtrl.getRight();
            int viewerTop = mViewCtrl.getTop();
            int viewerBottom = mViewCtrl.getBottom();
            mPdfViewerRect.set(viewerLeft, viewerTop, viewerRight, viewerBottom);
        }

        private void calcCurPageViewRect() {
            mCurPageViewRect.set(mViewCtrl.getPageViewRect(curPageIndex));
        }

        private void calcPanZoomRect() {
            Rect rect = new Rect();
            rect.set(mCurPageViewRect);
            rect.intersect(mPdfViewerRect);

            RectF pvRect = new RectF();
            mViewCtrl.convertDisplayViewRectToPageViewRect(AppDmUtil.rectToRectF(rect), pvRect, curPageIndex);

            try {
                DocumentManager documentManager = ((UIExtensionsManager) mViewCtrl.getUIExtensionsManager()).getDocumentManager();
                PDFPage pdfPage = documentManager.getPage(curPageIndex, false);
                if (pdfPage == null || pdfPage.isEmpty()) {
                    return;
                }
                float pageWidth = pdfPage.getWidth();
                float pageHeight = pdfPage.getHeight();
                if (isRotationVertical(mViewCtrl.getViewRotation())) {
                    float tmp = pageWidth;
                    pageWidth = pageHeight;
                    pageHeight = tmp;
                }

                float widthScale = (float) mMaxWidth / pageWidth;
                float heightScale = (float) mMaxHeight / pageHeight;
                float scale = Math.min(widthScale, heightScale);
                mPanZoomWidth = pageWidth * scale;
                mPanZoomHeight = pageHeight * scale;
            } catch (PDFException e) {
                e.printStackTrace();
            }

            float left = pvRect.left / mViewCtrl.getPageViewWidth(curPageIndex) * mPanZoomWidth;
            float top = pvRect.top / mViewCtrl.getPageViewHeight(curPageIndex) * mPanZoomHeight;
            float right = pvRect.right / mViewCtrl.getPageViewWidth(curPageIndex) * mPanZoomWidth;
            float bottom = pvRect.bottom / mViewCtrl.getPageViewHeight(curPageIndex) * mPanZoomHeight;

            mPanZoomRect.set(left, top, right, bottom);
        }

        Rect mClipRect = new Rect();
        RectF mDrawRect = new RectF();

        @Override
        protected void onDraw(Canvas canvas) {
            if (mBitmap == null) {
                if (Build.VERSION.SDK_INT >= 24) {
                    post(this);
                }
                return;
            }
            canvas.getClipBounds(mClipRect);
            canvas.drawBitmap(mBitmap, mClipRect.left, mClipRect.top, mBitmapPaint);

            mRectPaint.setColor(mRectColor);
            canvas.save();
            mDrawRect.set(mPanZoomRect);
            float dx = mLastPoint.x - mDownPoint.x;
            float dy = mLastPoint.y - mDownPoint.y;
            mDrawRect.offset(dx, dy);
            mDrawRect.inset(mLineWith / 2.0f, mLineWith / 2.0f);
            canvas.drawRect(mDrawRect, mRectPaint);
            canvas.restore();
        }

        private PointF mAdjustPointF = new PointF(0, 0);

        private PointF adjustScalePointF(RectF rectF, float dxy) {
            float adjustx = 0;
            float adjusty = 0;

            if ((int) rectF.left < dxy) {
                adjustx = -rectF.left + dxy;
                rectF.left = dxy;
            }
            if ((int) rectF.top < dxy) {
                adjusty = -rectF.top + dxy;
                rectF.top = dxy;
            }

            if ((int) rectF.right > getWidth() - dxy) {
                adjustx = getWidth() - rectF.right - dxy;
                rectF.right = getWidth() - dxy;
            }
            if ((int) rectF.bottom > getHeight() - dxy) {
                adjusty = getHeight() - rectF.bottom - dxy;
                rectF.bottom = getHeight() - dxy;
            }
            mAdjustPointF.set(adjustx, adjusty);
            return mAdjustPointF;
        }

        private PointF mDownPoint = new PointF();
        private PointF mLastPoint = new PointF();
        private RectF mInvalidateRect = new RectF(0, 0, 0, 0);
        private RectF mAdjustRect = new RectF(0, 0, 0, 0);
        private boolean mTouchCaptured = false;

        private PointF mLastDownPoint = new PointF();

        @Override
        public boolean onTouchEvent(MotionEvent event) {
            float x = event.getX();
            float y = event.getY();
            int action = event.getAction();
            switch (action) {
                case MotionEvent.ACTION_DOWN:
                    mDownPoint.set(x, y);
                    mLastPoint.set(x, y);

                    mLastDownPoint.set(x, y);
                    boolean isMaxRect = mPanZoomRect.width() == mPanZoomWidth && mPanZoomRect.height() == mPanZoomHeight;
                    if (!isMaxRect && isInPanZoomRect(x, y)) {
                        ((UIExtensionsManager)mPdfViewCtrl.getUIExtensionsManager()).stopHideToolbarsTimer();
                        mTouchCaptured = true;
                    } else {
                        mTouchCaptured = false;
                        return false;
                    }

                    return true;
                case MotionEvent.ACTION_MOVE:
                    if (mTouchCaptured) {
                        if (x != mLastPoint.x && y != mLastPoint.y) {
                            RectF drawBBox = new RectF(mPanZoomRect);
                            mInvalidateRect.set(drawBBox);
                            mInvalidateRect.offset(mLastPoint.x - mDownPoint.x, mLastPoint.y - mDownPoint.y);

                            mAdjustRect.set(drawBBox);
                            mAdjustRect.offset(x - mDownPoint.x, y - mDownPoint.y);
                            float deltaXY = 0;// Judging border value
                            PointF adjustXY = adjustScalePointF(mAdjustRect, deltaXY);
                            mInvalidateRect.union(mAdjustRect);

                            mInvalidateRect.inset(-deltaXY, -deltaXY);
                            invalidate(AppDmUtil.rectFToRect(mInvalidateRect));

                            mLastPoint.set(x, y);
                            mLastPoint.offset(adjustXY.x, adjustXY.y);

                            float offsetX = (mLastPoint.x - mLastDownPoint.x) * mViewCtrl.getPageViewWidth(curPageIndex) / mPanZoomWidth;
                            float offsetY = (mLastPoint.y - mLastDownPoint.y) * mViewCtrl.getPageViewHeight(curPageIndex) / mPanZoomHeight;
                            if (mPanZoomRectListener != null) {
                                mPanZoomRectListener.onPanZoomRectMove(offsetX, offsetY);
                            }

                            mLastDownPoint.set(mLastPoint);
                        }
                    }
                    return true;
                case MotionEvent.ACTION_UP:
                case MotionEvent.ACTION_CANCEL:
                    if (mTouchCaptured) {
                        ((UIExtensionsManager)mPdfViewCtrl.getUIExtensionsManager()).startHideToolbarsTimer();
                        RectF drawBox = new RectF(mPanZoomRect);
                        drawBox.inset(mLineWith / 2.0f, mLineWith / 2.0f);
                        drawBox.offset(mLastPoint.x - mDownPoint.x, mLastPoint.y - mDownPoint.y);

                        mPanZoomRect.set(drawBox);
                        mPanZoomRect.sort();
                        mPanZoomRect.inset(-mLineWith / 2.0f, -mLineWith / 2.0f);

                        mTouchCaptured = false;
                        mDownPoint.set(0, 0);
                        mLastPoint.set(0, 0);
                        mLastDownPoint.set(0, 0);
                        return true;
                    }

                    mTouchCaptured = false;
                    mDownPoint.set(0, 0);
                    mLastPoint.set(0, 0);
                    mLastDownPoint.set(0, 0);
                    return true;
                default:
            }

            return true;
        }

        @Override
        public void run() {
            drawPage(curPageIndex);
        }

        public void setCurPageIndex(int pageIndex) {
            curPageIndex = pageIndex;
        }

        public int getCurPageIndex() {
            return curPageIndex;
        }

        public void drawPage(int pageIndex) {
            PDFViewCtrl.lock();
            try {
                DocumentManager documentManager = ((UIExtensionsManager) mViewCtrl.getUIExtensionsManager()).getDocumentManager();
                PDFPage pdfPage = documentManager.getPage(pageIndex, false);
                if (pdfPage == null) {
                    return;
                }

                float width = pdfPage.getWidth();
                float height = pdfPage.getHeight();

                if (isRotationVertical(mViewCtrl.getViewRotation())) {
                    float tmp = width;
                    width = height;
                    height = tmp;
                }

                Matrix2D page2device = pdfPage.getDisplayMatrix(0, 0, (int) width, (int) height, mViewCtrl.getViewRotation());
                Bitmap bitmap = Bitmap.createBitmap((int) width, (int) height, Bitmap.Config.RGB_565);
                int backgroundColor;
                int foregroundColor;
                if (mPdfViewCtrl.getColorMode() == Renderer.e_ColorModeMappingGray
                        || mPdfViewCtrl.getColorMode() == Renderer.e_ColorModeMapping) {
                    backgroundColor = mPdfViewCtrl.getMappingModeBackgroundColor();
                    foregroundColor = mPdfViewCtrl.getMappingModeForegroundColor();
                    //} else if (mPdfViewCtrl.getColorMode() == Renderer.e_ColorModeMapping && mPdfViewCtrl.isNightMode()) {
                    //    backgroundColor = Color.argb(0xff, 0x0a, 0x0a, 0x0a);
                    //    foregroundColor = Color.argb(0xff, 0x9b, 0x9b, 0x9b);
                } else {
                    backgroundColor = Color.WHITE;
                    foregroundColor = Color.BLACK;
                }
                bitmap.eraseColor(backgroundColor);
                Renderer renderer = new Renderer(bitmap, true);
                renderer.setColorMode(mPdfViewCtrl.getColorMode());
                renderer.setMappingModeColors(backgroundColor, foregroundColor);
                Progressive progressive = renderer.startRender(pdfPage, page2device, null);
                int state = Progressive.e_ToBeContinued;
                while (state == Progressive.e_ToBeContinued) {
                    state = progressive.resume();
                }

                mViewCtrl.renderRmsWatermark(pdfPage, renderer, page2device);

                mBitmap = Bitmap.createScaledBitmap(bitmap, (int) mPanZoomWidth, (int) mPanZoomHeight, true);
                if (!bitmap.isRecycled() && !mBitmap.equals(bitmap)) {
                    bitmap.recycle();
                }

                LinearLayout.LayoutParams layoutParams = (LinearLayout.LayoutParams) this.getLayoutParams();
                layoutParams.width = (int) mPanZoomWidth;
                layoutParams.height = (int) mPanZoomHeight;
                setLayoutParams(layoutParams);
                invalidate();
            } catch (PDFException e) {
                e.printStackTrace();
            } finally {
                PDFViewCtrl.unlock();
            }
        }


        public boolean isInPanZoomRect(float x, float y) {
            return mPanZoomRect.contains(x, y);
        }

        IPanZoomRectEventListener mPanZoomRectListener;

        public void setPanZoomRectEvent(IPanZoomRectEventListener listener) {
            mPanZoomRectListener = listener;
        }
    }


    public interface IPanZoomRectEventListener {
        void onPanZoomRectMove(float offsetX, float offsetY);
    }

    private boolean isRotationVertical(int rotation) {
        return rotation == com.foxit.sdk.common.Constants.e_Rotation90
                || rotation == com.foxit.sdk.common.Constants.e_Rotation270;
    }

    void updateTheme() {
        UIExtensionsManager uiExtensionsManager = (UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager();
        IViewSettingsWindow viewSettingsWindow = uiExtensionsManager.getMainFrame().getSettingWindow();
        if (viewSettingsWindow.getColorMode() == IViewSettingsWindow.NIGHT) {
            mPanZoom_ll_center.setBackgroundColor(Color.parseColor("#36404A"));
        } else {
            mPanZoom_ll_center.setBackgroundColor(AppResource.getColor(mContext, R.color.ux_bg_color_docviewer));
        }
        mOverlayView.setBackgroundColor(Color.argb(0xff, 0xe1, 0xe1, 0xe1));
    }

}

