/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.modules.signature;

import android.app.Activity;
import android.content.Context;
import android.content.Intent;
import android.content.res.ColorStateList;
import android.graphics.Bitmap;
import android.graphics.Bitmap.Config;
import android.graphics.BitmapFactory;
import android.graphics.Canvas;
import android.graphics.Color;
import android.graphics.Matrix;
import android.graphics.Paint;
import android.graphics.PaintFlagsDrawFilter;
import android.graphics.PointF;
import android.graphics.PorterDuff;
import android.graphics.Rect;
import android.graphics.RectF;
import android.graphics.Shader;
import android.graphics.Typeface;
import android.net.Uri;
import android.os.Build;
import android.os.Handler;
import android.os.Looper;
import android.os.Message;
import android.provider.MediaStore;
import android.text.TextUtils;
import android.util.Log;
import android.util.TypedValue;
import android.view.Gravity;
import android.view.MotionEvent;
import android.view.View;
import android.view.View.OnClickListener;
import android.view.ViewGroup;
import android.view.ViewGroup.LayoutParams;
import android.widget.ImageView;
import android.widget.LinearLayout;
import android.widget.RelativeLayout;
import android.widget.Toast;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.core.content.FileProvider;
import androidx.core.view.OnApplyWindowInsetsListener;
import androidx.core.view.ViewCompat;
import androidx.core.view.WindowInsetsCompat;
import androidx.fragment.app.FragmentActivity;

import com.foxit.sdk.PDFException;
import com.foxit.sdk.PDFViewCtrl;
import com.foxit.sdk.common.Path;
import com.foxit.sdk.pdf.PSI;
import com.foxit.uiextensions.IPermissionProvider;
import com.foxit.uiextensions.Module;
import com.foxit.uiextensions.R;
import com.foxit.uiextensions.ToolHandler;
import com.foxit.uiextensions.UIExtensionsManager;
import com.foxit.uiextensions.annots.common.UIFillView;
import com.foxit.uiextensions.annots.fillsign.FillSignModule;
import com.foxit.uiextensions.annots.fillsign.FillSignToolHandler;
import com.foxit.uiextensions.controls.dialog.UIDialog;
import com.foxit.uiextensions.controls.dialog.UIPopoverFrag;
import com.foxit.uiextensions.controls.dialog.UITextEditDialog;
import com.foxit.uiextensions.controls.propertybar.PropertyBar;
import com.foxit.uiextensions.controls.propertybar.imp.PropertyBarImpl;
import com.foxit.uiextensions.controls.toolbar.BaseBar;
import com.foxit.uiextensions.controls.toolbar.IBaseItem;
import com.foxit.uiextensions.controls.toolbar.ToolbarItemConfig;
import com.foxit.uiextensions.controls.toolbar.impl.BaseItemImpl;
import com.foxit.uiextensions.controls.toolbar.impl.BottomBarImpl;
import com.foxit.uiextensions.controls.toolbar.impl.TopBarImpl;
import com.foxit.uiextensions.controls.toolbar.impl.UIColorItem;
import com.foxit.uiextensions.modules.signature.appearance.SignatureAppearanceDialog;
import com.foxit.uiextensions.pdfreader.config.ActRequestCode;
import com.foxit.uiextensions.pdfreader.config.ReadStateConfig;
import com.foxit.uiextensions.security.ICertificateSupport;
import com.foxit.uiextensions.security.certificate.CertificateFileInfo;
import com.foxit.uiextensions.security.digitalsignature.DigitalSignatureModule;
import com.foxit.uiextensions.security.digitalsignature.DigitalSignatureUtil;
import com.foxit.uiextensions.security.digitalsignature.IDigitalSignatureCallBack;
import com.foxit.uiextensions.theme.ThemeConfig;
import com.foxit.uiextensions.theme.ThemeUtil;
import com.foxit.uiextensions.utils.AppDevice;
import com.foxit.uiextensions.utils.AppDisplay;
import com.foxit.uiextensions.utils.AppDmUtil;
import com.foxit.uiextensions.utils.AppFileUtil;
import com.foxit.uiextensions.utils.AppIntentUtil;
import com.foxit.uiextensions.utils.AppResource;
import com.foxit.uiextensions.utils.AppSharedPreferences;
import com.foxit.uiextensions.utils.AppUtil;
import com.foxit.uiextensions.utils.Event;
import com.foxit.uiextensions.utils.SystemUiHelper;
import com.foxit.uiextensions.utils.UIToast;
import com.foxit.uiextensions.utils.thread.AppThreadManager;

import java.io.File;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.List;

public class SignatureDrawView {
    private static final int MSG_DRAW = 0x01;
    private static final int MSG_CLEAR = 0x02;
    private static final int MSG_COLOR = 0x04;
    private static final int MSG_DIAMETER = 0x08;
    private static final int MSG_RELEASE = 0x10;
    private boolean mIsFromSignatureField = false;
    private static final int INK_DIAMETER_SCALE = 10;

    public interface OnDrawListener {
        boolean canDraw();

        void moveToTemplate();

        void onBackPressed();

        void result(SignatureInkItem inkItem);
    }

    public void setOnDrawListener(OnDrawListener listener) {
        mListener = listener;
    }

    private Context mContext;
    private Activity mActivity;
    private SignatureFragment mFragment;
    private ViewGroup mParent;
    private PDFViewCtrl mPdfViewCtrl;
    private UIExtensionsManager mUiExtensionsManager;
    private SignatureToolHandler mToolHandler;
    private List<ISigBitmapChangeCallback> mBitmapChangeCallbackList = new ArrayList<>();
    private OnDrawListener mListener;

    private UIFillView mPropView;
    private UIColorItem mPropItem;
    private IBaseItem mBackItem;
    private IBaseItem mTitleItem;
    private IBaseItem mSaveItem;
    private IBaseItem mCertificateItem;

    private BaseBar mTopTitleBar;
    private TopBarImpl mTopToolBar;
    private PropertyBar mPropertyBar;
    private BottomBarImpl mBottomToolBar;

    private DrawView mDrawView;
    private ViewGroup mContentView;
    private RelativeLayout mTopBarLayout;
    private RelativeLayout mTopToolLayout;
    private RelativeLayout mBottomToolLayout;
    private RelativeLayout mSwitchLayout;
    private ImageView mSwitchBtn;

    private View switchView;

    private Bitmap mBitmap;
    private Rect mRect = new Rect();
    private Rect mValidRect = new Rect();

    private String mKey;
    private String mCurDsgPath;
    private String mCapturePath;

    private int mBmpHeight;
    private int mBmpWidth;
    private boolean mCanDraw = false;

    private PSI mPsi = null;
    private DigitalSignatureUtil mDsgUtil;
    private com.foxit.uiextensions.config.Config mModuleConfig;

    private SignatureAppearanceDialog mSignAppearanceDialog;
    private SignatureFragment.SignatureInkCallback mInkCallback;
    private SignatureInkItem mInkItem;
    private CertificateFileInfo mFileInfo;
    private ICertificateSupport.CertificateInfo mInfo;
    DigitalSignatureModule mDsgModule;

    private boolean mIsFromDraw;
    private Handler mHandler = new Handler(Looper.getMainLooper()) {

        @Override
        public void handleMessage(Message msg) {
            int what = msg.what;
            switch (what) {
                case MSG_DRAW:
                    if (mDrawView == null) return;
                    mCanDraw = true;
                    mDrawView.invalidate();
                    break;
                case MSG_CLEAR:
                    mCanDraw = true;
                    changeDrawBitmap(null, true);
                    mDrawView.invalidate();
                    break;
                case MSG_COLOR:
                    mCanDraw = true;

                    break;
                case MSG_DIAMETER:
                    mCanDraw = true;
                    break;
                case MSG_RELEASE:
                    mCanDraw = false;
                    if (mBitmap != null && mBitmap.isRecycled()) {
                        mBitmap.recycle();
                    }
                    mBitmap = null;
                    break;
                default:
                    break;
            }
        }
    };

    public SignatureDrawView(Context context, ViewGroup parent, PDFViewCtrl pdfViewCtrl) {
        mContext = context.getApplicationContext();
        mPdfViewCtrl = pdfViewCtrl;
        mParent = parent;
        mPropertyBar = new PropertyBarImpl(mContext, pdfViewCtrl);
        mUiExtensionsManager = (UIExtensionsManager) pdfViewCtrl.getUIExtensionsManager();
        mCapturePath = AppFileUtil.getAppCacheDir(mContext) + "/camera_photos/sign_capture_img.png";
        mModuleConfig = mUiExtensionsManager.getConfig();
        mContentView = (ViewGroup) View.inflate(mContext, R.layout.rv_sg_create, null);
        mTopBarLayout = mContentView.findViewById(R.id.sig_create_top_title_bar_layout);
        mTopToolLayout = mContentView.findViewById(R.id.sig_create_top_tool_bar_layout);
        mBottomToolLayout = mContentView.findViewById(R.id.sig_create_bottom_tool_bar_layout);
        mTopBarLayout.setBackgroundColor(AppResource.getColor(mContext, R.color.ui_color_top_bar_main));

        ViewGroup drawContainer = mContentView.findViewById(R.id.sig_create_canvas);
        mDrawView = new DrawView(mContext);
        drawContainer.addView(mDrawView);
        if (Build.VERSION.SDK_INT >= 29) { // Build.VERSION_CODES.Q
            drawContainer.setForceDarkAllowed(false);
        }
        mDsgModule = (DigitalSignatureModule) mUiExtensionsManager.getModuleByName(Module.MODULE_NAME_DIGITALSIGNATURE);
        if (mDsgModule != null)
            mDsgUtil = mDsgModule.getDigitalSignatureUtil();
        mBitmapChangeCallbackList.clear();
        SignatureModule sigModule = (SignatureModule) mUiExtensionsManager.getModuleByName(Module.MODULE_NAME_PSISIGNATURE);
        if (sigModule != null) {
            mToolHandler = (SignatureToolHandler) sigModule.getToolHandler();
            mBitmapChangeCallbackList.add(sigModule.getBitmapChangeCallback());
        }
        initBarLayout();

        ViewCompat.setOnApplyWindowInsetsListener(mContentView, new OnApplyWindowInsetsListener() {
            @Override
            public WindowInsetsCompat onApplyWindowInsets(View v, WindowInsetsCompat windowInsets) {
                androidx.core.graphics.Insets insets;
                if (Build.VERSION.SDK_INT >= 30) {
                    insets = windowInsets.getInsets(WindowInsetsCompat.Type.systemBars() | WindowInsetsCompat.Type.displayCutout());
                } else {
                    insets = windowInsets.getInsets(WindowInsetsCompat.Type.displayCutout());
                }
                mTopBarLayout.setPadding(0,
                        insets.top,
                        0,
                        0);
                mContentView.setPadding(insets.left,
                        0,
                        insets.right,
                        insets.bottom);
                return WindowInsetsCompat.CONSUMED;
            }
        });
    }

    public void setInkCallback(SignatureFragment.SignatureInkCallback callback) {
        mInkCallback = callback;
    }


    private void initBarLayout() {
        initTitleBar();
        initToolBar();

        mTopBarLayout.addView(mTopTitleBar.getContentView());
        mTopToolLayout.addView(mTopToolBar.getContentView());
        mBottomToolLayout.addView(mBottomToolBar.getContentView());
    }

    private void initTitleBar() {
        SignatureModule signatureModule = (SignatureModule) mUiExtensionsManager.getModuleByName(Module.MODULE_NAME_PSISIGNATURE);
        if (signatureModule != null) {
            mTopTitleBar = signatureModule.getSignatureViewTopBar();

            View contentView = mTopTitleBar.getContentView();
            if (contentView.getParent() != null) {
                ViewGroup vg = (ViewGroup) contentView.getParent();
                vg.removeView(contentView);
                mTopTitleBar.removeItemByTag(ToolbarItemConfig.SIGNATURE_VIEW_TOP_LEFT_BACK_ITEM);
                mTopTitleBar.removeItemByTag(ToolbarItemConfig.SIGNATURE_VIEW_TOP_CENTER_TITLE_ITEM);
                mTopTitleBar.removeItemByTag(ToolbarItemConfig.SIGNATURE_VIEW_TOP_RIGHT_SAVE_ITEM);
            }
        } else {
            mTopTitleBar = new TopBarImpl(mContext);
            mTopTitleBar.setStartMargin(AppResource.getDimensionPixelSize(mContext, R.dimen.ux_margin_16dp));
            mTopTitleBar.setEndMargin(AppResource.getDimensionPixelSize(mContext, R.dimen.ux_margin_16dp));

            ((TopBarImpl) mTopTitleBar).setShowSolidLine(false);
        }
        mTopTitleBar.setBackgroundColor(ThemeConfig.getInstance(mContext).getB2());
        mBackItem = new BaseItemImpl(mContext, AppResource.getString(mContext, R.string.fx_string_cancel));
        mBackItem.setTextColor(ThemeUtil.getPrimaryTextColor(mContext));
        mBackItem.setTextSize(TypedValue.COMPLEX_UNIT_PX, AppResource.getDimensionPixelSize(mContext, R.dimen.ux_text_size_15sp));
        mBackItem.setId(R.id.sig_create_back);
        mBackItem.setTag(ToolbarItemConfig.SIGNATURE_VIEW_TOP_LEFT_BACK_ITEM);
        mBackItem.setOnClickListener(mOnClickListener);

        mTitleItem = new BaseItemImpl(mContext, AppResource.getString(mContext, R.string.rd_security_dsg_addSig));
        mTitleItem.setTextColorResource(R.color.t4);
        mTitleItem.setTextSize(TypedValue.COMPLEX_UNIT_PX, AppResource.getDimensionPixelSize(mContext, R.dimen.ux_text_size_18sp));
        mTitleItem.setTag(ToolbarItemConfig.SIGNATURE_VIEW_TOP_CENTER_TITLE_ITEM);
        mTitleItem.setTypeface(Typeface.defaultFromStyle(Typeface.BOLD));

        mSaveItem = new BaseItemImpl(mContext, AppResource.getString(mContext, R.string.fx_string_save));
        mSaveItem.setTextColor(ThemeUtil.getPrimaryTextColor(mContext));
        mSaveItem.setTextSize(TypedValue.COMPLEX_UNIT_PX, AppResource.getDimensionPixelSize(mContext, R.dimen.ux_text_size_15sp));
        mSaveItem.setId(R.id.sig_create_save);
        mSaveItem.setTag(ToolbarItemConfig.SIGNATURE_VIEW_TOP_RIGHT_SAVE_ITEM);
        mSaveItem.setOnClickListener(mOnClickListener);

        mTopTitleBar.addView(mBackItem, BaseBar.TB_Position.Position_LT);
        mTopTitleBar.addView(mTitleItem, BaseBar.TB_Position.Position_CENTER);
        mTopTitleBar.addView(mSaveItem, BaseBar.TB_Position.Position_RB);
    }

    private void initToolBar() {
        mTopToolBar = new TopBarImpl(mContext);
        mTopToolBar.setStartMargin(AppResource.getDimensionPixelSize(mContext, R.dimen.ux_margin_16dp));
        mTopToolBar.setEndMargin(AppResource.getDimensionPixelSize(mContext, R.dimen.ux_margin_8dp));
        mTopToolBar.setBackgroundColor(ThemeConfig.getInstance(mContext).getB2());
        mTopToolBar.setAutoCompressItemsInterval(true);
        mTopToolBar.setMiddleButtonCenter(true);
        mTopToolBar.setCenterItemInterval(AppResource.getDimensionPixelSize(mContext, R.dimen.ux_margin_24dp));
        mTopToolBar.setEndItemiInterval(AppResource.getDimensionPixelSize(mContext, R.dimen.ux_margin_4dp));

        IBaseItem pictureItem = new BaseItemImpl(mContext, R.drawable.commont_ic_picture);
        pictureItem.setId(R.id.sig_from_picture);
        pictureItem.setOnClickListener(mOnClickListener);
        pictureItem.setImageTintList(ThemeUtil.getPrimaryIconColor(mContext));
        IBaseItem cameraItem = new BaseItemImpl(mContext, R.drawable.commont_ic_camera);
        cameraItem.setId(R.id.sig_from_camera);
        cameraItem.setOnClickListener(mOnClickListener);
        cameraItem.setImageTintList(ThemeUtil.getPrimaryIconColor(mContext));
        if (mDsgUtil != null && mModuleConfig.modules.isLoadForm) {
            mCertificateItem = new BaseItemImpl(mContext, R.drawable.sign_create_add_cert);
            mCertificateItem.setId(R.id.sig_from_cert);
            mCertificateItem.setOnClickListener(mOnClickListener);
            mCertificateItem.setImageTintList(ThemeUtil.getPrimaryIconColor(mContext));
        }

        IBaseItem eraseItem = new BaseItemImpl(mContext, R.drawable.drawing_tool_eraser);
        eraseItem.setId(R.id.sig_create_delete);
        eraseItem.setOnClickListener(mOnClickListener);
        eraseItem.setImageTintList(ThemeUtil.getEnableIconColor(mContext));
        mPropView = new UIFillView(mContext);
        mPropView.setForceDarkAllowed(false);
        mPropView.setFillResource(R.drawable.annot_prop_circle_border_bg);
        mPropView.setBorderResource(R.drawable.annot_prop_circle_border_bg);
        mPropView.setBorderWidth(AppResource.getDimensionPixelSize(mContext, R.dimen.ux_item_border_unselected_width));
        mPropView.setBorderColor(AppResource.getColor(mContext, R.color.i3));
        mPropView.setFillColorFilter(AppResource.getColor(mContext, R.color.b2));
        LinearLayout linearLayout = new LinearLayout(mContext);
        linearLayout.setGravity(Gravity.CENTER);
        linearLayout.addView(mPropView);
        linearLayout.setId(R.id.sig_create_property);
        linearLayout.setOnClickListener(mOnClickListener);
        mPropItem = new UIColorItem(mContext, linearLayout) {
            @Override
            public void onItemLayout(int l, int t, int r, int b) {
                if (mUiExtensionsManager.getCurrentToolHandler() != null
                        && mUiExtensionsManager.getCurrentToolHandler().getType().equals(ToolHandler.TH_TYPE_SIGNATURE)) {
                    if (mPropertyBar.isShowing()) {
                        Rect rect = new Rect();
                        mPropItem.getContentView().getGlobalVisibleRect(rect);
                        if (SystemUiHelper.getInstance().isStatusBarShown(mActivity))
                            rect.offset(0, SystemUiHelper.getInstance().getStatusBarHeight(mActivity));
                        mPropertyBar.update(new RectF(rect), UIPopoverFrag.ARROW_TOP);
                    }
                }
            }
        };
        mPropItem.setBackgroundResource(R.drawable.tool_bar_drop_right);

//        {
//            pictureItem.setId(R.id.id_signature_toolbar_picture);
//            cameraItem.setId(R.id.id_signature_toolbar_camera);
//            if (mCertificateItem != null)
//                mCertificateItem.setId(R.id.id_signature_toolbar_cert);
//            eraseItem.setId(R.id.id_signature_toolbar_cert);
//            mPropItem.setId(R.id.id_signature_toolbar_prop);
//        }

        mTopToolBar.addView(pictureItem, BaseBar.TB_Position.Position_CENTER);
        if (mActivity == null)
            mActivity = mUiExtensionsManager.getAttachedActivity();
        if (AppDevice.hasCamera(mActivity)) {
            mTopToolBar.addView(cameraItem, BaseBar.TB_Position.Position_CENTER);
        }
        if (mCertificateItem != null && mModuleConfig.modules.isLoadForm)
            mTopToolBar.addView(mCertificateItem, BaseBar.TB_Position.Position_RB);
        mTopToolBar.addView(eraseItem, BaseBar.TB_Position.Position_RB);
        mTopToolBar.addView(mPropItem, BaseBar.TB_Position.Position_RB);

        mBottomToolBar = new BottomBarImpl(mContext);
        mBottomToolBar.setStartMargin(AppResource.getDimensionPixelSize(mContext, R.dimen.ux_margin_16dp));
        mBottomToolBar.setBackgroundColor(ThemeConfig.getInstance(mContext).getB2());
        mBottomToolBar.setAutoCompressItemsInterval(true);

        switchView  = View.inflate(mContext, R.layout.nui_sign_create_remove_white_bg, null);
        mSwitchLayout = switchView.findViewById(R.id.sign_create_remove_white_bg_btn_ll);
        ThemeUtil.setBackgroundTintList(mSwitchLayout, getSelectedButtonColorStateList());
        mSwitchBtn = switchView.findViewById(R.id.sign_create_remove_white_bg_btn);
        mSwitchLayout.setSelected(false);

        switchView.setOnClickListener(new OnClickListener() {
            @Override
            public void onClick(View v) {
                if (mSwitchLayout.isSelected()) {
                    setSelectedButtonState(false, mSwitchLayout, mSwitchBtn);
                } else {
                    setSelectedButtonState(true, mSwitchLayout, mSwitchBtn);
                }
            }
        });
        IBaseItem item = new BaseItemImpl(mContext, switchView);
        mBottomToolBar.addView(item, BaseBar.TB_Position.Position_LT);

    }

    private void setSelectedButtonState(boolean isSelected, RelativeLayout switchLayout, ImageView switchIcon) {
        if (switchLayout == null || switchIcon == null) return;
        switchLayout.setSelected(isSelected);
        RelativeLayout.LayoutParams params = (RelativeLayout.LayoutParams) switchIcon.getLayoutParams();
        if (params == null) return;
        params.removeRule(isSelected ? RelativeLayout.ALIGN_PARENT_LEFT : RelativeLayout.ALIGN_PARENT_RIGHT);
        params.addRule(isSelected ? RelativeLayout.ALIGN_PARENT_RIGHT : RelativeLayout.ALIGN_PARENT_LEFT, RelativeLayout.TRUE);
        switchIcon.setLayoutParams(params);
    }

    private ColorStateList getSelectedButtonColorStateList() {
        int disabled = AppResource.getColor(mContext, R.color.p3);
        int selected = ThemeConfig.getInstance(mContext).getPrimaryColor();
        int normal = AppResource.getColor(mContext, R.color.p1);
        return AppResource.createColorStateList(selected, disabled, normal);
    }

    private void preparePropertyBar() {
        int[] colors = new int[PropertyBar.PB_COLORS_SIGN.length];
        System.arraycopy(PropertyBar.PB_COLORS_SIGN, 0, colors, 0, colors.length);
        colors[0] = PropertyBar.PB_COLORS_SIGN[0];
        mPropertyBar.setColors(colors);
        mPropertyBar.setProperty(PropertyBar.PROPERTY_COLOR, mToolHandler.getColor());
        mPropertyBar.setProperty(PropertyBar.PROPERTY_LINEWIDTH, translate2LineWidth(mToolHandler.getDiameter()));
        mPropertyBar.setPropertyTitle(PropertyBar.PROPERTY_LINEWIDTH, AppResource.getString(mContext, R.string.pb_border_thickness));
        mPropertyBar.setArrowVisible(true);
        mPropertyBar.setAutoResetSystemUiOnDismiss(false);
        mPropertyBar.reset(getSupportedProperties());
        mPropertyBar.setPropertyChangeListener(propertyChangeListener);
    }

    private long getSupportedProperties() {
        return PropertyBar.PROPERTY_COLOR
                | PropertyBar.PROPERTY_LINEWIDTH;
    }

    private PropertyBar.PropertyChangeListener propertyChangeListener = new PropertyBar.PropertyChangeListener() {
        @Override
        public void onValueChanged(long property, int value) {
            if (property == PropertyBar.PROPERTY_COLOR) {
                if (value == mToolHandler.getColor()) return;
                mToolHandler.setColor(value);
                setInkColor(value);
                mPropView.setFillColorFilter(value);
            } else if (property == PropertyBar.PROPERTY_SELF_COLOR) {
                if (value == mToolHandler.getColor()) return;
                mToolHandler.setColor(value);
                setInkColor(value);
                mPropView.setFillColorFilter(value);
            }
        }

        @Override
        public void onValueChanged(long property, float value) {
            if (property == PropertyBar.PROPERTY_LINEWIDTH) {
                if (mToolHandler.getDiameter() == unTranslate(value)) return;
                float diameter = unTranslate(value);
                mToolHandler.setDiameter(diameter);
                setInkDiameter(diameter);
            }
        }

        @Override
        public void onValueChanged(long property, String value) {

        }
    };

    private float unTranslate(float r) {
        if (r <= 1) {
            r = 1.4999f;
        }
        return (r - 1) / 2;
    }


    private float translate2LineWidth(float d) {
        return (2 * d + 1);
    }

    public ViewGroup getView() {
        return mContentView;
    }

    public void init(int width, int height, String dsgPath) {
        if (!"eSign".equals(dsgPath)) {
            mInkItem = new SignatureInkItem();
            mInkItem.bitmap = mDrawView.getBmp();
            mRect.setEmpty();
            mInkItem.rect = mRect;
            mInkItem.color = mToolHandler.getColor();
            mInkItem.diameter = mToolHandler.getDiameter();
            mInkItem.dsgPath = mCurDsgPath;
        } else {
            dsgPath = null;
        }
        if ("".equals(dsgPath)) {
            dsgPath = null;
        }

        if (mInkItem == null) {
            mInkItem = new SignatureInkItem();
            mInkItem.bitmap = mDrawView.getBmp();
            mRect.setEmpty();
            mInkItem.rect = mRect;
            mInkItem.color = mToolHandler.getColor();
            mInkItem.diameter = mToolHandler.getDiameter();
            mInkItem.dsgPath = mCurDsgPath;
        }
        _enableControlViews(true);
        mBmpWidth = width;
        mBmpHeight = height - (int) AppResource.getDimension(mContext, R.dimen.ux_topbar_height) * 2;

        mValidRect.set(AppDisplay.dp2px(3),
                AppDisplay.dp2px(7),
                mBmpWidth - AppDisplay.dp2px(3),
                mBmpHeight - AppDisplay.dp2px(7));
        mKey = null;
        mRect.setEmpty();
        changeDrawBitmap(null, false);

        com.foxit.uiextensions.config.Config config = mUiExtensionsManager.getConfig();
        if (mToolHandler.getColor() == 0)
            mToolHandler.setColor(config.uiSettings.signature.color);
        mPropView.setFillColorFilter(mToolHandler.getColor());
        if (mToolHandler.getDiameter() == 0) {
            float diameter = unTranslate(config.uiSettings.signature.thickness);
            mToolHandler.setDiameter(diameter);
        }
        if (mBitmap == null) {
            try {
                mBitmap = Bitmap.createBitmap(mBmpWidth, mBmpHeight, Config.ARGB_8888);
            } catch (OutOfMemoryError error) {
                error.printStackTrace();
                if (mListener != null) {
                    mListener.onBackPressed();
                }
                return;
            }
        }
        mBitmap.eraseColor(0x00FFFFFF);
        mCanDraw = false;
        initCanvas();
        mCurDsgPath = dsgPath;
        setCertificateItem(mCurDsgPath);
        if (mDsgModule != null && !TextUtils.isEmpty(mCurDsgPath)) {

            CertificateFileInfo fileInfo = mDsgModule.getDigitalSignatureUtil().getCertInfo(mCurDsgPath);
            mInfo = mDsgModule.getDigitalSignatureUtil().getCertSupport().verifyPwd(fileInfo.filePath, fileInfo.password);
        }
    }

    public void init(int width, int height, SignatureInkItem inkItem) {
        mInkItem = inkItem;
        init(width, height, inkItem.key, inkItem.bitmap, inkItem.rect, inkItem.color, inkItem.diameter, inkItem.dsgPath);
    }

    public void init(int width, int height, String key, Bitmap bitmap, Rect rect, int color, float diameter, String dsgPath) {
        if (bitmap == null || rect == null) {
            init(width, height, dsgPath);
            return;
        }
        _enableControlViews(true);
        mBmpWidth = width;
        mBmpHeight = height - (int) AppResource.getDimension(mContext, R.dimen.ux_topbar_height) * 2;

        mValidRect.set(AppDisplay.dp2px(3),
                AppDisplay.dp2px(7),
                mBmpWidth - AppDisplay.dp2px(3),
                mBmpHeight - AppDisplay.dp2px(7));
        mKey = key;
        mRect.set(rect);
        if (mBitmap != null) {
            if (!mBitmap.isRecycled()) mBitmap.recycle();
            mBitmap = null;
        }
        int[] colors;
        try {
            mBitmap = Bitmap.createBitmap(mBmpWidth, mBmpHeight, Config.ARGB_8888);
            colors = new int[mBmpWidth * mBmpHeight];
        } catch (OutOfMemoryError error) {
            error.printStackTrace();
            if (mListener != null) {
                mListener.onBackPressed();
            }
            return;
        }
        try {
            int bWidth = Math.min(bitmap.getWidth(), mBmpWidth);
            int bHeight = Math.min(bitmap.getHeight(), mBmpHeight);
            bitmap.getPixels(colors, 0, mBmpWidth, 0, 0, bWidth, bHeight);
            mBitmap.setPixels(colors, 0, mBmpWidth, 0, 0, bWidth, bHeight);
        } catch (Exception e) {
            int oldVerBmpHeight = height - (int) AppResource.getDimension(mContext, R.dimen.ux_toolbar_height_phone);//for supper old version
            if (oldVerBmpHeight > bitmap.getHeight()) {
                bitmap.getPixels(colors, 0, mBmpWidth, 0, 0, mBmpWidth, bitmap.getHeight());
                mBitmap.setPixels(colors, 0, mBmpWidth, 0, 0, mBmpWidth, bitmap.getHeight());
            } else {
                bitmap.getPixels(colors, 0, mBmpWidth, 0, 0, mBmpWidth, oldVerBmpHeight);
                mBitmap.setPixels(colors, 0, mBmpWidth, 0, 0, mBmpWidth, oldVerBmpHeight);
            }

        }
        bitmap.recycle();
        bitmap = null;
        mToolHandler.setColor(color);
        mPropView.setFillColorFilter(color);
        mToolHandler.setDiameter(diameter);
        mCanDraw = false;

        adjustCanvasRect();
        if (mInkItem != null) {
            mInkItem.setRect(mRect);
        }
        initCanvas();
        mCurDsgPath = dsgPath;
        changeDrawBitmap(mBitmap, false);
        setCertificateItem(mCurDsgPath);
        if (mInkItem != null) {
            mInkItem.dsgPath = mCurDsgPath;
            mIsFromDraw =  mInkItem.isPic;
            if(mIsFromDraw && mInkItem.isClearBg){
                switchView.setEnabled(false);
                mSwitchLayout.setEnabled(false);
                setSelectedButtonState(true, mSwitchLayout, mSwitchBtn);
            }
        }
        if (mDsgModule != null && !TextUtils.isEmpty(mCurDsgPath)) {
            CertificateFileInfo fileInfo = mDsgModule.getDigitalSignatureUtil().getCertInfo(mCurDsgPath);
            mInfo = mDsgModule.getDigitalSignatureUtil().getCertSupport().verifyPwd(fileInfo.filePath, fileInfo.password);
        }
    }

    public void reDraw(int width, int height) {
        mBmpWidth = width;
        mBmpHeight = height - (int) AppResource.getDimension(mContext, R.dimen.ux_topbar_height) * 2;

        mValidRect.set(AppDisplay.dp2px(3),
                AppDisplay.dp2px(7),
                mBmpWidth - AppDisplay.dp2px(3),
                mBmpHeight - AppDisplay.dp2px(7));

        if (mBitmap == null) return;

        int srcWidth = mBitmap.getWidth();
        int srcHeight = mBitmap.getHeight();
        float wscale = (float) mBmpWidth / srcWidth;
        float hscale = (float) mBmpHeight / srcHeight;
        float scale = Math.min(wscale, hscale);
        int dstWidth = (int) (srcWidth * scale);
        int dstHeight = (int) (srcHeight * scale);

        Bitmap newBitmap = zoomBitmap(mBitmap, dstWidth, dstHeight);
        Bitmap bgBitMap = Bitmap.createBitmap(mBmpWidth, mBmpHeight, Config.ARGB_8888);
        bgBitMap.eraseColor(0x00FFFFFF);
        Paint paint = new Paint();
        Canvas canvas = new Canvas(bgBitMap);
        PaintFlagsDrawFilter pfd = new PaintFlagsDrawFilter(0, Paint.ANTI_ALIAS_FLAG | Paint.FILTER_BITMAP_FLAG);
        paint.setFilterBitmap(true);
        paint.setAntiAlias(true);
        canvas.setDrawFilter(pfd);
        Rect baseRect = new Rect(0, 0, bgBitMap.getWidth(), bgBitMap.getHeight());
        Rect frontRect = new Rect(0, 0, newBitmap.getWidth(), newBitmap.getHeight());
        canvas.drawBitmap(newBitmap, frontRect, baseRect, paint);

        canvas.save();
        canvas.restore();
        newBitmap.recycle();
        mBitmap = bgBitMap;

        float x = Math.abs((float) mBmpWidth - dstWidth) / 2;
        float y = Math.abs((float) mBmpHeight - dstHeight) / 2;
        Rect contentRect = new Rect((int) x, (int) y, (int) (x + dstWidth + 0.5), (int) (y + dstHeight + 0.5));
        mRect.set(contentRect);
        initCanvas();
    }

    public void unInit() {
        releaseCanvas();
        mPropertyBar.setDismissListener(null);
        mFragment = null;
    }

    public void setMiddleContentViewOnly() {
        _enableControlViews(false);
    }

    private void _enableControlViews(boolean enable) {
        if (enable) {
            mTopBarLayout.setVisibility(View.VISIBLE);
            mTopToolLayout.setVisibility(View.VISIBLE);
            mBottomToolLayout.setVisibility(View.VISIBLE);
        } else {
            mTopBarLayout.setVisibility(View.GONE);
            mTopToolLayout.setVisibility(View.GONE);
            mBottomToolLayout.setVisibility(View.GONE);
        }
    }

    private void setCertificateItem(String dsgPath) {
        if (mCertificateItem != null) {
            if (!AppUtil.isEmpty(dsgPath)) {
                File file = new File(dsgPath);
                UIToast.getInstance(mContext).show(AppResource.getString(mContext,
                        R.string.sg_cert_current_name_title) + file.getName());
            }
        }
    }

    private OnClickListener mOnClickListener = new OnClickListener() {

        @Override
        public void onClick(View v) {
            if (AppUtil.isFastDoubleClick()) return;
            int id = v.getId();
            if (R.id.sig_create_back == id) {
                if (mListener != null) {
                    mListener.onBackPressed();
                }
            } else if (R.id.sig_create_property == id) {
                preparePropertyBar();
                Rect rect = new Rect();
                mPropItem.getContentView().getGlobalVisibleRect(rect);
                if (SystemUiHelper.getInstance().isStatusBarShown(mActivity))
                    rect.offset(0, SystemUiHelper.getInstance().getStatusBarHeight(mActivity));
                mPropertyBar.show(getView(), new RectF(rect), false, UIPopoverFrag.ARROW_TOP);
            } else if (R.id.sig_create_delete == id) {
                clearCanvas();
            } else if (R.id.sig_create_save == id) {
                if (mDrawView == null || mDrawView.getBmp() == null) return;
                if (mIsFromDraw &&
                        (mBitmap != null || mSrcBitmap != null) &&
                        mSwitchLayout != null && mSwitchLayout.isSelected() &&
                        switchView.isEnabled()) {
                    Bitmap srcBitmap = mSrcBitmap != null ? mSrcBitmap : mBitmap;
                    mInkItem.bitmap = removeWhiteBg(srcBitmap);
                } else{
                    mInkItem.bitmap = mDrawView.getBmp();
                }
                if (!TextUtils.isEmpty(mCurDsgPath) && mInkItem != null) {
                    showAppearanceDialog();
                    return;
                }
                saveSignCallback(false);
            } else if (R.id.sig_from_picture == id) {
                Intent intent = new Intent(Intent.ACTION_PICK, MediaStore.Images.Media.EXTERNAL_CONTENT_URI);
                mFragment.startActivityForResult(intent, ActRequestCode.REQ_FILE_FROM_GALLERY);
            } else if (R.id.sig_from_camera == id) {
                startCamera();
            } else if (R.id.sig_from_cert == id) {
                final Runnable doRunnable = new Runnable() {
                    @Override
                    public void run() {
                        mDsgUtil.addCertList(mFragment.isFullScreen(), new IDigitalSignatureCallBack() {
                            @Override
                            public void onCertSelect(String path, CertificateFileInfo fileInfo) {
                                if (mFragment != null
                                        && mFragment.isFullScreen()
                                        && mFragment.getDialog() != null)
                                    SystemUiHelper.getInstance().hideSystemUI(mFragment.getDialog().getWindow());
                                if (!AppUtil.isEmpty(path) && !AppUtil.isEmpty(fileInfo.fileName)) {
                                    UIToast.getInstance(mContext).show(AppResource.getString(mContext, R.string.sg_cert_current_name_title) + fileInfo.fileName);
                                    mCurDsgPath = path;

//                                    DigitalSignatureUtil.copyFile(mContext,fileInfo.filePath, mCurDsgPath);
                                    if (mInkItem != null) {
                                        mInkItem.dsgPath = mCurDsgPath;
                                    }
                                } else {
                                    mCurDsgPath = null;
                                }
                            }
                        });
                    }
                };

                if (((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getPermissionProvider() != null) {
                    ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getPermissionProvider().checkPermission(
                            IPermissionProvider.FUNCTION_CERT_VERITY, new IPermissionProvider.IPermissionState() {
                                @Override
                                public void onState(int state) {
                                    if (state == IPermissionProvider.PERMISSION_STATE_SHOW) {
                                        doRunnable.run();
                                    }
                                }
                            });
                } else {
                    doRunnable.run();
                }
            }
        }
    };

    private void saveSignCallback(boolean isSignTitle) {
        boolean firstCreate = AppSharedPreferences.getInstance(mContext).getBoolean("firstCreateSignature", "FIRST_CREATE_SIGNATURE", false);
        if(!firstCreate && !mIsFromSignatureField) {
            Toast.makeText(mContext, AppResource.getString(mContext, R.string.signature_tap_insert), Toast.LENGTH_SHORT).show();
            AppSharedPreferences.getInstance(mContext).setBoolean("firstCreateSignature", "FIRST_CREATE_SIGNATURE", true);
        }
        mToolHandler.showProgressDialog();
        saveSign(isSignTitle, new Event.Callback() {
            @Override
            public void result(Event event, boolean success) {
                mToolHandler.dismissProgressDialog();
                if (!mIsFromSignatureField) {
                    if (mUiExtensionsManager.getState() == ReadStateConfig.STATE_FILLSIGN) {
                        if (!(mUiExtensionsManager.getCurrentToolHandler() instanceof FillSignToolHandler)) {
                            FillSignModule fillSignModule = (FillSignModule) mUiExtensionsManager.getModuleByName(Module.MODULE_NAME_FIllSIGN);
                            if (fillSignModule != null)
                                fillSignModule.activateSignature();
                        }
                    } else if (!(mUiExtensionsManager.getCurrentToolHandler() instanceof SignatureToolHandler)) {
                        SignatureModule signatureModule = (SignatureModule) mUiExtensionsManager.getModuleByName(Module.MODULE_NAME_PSISIGNATURE);
                        mUiExtensionsManager.setCurrentToolHandler(signatureModule.getToolHandler());
                    }
                }
            }
        });
    }

    private void saveSign(final boolean isSignTitle, @NonNull final Event.Callback callback) {
        if (mIsFromSignatureField && TextUtils.isEmpty(mCurDsgPath)) {

            if (mActivity != null) {
                final UITextEditDialog dialog = new UITextEditDialog(mActivity, UIDialog.NO_INPUT);
                dialog.setTitle(AppResource.getString(mContext, R.string.fx_string_warning));
                dialog.getPromptTextView().setText(AppResource.getString(mContext, R.string.sg_cert_add_text));
                dialog.getCancelButton().setOnClickListener(new View.OnClickListener() {
                    @Override
                    public void onClick(View v) {
                        dialog.dismiss();
                    }
                });
                dialog.getOKButton().setOnClickListener(new View.OnClickListener() {
                    @Override
                    public void onClick(View v) {
                        mDsgUtil.addCertList(new IDigitalSignatureCallBack() {
                            @Override
                            public void onCertSelect(String path, CertificateFileInfo fileInfo) {
                                if (!AppUtil.isEmpty(path) && !AppUtil.isEmpty(fileInfo.fileName)) {
                                    UIToast.getInstance(mContext).show(AppResource.getString(mContext,
                                            R.string.sg_cert_current_name_title) + fileInfo.fileName);
                                    mCurDsgPath = path;
//                                    DigitalSignatureUtil.copyFile(mContext,fileInfo.filePath, mCurDsgPath);
                                    if (mInkItem != null) {
                                        mInkItem.dsgPath = mCurDsgPath;
                                    }
                                } else {
                                    mCurDsgPath = null;
                                }
                            }
                        });
                        dialog.dismiss();
                    }
                });
                dialog.show();
            } else {
                changeDrawBitmap(null, true);
            }

            callback.result(null, true);
            return;
        }

        final Bitmap bitmap = mInkItem.bitmap;
//        if (mSwitchLayout != null && mSwitchLayout.isSelected() && switchView.isEnabled()) {
//           Bitmap resultBitmap =  removeWhiteBg(bitmap);
//            bitmap = mInkItem.bitmap;
//            AppThreadManager.getInstance().startThread(new Runnable() {
//                @Override
//                public void run() {
//                    AppThreadManager.getInstance().runOnUiThread(new Runnable() {
//                        @Override
//                        public void run() {
//                            saveSignData(isSignTitle, callback, bitmap);
//                        }
//                    });
//                }
//            });
//
//        } else {
            saveSignData(isSignTitle, callback, bitmap);
//        }
    }

    private void saveSignData(boolean isSignTitle, @NonNull Event.Callback callback, Bitmap bitmap) {
        if (mKey == null) {
            if (isSignTitle) {
                mInkItem.setBitmap(bitmap);
                mInkItem.setRect(mRect);
                mInkItem.setColor(mToolHandler.getColor());
                mInkItem.setDiameter(mToolHandler.getDiameter());
                mInkItem.isClearBg = mSwitchLayout.isSelected();
                mInkItem.isPic = mIsFromDraw;
                SignatureDataUtil.insertData(mContext, mInkItem);
            } else {
                SignatureDataUtil.insertData(mContext, bitmap, mRect, mToolHandler.getColor(), mToolHandler.getDiameter(), mCurDsgPath,mSwitchLayout.isSelected(),mIsFromDraw);
            }
        } else {
            if (isSignTitle) {
                mInkItem.setBitmap(bitmap);
                mInkItem.setRect(mRect);
                mInkItem.setColor(mToolHandler.getColor());
                mInkItem.setDiameter(mToolHandler.getDiameter());
                mInkItem.isClearBg = mSwitchLayout.isSelected();
                mInkItem.isPic = mIsFromDraw;
                SignatureDataUtil.updateByKey(mContext, mInkItem);
            } else {
                SignatureDataUtil.updateByKey(mContext, mKey, bitmap, mRect, mToolHandler.getColor(), mToolHandler.getDiameter(), mCurDsgPath,mSwitchLayout.isSelected(),mIsFromDraw);
            }
        }

        if (mListener != null) {
            if (!isSignTitle) {
                if (mInkItem == null) {
                    mInkItem = new SignatureInkItem();
                    mInkItem.setBitmap(bitmap);
                    mInkItem.setRect(mRect);
                    mInkItem.setColor(mToolHandler.getColor());
                    mInkItem.setDiameter(mToolHandler.getDiameter());
                }
            }

            mListener.result(mInkItem);

        }
        callback.result(null, true);
    }

    public byte[] bitmapToByteArrayARGB8888(Bitmap bitmap) {

        ByteBuffer byteBuffer = ByteBuffer.allocate(bitmap.getByteCount());

        bitmap.copyPixelsToBuffer(byteBuffer);
//        byteBuffer.rewind();
        return byteBuffer.array();
    }

    private Bitmap removeWhiteBg(Bitmap sourceBitmap) {
        try {
            Bitmap srcBitmap = Bitmap.createBitmap(sourceBitmap.getWidth(), sourceBitmap.getHeight(), Bitmap.Config.ARGB_8888);
            Canvas canvas = new Canvas(srcBitmap);
            Paint paint = new Paint(Paint.ANTI_ALIAS_FLAG);
            canvas.drawBitmap(sourceBitmap, 0, 0, paint);

            int format =  com.foxit.sdk.common.Bitmap.e_DIBArgb;
            int pitch = srcBitmap.getRowBytes();
            byte[]  buffer = null;
            buffer = bitmapToByteArrayARGB8888(srcBitmap);

            com.foxit.sdk.common.Bitmap bitmap = new com.foxit.sdk.common.Bitmap(srcBitmap.getWidth(),srcBitmap.getHeight(),format,buffer,pitch);
            Bitmap maskBitmap =  bitmap.fadeOutBackground(com.foxit.sdk.common.Bitmap.e_EnhanceAlgoGrayLevelTransformation, com.foxit.sdk.common.Bitmap.e_ThresholdAlgoFixedThreshold, 1.0f);
            int srcWidth = maskBitmap.getWidth();
            int srcHeight = maskBitmap.getHeight();
            float wscale = (float) mBmpWidth / srcWidth;
            float hscale = (float) mBmpHeight / srcHeight;
            float scale = Math.min(wscale, hscale);
            int dstWidth = (int) (srcWidth * scale);
            int dstHeight = (int) (srcHeight * scale);
            Bitmap newBitmap = zoomBitmap(maskBitmap, dstWidth, dstHeight);
            Bitmap bgBitMap = Bitmap.createBitmap(mBmpWidth, mBmpHeight, Config.ARGB_8888);
            bgBitMap.eraseColor(0x00FFFFFF);
            Canvas canvas2 = new Canvas(bgBitMap);
            float x = Math.abs((float) mBmpWidth - dstWidth) / 2;
            float y = Math.abs((float) mBmpHeight - dstHeight) / 2;
            canvas2.drawBitmap(newBitmap, x, y, null);
            maskBitmap.recycle();
            newBitmap.recycle();
            srcBitmap.recycle();
            return  bgBitMap;

        } catch (PDFException e) {
            throw new RuntimeException(e);
        }

    }

    private void adjustCanvasRect() {
        if (mBitmap == null) return;
        if (mRect.left < 0) mRect.left = 0;
        if (mRect.top < 0) mRect.top = 0;
        if (mRect.right > mBmpWidth) mRect.right = mBmpWidth;
        if (mRect.bottom > mBmpHeight) mRect.bottom = mBmpHeight;
    }

    private void initCanvas() {
        if (mBitmap == null) return;

        try {
            mPsi = new PSI(mBitmap, true);
            mPsi.setColor(mToolHandler.getColor());
            int diameter = (int) (mToolHandler.getDiameter() * INK_DIAMETER_SCALE);
            if (diameter == 0) {
                diameter = 1;
            }
            mPsi.setDiameter(diameter);
            mPsi.setOpacity(1f);

            mHandler.sendEmptyMessage(MSG_DRAW);
        } catch (PDFException e) {
            e.printStackTrace();
        }

    }

    private void setInkColor(int color) {
        if (mPsi == null || mPsi.isEmpty()) return;
        try {
            mPsi.setColor(color);
            mHandler.sendEmptyMessage(MSG_COLOR);
        } catch (PDFException e) {
            e.printStackTrace();
        }
    }

    private void setInkDiameter(float diameter) {
        if (mPsi == null || mPsi.isEmpty()) return;
        try {
            int tmp = (int) (diameter * INK_DIAMETER_SCALE);
            if (tmp == 0) {
                tmp = 1;
            }
            mPsi.setDiameter(tmp);

            mHandler.sendEmptyMessage(MSG_DIAMETER);
        } catch (PDFException e) {
            e.printStackTrace();
        }
    }

    public void clearCanvas() {
        mRect.setEmpty();
        mBitmap.eraseColor(0x00FFFFFF);
        mHandler.sendEmptyMessage(MSG_CLEAR);
        if(mInkItem != null) {
            mInkItem.isClearBg = false;
            if(mSwitchLayout.isSelected()){
                setSelectedButtonState(false, mSwitchLayout, mSwitchBtn);
            }
        }
        switchView.setEnabled(true);
        mSwitchLayout.setEnabled(true);
        mIsFromDraw = false;
    }

    private boolean mBmpChanged;
    private void addPoint(final List<PointF> points, final List<Float> pressures, final int flag) {
        try {
            if (mPsi == null || mPsi.isEmpty()) return;
            for (int i = 0; i < points.size(); i++) {
                PointF point = points.get(i);
                Float pressure = pressures != null ? pressures.get(i) : 1.0f;
                mPsi.addPoint(new com.foxit.sdk.common.fxcrt.PointF(point.x, point.y), flag, pressure);
            }

            com.foxit.sdk.common.fxcrt.RectF rect = mPsi.getContentsRect();
            Rect contentRect = new Rect((int) rect.getLeft(), (int) rect.getTop(), (int) (rect.getRight() + 0.5), (int) (rect.getBottom() + 0.5));
            if (mRect.isEmpty())
                mRect.set(contentRect);
            else
                mRect.union(contentRect);

            if (!mRect.isEmpty()) {
                adjustCanvasRect();
                changeDrawBitmap(mBitmap, true);
            }
            mDrawView.invalidate(contentRect);
        } catch (PDFException e) {
            e.printStackTrace();
        }
    }

    private void releaseCanvas() {
        mHandler.sendEmptyMessage(MSG_RELEASE);
    }

    class DrawView extends View {

        private Paint mPaint;
        private PointF mDownPt;
        private PointF mLastPt;

        public DrawView(Context context) {
            super(context);
            this.setLayoutParams(new LayoutParams(LayoutParams.MATCH_PARENT, LayoutParams.MATCH_PARENT));
            if (Build.VERSION.SDK_INT >= 11) {
                this.setLayerType(View.LAYER_TYPE_SOFTWARE, null);
            }
            mPaint = new Paint();
            mPaint.setAntiAlias(true);
            mPaint.setFilterBitmap(true);

            mDownPt = new PointF();
            mLastPt = new PointF();
        }

        @Override
        protected void onDraw(Canvas canvas) {
            if (mPsi != null && !mPsi.isEmpty()) {
                canvas.drawBitmap(mBitmap, 0, 0, mPaint);
            }
        }

        private float getDistanceOfTwoPoint(PointF p1, PointF p2) {
            return (float) Math.sqrt((p1.x - p2.x) * (p1.x - p2.x) + (p1.y - p2.y) * (p1.y - p2.y));
        }

        @Override
        protected void onAttachedToWindow() {
            super.onAttachedToWindow();
        }

        @Override
        protected void onDetachedFromWindow() {
            super.onDetachedFromWindow();
        }

        @Override
        public boolean onTouchEvent(MotionEvent event) {
            if (!mCanDraw || mListener == null || !mListener.canDraw()) return false;
            if(mIsFromDraw) return false;
            int count = event.getPointerCount();
            PointF point = new PointF(event.getX(), event.getY());
            int action = event.getAction();
            switch (action) {
                case MotionEvent.ACTION_DOWN:
                    if (mValidRect.contains((int) event.getX(), (int) event.getY())) {
                        List<PointF> points = new ArrayList<PointF>();
                        points.add(point);
                        addPoint(points, null, Path.e_TypeMoveTo);

                        mDownPt.set(point);
                        mLastPt.set(point);
                    }
                    break;
                case MotionEvent.ACTION_MOVE:
                    if (mValidRect.contains((int) event.getX(), (int) event.getY())) {
                        List<PointF> points = new ArrayList<PointF>();
                        points.add(point);
                        addPoint(points, null, Path.e_TypeLineTo);

                        mLastPt.set(point);
                    }
                    break;
                case MotionEvent.ACTION_UP:
                    if (mDownPt.equals(mLastPt)) {
                        PointF movePoint = new PointF(point.x, point.y);
                        movePoint.offset(translate2LineWidth(mToolHandler.getDiameter()) / 2, 0);
                        List<PointF> points = new ArrayList<PointF>();
                        points.add(movePoint);
                        addPoint(points, null, Path.e_TypeLineTo);
                    }
                    List<PointF> points = new ArrayList<PointF>();
                    points.add(point);
                    addPoint(points, null, Path.e_TypeLineToCloseFigure);

                    mDownPt.set(0, 0);
                    mLastPt.set(0, 0);
                    break;
                case MotionEvent.ACTION_CANCEL:
                    mDownPt.set(0, 0);
                    mLastPt.set(0, 0);
                default:
                    break;
            }
            return true;
        }

        public Bitmap getBmp() {
            Bitmap bitmap = null;
            if (mBitmap != null) {
                bitmap = Bitmap.createBitmap(mBitmap);
            }
            return bitmap;
        }

    }

    public void setIsFromSignatureField(boolean isFromSignatureField) {
        mIsFromSignatureField = isFromSignatureField;
        if (mIsFromSignatureField) {
            if (mDsgUtil != null && !mModuleConfig.modules.isLoadForm) {
                mCertificateItem = new BaseItemImpl(mContext, R.drawable.sign_create_add_cert);
                mCertificateItem.setId(R.id.sig_from_cert);
                mCertificateItem.setOnClickListener(mOnClickListener);
                mCertificateItem.setImageTintList(ThemeUtil.getPrimaryIconColor(mContext));
                mTopToolBar.addView(mCertificateItem, BaseBar.TB_Position.Position_RB, 0);
            }
        } else {
            if (mCertificateItem != null && !mUiExtensionsManager.getDocumentManager().withAddPermission())
                mCertificateItem.setEnable(false);
        }
    }

    public void setActivity(Activity activity, SignatureFragment fragment) {
        mActivity = activity;
        mFragment = fragment;
    }

    public void onActivityResult(int requestCode, int resultCode, @Nullable Intent data) {
        if (mPdfViewCtrl.getDoc() == null) return;
        if (resultCode != Activity.RESULT_OK) return;

        String path = "";
        if (requestCode == ActRequestCode.REQ_FILE_FROM_GALLERY) {
            if (data == null) return;
            Uri uri = data.getData();
            path = AppFileUtil.getFilePathFromUri(mContext, uri);
            path = AppFileUtil.saveToScopedCache(path, uri);
        } else if (requestCode == ActRequestCode.REQ_FILE_FROM_CAMERA) {
            path = mCapturePath;
        }

        if (!AppUtil.isEmpty(path)) {
            if(mSwitchLayout.isSelected()){
                setSelectedButtonState(false, mSwitchLayout, mSwitchBtn);
            }
            mIsFromDraw = true;
            mBitmap.eraseColor(0x00FFFFFF);
            copyBitmap(path);
            changeDrawBitmap(mBitmap, true);
            AppFileUtil.deleteScopedCacheFile(path);
            switchView.setEnabled(true);
            mSwitchLayout.setEnabled(true);
        }

    }

    Bitmap mSrcBitmap = null;
    private void copyBitmap(String path) {
        if (mBitmap != null && !mBitmap.isRecycled())
            mBitmap.recycle();

        Bitmap srcBitmap = BitmapFactory.decodeFile(path);
        mSrcBitmap = BitmapFactory.decodeFile(path);
        int srcWidth = srcBitmap.getWidth();
        int srcHeight = srcBitmap.getHeight();
        float wscale = (float) mBmpWidth / srcWidth;
        float hscale = (float) mBmpHeight / srcHeight;
        float scale = Math.min(wscale, hscale);
        int dstWidth = (int) (srcWidth * scale);
        int dstHeight = (int) (srcHeight * scale);
        Bitmap newBitmap = zoomBitmap(srcBitmap, dstWidth, dstHeight);

        Bitmap bgBitMap = Bitmap.createBitmap(mBmpWidth, mBmpHeight, Config.ARGB_8888);
        bgBitMap.eraseColor(0x00FFFFFF);

        Canvas canvas = new Canvas(bgBitMap);
        float x = Math.abs((float) mBmpWidth - dstWidth) / 2;
        float y = Math.abs((float) mBmpHeight - dstHeight) / 2;
        canvas.drawBitmap(newBitmap, x, y, null);

        srcBitmap.recycle();
        newBitmap.recycle();
        mBitmap = bgBitMap;

        Rect contentRect = new Rect((int) x, (int) y, (int) (x + dstWidth + 0.5), (int) (y + dstHeight + 0.5));
        mRect.set(contentRect);
        initCanvas();
    }

    public static Bitmap zoomBitmap(Bitmap bm, int newWidth, int newHeight) {
        int width = bm.getWidth();
        int height = bm.getHeight();
        float scaleWidth = ((float) newWidth) / width;
        float scaleHeight = ((float) newHeight) / height;
        Matrix matrix = new Matrix();
        matrix.postScale(scaleWidth, scaleHeight);
        Bitmap newbm = Bitmap.createBitmap(bm, 0, 0, width, height, matrix, true);
        return newbm;
    }

    private void startCamera() {
        mFragment.requestPermissions(android.Manifest.permission.CAMERA,
                new String[]{android.Manifest.permission.CAMERA},
                ActRequestCode.REQ_CAMERA_PERMISSION,
                new Event.Callback() {
                    @Override
                    public void result(Event event, boolean success) {
                        if (success) {
                            File outFile = new File(mCapturePath);
                            if (outFile.getParentFile() == null || !outFile.getParentFile().exists()) {
                                if (!outFile.mkdirs()) return;
                            }
                            try {
                                if (outFile.exists())
                                    outFile.delete();
                                outFile.createNewFile();
                            } catch (Exception e) {
                                e.printStackTrace();
                            }

                            Intent intent = new Intent();
                            Uri uri;
                            if (Build.VERSION.SDK_INT > 23) {//Build.VERSION_CODES.M
                                uri = FileProvider.getUriForFile(mContext, AppIntentUtil.getFileProviderName(mContext), outFile);
                                intent.addFlags(Intent.FLAG_GRANT_READ_URI_PERMISSION);
                            } else
                                uri = Uri.fromFile(outFile);
                            // set the action to open system camera.
                            intent.setAction(MediaStore.ACTION_IMAGE_CAPTURE);
                            intent.putExtra(MediaStore.EXTRA_OUTPUT, uri);
                            mFragment.startActivityForResult(intent, ActRequestCode.REQ_FILE_FROM_CAMERA);
                        } else {
                            UIToast.getInstance(mContext).show(AppResource.getString(mContext, R.string.fx_permission_denied));
                        }
                    }
                });
    }

    public Bitmap getBitmap() {
        return mDrawView.getBmp();
    }

    public Rect getBitmapRect() {
        Rect rect = new Rect();
        if (!mRect.isEmpty()) {
            rect.set(mRect);
        } else {
            if (mBitmap != null) {
                rect.set(0, 0, mBitmap.getWidth(), mBitmap.getHeight());
            }
        }
        return rect;
    }

    public boolean isEmpty() {
        return mRect.isEmpty() || mBitmap == null || mBitmap.getWidth() < 1;
    }

    void updateTheme() {
        if (mTopTitleBar != null) {
            mTopTitleBar.setBackgroundColor(ThemeConfig.getInstance(mContext).getB2());
            mBackItem.setTextColor(ThemeUtil.getPrimaryTextColor(mContext));
            mTitleItem.setTextColorResource(R.color.t4);
            mSaveItem.setTextColor(ThemeUtil.getPrimaryTextColor(mContext));
        }
        if (mPropertyBar != null) {
            if (mPropertyBar instanceof PropertyBarImpl) {
                ((PropertyBarImpl) mPropertyBar).dismissPopup();
            }
            mPropertyBar.updateTheme();
        }
        if (mListener != null) {
            mListener.onBackPressed();
        }
        if (mDsgUtil != null) {
            mDsgUtil.dismissPfxDialog();
        }
    }

    private void changeDrawBitmap(Bitmap bitmap, boolean modify) {
        mSaveItem.setEnable(bitmap != null);
        if (mBitmapChangeCallbackList.size() > 0) {
            for (ISigBitmapChangeCallback callback : mBitmapChangeCallbackList) {
                if (callback != null)
                    callback.onBitmapChanged(bitmap, modify);
            }
        }
    }

    public void addBitmapChangedCallback(@NonNull ISigBitmapChangeCallback callback) {
        mBitmapChangeCallbackList.add(callback);
    }

    public void showAppearanceDialog() {
        Activity activity = mUiExtensionsManager.getAttachedActivity();

        mSignAppearanceDialog = new SignatureAppearanceDialog(activity, mParent, mPdfViewCtrl, mInkItem);
        mSignAppearanceDialog.setSignCallback(new SignatureAppearanceDialog.ISignAppearanceCallback() {
            @Override
            public void onDismiss(SignatureInkItem inkItem) {
                Context context = mUiExtensionsManager.getAttachedActivity();
                if (context == null) {
                    return;
                }
                String recentTitle = inkItem.getTitle();
                if (TextUtils.isEmpty(recentTitle)) {
                    saveSignCallback(false);
                } else {
                    SignatureInkItem titleInkItem = SignatureDataUtil.getTitleByKey(mContext, inkItem.getTitle());
                    mInkItem.setTitle(titleInkItem.getTitle());
                    mInkItem.setNameOpen(titleInkItem.isNameOpen() ? 1 : 0);
                    mInkItem.setDistinguishedNameOpen(titleInkItem.isDistinguishedNameOpen() ? 1 : 0);
                    mInkItem.setDateOpen(titleInkItem.isDateOpen() ? 1 : 0);
                    mInkItem.setVersionOpen(titleInkItem.isVersionOpen() ? 1 : 0);
                    mInkItem.setLogo(titleInkItem.isLogo() ? 1 : 0);
                    mInkItem.setLabels(titleInkItem.isLabels() ? 1 : 0);
                    if (inkItem.isNameOpen()) {
                        String publisher = mInfo.publisher == null ? "":mInfo.publisher;
                        mInkItem.setName(publisher);
                    }
                    if (inkItem.isDateOpen()) {
                        mInkItem.setDate(AppDmUtil.currentDateToDocumentDateString());
                    }
                    if (inkItem.isVersionOpen()) {
                        if (mUiExtensionsManager.getAPPInfoProvider() != null) {
                            mInkItem.setVersion(mUiExtensionsManager.getAPPInfoProvider().getAppVersion());
                        }
                    }
                    if (inkItem.isDistinguishedNameOpen) {
                        mInkItem.setDistinguishedName(inkItem.distinguishedName);
                    }
                    saveSignCallback(true);
                }
                FragmentActivity act = ((FragmentActivity) context);
                SignatureFragment fragment = (SignatureFragment) act.getSupportFragmentManager().findFragmentByTag("InkSignFragment");
                fragment.dismiss();
                mSignAppearanceDialog.dismiss();
            }

            @Override
            public void onSaveSign(SignatureInkItem inkItem) {
                mInkItem = inkItem;
                if (inkItem.isDateOpen()) {
                    mInkItem.setDate(AppDmUtil.currentDateToDocumentDateString());
                }

                saveSignCallback(true);
                Context context = mUiExtensionsManager.getAttachedActivity();
                if (context == null) {
                    return;
                }
                FragmentActivity act = ((FragmentActivity) context);
                SignatureFragment fragment = (SignatureFragment) act.getSupportFragmentManager().findFragmentByTag("InkSignFragment");
                fragment.dismiss();
                mSignAppearanceDialog.dismiss();
            }
        });
        mSignAppearanceDialog.loadData();
        mSignAppearanceDialog.showDialog();
        new Handler().postDelayed(new Runnable() {
            @Override
            public void run() {
                mSignAppearanceDialog.titleTextAdaptive();
            }
        }, 300);

    }


}
