/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.modules.signature;

import android.content.Context;
import android.graphics.Bitmap;
import android.graphics.Rect;
import android.text.TextUtils;
import android.view.View;
import android.view.ViewGroup;
import android.widget.ImageView;
import android.widget.LinearLayout;

import com.foxit.sdk.PDFViewCtrl;
import com.foxit.uiextensions.IThemeEventListener;
import com.foxit.uiextensions.R;
import com.foxit.uiextensions.UIExtensionsManager;
import com.foxit.uiextensions.config.Config;
import com.foxit.uiextensions.controls.dialog.AppDialogManager;
import com.foxit.uiextensions.controls.dialog.UIMatchDialog;
import com.foxit.uiextensions.controls.toolbar.BaseBar;
import com.foxit.uiextensions.theme.ThemeConfig;
import com.foxit.uiextensions.utils.AppDisplay;
import com.foxit.uiextensions.utils.AppResource;
import com.foxit.uiextensions.utils.AppUtil;
import com.foxit.uiextensions.utils.SystemUiHelper;
import com.foxit.uiextensions.utils.thread.AppThreadManager;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;

import androidx.fragment.app.FragmentActivity;
import androidx.recyclerview.widget.DefaultItemAnimator;
import androidx.recyclerview.widget.LinearLayoutManager;
import androidx.recyclerview.widget.RecyclerView;

public class SignatureListDialog extends UIMatchDialog {
    private Context mContext;
    private ViewGroup mParent;
    private PDFViewCtrl mPdfViewCtrl;
    private UIExtensionsManager mUIExtensionsManager;
    private ViewGroup mContentView;
    private RecyclerView mSignListView;
    private SignatureListAdapter mSignListAdapter;
    private LinearLayout mSignListEmptyLl;
    private ImageView mSignListEmptyIv;
    private LinearLayoutManager mLayoutManager;

    private SignatureFragment.SignatureInkCallback mInkCallback;
    private ISignListPickerDismissCallback mSignListPickerDismissCallback;

    private ArrayList<SignatureInkItem> mAllInkItems = new ArrayList<>();
    private ArrayList<SignatureInkItem> mDsgInkItems = new ArrayList<>();
    private ArrayList<SignatureInkItem> mHandwritingInkItems = new ArrayList<>();
    private Config mModuleConfig;
    private boolean mIsFromSignatureField = false;
    private boolean mIsShowMenuWhenDismiss = true;

    public SignatureListDialog(Context context,
                               ViewGroup parent,
                               PDFViewCtrl pdfViewCtrl,
                               SignatureFragment.SignatureInkCallback inkCallback,
                               boolean isFromSignatureField) {
        super(context);
        mContext = context.getApplicationContext();
        mParent = parent;
        mPdfViewCtrl = pdfViewCtrl;
        mUIExtensionsManager = (UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager();
        mInkCallback = inkCallback;
        mModuleConfig = ((UIExtensionsManager) pdfViewCtrl.getUIExtensionsManager()).getConfig();
        mIsFromSignatureField = isFromSignatureField;
        if (!AppDisplay.isPad())
            SystemUiHelper.getInstance().showNavigationBar(mUIExtensionsManager.getAttachedActivity());

        initStyle();
        initView();

        setOnDLDismissListener(new DismissListener() {
            @Override
            public void onDismiss() {
                mSignListPickerDismissCallback.onDismiss(mIsShowMenuWhenDismiss);
                mUIExtensionsManager.unregisterThemeEventListener(mThemeColorChangedListener);
                if (getSignatureCount() == 0) {
                    mUIExtensionsManager.setCurrentToolHandler(null);
                }
            }
        });
    }

    private void initStyle() {
        setBackButtonStyle(TEXT_BACK);
        setBackButtonText(AppResource.getString(mContext, R.string.fx_string_close));
        setTitle(AppResource.getString(mContext, R.string.rv_sign_model));
        setTitlePosition(BaseBar.TB_Position.Position_CENTER);
        setRightButtonVisible(View.VISIBLE);
        setRightButtonText(AppResource.getString(mContext, R.string.fx_string_create));
        setStyle(DLG_TITLE_STYLE_BG_DEFAULT);

        setListener(new DialogListener() {
            @Override
            public void onResult(long btType) {
            }

            @Override
            public void onBackClick() {
                mIsShowMenuWhenDismiss = true;
            }

            @Override
            public void onTitleRightButtonClick() {
                mIsShowMenuWhenDismiss = false;
                dismiss();
                Context context = mUIExtensionsManager.getAttachedActivity();
                if (context == null) return;
                FragmentActivity act = ((FragmentActivity) context);
                SignatureFragment fragment = (SignatureFragment) act.getSupportFragmentManager().findFragmentByTag(SignatureFragment.TAG);
                if (fragment == null) {
                    fragment = new SignatureFragment();
                    fragment.init(mContext, mParent, mPdfViewCtrl, mIsFromSignatureField);
                }
                fragment.setInkCallback(mInkCallback);
                AppDialogManager.getInstance().showAllowManager(fragment, act.getSupportFragmentManager(), SignatureFragment.TAG, null);
            }
        });
    }

    private void initView() {
        mContentView = (ViewGroup) View.inflate(mContext, R.layout.sign_list_layout, null);

        mSignListEmptyLl = mContentView.findViewById(R.id.sign_list_empty_ll);
        mSignListEmptyIv = mContentView.findViewById(R.id.sign_list_empty_iv);
        mSignListEmptyIv.setColorFilter(ThemeConfig.getInstance(mContext).getPrimaryColor());

        mSignListView = mContentView.findViewById(R.id.sign_list_listview);
        mSignListAdapter = new SignatureListAdapter(mContext, mPdfViewCtrl);
        mSignListView.setAdapter(mSignListAdapter);
        mLayoutManager = new LinearLayoutManager(mContext, LinearLayoutManager.VERTICAL, false);
        mSignListView.setLayoutManager(mLayoutManager);
        mSignListView.setItemAnimator(new DefaultItemAnimator());
        mSignListAdapter.setOnItemClickCallback(new SignatureListAdapter.OnItemClickCallback() {
            @Override
            public void onItemClick(int type, int position, SignatureInkItem inkItem) {
                if (inkItem == null) return;
                if (type == SignatureListAdapter.CLICK_TYPE_CONTENT) {
                    applySign(inkItem);
                } else if (type == SignatureListAdapter.CLICK_TYPE_EDIT) {
                    AppThreadManager.getInstance().getMainThreadHandler().post(new Runnable() {
                        @Override
                        public void run() {
                            mIsShowMenuWhenDismiss = false;
                            dismiss();
                        }
                    });

                    initItem(inkItem);
                    Context context = mUIExtensionsManager.getAttachedActivity();
                    if (context == null) return;
                    FragmentActivity act = ((FragmentActivity) context);
                    SignatureFragment fragment = (SignatureFragment) act.getSupportFragmentManager().findFragmentByTag(SignatureFragment.TAG);
                    if (fragment == null) {
                        fragment = new SignatureFragment();
                        fragment.init(mContext, mParent, mPdfViewCtrl, mIsFromSignatureField);
                    }
                    fragment.setInkCallback(mInkCallback, inkItem);
                    AppDialogManager.getInstance().showAllowManager(fragment, act.getSupportFragmentManager(), SignatureFragment.TAG, null);
                } else if (type == SignatureListAdapter.CLICK_TYPE_DELETE) {
                    SignatureDataUtil.deleteByKey(mContext, SignatureConstants.getModelTableName(), inkItem.key);
                    mAllInkItems.remove(inkItem);
                    if (inkItem.getFlag() == SignatureInkItem.FLAG_NORMAL) {
                        mHandwritingInkItems.remove(inkItem);
                    } else {
                        mDsgInkItems.remove(inkItem);
                    }

                    if (getSignatureCount() == 0) {
                        mSignListView.setVisibility(View.GONE);
                        mSignListEmptyLl.setVisibility(View.VISIBLE);
                    } else {
                        List<String> recent = SignatureDataUtil.getRecentKeys(mContext);
                        String recentKey = recent == null ? null : recent.get(0);

                        for (int i = 0; i < mAllInkItems.size(); i++) {
                            SignatureInkItem item = mAllInkItems.get(i);
                            if (item.getFlag() != SignatureInkItem.FLAG_LABEL)
                                item.selected = item.key.equals(recentKey);
                        }
                    }
                    mSignListAdapter.notifyUpdateData();
                }
            }
        });

        setContentView(mContentView);
    }

    public void loadData() {
        mAllInkItems.clear();
        mHandwritingInkItems.clear();
        mDsgInkItems.clear();
        mIsShowMenuWhenDismiss = true;
        mUIExtensionsManager.registerThemeEventListener(mThemeColorChangedListener);

        List<String> recentList = SignatureDataUtil.getRecentKeys(mContext);
        String recentKey = null;
        if (recentList != null && recentList.size() > 0) {
            if (mIsFromSignatureField) {
                for (String key : recentList) {
                    HashMap<String, Object> map = SignatureDataUtil.getBitmapByKey(mContext, key);
                    if (map != null && map.get("dsgPath") != null && !AppUtil.isEmpty((String) map.get("dsgPath"))) {
                        recentKey = key;
                        break;
                    }
                }
            } else {
                recentKey = recentList.get(0);
            }
        }

        List<String> keys = SignatureDataUtil.getModelKeys(mContext);
        if (keys == null) return;
        for (String key : keys) {
            SignatureInkItem item = new SignatureInkItem();
            if (!AppUtil.isEmpty(recentKey) && key.equals(recentKey)) {
                item.selected = true;
            }
            item.key = key;
            HashMap<String, Object> map = SignatureDataUtil.getBitmapByKey(mContext, key);
            if (map != null) {
                Object dsgPathObj = map.get("dsgPath");
                Object rect = map.get("rect");
                item.setRect((Rect) rect);
                item.color = (int) map.get("color");
                item.diameter = (float) map.get("diameter");
                Object clearBg = map.get("clearBg");
                Object pic = map.get("pic");
                item.setClearBg((Integer) clearBg);
                item.setPic((Integer) pic);
                if (dsgPathObj != null && !AppUtil.isEmpty((String) dsgPathObj) && map.get("rect") != null) {
                    item.dsgPath = (String) dsgPathObj;
                    item.setFlag(SignatureInkItem.FLAG_DSG);
                    Object dsgTitle = map.get("title");
                    if (!TextUtils.isEmpty((CharSequence) dsgTitle)) {
                        Object dsgLocation = map.get("location");
                        Object dsgOpenLocation = map.get("openLocation");
                        Object dsgReason = map.get("reason");
                        Object dsgOpenReason = map.get("openReason");
                        Object dsgDn = map.get("dn");
                        Object name= map.get("name");
                        Object date = map.get("date");
                        Object version = map.get("version");
                        item.setTitle((String) dsgTitle);
                        item.setLocation((String) dsgLocation);
                        item.setLocationOpen((Integer) dsgOpenLocation);
                        item.setReason((Integer) dsgReason);
                        item.setReasonOpen((Integer) dsgOpenReason);
                        SignatureInkItem titleInkItem = SignatureDataUtil.getTitleByKey(mContext, item.getTitle());
                        item.setNameOpen(titleInkItem.isNameOpen() ? 1 : 0);
                        if(item.isNameOpen()){
                            item.setName((String) name);
                        }
                        item.setDistinguishedNameOpen(titleInkItem.isDistinguishedNameOpen() ? 1 : 0);
                        if (item.isDistinguishedNameOpen()&&dsgDn!=null) {
                            item.setDistinguishedName((String) dsgDn);
                        }
                        item.setDateOpen(titleInkItem.isDateOpen() ? 1 : 0);
                        if(item.isDateOpen()){
                            item.setDate((String) date);
                        }

                        item.setVersionOpen(titleInkItem.isVersionOpen() ? 1 : 0);
                        if(item.isVersionOpen()){
                            item.setVersion((String) version);
                        }
                        item.setLogo(titleInkItem.isLogo() ? 1 : 0);
                        item.setLabels(titleInkItem.isLabels() ? 1 : 0);
                    }
                    mDsgInkItems.add(item);
                } else {
                    item.dsgPath = null;
                    item.setFlag(SignatureInkItem.FLAG_NORMAL);
                    mHandwritingInkItems.add(item);
                }
            }
        }

        if (mIsFromSignatureField) {
            SignatureInkItem dsgLabelItem = new SignatureInkItem();
            dsgLabelItem.setFlag(SignatureInkItem.FLAG_LABEL);
            dsgLabelItem.setTag(AppResource.getString(getContext(), R.string.sign_list_signer_dsg_group_title).toUpperCase());
            mAllInkItems.add(dsgLabelItem);
            mAllInkItems.addAll(mDsgInkItems);
        } else {
            SignatureInkItem normalLableItem = new SignatureInkItem();
            normalLableItem.setFlag(SignatureInkItem.FLAG_LABEL);
            normalLableItem.setTag(AppResource.getString(getContext(), R.string.sign_list_signer_handwriting_group_title).toUpperCase());
            mAllInkItems.add(normalLableItem);
            mAllInkItems.addAll(mHandwritingInkItems);

            if (mModuleConfig.modules.isLoadForm && mUIExtensionsManager.getDocumentManager().withAddPermission()) {
                SignatureInkItem dsgLabelItem = new SignatureInkItem();
                dsgLabelItem.setFlag(SignatureInkItem.FLAG_LABEL);
                dsgLabelItem.setTag(AppResource.getString(getContext(), R.string.sign_list_signer_dsg_group_title).toUpperCase());
                mAllInkItems.add(dsgLabelItem);
                mAllInkItems.addAll(mDsgInkItems);
            }
        }

        mSignListAdapter.setSignList(mAllInkItems);
        mSignListAdapter.notifyUpdateData();
        if (getSignatureCount() == 0) {
            mSignListView.setVisibility(View.GONE);
            mSignListEmptyLl.setVisibility(View.VISIBLE);
        } else {
            mSignListView.setVisibility(View.VISIBLE);
            mSignListEmptyLl.setVisibility(View.GONE);
        }
    }


    private void initItem(SignatureInkItem item) {
        HashMap<String, Object> map = SignatureDataUtil.getBitmapByKey(mContext, item.key);
        item.bitmap = (Bitmap) map.get("bitmap");
        item.rect = (Rect) map.get("rect");
        item.color = (int) map.get("color");
        item.diameter = (float) map.get("diameter");
        Object clearBg = map.get("clearBg");
        Object pic = map.get("pic");
        item.setClearBg((Integer) clearBg);
        item.setPic((Integer) pic);
        Object dsgPathObj = map.get("dsgPath");
        if (dsgPathObj != null && !AppUtil.isEmpty((String) dsgPathObj)) {
            item.dsgPath = (String) dsgPathObj;
            Object dsgTitle = map.get("title");
            if (!TextUtils.isEmpty((CharSequence) dsgTitle)) {
                Object dsgLocation = map.get("location");
                Object dsgOpenLocation = map.get("openLocation");
                Object dsgReason = map.get("reason");
                Object dsgOpenReason = map.get("openReason");
                Object dsgDn = map.get("dn");
                Object name= map.get("name");
                Object date = map.get("date");
                Object version = map.get("version");
                item.setTitle((String) dsgTitle);
                item.setLocation((String) dsgLocation);
                item.setLocationOpen((Integer) dsgOpenLocation);
                item.setReason((Integer) dsgReason);
                item.setReasonOpen((Integer) dsgOpenReason);
                SignatureInkItem titleInkItem = SignatureDataUtil.getTitleByKey(mContext, item.getTitle());
                item.setNameOpen(titleInkItem.isNameOpen() ? 1 : 0);
                item.setDistinguishedNameOpen(titleInkItem.isDistinguishedNameOpen() ? 1 : 0);
                if(item.isNameOpen()){
                    item.setName((String) name);
                }
                item.setDistinguishedNameOpen(titleInkItem.isDistinguishedNameOpen() ? 1 : 0);
                if (item.isDistinguishedNameOpen()&&dsgDn!=null) {
                    item.setDistinguishedName((String) dsgDn);
                }
                item.setDateOpen(titleInkItem.isDateOpen() ? 1 : 0);
                if(item.isDateOpen()){
                    item.setDate((String) date);
                }

                item.setVersionOpen(titleInkItem.isVersionOpen() ? 1 : 0);
                if(item.isVersionOpen()){
                    item.setVersion((String) version);
                }
                item.setLogo(titleInkItem.isLogo() ? 1 : 0);
                item.setLabels(titleInkItem.isLabels() ? 1 : 0);
            }
        } else {
            item.dsgPath = null;
        }
    }

    private int getSignatureCount() {
        if (mIsFromSignatureField)
            return mDsgInkItems.size();
        else
            return mHandwritingInkItems.size() + mDsgInkItems.size();
    }

    private void applySign(SignatureInkItem item) {
        initItem(item);
        mInkCallback.onSuccess(false, item);
        SignatureDataUtil.insertRecent(mContext, item.key);
        mIsShowMenuWhenDismiss = true;
        dismiss();
    }

    @Override
    public void dismiss() {
        super.dismiss();
        mSignListAdapter.clear();
    }

    /**
     * must use
     */
    public void setDismissCallback(ISignListPickerDismissCallback signListPickerDismissCallback) {
        mSignListPickerDismissCallback = signListPickerDismissCallback;
    }

    public interface ISignListPickerDismissCallback {
        void onDismiss(boolean isShowAnnotMenu);
    }

    public void onLayoutChanged() {
        resetWH();
        mSignListAdapter.onLayoutChanged(mLayoutManager);
    }

    private IThemeEventListener mThemeColorChangedListener = new IThemeEventListener() {
        @Override
        public void onThemeColorChanged(String type, int color) {
            mIsShowMenuWhenDismiss = true;
            dismiss();
        }
    };

}
