/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.modules.signature.appearance;

import android.content.Context;
import android.graphics.Rect;
import android.text.TextUtils;
import android.view.View;
import android.view.ViewGroup;

import androidx.fragment.app.FragmentActivity;
import androidx.recyclerview.widget.LinearLayoutManager;
import androidx.recyclerview.widget.RecyclerView;

import com.foxit.sdk.PDFViewCtrl;
import com.foxit.uiextensions.IThemeEventListener;
import com.foxit.uiextensions.R;
import com.foxit.uiextensions.UIExtensionsManager;
import com.foxit.uiextensions.controls.dialog.UIDialog;
import com.foxit.uiextensions.controls.dialog.UIMatchDialog;
import com.foxit.uiextensions.controls.dialog.UITextEditDialog;
import com.foxit.uiextensions.controls.dialog.sheetmenu.ISheetMenu;
import com.foxit.uiextensions.controls.dialog.sheetmenu.UIBottomSheetMenu;
import com.foxit.uiextensions.controls.dialog.sheetmenu.UISheetMenu;
import com.foxit.uiextensions.controls.toolbar.BaseBar;
import com.foxit.uiextensions.modules.signature.SignatureDataUtil;
import com.foxit.uiextensions.theme.ThemeUtil;
import com.foxit.uiextensions.utils.AppDevice;
import com.foxit.uiextensions.utils.AppDisplay;
import com.foxit.uiextensions.utils.AppResource;
import com.foxit.uiextensions.utils.AppUtil;
import com.foxit.uiextensions.utils.SystemUiHelper;

import java.util.ArrayList;
import java.util.List;

public class SignatureAppearanceTitleDialog extends UIMatchDialog {
    private Context mContext;
    private ViewGroup mParent;
    private PDFViewCtrl mPdfViewCtrl;
    private UIExtensionsManager mUIExtensionsManager;
    private ViewGroup mContentView;
    private RecyclerView mSignTitleRecyclerView;
    private SignatureTitleAdapter mSignatureTitleAdapter;
    private LinearLayoutManager mLayoutManager;
    private ISignSelectorTitleCallback mSignSelectorTitleCallback;
    private String mSelectName;
    private String mTitle = "";
    private UITextEditDialog mDialog;
    private List<SignatureTitleBean> mTitleList = new ArrayList<>();
    private String mEditedTitle;
    public SignatureAppearanceTitleDialog(Context context,
                                          ViewGroup parent,
                                          PDFViewCtrl pdfViewCtrl,
                                          String title,
                                          String editedTitle) {
        super(context);
        mContext = context.getApplicationContext();
        mParent = parent;
        mPdfViewCtrl = pdfViewCtrl;
        mUIExtensionsManager = (UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager();
        mTitle = title;
        mEditedTitle = editedTitle;
        initStyle();
        initView();


        setOnDLDismissListener(new DismissListener() {
            @Override
            public void onDismiss() {
                mUIExtensionsManager.unregisterThemeEventListener(mThemeColorChangedListener);
            }
        });
    }


    @Override
    public void onBackPressed() {
        if(!TextUtils.isEmpty(mSelectName)) {
            mSignSelectorTitleCallback.onSelectorTitle(mSelectName);
        }
        dismiss();
    }

    private void initStyle() {
        setBackButtonTintList(ThemeUtil.getItemIconColor(mContext));
        setTitle(AppResource.getString(mContext, R.string.appearance_type));
        setTitlePosition(BaseBar.TB_Position.Position_CENTER);
        setStyle(DLG_TITLE_STYLE_BG_DEFAULT);
        setListener(new DialogListener() {
            @Override
            public void onResult(long btType) {
            }

            @Override
            public void onBackClick() {
                if(!TextUtils.isEmpty(mSelectName)) {
                    mSignSelectorTitleCallback.onSelectorTitle(mSelectName);
                }
                dismiss();
            }

            @Override
            public void onTitleRightButtonClick() {

            }
        });
    }

    private void initView() {
        mContentView = (ViewGroup) View.inflate(mContext, R.layout.sign_appearance_title_layout, null);
        mSignTitleRecyclerView = mContentView.findViewById(R.id.sign_appearance_title_rv);
        mLayoutManager = new LinearLayoutManager(mContext, LinearLayoutManager.VERTICAL, false);
        mSignTitleRecyclerView.setLayoutManager(mLayoutManager);
        mSignatureTitleAdapter = new SignatureTitleAdapter(mContext, mPdfViewCtrl);
        mSignTitleRecyclerView.setAdapter(mSignatureTitleAdapter);
        mSignatureTitleAdapter.setOnItemClickCallback(onItemClickCallback);
        setContentView(mContentView);

    }

    public void loadData() {
        mUIExtensionsManager.registerThemeEventListener(mThemeColorChangedListener);
        List<String> list = SignatureDataUtil.getTitleKeys(mContext);
        for (int i = 0; i < list.size(); i++) {
            if(mTitle.equals(list.get(i))){
                mTitleList.add(new SignatureTitleBean(list.get(i), true));
            }else{
                mTitleList.add(new SignatureTitleBean(list.get(i), false));
            }
        }
        mSignatureTitleAdapter.setTitleList(mTitleList);
    }

    private IThemeEventListener mThemeColorChangedListener = new IThemeEventListener() {
        @Override
        public void onThemeColorChanged(String type, int color) {
            dismiss();
        }
    };

    public void setSelectorTitleCallback(ISignSelectorTitleCallback signSelectorTitleCallback) {
        mSignSelectorTitleCallback = signSelectorTitleCallback;
    }

    public interface ISignSelectorTitleCallback {
        void onSelectorTitle(String title);
    }

    private SignatureTitleAdapter.OnItemClickCallback onItemClickCallback = new SignatureTitleAdapter.OnItemClickCallback() {
        @Override
        public void onItemClick(int position, SignatureTitleBean item) {
            mSelectName = item.getName();
            SignatureDataUtil.insertRecentTitle(mContext, mSelectName);
        }

        @Override
        public void onItemMoreClick(int position, SignatureTitleBean item, View view) {
            showSheetMenu(position,item,view);
        }
    };

    ISheetMenu mSheetMenu;
    private void showSheetMenu(final int position, final SignatureTitleBean item, View view) {
        mSheetMenu = UISheetMenu.newInstance((FragmentActivity) mUIExtensionsManager.getAttachedActivity());
        if (AppDisplay.isPad())
            mSheetMenu.setWidth(AppResource.getDimensionPixelSize(getContext(), R.dimen.ux_pad_more_menu_width));
        List<Integer> items = new ArrayList<>();
        items.add(ISheetMenu.DELETE_MENU);
        mSheetMenu.setSheetItems(items);
        mSheetMenu.setAutoResetSystemUiOnShow(false);
        mSheetMenu.setSheetItemClickListener(new UIBottomSheetMenu.OnSheetItemClickListener() {
            @Override
            public void onItemClick(int type) {
                dismissSheetMenu();
                if (AppUtil.isFastDoubleClick()) {
                    return;
                }
                if (!TextUtils.isEmpty(mEditedTitle) && mEditedTitle.equals(item.getName())) {
                    showDialogDeleteWarning();
                }else {
                    showDialogDelete(position, item);
                }
            }
        });
        mSheetMenu.setOnSheetDismissListener(new ISheetMenu.OnSheetDismissListener() {
            @Override
            public void onDismiss(ISheetMenu sheetMenu) {
            }
        });
        showSheetMenu(view);
    }
    private void showSheetMenu(View view) {
        Rect rect = new Rect();
        view.getGlobalVisibleRect(rect);
        int[] location = new int[2];
        view.getLocationOnScreen(location);
        int screen_x = location[0];
        int screen_y = location[1];
        if (AppDevice.isChromeOs(mUIExtensionsManager.getAttachedActivity())) {
            mUIExtensionsManager.getRootView().getLocationOnScreen(location);
            screen_x -= location[0];
            screen_y -= location[1];
            rect.set(screen_x, screen_y, screen_x + rect.width(), screen_y + rect.height());
        } else {
            view.getLocationInWindow(location);
            int window_x = location[0];
            int window_y = location[1];
            rect.offset(screen_x - window_x, screen_y - window_y);
        }
        if (!SystemUiHelper.getInstance().isFullScreenMode(mUIExtensionsManager.getAttachedActivity())
                && SystemUiHelper.getInstance().isFullScreen())
            rect.offset(0, -SystemUiHelper.getInstance().getStatusBarHeight(mUIExtensionsManager.getAttachedActivity()));
        mSheetMenu.show(mUIExtensionsManager.getRootView(), rect);
    }

    private void dismissSheetMenu() {
        if (mSheetMenu != null && mSheetMenu.isShowing()) {
            mSheetMenu.dismiss();
        }
    }

    private  void showDialogDelete(final int position, final SignatureTitleBean item){
        if (mDialog == null) {
            final Context context = mUIExtensionsManager.getAttachedActivity();
            String title = AppResource.getString(mContext, R.string.menu_more_confirm);
            String prompt = AppResource.getString(mContext, R.string.appearance_delete_style);
            mDialog = new UITextEditDialog(context, UIDialog.NO_INPUT);
            mDialog.setTitle(title);
            mDialog.getPromptTextView().setText(prompt);
        }
        mDialog.getOKButton().setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                if (AppUtil.isFastDoubleClick()) {
                    return;
                }
                SignatureDataUtil.deleteSigByTitle(mContext,item.getName());
                SignatureDataUtil.deleteTitle(mContext,item.getName());

                mTitleList.remove(position);
                if(item.isSelect()){
                    int selectPosition = position > 0 ? position -1:0;
                    mSignatureTitleAdapter.mOldSelectorPosition = selectPosition;
                    mTitleList.get(selectPosition).setSelect(true);
                    mSelectName = mTitleList.get(selectPosition).getName();
                    SignatureDataUtil.insertRecentTitle(mContext, mSelectName);
                }
                mSignatureTitleAdapter.notifyUpdateData();
                mDialog.dismiss();
            }
        });
        mDialog.show();
    }

    private  void showDialogDeleteWarning(){
        final UITextEditDialog dialog = new UITextEditDialog(mUIExtensionsManager.getAttachedActivity(), UIDialog.NO_INPUT);
        dialog.setTitle(AppResource.getString(mContext, R.string.fx_string_warning_title));
        dialog.getPromptTextView().setText(AppResource.getString(mContext, R.string.appearance_cannot_delete_style));
        dialog.getCancelButton().setVisibility(View.GONE);
        dialog.getOKButton().setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                dialog.dismiss();
            }
        });
        dialog.show();
    }


}
