/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.modules.thumbnail;


import android.app.Service;
import android.content.Context;
import android.content.res.Configuration;
import android.graphics.Point;
import android.graphics.Rect;
import android.os.Bundle;
import android.os.Vibrator;
import android.util.SparseArray;
import android.view.ContextThemeWrapper;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.view.animation.AnimationSet;
import android.view.animation.TranslateAnimation;
import android.widget.CheckBox;
import android.widget.ImageView;
import android.widget.LinearLayout;
import android.widget.TextView;

import com.foxit.sdk.PDFException;
import com.foxit.sdk.PDFViewCtrl;
import com.foxit.sdk.Task;
import com.foxit.sdk.pdf.PDFDoc;
import com.foxit.sdk.pdf.PDFPage;
import com.foxit.sdk.pdf.ReadingBookmark;
import com.foxit.uiextensions.Module;
import com.foxit.uiextensions.R;
import com.foxit.uiextensions.UIExtensionsManager;
import com.foxit.uiextensions.modules.pagenavigation.PageNavigationModule;
import com.foxit.uiextensions.theme.ThemeConfig;
import com.foxit.uiextensions.utils.AppResource;
import com.foxit.uiextensions.utils.AppUtil;
import com.foxit.uiextensions.utils.Event;
import com.foxit.uiextensions.utils.thread.AppThreadManager;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.fragment.app.Fragment;
import androidx.recyclerview.widget.GridLayoutManager;
import androidx.recyclerview.widget.ItemTouchHelper;
import androidx.recyclerview.widget.RecyclerView;

public class ThumbnailFragment extends Fragment {
    private List<ThumbnailItem> mItems;

    private Context mContext;
    private PDFViewCtrl mPDFView;
    private UIExtensionsManager mUIExtensionsManager;
    private LinearLayout mNoInfoContainer;
    private TextView mLoadingTextView;

    private ThumbnailAdapter mAdapter;
    private RecyclerView mThumbnailGridView;
    private GridLayoutManager mGridLayoutManager;
    private IThumbnailSupport mSupport;
    private ThumbnailItem mDraggingItem;

    private int mSpanCount;
    private int mVerSpacing;
    private int mHorSpacing;
    private int mTabPosition;
    private int mCurPagePosition;
    private boolean mIsPageCopied;
    private boolean mbCanPaste = false;
    private boolean mIsSingleChoice = false;
    private int mSelectedPagePosition = -1;
    private ImageView mNoInfoIv;
    private TextView mNoInfoTv;
    private boolean isMoveTo = false;
    private boolean mIsLoadingData = false;

    public static ThumbnailFragment newInstance(int tabPosition, int curPagePosition,
                                                IThumbnailSupport support, PDFViewCtrl pdfViewCtrl, List<ThumbnailItem> items) {
        ThumbnailFragment fragment = new ThumbnailFragment();
        fragment.init(tabPosition, curPagePosition, support, pdfViewCtrl, items);
        return fragment;
    }

    private void init(int tabPosition, int curPagePosition, IThumbnailSupport support, PDFViewCtrl pdfViewCtrl, List<ThumbnailItem> items) {
        mTabPosition = tabPosition;
        mCurPagePosition = curPagePosition;
        mItems = items;
        mPDFView = pdfViewCtrl;
        mSupport = support;
        mUIExtensionsManager = (UIExtensionsManager) pdfViewCtrl.getUIExtensionsManager();
    }

    @Override
    public void onCreate(@Nullable Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        mContext = getActivity().getApplicationContext();
        mHorSpacing = AppResource.getDimensionPixelSize(mContext, R.dimen.ux_margin_16dp);
        mAdapter = new ThumbnailAdapter(mTabPosition, mSupport, mItems);
        mAdapter.mCurPagePosition = mCurPagePosition;
        computeSize();

    }


    @Nullable
    @Override
    public View onCreateView(@NonNull LayoutInflater inflater, @Nullable ViewGroup container, @Nullable Bundle savedInstanceState) {
        final Context contextThemeWrapper = new ContextThemeWrapper(getActivity(), R.style.ThumbnailDialogTheme);
        LayoutInflater localInflater = inflater.cloneInContext(contextThemeWrapper);
        View view = localInflater.inflate(R.layout.thumbnail_content_frag_layout, container, false);

        initNoInfoView(view);
        initThumbnailList(view);
        moveToPosition(mPDFView.getCurrentPage());
        return view;
    }

    public void setMoveTo() {
        if (!isMoveTo) {
            for (int i = 0; i < mItems.size(); i++) {
                if (mItems.get(i).getPageIndex() == mPDFView.getCurrentPage()) {
                    final int position = i;
                    mPDFView.postDelayed(new Runnable() {
                        @Override
                        public void run() {
                            moveToPosition(position);
                        }
                    }, 200);
                    isMoveTo = true;
                    break;
                }
            }
        }
    }

    @Override
    public void onConfigurationChanged(@NonNull Configuration newConfig) {
        super.onConfigurationChanged(newConfig);
        mNoInfoIv.setColorFilter(ThemeConfig.getInstance(mContext).getPrimaryColor());
        mNoInfoTv.setTextColor(mContext.getResources().getColor(R.color.t2));
        if (mAdapter != null) {
            mAdapter.notifyDataSetChanged();
        }
    }

    private void initNoInfoView(View contentView) {
        mLoadingTextView = contentView.findViewById(R.id.thumbnail_loading_text);
        mNoInfoContainer = contentView.findViewById(R.id.thumbnail_no_content_ll);
        mNoInfoIv = contentView.findViewById(R.id.thumbnail_no_content_iv);
        mNoInfoTv = contentView.findViewById(R.id.thumbnail_no_content_tv);
        mNoInfoIv.setColorFilter(ThemeConfig.getInstance(mContext).getPrimaryColor());

        if (ThumbnailSupport.ANNOTTATED_TAB_POSITION == mTabPosition) {
            mNoInfoIv.setImageResource(R.drawable.panel_no_annotation);
            mNoInfoTv.setText(AppResource.getString(mContext, R.string.thumbnail_no_annotated_pages));
        } else {
            mNoInfoIv.setImageResource(R.drawable.panel_no_bookmark);
            mNoInfoTv.setText(AppResource.getString(mContext, R.string.thumbnail_no_bookmarked_pages));
        }
    }

    private void initThumbnailList(View contentView) {
        mThumbnailGridView = contentView.findViewById(R.id.thumbnail_grid_view);
        mThumbnailGridView.setHasFixedSize(true);
        mThumbnailGridView.setAdapter(mAdapter);
        mGridLayoutManager = new GridLayoutManager(mContext, mSpanCount);
        mThumbnailGridView.setLayoutManager(mGridLayoutManager);
        SpacesItemDecoration spacesItemDecoration = new SpacesItemDecoration();
        mThumbnailGridView.addItemDecoration(spacesItemDecoration);
        mThumbnailGridView.addOnScrollListener(new RecyclerView.OnScrollListener() {
            @Override
            public void onScrolled(RecyclerView recyclerView, int dx, int dy) {
                super.onScrolled(recyclerView, dx, dy);
                for (int position = mGridLayoutManager.findFirstVisibleItemPosition(); position <= mGridLayoutManager.findLastVisibleItemPosition(); position++) {
                    ThumbnailAdapter.ThumbViewHolder viewHolder = (ThumbnailAdapter.ThumbViewHolder) mThumbnailGridView.findViewHolderForAdapterPosition(position);
                    if (viewHolder != null)
                        viewHolder.drawThumbnail(mAdapter.getThumbnailItem(position), position);
                }
            }
        });
        if (mSupport.isSelectMode()) {
            mCurPagePosition = mSelectedPagePosition >= 0 && mSelectedPagePosition < mPDFView.getPageCount()
                    ? mSelectedPagePosition : mPDFView.getCurrentPage();
            mAdapter.mCurPagePosition = mCurPagePosition;
            mThumbnailGridView.smoothScrollToPosition(mCurPagePosition);
            mAdapter.updateSelectListInfo(mCurPagePosition, mAdapter.getThumbnailItem(mCurPagePosition), true);
            mSupport.setSelectViewMode(mAdapter.isSelectedAll());
            mThumbnailGridView.addOnItemTouchListener(new OnThumbnailItemTouchListener(mThumbnailGridView) {
                @Override
                public void onLongPress(RecyclerView.ViewHolder vh) {
                    ThumbnailFragment.this.onItemClick(vh);
                }

                @Override
                public boolean onItemClick(RecyclerView.ViewHolder vh) {
                    return ThumbnailFragment.this.onItemClick(vh);
                }

                @Override
                boolean onToRightFling(RecyclerView.ViewHolder vh) {
                    return false;
                }

                @Override
                boolean onToLeftFling(RecyclerView.ViewHolder vh) {
                    return false;
                }
            });
            updateUIState();
            return;
        }
        ThumbnailItemTouchCallback.OnDragListener dragListener = new ThumbnailItemTouchCallback.OnDragListener() {
            @Override
            public void onFinishDrag() {
                AppThreadManager.getInstance().getMainThreadHandler().post(new Runnable() {
                    @Override
                    public void run() {
                        if (mDraggingItem != null && mDraggingItem.itemType == ThumbnailItem.CONTENT_ITEM) {
                            mDraggingItem.setDragging(false);
                            int movedPosition = mAdapter.mMoveToPosition;
                            final SparseArray<ThumbnailItem> selectedItems = mAdapter.mSelectedArray;
                            int selectedCount = selectedItems.size();
                            if (movedPosition != -1) {
                                MovePageTask movePageTask = new MovePageTask(movedPosition, mDraggingItem, mAdapter, new Event.Callback() {
                                    @Override
                                    public void result(Event event, boolean success) {
                                        selectedItems.clear();
                                        List<ThumbnailItem> items = mAdapter.mThumbnailList;
                                        int thumbSize = items.size();
                                        if (mTabPosition == ThumbnailSupport.ALL_TAB_POSITION) {
                                            for (int i = 0; i < thumbSize; i++) {
                                                ThumbnailItem item = items.get(i);
                                                item.setPageIndex(i);
                                                if (item.isSelected())
                                                    selectedItems.put(i, item);
                                            }
                                        } else if (mTabPosition == ThumbnailSupport.ANNOTTATED_TAB_POSITION) {
                                            try {
                                                int pageCount = mPDFView.getPageCount();
                                                PDFDoc doc = mPDFView.getDoc();

                                                int itemIndex = 0;
                                                for (int i = 0; i < pageCount; i++) {
                                                    PDFPage page = doc.getPage(i);
                                                    int pageIndex = page.getIndex();
                                                    if (page.getAnnotCount() > 0) {
                                                        if (itemIndex < thumbSize) {
                                                            ThumbnailItem item = items.get(itemIndex);
                                                            item.setPageIndex(pageIndex);
                                                            if (item.isSelected())
                                                                selectedItems.put(itemIndex, item);
                                                            itemIndex++;
                                                        }
                                                    }
                                                }
                                            } catch (PDFException ignored) {
                                            }
                                        } else {
                                            try {
                                                PDFDoc doc = mPDFView.getDoc();
                                                int readingBookmarkCount = doc.getReadingBookmarkCount();
                                                List<Integer> bookmarkIndexs = new ArrayList<>();
                                                for (int i = 0; i < readingBookmarkCount; i++) {
                                                    ReadingBookmark bookmark = doc.getReadingBookmark(i);
                                                    if (bookmark != null && !bookmark.isEmpty()) {
                                                        int pageIndex = bookmark.getPageIndex();
                                                        bookmarkIndexs.add(pageIndex);
                                                    }
                                                }
                                                Collections.sort(bookmarkIndexs);
                                                int bookmarkSize = bookmarkIndexs.size();
                                                for (int i = 0; i < bookmarkSize; i++) {
                                                    if (i < thumbSize) {
                                                        ThumbnailItem item = items.get(i);
                                                        item.setPageIndex(bookmarkIndexs.get(i));
                                                        if (item.isSelected())
                                                            selectedItems.put(i, item);
                                                    }
                                                }
                                            } catch (PDFException ignored) {
                                            }
                                        }
                                        mAdapter.mMoveToPosition = -1;
                                        mAdapter.notifyDataSetChanged();
                                        mSupport.refreshOtherTab();
                                    }
                                });
                                mPDFView.addTask(movePageTask);
                            } else {
                                for (int i = 0; i < selectedCount; i++) {
                                    int position = selectedItems.keyAt(i);
                                    ThumbnailItem item = selectedItems.valueAt(i);
                                    if (mDraggingItem == item) continue;
                                    mAdapter.mThumbnailList.add(position, item);
                                }
                                mAdapter.notifyDataSetChanged();
                            }
                        } else {
                            AppThreadManager.getInstance().getMainThreadHandler().post(new Runnable() {
                                @Override
                                public void run() {
                                    mAdapter.notifyDataSetChanged();
                                }
                            });
                        }
                    }
                });
            }
        };

        ThumbnailItemTouchCallback touchCallback = new ThumbnailItemTouchCallback(mAdapter).setOnDragListener(dragListener);
        touchCallback.setLongPressDragEnabled(false);
        final ItemTouchHelper itemTouchHelper = new ItemTouchHelper(touchCallback);
        itemTouchHelper.attachToRecyclerView(mThumbnailGridView);

        mThumbnailGridView.addOnItemTouchListener(new OnThumbnailItemTouchListener(mThumbnailGridView) {
            @Override
            public void onLongPress(RecyclerView.ViewHolder vh) {
                if (mSupport.isEditMode()) {
                    ThumbnailAdapter.ThumbViewHolder viewHolder = (ThumbnailAdapter.ThumbViewHolder) vh;
                    int pressPosition = vh.getAdapterPosition();
                    ThumbnailItem thumbnailItem = mAdapter.getThumbnailItem(pressPosition);

                    if (mSupport.isPasteState()) {
                        if (thumbnailItem.itemType == ThumbnailItem.PASTE_ITEM) {
                            mDraggingItem = thumbnailItem;
                            itemTouchHelper.startDrag(vh);
                            Vibrator vib = (Vibrator) getActivity().getSystemService(Service.VIBRATOR_SERVICE);
                            vib.vibrate(70);
                        }
                    } else {
                        mDraggingItem = thumbnailItem;
                        if (!thumbnailItem.isSelected()) {
                            mIsPageCopied = false;
                            mAdapter.updateSelectListInfo(pressPosition, thumbnailItem, true);
                            mSupport.setSelectViewMode(mAdapter.isSelectedAll());
                        }
                        thumbnailItem.setDragging(true);

                        View itemView = viewHolder.itemView;
                        TextView selectedCount = itemView.findViewById(R.id.thumbnail_select_count);
                        selectedCount.setVisibility(View.VISIBLE);
                        selectedCount.setText(String.format("%s", mAdapter.getSelectedItemCount()));
                        CheckBox checkBox = itemView.findViewById(R.id.thumbnail_select_view);
                        checkBox.setChecked(true);
                        float x = itemView.getX();
                        float y = itemView.getY();
                        SparseArray<ThumbnailItem> selectedItems = mAdapter.mSelectedArray;
                        int selectedSize = selectedItems.size();

                        List<ThumbnailItem> items = mAdapter.mThumbnailList;
                        int firstVisiblePosition = mGridLayoutManager.findFirstVisibleItemPosition();
                        int lastVisiblePosition = mGridLayoutManager.findLastVisibleItemPosition();
                        for (int position = firstVisiblePosition; position <= lastVisiblePosition; position++) {
                            ThumbnailItem item = items.get(position);
                            if (item.isSelected() && item != mDraggingItem) {
                                removeAnimator(position, x, y);
                            }
                        }

                        for (int i = selectedSize - 1; i >= 0; i--) {
                            int itemPosition = selectedItems.keyAt(i);
                            if (itemPosition == pressPosition) continue;
                            items.remove(itemPosition);
                            mAdapter.notifyItemRemoved(itemPosition);
                        }
                        itemTouchHelper.startDrag(vh);
                        Vibrator vib = (Vibrator) getActivity().getSystemService(Service.VIBRATOR_SERVICE);
                        vib.vibrate(70);
                    }
                }
            }

            @Override
            public boolean onItemClick(RecyclerView.ViewHolder vh) {
                return ThumbnailFragment.this.onItemClick(vh);
            }

            @Override
            public boolean onToRightFling(RecyclerView.ViewHolder vh) {
                return false;
            }

            @Override
            public boolean onToLeftFling(RecyclerView.ViewHolder vh) {
                return false;
            }
        });
        updateUIState();
    }

    public boolean onItemClick(RecyclerView.ViewHolder vh) {
        ThumbnailAdapter.ThumbViewHolder viewHolder = (ThumbnailAdapter.ThumbViewHolder) vh;
        int position = vh.getAdapterPosition();
        ThumbnailItem thumbnailItem = mAdapter.getThumbnailItem(position);
        if (mSupport.isEditMode()) {
            if (!mSupport.isPasteState() && thumbnailItem.itemType == ThumbnailItem.CONTENT_ITEM) {
                mIsPageCopied = false;
                boolean isSelected = !thumbnailItem.isSelected();
                if (mIsSingleChoice && isSelected) {
                    for (int index : mAdapter.getSelectedItemIndexes()) {
                        mAdapter.updateSelectListInfo(index, mAdapter.getThumbnailItem(index), !isSelected);
                        View view = mGridLayoutManager.findViewByPosition(index);
                        if (view != null) {
                            ((ThumbnailAdapter.ThumbViewHolder) mThumbnailGridView.getChildViewHolder(view)).changeSelectView(!isSelected);
                        }
                    }
                }
                mAdapter.updateSelectListInfo(position, thumbnailItem, isSelected);
                mSupport.setSelectViewMode(mAdapter.isSelectedAll());
                viewHolder.changeSelectView(isSelected);
            }
        } else {
            mSupport.updateRecycleLayout();
            mPDFView.gotoPage(thumbnailItem.getPageIndex());
            PageNavigationModule module = (PageNavigationModule) mUIExtensionsManager.getModuleByName(Module.MODULE_NAME_PAGENAV);
            if (module != null)
                module.resetJumpView();

            mSupport.exitThumbnailDialog();
        }
        return true;
    }

    public ThumbnailAdapter.ThumbViewHolder getViewHolderByItem(ThumbnailItem item) {
        int position = mAdapter.mThumbnailList.indexOf(item);
        return (ThumbnailAdapter.ThumbViewHolder) mThumbnailGridView.findViewHolderForAdapterPosition(position);
    }

    public boolean isThumbnailItemVisible(ThumbnailItem item) {
        int position = mAdapter.mThumbnailList.indexOf(item);
        return position >= mGridLayoutManager.findFirstVisibleItemPosition() && position <= mGridLayoutManager.findLastVisibleItemPosition();
    }

    public void onLayoutChange(View v, int newWidth, int newHeight, int oldWidth, int oldHeight) {
        if (oldWidth != newWidth || oldHeight != newHeight) {
            computeSize();
            if (mGridLayoutManager != null) {
                mGridLayoutManager.setSpanCount(mSpanCount);
                mGridLayoutManager.requestLayout();
            }
            if (mAdapter != null) {
                mAdapter.notifyDataSetChanged();
            }
        }
    }

    int getTabPosition() {
        return mTabPosition;
    }

    public ThumbnailAdapter getAdapter() {
        return mAdapter;
    }

    void updateData(int curPagePosition, List<ThumbnailItem> annotList) {
        mAdapter.mCurPagePosition = curPagePosition;
        mAdapter.mThumbnailList = annotList;
        mAdapter.notifyDataSetChanged();

        updateUIState();
    }

    void setLoadingState(boolean loading) {
        mIsLoadingData = loading;
    }

    void setLoadingText(int loadingIndex, int total){
        if (mLoadingTextView != null)
            mLoadingTextView.setText(AppResource.getString(mContext, R.string.rv_panel_annot_loading_pagenum, loadingIndex, total));
    }

    void updateUIState() {
        if (mIsLoadingData) {
            showLoadingView();
        } else if (mItems.size() > 0) {
            hideNoInfoView();
        } else {
            showNoInfoView();
        }
    }

    private void showLoadingView() {
        mLoadingTextView.setVisibility(View.VISIBLE);
        mThumbnailGridView.setVisibility(View.GONE);
        mNoInfoContainer.setVisibility(View.GONE);
    }

    private void showNoInfoView() {
        mLoadingTextView.setVisibility(View.GONE);
        mThumbnailGridView.setVisibility(View.GONE);
        mNoInfoContainer.setVisibility(View.VISIBLE);
    }

    private void hideNoInfoView() {
        mThumbnailGridView.setVisibility(View.VISIBLE);
        mNoInfoContainer.setVisibility(View.GONE);
        mLoadingTextView.setVisibility(View.GONE);
    }

    private void computeSize() {
        View rootView = mUIExtensionsManager.getRootView();
        int width = rootView.getWidth();
        int height = rootView.getHeight();
        Point size = mSupport.getThumbnailBackgroundSize();
        int margin = AppResource.getDimensionPixelSize(mContext, R.dimen.ux_margin_8dp);
        mSpanCount = Math.max(1, (width - margin) / (margin + size.x + 2));
        int tasksMax = mSpanCount * (height / size.y + 2);
        int bitmapMax = Math.max(64, tasksMax);
        if (mAdapter != null)
            mAdapter.setCacheSize(tasksMax, bitmapMax);
        mVerSpacing = (width - size.x * mSpanCount) / (mSpanCount + 1);
    }

    private class SpacesItemDecoration extends RecyclerView.ItemDecoration {
        @Override
        public void getItemOffsets(@NonNull Rect outRect, @NonNull View view,
                                   @NonNull RecyclerView parent, @NonNull RecyclerView.State state) {
            if (mSpanCount > 0) {
                int position = parent.getChildAdapterPosition(view);
                int spanIndex = position % mSpanCount;

                outRect.left = mVerSpacing - spanIndex * mVerSpacing / mSpanCount;
                outRect.right = (spanIndex + 1) * mVerSpacing / mSpanCount;

                outRect.top = mHorSpacing;
                outRect.bottom = mHorSpacing;
            } else {
                outRect.setEmpty();
            }
        }
    }

    private void removeAnimator(final int position, float toX, float toY) {
        ThumbnailAdapter.ThumbViewHolder viewHolder = (ThumbnailAdapter.ThumbViewHolder) mThumbnailGridView.findViewHolderForAdapterPosition(position);
        if (viewHolder != null) {
            View itemView = viewHolder.itemView;
            float fromX = itemView.getX();
            float fromY = itemView.getY();

            TranslateAnimation translateAnimation = new TranslateAnimation(0, toX - fromX, 0, toY - fromY);
            AnimationSet animationSet = new AnimationSet(true);
            animationSet.setDuration(500);
            animationSet.addAnimation(translateAnimation);
            animationSet.setFillAfter(true);
            itemView.startAnimation(animationSet);
        }
    }

    private static class MovePageTask extends Task {
        private ThumbnailAdapter adapter;
        private ThumbnailItem dragedItem;

        private int movedPosition;
        private boolean success = true;

        public MovePageTask(int movedPosition, ThumbnailItem dragedItem,
                            ThumbnailAdapter adapter, final Event.Callback callback) {
            super(new CallBack() {
                @Override
                public void result(Task task) {
                    callback.result(null, ((MovePageTask) task).success);
                }
            });

            this.dragedItem = dragedItem;
            this.adapter = adapter;
            this.movedPosition = movedPosition;
        }

        @Override
        protected void execute() {
            List<ThumbnailItem> items = adapter.mThumbnailList;
            if (items.size() <= 1) return;

            SparseArray<ThumbnailItem> selectedItems = adapter.mSelectedArray;
            int selectedCount = selectedItems.size();
            int dstIndex;
            if (movedPosition == 0) {
                dstIndex = items.get(movedPosition + 1).getPageIndex();
                if (dstIndex > 0)
                    dstIndex -= 1;
            } else
                dstIndex = items.get(movedPosition - 1).getPageIndex();

            int[] pageIndexs = new int[selectedCount];
            List<ThumbnailItem> seletedList = new ArrayList<>();
            for (int i = 0; i < selectedItems.size(); i++) {
                ThumbnailItem item = selectedItems.valueAt(i);
                pageIndexs[i] = item.getPageIndex();
                seletedList.add(item);
            }
            items.remove(dragedItem);
            items.addAll(movedPosition, seletedList);

            int forwardCount = 0;
            int tempDstIndex = dstIndex;
            for (int i = 0; i < selectedCount; i++) {
                int pageIndex = pageIndexs[i];
                if (pageIndex <= dstIndex) {
                    int srcIndex = pageIndex - forwardCount;
                    if (srcIndex != dstIndex)
                        success = adapter.movePage(srcIndex, dstIndex);
                    forwardCount++;
                } else {
                    if (forwardCount > 0 || tempDstIndex != 0) {
                        dstIndex++;
                        forwardCount = 0;
                        tempDstIndex = 0;
                    }

                    if (pageIndex != dstIndex)
                        success = adapter.movePage(pageIndex, dstIndex);
                    dstIndex++;
                }
                if (!success) break;
            }
            success = true;
        }
    }

    boolean isPageCopied() {
        return mIsPageCopied;
    }

    void setPageCopied(boolean isCopied) {
        this.mIsPageCopied = isCopied;
    }

    void setCanPaste(boolean canPaste) {
        mbCanPaste = canPaste;
    }

    boolean canPaste() {
        return mbCanPaste;
    }

    void showPasteItem() {
        int insertPositon;
        if (mTabPosition == ThumbnailSupport.ALL_TAB_POSITION) {
            insertPositon = mCurPagePosition >= 0
                    ? Math.min(mCurPagePosition + 1, mAdapter.getItemCount()) : mAdapter.getItemCount();
        } else {
            insertPositon = mAdapter.getItemCount();
        }
        mAdapter.insertPasteItem(insertPositon);
        moveToPosition(insertPositon);
        updateUIState();
    }

    void removePasteItem() {
        mAdapter.removePasteItem();
    }

    void reset() {
        mbCanPaste = false;
        mIsPageCopied = false;
        if (mAdapter != null) {
            mAdapter.reset();
            mAdapter.notifyDataSetChanged();
        }
    }

    void notifyDataSetChanged() {
        if (getActivity() != null)
            getActivity().runOnUiThread(new Runnable() {
                @Override
                public void run() {
                    mAdapter.notifyDataSetChanged();
                }
            });
    }

    private void moveToPosition(int position) {
        int firstItem = mGridLayoutManager.findFirstVisibleItemPosition();
        int lastItem = mGridLayoutManager.findLastVisibleItemPosition();
        if (position <= firstItem) {
            mThumbnailGridView.scrollToPosition(position);
        } else if (position <= lastItem) {
            int top = mThumbnailGridView.getChildAt(position - firstItem).getTop();
            mThumbnailGridView.scrollBy(0, top);
        } else {
            mThumbnailGridView.scrollToPosition(position);
        }
    }

    public void setSingleChoice(boolean flag) {
        mIsSingleChoice = flag;
    }

    public void setSelectedPagePosition(int position) {
        mSelectedPagePosition = position;
    }

    @Override
    public void onDetach() {
        super.onDetach();
        if (mAdapter != null)
            mAdapter.release();
        mAdapter = null;
        AppUtil.removeViewFromParent(mThumbnailGridView);
    }
}
