/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.modules.thumbnail;

import android.graphics.Bitmap;
import android.graphics.Point;
import android.graphics.Rect;

import com.foxit.sdk.PDFException;
import com.foxit.sdk.PDFViewCtrl;
import com.foxit.sdk.addon.xfa.XFAPage;
import com.foxit.sdk.common.Constants;
import com.foxit.sdk.pdf.PDFPage;

import androidx.annotation.NonNull;

public class ThumbnailItem implements Comparable<ThumbnailItem> {
    private final PDFViewCtrl mPDFView;
    private Point mThumbnailSize;
    private final Point mBackgroundSize;
    private Rect mThumbnailRect;
    private Bitmap mBitmap;

    final static int CONTENT_ITEM = 0;
    final static int PASTE_ITEM = 1;

    public int itemType;
    private int mPageIndex = -1;
    private boolean isSelected;
    private boolean mbNeedCompute;
    private boolean isReadingBookmark;
    private boolean isRendering = false;
    private boolean isDragging = false;
    private boolean isBlank = false;

    public ThumbnailItem(Point backgroundSize, PDFViewCtrl pdfViewCtrl) {
        itemType = PASTE_ITEM;
        mPDFView = pdfViewCtrl;
        mBackgroundSize = backgroundSize;
    }

    public ThumbnailItem(int pageIndex, Point backgroundSize, PDFViewCtrl pdfViewCtrl) {
        itemType = CONTENT_ITEM;
        mPageIndex = pageIndex;
        mPDFView = pdfViewCtrl;
        mBackgroundSize = backgroundSize;
        isSelected = false;
        mbNeedCompute = true;
    }

    public ThumbnailItem(boolean isblank, int pageIndex, Point backgroundSize, PDFViewCtrl pdfViewCtrl) {
        this.isBlank = isblank;
        itemType = CONTENT_ITEM;
        mPageIndex = pageIndex;
        mPDFView = pdfViewCtrl;
        mBackgroundSize = backgroundSize;
        isSelected = false;
        mbNeedCompute = true;
    }

    public boolean isBlank(){
        return isBlank;
    }

    public int getPageIndex() {
        return mPageIndex;
    }

    public void setPageIndex(int pageIndex) {
        this.mPageIndex = pageIndex;
    }

    public PDFViewCtrl getPDFView() {
        return mPDFView;
    }

    public boolean isRendering() {
        return isRendering;
    }

    public void resetRending(boolean rendering) {
        isRendering = rendering;
    }

    public boolean isReadingBookmark() {
        return isReadingBookmark;
    }

    public void setReadingBookmark(boolean bookmark) {
        this.isReadingBookmark = bookmark;
    }

    public void setDragging(boolean isDragging) {
        this.isDragging = isDragging;
    }

    public boolean isDragging() {
        return isDragging;
    }

    public Bitmap getBitmap() {
        return mBitmap;
    }

    public void setBitmap(Bitmap bitmap) {
        mBitmap = bitmap;
    }

    public boolean needRecompute() {
        return mbNeedCompute;
    }

    public void setNeedCompute(boolean needCompute) {
        mbNeedCompute = needCompute;
    }

    public PDFPage getPage() {
        try {
            return mPDFView.getDoc().getPage(mPageIndex);
        } catch (PDFException e) {
            e.printStackTrace();
        }
        return null;
    }

    public void setSelected(boolean isSelected) {
        this.isSelected = isSelected;
    }

    public boolean isSelected() {
        return this.isSelected;
    }

    public boolean setRotation(int rotation) {
        int[] pageIndexes = new int[1];
        pageIndexes[0] = getPageIndex();
        boolean success = mPDFView.rotatePages(pageIndexes, rotation);
        mbNeedCompute = true;
        return success;
    }

    public int getRotation() {
        try {
            return getPage() != null ? getPage().getRotation() : 0;
        } catch (PDFException e) {
            e.printStackTrace();
        }
        return 0;
    }

    private void compute() {
        if (mThumbnailRect == null) {
            mThumbnailRect = new Rect();
        }
        if (mThumbnailSize == null) {
            mThumbnailSize = new Point();
        }

        try {
            float psWidth;
            float psHeight;
            if (mPDFView.isDynamicXFA()) {
                XFAPage page = mPDFView.getXFADoc().getPage(mPageIndex);
                if (page.isEmpty()) return;
                psWidth = page.getWidth();
                psHeight = page.getHeight();
            } else {
                PDFPage page = getPage();
                if (page == null || page.isEmpty())
                    return;
                psWidth = page.getWidth();
                psHeight = page.getHeight();
            }

            if (mPDFView.getViewRotation() == Constants.e_Rotation90 ||
                    mPDFView.getViewRotation() == Constants.e_Rotation270) {
                float tmp = psWidth;
                psWidth = psHeight;
                psHeight = tmp;
            }

            float scale = Math.min(mBackgroundSize.x / psWidth, mBackgroundSize.y / psHeight);
            psWidth *= scale;
            psHeight *= scale;
            int left = (int) (mBackgroundSize.x / 2.0f - psWidth / 2.0f);
            int top = (int) (mBackgroundSize.y / 2.0f - psHeight / 2.0f);
            final int right = mBackgroundSize.x - left;
            final int bottom = mBackgroundSize.y - top;
            mThumbnailRect.set(left, top, right, bottom);
            mThumbnailSize.set((int) psWidth, (int) psHeight);

            mbNeedCompute = false;
        } catch (PDFException e) {
            e.printStackTrace();
        }
    }

    public Point getSize() {
        if (mbNeedCompute)
            compute();
        return new Point(mThumbnailSize);
    }

    public Point getBackgroundSize() {
        return mBackgroundSize;
    }

    public Rect getRect() {
        if (mbNeedCompute)
            compute();
        return new Rect(mThumbnailRect);
    }

    @Override
    public boolean equals(Object o) {
        if (!(o instanceof ThumbnailItem)) return false;
        return this == o || this.getPageIndex() == ((ThumbnailItem) o).getPageIndex();
    }

    @Override
    public int compareTo(@NonNull ThumbnailItem another) {
        return getPageIndex() - another.getPageIndex();
    }
}
