/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.modules.thumbnail.createpage;


import android.content.Context;
import android.content.res.ColorStateList;
import android.graphics.PorterDuff;
import android.graphics.PorterDuffColorFilter;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.ImageView;
import android.widget.TextView;

import com.foxit.sdk.PDFViewCtrl;
import com.foxit.uiextensions.R;
import com.foxit.uiextensions.browser.adapter.SuperAdapter;
import com.foxit.uiextensions.browser.adapter.viewholder.SuperViewHolder;
import com.foxit.uiextensions.modules.panel.bean.BaseBean;
import com.foxit.uiextensions.theme.ThemeConfig;
import com.foxit.uiextensions.utils.AppResource;

import java.util.ArrayList;
import java.util.List;

import androidx.annotation.NonNull;

public class PageStyleAdapter extends SuperAdapter<PageStyleAdapter.PageStyleBean> {

    private OnPageStyleItemClickListener mItemClickListener;
    private List<PageStyleBean> mItems;

    private int mLastSelectedPosition = -1;

    public PageStyleAdapter(Context context) {
        super(context);
        mItems = new ArrayList<>();
    }

    @Override
    public void notifyUpdateData() {
        notifyDataSetChanged();
    }

    @Override
    public PageStyleBean getDataItem(int position) {
        return mItems.get(position);
    }

    @Override
    public int getItemCount() {
        return mItems.size();
    }

    @NonNull
    @Override
    public SuperViewHolder onCreateViewHolder(@NonNull ViewGroup parent, int viewType) {
        return new PageStyleViewHolder(LayoutInflater.from(getContext()).inflate(R.layout.thumbnail_page_style_item, parent, false));
    }

    public class PageStyleViewHolder extends SuperViewHolder {
        private final ImageView mStyleIcon;
        private final TextView mStyleName;

        public PageStyleViewHolder(View itemView) {
            super(itemView);

            mStyleIcon = itemView.findViewById(R.id.thumbnail_blank_page_icon);
            mStyleName = itemView.findViewById(R.id.thumbnail_blank_page_name);
        }

        @Override
        public void bind(BaseBean data, int position) {
            PageStyleBean styleBean = (PageStyleBean) data;
            mStyleIcon.setImageResource(styleBean.mIconId);
            mStyleName.setText(styleBean.mName);
            int bgColor;
            if (styleBean.mSelected)
                bgColor = ThemeConfig.getInstance(getContext()).getPrimaryColor();
            else
                bgColor = AppResource.getColor(getContext(), R.color.ux_color_translucent);
            mStyleIcon.getBackground().setColorFilter(new PorterDuffColorFilter(bgColor, PorterDuff.Mode.SRC_IN));
        }

        @Override
        public void onClick(View v) {
            int positon = getAdapterPosition();
            if (mLastSelectedPosition == positon) return;

            if (mLastSelectedPosition != -1) {
                PageStyleBean lastSelectedStyleBean = mItems.get(mLastSelectedPosition);
                lastSelectedStyleBean.mSelected = false;
                notifyItemChanged(mLastSelectedPosition);
            }

            PageStyleBean styleBean = mItems.get(positon);
            styleBean.mSelected = true;
            notifyItemChanged(positon);
            mLastSelectedPosition = positon;

            if (mItemClickListener != null)
                mItemClickListener.onItemClick(styleBean.mStyle);
        }
    }

    static class PageStyleBean extends BaseBean {
        public int mStyle;
        public int mIconId;
        public String mName;
        public boolean mSelected;
    }

    public static ColorStateList getIconStateList(Context context) {
        int pressed = AppResource.getColor(context, R.color.ux_color_translucent);
        int disabled = ThemeConfig.getInstance(context).getI2();
        int selected = ThemeConfig.getInstance(context).getPrimaryColor();
        int normal = AppResource.getColor(context, R.color.ux_color_translucent);
        return AppResource.createColorStateList(pressed, selected, pressed, disabled, normal);
    }

    public void initData() {
        PageStyleBean blank = new PageStyleBean();
        blank.mSelected = true;
        blank.mIconId = R.drawable.rd_createpdf_bk_blank;
        blank.mName = AppResource.getString(getContext(), R.string.createpdf_new_page_blank);
        blank.mStyle = PDFViewCtrl.PDF_PAGE_STYLE_TYPE_BLANK;

        PageStyleBean line = new PageStyleBean();
        line.mSelected = false;
        line.mIconId = R.drawable.rd_createpdf_bk_lined;
        line.mName = AppResource.getString(getContext(), R.string.createpdf_new_page_lined);
        line.mStyle = PDFViewCtrl.PDF_PAGE_STYLE_TYPE_LINED;

        PageStyleBean grid = new PageStyleBean();
        grid.mSelected = false;
        grid.mIconId = R.drawable.rd_createpdf_bk_grid;
        grid.mName = AppResource.getString(getContext(), R.string.createpdf_new_page_grid);
        grid.mStyle = PDFViewCtrl.PDF_PAGE_STYLE_TYPE_GRID;

        PageStyleBean graph = new PageStyleBean();
        graph.mSelected = false;
        graph.mIconId = R.drawable.rd_createpdf_bk_graph;
        graph.mName = AppResource.getString(getContext(), R.string.createpdf_new_page_graph);
        graph.mStyle = PDFViewCtrl.PDF_PAGE_STYLE_TYPE_GRAPH;

        PageStyleBean music = new PageStyleBean();
        music.mSelected = false;
        music.mIconId = R.drawable.rd_createpdf_bk_music;
        music.mName = AppResource.getString(getContext(), R.string.createpdf_new_page_music);
        music.mStyle = PDFViewCtrl.PDF_PAGE_STYLE_TYPE_MUSIC;

        mItems.add(blank);
        mItems.add(line);
        mItems.add(grid);
        mItems.add(graph);
        mItems.add(music);

        mLastSelectedPosition = 0;
    }

    public void setItemClickListener(OnPageStyleItemClickListener itemClickListener) {
        mItemClickListener = itemClickListener;
    }

    public interface OnPageStyleItemClickListener {
        void onItemClick(int type);
    }

}
