/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.pdfreader;

import android.app.Activity;
import android.content.Context;
import android.view.ViewGroup;
import android.view.animation.Animation;
import android.widget.RelativeLayout;

import com.foxit.uiextensions.controls.propertybar.IViewSettingsWindow;
import com.foxit.uiextensions.controls.propertybar.PropertyBar;
import com.foxit.uiextensions.controls.toolbar.BaseBar;
import com.foxit.uiextensions.controls.toolbar.ToolbarItemConfig;
import com.foxit.uiextensions.controls.toolbar.drag.UIDragToolBar;
import com.foxit.uiextensions.controls.toolbar.drag.UIToolBaseBar;
import com.foxit.uiextensions.modules.panel.IPanelManager;
import com.foxit.uiextensions.pdfreader.tab.ITabSelectedListener;

import java.util.List;

/**
 * Interface definition for main frame of the complete pdf reader. It provides an entrance to access the UI of reader. For example,
 * getTopToolbar/getBottomToolbar will provide caller the interface for top bar and bottom bar of pdf view control. The SDK user
 * is able to modify or customize the reader UI through these interfaces.
 */
public interface IMainFrame {

    /**
     * @return Return the {@link Activity} this frame is currently associated with.
     */
    Activity getAttachedActivity();

    /**
     * Set the {@link Activity} associated with this frame.
     *
     * @param act {@link Activity}
     */
    void setAttachedActivity(Activity act);

    /**
     * @return {@link Context} this frame is currently associated with.
     */
    Context getContext();

    /**
     * @return Get the root view for the frame layout
     */
    RelativeLayout getContentView();

    /**
     * Show th tool bars, exit full screen mode
     */
    void showToolbars();

    /**
     * Hide the tool bars, enter the full screen mode
     */
    void hideToolbars();

    /**
     * @return If it's full screen mode, reture true, false otherwise.
     */
    boolean isToolbarsVisible();

    /**
     * @return The top toolbar
     */
    BaseBar getTopToolbar();

    /**
     * @return The bottom toolbar
     */
    BaseBar getBottomToolbar();

    /**
     * @param isEnabled If set false, the top toolbar will be hidden, true otherwise.
     */
    void enableTopToolbar(boolean isEnabled);

    /**
     * @param isEnabled If set false, the bottom toolbar will be hidden, true otherwise.
     */
    void enableBottomToolbar(boolean isEnabled);

    /**
     * @return True indicates that the top toolbar is the display state. false otherwise.
     */
    boolean isShowTopToolbar();

    /**
     * @return True indicates that the bottom toolbar is the display state. false otherwise.
     */
    boolean isShowBottomToolbar();

    /**
     * @return The custom top bar
     */
    BaseBar getCustomTopbar();

    /**
     * @return The custom bottom bar
     */
    BaseBar getCustomBottombar();

    /**
     * @return The manager of the panel{@link IPanelManager}
     */
    IPanelManager getPanelManager();

    /**
     * @return The settings window.
     */
    IViewSettingsWindow getSettingWindow();

    /**
     * Hide the settings window.
     */
    void hideSettingWindow();

    /**
     * @return Get the action view, it is below the top bar and its sub-toolbar, and can be used to insert a custom views,
     */
    ViewGroup getTopActionView();

/// @cond DEV

    /**
     * @return Get money from the Toolbar
     */
    public UIToolBaseBar getCurToolbar();

    /**
     * Get toolbar by type
     *
     * @param tabType {@link ToolbarItemConfig#ITEM_HOME_TAB}
     * @return Get toolbar by type
     */
    public UIToolBaseBar getToolbar(int tabType);

    /**
     * Get the dragged toolbar. When {@link UIToolBaseBar} can be dragged, you can use this to get
     * the dragged toolbar and do some drag related events.
     *
     * @return Get the dragged toolbar
     */
    public UIDragToolBar getDragToolBar();

    public void setToolBar(UIToolBaseBar toolBar);
/// @endcond

    /**
     * Set the toolbar position, Currently supported locations can refer to {@link UIDragToolBar.ToolbarPos}.
     *
     * @param position The position of  toolbar. {@link UIDragToolBar.ToolbarPos}
     */
    public void setToolBarPosition(@UIDragToolBar.ToolbarPos int position);

    /**
     * Get the position of the toolbar.
     *
     * @return The position of the toolbar.
     */
    @UIDragToolBar.ToolbarPos
    public int getToolBarPosition();

    /**
     * @return {@link ToolbarItemConfig#ITEM_HOME_TAB}
     */
    public int getCurrentTab();

    /**
     * @param tabType {@link ToolbarItemConfig#ITEM_HOME_TAB}
     */
    public void setCurrentTab(int tabType);

    /**
     * <p>
     * Remove the tab in the center of the topbar and its sub-toolbar
     * </p>
     * Note:Removing the tab of the topbar will not affect the full screen toolbar
     *
     * @param tabType Please refer to values starting from {@link ToolbarItemConfig#ITEM_HOME_TAB}
     */
    public void removeTab(int tabType);

    public void setCenterItems(List<MainCenterItemBean> items);

    public List<MainCenterItemBean> getCenterItems();

    public void setAddToolsProperty(List<AddToolsBean> items);

    public List<AddToolsBean> getAddToolsProperty();

    PropertyBar getPropertyBar();

    void showMaskView();

    void hideMaskView();

    boolean isMaskViewShowing();

    int getMaskViewIndex();

    public Animation getTopbarShowAnimation();

    public Animation getBottombarShowAnimation();

    public Animation getTopbarHideAnimation();

    public Animation getBottombarHideAnimation();

    /**
     * Enable or disable small top toolbar.
     *
     * @param isEnabled Whether small top toolbar is enabled or not.
     */
    void enableSmallTopToolbar(boolean isEnabled);

    /**
     * Enable or disable small bottom toolbar.
     *
     * @param isEnabled Whether small bottom toolbar is enabled or not.
     */
    void enableSmallBottomToolbar(boolean isEnabled);

    /**
     * Returns the enabled status for the small top toolbar
     *
     * @return True if this view is enabled, false otherwise.
     */
    boolean isSmallTopToolbarEnabled();

    /**
     * Returns the enabled status for the small bottom toolbar
     *
     * @return True if this view is enabled, false otherwise.
     */
    boolean isSmallBottomToolbarEnabled();

    void addTabSelectedListener(ITabSelectedListener tabSelectedListener);

    void removeTabSelectedListener(ITabSelectedListener tabSelectedListener);

    void setPageBackgroundColor(int color);

}