/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.pdfreader.impl;

import android.app.Activity;
import android.content.Context;
import android.content.Intent;
import android.text.TextUtils;
import android.view.MotionEvent;
import android.view.View;
import android.view.ViewGroup;
import android.widget.LinearLayout;
import android.widget.TextView;
import android.widget.Toast;

import com.foxit.sdk.PDFViewCtrl;
import com.foxit.sdk.common.Constants;
import com.foxit.sdk.pdf.PDFDoc;
import com.foxit.uiextensions.R;
import com.foxit.uiextensions.UIExtensionsManager;
import com.foxit.uiextensions.controls.dialog.PasswordDialog;
import com.foxit.uiextensions.event.DocEventListener;
import com.foxit.uiextensions.pdfreader.ILifecycleEventListener;
import com.foxit.uiextensions.theme.ThemeUtil;
import com.foxit.uiextensions.utils.AppFileUtil;
import com.foxit.uiextensions.utils.AppResource;
import com.foxit.uiextensions.utils.AppUtil;
import com.foxit.uiextensions.utils.UIToast;

public class SimpleViewer extends View {
    private View mOpenView;
    private TextView mTvFileName;
    private LinearLayout mLlContent;

    private PDFViewCtrl mAttachPdfViewCtrl;
    private PDFViewCtrl mPdfViewCtrl;


    private Context mContext;
    private ViewGroup mParent;

    public SimpleViewer(Context context, PDFViewCtrl pdfViewCtrl, ViewGroup parent) {
        super(context);
        this.mContext = context;
        this.mPdfViewCtrl = pdfViewCtrl;
        this.mParent = parent;

        initOpenView();
    }


    private void initOpenView() {
        mOpenView = View.inflate(mContext, R.layout.attachment_view, null);
        mOpenView.findViewById(R.id.panel_edit_tv).setVisibility(View.INVISIBLE);
        TextView closeTv = mOpenView.findViewById(R.id.panel_close_tv);
        closeTv.setTextColor(ThemeUtil.getPrimaryTextColor(mContext));
        closeTv.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                mOpenView.setVisibility(View.GONE);
                mAttachPdfViewCtrl.closeDoc();
            }
        });

        mLlContent = mOpenView.findViewById(R.id.attachment_view_content_ly);
        mTvFileName = (TextView) mOpenView.findViewById(R.id.panel_title_tv);
        mParent.addView(mOpenView);
        mOpenView.setVisibility(View.GONE);
        mOpenView.setOnTouchListener(new View.OnTouchListener() {
            @Override
            public boolean onTouch(View v, MotionEvent event) {
                return true;
            }
        });
        setVisibility(GONE);

    }

    private boolean mPasswordError = false;

    private void create(final String filePath, final String fileName, final ISimpleViewerCallBack callBack) {
        mAttachPdfViewCtrl = new PDFViewCtrl(mContext);
        mTvFileName.setText(fileName);
        mLlContent.removeAllViews();
        mLlContent.addView(mAttachPdfViewCtrl);
        mOpenView.setVisibility(View.VISIBLE);
        mLlContent.setVisibility(View.VISIBLE);
        mAttachPdfViewCtrl.setAttachedActivity(mPdfViewCtrl.getAttachedActivity());
        ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).registerLifecycleListener(mLifecycleEventListener);
//        mAttachPdfViewCtrl.setConnectedPDFEventListener(mPdfViewCtrl.getConnectedPdfEventListener()); //unsupported
        mAttachPdfViewCtrl.registerDocEventListener(new DocEventListener() {
            @Override
            public void onDocWillOpen() {
                if (callBack != null) {
                    callBack.onDocWillOpen();
                }
            }

            @Override
            public void onDocOpened(PDFDoc document, int errCode) {
                if (callBack != null) {
                    callBack.onDocOpened();
                }
                ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).unregisterLifecycleListener(mLifecycleEventListener);
                switch (errCode) {
                    case Constants.e_ErrSuccess:
                        mAttachPdfViewCtrl.setContinuous(true);
                        mAttachPdfViewCtrl.setPageLayoutMode(PDFViewCtrl.PAGELAYOUTMODE_SINGLE);
                        mPasswordError = false;
                        return;
                    case Constants.e_ErrPassword:
                        if (TextUtils.isEmpty(filePath)) {
                            Toast.makeText(mContext.getApplicationContext(), R.string.rv_document_open_failed, Toast.LENGTH_SHORT).show();
                            return;
                        }
                        String tips;
                        if (mPasswordError) {
                            tips = AppResource.getString(mContext.getApplicationContext(), R.string.rv_tips_password_error);
                        } else {
                            tips = AppResource.getString(mContext.getApplicationContext(), R.string.rv_tips_password);
                        }

                        Activity activity = ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getAttachedActivity();
                        PasswordDialog passwordDialog = new PasswordDialog(activity, new PasswordDialog.IPasswordDialogListener() {
                            @Override
                            public void onConfirm(byte[] password) {
                                String openPath = AppFileUtil.getAdaptedFilePath(mContext, filePath);
                                mAttachPdfViewCtrl.openDoc(openPath, password);
                            }

                            @Override
                            public void onDismiss() {
                                mPasswordError = false;
                                Toast.makeText(mContext.getApplicationContext(), R.string.rv_document_open_failed, Toast.LENGTH_SHORT).show();
                            }

                            @Override
                            public void onKeyBack() {
                                mPasswordError = false;
                                Toast.makeText(mContext.getApplicationContext(), R.string.rv_document_open_failed, Toast.LENGTH_SHORT).show();
                            }
                        });
                        passwordDialog.setTitle(AppResource.getString(mContext.getApplicationContext(), R.string.rv_password_dialog_title))
                                .setPromptTips(tips)
                                .show();

                        if (!mPasswordError) {
                            mPasswordError = true;
                        }
                        return;
                    default:
                        String message = AppUtil.getMessage(mContext, errCode);
                        UIToast.getInstance(mContext).show(message);
                        break;
                }
            }

            @Override
            public void onDocWillClose(PDFDoc document) {
            }

            @Override
            public void onDocClosed(PDFDoc document, int errCode) {
                mAttachPdfViewCtrl = null;
                if (callBack != null) {
                    callBack.onDocClosed();
                }
            }

        });
        mAttachPdfViewCtrl.setContinuous(true);
        mAttachPdfViewCtrl.setPageLayoutMode(PDFViewCtrl.PAGELAYOUTMODE_SINGLE);
    }

    public void open(final String filePath, final String fileName, final ISimpleViewerCallBack callBack) {
        create(filePath, fileName, callBack);
        String openPath = AppFileUtil.getAdaptedFilePath(mContext, filePath);
        mAttachPdfViewCtrl.openDoc(openPath, null);
        setVisibility(View.VISIBLE);
    }

    public void open(final PDFDoc pdfDoc, final String fileName, final ISimpleViewerCallBack callBack) {
        create(null, fileName, callBack);
        mAttachPdfViewCtrl.setDoc(pdfDoc);
        setVisibility(View.VISIBLE);
    }

    public void close() {
        if (mAttachPdfViewCtrl != null) {
            mAttachPdfViewCtrl.closeDoc();
            mOpenView.setVisibility(GONE);
        }
        setVisibility(GONE);
    }

    private ILifecycleEventListener mLifecycleEventListener = new LifecycleEventListener() {
        @Override
        public void onActivityResult(Activity act, int requestCode, int resultCode, Intent data) {
            if (mAttachPdfViewCtrl != null) {
                mAttachPdfViewCtrl.handleActivityResult(requestCode, resultCode, data);
            }
        }
    };

    public interface ISimpleViewerCallBack {
        void onDocWillOpen();

        void onDocOpened();

        void onDocClosed();
    }
}
