/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.pdfreader.tab;


import android.annotation.SuppressLint;
import android.content.Context;
import android.view.LayoutInflater;
import android.view.MotionEvent;
import android.view.View;
import android.view.ViewGroup;
import android.widget.ImageView;
import android.widget.TextView;

import com.foxit.uiextensions.R;
import com.foxit.uiextensions.browser.adapter.SuperAdapter;
import com.foxit.uiextensions.browser.adapter.viewholder.SuperViewHolder;
import com.foxit.uiextensions.controls.toolbar.IBaseItem;
import com.foxit.uiextensions.modules.panel.bean.BaseBean;
import com.foxit.uiextensions.theme.ThemeUtil;
import com.foxit.uiextensions.utils.AppResource;
import com.foxit.uiextensions.utils.Event;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import androidx.annotation.NonNull;
import androidx.recyclerview.widget.RecyclerView;

class TabMenuAdapter extends SuperAdapter<TabMenuAdapter.TabItemBean>
        implements TabMenuItemTouchHelper.ItemTouchHelperCallback {

    private ITabMenuChangedListener mTabMenuChangedListener;
    private ITabMenuClickListener mTabMenuClickListener;
    private List<TabItemBean> mOnTools;
    private List<TabItemBean> mMoretools;
    private List<TabItemBean> mAllTools;

    private TabItemBean mHeaderItem;
    private TabItemBean mFooterItem;
    private TabItemBean mMoreToolsTitle;
    private int mMoveStartPosition;
    private boolean mIsLongPressDragEnabled = true;

    public TabMenuAdapter(Context context, List<TabItemBean> on_tools, List<TabItemBean> more_tools) {
        super(context);
        initItems(on_tools, more_tools);
    }

    public void initItems(List<TabItemBean> on_tools, List<TabItemBean> more_tools) {
        mOnTools = on_tools;
        mMoretools = more_tools;
        mAllTools = new ArrayList<>();

        TabItemBean onToolsTitle = new TabItemBean();
        onToolsTitle.mViewType = TabConstants.Tab_Title;
        onToolsTitle.mText = AppResource.getString(getContext(), R.string.top_tab_menu_on_tools_title);
        mAllTools.add(onToolsTitle);
        if (on_tools.size() == 0) {
            mAllTools.add(getHeaderItem());
        } else {
            mAllTools.addAll(on_tools);
        }

        mMoreToolsTitle = new TabItemBean();
        mMoreToolsTitle.mViewType = TabConstants.Tab_Title;
        mMoreToolsTitle.mText = AppResource.getString(getContext(), R.string.top_tab_menu_more_tools_title);
        mAllTools.add(mMoreToolsTitle);
        if (more_tools.size() == 0) {
            mAllTools.add(getFooterItem());
        } else {
            mAllTools.addAll(more_tools);
        }
    }

    private TabItemBean getHeaderItem() {
        if (mHeaderItem == null) {
            mHeaderItem = new TabItemBean();
            mHeaderItem.mViewType = TabConstants.Tab_Header;
            mHeaderItem.mText = AppResource.getString(getContext(), R.string.top_tab_menu_add_on_tools_tips);
        }
        return mHeaderItem;
    }

    private TabItemBean getFooterItem() {
        if (mFooterItem == null) {
            mFooterItem = new TabItemBean();
            mFooterItem.mViewType = TabConstants.Tab_Footer;
            mFooterItem.mText = AppResource.getString(getContext(), R.string.top_tab_menu_add_more_tools_tips);
        }
        return mFooterItem;
    }

    public List<TabItemBean> getAllToolItems() {
        return mAllTools;
    }

    public List<TabItemBean> getOnToolItems() {
        return mOnTools;
    }

    public List<TabItemBean> getMoreToolItems() {
        return mMoretools;
    }

    public void removeItem(int tabId) {
        for (int i = 0; i < mAllTools.size(); i++) {
            TabItemBean tabItemBean = mAllTools.get(i);
            if (tabItemBean.mViewType == TabConstants.Tab_Content) {
                if (tabItemBean.mTabItem.getTag() == tabId) {
                    mAllTools.remove(tabItemBean);
                    mMoretools.remove(tabItemBean);
                    mOnTools.remove(tabItemBean);
                    notifyItemRemoved(i);
                    break;
                }
            }
        }
    }

    public void changeMoreToolsPosition(int toPosition) {
        int fromPostion = mAllTools.indexOf(mMoreToolsTitle);
        if (fromPostion == toPosition) return;

        mAllTools.remove(mMoreToolsTitle);
        notifyItemRemoved(fromPostion);
        mAllTools.add(toPosition, mMoreToolsTitle);
        notifyItemInserted(toPosition);
        if (mAllTools.contains(getFooterItem())) {
            int removePosition = mAllTools.indexOf(getFooterItem());
            mAllTools.remove(getFooterItem());
            notifyItemRemoved(removePosition);
        }

        mOnTools.clear();
        for (int i = 0; i < toPosition; i++) {
            TabItemBean tabItemBean = mAllTools.get(i);
            if (tabItemBean.mViewType == TabConstants.Tab_Content) {
                mOnTools.add(tabItemBean);
            }
        }
        mMoretools.clear();
        for (int i = toPosition; i < mAllTools.size(); i++) {
            TabItemBean tabItemBean = mAllTools.get(i);
            if (tabItemBean.mViewType == TabConstants.Tab_Content) {
                mMoretools.add(tabItemBean);
            }
        }
    }

    @Override
    public void notifyUpdateData() {
        notifyDataSetChanged();
    }

    @Override
    public TabItemBean getDataItem(int position) {
        return mAllTools.get(position);
    }

    @Override
    public int getItemViewType(int position) {
        TabItemBean itemBean = mAllTools.get(position);
        return itemBean.mViewType;
    }

    @NonNull
    @Override
    public SuperViewHolder onCreateViewHolder(@NonNull ViewGroup parent, int viewType) {
        if (viewType == TabConstants.Tab_Header || viewType == TabConstants.Tab_Footer) {
            View view = LayoutInflater.from(getContext()).inflate(R.layout.top_bar_tab_menu_header, parent, false);
            return new HeaderFooterViewHolder(view);
        } else if (viewType == TabConstants.Tab_Title) {
            View view = LayoutInflater.from(getContext()).inflate(R.layout.top_bar_tab_menu_title, parent, false);
            return new TitleItemViewHolder(view);
        } else {
            View view = LayoutInflater.from(getContext()).inflate(R.layout.top_bar_tab_menu_content, parent, false);
            return new ContentItemViewHolder(view);
        }
    }

    @Override
    public int getItemCount() {
        return mAllTools.size();
    }

    @Override
    public boolean isLongPressDragEnabled() {
        return mIsLongPressDragEnabled;
    }

    @Override
    public void onMoveStart(RecyclerView.ViewHolder viewHolder) {
        viewHolder.itemView.setScaleX(1.05f);
        viewHolder.itemView.setScaleY(1.05f);
        mMoveStartPosition = viewHolder.getAdapterPosition();
    }

    @Override
    public boolean onMove(RecyclerView.ViewHolder viewHolder, int fromPosition, int toPosition) {
        if (toPosition == 0) return false;

        Collections.swap(mAllTools, fromPosition, toPosition);
        notifyItemMoved(fromPosition, toPosition);
        return true;
    }

    @Override
    public void onMoveEnd(RecyclerView.ViewHolder viewHolder) {
        viewHolder.itemView.setScaleX(1.0f);
        viewHolder.itemView.setScaleY(1.0f);

        int moveEndPosition = viewHolder.getAdapterPosition();
        if (moveEndPosition == -1) return;

        if (mMoveStartPosition == moveEndPosition) {
            mMoveStartPosition = 0;
            return;
        }

        int moreTitleIndex = mAllTools.indexOf(mMoreToolsTitle);
        TabItemBean moveItemBean = mAllTools.get(moveEndPosition);
        if (moveEndPosition > moreTitleIndex) {
            if (mAllTools.contains(getFooterItem())) {
                int removePosition = mAllTools.indexOf(getFooterItem());
                mAllTools.remove(getFooterItem());
                notifyItemRemoved(removePosition);
            }

            if (moreTitleIndex == 1) {
                mAllTools.add(1, getHeaderItem());
                notifyItemInserted(1);
            }

            mMoretools.clear();
            int fromIndex = mAllTools.indexOf(mMoreToolsTitle);
            for (int i = fromIndex + 1; i < mAllTools.size(); i++) {
                mMoretools.add(mAllTools.get(i));
            }

            if (mMoveStartPosition <= moreTitleIndex) {
                int removeIndex = mOnTools.indexOf(moveItemBean);
                mOnTools.remove(moveItemBean);
                if (mTabMenuChangedListener != null)
                    mTabMenuChangedListener.onItemRemoved(removeIndex, moveItemBean);
            }
        } else {
            if (moreTitleIndex == mAllTools.size() - 1) {
                mAllTools.add(getFooterItem());
                notifyItemInserted(mAllTools.size());
            }

            if (mAllTools.contains(getHeaderItem())) {
                int removePosition = mAllTools.indexOf(getHeaderItem());
                mAllTools.remove(getHeaderItem());
                notifyItemRemoved(removePosition);
            }

            mOnTools.clear();
            int endIndex = mAllTools.indexOf(mMoreToolsTitle);
            for (int i = 1; i < endIndex; i++) {
                mOnTools.add(mAllTools.get(i));
            }

            if (mMoveStartPosition >= moreTitleIndex) {
                mMoretools.remove(moveItemBean);
                if (mTabMenuChangedListener != null)
                    mTabMenuChangedListener.onItemInserted(mOnTools.indexOf(moveItemBean), moveItemBean);
            } else {
                if (mTabMenuChangedListener != null)
                    mTabMenuChangedListener.onItemMoved(mOnTools.indexOf(moveItemBean), moveItemBean);
            }
        }
        mMoveStartPosition = 0;
    }

    class ContentItemViewHolder extends SuperViewHolder {

        private final TextView mItemDetailTv;
        private final ImageView mItemIconView;

        public ContentItemViewHolder(View itemView) {
            super(itemView);
            mItemDetailTv = itemView.findViewById(R.id.tab_item_content_text);
            mItemIconView = itemView.findViewById(R.id.tab_item_content_icon);
            itemView.findViewById(R.id.tab_item_content_ll).setOnClickListener(this);
        }

        @Override
        public void bind(BaseBean data, int position) {
            TabItemBean itemBean = (TabItemBean) data;

            IBaseItem tabItem = itemBean.mTabItem;
            mItemIconView.setImageResource(tabItem.getHighlightImageResource());
            mItemIconView.setSelected(tabItem.isSelected());
            mItemDetailTv.setText(tabItem.getText());
            mItemDetailTv.setSelected(tabItem.isSelected());
            ThemeUtil.setTintList(mItemIconView, ThemeUtil.getListIconColor(getContext()));
            ThemeUtil.setTintList(mItemDetailTv, ThemeUtil.getListTextColor(getContext()));
        }

        @Override
        public void onClick(View v) {
            int id = v.getId();
            final int position = getAdapterPosition();
            if (position == -1) return;

            if (id == R.id.tab_item_content_ll) {
                int moreTitleIndex = mAllTools.indexOf(mMoreToolsTitle);
                final TabItemBean tabItemBean = mAllTools.get(position);
                if (position < moreTitleIndex) {
                    if (mTabMenuClickListener != null)
                        mTabMenuClickListener.onOnToolClick(position, tabItemBean);
                } else {
                    if (mTabMenuClickListener != null) {
                        mTabMenuClickListener.checkSubscriptionPermission(tabItemBean.mTabItem.getTag(), new Event.Callback() {
                            @Override
                            public void result(Event event, boolean success) {
                                if (success) {
                                    mAllTools.remove(tabItemBean);
                                    mAllTools.add(mAllTools.indexOf(mMoreToolsTitle), tabItemBean);
                                    mAllTools.remove(getHeaderItem());
                                    int moreTitleIndex = mAllTools.indexOf(mMoreToolsTitle);
                                    if (moreTitleIndex == mAllTools.size() - 1) {
                                        mAllTools.add(getFooterItem());
                                    }
                                    mOnTools.add(tabItemBean);
                                    mMoretools.remove(tabItemBean);
                                    notifyUpdateData();
                                    if (mTabMenuClickListener != null)
                                        mTabMenuClickListener.onMoreToolClick(position, tabItemBean);
                                }
                            }
                        });
                    }
                }
            }
        }
    }

    class TitleItemViewHolder extends SuperViewHolder {

        private final TextView mItemTitleTv;

        public TitleItemViewHolder(View itemView) {
            super(itemView);
            mItemTitleTv = itemView.findViewById(R.id.tab_item_title_tv);
            itemView.setOnTouchListener(new View.OnTouchListener() {
                @SuppressLint("ClickableViewAccessibility")
                @Override
                public boolean onTouch(View v, MotionEvent event) {
                    if (event.getAction() == MotionEvent.ACTION_DOWN) {
                        mIsLongPressDragEnabled = false;
                    } else if (event.getAction() == MotionEvent.ACTION_UP
                            || event.getAction() == MotionEvent.ACTION_CANCEL) {
                        mIsLongPressDragEnabled = true;
                    }
                    return false;
                }
            });
        }

        @Override
        public void bind(BaseBean data, int position) {
            TabItemBean itemBean = (TabItemBean) data;
            mItemTitleTv.setText(itemBean.mText);
        }

        @Override
        public void onClick(View v) {
        }
    }

    class HeaderFooterViewHolder extends SuperViewHolder {

        private final TextView mFrameTextView;

        public HeaderFooterViewHolder(View itemView) {
            super(itemView);
            mFrameTextView = itemView.findViewById(R.id.tab_item_header_frame);
            itemView.setOnTouchListener(new View.OnTouchListener() {
                @SuppressLint("ClickableViewAccessibility")
                @Override
                public boolean onTouch(View v, MotionEvent event) {
                    if (event.getAction() == MotionEvent.ACTION_DOWN) {
                        mIsLongPressDragEnabled = false;
                    } else if (event.getAction() == MotionEvent.ACTION_UP
                            || event.getAction() == MotionEvent.ACTION_CANCEL) {
                        mIsLongPressDragEnabled = true;
                    }
                    return false;
                }
            });
        }

        @Override
        public void bind(BaseBean data, int position) {
            TabItemBean itemBean = (TabItemBean) data;
            mFrameTextView.setText(itemBean.mText);
        }

        @Override
        public void onClick(View v) {
        }
    }

    static class TabItemBean extends BaseBean {
        int mViewType;
        String mText;
        IBaseItem mTabItem;
    }

    void setTabMenuChangedListener(ITabMenuChangedListener menuChangedListener) {
        mTabMenuChangedListener = menuChangedListener;
    }

    void setTabMenuClickListener(ITabMenuClickListener menuClickListener) {
        mTabMenuClickListener = menuClickListener;
    }

    interface ITabMenuClickListener {
        void onOnToolClick(int position, TabItemBean tabItemBean);

        void onMoreToolClick(int position, TabItemBean tabItemBean);

        void checkSubscriptionPermission(int tabId, Event.Callback callback);
    }

}
