/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.security;

import org.bouncycastle.jce.provider.BouncyCastleProvider;

import java.io.InputStream;
import java.security.Key;
import java.security.KeyPair;
import java.security.KeyStore;
import java.security.PrivateKey;
import java.security.Security;
import java.security.cert.X509Certificate;
import java.security.interfaces.RSAPrivateKey;
import java.security.interfaces.RSAPublicKey;
import java.util.Enumeration;
import java.util.Iterator;
import java.util.Map;
import java.util.TreeMap;

public class PfxCrypto {
    static String KEYSTORE_TYPE = "PKCS12";
    static String PROVIDER_NAME = "BC";;

    InputStream mPfxInputStream;
    String mPfxPassword;

    public KeyStore mKeyStore;

    KeyPair mKeyPair;
    X509Certificate mUserCert;
    public String mAlias;

    Map<Long, KeyPair> mKeyPairMap = new TreeMap();
    Map<Long, X509Certificate> mKSCertMap = new TreeMap();
    Map<String, Long> mAliasMap = new TreeMap();

    static {
        Security.addProvider(new BouncyCastleProvider());
    }

    public PfxCrypto(InputStream keyStoreInputStream) {
        mPfxInputStream = keyStoreInputStream;
    }

    public synchronized void load(String password) throws Exception {
        mPfxPassword = password;

        try {
            mKeyStore = KeyStore.getInstance(KEYSTORE_TYPE, PROVIDER_NAME);
            mKeyStore.load(mPfxInputStream, mPfxPassword.toCharArray());
        } catch (Exception e) {
            throw new Exception("load error", e);
        }

        try {
            findKeyPair();
        } catch (Exception e2) {
            if (e2 instanceof Exception) {
                throw e2;
            } else {
                throw new Exception(e2);
            }
        }
    }

    public synchronized void reset() {
        mPfxInputStream = null;
        mPfxPassword = null;
        mKeyPair = null;
        mUserCert = null;
    }

    private synchronized void findKeyPair() throws Exception {
        String hexPubModules = null;
        X509Certificate tmpCert = null;

        Enumeration<String> aliases = mKeyStore.aliases();
        RSAPrivateKey privateKey = null;

        while(aliases.hasMoreElements()) {
            String alias = aliases.nextElement();

            try {
                Key key = mKeyStore.getKey(alias, mPfxPassword.toCharArray());

                if (key instanceof PrivateKey) {
                    privateKey = (RSAPrivateKey)mKeyStore.getKey(alias, mPfxPassword.toCharArray());
                    String hexPriModules = privateKey.getModulus().toString(16);

                    try {
                        tmpCert = (X509Certificate)mKeyStore.getCertificate(alias);
                        if (tmpCert != null) {
                            hexPubModules = ((RSAPublicKey)tmpCert.getPublicKey()).getModulus().toString(16);
                        } else {
                            hexPubModules = null;
                        }
                    } catch (Exception var12) {
                    }

                    if (hexPubModules != null && hexPubModules.equals(hexPriModules)) {
                        mAliasMap.put(alias, tmpCert.getNotBefore().getTime());
                        mKeyPairMap.put(tmpCert.getNotBefore().getTime(), new KeyPair(tmpCert.getPublicKey(), privateKey));
                        mKSCertMap.put(tmpCert.getNotBefore().getTime(), tmpCert);
                        hexPubModules = null;
                    }
                }
            } catch (Exception var13) {
                var13.printStackTrace();
                throw new Exception(var13);
            }
        }

        if (mKeyPairMap.size() <= 0) {
            throw new Exception("there is no key pairs!");
        } else {
            Iterator<Long> itor = mKeyPairMap.keySet().iterator();

            Long keyId;
            for(keyId = null; itor.hasNext(); keyId = itor.next()) {
            }

            mKeyPair = mKeyPairMap.get(keyId);
            mUserCert = mKSCertMap.get(keyId);

            String[] aliasArray = mAliasMap.keySet().toArray(new String[mAliasMap.values().size()]);
            String[] arrs = aliasArray;
            int lens = aliasArray.length;

            for(int ii = 0; ii < lens; ++ii) {
                String getAlias = arrs[ii];
                if ((mAliasMap.get(getAlias)).compareTo(keyId) == 0) {
                    mAlias = getAlias;
                    break;
                }
            }
        }
    }

    public synchronized PrivateKey getPrivateKey() {
        try {
            Key key = mKeyStore.getKey(mAlias, mPfxPassword.toCharArray());
            if (key instanceof PrivateKey) {
                PrivateKey pkey = (RSAPrivateKey) mKeyStore.getKey(mAlias, mPfxPassword.toCharArray());
                return pkey;
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return null;
    }

    public synchronized X509Certificate getX509Certificate() {
        return mUserCert;
    }
}
