/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.security.trustcertificate;


import android.app.Activity;
import android.content.Context;
import android.content.res.Configuration;
import android.graphics.Rect;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.ImageView;
import android.widget.TextView;

import com.foxit.sdk.PDFViewCtrl;
import com.foxit.uiextensions.R;
import com.foxit.uiextensions.UIExtensionsManager;
import com.foxit.uiextensions.browser.adapter.SuperAdapter;
import com.foxit.uiextensions.browser.adapter.viewholder.SuperViewHolder;
import com.foxit.uiextensions.controls.dialog.UIDialog;
import com.foxit.uiextensions.controls.dialog.UIPopoverWin;
import com.foxit.uiextensions.controls.dialog.UITextEditDialog;
import com.foxit.uiextensions.controls.dialog.sheetmenu.ISheetMenu;
import com.foxit.uiextensions.controls.dialog.sheetmenu.UISheetMenu;
import com.foxit.uiextensions.modules.panel.bean.BaseBean;
import com.foxit.uiextensions.security.certificate.CertificateFileInfo;
import com.foxit.uiextensions.theme.ThemeUtil;
import com.foxit.uiextensions.utils.AppDevice;
import com.foxit.uiextensions.utils.AppDisplay;
import com.foxit.uiextensions.utils.AppDmUtil;
import com.foxit.uiextensions.utils.AppResource;
import com.foxit.uiextensions.utils.Event;
import com.foxit.uiextensions.utils.IResult;
import com.foxit.uiextensions.utils.SystemUiHelper;

import java.util.ArrayList;
import java.util.List;

import androidx.annotation.NonNull;
import androidx.fragment.app.FragmentActivity;
import androidx.recyclerview.widget.RecyclerView;

import static com.foxit.uiextensions.utils.AppResource.getString;

public class TrustCertificateListAdapter extends SuperAdapter<CertificateFileInfo> {

    private PDFViewCtrl mPDFViewCtrl;
    private Context mContext;
    private TrustCertificatePresenter mPresenter;
    private ArrayList<CertificateFileInfo> mTrustCertInfos = new ArrayList<>();
    private View mNoContentView;
    private RecyclerView mRecyclerView;

    private int mIndex = -1;
    private ISheetMenu mSheetMenu;
    private UIExtensionsManager.ConfigurationChangedListener mConfigurationChangeListener;
    private View mSheetMenuAnchorView;

    public TrustCertificateListAdapter(Context context, PDFViewCtrl pdfViewCtrl) {
        super(context);
        this.mContext = context;
        this.mPDFViewCtrl = pdfViewCtrl;
        mPresenter = new TrustCertificatePresenter(context, pdfViewCtrl);
    }

    public void reset() {
        mIndex = -1;
    }

    public int getIndex() {
        return mIndex;
    }

    @Override
    public void notifyUpdateData() {
        notifyDataSetChanged();
    }

    @Override
    public CertificateFileInfo getDataItem(int position) {
        return mTrustCertInfos.get(position);
    }

    @NonNull
    @Override
    public SuperViewHolder onCreateViewHolder(@NonNull ViewGroup parent, int viewType) {
        SuperViewHolder viewHolder = new ItemViewHolder(LayoutInflater.from(mContext).inflate(R.layout.trust_certificate_list_item, parent, false));
        return viewHolder;
    }

    @Override
    public int getItemCount() {
        return mTrustCertInfos.size();
    }

    public void setNoContentView(View mNoContentView) {
        this.mNoContentView = mNoContentView;
    }

    class ItemViewHolder extends SuperViewHolder {
        private TextView mTvCertTitle;
        private TextView mTvCertDate;

        private ImageView mMore;
        private View mCertInfoView;
        private View mDeleteCertView;
        private ImageView mIvDetails;
        private TextView mTvDetails;
        private ImageView mIvDelete;
        private TextView mTvDelete;

        private View mMoreLayoutView;
        private View mCertContainer;

        public ItemViewHolder(View viewHolder) {
            super(viewHolder);
            mTvCertTitle = viewHolder.findViewById(R.id.panel_item_trust_certificate_title);
            mTvCertDate = viewHolder.findViewById(R.id.panel_item_trust_certificate_date);
            mMore = viewHolder.findViewById(R.id.panel_item_trust_cert_more);
            mMoreLayoutView = viewHolder.findViewById(R.id.trust_certificate_more_view);
            mCertInfoView = viewHolder.findViewById(R.id.rd_trust_certificate_detail);
            mDeleteCertView = viewHolder.findViewById(R.id.rd_delete_trust_certificate);
            mCertContainer = viewHolder.findViewById(R.id.rela_trust_cert);
            mIvDetails = viewHolder.findViewById(R.id.rd_trust_certificate_detail_icon);
            mTvDetails = viewHolder.findViewById(R.id.rd_trust_certificate_detail_text);
            mIvDelete = viewHolder.findViewById(R.id.rd_delete_trust_certificate_icon);
            mTvDelete = viewHolder.findViewById(R.id.rd_delete_trust_certificate_text);
            mMore.setOnClickListener(this);
            mDeleteCertView.setOnClickListener(this);
            mCertInfoView.setOnClickListener(this);
//            mMoreLayoutView.setOnClickListener(this);
            mCertContainer.setOnClickListener(this);
        }

        @Override
        public void bind(BaseBean data, int position) {
            CertificateFileInfo certInfo = (CertificateFileInfo) data;

            mTvCertTitle.setText(certInfo.subject);
            mTvCertDate.setText(AppDmUtil.formatStrDate(AppDmUtil.FORMAT_YYYY_MM_DD_HH_MM_SS, AppDmUtil.FORMAT_MMM_DD_YYYY_HH_MM_SS, certInfo.validTo));
//            int visibility = getAdapterPosition() == mIndex ? View.VISIBLE : View.GONE;
//            mMoreLayoutView.setVisibility(visibility);
            ThemeUtil.setTintList(mMore, ThemeUtil.getPrimaryIconColor(mContext));
            mTvCertTitle.setTextColor(mContext.getResources().getColor(R.color.t4));
            mTvCertDate.setTextColor(mContext.getResources().getColor(R.color.t3));
            mMoreLayoutView.setBackgroundColor(mContext.getResources().getColor(R.color.ux_bg_color_toolbar_light));
            mCertInfoView.setBackground(mContext.getResources().getDrawable(R.drawable.fb_item_func_layout_selector));
            mDeleteCertView.setBackground(mContext.getResources().getDrawable(R.drawable.fb_item_func_layout_selector));
            ThemeUtil.setTintList(mIvDetails, ThemeUtil.getPrimaryIconColor(mContext));
            ThemeUtil.setTintList(mIvDelete, ThemeUtil.getPrimaryIconColor(mContext));
            mTvDetails.setTextColor(mContext.getResources().getColor(R.color.hm_bottom_bar_text_color_selector));
            mTvDelete.setTextColor(mContext.getResources().getColor(R.color.hm_bottom_bar_text_color_selector));
        }

        @Override
        public void onClick(View v) {
            int id = v.getId();
            if (id == R.id.rd_trust_certificate_detail) {
//                ((LinearLayout) v.getParent()).setVisibility(View.GONE);
                viewCertInfo(getAdapterPosition());
            } else if (id == R.id.rd_delete_trust_certificate) {
//                ((LinearLayout) v.getParent()).setVisibility(View.GONE);
//                deleteTrustCert(getAdapterPosition());
            } else if (id == R.id.panel_item_trust_cert_more) {
//                int temp = mIndex;
                mIndex = getAdapterPosition();
//                notifyItemChanged(temp);
//                notifyItemChanged(mIndex);
                showActionMenu(v);
            } else if (id == R.id.rela_trust_cert) {
//                if (mIndex != -1) {
//                    int temp = mIndex;
//                    reset();
//                    notifyItemChanged(temp);
//                    return;
//                }
            }
        }
    }

    private void showActionMenu(View view) {
        mSheetMenuAnchorView = view;
        final UIExtensionsManager extensionsManager = (UIExtensionsManager) mPDFViewCtrl.getUIExtensionsManager();
        if (mSheetMenu == null) {
            mSheetMenu = UISheetMenu.newInstance((FragmentActivity) extensionsManager.getAttachedActivity());
            if (AppDisplay.isPad())
                mSheetMenu.setWidth(AppResource.getDimensionPixelSize(mContext, R.dimen.ux_pad_more_menu_width));
            mSheetMenu.setAutoResetSystemUiOnShow(false);
            mSheetMenu.addSheetItem(ISheetMenu.INFO_SIGNATURE_MENU);
            mSheetMenu.addSheetItem(ISheetMenu.DELETE_MENU);
            mSheetMenu.setSheetItemClickListener(new ISheetMenu.OnSheetItemClickListener() {
                @Override
                public void onItemClick(int type) {
                    switch (type) {
                        case ISheetMenu.INFO_SIGNATURE_MENU:
                            mSheetMenu.dismiss();
                            viewCertInfo(getIndex());
                            break;
                        case ISheetMenu.DELETE_MENU:
                            mSheetMenu.dismiss();
                            deleteTrustCert(getIndex());
                            break;
                        default:
                            break;
                    }
                }
            });
            mSheetMenu.setOnSheetDismissListener(new ISheetMenu.OnSheetDismissListener() {
                @Override
                public void onDismiss(ISheetMenu sheetMenu) {
                    if (mConfigurationChangeListener != null) {
                        extensionsManager.unregisterConfigurationChangedListener(mConfigurationChangeListener);
                        mConfigurationChangeListener = null;
                    }
                }
            });
        }
        Rect rect = getViewRect(view);
        if (mConfigurationChangeListener == null) {
            mConfigurationChangeListener = new UIExtensionsManager.ConfigurationChangedListener() {
                @Override
                public void onConfigurationChanged(Configuration newConfig) {
                    mPDFViewCtrl.postDelayed(new Runnable() {
                        @Override
                        public void run() {
                            mSheetMenu.show(extensionsManager.getRootView(), getViewRect(mSheetMenuAnchorView));
                        }
                    }, 100);
                }
            };
            extensionsManager.registerConfigurationChangedListener(mConfigurationChangeListener);
        }
        mSheetMenu.show(extensionsManager.getRootView(), rect, UIPopoverWin.ARROW_RIGHT, 0);
    }

    private Rect getViewRect(View view) {
        UIExtensionsManager extensionsManager = (UIExtensionsManager) mPDFViewCtrl.getUIExtensionsManager();
        Rect rect = new Rect();
        view.getGlobalVisibleRect(rect);

        int[] location = new int[2];
        view.getLocationOnScreen(location);
        int screen_x = location[0];
        int screen_y = location[1];
        if (AppDevice.isChromeOs(extensionsManager.getAttachedActivity())){
            extensionsManager.getRootView().getLocationOnScreen(location);
            screen_x -= location[0];
            screen_y -= location[1];
            rect.set(screen_x, screen_y, screen_x + rect.width(), screen_y + rect.height());
        }else {
            view.getLocationInWindow(location);
            int window_x = location[0];
            int window_y = location[1];
            rect.offset(screen_x - window_x, screen_y - window_y);
        }

        if (!SystemUiHelper.getInstance().isFullScreenMode(extensionsManager.getAttachedActivity())
                && SystemUiHelper.getInstance().isFullScreen())
            rect.offset(0, -SystemUiHelper.getInstance().getStatusBarHeight(extensionsManager.getAttachedActivity()));
        return rect;
    }

    public void loadData() {
        mPresenter.loadTrustCert(new IResult<List<CertificateFileInfo>, Object, Object>() {
            @Override
            public void onResult(boolean success, List<CertificateFileInfo> p1, Object p2, Object p3) {
                mIndex = -1;
                mTrustCertInfos.clear();
                boolean hasTrustCert = success && p1.size() > 0;
                if (hasTrustCert) {
                    mTrustCertInfos.addAll(p1);
                }
                showNoContentOrNot();
                notifyDataSetChanged();
            }
        });
    }

    private void viewCertInfo(int position) {
        mPresenter.viewCertInfo(mTrustCertInfos.get(position));
    }

    private void deleteTrustCert(final int position) {
        Activity activity = ((UIExtensionsManager) mPDFViewCtrl.getUIExtensionsManager()).getAttachedActivity();
        final UITextEditDialog dialog = new UITextEditDialog(activity, UIDialog.NO_INPUT);
        dialog.setTitle(getString(mContext, R.string.fx_string_delete));
        dialog.getPromptTextView().setText(getString(mContext, R.string.rv_remove_trust_certificate_prompt));
        dialog.getOKButton().setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                mPresenter.deleteTrustCert(mTrustCertInfos.get(position), new Event.Callback() {
                    @Override
                    public void result(Event event, boolean success) {
                        if (success) {
                            mIndex = -1;
                            mTrustCertInfos.remove(position);
                            showNoContentOrNot();
                            notifyItemRemoved(position);
                        }
                    }
                });
                dialog.dismiss();
            }
        });
        dialog.show();
    }

    @Override
    public void onAttachedToRecyclerView(@NonNull RecyclerView recyclerView) {
        super.onAttachedToRecyclerView(recyclerView);
        mRecyclerView = recyclerView;
    }

    private void showNoContentOrNot() {
        if (mNoContentView != null)
            mNoContentView.setVisibility(mTrustCertInfos.isEmpty() ? View.VISIBLE : View.GONE);
        mRecyclerView.setVisibility(mTrustCertInfos.isEmpty() ? View.INVISIBLE : View.VISIBLE);
    }

    public void addCert() {
        mPresenter.addTrustCert(new Event.Callback() {
            @Override
            public void result(Event event, boolean success) {
                if (success) {
                    loadData();
                } else {
                    if (event != null) {
                        int errorCode = event.mType;
                        if (errorCode == TrustCertificatePresenter.e_ErrTrustCertExisted) {
                            //
                        }
                    }
                }
            }
        });
    }
}
