/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.utils;

import android.content.Context;
import android.graphics.DashPathEffect;
import android.graphics.PathEffect;
import android.graphics.PointF;
import android.graphics.RectF;
import android.view.Gravity;
import android.view.LayoutInflater;
import android.view.MotionEvent;
import android.view.View;
import android.widget.TextView;
import android.widget.Toast;

import com.foxit.sdk.PDFException;
import com.foxit.sdk.PDFViewCtrl;
import com.foxit.sdk.addon.xfa.XFADoc;
import com.foxit.sdk.addon.xfa.XFAPage;
import com.foxit.sdk.common.Font;
import com.foxit.sdk.common.Library;
import com.foxit.sdk.common.fxcrt.Matrix2D;
import com.foxit.sdk.fdf.FDFDoc;
import com.foxit.sdk.pdf.PDFDoc;
import com.foxit.sdk.pdf.PDFPage;
import com.foxit.sdk.pdf.Signature;
import com.foxit.sdk.pdf.annots.Annot;
import com.foxit.sdk.pdf.annots.BorderInfo;
import com.foxit.sdk.pdf.annots.Caret;
import com.foxit.sdk.pdf.annots.DefaultAppearance;
import com.foxit.sdk.pdf.annots.FreeText;
import com.foxit.sdk.pdf.annots.Highlight;
import com.foxit.sdk.pdf.annots.Line;
import com.foxit.sdk.pdf.annots.Markup;
import com.foxit.sdk.pdf.annots.MarkupArray;
import com.foxit.sdk.pdf.annots.Note;
import com.foxit.sdk.pdf.annots.Redact;
import com.foxit.sdk.pdf.annots.RichMedia;
import com.foxit.sdk.pdf.annots.Screen;
import com.foxit.sdk.pdf.annots.Sound;
import com.foxit.sdk.pdf.annots.StrikeOut;
import com.foxit.sdk.pdf.annots.Widget;
import com.foxit.sdk.pdf.interform.Control;
import com.foxit.sdk.pdf.interform.Field;
import com.foxit.sdk.pdf.interform.Form;
import com.foxit.sdk.pdf.objects.PDFDictionary;
import com.foxit.sdk.pdf.objects.PDFObject;
import com.foxit.uiextensions.R;
import com.foxit.uiextensions.ToolHandler;
import com.foxit.uiextensions.annots.AnnotHandler;
import com.foxit.uiextensions.annots.line.LineConstants;
import com.foxit.uiextensions.annots.multimedia.screen.multimedia.MultimediaManager;
import com.foxit.uiextensions.annots.multiselect.GroupManager;
import com.foxit.uiextensions.annots.textmarkup.TextmarkupConstants;
import com.foxit.uiextensions.config.JsonConstants;

import java.util.ArrayList;

import androidx.annotation.NonNull;

public class AppAnnotUtil {
    public static float ANNOT_SELECT_TOLERANCE = 10.0f;
    private static AppAnnotUtil mAppAnnotUtil = null;
    private Context mContext;

    public static AppAnnotUtil getInstance(Context context) {
        if (mAppAnnotUtil == null) {
            mAppAnnotUtil = new AppAnnotUtil(context);
        }
        return mAppAnnotUtil;
    }

    public AppAnnotUtil(Context context) {
        mContext = context;
    }

    public static PathEffect getAnnotBBoxPathEffect() {
        return new DashPathEffect(new float[]{6, 2}, 0);
    }

    public static int getAnnotBBoxSpace() {
        return 5;
    }

    public float getAnnotBBoxStrokeWidth() {
        return AppDisplay.dp2px(1.0f);
    }

    private static PathEffect mPathEffect;

    public static PathEffect getBBoxPathEffect2() {
        if (mPathEffect == null) {
            mPathEffect = new DashPathEffect(new float[]{6.0f, 6.0f}, 0);
        }
        return mPathEffect;
    }

    public static void toastAnnotCopy(Context context) {
        UIToast.getInstance(context).show(context.getApplicationContext().getString(R.string.fm_annot_copy));
    }

    public static boolean isSupportReply(PDFViewCtrl pdfViewCtrl, Annot annot) {
        if (annot == null || annot.isEmpty())
            return false;

        if (GroupManager.getInstance().isGrouped(pdfViewCtrl, annot))
            return GroupManager.getInstance().canReply(pdfViewCtrl, annot);
        else
            return isAnnotSupportReply(annot) && !AppAnnotUtil.isReadOnly(annot);
    }

    public static boolean isAnnotSupportReply(Annot annot) {
        try {
            switch (annot.getType()) {
                case Annot.e_Note: {
                    Note note = (Note) annot;
                    if (note.isStateAnnot())
                        return false;
                }
                case Annot.e_Highlight:
                case Annot.e_Underline:
                case Annot.e_Squiggly:
                case Annot.e_StrikeOut:
                case Annot.e_Circle:
                case Annot.e_Square:
//                case Annot.e_Screen:
//                    return !isSupportGroupElement(annot);
                case Annot.e_Stamp:
                case Annot.e_Caret:
                case Annot.e_Line:
                case Annot.e_Ink:
                case Annot.e_Polygon:
                case Annot.e_PolyLine:
                case Annot.e_Redact:
                    return true;
                default:
                    return false;
            }
        } catch (PDFException e) {
            e.printStackTrace();
        }
        return false;
    }

    public static boolean isSupportComment(PDFViewCtrl pdfViewCtrl, Annot annot) {
        if (annot == null || annot.isEmpty())
            return false;

        if (GroupManager.getInstance().isGrouped(pdfViewCtrl, annot)) {
            return false;
        } else {
            return !isLocked(annot) && !isReadOnly(annot);
        }
    }

    public static String getTypeString(Annot annot) {
        try {
            switch (annot.getType()) {
                case Annot.e_Note:
                    return JsonConstants.TYPE_NOTE;
                case Annot.e_Link:
                    return JsonConstants.TYPE_LINK;
                case Annot.e_FreeText: {
                    String name;
                    String intent = ((FreeText) annot).getIntent();
                    if ("FreeTextCallout".equalsIgnoreCase(intent)) {
                        name = JsonConstants.TYPE_CALLOUT;
                    } else if ("FreeTextTypewriter".equalsIgnoreCase(intent)) {
                        name = JsonConstants.TYPE_TYPEWRITER;
                    } else {
                        name = JsonConstants.TYPE_TEXTBOX;
                    }
                    return name;
                }
                case Annot.e_Line: {
                    String intent = ((Line) annot).getIntent();
                    if (LineConstants.INTENT_LINE_ARROW.equals(intent)) {
                        return JsonConstants.TYPE_LINEARROW;
                    } else if (LineConstants.INTENT_LINE_DIMENSION.equals(intent)) {
                        return JsonConstants.TYPE_LINEDIMENSION;
                    } else {
                        return JsonConstants.TYPE_LINE;
                    }
                }
                case Annot.e_Square:
                    return JsonConstants.TYPE_SQUARE;
                case Annot.e_Circle:
                    return JsonConstants.TYPE_CIRCLE;
                case Annot.e_Polygon:
                    BorderInfo borderInfo = annot.getBorderInfo();
                    if (borderInfo != null && borderInfo.getStyle() == BorderInfo.e_Cloudy) {
                        return JsonConstants.TYPE_POLYGONCLOUD;
                    }
                    return JsonConstants.TYPE_POLYGON; //"PolyLineDimension"
                case Annot.e_PolyLine:
                    return JsonConstants.TYPE_POLYLINE;
                case Annot.e_Highlight:
                    if (annot.getDict().hasKey(TextmarkupConstants.AREA_HIGHLIGHT)
                            || TextmarkupConstants.AREA_HIGHLIGHT.equals(((Highlight) annot).getIntent()))
                        return JsonConstants.TYPE_AREA_HIGHLIGHT;
                    else
                        return JsonConstants.TYPE_HIGHLIGHT;
                case Annot.e_Underline:
                    return JsonConstants.TYPE_UNDERLINE;
                case Annot.e_Squiggly:
                    return JsonConstants.TYPE_SQUIGGLY;
                case Annot.e_StrikeOut:
                    return JsonConstants.TYPE_STRIKEOUT;
                case Annot.e_Stamp:
                    return JsonConstants.TYPE_STAMP;
                case Annot.e_Caret:
                    return isReplaceCaret(annot) ? JsonConstants.TYPE_REPLACE : JsonConstants.TYPE_CARET;
                case Annot.e_Ink:
                    return JsonConstants.TYPE_INK;
                case Annot.e_PSInk:
                    return JsonConstants.TYPE_PSINK;
                case Annot.e_FileAttachment:
                    return JsonConstants.TYPE_ATTACHMENT;
                case Annot.e_Sound:
                    return JsonConstants.TYPE_SOUND;
                case Annot.e_Movie:
                    return JsonConstants.TYPE_MOVIE;
                case Annot.e_Widget:
                    return JsonConstants.TYPE_WIDGET;
                case Annot.e_Screen:
                case Annot.e_RichMedia:
                    return MultimediaManager.getInstance().getTypeString(annot);
                case Annot.e_PrinterMark:
                    return JsonConstants.TYPE_PRINTER_MARK;
                case Annot.e_TrapNet:
                    return JsonConstants.TYPE_TRAPNET;
                case Annot.e_Watermark:
                    return JsonConstants.TYPE_WATERMARK;
                case Annot.e_3D:
                    return JsonConstants.TYPE_3D;
                case Annot.e_Redact:
                    return JsonConstants.TYPE_REDACTION;
                case Annot.e_Popup:
                    return JsonConstants.TYPE_POPUP;
                default:
                    return JsonConstants.TYPE_UNKNOWN;
            }
        } catch (PDFException e) {
            e.printStackTrace();
        }
        return JsonConstants.TYPE_UNKNOWN;
    }

    public static String getTypeToolName(Annot annot) {
        try {
            switch (annot.getType()) {
                case Annot.e_Line: {
                    String intent = ((Line) annot).getIntent();
                    if ("LineArrow".equals(intent))
                        return ToolHandler.TH_TYPE_ARROW;
                    if ("LineDimension".equals(intent))
                        return ToolHandler.TH_TYPE_DISTANCE;
                    return ToolHandler.TH_TYPE_LINE;
                }
                case Annot.e_Polygon:
                    BorderInfo borderInfo = annot.getBorderInfo();
                    if (borderInfo != null && borderInfo.getStyle() == BorderInfo.e_Cloudy) {
                        return ToolHandler.TH_TYPE_POLYGONCLOUD;
                    }
                    return ToolHandler.TH_TYPE_POLYGON;
                case Annot.e_Caret:
                    return isReplaceCaret(annot) ? ToolHandler.TH_TYPE_REPLACE : ToolHandler.TH_TYPR_INSERTTEXT;
                case Annot.e_Screen:
                    String typeString = MultimediaManager.getInstance().getTypeString(annot);
                    if (JsonConstants.TYPE_SCREEN_IMAGE.equals(typeString)) {
                        return ToolHandler.TH_TYPE_PDFIMAGE;
                    } else if (JsonConstants.TYPE_AUDIO.equals(typeString)) {
                        return ToolHandler.TH_TYPE_SCREEN_AUDIO;
                    } else {
                        return ToolHandler.TH_TYPE_SCREEN_VIDEO;
                    }
                case Annot.e_RichMedia:
                     typeString = MultimediaManager.getInstance().getTypeString(annot);
                    if (JsonConstants.TYPE_RICH_MEDIA.equals(typeString)) {
                        return ToolHandler.TH_TYPE_SCREEN_AUDIO;
                    }
                default:
                    return "Unknown";
            }
        } catch (PDFException e) {
            e.printStackTrace();
        }
        return "Unknown";
    }

    public static boolean contentsModifiable(PDFViewCtrl pdfViewCtrl, Annot annot) {
        if (annot == null || annot.isEmpty())
            return false;

        if (GroupManager.getInstance().isGrouped(pdfViewCtrl, annot))
            return GroupManager.getInstance().contentsModifiable(pdfViewCtrl, annot);
        else
            return contentsModifiable(getTypeString(annot));
    }

    public static boolean contentsModifiable(String type) {
        return JsonConstants.TYPE_NOTE.equals(type)
                || JsonConstants.TYPE_LINE.equals(type)
                || JsonConstants.TYPE_LINEARROW.equals(type)
                || JsonConstants.TYPE_LINEDIMENSION.equals(type)
                || JsonConstants.TYPE_SQUARE.equals(type)
                || JsonConstants.TYPE_CIRCLE.equals(type)
                || JsonConstants.TYPE_HIGHLIGHT.equals(type)
                || JsonConstants.TYPE_AREA_HIGHLIGHT.equals(type)
                || JsonConstants.TYPE_UNDERLINE.equals(type)
                || JsonConstants.TYPE_SQUIGGLY.equals(type)
                || JsonConstants.TYPE_STRIKEOUT.equals(type)
                || JsonConstants.TYPE_STAMP.equals(type)
                || JsonConstants.TYPE_CARET.equals(type)
                || JsonConstants.TYPE_REPLACE.equals(type)
                || JsonConstants.TYPE_INK.equals(type)
//                || "Screen".equals(type)
                || JsonConstants.TYPE_POLYGON.equals(type)
                || JsonConstants.TYPE_POLYGONCLOUD.equals(type)
                || JsonConstants.TYPE_POLYLINE.equals(type)
                || JsonConstants.TYPE_REDACTION.equals(type);
    }

    public static Annot getAnnot(PDFPage page, String UID) {
        if (page == null) return null;
        try {
            long nCount = page.getAnnotCount();
            for (int i = 0; i < nCount; i++) {
                Annot annot = AppAnnotUtil.createAnnot(page.getAnnot(i));
                if (annot != null) {
                    if (AppUtil.isEmpty(annot.getUniqueID())) {
                        if (annot.getDict() != null && String.valueOf(annot.getDict().getObjNum()).compareTo(UID) == 0) {
                            return annot;
                        }
                    } else if (annot.getUniqueID().compareTo(UID) == 0) {
                        return annot;
                    }
                }
            }
        } catch (PDFException e) {
            e.printStackTrace();
        }
        return null;
    }

    public static Annot getAnnot(PDFPage page, int objNum) {
        if (page == null) return null;
        try {
            long nCount = page.getAnnotCount();
            for (int i = 0; i < nCount; i++) {
                Annot annot = AppAnnotUtil.createAnnot(page.getAnnot(i));
                if (annot != null && annot.getDict() != null) {
                    if (annot.getDict().getObjNum() == objNum) {
                        return annot;
                    }
                }
            }
        } catch (PDFException e) {
            e.printStackTrace();
        }
        return null;
    }

    public static Control getControlAtPos(PDFPage page, PointF point, float tolerance) throws PDFException {
        Annot annot = AppAnnotUtil.createAnnot(page.getAnnotAtPoint(AppUtil.toFxPointF(point), tolerance));
        if (annot != null && annot.getType() == Annot.e_Widget) {
            return ((Widget) annot).getControl();
        }
        return null;
    }


    public static Signature getSignatureAtPos(PDFPage page, PointF point, float tolerance) throws PDFException {
        Annot annot = AppAnnotUtil.createAnnot(page.getAnnotAtPoint(AppUtil.toFxPointF(point), tolerance));
        if (annot != null && annot.getType() == Annot.e_Widget) {
            Field field = ((Widget) annot).getField();
            if (field != null && field.getType() == Field.e_TypeSignature)
                return (Signature) field;
        }
        return null;
    }

    public static boolean isSameAnnot(Annot annot, Annot comparedAnnot) {
        boolean ret = false;
        try {
            long objNumA = 0;
            if (annot != null && !annot.isEmpty())
                objNumA = annot.getDict().getObjNum();
            long objNumB = 0;
            if (comparedAnnot != null && !comparedAnnot.isEmpty())
                objNumB = comparedAnnot.getDict().getObjNum();
            if (objNumA == objNumB)
                ret = true;
        } catch (PDFException e) {
            e.printStackTrace();
        }
        return ret;
    }

    public static boolean isSameAnnot(int objNumA, int  objNumB) {
        return objNumA == objNumB;
    }

    // if the annot in a support group by rdk and the annot is not the header of group  return true, otherwise return false.
    public static boolean isSupportGroupElement(Annot annot) {
        if (!isSupportGroup(annot))
            return false;
        try {
            return !isSameAnnot(annot, ((Markup) annot).getGroupHeader());
        } catch (PDFException e) {
            e.printStackTrace();
        }
        return false;
    }

    public static boolean isSupportGroup(Annot annot) {
        if (annot == null || annot.isEmpty()) return false;
        try {
            if (!annot.isMarkup() || !((Markup) annot).isGrouped()) return false;
            Markup head = ((Markup) annot).getGroupHeader();

            //now just support replace annot (Caret, StikeOut)
            switch (head.getType()) {
                case Annot.e_Caret:
                    return isReplaceCaret(head);
                default:
                    return false;
            }

        } catch (PDFException e) {
            e.printStackTrace();
        }
        return false;
    }

    public static boolean isGrouped(Annot annot) {
        if (annot == null || annot.isEmpty()) return false;
        try {
            return annot.isMarkup() && ((Markup) annot).isGrouped();
        } catch (PDFException e) {
            e.printStackTrace();
        }
        return false;
    }

    public static boolean isSupportAnnotGroup(Annot annot) {
        int annotType = Annot.e_UnknownType;
        try {
            annotType = annot.getType();
        } catch (PDFException e) {
            e.printStackTrace();
        }
        return Annot.e_Note == annotType
                || Annot.e_FileAttachment == annotType
                || Annot.e_Stamp == annotType
                || Annot.e_FreeText == annotType
                || Annot.e_Line == annotType
                || Annot.e_Square == annotType
                || Annot.e_Circle == annotType
                || Annot.e_Polygon == annotType
                || Annot.e_PolyLine == annotType
                || Annot.e_Ink == annotType;
    }

    public static boolean isReplaceCaret(Annot annot) {
        try {
            if (annot == null || annot.getType() != Annot.e_Caret || !((Markup) annot).isGrouped())
                return false;
            Caret caret = (Caret) AppAnnotUtil.createAnnot(annot, Annot.e_Caret);
            Markup head = caret.getGroupHeader();
            MarkupArray markupArray = head.getGroupElements();
            if (head.getType() != Annot.e_Caret || markupArray.getSize() != 2 || !isSameAnnot(head, caret))
                return false;
            for (int i = 0; i < 2; i++) {
                Markup markup = markupArray.getAt(i);//caret.getGroupElement(i);
                if (markup.getType() == Annot.e_StrikeOut) {
                    return true;
                }
            }
        } catch (PDFException e) {
            e.printStackTrace();
        }
        return false;
    }

    public static StrikeOut getStrikeOutFromCaret(@NonNull Caret caret) {
        if (caret.isEmpty()) return null;
        try {
            MarkupArray markupArray = caret.getGroupElements();
            long nCount = markupArray.getSize();
            for (int i = 0; i < nCount; i++) {
                Markup groupAnnot = markupArray.getAt(i);
                if (groupAnnot.getType() == Annot.e_StrikeOut)
                    return (StrikeOut) AppAnnotUtil.createAnnot(groupAnnot, Annot.e_StrikeOut);
            }
        } catch (PDFException e) {
            e.printStackTrace();
        }

        return null;
    }

    public static Annot getReplyToAnnot(Annot annot) {
        if (annot == null || annot.isEmpty())
            return null;
        try {
            if (annot.getType() == Annot.e_Note)
                return ((Note) annot).getReplyTo();
        } catch (PDFException e) {
            e.printStackTrace();
        }
        return null;

    }

    public static PointF getPageViewPoint(PDFViewCtrl pdfViewCtrl, int pageIndex, MotionEvent motionEvent) {
        PointF devPt = new PointF(motionEvent.getX(), motionEvent.getY());
        PointF point = new PointF();
        pdfViewCtrl.convertDisplayViewPtToPageViewPt(devPt, point, pageIndex);
        return point;
    }

    public static PointF getPdfPoint(PDFViewCtrl pdfViewCtrl, int pageIndex, MotionEvent motionEvent) {
        PointF devPt = new PointF(motionEvent.getX(), motionEvent.getY());
        PointF pageViewPt = new PointF();
        pdfViewCtrl.convertDisplayViewPtToPageViewPt(devPt, pageViewPt, pageIndex);
        PointF point = new PointF();
        pdfViewCtrl.convertPageViewPtToPdfPt(pageViewPt, point, pageIndex);
        return point;
    }

    public static int getIconId(String type) {
        int resId;
        if (JsonConstants.TYPE_NOTE.equals(type)) {
            resId = R.drawable.comment_tool_note_bg;
        } else if (JsonConstants.TYPE_ATTACHMENT.equals(type)) {
            resId = R.drawable.comment_tool_attachment_bg;
        } else if (JsonConstants.TYPE_STAMP.equals(type)) {
            resId = R.drawable.comment_tool_stamp_bg;
        } else if (JsonConstants.TYPE_HIGHLIGHT.equals(type)) {
            resId = R.drawable.comment_tool_highlight_bg;
        } else if (JsonConstants.TYPE_AREA_HIGHLIGHT.equals(type)) {
            resId = R.drawable.comment_tool_area_highlight_bg;
        } else if (JsonConstants.TYPE_STRIKEOUT.equals(type)) {
            resId = R.drawable.comment_tool_strikeout_bg;
        } else if (JsonConstants.TYPE_UNDERLINE.equals(type)) {
            resId = R.drawable.comment_tool_underline_bg;
        } else if (JsonConstants.TYPE_SQUIGGLY.equals(type)) {
            resId = R.drawable.comment_tool_squiggly_bg;
        } else if (JsonConstants.TYPE_CARET.equals(type)) {
            resId = R.drawable.comment_tool_insert_text_bg;
        } else if (JsonConstants.TYPE_REPLACE.equals(type)) {
            resId = R.drawable.comment_tool_replace_text_bg;
        } else if (JsonConstants.TYPE_TYPEWRITER.equals(type)) {
            resId = R.drawable.comment_tool_typewriter_bg;
        } else if (JsonConstants.TYPE_CALLOUT.equals(type)) {
            resId = R.drawable.comment_tool_callout_bg;
        } else if (JsonConstants.TYPE_TEXTBOX.equals(type)) {
            resId = R.drawable.comment_tool_textbox_bg;
        } else if (JsonConstants.TYPE_LINEDIMENSION.equals(type)) {
            resId = R.drawable.comment_tool_line_measure_bg;
        } else if (JsonConstants.TYPE_POLYGON.equals(type)) {
            resId = R.drawable.drawing_tool_polygon;
        } else if (JsonConstants.TYPE_POLYGONCLOUD.equals(type)) {
            resId = R.drawable.drawing_tool_polygon_cloud;
        } else if (JsonConstants.TYPE_POLYLINE.equals(type)) {
            resId = R.drawable.drawing_tool_polygonline;
        } else if (JsonConstants.TYPE_CIRCLE.equals(type)) {
            resId = R.drawable.drawing_tool_oval;
        } else if (JsonConstants.TYPE_SQUARE.equals(type)) {
            resId = R.drawable.drawing_tool_square;
        } else if (JsonConstants.TYPE_INK.equals(type)) {
            resId = R.drawable.drawing_tool_pencil_bg;
        } else if (JsonConstants.TYPE_LINE.equals(type)) {
            resId = R.drawable.drawing_tool_line;
        } else if (JsonConstants.TYPE_LINEARROW.equals(type)) {
            resId = R.drawable.drawing_tool_line_arrow;
        } else if (JsonConstants.TYPE_SCREEN_IMAGE.equals(type)) {
            resId = R.drawable.comment_tool_image_bg;
        } else if (JsonConstants.TYPE_AUDIO.equals(type)) {
            resId = R.drawable.edit_tool_audio;
        } else if (JsonConstants.TYPE_VIDEO.equals(type)) {
            resId = R.drawable.edit_tool_video;
        } else if (JsonConstants.TYPE_SOUND.equals(type)) {
            resId = R.drawable.comment_tool_sound_type;
        } else if (JsonConstants.TYPE_REDACTION.equals(type)) {
            resId = R.drawable.comment_tool_redact_type;
        } else {
            resId = R.drawable.rv_panel_annot_not_edit_type;
        }
        return resId;
    }

    public static boolean isSupportDelete(PDFViewCtrl pdfViewCtrl, Annot annot) {
        if (annot == null || annot.isEmpty())
            return false;

        if (GroupManager.getInstance().isGrouped(pdfViewCtrl, annot))
            return GroupManager.getInstance().canDelete(pdfViewCtrl, annot);
        else
            return isSupportDeleteAnnot(annot) && !(isLocked(annot) || isReadOnly(annot));
    }

    public static boolean isSupportDeleteAnnot(Annot annot) {
        try {
            switch (annot.getType()) {
                case Annot.e_Sound:
                    return false;
                default:
                    return true;
            }
        } catch (PDFException e) {
            e.printStackTrace();
        }
        return true;
    }

    public static boolean isSupportEditAnnot(Annot annot) {
        if (annot == null || annot.isEmpty())
            return false;
        try {
            switch (annot.getType()) {
                case Annot.e_Note: {
                    Note note = (Note) annot;
                    if (note.isStateAnnot())
                        return false;
                }

                case Annot.e_Highlight:
                case Annot.e_Underline:
                case Annot.e_Squiggly:
                case Annot.e_StrikeOut:
                case Annot.e_Circle:
                case Annot.e_Square:
                    return !isSupportGroupElement(annot);
                case Annot.e_FreeText:
//                    String intent = ((Markup)annot).getIntent();
//                    return intent == null || "FreeTextTypewriter".equals(intent);
                case Annot.e_Stamp:
                case Annot.e_Caret:
                case Annot.e_Line:
                case Annot.e_Ink:
                case Annot.e_FileAttachment:
                case Annot.e_Polygon:
                case Annot.e_Redact:
                case Annot.e_PolyLine:
                case Annot.e_Sound:
                    return !isSupportGroupElement(annot);
                case Annot.e_Screen:
                    String typeString = MultimediaManager.getInstance().getTypeString(annot);
                    return !(JsonConstants.TYPE_AUDIO.equals(typeString)
                            || JsonConstants.TYPE_VIDEO.equals(typeString));
                default:
                    return false;
            }
        } catch (PDFException e) {
            e.printStackTrace();
        }
        return false;
    }

    private Toast mAnnotToast;

    /**
     * Only for annot continue create toast
     */
    public void showAnnotContinueCreateToast(boolean isContinuousCreate) {
        if (mAnnotToast == null) {
            initAnnotToast();
        }
        if (mAnnotToast == null) {
            return;
        }
        String str;
        if (isContinuousCreate) {
            str = AppResource.getString(mContext.getApplicationContext(), R.string.annot_continue_create);
        } else {
            str = AppResource.getString(mContext.getApplicationContext(), R.string.annot_single_create);
        }
        TextView tv = (TextView) mAnnotToast.getView().findViewById(R.id.annot_continue_create_toast_tv);
        int yOffset;
        if (AppDisplay.isPad()) {
            yOffset = AppResource.getDimensionPixelSize(mContext, R.dimen.ux_toolbar_height_pad) + AppDisplay.dp2px(16) * (2 + 1);
        } else {
            yOffset = AppResource.getDimensionPixelSize(mContext, R.dimen.ux_toolbar_height_phone) + AppDisplay.dp2px(16) * (2 + 1);
        }
        mAnnotToast.setGravity(Gravity.BOTTOM, 0, yOffset);
        tv.setText(str);
        mAnnotToast.show();
    }

    private void initAnnotToast() {
        try {
            mAnnotToast = new Toast(mContext);
            LayoutInflater inflate = (LayoutInflater) mContext.getSystemService(Context.LAYOUT_INFLATER_SERVICE);
            View toastlayout = inflate.inflate(R.layout.annot_continue_create_tips, null);
            mAnnotToast.setView(toastlayout);
            mAnnotToast.setDuration(Toast.LENGTH_SHORT);
            int yOffset;
            if (AppDisplay.isPad()) {
                yOffset = AppResource.getDimensionPixelSize(mContext, R.dimen.ux_toolbar_height_pad) + AppDisplay.dp2px(16) * (2 + 1);
            } else {
                yOffset = AppResource.getDimensionPixelSize(mContext, R.dimen.ux_toolbar_height_phone) + AppDisplay.dp2px(16) * (2 + 1);
            }
            mAnnotToast.setGravity(Gravity.BOTTOM, 0, yOffset);
        } catch (Exception e) {
            mAnnotToast = null;
        }
    }

    public static int getAnnotHandlerType(Annot annot) {
        if (annot == null || annot.isEmpty()) return Annot.e_UnknownType;
        int type = Annot.e_UnknownType;
        try {
            type = annot.getType();
            if (type == Annot.e_FreeText) {
                String intent = ((FreeText) annot).getIntent();
                if ("FreeTextCallout".equalsIgnoreCase(intent)) {
                    type = AnnotHandler.TYPE_FREETEXT_CALLOUT; // FreeTextCallout annot handler type
                } else if (!"FreeTextTypewriter".equalsIgnoreCase(intent)) {
                    type = AnnotHandler.TYPE_FREETEXT_TEXTBOX; // text box;
                }
            } else if (type == Annot.e_Widget) {
                Field field = ((Widget) annot).getField();
                if (field != null) {
                    int ft = field.getType();
                    if (ft == Field.e_TypeSignature) {
                        type = AnnotHandler.TYPE_FORMFIELD_SIGNATURE;//signature handle type
                    }
                }
            } else if (type == Annot.e_Screen || type == Annot.e_RichMedia) {
                String typeString = MultimediaManager.getInstance().getTypeString(annot);
                if (JsonConstants.TYPE_SCREEN_IMAGE.equals(typeString)) {
                    type = AnnotHandler.TYPE_SCREEN_IMAGE; // Screen -Image;
                } else {
                    type = AnnotHandler.TYPE_SCREEN_MULTIMEDIA; // Screen - Multimedia
                }
            }
        } catch (PDFException e) {
            e.printStackTrace();
        }

        return type;
    }

    public static Annot createAnnot(Annot annot) {
        if (annot == null || annot.isEmpty()) return null;
        try {
            return createAnnot(annot, annot.getType());
        } catch (PDFException e) {
            e.printStackTrace();
        }
        return null;
    }

    public static Annot createAnnot(Annot annot, int type) {
        if (annot == null || annot.isEmpty()) return null;
        Annot object = null;
        switch (type) {
            case Annot.e_Note:
                object = new com.foxit.sdk.pdf.annots.Note(annot);
                break;
            case Annot.e_Highlight:
                object = new com.foxit.sdk.pdf.annots.Highlight(annot);
                break;
            case Annot.e_Underline:
                object = new com.foxit.sdk.pdf.annots.Underline(annot);
                break;
            case Annot.e_StrikeOut:
                object = new com.foxit.sdk.pdf.annots.StrikeOut(annot);
                break;
            case Annot.e_Squiggly:
                object = new com.foxit.sdk.pdf.annots.Squiggly(annot);
                break;
            case Annot.e_Link:
                object = new com.foxit.sdk.pdf.annots.Link(annot);
                break;
            case Annot.e_Circle:
                object = new com.foxit.sdk.pdf.annots.Circle(annot);
                break;
            case Annot.e_Square:
                object = new com.foxit.sdk.pdf.annots.Square(annot);
                break;
            case Annot.e_FreeText:
                object = new com.foxit.sdk.pdf.annots.FreeText(annot);
                break;
            case Annot.e_Line:
                object = new com.foxit.sdk.pdf.annots.Line(annot);
                break;
            case Annot.e_Ink:
                object = new com.foxit.sdk.pdf.annots.Ink(annot);
                break;
            case Annot.e_Caret:
                object = new com.foxit.sdk.pdf.annots.Caret(annot);
                break;
            case Annot.e_Polygon:
                object = new com.foxit.sdk.pdf.annots.Polygon(annot);
                break;
            case Annot.e_PolyLine:
                object = new com.foxit.sdk.pdf.annots.PolyLine(annot);
                break;
            case Annot.e_Stamp:
                object = new com.foxit.sdk.pdf.annots.Stamp(annot);
                break;
            case Annot.e_Popup:
                object = new com.foxit.sdk.pdf.annots.Popup(annot);
                break;
            case Annot.e_PSInk:
                object = new com.foxit.sdk.pdf.annots.PSInk(annot);
                break;
            case Annot.e_FileAttachment:
                object = new com.foxit.sdk.pdf.annots.FileAttachment(annot);
                break;
            case Annot.e_Widget:
                object = new Widget(annot);
                break;
            case Annot.e_Screen:
                object = new Screen(annot);
                break;
            case Annot.e_Redact:
                object = new Redact(annot);
                break;
            case Annot.e_Sound:
                object = new Sound(annot);
                break;
            case Annot.e_RichMedia:
                object = new RichMedia(annot);
                break;
            default:
                try {
                    if (annot.isMarkup())
                        object = new Markup(annot);
                } catch (PDFException e) {
                    e.printStackTrace();
                }
                break;
        }
        return object;
    }

    public static boolean equals(Annot annot, Annot other) {

        try {
            if (annot == null || annot.isEmpty() || other == null || other.isEmpty()) return false;
            if (annot.getIndex() == other.getIndex() && annot.getPage().getIndex() == other.getPage().getIndex()) {
                return true;
            }

        } catch (PDFException e) {
            e.printStackTrace();
        }
        return false;
    }

    public static boolean isLocked(Annot annot) {
        try {
            if (annot == null || annot.isEmpty()) return false;
            return (annot.getFlags() & Annot.e_FlagLocked) != 0;
        } catch (PDFException e) {
            e.printStackTrace();
        }
        return false;
    }

    public static boolean isReadOnly(Annot annot) {
        try {
            if (annot == null || annot.isEmpty()) return false;
            return (annot.getFlags() & Annot.e_FlagReadOnly) != 0;
        } catch (PDFException e) {
            e.printStackTrace();
        }
        return false;
    }

    public static void convertPageViewRectToPdfRect(PDFViewCtrl pdfViewCtrl, Annot annot, RectF pvRect, RectF pdfRect) {
        try {
            com.foxit.sdk.common.fxcrt.RectF _pdfRect = AppUtil.toFxRectF(pvRect);
            PDFPage pdfPage = annot.getPage();
            int pageIndex = pdfPage.getIndex();
            Matrix2D matrix2D = annot.getDisplayMatrix(AppUtil.toMatrix2D(pdfViewCtrl.getDisplayMatrix(pageIndex)));
            Matrix2D pdfMatrix = new Matrix2D();
            pdfMatrix.setReverse(matrix2D);
            pdfMatrix.transformRect(_pdfRect);
            pdfRect.set(AppUtil.toRectF(_pdfRect));
            int flag = annot.getFlags();
//          if ((flag & Annot.e_FlagNoRotate) != 0 || (flag & Annot.e_FlagNoZoom) != 0) {
            RectF _pvRect = new RectF(pvRect);
            pdfViewCtrl.convertPageViewRectToPdfRect(_pvRect, _pvRect, pageIndex);
            int rotation = (pdfViewCtrl.getViewRotation() + pdfPage.getRotation()) % 4;
            float width = Math.abs(pdfRect.width());
            float height = Math.abs(pdfRect.height());
            if ((flag & Annot.e_FlagNoZoom) != 0) {
                RectF annotRect = AppUtil.toRectF(annot.getRect());
                width = Math.abs(annotRect.width());
                height = Math.abs(annotRect.height());
            }

            switch (rotation) {
                case com.foxit.sdk.common.Constants.e_Rotation0:
                    pdfRect.set(_pvRect.left, _pvRect.top, _pvRect.left + width, _pvRect.top - height);
                    break;
                case com.foxit.sdk.common.Constants.e_Rotation90:
                    pdfRect.set(_pvRect.left, _pvRect.bottom, _pvRect.left + width, _pvRect.bottom - height);
                    break;
                case com.foxit.sdk.common.Constants.e_Rotation180:
                    pdfRect.set(_pvRect.right, _pvRect.bottom, _pvRect.right + width, _pvRect.bottom - height);
                    break;
                case com.foxit.sdk.common.Constants.e_Rotation270:
                    pdfRect.set(_pvRect.right, _pvRect.top, _pvRect.right + width, _pvRect.top - height);
                    break;
                default:
            }
//          }
        } catch (PDFException e) {
            e.printStackTrace();
        }
    }

    public static boolean hasModuleLicenseRight(int module) {
        try {
            return Library.hasModuleLicenseRight(module);
        } catch (PDFException e) {
            e.printStackTrace();
        }
        return false;
    }

    public static ArrayList<Annot> getAnnotsByNMs(PDFPage page, ArrayList<String> nms) {
        ArrayList<Annot> annots = new ArrayList<>();
        for (int i = 0; i < nms.size(); i++) {
            Annot annot = getAnnot(page, nms.get(i));
            if (annot == null || annot.isEmpty())
                continue;
            annots.add(annot);
        }
        return annots;
    }

    public static String getAnnotUniqueID(Annot annot) {
        if (annot == null || annot.isEmpty())
            return "";

        try {
            String uniqueID = annot.getUniqueID();
            if (AppUtil.isEmpty(uniqueID)) {
                if (annot.getDict() != null) {
                    int objNum = annot.getDict().getObjNum();
                    uniqueID = String.valueOf(objNum);
                } else {
                    uniqueID = AppDmUtil.randomUUID(null);
                    annot.setUniqueID(uniqueID);
                }
            }
            return uniqueID;
        } catch (PDFException e) {
            e.printStackTrace();
        }
        return "";
    }

    public static int getAnnotObjNum(Annot annot) {
        if (annot == null || annot.isEmpty())
            return -1;

        try {
            if (annot.getDict() != null) {
                return annot.getDict().getObjNum();
            }
        } catch (PDFException e) {
            e.printStackTrace();
        }
        return -1;
    }

    public static int getStandard14Font(DefaultAppearance da, PDFDoc doc) {
        int id = Font.e_StdIDCourier;
        try {
            Font font = da != null ? da.getFont() : null;
            if (font != null && !font.isEmpty())
                id = font.getStandard14Font(doc);
        } catch (PDFException e) {
            e.printStackTrace();
        }
        return id;
    }

    public static PDFDictionary clonePDFDict(PDFDictionary dict) {
        try {
            PDFDictionary cloneDict = PDFDictionary.create();
            if (cloneDict == null) return null;

            long pos = dict.moveNext(0);
            while (pos != 0) {
                String key = dict.getKey(pos);
                PDFObject object = dict.getValue(pos);
                if (object != null)
                    cloneDict.setAt(key, object.cloneObject());
                pos = dict.moveNext(pos);
            }
            return cloneDict;
        } catch (PDFException e) {
            e.printStackTrace();
        }
        return null;
    }

    public static boolean resetPDFDict(Annot annot, PDFDictionary dict) {
        return resetPDFDict(annot, dict, false);
    }

    public static boolean resetPDFDict(Annot annot, PDFDictionary dict, boolean clone) {
        try {
            if (dict == null) return false;

            PDFDictionary annotDict = annot.getDict();
            long pos = dict.moveNext(0);
            while (pos != 0) {
                String key = dict.getKey(pos);
                if ("Popup".equals(key)) {
                    pos = dict.moveNext(pos);
                    continue;
                }

                PDFObject object = dict.getValue(pos);
                if (object != null) {
                    if (clone) {
                        annotDict.setAt(key, object.cloneObject());
                    } else {
                        annotDict.setAt(key, object);
                    }
                }
                pos = dict.moveNext(pos);
            }
            return true;
        } catch (PDFException e) {
            e.printStackTrace();
        }
        return false;
    }

    public static boolean noAnnotation(PDFDoc doc) {
        boolean result = true;
        if (doc == null)return result;
        try {
            int pageCount = doc.getPageCount();
            int annotationCount = 0;
            for (int i = 0; i < pageCount; i++) {
                annotationCount += doc.getPage(i).getAnnotCount();
                if (annotationCount > 0)
                    return false;
            }
        } catch (PDFException e) {
            e.printStackTrace();
        }
        return result;
    }

    public static int getAnnotationCount(PDFDoc doc) {
        int annotationCount = 0;
        if (doc == null)return annotationCount;
        try {
            int pageCount = doc.getPageCount();
            for (int i = 0; i < pageCount; i++) {
                annotationCount += doc.getPage(i).getAnnotCount();
            }
            return annotationCount;
        } catch (PDFException e) {
            e.printStackTrace();
        }
        return annotationCount;
    }

    public static boolean noRedaction(PDFDoc doc){
        boolean result = true;
        if (doc == null)return result;
        int count;
        PDFPage page;
        try {
            int pageCount = doc.getPageCount();
            for (int i = 0; i < pageCount; i++) {
                page = doc.getPage(i);
                count = page.getAnnotCount();
                if (count == 0){
                    continue;
                }
                for (int j = 0; j < count; j++) {
                    Annot annot = page.getAnnot(j);
                    if (annot.getType() == Annot.e_Redact){
                        return false;
                    }
                }
            }
        } catch (PDFException e) {
            e.printStackTrace();
        }
        return result;
    }

    public static boolean noComment(PDFDoc doc){
        boolean result = true;
        if (doc == null)return result;
        int count;
        PDFPage page;
        ArrayList<Integer> excludedTypeList = new ArrayList<>();
        excludedTypeList.add(Annot.e_Link);
        excludedTypeList.add(Annot.e_PSInk);
        excludedTypeList.add(Annot.e_Sound);
        excludedTypeList.add(Annot.e_Movie);
        excludedTypeList.add(Annot.e_Widget);
        excludedTypeList.add(Annot.e_Screen);
        excludedTypeList.add(Annot.e_PrinterMark);
        excludedTypeList.add(Annot.e_TrapNet);
        excludedTypeList.add(Annot.e_Watermark);
        excludedTypeList.add(Annot.e_3D);
        excludedTypeList.add(Annot.e_Popup);
        excludedTypeList.add(Annot.e_Redact);
        try {
            int pageCount = doc.getPageCount();
            for (int i = 0; i < pageCount; i++) {
                page = doc.getPage(i);
                count = page.getAnnotCount();
                if (count == 0){
                    continue;
                }
                for (int j = 0; j < count; j++) {
                    Annot annot = page.getAnnot(j);
                    if (!excludedTypeList.contains(annot.getType())){
                        return false;
                    }
                }
            }
        } catch (PDFException e) {
            e.printStackTrace();
        }
        return result;
    }

    public static boolean noAnnotation(FDFDoc doc){
        boolean result = true;
        if (doc == null || doc.isEmpty())return true;
        try {
            PDFDictionary dictionary = doc.getFDFDict();
            if (dictionary != null && dictionary.hasKey("Annots")
                    && dictionary.getElement("Annots").getArray().getElementCount() > 0){
                result =  false;
            }
        } catch (PDFException e) {
            e.printStackTrace();
        }
        return result;
    }

    public static boolean containWidgets(PDFViewCtrl viewCtrl){
        if (viewCtrl == null || viewCtrl.getDoc() == null)return false;

        try {
            if (viewCtrl.isDynamicXFA()) {
                XFADoc doc = viewCtrl.getXFADoc();
                int pageCount = doc.getPageCount();
                int widgetCount = 0;
                for (int i = 0; i < pageCount; i++) {
                    XFAPage page = doc.getPage(i);
                    int count = page.getWidgetCount();
                    if (count == 0) {
                        continue;
                    }

                    widgetCount += count;
                    if (widgetCount > 1){
                        return true;
                    }
                }
            } else {
                Form form = new Form(viewCtrl.getDoc());
                int fieldCount = form.getFieldCount(null);
                form.delete();
                return fieldCount > 1;
            }
        } catch (PDFException e) {
            e.printStackTrace();
        }
        return false;
    }
}
