/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.utils;

import android.annotation.SuppressLint;
import android.app.Activity;
import android.content.Context;
import android.content.res.Configuration;
import android.content.res.Resources;
import android.graphics.Point;
import android.os.Build;
import android.provider.Settings;
import android.util.DisplayMetrics;
import android.util.Log;
import android.view.Display;
import android.view.WindowManager;

import java.lang.ref.WeakReference;
import java.lang.reflect.Method;

@SuppressLint("NewApi")
public class AppDisplay {
    private static DisplayMetrics mMetrics;
    private static int mWidthPixels;
    private static int mHeightPixels;
    private static boolean mPadDevice;
    private static WeakReference<Activity> mAct;
    private static Context mAppContext;
    private static AppDisplay mAppDisplay = null;

    public static void Instance(Activity act) {
        if (mAppDisplay == null || mAct == null ||mAct.get() != act) {
            mAppDisplay = new AppDisplay(act, null);
        }
    }

    public static void Instance(Context context) {
        if (mAct != null && mAct.get() != null) return;

        if (mAppDisplay == null || mAppContext != context) {
            mAppDisplay = new AppDisplay(null, context);
        }
    }

    /**
     * Device DPI
     * 120 === LDPI
     * 160 === MDPI
     * 213 === TVDPI
     * 240 === HDPI
     * 320 === XHDPI
     * 480 === XXHDPI
     */
    private AppDisplay(Activity act, Context context) {
        mAct = new WeakReference<>(act);
        mAppContext  = act != null ? act.getApplicationContext() : context.getApplicationContext();
        mMetrics = mAppContext.getResources().getDisplayMetrics();
        Log.d("AppDisplay", "DPI:" + mMetrics.densityDpi);
        Display display = ((WindowManager) mAppContext.getSystemService(Context.WINDOW_SERVICE)).getDefaultDisplay();
        if (Build.VERSION.SDK_INT < 13) {
            mWidthPixels = mMetrics.widthPixels;
            mHeightPixels = mMetrics.heightPixels;
        } else if (Build.VERSION.SDK_INT == 13) {
            try {
                Method methodWidth = Display.class.getMethod("getRealWidth");
                Method methodHeight = Display.class.getMethod("getRealHeight");
                mWidthPixels = (Integer) methodWidth.invoke(display);
                mHeightPixels = (Integer) methodHeight.invoke(display);
            } catch (Exception e) {
                mWidthPixels = mMetrics.widthPixels;
                mHeightPixels = mMetrics.heightPixels;
            }
        } else if (Build.VERSION.SDK_INT > 13 && Build.VERSION.SDK_INT < 17) {
            try {
                Method methodWidth = Display.class.getMethod("getRawWidth");
                Method methodHeight = Display.class.getMethod("getRawHeight");
                mWidthPixels = (Integer) methodWidth.invoke(display);
                mHeightPixels = (Integer) methodHeight.invoke(display);
            } catch (Exception e) {
                mWidthPixels = mMetrics.widthPixels;
                mHeightPixels = mMetrics.heightPixels;
            }
        } else if (Build.VERSION.SDK_INT >= 17) {
            display.getRealMetrics(mMetrics);

            mWidthPixels = mMetrics.widthPixels;
            mHeightPixels = mMetrics.heightPixels;
        }

        int screenLayout = mAppContext.getResources().getConfiguration().screenLayout;
        mPadDevice = (screenLayout & Configuration.SCREENLAYOUT_SIZE_MASK) >= Configuration.SCREENLAYOUT_SIZE_LARGE;
    }

    public static int getActivityWidth() {
        if (AppDevice.isChromeOs(mAct.get())) {
            DisplayMetrics metrics = mAct.get().getResources().getDisplayMetrics();
            int actWidth = metrics.widthPixels;
            return actWidth;
        } else {
            return getScreenWidth();
        }
    }

    public static int getActivityHeight() {
        if (AppDevice.isChromeOs(mAct.get())) {
            DisplayMetrics metrics = mAct.get().getResources().getDisplayMetrics();
            int actHeight = metrics.heightPixels;
            return actHeight;
        } else {
            return getScreenHeight();
        }
    }

    public static int dp2px(float value) {
        return (int) (value * mMetrics.density + 0.5);
    }

    public static float px2dp(float pxValue) {
        return pxValue / mMetrics.density;
    }

    public static int px2sp(float pxValue) {
        final float fontScale = mMetrics.scaledDensity;
        return (int) (pxValue / fontScale + 0.5f);
    }

    public static int sp2px(float spValue) {
        final float fontScale = mMetrics.scaledDensity;
        return (int) (spValue * fontScale + 0.5f);
    }

    public static int getDisplayWidth() {
        Point screenSize = new Point();
        ((WindowManager) mAppContext.getSystemService(Context.WINDOW_SERVICE)).getDefaultDisplay().getSize(screenSize);
        return screenSize.x;
    }

    public static int getDisplayHeight() {
        Point screenSize = new Point();
        ((WindowManager) mAppContext.getSystemService(Context.WINDOW_SERVICE)).getDefaultDisplay().getSize(screenSize);
        return screenSize.y;
    }

    public static int getScreenWidth() {
        mMetrics = mAppContext.getResources().getDisplayMetrics();
        return mMetrics.widthPixels;
    }

    public static int getScreenHeight() {
        mMetrics = mAppContext.getResources().getDisplayMetrics();
        return mMetrics.heightPixels;
    }

    public static int getDialogWidth() {
        int dlgWidth = 100;
        if (mAppContext.getResources().getConfiguration().orientation == Configuration.ORIENTATION_LANDSCAPE) {
            dlgWidth = getScreenHeight() * 4 / 5;
        } else {
            dlgWidth = getScreenWidth() * 4 / 5;
        }

        int actWidth = getActivityWidth();
        if (dlgWidth > actWidth) {
            dlgWidth = actWidth * 4 / 5;
        }
        return dlgWidth;
    }

    public static int getDialogHeight() {
        int dlgHeight = 100;

        if (mAppContext.getResources().getConfiguration().orientation == Configuration.ORIENTATION_LANDSCAPE) {
            dlgHeight = getScreenWidth() * 3 / 4;
        } else {
            dlgHeight = getScreenHeight() * 3 / 4;
        }

        int actHeight = getActivityHeight();
        if (dlgHeight >= actHeight) {
            dlgHeight = actHeight * 3 / 4;
        }

        return dlgHeight;
    }

    public static int getUITextEditDialogWidth() {
        int dlgWidth = 100;

        if (isPad()) {
            if (mAppContext.getResources().getConfiguration().orientation == Configuration.ORIENTATION_LANDSCAPE) {
                dlgWidth = getScreenHeight() / 2;
            } else {
                dlgWidth = getScreenWidth() / 2;
            }
        } else {
            if (mAppContext.getResources().getConfiguration().orientation == Configuration.ORIENTATION_LANDSCAPE) {
                dlgWidth = getScreenHeight() * 4 / 5;
            } else {
                dlgWidth = getScreenWidth() * 4 / 5;
            }
        }

        int actWidth = getActivityWidth();
        if (dlgWidth >= actWidth) {
            dlgWidth = actWidth * 4 / 5;
        }
        return dlgWidth;
    }

    public static boolean isLandscape() {
        if (getScreenWidth() > getScreenHeight()) {
            return true;
        } else {
            return false;
        }
    }

    public static int getRawScreenWidth() {
        if (isLandscape()) {
            return Math.max(mWidthPixels, mHeightPixels);
        } else {
            return Math.min(mWidthPixels, mHeightPixels);
        }
    }

    public static int getRawScreenHeight() {
        if (isLandscape()) {
            return Math.min(mWidthPixels, mHeightPixels);
        } else {
            return Math.max(mWidthPixels, mHeightPixels);
        }
    }

    public static boolean isPad() {
        return mPadDevice;
    }

    public static float getFingerArea() {
        // dpi / per cm
        return mMetrics.densityDpi / 2.54f;
    }

    public static int getRealNavBarHeight() {
        int result = 0;

        if (hasNavigationBar()) {
            int resourceId = mAppContext.getResources().getIdentifier("navigation_bar_height", "dimen", "android");
            if (resourceId > 0) {
                return mAppContext.getResources().getDimensionPixelSize(resourceId);
            }
        }
        return result;
    }

    public static int getNavBarHeight() {
        int result = 0;

        if (hasNavigationBar()) {
            if (isLandscape()) {
                return 0;
            }

            int resourceId = mAppContext.getResources().getIdentifier("navigation_bar_height", "dimen", "android");
            if (resourceId > 0) {
                return mAppContext.getResources().getDimensionPixelSize(resourceId);
            }
        }
        return result;
    }

    @SuppressLint("PrivateApi")
    private static boolean hasNavigationBar() {
        // Emulator
        if (isEmulator()) {
            return true;
        }

        boolean has_navigation_bar = false;
        int id = mAppContext.getResources().getIdentifier("config_showNavigationBar", "bool", "android");
        if (id > 0) {
            has_navigation_bar = mAppContext.getResources().getBoolean(id);
        }
        try {
            Class systemProperties = Class.forName("android.os.SystemProperties");
            Method method = systemProperties.getMethod("get", String.class);
            String navBarOverride = (String) method.invoke(systemProperties, "qemu.hw.mainkeys");
            int navigation_bar_is_min = 0;
            if (Build.VERSION.SDK_INT < Build.VERSION_CODES.LOLLIPOP) {
                navigation_bar_is_min = Settings.System.getInt(mAppContext.getContentResolver(), "navigationbar_is_min", 0);
            } else {
                navigation_bar_is_min = Settings.Global.getInt(mAppContext.getContentResolver(), "navigationbar_is_min", 0);
            }
            if ("1".equals(navBarOverride) || 1 == navigation_bar_is_min) {
                has_navigation_bar = false;
            } else if ("0".equals(navBarOverride)) {
                has_navigation_bar = true;
            }
        } catch (Exception ignored) {
        }
        return has_navigation_bar;
    }

    private static boolean isEmulator() {
        return Build.FINGERPRINT.startsWith("generic")
                || Build.FINGERPRINT.toLowerCase().contains("vbox")
                || Build.FINGERPRINT.toLowerCase().contains("test-keys")
                || Build.MODEL.contains("google_sdk")
                || Build.MODEL.contains("Emulator")
                || Build.MODEL.contains("Android SDK built for x86")
                || Build.MANUFACTURER.contains("Genymotion")
                || (Build.BRAND.startsWith("generic") && Build.DEVICE.startsWith("generic"))
                || "google_sdk".equals(Build.PRODUCT);
    }

    public static boolean isStatusBarShown(Activity context) {
        WindowManager.LayoutParams params = context.getWindow().getAttributes();
        int paramsFlag = params.flags & (~WindowManager.LayoutParams.FLAG_FULLSCREEN);
        return paramsFlag == params.flags;
    }

    public static int getStatusBarHeight() {
        int height = 0;
        int resourceId = mAppContext.getResources().getIdentifier("status_bar_height", "dimen", "android");
        if (resourceId > 0) {
            height = mAppContext.getResources().getDimensionPixelSize(resourceId);
        }
        return height;
    }

}
