/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.utils;

import android.app.Activity;
import android.content.ContentResolver;
import android.content.ContentUris;
import android.content.Context;
import android.content.Intent;
import android.database.Cursor;
import android.graphics.Bitmap;
import android.media.ExifInterface;
import android.media.MediaMetadataRetriever;
import android.net.Uri;
import android.os.Build;
import android.os.Environment;
import android.os.ParcelFileDescriptor;
import android.provider.DocumentsContract;
import android.provider.MediaStore;
import android.text.TextUtils;
import android.webkit.MimeTypeMap;

import com.foxit.sdk.common.Constants;

import java.io.File;
import java.io.FileDescriptor;
import java.io.FileFilter;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.List;

import androidx.documentfile.provider.DocumentFile;

public class AppFileUtil {
    //Check whether the SD is available.
    public static boolean isSDAvailable() {
        return Environment.getExternalStorageState().equals(Environment.MEDIA_MOUNTED);
    }

    public static String getAppCacheDir(Context context) {
        return context.getApplicationContext().getCacheDir().getAbsolutePath();
    }

    public static String getSDPath() {
        return Environment.getExternalStorageDirectory().getPath();
    }

    public static String getFilePath(Context context, Intent intent, String stringExtra) {
        String filePath = null;
        String action = intent.getAction();
        if (Intent.ACTION_VIEW.equals(action)) {
            Uri uri = intent.getData();
            if (uri.getScheme().equals("file")) {
                filePath = uri.getPath();
            } else {
                try {
                    ContentResolver resolver = context.getContentResolver();

                    String[][] projections = {new String[]{"_display_name"},
                            new String[]{"filename"},
                    };
                    String filename = null;
                    for (String[] projection : projections) {
                        try {
                            Cursor cursor = resolver.query(uri, projection, null, null, null);
                            if (cursor == null) continue;
                            cursor.moveToFirst();
                            int column_index = cursor.getColumnIndex(projection[0]);
                            if (column_index >= 0) {
                                filename = cursor.getString(column_index);
                                cursor.close();
                                break;
                            }
                            cursor.close();
                        } catch (Exception ignored) {
                        }
                    }
                    if (filename == null) {
                        filename = AppDmUtil.randomUUID(null) + ".pdf";
                    }
                    ParcelFileDescriptor parcelFileDescriptor = resolver.openFileDescriptor(uri, "r");
                    filePath = cacheContentPdfFile(context, parcelFileDescriptor != null ? parcelFileDescriptor.getFileDescriptor() : null, filename);
                } catch (Exception e) {
                    e.printStackTrace();
                }
            }
        }

        if (filePath == null && stringExtra != null) {
            filePath = intent.getStringExtra(stringExtra);
        }
        return filePath;
    }

    private static String cacheContentPdfFile(Context context, FileDescriptor fileDescriptor, String filename) {
        String cacheDir = context.getCacheDir().getPath() + File.separator + "contentfile";
        return cacheFile(context, fileDescriptor, filename, cacheDir, true);
    }

    private static String cacheFile(Context context, Uri uri, String srcPath, String cacheDir, String dstName) {
        String dstPath = srcPath;
        if (existsDocumentFile(context, uri)) {
            ContentResolver resolver = context.getContentResolver();
            try {
                ParcelFileDescriptor parcelFileDescriptor = resolver.openFileDescriptor(uri, "r");
                dstPath = cacheFile(context, parcelFileDescriptor != null ? parcelFileDescriptor.getFileDescriptor() : null, dstName,
                        cacheDir, false);
            } catch (Exception e) {
                e.printStackTrace();
            }
        } else {
            dstPath = AppStorageManager.getInstance(context).getScopedCacheFilePath(srcPath);
            File parent = new File(dstPath).getParentFile();
            if (parent != null && !parent.exists()) {
                parent.mkdirs();
            }
        }
        return dstPath;
    }

    private static String cacheFile(Context context, FileDescriptor fileDescriptor, String filename, String cacheDir, boolean forceDelete) {
        File dirFile;
        File file = null;

        dirFile = new File(cacheDir);
        if (dirFile.exists() && forceDelete) {
            deleteFolder(dirFile, false);
        }

        dirFile.mkdirs();
        if (dirFile.exists() && dirFile.isDirectory()) {
            String filePath = cacheDir + "/" + filename;
            file = new File(filePath);
        }

        if (file != null) {
            try {
                if (file.exists() && forceDelete) {
                    file.delete();
                }
                FileInputStream fis = new FileInputStream(fileDescriptor);
                int dataSize = fis.available();
                if (!file.createNewFile()) {
                    if (dataSize == file.length()) {
                        fis.close();
                        return file.getAbsolutePath();
                    }
                }
                cleanDirectory(dirFile, getDirectorySize(dirFile) + dataSize - 2L * 1024 * 1024 * 1024);

                FileOutputStream fos = new FileOutputStream(file);
                byte[] read = new byte[8 * 1024];
                int byteCount;
                while ((byteCount = fis.read(read)) > 0) {
                    fos.write(read, 0, byteCount);
                }
                fis.close();
                fos.flush();
                fos.close();
                return file.getAbsolutePath();
            } catch (IOException e) {
                e.printStackTrace();
                return null;
            }
        }

        return null;
    }

    public static long getFileSize(String filePath) {
        if (AppUtil.isEmpty(filePath))
            return 0;
        long size = 0;
        File file = new File(filePath);
        if (!file.exists()) return size;
        size = file.length();
        return size;
    }

    //Format file size to string.
    public static String formatFileSize(long fileSize) {
        String sizeStr = null;
        float sizeFloat = 0;
        if (fileSize < 1024) {
            sizeStr = Long.toString(fileSize);
            sizeStr += "B";
        } else if (fileSize < (1 << 20)) {
            sizeFloat = (float) fileSize / (1 << 10);
            sizeFloat = (float) (Math.round(sizeFloat * 100)) / 100;
            sizeStr = Float.toString(sizeFloat) + "KB";
        } else if (fileSize < (1 << 30)) {
            sizeFloat = (float) fileSize / (1 << 20);
            sizeFloat = (float) (Math.round(sizeFloat * 100)) / 100;
            sizeStr = Float.toString(sizeFloat) + "MB";
        } else {
            sizeFloat = (float) fileSize / (1 << 30);
            sizeFloat = (float) (Math.round(sizeFloat * 100)) / 100;
            sizeStr = Float.toString(sizeFloat) + "GB";
        }
        return sizeStr;
    }

    public static long getFolderSize(String filePath) {
        long size = 0;
        File file = new File(filePath);
        if (!file.exists()) return size;
        File[] fileList = file.listFiles();
        if (fileList != null) {
            for (File subFile : fileList) {
                if (subFile.isDirectory()) {
                    size += getFolderSize(subFile.getPath());
                } else {
                    size += subFile.length();
                }
            }
        }
        return size;
    }

    public static String getFileFolder(String filePath) {
        int index = filePath.lastIndexOf('/');
        if (index < 0) return "";
        return filePath.substring(0, index);
    }

    public static String getFileName(String filePath) {
        int index = filePath.lastIndexOf('/');
        return (index < 0) ? filePath : filePath.substring(index + 1);
    }

    public static String getFileExt(String filePath) {
        int index = filePath.lastIndexOf('.');
        return (index < 0) ? filePath : filePath.substring(index + 1);
    }

    public static String getFileNameWithoutExt(String filePath) {
        if (AppUtil.isEmpty(filePath)) return null;
        int index = filePath.lastIndexOf('/');
        String name = filePath.substring(index + 1);
        index = name.lastIndexOf('.');
        if (index > 0) {
            name = name.substring(0, index);
        }
        return name;
    }

    //Rename the file path as "xxxxx(num).xxx".
    public static String getFileDuplicateName(String filePath) {
        String newPath = filePath;
        while (true) {
            File file2 = new File(newPath);
            if (file2.exists()) {
                String ext = newPath.substring(newPath.lastIndexOf('.'));
                newPath = newPath.substring(0, newPath.lastIndexOf('.'));
                int begin = 0;
                int end = newPath.length() - 1;
                if (newPath.charAt(end) == ')') {
                    for (int i = end - 1; i >= 0; i--) {
                        char c = newPath.charAt(i);
                        if (c == '(') {
                            begin = i;
                            break;
                        }
                        if (c < '0' || c > '9')
                            break;
                    }
                }
                if (begin > 0 && end - begin < 32) {
                    String num = newPath.substring(begin + 1, end);
                    int index = Integer.parseInt(num, 10) + 1;
                    newPath = newPath.substring(0, begin) + "(" + index + ")" + ext;
                    continue;
                }
                newPath = newPath + "(" + 1 + ")" + ext;
                continue;
            }
            break;
        }
        return newPath;
    }

    public static boolean isFileExist(String filePath) {
        File file = new File(filePath);
        return file.exists();
    }

    public static boolean renameFile(String path, String newPath) {
        File file = new File(path);
        return file.renameTo(new File(newPath));
    }

    public static boolean deleteFile(String path) {
        try {
            File file = new File(path);
            if (file.exists()) {
                return file.delete();
            }
            return true;
        } catch (Exception ignored) {
        }
        return false;
    }

    public static boolean deleteFolder(File dirPath, boolean deleteHistory) {
        boolean flag = false;
        if (!dirPath.isDirectory()) {
            return flag;
        }
        File[] fileList = dirPath.listFiles();
        if (fileList != null) {
            for (File file : fileList) {
                if (file.isFile()) {
                    flag = file.delete();
                } else if (file.isDirectory()) {
                    flag = deleteFolder(file, deleteHistory);
                }
                if (!flag) {
                    break;
                }
            }
        }
        flag = dirPath.delete();
        return flag;
    }

    public static AppFileUtil getInstance() {
        return INSTANCE;
    }

    private AppFileUtil() {
    }

    private static AppFileUtil INSTANCE = new AppFileUtil();

    public static String getDiskCachePath(Context context) {
        if (Environment.MEDIA_MOUNTED.equals(Environment.getExternalStorageState())) {
            File ecd = context.getExternalCacheDir();
            if (ecd != null)
                return context.getExternalCacheDir().getPath();
            else
                return context.getCacheDir().getPath();

        } else {
            return context.getCacheDir().getPath();
        }
    }

    public static String getExternalFilePath(Context context, String type) {
        if (Environment.MEDIA_MOUNTED.equals(Environment.getExternalStorageState())) {
            File ecd = context.getExternalCacheDir();
            if (ecd != null)
                return context.getExternalFilesDir(type).getAbsolutePath();
            else
                return context.getFilesDir().getAbsolutePath();
        } else {
            return context.getFilesDir().getAbsolutePath();
        }
    }

    public static String getFileExtension(String filePath) {
        if (filePath == null) return "";
        int index = filePath.lastIndexOf(".");
        if (index != -1) {
            return filePath.substring(index + 1);
        } else {
            return "";
        }
    }

    public static String replaceFileExtension(String filePath, String ext) {
        if (AppUtil.isEmpty(filePath) || AppUtil.isEmpty(ext)) return "";
        int index = filePath.lastIndexOf(".");
        if (index != -1) {
            return filePath.substring(0, index) + ext;
        }
        return "";
    }

    public static String getFilePathFromUri(Context context, Uri uri) {
        if (null == uri)
            return null;

        String path = null;
        // DocumentProvider
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.KITKAT
                && DocumentsContract.isDocumentUri(context, uri)) {
            // ExternalStorageProvider
            if (isExternalStorageDocument(uri)) {
                final String docId = DocumentsContract.getDocumentId(uri);
                final String[] split = docId.split(":");
                final String type = split[0];
                if ("primary".equalsIgnoreCase(type)) {
                    path = getSDPath() + "/" + split[1];
                    return path;
                }
            }
            // DownloadsProvider
            else if (isDownloadsDocument(uri)) {
                final String id = DocumentsContract.getDocumentId(uri);
                if (id.startsWith("raw:")) {
                    return id.replaceFirst("raw:", "");
                }
                final Uri contentUri = ContentUris.withAppendedId(Uri.parse("content://downloads/public_downloads"), Long.parseLong(id));
                path = getDataColumn(context, contentUri, null, null);
                return path;
            }
            // MediaProvider
            else if (isMediaDocument(uri)) {
                final String docId = DocumentsContract.getDocumentId(uri);
                final String[] split = docId.split(":");
                final String type = split[0];

                Uri contentUri = null;
                if ("image".equals(type)) {
                    contentUri = MediaStore.Images.Media.EXTERNAL_CONTENT_URI;
                } else if ("video".equals(type)) {
                    contentUri = MediaStore.Video.Media.EXTERNAL_CONTENT_URI;
                } else if ("audio".equals(type)) {
                    contentUri = MediaStore.Audio.Media.EXTERNAL_CONTENT_URI;
                }

                final String selection = "_id=?";
                final String[] selectionArgs = new String[]{
                        split[1]
                };
                path = getDataColumn(context, contentUri, selection, selectionArgs);
                return path;
            }
        }

        final String scheme = uri.getScheme();
        if (scheme == null) {
            path = uri.getPath();
        } else if (ContentResolver.SCHEME_FILE.equals(scheme)) {
            path = uri.getPath();
        } else if (ContentResolver.SCHEME_CONTENT.equals(scheme)) {
            path = getDataColumn(context, uri, null, null);
        }
        return path;
    }

    public static String getDataColumn(Context context, Uri uri, String selection, String[] selectionArgs) {
        Cursor cursor = null;
        final String column = "_data";
        final String[] projection = {column};

        try {
            cursor = context.getContentResolver().query(uri, projection, selection, selectionArgs, null);
            if (cursor != null && cursor.moveToFirst()) {
                final int index = cursor.getColumnIndexOrThrow(column);
                if (index > -1)
                    return cursor.getString(index);
            }
        } catch (Exception e) {
            e.printStackTrace();
        } finally {
            if (cursor != null) {
                cursor.close();
            }
        }
        return null;
    }

    /**
     * @param uri The Uri to check.
     * @return Whether the Uri authority is ExternalStorageProvider.
     */
    public static boolean isExternalStorageDocument(Uri uri) {
        return DOCUMENT_EXTERNAL_URI_AUTHORITY_PREFIX.equals(uri.getAuthority());
    }

    /**
     * @param uri The Uri to check.
     * @return Whether the Uri authority is DownloadsProvider.
     */
    public static boolean isDownloadsDocument(Uri uri) {
        return "com.android.providers.downloads.documents".equals(uri.getAuthority());
    }

    /**
     * @param uri The Uri to check.
     * @return Whether the Uri authority is MediaProvider.
     */
    public static boolean isMediaDocument(Uri uri) {
        return "com.android.providers.media.documents".equals(uri.getAuthority());
    }

    public static String getMimeType(String filePath) {
        if (filePath == null) {
            return null;
        }
        int lastIndex = filePath.lastIndexOf('.');
        String suffix = lastIndex >= 0 ? filePath.substring(lastIndex + 1).toLowerCase() : "";
        return MimeTypeMap.getSingleton().getMimeTypeFromExtension(suffix);
    }

    public static Bitmap getVideoThumbnail(String filePath) {
        if (!isFileExist(filePath)) {
            return null;
        }
        MediaMetadataRetriever retriever = new MediaMetadataRetriever();
        retriever.setDataSource(filePath);
        return retriever.getFrameAtTime();

    }

    public static final String DOCUMENT_TREE_URI_PATH_PREFIX = "/tree/primary:";
    public static final String DOCUMENT_URI_SCHEME_PREFIX = "content://";
    public static final String DOCUMENT_EXTERNAL_URI_AUTHORITY_PREFIX = "com.android.externalstorage.documents";

    public static boolean checkCallDocumentTreeUriPermission(Activity activity, int requestCode, Uri treeUri) {
        if (Build.VERSION.SDK_INT < 21 || !needScopedStorageAdaptation())
            return true;
        if (treeUri == null) treeUri = Uri.parse("");
        if (existsDocumentFile(AppUtil.getApplicationContext(), treeUri)) return true;
        if (activity == null)
            return false;
        Intent intent = new Intent(Intent.ACTION_OPEN_DOCUMENT_TREE);
        intent.addFlags(Intent.FLAG_GRANT_READ_URI_PERMISSION);
        intent.addFlags(Intent.FLAG_GRANT_PERSISTABLE_URI_PERMISSION);
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
            intent.putExtra(DocumentsContract.EXTRA_INITIAL_URI, treeUri);
        }
        activity.startActivityForResult(intent, requestCode);
        return false;
    }

    public static Uri toDocumentUriFromPath(String path) {
        Context context = AppUtil.getApplicationContext();
        StringBuilder uriBuilder = new StringBuilder();
        if (TextUtils.isEmpty(path)) return Uri.parse("");
        AppStorageManager storageManager = AppStorageManager.getInstance(context);
        if (storageManager.isRootVolumePath(path)) {
            uriBuilder.append(DOCUMENT_URI_SCHEME_PREFIX + DOCUMENT_EXTERNAL_URI_AUTHORITY_PREFIX + "/tree/primary%3A/document/primary%3A");
        } else {
            String volumePath = storageManager.startWithVolumePath(path);
            if (volumePath != null) {
                uriBuilder.append(DOCUMENT_URI_SCHEME_PREFIX + DOCUMENT_EXTERNAL_URI_AUTHORITY_PREFIX);
                boolean isFile = path.lastIndexOf(".") >= 0;
                String relativePath = path.substring(volumePath.length() + 1);
                int splitSize = relativePath.split("/").length;
                boolean isParentVolume = splitSize == 1 && isFile;
                if (!isParentVolume) {
                    uriBuilder.append("/tree/primary%3A");
                    uriBuilder.append(splitSize > 1 ? relativePath.substring(0, relativePath.indexOf("/")) : relativePath);
                }
                uriBuilder.append("/document/primary%3A");
                uriBuilder.append(Uri.encode(relativePath));
            }
        }
        return Uri.parse(uriBuilder.toString());
    }

    public static boolean needScopedStorageAdaptation() {
        if (Build.VERSION.SDK_INT >= 30) {
            return !mIsExternalStorageLegacy && !mIsExternalStorageManager;
        }
        return !mIsExternalStorageLegacy;
    }

    public static File[] listFiles(Context context, File file, FileFilter fileFilter) {
        if (file != null && !needScopedStorageAdaptation()) {
            return file.listFiles(fileFilter);
        } else if (file != null) {
            if (AppStorageManager.getInstance(context).isRootVolumePath(file.getPath()))
                return file.listFiles(fileFilter);
            List<Uri> subUriList = listUriFroDocumentTree(context, file);
            String[] subFileString = file.list();
            int subFileCount = subFileString == null ? 0 : subFileString.length;
            int subDocumentCount = subUriList.size();
            if (subDocumentCount > 0 && subDocumentCount > subFileCount) {
                File tempFile;
                List<File> fileList = new ArrayList<>();
                for (int i = 0; i < subDocumentCount; i++) {
                    tempFile = new File(getFilePathFromUri(context, subUriList.get(i)));
                    if (fileFilter != null && !fileFilter.accept(tempFile)) {
                        continue;
                    }
                    fileList.add(tempFile);
                }
                subDocumentCount = fileList.size();
                return fileList.toArray(new File[subDocumentCount]);
            } else if (subFileCount > 0) {
                return file.listFiles(fileFilter);
            }
        }
        return new File[0];
    }

    public static boolean isDocumentTreeUri(Uri uri) {
        final List<String> paths = uri.getPathSegments();
        return (paths.size() >= 2 && "tree".equals(paths.get(0)));
    }

    public static String getScopedCachePath(Context context, String path) {
        Uri uri = toDocumentUriFromPath(path);
        String name = getFileName(path);
        String relativeDirectory = getFileFolder(uri.getLastPathSegment().substring(8));
        String cacheDir = context.getCacheDir().getPath()
                + File.separator
                + AppStorageManager.DIRECTORY_SCOPED_CACHE
                + File.separator
                + relativeDirectory;
        return cacheFile(context, uri, path, cacheDir, name);
    }

    public static long getDirectorySize(File directory) {
        long size = 0;
        if (directory != null) {
            File[] files = directory.listFiles();
            if (files != null) {
                for (File file : files) {
                    size += file.length();
                }
            }
        }
        return size;
    }

    public static void cleanDirectory(File directory, long cleanSize) {
        long sizeCleaned = 0;
        File[] files = directory.listFiles();
        if (files != null && cleanSize > 0) {
            for (File file : files) {
                if (file.length() > 0 && file.delete()) {
                    sizeCleaned += file.length();
                    if (sizeCleaned >= cleanSize) {
                        break;
                    }
                }
            }
        }
    }

    public static boolean copyFile(File srcFile, File dstFile) {
        try {
            FileInputStream fis = new FileInputStream(srcFile);
            FileOutputStream fos = new FileOutputStream(dstFile);
            fis.getChannel().transferTo(0, srcFile.length(), fos.getChannel());
            return true;
        } catch (Exception e) {
            e.printStackTrace();
        }
        return false;
    }

    public static int copy(InputStream input, OutputStream output) throws IOException {
        return copyLarge(input, output, new byte[8 * 1024]);
    }

    public static int copyLarge(InputStream input, OutputStream output, byte[] buffer) throws IOException {
        int count = 0;
        int block;
        while (-1 != (block = input.read(buffer))) {
            output.write(buffer, 0, block);
            count += block;
        }
        return count;
    }

    public static void callCreateDocumentAction(Activity activity, int requestCode, String name, String type) {
        if (Build.VERSION.SDK_INT < 21 || activity == null)
            return;
        Intent intent = new Intent(Intent.ACTION_CREATE_DOCUMENT);
        intent.addFlags(Intent.FLAG_GRANT_READ_URI_PERMISSION);
        intent.addFlags(Intent.FLAG_GRANT_WRITE_URI_PERMISSION);
        intent.addFlags(Intent.FLAG_GRANT_PERSISTABLE_URI_PERMISSION);
        intent.setType(type);
        intent.putExtra(Intent.EXTRA_TITLE, name);
        activity.startActivityForResult(intent, requestCode);
    }

    public static String getDefaultDocumentDirectory() {
        return needScopedStorageAdaptation() ? AppFileUtil.getSDPath() : (getSDPath() + File.separator + "FoxitSDK");
    }

    public static String getExternalRootDocumentTreeUriPath() {
        return DOCUMENT_URI_SCHEME_PREFIX + DOCUMENT_EXTERNAL_URI_AUTHORITY_PREFIX + "/tree/primary%3A/document/primary%3A";
    }

    public static String toPathFromDocumentTreeUri(Uri uri) {
        String result = null;
        if (isDocumentTreeUri(uri)) {
            String path = uri.getLastPathSegment();
            if (path != null && path.startsWith("primary:")) {
                result = getSDPath() + File.separator + path.substring(8);
            }
        }
        return result;
    }

    public static boolean existsFileOrDocument(Context context, String path) {
        if (!needScopedStorageAdaptation()) {
            return new File(path).exists();
        } else {
            return existsDocumentFile(context, toDocumentUriFromPath(path));
        }
    }

    public static boolean existsDocumentFile(Context context, Uri uri) {
        Cursor cursor = null;
        try {
            cursor = context.getContentResolver().query(uri, new String[]{
                    DocumentsContract.Document.COLUMN_DOCUMENT_ID}, null, null, null);
            return true;
        } catch (Exception ignored) {
        } finally {
            closeQuietly(cursor);
        }
        return false;
    }

    public static void closeQuietly(AutoCloseable closeable) {
        if (closeable != null) {
            try {
                closeable.close();
            } catch (Exception ignored) {
            }
        }
    }

    public static int getDocumentTreeChildCount(Context context, File file, FileFilter fileFilter) {
        int count = 0;
        if (android.os.Build.VERSION.SDK_INT >= android.os.Build.VERSION_CODES.LOLLIPOP
                && needScopedStorageAdaptation() && file != null && file.exists()) {
            Uri treeUri = toDocumentUriFromPath(file.getPath());
            if (isDocumentTreeUri(treeUri)) {
                final Uri childrenUri = DocumentsContract.buildChildDocumentsUriUsingTree(treeUri,
                        DocumentsContract.getDocumentId(treeUri));
                final ContentResolver resolver = context.getContentResolver();
                Cursor cursor = null;
                try {
                    cursor = resolver.query(childrenUri, new String[]{
                            DocumentsContract.Document.COLUMN_DOCUMENT_ID}, null, null, null);
                    if (fileFilter == null) {
                        count = cursor.getCount();
                    } else {
                        while (cursor.moveToNext()) {
                            if (!fileFilter.accept(new File(cursor.getString(0).replace("primary:", getSDPath() + File.separator))))
                                continue;
                            count++;
                        }
                    }
                } catch (Exception ignored) {
                } finally {
                    closeQuietly(cursor);
                }
            }
        }
        return count;
    }

    public static List<Uri> listUriFroDocumentTree(Context context, File file) {
        List<Uri> results = new ArrayList<>();
        if (android.os.Build.VERSION.SDK_INT >= android.os.Build.VERSION_CODES.LOLLIPOP
                && needScopedStorageAdaptation() && file != null) {
            Uri treeUri = toDocumentUriFromPath(file.getPath());
            if (isDocumentTreeUri(treeUri)) {
                final Uri childrenUri = DocumentsContract.buildChildDocumentsUriUsingTree(treeUri,
                        DocumentsContract.getDocumentId(treeUri));
                final ContentResolver resolver = context.getContentResolver();
                Cursor cursor = null;
                try {
                    cursor = resolver.query(childrenUri, new String[]{
                            DocumentsContract.Document.COLUMN_DOCUMENT_ID}, null, null, null);
                    while (cursor.moveToNext()) {
                        final String documentId = cursor.getString(0);
                        final Uri documentUri = DocumentsContract.buildDocumentUriUsingTree(treeUri,
                                documentId);
                        results.add(documentUri);
                    }
                } catch (Exception ignored) {
                } finally {
                    closeQuietly(cursor);
                }
            }
        }
        return results;
    }

    public static String getAdaptedFilePath(Context context, String path) {
        if (!needScopedStorageAdaptation())
            return path;
        if (new File(path).canRead()) {
            return path;
        }
        return getScopedCachePath(context, path);
    }

    public static boolean canRead(File file) {
        if (file == null) return false;
        boolean canRead = file.canRead();
        if (needScopedStorageAdaptation() && file.exists() && file.isFile()) {
            canRead = true;
        }
        return canRead;
    }

    private static boolean mIsExternalStorageLegacy = true;
    private static boolean mIsExternalStorageManager = false;

    static {
        if (Build.VERSION.SDK_INT >= 29) {
            mIsExternalStorageLegacy = Environment.isExternalStorageLegacy();
        }
        if (Build.VERSION.SDK_INT >= 30) {
            // feature//reader//b
            mIsExternalStorageManager = Environment.isExternalStorageManager();
        }
    }

    public static void updateIsExternalStorageManager() {
        if (Build.VERSION.SDK_INT >= 30) {
            // feature//reader//b
            mIsExternalStorageManager = Environment.isExternalStorageManager();
        } else {
            mIsExternalStorageManager = false;
        }
    }


    public static boolean isExternalStorageLegacy() {
        return mIsExternalStorageLegacy;
    }

    public static boolean isExternalStorageManager() {
        return mIsExternalStorageManager;
    }

    public static boolean createNewDocument(Uri uri, boolean isDirectory) {
        boolean result = false;
        Context context = AppUtil.getApplicationContext();
        DocumentFile documentFile = null;
        List<String> pathSegments;
        if (uri != null && isExternalStorageDocument(uri)) {
            pathSegments = uri.getPathSegments();
            if (pathSegments != null && pathSegments.size() == 4) {
                String[] subPathSegments = null;
                String treePath = pathSegments.get(1);
                String documentPath = pathSegments.get(3);
                if (!treePath.equals(documentPath)) {
                    subPathSegments = documentPath.substring(treePath.length() + 1).split("/");
                }
                DocumentFile parentFile = DocumentFile.fromTreeUri(context, uri);
                if (parentFile != null && parentFile.getUri().equals(uri)) {
                    String uriStr = uri.toString();
                    String parentPath = uriStr.substring(0, uriStr.lastIndexOf(Uri.encode(pathSegments.get(2))));
                    parentFile = DocumentFile.fromTreeUri(context, Uri.parse(parentPath));
                }

                String name;
                if (subPathSegments != null) {
                    int size = subPathSegments.length;
                    for (int i = 0; i < size; i++) {
                        if (parentFile != null) {
                            name = subPathSegments[i];
                            documentFile = parentFile.findFile(name);
                            if (documentFile == null) {
                                if (i < size - 1) {
                                    parentFile = parentFile.createDirectory(name);
                                } else if (i == size - 1) {
                                    if (isDirectory) {
                                        documentFile = parentFile.createDirectory(name);
                                    } else {
                                        documentFile = parentFile.createFile(getMimeType(name), name);
                                    }
                                    result = documentFile != null && documentFile.exists();
                                }
                            } else {
                                parentFile = documentFile;
                            }
                        }
                    }
                }
            }
        }
        return result;
    }

    public static boolean deleteDocumentFile(String path) {
        boolean result = false;
        try {
            Context context = AppUtil.getApplicationContext();
            DocumentFile documentFile = AppStorageManager.getInstance(context).getExistingDocumentFile(toDocumentUriFromPath(path));
            if (documentFile != null && documentFile.delete()) {
                result = true;
            }
        } catch (Exception ignored) {
        }
        return result;
    }

    public static boolean isScopedCachePath(String path) {
        return !TextUtils.isEmpty(path) && path.startsWith(AppStorageManager.getInstance(AppUtil.getApplicationContext()).getScopedCacheDir());
    }

    public static String saveToScopedCache(String path) {
        String cachePath = path;
        if (path == null) return null;
        File file = new File(path);
        if (file.exists() && !file.canRead()) {
            cachePath = AppFileUtil.getScopedCachePath(AppUtil.getApplicationContext(), path);
        }
        return cachePath;
    }

    public static String saveToScopedCache(String path, String dstDir, String dstName) {
        String dstPath = path;
        if (path == null) return null;
        File file = new File(path);
        if (file.exists() && !file.canRead()) {
            Uri uri = toDocumentUriFromPath(path);
            dstPath = cacheFile(AppUtil.getApplicationContext(), uri, path, dstDir, dstName);
        }
        return dstPath;
    }

    public static boolean deleteScopedCacheFile(String path) {
        if (AppFileUtil.isScopedCachePath(path)) {
            try {
                return new File(path).delete();
            } catch (Exception ignored) {
            }
        }
        return false;
    }

    public static String saveToScopedCache(String path, Uri uri) {
        String cachePath = path;
        if (uri == null) return path;
        Uri documentUri = toDocumentUriFromPath(path);
        String relativeDirectory = getFileFolder(documentUri.getLastPathSegment().substring(8));
        String name = getFileName(path);
        Context context = AppUtil.getApplicationContext();
        if (existsDocumentFile(context, uri)) {
            ContentResolver resolver = context.getContentResolver();
            try {
                ParcelFileDescriptor parcelFileDescriptor = resolver.openFileDescriptor(uri, "r");
                String cacheDirPath = context.getCacheDir().getPath()
                        + File.separator
                        + AppStorageManager.DIRECTORY_SCOPED_CACHE
                        + File.separator
                        + relativeDirectory;
                cachePath = cacheFile(context, parcelFileDescriptor != null ? parcelFileDescriptor.getFileDescriptor() : null, name,
                        cacheDirPath, false);
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
        return cachePath;
    }

    public static int readPictureDegree(String path) {
        int degree = 0;
        try {
            ExifInterface exifInterface = new ExifInterface(path);
            int orientation = exifInterface.getAttributeInt(ExifInterface.TAG_ORIENTATION, ExifInterface.ORIENTATION_NORMAL);
            switch (orientation) {
                case ExifInterface.ORIENTATION_ROTATE_90:
                    degree = 90;
                    break;
                case ExifInterface.ORIENTATION_ROTATE_180:
                    degree = 180;
                    break;
                case ExifInterface.ORIENTATION_ROTATE_270:
                    degree = 270;
                    break;
                default:
                    break;
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return degree;
    }

    public static int readPictureRotation(String path) {
        int rotation = Constants.e_Rotation0;
        try {
            ExifInterface exifInterface = new ExifInterface(path);
            int orientation = exifInterface.getAttributeInt(ExifInterface.TAG_ORIENTATION, ExifInterface.ORIENTATION_NORMAL);
            switch (orientation) {
                case ExifInterface.ORIENTATION_ROTATE_90:
                    rotation = Constants.e_Rotation90;
                    break;
                case ExifInterface.ORIENTATION_ROTATE_180:
                    rotation = Constants.e_Rotation180;
                    break;
                case ExifInterface.ORIENTATION_ROTATE_270:
                    rotation = Constants.e_Rotation270;
                    break;
                default:
                    break;
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return rotation;
    }

    public static String saveBitmap(String parent, String child, Bitmap bm) {
        if (bm == null) return "";
        File rootDir = new File(parent);
        if (!rootDir.exists()) {
            rootDir.mkdirs();
        }
        File saveFile = new File(rootDir, child);
        FileOutputStream fos = null;
        try {
            fos = new FileOutputStream(saveFile);
            bm.compress(Bitmap.CompressFormat.PNG, 100, fos);
            fos.flush();
        } catch (IOException e) {
            e.printStackTrace();
        } finally {
            try {
                if (fos != null) {
                    fos.close();
                }
            } catch (IOException e) {
                e.printStackTrace();
            }
        }
        return saveFile.getAbsolutePath();
    }

    public static boolean copyFile(String oldPath, String newPath) {
        try {
            File oldFile = new File(oldPath);
            if (!oldFile.exists()) return false;

            File newFile = new File(newPath);
            if (!newFile.getParentFile().exists()) {
                newFile.getParentFile().mkdirs();
            }
            FileInputStream fileInputStream = new FileInputStream(oldPath);
            FileOutputStream fileOutputStream = new FileOutputStream(newPath);
            byte[] buffer = new byte[1024];
            int byteRead;
            while ((byteRead = fileInputStream.read(buffer)) != -1) {
                fileOutputStream.write(buffer, 0, byteRead);
            }
            fileInputStream.close();
            fileOutputStream.flush();
            fileOutputStream.close();
            return true;
        } catch (Exception e) {
            e.printStackTrace();
            return false;
        }
    }

}
